/*
 * Cobham Gaisler GRSPW-ROUTER APB-Register access driver.
 *
 * Copyright (c) 2016-2022 Cobham Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <asm/uaccess.h>
#include <linux/ioctl.h>
#include <linux/slab.h>

#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/version.h>

#include <linux/grlib/grspw_router.h>
#include <linux/grlib/devno.h>

#define REG_READ(adr) (*(volatile u32 *)(adr))
#define REG_WRITE(adr, value) (*(volatile u32 *)(adr) = (value))

struct router_regs {
	u32 resv1;		/* 0x000 */
	u32 psetup[255];	/* 0x004 */
	u32 resv2[32];		/* 0x400 */
	u32 routes[224];	/* 0x480 */
	u32 pctrl[32];		/* 0x800 */
	u32 psts[32];		/* 0x880 */
	u32 tprescaler;		/* 0x900 */
	u32 treload[31];	/* 0x904 */
	u32 resv3[32];		/* 0x980 */
	u32 cfgsts;		/* 0xA00 */
	u32 timecode;		/* 0xA04 */
	u32 ver;		/* 0xA08 */
	u32 idiv;		/* 0xA0C */
	u32 cfgwe;		/* 0xA10 */
};

struct router_priv {
	struct router_regs *regs;
	struct device *dev;
	int minor;
	struct cdev cdev;	/* Char device */
	int open;
	struct router_hw_info hwinfo;
	int nports;
};

static int router_count;

static int router_open(struct inode *inode, struct file *file);
static int router_release(struct inode *inode, struct file *file);
static long router_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);

struct file_operations router_fops = {
	.owner   = THIS_MODULE,
	.open    = router_open,
	.release = router_release,
	.unlocked_ioctl   = router_ioctl,
};

static int router_open(struct inode *inode, struct file *file)
{
	struct router_priv *priv;

	priv = container_of(inode->i_cdev, struct router_priv, cdev);

	if (priv == NULL) {
		printk (KERN_WARNING "GRSPW_ROUTER: device doesn't exist\n");
		return -ENODEV;
	}

	if ( priv->open ) {
		return 0;
	}

	priv->open = 1;
	file->private_data = priv;

	return 0;
}

static int router_release(struct inode *inode, struct file *file)
{
	struct router_priv *priv;

	priv = container_of(inode->i_cdev, struct router_priv, cdev);

	priv->open = 0;

	return 0;
}

void router_hwinfo(struct router_priv *priv, struct router_hw_info *hwinfo)
{
	unsigned int tmp;

	tmp = REG_READ(&priv->regs->cfgsts);
	hwinfo->nports_spw   = (tmp >> 27) & 0x1f;
	hwinfo->nports_amba  = (tmp >> 22) & 0x1f;
	hwinfo->nports_fifo  = (tmp >> 17) & 0x1f;
	hwinfo->timers_avail = (tmp >>  1) & 0x1;
	hwinfo->pnp_avail    = (tmp >>  0) & 0x1;

	tmp = REG_READ(&priv->regs->ver);
	hwinfo->ver_major   = (tmp >> 24) & 0xff;
	hwinfo->ver_minor   = (tmp >> 16) & 0xff;
	hwinfo->ver_patch   = (tmp >>  8) & 0xff;
	hwinfo->iid         = (tmp >>  0) & 0xff;
}

int router_config_set(struct router_priv *priv, struct router_config *cfg)
{
	int i;

	/* Write only configuration bits in Config register */
	if ( cfg->flags & ROUTER_FLG_CFG ) {
		REG_WRITE(&priv->regs->cfgsts, cfg->config & ~0x4);
	}

	/* Write Instance ID to Version Register */
	if ( cfg->flags & ROUTER_FLG_IID ) {
		REG_WRITE(&priv->regs->ver, cfg->iid);
	}

	/* Write startup-clock-divisor Register */
	if ( cfg->flags & ROUTER_FLG_IDIV ) {
		REG_WRITE(&priv->regs->idiv, cfg->idiv);
	}

	/* Write Timer Prescaler Register */
	if ( cfg->flags & ROUTER_FLG_TPRES ) {
		REG_WRITE(&priv->regs->tprescaler, cfg->timer_prescaler);
	}

	/* Write Timer Reload Register */
	if ( cfg->flags & ROUTER_FLG_TRLD ) {
		for (i=0; i<priv->nports; i++)
			REG_WRITE(&priv->regs->treload[i], cfg->timer_reload[i-1]);
	}

	return 0;
}

int router_config_read(struct router_priv *priv, struct router_config *cfg)
{
	int i;

	cfg->config = REG_READ(&priv->regs->cfgsts) & ~0xffff0007;
	cfg->iid = REG_READ(&priv->regs->ver) & 0xff;
	cfg->idiv = REG_READ(&priv->regs->idiv) & 0xff;
	cfg->timer_prescaler = REG_READ(&priv->regs->tprescaler);
	for (i=0; i<priv->nports; i++)
		cfg->timer_reload[i] = REG_READ(&priv->regs->treload[i]);

	return 0;
}

int router_routes_set(struct router_priv *priv, struct router_routes *routes)
{
	int i;
	for (i=0; i<224; i++)
		REG_WRITE(&priv->regs->routes[i], routes->route[i]);
	return 0;
}

int router_routes_read(struct router_priv *priv, struct router_routes *routes)
{
	int i;
	for (i=0; i<224; i++)
		routes->route[i] = REG_READ(&priv->regs->routes[i]);
	return 0;
}

int router_ps_set(struct router_priv *priv, struct router_ps *ps)
{
	int i;
	unsigned int *p = &ps->ps[0];
	for (i=0; i<255; i++,p++) 
		REG_WRITE(&priv->regs->psetup[i], *p);
	return 0;
}

int router_ps_read(struct router_priv *priv, struct router_ps *ps)
{
	int i;
	unsigned int *p = &ps->ps[0];
	for (i=0; i<255; i++,p++) 
		REG_WRITE(&priv->regs->psetup[i], *p);
	return 0;
}

int router_we_set(struct router_priv *priv, int we)
{
	REG_WRITE(&priv->regs->cfgwe, we & 0x1);
	return 0;
}

int router_port_ctrl(struct router_priv *priv, struct router_port *port)
{
	unsigned int ctrl, sts;

	if ( port->port > priv->nports )
		return -EINVAL;

	ctrl = port->ctrl;
	if ( port->flag & ROUTER_PORTFLG_GET_CTRL ) {
		ctrl = REG_READ(&priv->regs->pctrl[port->port]);
	}
	sts = port->sts;
	if ( port->flag & ROUTER_PORTFLG_GET_STS ) {
		sts = REG_READ(&priv->regs->psts[port->port]);
	}

	if ( port->flag & ROUTER_PORTFLG_SET_CTRL ) {
		REG_WRITE(&priv->regs->pctrl[port->port], port->ctrl);
	}
	if ( port->flag & ROUTER_PORTFLG_SET_STS ) {
		REG_WRITE(&priv->regs->psts[port->port], port->sts);
	}

	port->ctrl = ctrl;
	port->sts = sts;
	return 0;
}

int router_cfgsts_set(struct router_priv *priv, unsigned int cfgsts)
{
	REG_WRITE(&priv->regs->cfgsts, cfgsts);
	return 0;
}

int router_cfgsts_read(struct router_priv *priv, unsigned int *cfgsts)
{
	*cfgsts = REG_READ(&priv->regs->cfgsts);
	return 0;
}

int router_tc_read(struct router_priv *priv, unsigned int *tc)
{
	*tc = REG_READ(&priv->regs->timecode);
	return 0;
}

static long router_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct router_priv *priv;
	void __user *argp = (void __user *)arg;

	priv = (struct router_priv *)filp->private_data;

	if (priv == NULL) {
		printk (KERN_WARNING "GRSPW_ROUTER: ioctl device doesn't exist\n");
		return -ENODEV;
	}

	/* Verify READ/WRITE Access to user provided buffer */
	if (_IOC_DIR(cmd) & (_IOC_WRITE | _IOC_READ))
		if (!access_ok((void __user *)arg, _IOC_SIZE(cmd)))
			return -EFAULT;

	switch (cmd) {


	/* Get Hardware support/information available */
	case GRSPWR_IOCTL_HWINFO:
	{
		struct router_hw_info hwinfo;
		router_hwinfo(priv, &hwinfo);
		if(copy_to_user(argp, &hwinfo, sizeof(struct router_hw_info)))
			return -EFAULT;
		break;
	}

	/* Set Router Configuration */
	case GRSPWR_IOCTL_CFG_SET:
	{
		struct router_config cfg;
		if(copy_from_user(&cfg, argp, sizeof(struct router_config)))
			return -EFAULT;
		return router_config_set(priv, &cfg);
	}

	/* Read Router Configuration */
	case GRSPWR_IOCTL_CFG_GET:
	{
		struct router_config cfg;
		router_config_read(priv, &cfg);
		if(copy_to_user(argp, &cfg, sizeof(struct router_config)))
			return -EFAULT;
		break;
	}

	/* Routes */
	case GRSPWR_IOCTL_ROUTES_SET:
	{
		struct router_routes routes;
		if(copy_from_user(&routes, argp, sizeof(struct router_routes)))
			return -EFAULT;
		return router_routes_set(priv, &routes);
	}

	case GRSPWR_IOCTL_ROUTES_GET:
	{
		struct router_routes routes;
		router_routes_read(priv, &routes);
		if(copy_to_user(argp, &routes, sizeof(struct router_routes)))
			return -EFAULT;
		break;
	}

	/* Port Setup */
	case GRSPWR_IOCTL_PS_SET:
	{
		struct router_ps ps;
		if(copy_from_user(&ps, argp, sizeof(struct router_ps)))
			return -EFAULT;
		return router_ps_set(priv, &ps);
	}

	case GRSPWR_IOCTL_PS_GET:
	{
		struct router_ps ps;
		router_ps_read(priv, &ps);
		if(copy_to_user(argp, &ps, sizeof(struct router_ps)))
			return -EFAULT;
		break;
	}

	/* Set configuration write enable */
	case GRSPWR_IOCTL_WE_SET:
	{
		return router_we_set(priv, (long)argp);
	}

	/* Set/Get Port Control/Status */
	case GRSPWR_IOCTL_PORT:
	{
		struct router_port port;
		int result;
		if(copy_from_user(&port, argp, sizeof(struct router_port)))
			return -EFAULT;
		if ( (result=router_port_ctrl(priv, &port)) )
			return result;
		if(copy_to_user(argp, &port, sizeof(struct router_port)))
			return -EFAULT;
		break;
	}

	/* Set Router Configuration/Status Register */
	case GRSPWR_IOCTL_CFGSTS_SET:
	{
		return router_cfgsts_set(priv, (long)argp);
	}

	/* Get Router Configuration/Status Register */
	case GRSPWR_IOCTL_CFGSTS_GET:
	{
		unsigned int cfgsts;
		router_cfgsts_read(priv, &cfgsts);
		if(copy_to_user(argp, &cfgsts, sizeof(unsigned int)))
			return -EFAULT;
		break;
	}

	/* Get Current Time-Code Register */
	case GRSPWR_IOCTL_TC_GET:
	{
		unsigned int tc;
		router_tc_read(priv, &tc);
		if(copy_to_user(argp, &tc, sizeof(unsigned int)))
			return -EFAULT;
		break;
	}

	default: return -ENOSYS;
	}

	return 0;
}

static int router_of_probe(struct platform_device *of_dev)
{
	struct router_priv *priv;
	int result, err, len;
	dev_t dev_id;
	const int *pampopts;

	/* Skip this device if user has set ampopts to zero, the device may
	 * be used by another OS instance in AMP systems.
	 */
	pampopts = of_get_property(of_dev->dev.of_node, "ampopts", &len);
	if ( pampopts && (len == 4) && (*pampopts == 0) )
		return 0;

	priv = (struct router_priv *)kmalloc(sizeof(*priv), GFP_KERNEL);
	if ( !priv )
		return -ENOMEM;

	memset(priv, 0, sizeof(*priv));

	/* MAP Registers into Kernel Virtual Address Space */
	priv->regs = devm_platform_ioremap_resource(of_dev, 0);
	if (IS_ERR(priv->regs)) {
		err = PTR_ERR(priv->regs);
		goto error;
	}

	/* Register character device in registered region */
	priv->minor = router_count;
	dev_id = MKDEV(SPWROUTER_MAJOR, SPWROUTER_MINOR + priv->minor);
	cdev_init(&priv->cdev, &router_fops);
	result = cdev_add(&priv->cdev, dev_id, 1);
	if ( result ) {
		printk(KERN_ERR "GRSPW_ROUTER: Failed adding CHAR dev\n");
		err = -ENODEV;
		goto error;
	}

	router_hwinfo(priv, &priv->hwinfo);
	priv->open = 0;
	priv->nports = priv->hwinfo.nports_spw + priv->hwinfo.nports_amba +
			priv->hwinfo.nports_fifo;

	/* Link of_dev with priv */
	priv->dev = &of_dev->dev;
	dev_set_drvdata(&of_dev->dev, priv);

	router_count++;

	printk (KERN_INFO "GRSPW_ROUTER[%d]: registered char device\n", priv->minor);
	printk (KERN_INFO "                 ports: %d, %x:%x:%x %x\n", priv->nports,
				priv->hwinfo.ver_major, priv->hwinfo.ver_minor,
				priv->hwinfo.ver_patch, priv->hwinfo.iid);

	return 0;

error:
	return err;
}

static int router_of_remove(struct platform_device *of_dev)
{
	struct router_priv *priv = dev_get_drvdata(&of_dev->dev);

	cdev_del(&priv->cdev);

	kfree(priv);
	dev_set_drvdata(&of_dev->dev, NULL);

	return 0;
}

static const struct of_device_id router_of_match[] = {
	{
	 .name = "GAISLER_SPWROUTER",
	 },
	{
	 .name = "01_08b",
	 },
	{
	 .compatible = "gaisler,spwrtr",
	 },
	{},
};

MODULE_DEVICE_TABLE(of, router_of_match);

static struct platform_driver router_of_driver = {
	.driver = {
		.name = "grlib-spwrouter",
		.owner = THIS_MODULE,
		.of_match_table = router_of_match,
	},
	.probe = router_of_probe,
	.remove = router_of_remove,
};

static int __init grspw_router_init(void)
{
	int result;
	dev_t dev_id;

	/* Register CHAR driver region */
	dev_id = MKDEV(SPWROUTER_MAJOR, SPWROUTER_MINOR);
	result = register_chrdev_region(dev_id, SPWROUTER_DEVCNT, "grspw_router");
	if ( result < 0 ) {
		printk(KERN_WARNING "ROUTER: Failed to register CHAR region\n");
		return 0;
	}

	router_count = 0;

	return platform_driver_register(&router_of_driver);
}

static void __exit grspw_router_cleanup(void)
{
	dev_t dev_id;

	/* Unregister CHAR driver region */
	dev_id = MKDEV(SPWROUTER_MAJOR, SPWROUTER_MINOR);
	unregister_chrdev_region(dev_id, SPWROUTER_DEVCNT);

	platform_driver_unregister(&router_of_driver);
}

module_init(grspw_router_init);
module_exit(grspw_router_cleanup);

MODULE_AUTHOR("Cobham Gaisler AB.");
MODULE_DESCRIPTION("Cobham Gaisler SpaceWire Router APB Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:grlib-grspwrouter");
