/*
 * Test MAPLIB driver
 *
 * Copyright (c) 2016-2023 Frontgrade Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <unistd.h>
#include <sys/time.h>

#include <linux/grlib/maplib.h>

#undef DEBUG_TEST

#ifdef DEBUG_TEST
#define PRINT(str...) printf(str); fflush(NULL)
#define PRINT2(str...) printf(str); fflush(NULL)
#define PRINT3(str...) printf(str); fflush(NULL)
#else
#define PRINT(str...)
#define PRINT2(str...)
#define PRINT3(str...)
#endif
/******** MAP LIBRARY ********/
#define MMAPLIB_POOL 0
struct map {
	void *start;
	size_t length;
	int ofs;
};

struct maplib_mmap_info mmapinfo;
struct map maps[256];
int map_fd = -1;
int map_alloc_idx = 0;
int map_alloc_ofs = 0;

/* Memory MAP memory to use for example */
int mmap_init(int idx, int blk_size, int blk_cnt)
{
	int i;
	void *adr;
	unsigned int start, end, len;
	struct maplib_mmap_info *mapi = &mmapinfo;
	char devname[32];
	struct maplib_setup setup;

	sprintf(devname, "/dev/maplib%d", idx);
	map_fd = open(devname, O_RDWR);
	if ( map_fd < 0 ) {
		printf("Failed to open MMAPLib\n");
		return -1;
	}

	/* Setup MAP */
	setup.blk_size = blk_size;
	setup.blk_cnt = blk_cnt;
	if ( ioctl(map_fd, MAPLIB_IOCTL_SETUP, &setup) ) {
		printf("Failed to Setup MMAPLib: %d\n", errno);
		return -2;
	}

	/* Get MMAP information calculated by driver */
	if ( ioctl(map_fd, MAPLIB_IOCTL_MMAPINFO, mapi) ) {
		printf("Failed to get MMAPINFO, errno: %d\n", errno);
		return -1;
	}

	printf("MMAP INFO: GRSPW Packet Buffers:\n");
	printf("    offset: 0x%x\n", mapi->buf_offset);
	printf("    length: %d (0x%x)\n", mapi->buf_length, mapi->buf_length);
	printf("    MMAP Block size: 0x%x\n", mapi->buf_blk_size);

	memset(maps, 0, sizeof(maps));

	/* Map all SpaceWire Packet Buffers */
	start = mapi->buf_offset;
	end = mapi->buf_offset + mapi->buf_length;

	/* Memory MAP driver's Buffers READ-and-WRITE  */
	adr = mmap(NULL, mapi->buf_length, PROT_READ|PROT_WRITE, MAP_SHARED,
			map_fd, start);
	if ( adr == MAP_FAILED ) {
		printf("MMAP Bufs Failed: %p, errno %d, %x\n", adr, errno, mapi->buf_length);
		return -1;
	}
	i=0;
	while ( start < end ) {
		/* Calc length of MMAP Block */
		if ( (end - start) >= mapi->buf_blk_size )
			len = mapi->buf_blk_size;
		else
			len = end - start;

		maps[i].ofs = start;
		maps[i].start = adr;
		maps[i].length = len;
		
		printf("MAPPED OFS=%x, BASE=%p, LEN=%lx\n", maps[i].ofs, maps[i].start, maps[i].length);
		fflush(NULL);
		i++;

		/* Next Block */
		start += mapi->buf_blk_size;
		adr += len;
	}

	printf("M-Mapping successful\n");

	return 0;
}

volatile int cnt = 0;

int mmap_test()
{
	unsigned long ustart;
	int i;

	fflush(NULL);
	usleep(200000);

	i=0;
	while ( maps[i].length > 0 ) { 

		ustart = (unsigned long)maps[i].start;
#if 1
		while ( (void *)ustart < (maps[i].start+maps[i].length) ) {
			*(volatile long *)ustart = 1;
			ustart += 4*1024;
		}
#endif
#if 0
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		cnt = 1;
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		*(volatile long *)ustart = 1;
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
		asm volatile("nop\n\t"::);
#endif
		i++;
	}

	return 0;
}

int main(void)
{
	pid_t pid;
	char str[64];

	/* Allocate 8Mbyte Memory mapped memory in 64 blocks 
	 * with maximal block size 128kB.
	 */
	if ( mmap_init(MMAPLIB_POOL, 128*1024, 64) ) {
		return 1;
	}

	pid = getpid();

	sprintf(str, "/bin/cat /proc/%d/maps", pid);
	if (system(str)) {
		printf("Calling /bin/cat failed: errno=%d\n", errno);
	}

	printf("TEST READ/Write to every PAGE within block\n");
	mmap_test();
	printf("Successfull\n");

	return 0;
}

