/*
 * Frontgrade Gaisler GRSPW user space access Driver.
 *
 * Copyright (c) 2016-2023 Frontgrade Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <asm/uaccess.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/workqueue.h>
#ifdef CONFIG_LEON
#include <asm/leon.h>
#endif
#include <linux/grlib/grspw.h>
#include <linux/grlib/grspw_user.h>
#include <linux/grlib/maplib.h>

#ifndef init_MUTEX
#define init_MUTEX(a) sema_init(a, 1);
#endif
#ifndef init_MUTEX_LOCKED
#define init_MUTEX_LOCKED(a) sema_init(a, 0);
#endif

/* Which default Memory MAP device node that should be used. User can
 * config maplib pool index per GRSPW Device during runtime using
 * GRSPW_IOCTL_BUFCFG ioctl command.
 */
#define MMAPLIB_POOL_DEFAULT 0

/* Do not enable Debug */
#undef GRSPWU_DEBUG

enum {
	STATE_STOPPED = 0,
	STATE_PAUSED = 1,
	STATE_STARTED = 2,
};

struct grspwu_priv {
	char name[8];		/* GRSPWU[N] Name */
	void *dh;		/* GRSPW Device Handle */
	int index;		/* Device Index */
	int chan_cnt;		/* Number of DMA Channels available in HW */
	unsigned int chan_mask;	/* Enabled (by user) DMA Channel Mask */
	unsigned int chan_avail_mask; /* DMA Channel Available Mask */
	void *dma[4];		/* DMA Channel device handles */
	int dma_rxmaxlen[4];	/* DMA RX Maxlength */
	int state;		/* If DMA Channels are open and ready */
	struct cdev cdev;	/* Char device */
	struct grspw_config cfg;/* Current Configuration */
	struct grspw_bufcfg bufcfg;/* Current Buffer Configuration */
	void *_pkt_tx_structs;	/* Base address of packet structures */
	void *_pkt_rx_structs;	/* Base address of packet structures */

	struct semaphore sem_rx_pktstrs;/* Free RX Pkt structures Semaphore */
	struct semaphore sem_tx_pktstrs;/* Free TX Pkt structure Semaphore */
	struct grspw_list tx_pktstrs;	/* Free TX Pkt structures list */
	struct grspw_list rx_pktstrs;	/* Free RX Pkt structures list */

	/* Memory MAP Library instance */
	struct maplib_drv mmapdrv;
};

/* Set this function pointer from another driver to do custom SpaceWire TimeCode
 * RX handling.
 */
void (*grspwu_tc_isr_callback)(void *data, int tc) = NULL;
void *grspwu_tc_isr_arg = NULL;

struct grspw_bufcfg grspwu_def_bufcfg = 
{
	.rx_pkts = 1024,
	.tx_pkts = 1024,
	.maplib_pool_idx = MMAPLIB_POOL_DEFAULT,
};

static struct grspwu_priv *privu_tab[GRSPW_MAX];
static struct class *grspw_class = NULL;
static int dev_major = 0;

static int grspwu_open(struct inode *inode, struct file *file);
static int grspwu_release(struct inode *inode, struct file *file);
static long grspwu_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);
static ssize_t grspwu_read(struct file *filp, char __user *buff, size_t count, loff_t *offp);
static ssize_t grspwu_write(struct file *filp, const char __user *buff, size_t count, loff_t *offp);


struct file_operations grspwu_fops = {
	.owner          = THIS_MODULE,
	.open           = grspwu_open,
	.release        = grspwu_release,
	.unlocked_ioctl = grspwu_ioctl,
	.compat_ioctl   = grspwu_ioctl,
	.mmap           = NULL,
	.read           = grspwu_read,
	.write          = grspwu_write,
};

void grspwu_stop(struct grspwu_priv *priv);
void grspwu_unmap(int idx, void *p);

static int grspwu_open(struct inode *inode, struct file *file)
{
	struct grspwu_priv *priv;
	struct grspw_hw_sup hwcfg;
	void *dh;

	priv = container_of(inode->i_cdev, struct grspwu_priv, cdev);

	if (priv == NULL) {
		printk (KERN_WARNING "GRSPWU: device doesn't exist\n");
		return -ENODEV;
	}

	/* Take a GRSPW Device, this routine will initialize/clear
	 * hardware registers of the device.
	 */
	dh = grspw_open(priv->index);
	if ( dh == NULL ) {
		printk (KERN_WARNING "GRSPWU: device %d already opened\n", priv->index);
		return -EBUSY;
	}
	priv->dh = dh;
	file->private_data = priv;
	priv->state = STATE_STOPPED;
	priv->dma[0] = grspw_dma_open(priv->dh, 0);
	if ( priv->dma[0] == NULL ) {
		printk (KERN_WARNING "GRSPWU[%d]: failed to open DMA CHAN[0]\n", priv->index);
		grspw_close(priv->dh);
		priv->dh = NULL;
		return -EIO;
	}
	priv->dma[1] = NULL;
	priv->dma[2] = NULL;
	priv->dma[3] = NULL;
	priv->chan_mask = 1; /* Default to only one DMA Channel */
	/* Get number of DMA Channels */
	grspw_hw_support(priv->dh, &hwcfg);
	priv->chan_cnt = hwcfg.ndma_chans;
	priv->chan_avail_mask = ~(-1 << priv->chan_cnt);

	/* Set Default Configuration */
	memcpy(&priv->bufcfg, &grspwu_def_bufcfg, sizeof(grspwu_def_bufcfg));
	priv->_pkt_tx_structs = NULL;
	priv->_pkt_rx_structs = NULL;

	grspw_list_clr(&priv->tx_pktstrs);
	grspw_list_clr(&priv->rx_pktstrs);

	init_MUTEX(&priv->sem_rx_pktstrs);
	init_MUTEX(&priv->sem_tx_pktstrs);

	return 0;
}

static int grspwu_release(struct inode *inode, struct file *file)
{
	struct grspwu_priv *priv = container_of(inode->i_cdev, struct grspwu_priv, cdev);
	int i;

	if ( priv->state == STATE_STARTED )
		grspwu_stop(priv);
	if ( priv->_pkt_tx_structs ) {
		kfree(priv->_pkt_tx_structs);
		priv->_pkt_tx_structs = NULL;
	}
	if ( priv->_pkt_rx_structs ) {
		kfree(priv->_pkt_rx_structs);
		priv->_pkt_rx_structs = NULL;
	}
	priv->state = STATE_STOPPED;
	for (i=0; i<priv->chan_cnt; i++) {
		if (priv->dma[i]) {
			grspw_dma_close(priv->dma[i]);
			priv->dma[i] = NULL;
		}
	}
	grspw_close(priv->dh);
	priv->dh = NULL;
	file->private_data = NULL;

	return 0;
}

/* Configure number of packet structures in driver (always called in stopped mode) */
static int grspwu_bufcfg(struct grspwu_priv *priv, struct grspw_bufcfg *bufcfg)
{
	/* Check buf configuration */
	if ( (bufcfg->rx_pkts < 1) || (bufcfg->tx_pkts < 1) )
		return -EINVAL;
	/* Configuration will take affect upon next start. If configuration
	 * differs from last configuration previous allocated memory (if any)
	 * will be freed.
	 */
	if ( (priv->bufcfg.rx_pkts != bufcfg->rx_pkts) ||
	     (priv->bufcfg.tx_pkts != bufcfg->tx_pkts) ) {
		priv->bufcfg.rx_pkts = bufcfg->rx_pkts;
		priv->bufcfg.tx_pkts = bufcfg->tx_pkts;
		if ( priv->_pkt_tx_structs ) {
			kfree(priv->_pkt_tx_structs);
			priv->_pkt_tx_structs = NULL;
			priv->state = STATE_STOPPED;
		}
		if ( priv->_pkt_rx_structs ) {
			kfree(priv->_pkt_rx_structs);
			priv->_pkt_rx_structs = NULL;
			priv->state = STATE_STOPPED;
		}
	}

	/* Basic sanity check of the input argument */
	if (maplib_drv_chkvalid(bufcfg->maplib_pool_idx))
		return -EINVAL;
	priv->bufcfg.maplib_pool_idx = bufcfg->maplib_pool_idx;

	return 0;
}

/* Configure Core and DMA Channels, RMAP and TimeCode setup */
static int grspwu_config_set(struct grspwu_priv *priv, struct grspw_config *cfg)
{
	int rmap_dstkey;
	int i, retval;

	/* Check RMAP Configuration */
	if ( cfg->rmap_cfg & ~(RMAPOPTS_EN_RMAP|RMAPOPTS_EN_BUF) )
		return -EINVAL;

	/* Check Timecode Configuration, enabling RX IRQ not allowed from here */
	if ( cfg->tc_cfg & ~(TCOPTS_EN_RX|TCOPTS_EN_TX) )
		return -EINVAL;

	/* Check Channel Enable Mask, report error if trying to enable a 
	 * channel that is not available. */
	if ( ~priv->chan_avail_mask & cfg->enable_chan_mask ) {
		return -EINVAL;
	}

	/* 1. Configure Node address and Node mask of DMA Channels
	 *    and default node address of core itself.
	 */
	cfg->adrcfg.promiscuous &= 1;
	grspw_addr_ctrl(priv->dh, &cfg->adrcfg);

	/* 2. Configure RMAP (Enable, Buffer, Destination Key) */
	rmap_dstkey = cfg->rmap_dstkey;
	if ( grspw_rmap_ctrl(priv->dh, &cfg->rmap_cfg, &rmap_dstkey) ) {
		/* RMAP not available in Core, but trying to enable */
		return -EIO;
	}

	/* 3. Timecode setup. If a custom timecode ISR callback handler is 
	 *    installed that routine is installed and RX IRQ is enabled.
	 */
	if ( grspwu_tc_isr_callback ) {
		grspw_tc_isr(priv->dh, grspwu_tc_isr_callback, grspwu_tc_isr_arg);
		cfg->tc_cfg |= TCOPTS_EN_RXIRQ;
	}
	grspw_tc_ctrl(priv->dh, &cfg->tc_cfg);

	/* 4. Configure DMA Channels and remember which channels will be
	 *    used. Only the enabled channels will be activated and started
	 *    during the ioctl(START) operation.
	 */
	retval = 0;
	for (i=0; i<priv->chan_cnt; i++) {
		/* Open or close DMA Channel */
		if ( cfg->enable_chan_mask & (1<<i) ) {
			/* Should be opened */
			if ( priv->dma[i] == NULL ) {
				priv->dma[i] = grspw_dma_open(priv->dh, i);
				if ( priv->dma[i] == NULL ) {
					retval = -EIO;
					continue;
				}
				priv->chan_mask |= (1<<i);
			}

			/* Configure channel */
			grspw_dma_config(priv->dma[i], &cfg->chan[i]);
		} else {
			/* Should be closed */
			if ( priv->dma[i] ) {
				grspw_dma_stop(priv->dma[i]);
				grspw_dma_close(priv->dma[i]);
				priv->dma[i] = NULL;
			}
			priv->chan_mask &= ~(1<<i);
		}
	}

	return retval;
}

static void grspwu_config_read(struct grspwu_priv *priv, struct grspw_config *cfg)
{
	int i;
	int rmap_dstkey;

	cfg->adrcfg.promiscuous = -1;
	grspw_addr_ctrl(priv->dh, &cfg->adrcfg);

	cfg->rmap_cfg = -1;
	rmap_dstkey = -1;
	grspw_rmap_ctrl(priv->dh, &cfg->rmap_cfg, &rmap_dstkey);
	cfg->rmap_dstkey = rmap_dstkey;

	cfg->tc_cfg = -1;
	grspw_tc_ctrl(priv->dh, &cfg->tc_cfg);

	cfg->enable_chan_mask = priv->chan_mask;
	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] ) {
			grspw_dma_config_read(priv->dma[i], &cfg->chan[i]);
		}
	}
}

static void grspwu_stats_read(struct grspwu_priv *priv, struct grspw_stats *stats)
{
	int i;

	grspw_stats_read(priv->dh, &stats->stats);

	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] )
			grspw_dma_stats_read(priv->dma[i], &stats->chan[i]);
		else
			memset(&stats->chan[i], 0, sizeof(struct grspw_dma_stats));
	}
}

static void grspwu_stats_clr(struct grspwu_priv *priv)
{
	int i;

	grspw_stats_clr(priv->dh);

	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] )
			grspw_dma_stats_clr(priv->dma[i]);
	}
}

static int grspwu_link_ctrl(struct grspwu_priv *priv, struct grspw_link_ctrl *ctrl)
{
	int clkdiv, stscfg;

	/* Check Configuration */
	if ( ctrl->ctrl & ~LINKOPTS_MASK )
		return -EINVAL;

	/* Configure */
	clkdiv = ctrl->clkdiv_start<<8 | ctrl->clkdiv_run;
	grspw_link_ctrl(priv->dh, &ctrl->ctrl, &stscfg, &clkdiv);

	return 0;
}

static int grspwu_port_ctrl(struct grspwu_priv *priv, int options)
{
	int port;


	if ( (options != 0) && (options != 1)  )
		port = 3; /* Let hardware select port */
	else
		port = options;

	/* Set selected port, if failure it indicates that the selected
	 * port does not exist (HW created with only one port).
	 */
	if ( grspw_port_ctrl(priv->dh, &port) ) 
		return -EIO;

	return 0;
}

static void grspwu_link_state_get(struct grspwu_priv *priv, struct grspw_link_state *state)
{
	int clkdiv;

	/* Get Current Link Configuration */
	state->link_ctrl = -1;
	clkdiv = -1;
	grspw_link_ctrl(priv->dh, &state->link_ctrl, NULL, &clkdiv);
	state->clkdiv_start = (clkdiv >> 8) & 0xff;
	state->clkdiv_run = clkdiv & 0xff;

	/* Get Current Link Status */
	state->link_state = grspw_link_state(priv->dh);

	/* Get Current Port configuration */
	state->port_cfg = -1;
	grspw_port_ctrl(priv->dh, &state->port_cfg);

	/* Get Current Active port */
	state->port_active = grspw_port_active(priv->dh);
}

/* Set TCTRL and TIMECNT, and Send Time-code */
static void grspwu_tc_send(struct grspwu_priv *priv, unsigned int options)
{
	int time;

	/* Set TCTRL and TIMECNT? */
	if ( options & GRSPW_TC_SEND_OPTION_SET_TCTRL ) {
		time = options & GRSPW_TC_SEND_OPTION_TCTRL;
		grspw_tc_time(priv->dh, &time);
	}

	/* Send Time Code (Generate Tick-In) */
	if ( options & GRSPW_TC_SEND_OPTION_TCTX )
		grspw_tc_tx(priv->dh);
}

static void grspwu_tc_read(struct grspwu_priv *priv, int *tc)
{
	*tc = -1;
	grspw_tc_time(priv->dh, tc);
}

static void grspwu_qpktcnt_read(struct grspwu_priv *priv,
				struct grspw_qpktcnt *qpktcnt)
{
	int i;
	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] ) {
			grspw_dma_rx_count(
				priv->dma[i],
				&qpktcnt->chan[i].rx_ready,
				&qpktcnt->chan[i].rx_sched,
				&qpktcnt->chan[i].rx_recv,
				&qpktcnt->chan[i].rx_hw);
			grspw_dma_tx_count(
				priv->dma[i],
				&qpktcnt->chan[i].tx_send,
				&qpktcnt->chan[i].tx_sched,
				&qpktcnt->chan[i].tx_sent,
				&qpktcnt->chan[i].tx_hw);
		}
	}
}

static int grspwu_start(struct grspwu_priv *priv, int options)
{
	int size, i;
	struct grspw_pkt *pktbase;
	struct grspw_dma_config cfg;

	if ( priv->state == STATE_STARTED )
		return 0;

	/* DMA Channels have been opened and configured at this point,
	 * or user is satisfied with the default configuration.
	 */

	/* Allocate Packet structures if not already allocated */
	if ( priv->_pkt_tx_structs == NULL ) {
		size = sizeof(struct grspw_pkt) * priv->bufcfg.tx_pkts;
		priv->_pkt_tx_structs = kmalloc(size, GFP_KERNEL);
		if ( priv->_pkt_tx_structs == NULL )
			return -ENOMEM;
	}
	if ( priv->_pkt_rx_structs == NULL ) {
		size = sizeof(struct grspw_pkt) * priv->bufcfg.rx_pkts;
		priv->_pkt_rx_structs = kmalloc(size, GFP_KERNEL);
		if ( priv->_pkt_rx_structs == NULL )
			return -ENOMEM;
	}
	/* Initialize Packet Structures to one long chain */
	pktbase = (struct grspw_pkt *)priv->_pkt_tx_structs;
	for (i=0; i<priv->bufcfg.tx_pkts; i++) {
		pktbase[i].next = &pktbase[i+1];
		pktbase[i].pkt_id = 0;
		pktbase[i].flags = 0;
		pktbase[i].reserved = 0;
		pktbase[i].hlen = 0;
		pktbase[i].dlen = 0;
		pktbase[i].data = GRSPW_PKT_DMAADDR_INVALID;
		pktbase[i].hdr = GRSPW_PKT_DMAADDR_INVALID;
	}
	/* Initialize TX Packet structure List */
	priv->tx_pktstrs.head = &pktbase[0];
	priv->tx_pktstrs.tail = &pktbase[priv->bufcfg.tx_pkts-1];
	priv->tx_pktstrs.tail->next = NULL;
	pktbase = (struct grspw_pkt *)priv->_pkt_rx_structs;
	for (i=0; i<priv->bufcfg.rx_pkts; i++) {
		pktbase[i].next = &pktbase[i+1];
		pktbase[i].pkt_id = 0;
		pktbase[i].flags = 0;
		pktbase[i].reserved = 0;
		pktbase[i].hlen = 0;
		pktbase[i].dlen = 0;
		pktbase[i].data = GRSPW_PKT_DMAADDR_INVALID;
		pktbase[i].hdr = GRSPW_PKT_DMAADDR_INVALID;
	}
	/* Initialize RX Packet structure List */
	priv->rx_pktstrs.head = &pktbase[0];
	priv->rx_pktstrs.tail = &pktbase[priv->bufcfg.rx_pkts-1];
	priv->rx_pktstrs.tail->next = NULL;
	priv->state = STATE_STOPPED;

	/* Get DMA Channel Configuration: RX Packet Max Length */
	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] ) {
			grspw_dma_config_read(priv->dma[i], &cfg);
			priv->dma_rxmaxlen[i] = cfg.rxmaxlen;
		}
	}

	/* Register at Memory MAP Library, this ensures that if user try to
	 * change memory map or similar this driver will be informed and
	 * all DMA activity to/from the mapped memory will be stopped, it
	 * will be interpreted as an ioctl(STOP) event.
	 */
	priv->mmapdrv.priv = priv;
	priv->mmapdrv.name = priv->name;
	priv->mmapdrv.unmap = grspwu_unmap;
	priv->mmapdrv.next = NULL;
	if ( maplib_drv_reg(priv->bufcfg.maplib_pool_idx,
			    &priv->mmapdrv) ) {
		/* Failed to register, this is because user has not setup
		 * or mapped all memory to user space. We cannot proceed
		 * until user has fixed this.
		 */
		return -EPERM;
	}

	/* START all Channels for DMA operations */
	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] ) {
			if ( grspw_dma_start(priv->dma[i]) ) {
				/* Failed to start DMA channel */
				goto err_out;
			}
		}
	}

	priv->state = STATE_STARTED;

	return 0;

err_out:
	/* Remove MMAP callback */
	maplib_drv_unreg(priv->bufcfg.maplib_pool_idx, &priv->mmapdrv);

	/* Something bad happened, stop all opened DMA channels */
	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] )
			grspw_dma_stop(priv->dma[i]);
	}
	return -EIO;
}

void grspwu_stop(struct grspwu_priv *priv)
{
	int i;

	if ( priv->state < STATE_STARTED )
		return; /* Already stopped */

	/* DMA operation has stopped but user may still read out
	 * buffers. writes are disabled by this. When internal
	 * structures are destroyed, then we enter STATE_STOPPED.
	 */
	priv->state = STATE_PAUSED;

	/* Stop all opened DMA channels */
	for (i=0; i<priv->chan_cnt; i++) {
		if ( priv->dma[i] )
			grspw_dma_stop(priv->dma[i]);
	}

	/* Remove MMAP callback now that we do not use any more of that
	 * memory, user may still access the memory, however this driver
	 * or hardware will not
	 */
	maplib_drv_unreg(priv->bufcfg.maplib_pool_idx, &priv->mmapdrv);
}

void grspwu_unmap(int idx, void *p)
{
	struct grspwu_priv *priv = (struct grspwu_priv *)p;

	if (idx != priv->bufcfg.maplib_pool_idx)
		return;

	/* User has done something bad: unmapped memory without stopping
	 * the current SpaceWire operation. To avoid memory overwriting
	 * all DMA activity to this memory must stop.
	 */
	grspwu_stop(priv);
}

static int grspwu_rx_wait(struct grspwu_priv *priv, struct grspw_rx_wait_chan *wait)
{
	int rc;

	if (wait->chan >= priv->chan_cnt)
		return -EINVAL;
	if (priv->dma[wait->chan] == NULL)
		return -EBUSY;

	rc = grspw_dma_rx_wait(priv->dma[wait->chan],
			       wait->recv_cnt, wait->op, wait->ready_cnt,
			       wait->timeout_ms * HZ/1000);
	if (rc == -1)
		return -EAGAIN; /* unknown error */
	if (rc == 1)
		return -EIO;
	else if (rc == 2)
		return -ETIME;
	else if (rc == 3)
		return -EBUSY;
	else
		return 0;
}

static int grspwu_tx_wait(struct grspwu_priv *priv, struct grspw_tx_wait_chan *wait)
{
	int rc;

	if (wait->chan >= priv->chan_cnt)
		return -EINVAL;
	if (priv->dma[wait->chan] == NULL)
		return -EBUSY;

	rc = grspw_dma_tx_wait(priv->dma[wait->chan],
			       wait->send_cnt, wait->op, wait->sent_cnt,
			       wait->timeout_ms * HZ/1000);
	if (rc == -1)
		return -EAGAIN; /* unknown error */
	if (rc == 1)
		return -EIO;
	else if (rc == 2)
		return -ETIME;
	else if (rc == 3)
		return -EBUSY;
	else
		return 0;
}

static long grspwu_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct grspwu_priv *priv;
	void __user *argp = (void __user *)arg;

	priv = (struct grspwu_priv *)filp->private_data;

	if (priv == NULL) {
		printk (KERN_WARNING "GRSPWU: device %d doesnt exist\n", priv->index);
		return -ENODEV;
	}

	/* Verify READ/WRITE Access to user provided buffer */
	if (_IOC_DIR(cmd) & (_IOC_WRITE | _IOC_READ))
		if (!access_ok((void __user *)arg, _IOC_SIZE(cmd)))
			return -EFAULT;

	switch (cmd) {

	/* Read hardware support */
	case GRSPW_IOCTL_HWSUP:
	{
		struct grspw_hw_sup hwcfg;
		grspw_hw_support(priv->dh, &hwcfg);
		if(copy_to_user(argp, &hwcfg, sizeof(struct grspw_hw_sup)))
			return -EFAULT;
		break;
	}

	/* Set buffer configuration */
	case GRSPW_IOCTL_BUFCFG:
	{
		struct grspw_bufcfg bufcfg;
		if ( priv->state == STATE_STARTED)
			return -EBUSY;
		if(copy_from_user(&bufcfg, argp, sizeof(struct grspw_bufcfg)))
			return -EFAULT;
		return grspwu_bufcfg(priv, &bufcfg);
	}

	/* Set Configuration */
	case GRSPW_IOCTL_CONFIG_SET:
	{
		struct grspw_config cfg;
		if ( priv->state == STATE_STARTED )
			return -EBUSY;
		if(copy_from_user(&cfg, argp, sizeof(struct grspw_config)))
			return -EFAULT;
		return grspwu_config_set(priv, &cfg);
	}

	/* Read current configuration */
	case GRSPW_IOCTL_CONFIG_READ:
	{
		struct grspw_config cfg;
		grspwu_config_read(priv, &cfg);
		if(copy_to_user(argp, &cfg, sizeof(struct grspw_config)))
			return -EFAULT;
		break;
	}

	/* Read statistics from all DMA channels and global Core paramters */
	case GRSPW_IOCTL_STATS_READ:
	{
		struct grspw_stats stats;
		grspwu_stats_read(priv, &stats);
		if(copy_to_user(argp, &stats, sizeof(struct grspw_stats)))
			return -EFAULT;
		break;
	}

	/* Clear Statistics for all DMA channels and Core */
	case GRSPW_IOCTL_STATS_CLR:
	{
		grspwu_stats_clr(priv);
		break;
	}

	case GRSPW_IOCTL_LINKCTRL:
	{
		struct grspw_link_ctrl ctrl;
		if(copy_from_user(&ctrl, argp, sizeof(struct grspw_link_ctrl)))
			return -EFAULT;
		return grspwu_link_ctrl(priv, &ctrl);
	}

	case GRSPW_IOCTL_PORTCTRL:
	{
		return grspwu_port_ctrl(priv, (long)argp);
	}

	case GRSPW_IOCTL_LINKSTATE:
	{
		struct grspw_link_state state;
		grspwu_link_state_get(priv, &state);
		if(copy_to_user(argp, &state, sizeof(struct grspw_link_state)))
			return -EFAULT;
		break;
	}

	case GRSPW_IOCTL_TC_SEND:
	{
		grspwu_tc_send(priv, (unsigned long)argp);
		break;
	}

	case GRSPW_IOCTL_TC_READ:
	{
		int tc;
		grspwu_tc_read(priv, &tc);
		if(copy_to_user(argp, &tc, sizeof(int)))
			return -EFAULT;
		break;
	}

	case GRSPW_IOCTL_QPKTCNT:
	{
		struct grspw_qpktcnt qpktcnt;
		grspwu_qpktcnt_read(priv, &qpktcnt);
		if(copy_to_user(argp, &qpktcnt, sizeof(struct grspw_qpktcnt)))
			return -EFAULT;
		break;
	}

	case GRSPW_IOCTL_STATUS_READ:
	{
		unsigned int status;
		status = grspw_link_status(priv->dh);
		if(copy_to_user(argp, &status, sizeof(unsigned int)))
			return -EFAULT;
		break;
	}

	case GRSPW_IOCTL_STATUS_CLR:
	{
		unsigned int clr_mask;
		if ( priv->state == STATE_STARTED)
			return -EBUSY;
		if(copy_from_user(&clr_mask, argp, sizeof(unsigned int)))
			return -EFAULT;
		grspw_link_status_clr(priv->dh, clr_mask);
		break;
	}

	case GRSPW_IOCTL_START:
	{
		return grspwu_start(priv, (long)argp);
	}

	case GRSPW_IOCTL_STOP:
	{
		grspwu_stop(priv);
		break;
	}

	case GRSPW_IOCTL_RX_WAIT:
	{
		struct grspw_rx_wait_chan wait;
		if(copy_from_user(&wait, argp, sizeof(struct grspw_rx_wait_chan)))
			return -EFAULT;
		return grspwu_rx_wait(priv, &wait);
	}

	case GRSPW_IOCTL_TX_WAIT:
	{
		struct grspw_tx_wait_chan wait;
		if(copy_from_user(&wait, argp, sizeof(struct grspw_tx_wait_chan)))
			return -EFAULT;
		return grspwu_tx_wait(priv, &wait);
	}
	
	default: return -ENOSYS;
	}

	return 0;
}

/* Reclaim Packets from one particular DMA Channel to user space */
static int grspwu_reclaim(
	struct grspwu_priv *priv, int dmachan,
	struct grspw_rtxpkt __user *dst, int max)
{
	struct grspw_list sentpkts;
	struct grspw_pkt *pkt;
	void *ch = priv->dma[dmachan];
	int count, pkts_cnt;
	struct grspw_rtxpkt pkts[64];
	int retval = 0;
#ifdef GRSPWU_DEBUG
	int tmp;
#endif

	/* Check that DMA Channel is open */
	if ( ch == NULL )
		return 0;

	grspw_list_clr(&sentpkts);

	/* Read up to MAX packets */
	count = max;
	grspw_dma_tx_reclaim(ch, 0, &sentpkts, &count);
#ifdef GRSPWU_DEBUG
	tmp = grspw_list_cnt(&sentpkts);
	printk(KERN_DEBUG "tx_reclaim(): req %d got %d packets, chain: %d\n", max, count, tmp);
	if ( tmp != count ) {
		printk(KERN_DEBUG "tx_reclaim(): check1 failed\n");
		return -EIO;
	}
#endif
	if ( count < 1 ) {
		/* IMPLEMENT BLOCKING RECLAIM PER DMA CHANNEL HERE.
		 * The ioctl() implements the TX_WAIT currenlty.
		 */

		/* If blocking wait until a number of Packets are available */
		
		return 0;
	}

	/* Convert packet list into "char-stream" in the format of
	 * 'struct grspw_rtxpkt'
	 */
	pkt = sentpkts.head;
	pkts_cnt = 0;
	while ( pkt ) {
		/* Convert Packet from grspw-lib packet to user space packet */
		pkts[pkts_cnt].flags = pkt->flags;
		pkts[pkts_cnt].dma_chan = dmachan;
		pkts[pkts_cnt].resv1 = 0;
		pkts[pkts_cnt].pkt_id = pkt->pkt_id;
		pkts_cnt++;
		
		/* Sanity check */
		if ( pkts_cnt > max ) {
			printk(KERN_DEBUG "grspwu_reclaim(): max=%d overriden %d: internal bug\n", max, count);
		}

		/* Copy packets to User space in chunks of 64 packets at a time */
		if ( pkts_cnt >= 64 ) {
			if ( copy_to_user(dst, &pkts[0], sizeof(pkts)) ) {
				retval = -EFAULT;
				goto out;
			}

			/* Write to user-space */
			dst += 64;
			pkts_cnt = 0;
		}
		pkt = pkt->next;
	}
	if ( pkts_cnt > 0 ) {
		/* Write remaining prepared packets to user space */
		if ( copy_to_user(dst, &pkts[0], pkts_cnt*sizeof(struct grspw_rtxpkt)) ) {
			retval = -EFAULT;
			goto out;
		}
	}
	retval = count;

out:
	/* Take List lock */
	while ( down_interruptible(&priv->sem_tx_pktstrs) )
		;

	/* Queue up packet structs for reuse */
	if ( grspw_list_is_empty(&sentpkts) == 0 )
		grspw_list_prepend_list(&priv->tx_pktstrs, &sentpkts);

	/* Release Lock */
	up(&priv->sem_tx_pktstrs);

	return retval;
}

/* Read Received Packets from one particular DMA Channel to user space */
static int grspwu_recv(
	struct grspwu_priv *priv, int dmachan,
	struct grspw_rrxpkt __user *dst, int max)
{
	struct grspw_list recvpkts;
	struct grspw_pkt *pkt;
	void *ch = priv->dma[dmachan];
	int count, pkts_cnt;
	struct grspw_rrxpkt pkts[32];
	int retval = 0;
#ifdef GRSPWU_DEBUG
	int tmp;
#endif

	/* Check that DMA Channel is open */
	if ( ch == NULL )
		return 0;

	grspw_list_clr(&recvpkts);

	/* Read up to MAX packets */
	count = max;
	grspw_dma_rx_recv(ch, 0, &recvpkts, &count);
#ifdef GRSPWU_DEBUG
	tmp = grspw_list_cnt(&recvpkts);
	printk(KERN_DEBUG "grspwu_recv(): req %d got %d packets, chain: %d\n", max, count, tmp);
	if ( tmp != count ) {
		printk(KERN_DEBUG "grspwu_recv(): check1 failed\n");
		return -EIO;
	}
#endif
	if ( count < 1 ) {
		/* IMPLEMENT BLOCKING RECV PER DMA CHANNEL HERE.
		 * Currently ioctl() implements the RX_WAIT instead.
		 */

		/* If blocking wait until a number of Packets are available */
		return 0;
	}

	/* Convert packet list into "char-stream" in the format of
	 * 'struct grspw_rrxpkt'
	 */
	pkt = recvpkts.head;
	pkts_cnt = 0;
	while ( pkt ) {
		/* Convert Packet from grspw-lib packet to user space packet */
		pkts[pkts_cnt].flags = pkt->flags;
		pkts[pkts_cnt].dma_chan = dmachan;
		pkts[pkts_cnt].resv1 = 0;
		pkts[pkts_cnt].dlen = pkt->dlen;
		/* UserSpaceAdr temporarily stored in HDR field */
		pkts[pkts_cnt].data = (void *)((unsigned long)pkt->hdr);
		pkts[pkts_cnt].pkt_id = pkt->pkt_id;
#ifdef GRSPWU_DEBUG
		if ( (pkts[pkts_cnt].flags | pkt->flags) & 0x0900 ) {
			printk(KERN_DEBUG "RECV: 0x%x - 0x%x\n", pkts[pkts_cnt].flags, pkt->flags);
		}
#endif
		pkts_cnt++;

		/* Copy packets to User space in chunks of 32 packets at a time */
		if ( pkts_cnt >= 32 ) {
			if ( copy_to_user(dst, &pkts[0], sizeof(pkts)) ) {
				retval = -EFAULT;
				goto out;
			}

			/* Write to user-space */
			dst += 32;
			pkts_cnt = 0;
		}
		pkt = pkt->next;
	}
	if ( pkts_cnt > 0 ) {
		/* Write remaining prepared packets to user space */
		if ( copy_to_user(dst, &pkts[0], pkts_cnt*sizeof(struct grspw_rrxpkt)) ) {
			retval = -EFAULT;
			goto out;
		}
	}
	retval = count;
out:
	/* Take List lock */
	while ( down_interruptible(&priv->sem_rx_pktstrs) )
		;

	/* Queue up packet structs for reuse */
	grspw_list_prepend_list(&priv->rx_pktstrs, &recvpkts);

	/* Release Lock */
	up(&priv->sem_rx_pktstrs);

	return retval;
}

static int grspwu_send(struct grspwu_priv *priv, int dmachan,
		struct grspw_wtxpkt __user *src, int cnt)
{
	struct grspw_list txpkts;
	struct grspw_pkt *pkt;
	void *ch = priv->dma[dmachan];
	int pktidx, count, size, left;
	struct grspw_wtxpkt pkts[32];
	int retval = 0;
	phys_addr_t pa;
#ifdef GRSPWU_DEBUG
	int tmp, tmp2;
#endif

	/* Check that DMA Channel is open */
	if ( ch == NULL )
		return 0;

	/* One list per DMA Channel */
	grspw_list_clr(&txpkts);
	count = 0;
#ifdef GRSPWU_DEBUG
	tmp2 = cnt;
#endif

	while ( down_interruptible(&priv->sem_tx_pktstrs) )
		;

	/* Take out as many Packet Structs from FREE Packet List as needed */
	cnt = grspw_list_take_head_list(&priv->tx_pktstrs, &txpkts, cnt);

	/* Release Lock */
	up(&priv->sem_tx_pktstrs);
#ifdef GRSPWU_DEBUG
	tmp = grspw_list_cnt(&txpkts);
	printk(KERN_DEBUG "grspwu_send(): req %d got %d packets, chain: %d\n", tmp2, count, tmp);
	if ( tmp != cnt ) {
		printk(KERN_DEBUG "grspwu_send(): check1 failed\n");
		return -EIO;
	}
#endif
	if ( cnt < 1 )
		return 0;

	/* Convert packet list into "char-stream" in the format of
	 * 'struct grspw_rrxpkt'
	 */
	pkt = txpkts.head;
	pktidx = 32;
	left = cnt;
	while ( (left > 0) && pkt ) {
		if ( pktidx >= 32 ) {
			if ( left < 32 ) {
				size = left * sizeof(struct grspw_wtxpkt);
			} else {
				size = sizeof(pkts);
			}
			if ( copy_from_user(&pkts[0], src, size) ) {
				retval = -EFAULT;
				goto out;
			}

			/* Write to user-space */
			src += 32; /* wrong to increase when left<32, but never used */
			pktidx = 0;
		}
		/* Convert Packet from grspw-lib packet to user space packet */
		pkt->flags = pkts[pktidx].flags & TXPKT_FLAG_INPUT_MASK;
		pkt->reserved = 0;
		pkt->hlen = pkts[pktidx].hlen;
		pkt->dlen = pkts[pktidx].dlen;
		if ( (pkt->dlen > 0) && maplib_lookup_kern(
				priv->bufcfg.maplib_pool_idx,
				pkts[pktidx].data,
				NULL, /* Kernel address Unused */
				&pa, /* hardware address used in DMA */
				pkt->dlen)) {
			/* Address translation Failed, memory may go over a 
			 * block boundary, or comletely outside. Bug in USER
			 * packet handling, or an attempt to destory kernel
			 * memory.
			 */
			printk(KERN_DEBUG "GRSPWU: send(data): %d,%p,%d,0x%x\n", pktidx,
				pkts[pktidx].data,pkt->dlen,pkt->flags);
			retval = -EPERM;
			goto out;
		}
		/* Truncate: Currently GRSPW2 IPs are only capable of performing DMA to
		 * lower 32-bit address space
		 */
		pkt->data = (u32)(pa & 0xFFFFFFFFUL);
		if ( (pkt->hlen > 0) && maplib_lookup_kern(
				priv->bufcfg.maplib_pool_idx,
				pkts[pktidx].hdr,
				NULL, /* Kernel address unused */
				&pa, /* hardware address used in DMA */
				pkt->hlen) ) {
			/* Address translation Failed, memory may go over a 
			 * block boundary, or comletely outside. Bug in USER
			 * packet handling, or an attempt to destory kernel
			 * memory.
			 */
			 printk(KERN_DEBUG "GRSPWU: send(hdr): %d,%p,%d,0x%x\n", pktidx,
			 	pkts[pktidx].hdr,pkt->hlen,pkt->flags);
			 retval = -EPERM;
			 goto out;
		}
		pkt->hdr = (u32)(pa & 0xFFFFFFFFUL);
		pkt->pkt_id = pkts[pktidx].pkt_id;

		pktidx++;
		pkt = pkt->next;
		left--;
	}
	count = cnt - left;

#ifdef GRSPWU_DEBUG
	tmp = grspw_list_cnt(&txpkts);
	printk(KERN_DEBUG "grspwu_send(): send %d packets, chain: %d\n", count, tmp);
	if ( tmp != count ) {
		printk(KERN_DEBUG "grspwu_send(): check2 failed\n");
		return -EIO;
	}
#endif

	/* Send the packets to respective DMA Channel */
	if ( count > 0 ) {
		grspw_dma_tx_send(priv->dma[dmachan], 0, &txpkts, count);
	}

	return count;

out:
	/* Return all Packet structures, send none of the read packets due
	 * to an error. " Rewind all work".
	 */
	while ( down_interruptible(&priv->sem_tx_pktstrs) )
		;

	grspw_list_prepend_list(&priv->tx_pktstrs, &txpkts);

	up(&priv->sem_tx_pktstrs);

	return retval;
}

static int grspwu_prepare(struct grspwu_priv *priv, int dmachan,
		struct grspw_wrxpkt __user *src, int cnt)
{
	struct grspw_list readypkts;
	struct grspw_pkt *pkt;
	void *ch = priv->dma[dmachan];
	int pktidx, count, size, left;
	struct grspw_wrxpkt pkts[64];
	int retval = 0;
	phys_addr_t pa;
#ifdef GRSPWU_DEBUG
	int tmp;
#endif

	/* Check that DMA Channel is open */
	if ( ch == NULL )
		return 0;

	/* One list per DMA Channel */
	grspw_list_clr(&readypkts);
	count = 0;

	while ( down_interruptible(&priv->sem_rx_pktstrs) )
		;

	/* Take out as many Packet Structs from FREE Packet List as needed */
	cnt = grspw_list_take_head_list(&priv->rx_pktstrs, &readypkts, cnt);

	up(&priv->sem_rx_pktstrs);
#ifdef GRSPWU_DEBUG
	tmp = grspw_list_cnt(&readypkts);
	printk(KERN_DEBUG "grspwu_prepare(): got %d packets, chain: %d\n", cnt, tmp);
	if ( tmp != cnt ) {
		printk(KERN_DEBUG "grspwu_prepare(): check1 failed\n");
		return -EIO;
	}
#endif
	if ( cnt < 1 )
		return 0;

	/* Convert "char-stream" to in the format of
	 * 'struct grspw_wrxpkt'
	 */
	pkt = readypkts.head;
	pktidx = 64;
	left = cnt;
	while ( (left > 0) && pkt ) {
		if ( pktidx >= 64 ) {
			if ( left < 64 ) {
				size = left * sizeof(struct grspw_wrxpkt);
			} else {
				size = sizeof(pkts);
			}
			if ( copy_from_user(&pkts[0], src, size) ) {
				retval = -EFAULT;
				goto out;
			}

			/* Write to user-space */
			src += 64; /* wrong to increase when left<32, but never used */
			pktidx = 0;
		}
		/* Convert Packet from grspw-lib packet to user space packet */
		pkt->flags = pkts[pktidx].flags & RXPKT_FLAG_INPUT_MASK;
		pkt->reserved = 0;
		pkt->hlen = 0;
		pkt->dlen = 0; /* Will be overwritten */
		if ( maplib_lookup_kern(
				priv->bufcfg.maplib_pool_idx,
				pkts[pktidx].data,
				NULL, /* Kernel Address unused */
				&pa, /* hardware address used in DMA */
				priv->dma_rxmaxlen[dmachan]) ) {
			/* Address translation Failed, memory may go over a 
			 * block boundary, or comletely outside. Bug in USER
			 * packet handling, or an attempt to destory kernel
			 * memory.
			 */
			 printk(KERN_DEBUG "GRSPWU: prepare(): %d,%p,%d,0x%x\n", pktidx,
			 	pkts[pktidx].data,priv->dma_rxmaxlen[dmachan],pkt->flags);
			 retval = -EPERM;
			 goto out;
		}
		pkt->data = (u32)(pa & 0xFFFFFFFFUL);
		/* Remember UserSpaceAdr in unused Header Address to avoid one
		 * extra address translation in recv() function.
		 */
		pkt->hdr = (u32)((unsigned long)pkts[pktidx].data & 0xFFFFFFFFUL);
		pkt->pkt_id = pkts[pktidx].pkt_id;

		pktidx++;
		pkt = pkt->next;
		left--;
	}
	count = cnt - left;

#ifdef GRSPWU_DEBUG
	tmp = grspw_list_cnt(&readypkts);
	printk(KERN_DEBUG "grspwu_prepare(): prep %d packets, chain: %d\n", count, tmp);
	if ( tmp != count ) {
		printk(KERN_DEBUG "grspwu_prepare(): check2 failed\n");
		return -EIO;
	}
#endif

	/* Send the packets to respective DMA Channel */
	if ( count > 0 ) {
		if ( grspw_dma_rx_prepare(priv->dma[dmachan], 0, &readypkts, count) )
			return -EBUSY;
	}

	return count;

out:
	/* Return all Packet structures, pepare none of the read packets due
	 * to an error. " Rewind all work".
	 */
	while ( down_interruptible(&priv->sem_rx_pktstrs) )
		;

	grspw_list_prepend_list(&priv->rx_pktstrs, &readypkts);

	up(&priv->sem_rx_pktstrs);

	return retval;
}

static ssize_t grspwu_read(struct file *filp, char __user *buff, size_t count, loff_t *offp)
{
	struct grspwu_priv *priv;
	int operation, chan_mask;
	int cnt, left, i;

	priv = filp->private_data;

	if ( (unsigned long)buff & 0x3 )
		return -EINVAL;

	if ( priv->state == STATE_STOPPED )
		return -EBUSY;

	/* Get Operation and channel selection from MSB of Length */
	operation = count & GRSPW_READ_RECLAIM;
	chan_mask = (count >> GRSPW_READ_CHANMSK_BIT) & GRSPW_READ_CHANMSK;
	/* Limit Reads to 65kB */
	count = count & 0xffff;

	/* Assume first DMA Channel if none given. The most usual configuration 
	 * of the GRSPW hardware is to have only one Channel.
	 */
	if ( chan_mask == 0 )
		chan_mask = 1;

#ifdef GRSPWU_DEBUG
	printk(KERN_DEBUG "grspwu_read(): %d, %d, %d, %p\n", operation, chan_mask, count, buff);
#endif
	if ( operation ) {
		/* RECLAIM SENT PACKET BUFFERS */
		struct grspw_rtxpkt __user *dst = (void *)buff;

		count = count / sizeof(struct grspw_rtxpkt);
		if ( count < 1 )
			return -EINVAL;
		left = count;
		for (i=0; i<priv->chan_cnt; i++) {
			if ( chan_mask & (1<<i) ) {
				cnt = grspwu_reclaim(priv, i, dst, left);
				/* If Failure then return the number of 
				 * packets processed, or if none processed
				 * return an error.
				 */
#ifdef GRSPWU_DEBUG
				printk(KERN_DEBUG "grspwu_read(): recl %d, %d, %d\n", cnt, left, count);
#endif
				if ( cnt < 0 ) {
					if ( left != count )
						break;
					return cnt;
				}
				if ( cnt == 0 )
					continue;
				left -= cnt;
				dst += cnt;
			}
		}
#ifdef GRSPWU_DEBUG
		printk(KERN_DEBUG "grspwu_read(): recl return %d %d %d\n", left, count, (count - left) * sizeof(struct grspw_rtxpkt));
#endif
		return (count - left) * sizeof(struct grspw_rtxpkt);
	} else {
		/* READ RECEIVED PACKETS */
		struct grspw_rrxpkt __user *dst = (void *)buff;
		count = count / sizeof(struct grspw_rrxpkt);
		if ( count < 1 )
			return -EINVAL;
		left = count;
		for (i=0; i<priv->chan_cnt; i++) {
			if ( chan_mask & (1<<i) ) {
				cnt = grspwu_recv(priv, i, dst, left);
				/* If Failure then return the number of 
				 * packets processed, or if none processed
				 * return an error.
				 */
				if ( cnt < 0 ) {
					if ( left != count )
						break;
					return cnt;
				}
				if ( cnt == 0 )
					continue;
				left -= cnt;
				dst += cnt;
			}
		}
		return (count - left) * sizeof(struct grspw_rrxpkt);
	}
}

static ssize_t grspwu_write(struct file *filp, const char __user *buff, size_t count, loff_t *offp)
{
	struct grspwu_priv *priv;
	int operation, chan, cnt;

	priv = filp->private_data;

	/* The DMA Channels must be started */
	if ( priv->state < STATE_STARTED )
		return -EBUSY;

	if ( (unsigned long)buff & 0x3 )
		return -EINVAL;

	/* Get Operation from MSB of Length */
	operation = count & GRSPW_WRITE_SEND;
	chan = (count >> GRSPW_WRITE_CHAN_BIT) & GRSPW_WRITE_CHAN;
	/* Limit Reads to 65kB */
	count = count & 0xffff;

	/* The DMA Channels must be started and opened */
	if ( priv->dma[chan] == NULL )
		return -EBUSY;

	if ( operation ) {
		/* SEND PACKETS */
		struct grspw_wtxpkt __user *src = (void *)buff;
		int cnt;

		count = count / sizeof(struct grspw_wtxpkt);
		if ( count < 1 )
			return -EINVAL;
		cnt = grspwu_send(priv, chan, src, count);
		if ( cnt < 0 )
			return cnt;
		return cnt * sizeof(struct grspw_wtxpkt);
	} else {
		/* PREPARE RECEIVE PACKET BUFFERS */
		struct grspw_wrxpkt __user *src = (void *)buff;
		count = count / sizeof(struct grspw_wrxpkt);
		if ( count < 1 )
			return -EINVAL;
		cnt = grspwu_prepare(priv, chan, src, count);
		if ( cnt < 0 )
			return cnt;
		return cnt * sizeof(struct grspw_wrxpkt);
	}
}

static void *grspwu_dev_add(int index)
{
	struct grspwu_priv *priv;
	int err;
	dev_t dev_id;

	/* Allocate private memory for all GRSPWU cores */
	priv = (struct grspwu_priv *)kmalloc(sizeof(*priv), GFP_KERNEL);
	memset(priv, 0, sizeof(*priv));
	privu_tab[index] = priv;

	/* Init all private structures */
	priv->index = index;
	strcpy(priv->name, "GRSPW[N]");
	priv->name[6] = '0' + priv->index;

	/* Init and Register CHAR driver */
	dev_id = MKDEV(dev_major, priv->index);
	cdev_init(&priv->cdev, &grspwu_fops);
	err = cdev_add(&priv->cdev, dev_id, 1);
	if ( err ) {
		printk(KERN_NOTICE "GRSPWU: Failed adding CHAR dev: %d\n", err);
		privu_tab[priv->index] = 0;
		kfree(priv);
		return NULL;
	}

	device_create(grspw_class, NULL, dev_id, NULL, "grspw%d", priv->index);

	return priv;
}

static void grspwu_dev_del(int index, void *data)
{
	struct grspwu_priv *priv = data;

	/* Take some action here to throw out user and remove private
	 * structures.
	 */
	cdev_del(&priv->cdev);
	kfree(priv);
	privu_tab[index] = 0;
}

static int __init grspwu_init (void)
{
	int result;
	dev_t dev_id;

	/* Get Major Number of CHAR driver */
	result = alloc_chrdev_region(&dev_id, 0, GRSPW_MAX, "grspw");
	if ( result < 0 ) {
		printk(KERN_WARNING "GRSPWU: Failed to register CHAR region\n");
		return 0;
	}

	dev_major = MAJOR(dev_id);

	memset(privu_tab, 0, sizeof(privu_tab));

#if LINUX_VERSION_CODE < KERNEL_VERSION(6,4,0)
	grspw_class = class_create(THIS_MODULE, "grspw");
#else
	grspw_class = class_create("grspw");
#endif
	grspw_initialize_user(grspwu_dev_add,grspwu_dev_del);

	return 0;
}

static void __exit grspwu_exit (void)
{
	int i;
	struct grspwu_priv *priv;

	/* Delete devices */
	for (i = 0; i<GRSPW_MAX; i++) {
		priv = privu_tab[i];
		if ( priv )
			grspwu_dev_del(i, priv);
	}
	grspw_initialize_user(NULL, NULL);

	for (i = 0; i < GRSPW_MAX; i++) {
		device_destroy(grspw_class, MKDEV(dev_major, i));
	}

	class_unregister(grspw_class);
	class_destroy(grspw_class);

	unregister_chrdev_region(MKDEV(dev_major, 0), GRSPW_MAX);
}

module_init(grspwu_init);
module_exit(grspwu_exit);

MODULE_AUTHOR("Frontgrade Gaisler AB.");
MODULE_DESCRIPTION("Frontgrade Gaisler GRSPW SpaceWire Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:grlib-grspwu");
