/*
 * Copyright (C) 2002-2005 Roman Zippel <zippel@linux-m68k.org>
 * Copyright (C) 2002-2005 Sam Ravnborg <sam@ravnborg.org>
 *
 * Released under the terms of the GNU GPL v2.0.
 */

#include <string.h>
#include "lkc.h"

/* file already present in list? If not add it */
struct file *file_lookup(const char *name)
{
	struct file *file;

	for (file = file_list; file; file = file->next) {
		if (!strcmp(name, file->name))
			return file;
	}

	file = malloc(sizeof(*file));
	memset(file, 0, sizeof(*file));
	file->name = strdup(name);
	file->next = file_list;
	file_list = file;
	return file;
}

static char* br2_symbol_printer(const char * const in)
{
	ssize_t i, j, len = strlen(in);
	char *ret;
	if (len < 1)
		return NULL;
	ret = malloc(len);
	if (!ret) {
		printf("Out of memory!");
		exit(1);
	}
	memset(ret, 0, len);
	i = j = 0;
	if (strncmp("BR2_", in, 4) == 0)
		i += 4;
	if (strncmp("PACKAGE_", in + i, 8) == 0)
		i += 8;
	else if (strncmp("TARGET_", in + i, 7) == 0)
		i += 7;
	while (i <= len)
		ret[j++] = tolower(in[i++]);
	return ret;
}

/* write dependencies of the infividual config-symbols */
static int write_make_deps(const char *name)
{
	char *str;
	char dir[PATH_MAX+1], buf[PATH_MAX+1], buf2[PATH_MAX+1];
	struct menu *menu;
	struct symbol *sym;
	struct property *prop, *p;
	unsigned done;
	const char * const name_tmp = "..make.deps.tmp";
	FILE *out;
	if (!name)
		name = ".auto.deps";

	strcpy(dir, conf_get_configname());
	str = strrchr(dir, '/');
	if (str)
		str[1] = 0;
	else
		dir[0] = 0;

	sprintf(buf, "%s%s", dir, name_tmp);
	out = fopen(buf, "w");
	if (!out)
		return 1;
	fprintf(out, "# ATTENTION! This does not handle 'depends', just 'select'! \n"
		"# See package/config/util.c write_make_deps()\n#\n");
	menu = &rootmenu;//rootmenu.list;
	while (menu) {
		sym = menu->sym;
		if (!sym) {
			if (!menu_is_visible(menu))
				goto next;
		} else if (!(sym->flags & SYMBOL_CHOICE)) {
			sym_calc_value(sym);
			if (sym->type == S_BOOLEAN
			    && sym_get_tristate_value(sym) != no) {
			    done = 0;
			    for_all_prompts(sym, prop) {
			        struct expr *e;
//printf("\nname=%s\n", sym->name);
			        for_all_properties(sym, p, P_SELECT) {
				    e = p->expr;
				    if (e && e->left.sym->name) {
				        if (!done) {
					    fprintf(out, "%s: $(BASE_TARGETS)", br2_symbol_printer(sym->name));
					    done = 1;
					}
//printf("SELECTS %s\n",e->left.sym->name);
					fprintf(out, " %s",br2_symbol_printer(e->left.sym->name));
				    }
				}
				if (done)
				    fprintf(out, "\n");
#if 0
				e = sym->rev_dep.expr;
				if (e && e->type == E_SYMBOL
					&& e->left.sym->name) {
				    fprintf(out, "%s: %s", br2_symbol_printer(e->left.sym->name),
						br2_symbol_printer(sym->name));
printf("%s is Selected BY: %s", sym->name, e->left.sym->name);
				}
#endif
			    }
			}
		}
next:
		if (menu->list) {
			menu = menu->list;
			continue;
		}
		if (menu->next)
			menu = menu->next;
		else while ((menu = menu->parent)) {
			if (menu->next) {
				menu = menu->next;
				break;
			}
		}
	}
	fclose(out);
	sprintf(buf2, "%s%s", dir, name);
	rename(buf, buf2);
	printf(_("#\n"
		 "# make dependencies written to %s\n"
		 "# ATTENTION devels!\n"
		 "# See top of this file before playing with this auto-preprequisites!\n"
		 "#\n"), name);
	return 0;
}

/* write a dependency file as used by kbuild to track dependencies */
int file_write_dep(const char *name)
{
	char *str;
	char buf[PATH_MAX+1], buf2[PATH_MAX+1], dir[PATH_MAX+1];
	struct symbol *sym, *env_sym;
	struct expr *e;
	struct file *file;
	FILE *out;

	if (!name)
		name = ".kconfig.d";

	strcpy(dir, conf_get_configname());
	str = strrchr(dir, '/');
	if (str)
		str[1] = 0;
	else
		dir[0] = 0;

	sprintf(buf, "%s..config.tmp", dir);
	out = fopen(buf, "w");
	if (!out)
		return 1;
	fprintf(out, "deps_config := \\\n");
	for (file = file_list; file; file = file->next) {
		if (file->next)
			fprintf(out, "\t%s \\\n", file->name);
		else
			fprintf(out, "\t%s\n", file->name);
	}
	fprintf(out, "\n%s: \\\n"
		     "\t$(deps_config)\n\n", conf_get_autoconfig_name());

	expr_list_for_each_sym(sym_env_list, e, sym) {
		struct property *prop;
		const char *value;

		prop = sym_get_env_prop(sym);
		env_sym = prop_get_symbol(prop);
		if (!env_sym)
			continue;
		value = getenv(env_sym->name);
		if (!value)
			value = "";
		fprintf(out, "ifneq \"$(%s)\" \"%s\"\n", env_sym->name, value);
		fprintf(out, "%s: FORCE\n", conf_get_autoconfig_name());
		fprintf(out, "endif\n");
	}

	fprintf(out, "\n$(deps_config): ;\n");
	fclose(out);
	sprintf(buf2, "%s%s", dir, name);
	rename(buf, buf2);
	return write_make_deps(NULL);
}


/* Allocate initial growable string */
struct gstr str_new(void)
{
	struct gstr gs;
	gs.s = malloc(sizeof(char) * 64);
	gs.len = 64;
	gs.max_width = 0;
	strcpy(gs.s, "\0");
	return gs;
}

/* Allocate and assign growable string */
struct gstr str_assign(const char *s)
{
	struct gstr gs;
	gs.s = strdup(s);
	gs.len = strlen(s) + 1;
	gs.max_width = 0;
	return gs;
}

/* Free storage for growable string */
void str_free(struct gstr *gs)
{
	if (gs->s)
		free(gs->s);
	gs->s = NULL;
	gs->len = 0;
}

/* Append to growable string */
void str_append(struct gstr *gs, const char *s)
{
	size_t l;
	if (s) {
		l = strlen(gs->s) + strlen(s) + 1;
		if (l > gs->len) {
			gs->s   = realloc(gs->s, l);
			gs->len = l;
		}
		strcat(gs->s, s);
	}
}

/* Append printf formatted string to growable string */
void str_printf(struct gstr *gs, const char *fmt, ...)
{
	va_list ap;
	char s[10000]; /* big enough... */
	va_start(ap, fmt);
	vsnprintf(s, sizeof(s), fmt, ap);
	str_append(gs, s);
	va_end(ap);
}

/* Retrieve value of growable string */
const char *str_get(struct gstr *gs)
{
	return gs->s;
}

#define APPEND_STR(r,a,l) if (l) {int rl=r?strlen(r):0; r = realloc(r,rl+l+1); memcpy(r+rl,a,l);r[rl+l] = 0;}

char *resolve_vars(const char *n, struct conf_level *l) {
	char *r = 0, *var = 0; const char *val = 0; int nl = strlen(n); int i,j;
	struct symbol *sym;
	struct conf_level *o = current_conf_level;
	current_conf_level = l;
	
	for (i = 0, j = 0; i < nl; i++) {
		if (n[i] == '$' && n[i+1] == '(') {
			APPEND_STR(r,&n[j],i-j);
			for (i+=2, j = i; i < nl; i++) {
				if (n[i] == ')') {
					break;
				}
			}
			if (i < nl && i > j) {
				var = malloc(i-j+1);
				memcpy(var, &n[j], i-j);
				var[i-j] = 0;
				if ((sym = sym_lookup(var, 0))) {
					if (sym_get_type(sym) == S_STRING) {
						sym_calc_value(sym);
						val  = sym_get_string_value(sym);
						if (val) {
							APPEND_STR(r,val,strlen(val));
						}
						
					}
				}
				free(var);
				j=i+1;
			}
		}
	}
	APPEND_STR(r,&n[j],i-j);
	current_conf_level = o;
	return r;
error_out:
	if (r)
		free(r);
	if (var)
		free(var);
	return 0;
}

/* if a subsource tag defined a .config with variable substitutions in it register
 * a reload function to trigger config reload.
 */
void register_load_updates(const char *n) {
	int i, j; int nl = strlen(n);
	char *var; struct symbol *sym;
	
	for (i = 0, j = 0; i < nl; i++) {
		if (n[i] == '$' && n[i+1] == '(') {
			for (i+=2, j = i; i < nl; i++) {
				if (n[i] == ')') {
					break;
				}
			}
			if (i < nl && i > j) {
				var = malloc(i-j+1);
				memcpy(var, &n[j], i-j);
				var[i-j] = 0;
				if ((sym = sym_lookup_level(var, 0, current_conf_level->parent))) {
					struct sym_update *u = malloc(sizeof(struct sym_update));
					memset(u, 0, sizeof(struct sym_update));
					u->n = sym_updates;
					u->sym = sym;
					u->func = (void (*)(struct symbol *sym, void *a))conf_reload;
					u->arg = current_conf_level;
					sym_updates = u;
				}
				free(var);
				j=i+1;
			}
		}
	}
}

void execute_sym_question(char *cmd, char *b, int restart, int isterm) 
{
	if (isterm) {
		sprintf(b,"\nRun command \"%s\"?\nThe main window will be inresponsive during execution?\n"
			"(After the command finishes executing it waits for pressing <enter>).\n%s",cmd,
			restart ? "Note:\nThe command requires the configurator to be restarted after the command executed,\nthis will happen automatically\n" : "");
	} else {
		sprintf(b,"\nRun command \"%s\" in new xterm (yes) or the parent shell (no),\nthe main window will be inresponsive during execution?\n"
			"(After the command finishes executing it waits for pressing <enter>).\n%s",cmd,
			restart ? "Note:\nThe command requires the configurator to be restarted after the command executed,\nthis will happen automatically\n" : "");
	}
}

int _restart_val = 0;
int (*f_do_run_cmds)(char *cmd, int restart) = 0;
void (*f_restart_app)(void) = 0;
void execute_sym(struct symbol *sym, int isterm) 
{
	int re = 0,rer = 0;

	if (sym
	    && sym_get_string_value(sym)
	    && sym_get_type(sym) == S_EXECUTE) {
		const char *cmd = sym_get_string_value(sym);
		char *rcmd = resolve_vars(cmd, sym->level);
		char *view = 0;
		if (rcmd) {
			re = strstr(rcmd, "RESTART=1") ? 1 : 0;
			if (!f_do_run_cmds || (rer = f_do_run_cmds(rcmd,re))) {

				if (rer == 1) {
					char *tcmd;
					tcmd = malloc(strlen(rcmd) + 128);
					tcmd[0] = 0;
					strcat(tcmd, "xterm -e \"");
					strcat(tcmd, rcmd);
					strcat(tcmd, ";echo Press return to continue; read;\"");
					free(rcmd);
					rcmd = tcmd;
				}

				conf_write(NULL);
				system(rcmd);
				free(rcmd);
				_restart_val = re ? 129 : 0;
				if (re && f_restart_app)
					f_restart_app();
			}
		}
	}
}

int get_restart() {
	return _restart_val;
}
