/*
 * MAPLIB user space interface
 *
 * Copyright (c) 2016-2023 Frontgrade Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <unistd.h>

#include <linux/grlib/maplib.h>
#include "maplib_user.h"

/******** MAP LIBRARY ********/
struct maplib_mmap_info mmapinfo;
struct map maps[512];
int map_fd = -1;

/* Memory MAP memory to use for example */
int mmaplib_init(int idx, int blk_size, int blk_cnt)
{
	int i;
	void *adr;
	unsigned int start, end, len;
	struct maplib_mmap_info *mapi = &mmapinfo;
	char devname[32];
	struct maplib_setup setup;

	sprintf(devname, "/dev/maplib%d", idx);
	map_fd = open(devname, O_RDWR);
	if ( map_fd < 0 ) {
		perror("Failed to open MMAPLib");
		return -1;
	}

	/* Setup MAP */
	setup.blk_size = blk_size;
	setup.blk_cnt = blk_cnt;
	if ( ioctl(map_fd, MAPLIB_IOCTL_SETUP, &setup) ) {
		printf("Failed to Setup MMAPLib: %d\n", errno);
		return -2;
	}

	/* Get MMAP information calculated by driver */
	if ( ioctl(map_fd, MAPLIB_IOCTL_MMAPINFO, mapi) ) {
		printf("Failed to get MMAPINFO, errno: %d\n", errno);
		return -3;
	}

	printf("MMAP INFO: GRSPW Packet Buffers:\n");
	printf("    offset: 0x%x\n", mapi->buf_offset);
	printf("    length: %d (0x%x)\n", mapi->buf_length, mapi->buf_length);
	printf("    MMAP Block size: 0x%x\n", mapi->buf_blk_size);

	memset(maps, 0, sizeof(maps));

	/* Map all SpaceWire Packet Buffers */
	start = mapi->buf_offset;
	end = mapi->buf_offset + mapi->buf_length;

	/* Memory MAP driver's Buffers READ-and-WRITE  */
	adr = mmap(NULL, mapi->buf_length, PROT_READ|PROT_WRITE, MAP_SHARED,
			map_fd, start);
	if ( adr == MAP_FAILED ) {
		printf("MMAP Bufs Failed: %p, errno %d, %x\n", adr, errno, mapi->buf_length);
		return -4;
	}
	i=0;
	while ( start < end ) {
		/* Calc length of MMAP Block */
		if ( (end - start) >= mapi->buf_blk_size )
			len = mapi->buf_blk_size;
		else
			len = end - start;

		if ((sizeof(maps) / sizeof(maps[0])) <= i) {
			printf("Failed to map: map[%d] referenced\n", i);
			return -5;
		}
		maps[i].start = adr;
		maps[i].length = len;
		maps[i].ofs = start;
		/*printf("MAPPED maps[%d] OFS=%p, BASE=%x, LEN=%p\n", i, maps[i].ofs, maps[i].start, maps[i].length);*/
		i++;

		/* Next Block */
		start += mapi->buf_blk_size;
		adr += len;
	}

	printf("M-Mapping successful\n");

	return 0;
}

void mmaplib_cleanup(void)
{
	/* Close MAP Lib, this will automatically unmap all memory for us. */
	close(map_fd);

	maps[0].length = 0;
	maps[0].start = 0;
	maps[0].ofs = 0;
}
