
#include <prom_watchdog.h>

#ifdef CONFIG_WATCHDOG

#include <gptimer.h>
#include <prom.h>

#define GPTIMER_CONF_TIMERS_MASK 0x7
#define GPTIMER_CTRL_RD 4
#define TIMER_IDX_INVALID -1

/* Allowed number of pings when building the device tree */
#define MAX_PROM_NO_PING_CNT 500

/* Allowed number of pings on UART transmission */
#define MAX_UART_PING_CNT 2000

/* Allowed number of pings on boot */
#define MAX_BOOT_PING_CNT 1

struct ping_cnt {
	unsigned int curr;
	unsigned int max;
};

struct watchdog_priv {
	int timer_idx;
	struct ping_cnt ping_cnt[PING_ID_MAX];
};

static struct watchdog_priv watchdog_priv;

void watchdog_init(void)
{
	watchdog_priv.ping_cnt[PING_ID_PROM_NO].max = MAX_PROM_NO_PING_CNT;
	watchdog_priv.ping_cnt[PING_ID_UART].max = MAX_UART_PING_CNT;
	watchdog_priv.ping_cnt[PING_ID_BOOT].max = MAX_BOOT_PING_CNT;

	/* Fetch the nbr of timers from GPTIMER0 config register */
	watchdog_priv.timer_idx =
		BYPASS_LOAD_PA(&gptimer->config) & GPTIMER_CONF_TIMERS_MASK;

	if (watchdog_priv.timer_idx <=
		(sizeof(gptimer->timer) / sizeof(gptimer->timer[0]))) {
		/* The watchdog timer is connected to the last found timer */
		watchdog_priv.timer_idx = watchdog_priv.timer_idx - 1;
	} else {
		watchdog_priv.timer_idx = TIMER_IDX_INVALID;
	}
}

void watchdog_ping(enum ping_id id)
{
	struct ping_cnt *ping_cnt;
	struct gptimer_timer *timer;

	if (id < 0 || id >= PING_ID_MAX)
		return;

	if (watchdog_priv.timer_idx == TIMER_IDX_INVALID)
		return;

	ping_cnt = &watchdog_priv.ping_cnt[id];

	if (ping_cnt->curr < ping_cnt->max) {
		/* Ping the watchdog by reloading the ctrl register of the last timer */
		timer = &gptimer->timer[watchdog_priv.timer_idx];
		BYPASS_STORE_PA(&timer->ctrl,
				BYPASS_LOAD_PA(&timer->ctrl) | GPTIMER_CTRL_RD);

		ping_cnt->curr++;
	}
}

void watchdog_set_timeout(unsigned int timeout_ms)
{
	struct gptimer_timer *timer;

	if (watchdog_priv.timer_idx == TIMER_IDX_INVALID)
		return;

	timer = &gptimer->timer[watchdog_priv.timer_idx];

	/* Set the reload value */
	BYPASS_STORE_PA(&timer->rld, timeout_ms * 1000);
	/* Reload the timer */
	BYPASS_STORE_PA(&timer->ctrl,
		BYPASS_LOAD_PA(&timer->ctrl) | GPTIMER_CTRL_RD);
}

#endif
