/*
 * GRSPW DMA char driver,
 * implemented by using the GRSPW kernel library
 *
 * Copyright (c) 2016-2022 Cobham Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#ifndef __GRSPWU_CHAR_H__
#define __GRSPWU_CHAR_H__

#ifndef GRSPW_PKT_FLAGS
#define GRSPW_PKT_FLAGS
/*** TX Packet flags ***/

/* Enable IRQ generation */
#define TXPKT_FLAG_IE 0x0040

/* Enable Header CRC generation (if CRC is available in HW)
 * Header CRC will be appended (one byte at end of header)
 */
#define TXPKT_FLAG_HCRC 0x0100

/* Enable Data CRC generation (if CRC is available in HW)
 * Data CRC will be appended (one byte at end of packet)
 */
#define TXPKT_FLAG_DCRC 0x0200

/* Control how many bytes the beginning of the Header 
 * the CRC should not be calculated for */
#define TXPKT_FLAG_NOCRC_MASK 0x0000000f
#define TXPKT_FLAG_NOCRC_LEN0 0x00000000
#define TXPKT_FLAG_NOCRC_LEN1 0x00000001
#define TXPKT_FLAG_NOCRC_LEN2 0x00000002
#define TXPKT_FLAG_NOCRC_LEN3 0x00000003
#define TXPKT_FLAG_NOCRC_LEN4 0x00000004
#define TXPKT_FLAG_NOCRC_LEN5 0x00000005
#define TXPKT_FLAG_NOCRC_LEN6 0x00000006
#define TXPKT_FLAG_NOCRC_LEN7 0x00000007
#define TXPKT_FLAG_NOCRC_LEN8 0x00000008
#define TXPKT_FLAG_NOCRC_LEN9 0x00000009
#define TXPKT_FLAG_NOCRC_LENa 0x0000000a
#define TXPKT_FLAG_NOCRC_LENb 0x0000000b
#define TXPKT_FLAG_NOCRC_LENc 0x0000000c
#define TXPKT_FLAG_NOCRC_LENd 0x0000000d
#define TXPKT_FLAG_NOCRC_LENe 0x0000000e
#define TXPKT_FLAG_NOCRC_LENf 0x0000000f

/* Marks if packet was transmitted or not */
#define TXPKT_FLAG_TX 0x8000

#define TXPKT_FLAG_INPUT_MASK (TXPKT_FLAG_NOCRC_MASK | TXPKT_FLAG_IE | \
				TXPKT_FLAG_HCRC | TXPKT_FLAG_DCRC)

/* Link Error */
#define TXPKT_FLAG_LINKERR 0x4000

#define TXPKT_FLAG_OUTPUT_MASK (TXPKT_FLAG_LINKERR)

/*** RX Packet Flags ***/

/* Enable IRQ generation */
#define RXPKT_FLAG_IE 0x0010

#define RXPKT_FLAG_INPUT_MASK (RXPKT_FLAG_IE)

/* Packet was truncated */
#define RXPKT_FLAG_TRUNK 0x0800
/* Data CRC error (only valid if RMAP CRC is enabled) */
#define RXPKT_FLAG_DCRC 0x0400
/* Header CRC error (only valid if RMAP CRC is enabled) */
#define RXPKT_FLAG_HCRC 0x0200
/* Error in End-of-Packet */
#define RXPKT_FLAG_EEOP 0x0100
/* Marks if packet was recevied or not */
#define RXPKT_FLAG_RX 0x8000

#define RXPKT_FLAG_OUTPUT_MASK (RXPKT_FLAG_TRUNK | RXPKT_FLAG_DCRC | \
				RXPKT_FLAG_HCRC | RXPKT_FLAG_EEOP)
#endif

/* GRSPW Write TX-Packet Entry (SEND PACKET) */
struct grspw_wtxpkt {
	int pkt_id;		/* Custom Packet ID */
	unsigned short flags;	/* See TXPKT_FLAG* above */
	unsigned char resv;	/* Reserved */
	unsigned char hlen;	/* Header Length. Set to zero if none. */
	unsigned int dlen;	/* Data Length. Set to zero if none. */
	void *hdr;		/* Header Pointer (Address from MMAP Lib) */
	void *data;		/* Data Pointer (Address from MMAP Lib) */
} __attribute__((packed));

/* GRSPW Read TX-Packet Entry (RECLAIM TX BUFFER) */
struct grspw_rtxpkt {
	int pkt_id;		/* Custom Packet ID */
	unsigned short flags;	/* See TXPKT_FLAG* above */
	unsigned char dma_chan;	/* DMA Channel 0..3 */
	unsigned char resv1;	/* Reserved, must be zero */
} __attribute__((packed));

/* GRSPW Write RX-Packet Entry (PREPARE RX BUFFER) */
struct grspw_wrxpkt {
	int pkt_id;		/* Custom Packet ID */
	unsigned short flags;	/* See RXPKT_FLAG* above */
	unsigned short resv1;	/* Reserved, must be zero */
	void *data;		/* Data Pointer (Address from MMAP Lib). The
				 * buffer must have room for max-packet */
} __attribute__((packed));

/* GRSPW Read RX-Packet Entry (RECEIVE) */
struct grspw_rrxpkt {
	int pkt_id;		/* Custom Packet ID */
	unsigned short flags;	/* See RXPKT_FLAG* above */
	unsigned char dma_chan;	/* DMA Channel 0..3 */
	unsigned char resv1;	/* Reserved, must be zero */
	int dlen;		/* Data Length */
	void *data;		/* Data Pointer (Address from MMAP Lib) */
} __attribute__((packed));

/* The Read() and Write() calls are used to transfer buffer paramters and
 * pointers between the driver and user space application. Blow is a summary
 * of the operations possible:
 *
 * read()
 *  - RX: RECEIVE RX Packets                         (struct grspw_rrxpkt)
 *  - TX: RECLAIM TX Packet Buffers                  (struct grspw_rtxpkt)
 * write()
 *  - RX: PREPARE driver with RX Packet buffers      (struct grspw_wrxpkt)
 *  - TX: SEND TX Packets                            (struct grspw_wtxpkt)
 *
 * To select between the different actions that the read()/write() operation
 * will take, the MSB of the buffer length is used an input paramters.
 *   - Bit 28: RX (0) or TX (1) operation
 *   - read(): Bit 26..24: DMA Channel MASK
 *   - write(): Bit 25..24: DMA Channel select
 *
 * The data format of the read()/write() operations are described with the
 * grspw_XYxpkt structures, see above.
 *
 * The input/output is limited to 65kbytes for read()/write() operation to
 * the driver.
 *
 * The Data and Header pointers is be aquired from the Memory MAP Library
 * that comes with this driver. The MMAP Lib ensures that memory is linear
 * and capable of DMA.
 */
#define GRSPW_READ_RECLAIM_BIT 20
#define GRSPW_READ_RECLAIM (1<<GRSPW_READ_RECLAIM_BIT)
#define GRSPW_READ_CHANMSK_BIT 16
#define GRSPW_READ_CHANMSK (0xf<<GRSPW_READ_CHANMSK_BIT)
#define GRSPW_WRITE_SEND_BIT 20
#define GRSPW_WRITE_SEND (1<<GRSPW_WRITE_SEND_BIT)
#define GRSPW_WRITE_CHAN_BIT 16
#define GRSPW_WRITE_CHAN (0x3<<GRSPW_WRITE_CHAN_BIT)

/* Get Hardware support available */
#define GRSPW_IOCTL_HWSUP	_IOW('a', 0x01, struct grspw_hw_sup)

/* Number of Packet Buffers that the driver can hold simultaneously. THe Packet
 * buffers are shared between all DMA Channels.
 * Defaults to 1000 packets for RX and 1000 for TX per DMA channel.
 */
struct grspw_bufcfg {
	int rx_pkts;
	int tx_pkts;
	int maplib_pool_idx; /* MAPLIB Pool index used by SPW device */
};
#define GRSPW_IOCTL_BUFCFG	_IOR('a', 0x02, struct grspw_bufcfg)

struct grspw_config {
	struct grspw_addr_config adrcfg;
	unsigned int rmap_cfg;
	unsigned char rmap_dstkey;
	unsigned int tc_cfg;
	/* Mask of which DMA Channels to enable. Most often only
	 * one DMA Channel is available, then set this to 1.
	 *
	 * By enabling a DMA Channel, the DMA Channels will be
	 * able to receive SpaceWire packets after SpaceWire buffers 
	 * has been prepared for it, and transmitt SpaceWire packets
	 * when the user request transmission of a SpaceWire packet.
	 */
	unsigned int enable_chan_mask;
	/* Per Channel Configuration */
	struct grspw_dma_config chan[4];
};
#define GRSPW_IOCTL_CONFIG_SET	_IOR('a', 0x03, struct grspw_config)
#define GRSPW_IOCTL_CONFIG_READ	_IOW('a', 0x04, struct grspw_config)

/* Statisics and diagnostics gathered by driver */
struct grspw_stats {
	/* Statistics for the Core */
	struct grspw_core_stats stats;

	/* Per DMA channel statistics */
	struct grspw_dma_stats chan[4];
};
#define GRSPW_IOCTL_STATS_READ	_IOW('a', 0x10, struct grspw_stats)
/* Clear/Reset statistics */
#define GRSPW_IOCTL_STATS_CLR	_IO('a', 0x11)

/* Set link config, argument (int) is a bit mask of options, see LINKOPTS_* options */
struct grspw_link_ctrl {
	int ctrl;			/* Configuration */
	unsigned char clkdiv_start;	/* Clock Division during startup */
	unsigned char clkdiv_run;	/* Clock Division in run-state */
	int stscfg;			/* Link status control LINKSTS_* */
};
#define GRSPW_IOCTL_LINKCTRL	_IOR('a', 0x20, struct grspw_link_ctrl)

/* Set port config, argument (int) is 0=Port0, 1=Port1, else let hardware select
 * between Port0 and Port1 */
#define GRSPW_IOCTL_PORTCTRL	_IO('a', 0x21)

/* Current state of configuration and link */
struct grspw_link_state {
	int link_ctrl;			/* Link Configuration */
	unsigned char clkdiv_start;	/* Clock Division during startup (not 
					 * in state run) */
	unsigned char clkdiv_run;	/* Clock Division in run-state */
	spw_link_state_t link_state;	/* State (Error-Reset, Error-wait...) */
	int port_cfg;			/* Port Configuration */
	int port_active;		/* Currently active port */
};
#define GRSPW_IOCTL_LINKSTATE	_IOW('a', 0x22, struct grspw_link_state)

/* Generate Write TCTRL|TIMECNT and/or generate a Tick-In (sends a timecode)
 *
 * Write TCTRL|TIMECNT if bit 8 is 1:
 *   TCTRL   = bits 7 and 6
 *   TIMECNT = bits 5 to 0
 * Generate Tick-In if bit 9 is 1.
 *
 * The Tick-In is generated after writing the TCTRL|TIMECNT values to register.
 *
 * Argument is a int.
 */
#define GRSPW_TC_SEND_OPTION_TCTRL	0x0ff
#define GRSPW_TC_SEND_OPTION_SET_TCTRL	0x100
#define GRSPW_TC_SEND_OPTION_TCTX	0x200
#define GRSPW_IOCTL_TC_SEND	_IO('a', 0x23)

/* Read current value of TCTRL|TIMECNT:
 *
 *    [TIMECODE WAS RECEIVED SINCE LAST READ OUT = bit 8] NOT IMPLEMENTED
 *   TCTRL   = bits 7 and 6
 *   TIMECNT = bits 5 to 0
 */
#define GRSPW_IOCTL_TC_READ	_IOW('a', 0x24, int)

/* Number of Packets in the Packet Queues of the driver */
struct grspw_qpktcnt_chan {
	/* RX */
	int rx_ready;
	int rx_sched;
	int rx_recv;
	int rx_hw;
	/* TX */
	int tx_send;
	int tx_sched;
	int tx_sent;
	int tx_hw;
};
struct grspw_qpktcnt {
	/* DMA Channels that does not exists are not written, thus data is
	 * undefined.
	 */
	struct grspw_qpktcnt_chan chan[4];
};
/* Read current number of packets in all TX/RX queues of all enabled DMA
 * channels.
 */
#define GRSPW_IOCTL_QPKTCNT	_IOW('a', 0x25, struct grspw_qpktcnt)

/*
 * Read current value of GRSPW status register.
 */
#define GRSPW_IOCTL_STATUS_READ	_IOW('a', 0x26, unsigned int)

/*
 * Clear one/more bit(s) in the GRSPW status register. It is determined by
 * the input bit-mask which bits are cleared.
 */
#define GRSPW_IOCTL_STATUS_CLR	_IOR('a', 0x27, unsigned int)

/* START DMA operation, after this the followin functions will be available for
 * every enabled DMA Channel:
 *  - RX: PREPARE RX BUFFER
 *  - RX: RECEIVE
 *  - TX: SEND
 *  - TX: RECLAIM BUFFER
 *
 * After the link has started, the BUFCFG and CONFIG_SET ioctl commands are
 * disabled, calling them will result in an EBUSY error.
 */
#define GRSPW_IOCTL_START	_IO('a', 0x30)

/* STOP DMA operation, after this only the folling operation will be available
 * for every enabled DMA Channel:
 *  - RX: RECEIVE (will get PREPARED buffers and RECEIVED Packets)
 *  - TX: RECLAIM BUFFER (will get SENT packets and unsent Packets)
 *
 * Future calls to SEND and PREPARE will result in an error.
 */
#define GRSPW_IOCTL_STOP	_IO('a', 0x31)

/* Specifices a RX wait conditions by RX packet queue size */
struct grspw_rx_wait_chan {
	int chan;		/* which DMA channel to wait for */
	int reserved;		/* reserved, set to zero */
	long timeout_ms;	/* number of milliseconds timeout (converted
				 * to jiffies). 0 means no wait forever.
				 */
	int recv_cnt;
	int op;
	int ready_cnt;
};
/* Wait for TX queue packet counters to be fulfilled. This call may block until
 * workqueue unblocks the caller. See grspw_dma_tx_wait() documentation for
 * details.
 */
#define GRSPW_IOCTL_RX_WAIT	_IOR('a', 0x40, struct grspw_rx_wait_chan)

/* Specifices a TX wait conditions by TX packet queue size */
struct grspw_tx_wait_chan {
	int chan;		/* which DMA channel to wait for */
	int reserved;		/* reserved, set to zero */
	long timeout_ms;	/* number of milliseconds timeout (converted
				 * to jiffies). 0 means no wait forever.
				 */
	int send_cnt;
	int op;
	int sent_cnt;
};
/* Wait for TX queue packet counters to be fulfilled. This call may block until
 * workqueue unblocks the caller. See grspw_dma_tx_wait() documentation for
 * details.
 */
#define GRSPW_IOCTL_TX_WAIT	_IOR('a', 0x41, struct grspw_rx_wait_chan)

#endif
