/*
 * GRSPW Driver Access Help Library
 *
 * Copyright (c) 2016-2022 Cobham Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include <linux/grlib/grspw.h>
#include <linux/grlib/grspw_user.h>
#include "grspwlib.h"

#undef RX_RECV_ADD_END
#undef TX_RECLAIM_ADD_END

/* In newer versions of GLIBC read() and write() are not accessed directly,
 * instead some buffer-overlow checking are performed first. The checks does
 * not work with the GRSPW driver since the driver uses some bits in the
 * count argument to select which DMA channel is accessed, and which operation
 * (read incomming packet, read used transmitted packet buffer).
 */
ssize_t __read(int fd, void *buf, size_t count);
ssize_t __write(int fd, const void *buf, size_t count);
#define READ(fd, buf, count) __read(fd, buf, count)
#define WRITE(fd, buf, count) __write(fd, buf, count)

/* Used For debug */
#define PRINT(args...)
/*#define PRINT(args...) printf(args)*/

/* Used for printing errors */
#define PRINT_ERR(args...) printf(args)

int grspw_open(struct grspw_dev *dev, int idx)
{
	char devname[32];
	int fd;

	memset(dev, 0, sizeof(struct grspw_dev));
	dev->index = idx;

	sprintf(devname, "/dev/grspw%d", idx);
	fd = open(devname, O_RDWR);
	if ( fd < 0 ) {
		PRINT_ERR("Failed to open GRSPW device %d: %d\n", idx, errno);
		return -1;
	}
	dev->fd = fd;
	PRINT("Successfully opened GRSPW[%d]: %d\n", dev->index, dev->fd);
	fflush(NULL);

	/* Get Hardware support */
	if ( grspw_hwsup_get(dev, &dev->hwsup) ) {
		close(dev->fd);
		dev->fd = -1;
		return -1;
	}

	return 0;
}

void grspw_close(struct grspw_dev *dev)
{
	if ( dev->fd != -1 ) {
		grspw_stop(dev);
		close(dev->fd);
		dev->fd = -1;
	}
}

void grspw_assign_id2pkt(struct grspw_dev *dev, struct spwlib_pkt **tab)
{
	dev->pktid2pkt = tab;
}

int grspw_hwsup_get(struct grspw_dev *dev, struct grspw_hw_sup *hw)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_HWSUP, hw);
	if ( result < 0 ) {
		PRINT_ERR("Failed to get hardware supported for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_bufcfg_set(struct grspw_dev *dev, struct grspw_bufcfg *bufcfg)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_BUFCFG, bufcfg);
	if ( result < 0 ) {
		PRINT_ERR("Failed to configure %d dev buffers, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_cfg_get(struct grspw_dev *dev, struct grspw_config *cfg)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_CONFIG_READ, cfg);
	if ( result < 0 ) {
		PRINT_ERR("Failed to read configuration %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_cfg_set(struct grspw_dev *dev, struct grspw_config *cfg)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_CONFIG_SET, cfg);
	if ( result < 0 ) {
		PRINT_ERR("Failed to set configuration %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

void grspw_cfg_print(struct grspw_hw_sup *hw, struct grspw_config *cfg)
{
	int i;

	printf("HARDWARE SUPPORT:\n");
	printf(" RMAP:                    %s\n", hw->rmap ? "YES" : "NO");
	printf(" RMAP CRC:                %s\n", hw->rmap_crc ? "YES" : "NO");
	printf(" UNALIGNED RX ADR:        %s\n", hw->rx_unalign ? "YES" : "NO");
	printf(" NUMBER OF PORTS:         %d\n", hw->nports);
	printf(" DMA CHANNEL COUNT:       %d\n", hw->ndma_chans);
	printf(" HARDWARE VERSION:        0x%08x\n", hw->hw_version);

	printf("SOFWTARE CONFIGURATION:\n");
	printf(" Promiscuous Mode:        %s\n", cfg->adrcfg.promiscuous ? "YES" : "NO");
	printf(" Default Node Address:    0x%02x\n", cfg->adrcfg.def_addr);
	printf(" Default Node Mask:       0x%02x\n", cfg->adrcfg.def_mask);
	for (i=0; i<hw->ndma_chans; i++) {
		printf("  DMA[%d] NODE ADR EN:     %s\n", i, cfg->adrcfg.dma_nacfg[i].node_en ? "YES" : "NO");
		printf("  DMA[%d] NODE ADDR:       0x%02x\n", i, cfg->adrcfg.dma_nacfg[i].node_addr);
		printf("  DMA[%d] NODE MASK:       0x%02x\n", i, cfg->adrcfg.dma_nacfg[i].node_mask);
	}
	printf(" RMAP ENABLED:            %s\n", cfg->rmap_cfg & RMAPOPTS_EN_RMAP ? "YES" : "NO");
	printf(" RMAP BUFFER ENABLED:     %s\n", cfg->rmap_cfg & RMAPOPTS_EN_BUF ? "YES" : "NO");
	printf(" RMAP DESTKEY:            0x%02x\n", cfg->rmap_dstkey);
	printf(" TIMECODE RX ENABLED:     %s\n", cfg->tc_cfg & TCOPTS_EN_RX ? "YES" : "NO");
	printf(" TIMECODE RX IRQ:         %s\n", cfg->tc_cfg & TCOPTS_EN_RXIRQ ? "YES" : "NO");
	printf(" TIMECODE TX ENABLED:     %s\n", cfg->tc_cfg & TCOPTS_EN_TX ? "YES" : "NO");
	printf(" DMA CHAN MASK ENABLE:    0x%02x\n", cfg->enable_chan_mask);	
	for (i=0; i<hw->ndma_chans; i++) {
		if ( (cfg->enable_chan_mask & (1<<i)) == 0 )
			continue;
		printf("  DMA[%d] FLAGS:           0x%04x\n", i, cfg->chan[i].flags);
		printf("  DMA[%d] RX MAX PKTLEN:   %d Bytes\n", i, cfg->chan[i].rxmaxlen);
		if ( cfg->chan[i].rx_irq_en_cnt == 0 )
			printf("  DMA[%d] RX IRQ EN CNT:   IRQ DISABLED\n", i);
		else
			printf("  DMA[%d] RX IRQ EN CNT:   1 IRQ per %d pkts\n", i, cfg->chan[i].rx_irq_en_cnt);
		if ( cfg->chan[i].tx_irq_en_cnt == 0 )
			printf("  DMA[%d] TX IRQ EN CNT:   IRQ DISABLED\n", i);
		else
			printf("  DMA[%d] TX IRQ EN CNT:   1 IRQ per %d pkts\n", i, cfg->chan[i].tx_irq_en_cnt);
	}
	fflush(NULL);
}

int grspw_stats_get(struct grspw_dev *dev, struct grspw_stats *stats)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_STATS_READ, stats);
	if ( result < 0 ) {
		PRINT_ERR("Failed to read statistics on GRSPW%d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_stats_clear(struct grspw_dev *dev)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_STATS_CLR, 0);
	if ( result < 0 ) {
		PRINT_ERR("Failed to clear statistics on GRSPW%d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

void grspw_stats_print(struct grspw_dev *dev, struct grspw_stats *stats)
{
	int i;
	struct grspw_dma_stats *chan;

	printf("SpW%d STATISTICS:\n", dev->index);
	printf(" irq_cnt:                  %d\n", stats->stats.irq_cnt);
	printf(" err_credit:               %d\n", stats->stats.err_credit);
	printf(" err_eeop:                 %d\n", stats->stats.err_eeop);
	printf(" err_addr:                 %d\n", stats->stats.err_addr);
	printf(" err_parity:               %d\n", stats->stats.err_parity);
	printf(" err_escape:               %d\n", stats->stats.err_escape);

	for (i=0; i<dev->hwsup.ndma_chans; i++) {
		chan = &stats->chan[i];
		printf(" DMA[%d] irq_cnt:           %d\n", i, chan->irq_cnt);
		printf(" DMA[%d] tx_pkts:           %d\n", i, chan->tx_pkts);
		printf(" DMA[%d] tx_err_link:       %d\n", i, chan->tx_err_link);
		printf(" DMA[%d] rx_pkts:           %d\n", i, chan->rx_pkts);
		printf(" DMA[%d] rx_err_trunk:      %d\n", i, chan->rx_err_trunk);
		printf(" DMA[%d] rx_err_endpkt:     %d\n", i, chan->rx_err_endpkt);
		printf(" DMA[%d] send_cnt_min:      %d\n", i, chan->send_cnt_min);
		printf(" DMA[%d] send_cnt_max:      %d\n", i, chan->send_cnt_max);
		printf(" DMA[%d] tx_sched_cnt_min:  %d\n", i, chan->tx_sched_cnt_min);
		printf(" DMA[%d] tx_sched_cnt_max:  %d\n", i, chan->tx_sched_cnt_max);
		printf(" DMA[%d] sent_cnt_max:      %d\n", i, chan->sent_cnt_max);
		printf(" DMA[%d] ready_cnt_min:     %d\n", i, chan->ready_cnt_min);
		printf(" DMA[%d] ready_cnt_max:     %d\n", i, chan->ready_cnt_max);
		printf(" DMA[%d] rx_sched_cnt_min:  %d\n", i, chan->rx_sched_cnt_min);
		printf(" DMA[%d] rx_sched_cnt_max:  %d\n", i, chan->rx_sched_cnt_max);
		printf(" DMA[%d] recv_cnt_max:      %d\n", i, chan->recv_cnt_max);
	}
}

int grspw_qpktcnt_get(struct grspw_dev *dev, struct grspw_qpktcnt *qpktcnt)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_QPKTCNT, qpktcnt);
	if ( result < 0 ) {
		PRINT_ERR("Failed to GET Q Packet Count for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_linkstate_get(struct grspw_dev *dev, struct grspw_link_state *ls)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_LINKSTATE, ls);
	if ( result < 0 ) {
		PRINT_ERR("Failed to GET Link-State for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

void grspw_linkstate_print(struct grspw_link_state *ls)
{
	printf("link_ctrl:      0x%02x\n", ls->link_ctrl);
	printf("clkdiv_start:   0x%02x\n", ls->clkdiv_start);
	printf("clkdiv_run:     0x%02x\n", ls->clkdiv_run);
	printf("link_state:     %d\n", ls->link_state);
	printf("port_cfg:       %d\n", ls->port_cfg);
	printf("port_active:    Port%d\n", ls->port_active);
}

int grspw_link_set(struct grspw_dev *dev, struct grspw_link_ctrl *ctrl)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_LINKCTRL, ctrl);
	if ( result < 0 ) {
		PRINT_ERR("Failed to SET Link-State for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

/* 0=Port0, 1=Port1, Other=HW Select */
int grspw_port_set(struct grspw_dev *dev, int portctrl)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_PORTCTRL, portctrl);
	if ( result < 0 ) {
		PRINT_ERR("Failed to Control Port Configuration for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_start(struct grspw_dev *dev)
{
	if ( ioctl(dev->fd, GRSPW_IOCTL_START, 0) < 0 ) {
		PRINT_ERR("Failed to start SpW%d: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_stop(struct grspw_dev *dev)
{
	if ( ioctl(dev->fd, GRSPW_IOCTL_STOP, 0) < 0 ) {
		PRINT_ERR("Failed to start SpW%d: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

/* 0xff=TCTRL|TIMECNT, 0x100=Write-Enable, 0x200=Generate-Tick-In */
int grspw_tc_set(struct grspw_dev *dev, unsigned int options)
{
	int result;

	result = ioctl(dev->fd, GRSPW_IOCTL_TC_SEND, options);
	if ( result < 0 ) {
		PRINT_ERR("Failed to call TC_SEND for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_tc_read(struct grspw_dev *dev, int *time)
{
	int result;

	if (!time)
		return -1;
	*time = 0;

	result = ioctl(dev->fd, GRSPW_IOCTL_TC_READ, time);
	if ( result < 0 ) {
		PRINT_ERR("Failed to call TC_READ for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_status_read(struct grspw_dev *dev, unsigned int *status)
{
	int result;

	if (!status)
		return -1;
	*status = 0;

	result = ioctl(dev->fd, GRSPW_IOCTL_STATUS_READ, status);
	if ( result < 0 ) {
		PRINT_ERR("Failed to call STATUS_READ for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

int grspw_link_status_clr(struct grspw_dev *dev, unsigned int clearmask)
{
	int result;
	unsigned int mask = clearmask;

	result = ioctl(dev->fd, GRSPW_IOCTL_STATUS_CLR, &mask);
	if ( result < 0 ) {
		PRINT_ERR("Failed to call STATUS_CLR for dev %d, errno: %d\n", dev->index, errno);
		return -1;
	}

	return 0;
}

/* Put Packet Chain 'lst' into the GRSPW driver's SEND List. The packets on
 * the SEND list will at some point make it into a descriptor. What triggers
 * packets to be scheduled are:
 *  - calls to grspw_tx_send()
 *  - calls to grspw_tx_reclaim()
 *  - TX/RX DMA IRQ (Can be controlled from configure structure)
 *
 * dma_chan determines which DMA channel packet will be sent upon. Often only
 * one DMA Channel is available, then set the dma_chan to 0.
 *
 * Return Values:
 *  * positive   = Number of packets sent (lst list updated containing only
 *                 unsent packets). If all packets has been sent 'lst' does
 *                 not contain any packets.
 *  * -1         = Some error occured.
 */
int grspw_tx_send(struct grspw_dev *dev, int dma_chan, struct spwlib_list *lst)
{
	int i, result, size, tot;
	struct grspw_wtxpkt buf[256];
	struct spwlib_pkt *pkt;
	size_t count;

	PRINT("SpW%d: SEND %d\n", dev->index, spwlib_list_cnt(lst));

	tot = 0;
	/* Convert All packets into a "flat" array of packet descriptions
	 * that the GRSPW driver understand.
	 *
	 * To optimize performance packets are handed over to the GRSPW
	 * driver in chunks of 256 packets per write() call.
	 */
	while ( (pkt=lst->head) != NULL)  {
		/* Prepare a chunk of packets */
		for (i=0; i<256; i++) {
			/* Get Index of Frame */
			buf[i].flags = pkt->flags;
			buf[i].hlen = pkt->hlen;
			buf[i].resv = 0;
			buf[i].dlen = pkt->dlen;
			buf[i].hdr = pkt->hdr;
			buf[i].data = pkt->data;
			buf[i].pkt_id = pkt->user_data[0];

			pkt = pkt->next;
			if ( pkt == NULL )
				break;
		}

		/* Inform GRSPW Driver about the frames */
		if ( i >= 256 ) {
			size = sizeof(buf);
		} else {
			size = (i+1)*sizeof(struct grspw_wtxpkt);
		}

		/* The MSB Bits determine functionality (SEND or PREPARE) and
		 * which DMA Channel is used to transmit packets.
		 */
		count = size | GRSPW_WRITE_SEND | 
			((dma_chan << GRSPW_WRITE_CHAN_BIT) & GRSPW_WRITE_CHAN);
		result = WRITE(dev->fd, buf, count);
		if ( result < 0 ) {
			PRINT_ERR("SpW%d: SEND %d failed: %d\n", dev->index, spwlib_list_cnt(lst), errno);
			return -1;
		}
		tot += result / sizeof(struct grspw_wtxpkt);

		/* Update Packet List Accorind to how many packets were queued
		 * up in driver SEND Q.
		 */
		if ( result < size ) {
			/* Not all packets were sent, we rewind and return
			 * to caller indicating that not all packets were
			 * sent. This can happen if driver has not enough
			 * packet structures to hold the packet information
			 * we are giving it.
			 *
			 * NOT IMPLEMENTED.
			 */
			count = result / sizeof(struct grspw_wtxpkt);
			while ( count-- > 0 ) {
				lst->head = lst->head->next;
			}
			return 1;
		} else {
			lst->head = pkt;
			if ( pkt == NULL )
				lst->tail = NULL;
		}
	}

	return tot;
}

/* Get as many sent/error TX Packets previously shceudled for transmission with
 * grspw_tx_send(). The reclaimed packet buffers will be put in the Packet Chain
 * pointed to be 'lst'. The reclaimed packets are taken from the GRSPW driver's
 * SENT List.
 *
 * dma_chan_mask is a bit mask indicating which DMA channels sent packet buffers
 * are reclaimed from. Often only one DMA Channels is available, then set the
 * bit-mask to 1.
 *
 * Return Values:
 *  * positive   = Number of packets reclaimed (lst list updated containing
 *                 reclaimed packets). If no packets has been sent 'lst' does
 *                 not contain any packets.
 *  * -1         = Some error occured.
 */
int grspw_tx_reclaim(struct grspw_dev *dev, int dma_chan_mask, struct spwlib_list *lst)
{
	int len, i, tot;
	struct grspw_rtxpkt buf[512];
	struct spwlib_pkt *pkt = NULL;
	struct spwlib_pkt **prev;
	size_t count;

#ifdef TX_RECLAIM_ADD_END
	if ( lst->head == NULL ) {
		prev = &lst->head;
	} else {
		prev = &lst->tail->next;
	}
#else
	spwlib_list_clr(lst);
	prev = &lst->head;
#endif

	tot = 0;
	/* The MSB Bits determine functionality (RECLAIM or RECV) and
	 * which DMA Channels is used to reclaim packet buffers from.
	 */
	count = sizeof(buf) | GRSPW_READ_RECLAIM |
		((dma_chan_mask << GRSPW_READ_CHANMSK_BIT) & GRSPW_READ_CHANMSK);
	while ( (len=READ(dev->fd, &buf[0], count)) > 0 ) {

		/* Get number of Packets descriptorions copied */
		len = len/sizeof(struct grspw_rtxpkt);

		/* Convert one packet information outputed by driver into
		 * a application-packet description per loop.
		 */
		for (i=0; i<len; i++) {
#if 0
			/* Check driver internal problem... */
			if ( buf[i].pkt_id >= dev->pkt_id_max ) {
				PRINT_ERR("TX[%d] RECLAIM: Invalid Packet ID\n", dev->index);
				return -1;
			}
#endif

			/* Get Application-"Packet structure" from packet ID
			 * written to driver previously.
			 */
			pkt = dev->pktid2pkt[buf[i].pkt_id];
			pkt->flags = buf[i].flags;

			/* Check Flags for errors */
			if ( pkt->flags & TXPKT_FLAG_LINKERR ) {
				/* Something bad happended. Handle this somewhere
				 * else? Where packets are handled.
				 */
				PRINT_ERR("TX[%d] RECLAIM: packet error: 0x%x\n", dev->index, pkt->flags);
				return -1;
			}

			/* Insert Packet last into List */
			*prev = pkt;
			prev = &pkt->next;
		}
		tot += len;
		pkt->next = NULL;
		lst->tail = pkt;

		if ( len < 512 ) /* Stop early to save CPU */
			break;
	}
	if ( len < 0 ) {
		PRINT_ERR("SpW%d: TX RECLAIM: Error: %d\n", dev->index, errno);
		return -1;
	}

	PRINT("SpW%d: TX-RECLAIM %d packets\n", dev->index, tot);

	return tot;
}

int grspw_rx_prepare(struct grspw_dev *dev, int dma_chan, struct spwlib_list *lst)
{
	int i, result;
	struct grspw_wrxpkt buf[512];
	struct spwlib_pkt *pkt;
	size_t count;

	PRINT("SpW%d: Prepare %d\n", dev->index, spwlib_list_cnt(lst));

	while ( (pkt=lst->head) != NULL)  {
		for (i=0; i<512; i++) {
			/* Get Index of Frame */
			buf[i].flags = pkt->flags;
			buf[i].resv1 = 0;
			buf[i].data = pkt->data;
			buf[i].pkt_id = pkt->user_data[0];

			pkt = pkt->next;
			if ( pkt == NULL )
				break;
		}

		/* Update Free Frame List */
		lst->head = pkt;
		if ( pkt == NULL )
			lst->tail = NULL;

		/* Inform GRSPW Driver about the frames */
		if ( i >= 512 ) {
			count = sizeof(buf);
		} else {
			count = (i+1)*sizeof(struct grspw_wrxpkt);
		}
		count |= (dma_chan << GRSPW_WRITE_CHAN_BIT) & GRSPW_WRITE_CHAN;
		result = WRITE(dev->fd, buf, count);
		if ( result < 0 ) {
			PRINT_ERR("SpW%d: Prepare %d failed: %d\n", dev->index, spwlib_list_cnt(lst), errno);
			return -1;
		}
	}

	return 0;
}

int grspw_rx_recv(struct grspw_dev *dev, int dma_chan_mask, struct spwlib_list *lst)
{
	int len, i, tot;
	struct grspw_rrxpkt buf[512];
	struct spwlib_pkt *pkt = NULL;
	struct spwlib_pkt **prev;
	size_t count;

#ifdef RX_RECV_ADD_END
	if ( lst->head == NULL ) {
		prev = &lst->head;
	} else {
		prev = &lst->tail->next;
	}
#else
	spwlib_list_clr(lst);
	prev = &lst->head;
#endif

	tot = 0;
	count = sizeof(buf) | ((dma_chan_mask << GRSPW_READ_CHANMSK_BIT) & GRSPW_READ_CHANMSK);
	while ( (len=READ(dev->fd, buf, count)) > 0 ) {
		len = len/sizeof(struct grspw_rrxpkt);
		for (i=0; i<len; i++) {
#if 0
			/* Check driver internal problem... */
			if ( buf[i].pkt_id >= dev->pkt_id_max ) {
				PRINT_ERR("RX[%d] RECLAIM: Invalid Packet ID\n", dev->index);
				return -1;
			}
#endif
			pkt = dev->pktid2pkt[buf[i].pkt_id];
			pkt->flags = buf[i].flags;
			pkt->dlen = buf[i].dlen;

			/* Check Flags for errors */
			if ( pkt->flags & (RXPKT_FLAG_TRUNK|RXPKT_FLAG_EEOP) ) {
				/* Something bad happended */
				PRINT_ERR("RX[%d] RECV: pkt->flags error: 0x%08x\n", dev->index, pkt->flags);
				return -1;
			}

			/* Insert Packet last into List */
			*prev = pkt;
			prev = &pkt->next;
		}
		tot += len;
		pkt->next = NULL;
		lst->tail = pkt;
		if ( len < 512 )
			break; /* Stop process incomming packets just to save CPU */
	}
	if ( len < 0 ) {
		PRINT_ERR("RX[%d] RECV: Error: %d\n", dev->index, errno);
		return -1;
	}

	PRINT("SpW%d: RECV %d\n", dev->index, tot);

	return 0;
}

int grspw_tx_wait(struct grspw_dev *dev, int dma_chan, int send_cnt, int op, int sent_cnt, long timeout_ms)
{
	int result;
	struct grspw_tx_wait_chan wait;

	wait.chan = dma_chan;
	wait.reserved = 0;
	wait.timeout_ms = timeout_ms;
	wait.send_cnt = send_cnt;
	wait.op = op;
	wait.sent_cnt = sent_cnt;

	result = ioctl(dev->fd, GRSPW_IOCTL_TX_WAIT, &wait);
	if ( result == -ETIME) {
		return 2;
	} else if ( result == -EIO) {
		return 1;
	} else if ( result == -EBUSY) {
		return -1;
	} else if ( result < 0 ) {
		PRINT_ERR("Failed to TX-wait for dev %d.%d, errno: %d\n", dev->index, dma_chan, errno);
		return -1;
	}

	return 0;
}

int grspw_rx_wait(struct grspw_dev *dev, int dma_chan, int recv_cnt, int op, int ready_cnt, long timeout_ms)
{
	int result;
	struct grspw_rx_wait_chan wait;

	wait.chan = dma_chan;
	wait.reserved = 0;
	wait.timeout_ms = timeout_ms;
	wait.recv_cnt = recv_cnt;
	wait.op = op;
	wait.ready_cnt = ready_cnt;

	result = ioctl(dev->fd, GRSPW_IOCTL_RX_WAIT, &wait);
	if ( result == -ETIME) {
		return 2;
	} else if ( result == -EIO) {
		return 1;
	} else if ( result == -EBUSY) {
		return -1;
	} else if ( result < 0 ) {
		PRINT_ERR("Failed to RX-wait for dev %d.%d, errno: %d\n", dev->index, dma_chan, errno);
		return -1;
	}

	return 0;
}
