/* AMBA Plug&Play scanning and initialization
 */

#include <common.h>
#include <startup.h>
#include <prom.h>
#include <ambapp.h>
#include <gptimer.h>
#include <irqmp.h>
#include <apbuart.h>
#include <iommu.h>

#ifdef CONFIG_AMPOPTS
struct amp_opt {
	short idx, val;
} ampopts[] = {
#include "config_ampopts.h"
	{-1, -1} /* END */
};
#endif

void _memcpy4_pa(void *dst, void *src, int len);

ambapp_dev *ambapp_get_first_dev(ambapp_core *core)
{
	if (core->ahbmst)
		return (ambapp_dev *)core->ahbmst;
	else if (core->ahbslv)
		return (ambapp_dev *)core->ahbslv;
	else if (core->apbslv)
		return (ambapp_dev *)core->apbslv;
	else {
		/* this should not happen */
		while (1)
			;
	}
}

void leon_init(ambapp_core *core, int level)
{
	unsigned long leon_freq_hz;

	if (level != 1)
		return;

	/* Get frequency for the CPU */
	leon_freq_hz = ambapp_dev_freq((ambapp_dev *)core->ahbmst);
	cpu_freq_khz = int_div(leon_freq_hz, 1000);
}

static int irqmp_initied = 0;

void irqmp_init(ambapp_core *core, int level)
{
	/* Only process the first GPTIMER core */
	if (irqmp_initied)
		return;

	irqmp_initied = 1;
	irqmp = (struct irqmp_regs *)core->apbslv->address;
}

static int gptimer_inited = 0;

/* Fixup IRQ of GPTIMER APBSLV interface */
int gptimer_apb_fixup(ambapp_dev *dev)
{
	struct gptimer_regs *regs;
	ambapp_apbdev *apbslv;
	uint32_t config;
	int ntimers, i;

	if (dev->dev_type != DEV_APB_SLV)
		return 0;
	apbslv = (ambapp_apbdev *)dev;
	regs = (struct gptimer_regs *)apbslv->address;
	config = BYPASS_LOAD_PA(&regs->config);
	if ((config & (1<<8)) == 0)
		return 0; /* All timers share the first IRQ */

	ntimers = config & 0x7;
	for (i = 1; i < ntimers; i++) {
		apbslv->irq[i] = apbslv->irq[0] + i;
	}
	return 0;
}

void gptimer_init(ambapp_core *core, int level)
{
	/* Only process the first GPTIMER core */
	if (gptimer_inited)
		return;

	gptimer_inited = 1;
	gptimer = (struct gptimer_regs *)core->apbslv->address;

#ifndef CONFIG_BOOTLOADER_FREQ
	{
		unsigned long timer_freq, scalar;
		struct apb_bus *apbbus;

		scalar = BYPASS_LOAD_PA(&gptimer->scalar_reload);
		if (scalar == 0)
			while (1) ; /* Frequency never initialized */
		timer_freq = int_mul(scalar + 1, 1000000);

		/* Set the frequency for the AHB bus that GPTIMER is bridged
		 * from. The APB frequency is always the same as the AHB
		 * frequency.
		 */
		apbbus = &ambapp_plb.apbs[core->apbslv->bus_idx];
		ambapp_freq_init(apbbus->parent_bus_idx, timer_freq);
	}
#endif
}

static int apbuart_initied = CONFIG_UART_INDEX + 1;

/* Find first APBUART for console */
void apbuart_init(ambapp_core *core, int level)
{
	if (apbuart_initied != 0 && (--apbuart_initied == 0) && core->apbslv) {
		apbuart = (struct apbuart_regs *)core->apbslv->address;
	}
}

/* Turn off GRETH RX/TM DMA operations. This should not have to be done here,
 * however during debug (no proper shutdown or reset/boot) the GRETH may still
 * be active. This may lead to DMA overwrites of early Linux initalization
 * memory since the GRETH may overwrite them.
 */
void greth_init(ambapp_core *core, int level)
{
	struct greth_regs {
		uint32_t ctrl;
		uint32_t status;
		/* ... */
	} *regs;

	if (level == 0 && core->apbslv) {
		/* Turn off RX/TX DMA */
		regs = (struct greth_regs *)core->apbslv->address;
		BYPASS_STORE_PA(&regs->ctrl, BYPASS_LOAD_PA(&regs->ctrl)&~0xf);
		/* Clear status */
		BYPASS_STORE_PA(&regs->status, 0xffffffff);
	}
}

struct amba_driver {
	unsigned short vendor;
	unsigned short device;
	unsigned char ver_min;
	unsigned char ver_max;
	unsigned char drv_type; /* 0x1=MST, 0x2=AHBSLV, 0x4=APBSLV, 0x8=CORE */
	int (*fixup_dev)(ambapp_dev *dev);
	int (*fixup_core)(ambapp_core *core);
	void (*init)(ambapp_core *core, int level);
	ambapp_core *cores;
};

/* Driver for a special version */
#define AMBAPP_DRV_VER(ven, dev, vmin, vmax, type, fixdev, fixcore, init) \
		{ven, dev, vmin, vmax, type, fixdev, fixcore, init, NULL}
/* Driver for all versions */
#define AMBAPP_DRV(ven, dev, type, fixdev, fixcore, init) \
		{ven, dev, 0, 0xff, type, fixdev, fixcore, init, NULL}

struct amba_driver amba_drivers[] = {
 AMBAPP_DRV(VENDOR_GAISLER, GAISLER_IRQMP, 8, NULL, NULL, irqmp_init),
 AMBAPP_DRV(VENDOR_GAISLER, GAISLER_GPTIMER, 8, gptimer_apb_fixup, NULL, gptimer_init),
 AMBAPP_DRV(VENDOR_GAISLER, GAISLER_LEON4, 8, NULL, NULL, leon_init),
 AMBAPP_DRV(VENDOR_GAISLER, GAISLER_LEON3, 8, NULL, NULL, leon_init),
 AMBAPP_DRV(VENDOR_GAISLER, GAISLER_LEON3FT, 8, NULL, NULL, leon_init),
 AMBAPP_DRV(VENDOR_GAISLER, GAISLER_APBUART, 8, NULL, NULL, apbuart_init),
 AMBAPP_DRV(VENDOR_GAISLER, GAISLER_ETHMAC, 8, NULL, NULL, greth_init),
};
#define AMBAPP_DRV_NUM (sizeof(amba_drivers)/sizeof(struct amba_driver))

/* Fixup AMBA PnP device information
 *
 * Return values:
 *  0 = Proceed as usual
 *  1 = Skip device
 */
int ambapp_fixup_dev(ambapp_dev *dev)
{
	struct amba_driver *drv;
	int i;

	for (i = 0, drv = amba_drivers; i < AMBAPP_DRV_NUM; i++, drv++) {
		if ((drv->device == dev->device) &&
		    (drv->fixup_dev != NULL) &&
		    (drv->vendor == dev->vendor) &&
		    ((dev->ver >= drv->ver_min) &&
		    (dev->ver <= drv->ver_max))) {
			return drv->fixup_dev(dev);
		}
	}

	return 0;
}

/* Fixup AMBA PnP core information
 *
 * Return values:
 *  0 = Proceed as usual
 *  1 = Skip core
 */
int ambapp_fixup_core(ambapp_core *core)
{
	struct amba_driver *drv;
	ambapp_dev *dev;
	int i;

	/* Find a device so that we get a VENDOR:DEVICE:VERSION */
	if (core->ahbmst)
		dev = (ambapp_dev *)core->ahbmst;
	else if (core->ahbslv)
		dev = (ambapp_dev *)core->ahbslv;
	else if (core->apbslv)
		dev = (ambapp_dev *)core->apbslv;
	else
		while (1) ; /* this should not happen */

	for (i = 0, drv = amba_drivers; i < AMBAPP_DRV_NUM; i++, drv++) {
		if ((drv->device == dev->device) &&
		    (drv->fixup_core != NULL) &&
		    (drv->vendor == dev->vendor) &&
		    ((dev->ver >= drv->ver_min) &&
		    (dev->ver <= drv->ver_max))) {
			return drv->fixup_core(core);
		}
	}

	return 0;
}

int ambapp_node_fixup(ambapp_core *core, struct node *node)
{
	ambapp_dev *dev = ambapp_get_first_dev(core);

	if (dev->vendor == VENDOR_GAISLER) {
#ifdef CONFIG_APBUART_LINUX_MAX
		/* One you the first N APBUART cores */
		static int apbuart_cnt = 0;
		if (dev->device == GAISLER_APBUART) {
			if (apbuart_cnt++ >= CONFIG_APBUART_LINUX_MAX)
				return 1;
		}
#endif

#ifdef CONFIG_MINIMAL_CORES
		/* Ignore other cores that what is required */
		if (dev->device != GAISLER_APBUART &&
		    dev->device != GAISLER_IRQMP &&
		    dev->device != GAISLER_GPTIMER)
			return 1;
#endif
	}
	return 0;
}

/*  */
struct ambapp_bridge_driver {
	int vendor;		/* VENDOR */
	int device;		/* DEVICE */
	unsigned char ver_min;	/* Version Minimum */
	unsigned char ver_max;	/* Version Maximum */
	unsigned char type;	/* 0=AHB-AHB, 1=AHB-APB */
	void (*create)(struct ambapp_bus *ainfo,
		struct ahb_bus *parent,
		ambapp_ahbdev *bridge);
};

/* proc.c */
extern unsigned long amba_systemid;
extern unsigned long ipi_num;
struct propa_ptr props_ambapp_templ[] = {
	PROPA_PTR("device_type", "ambapp", 7),
	PROPA_PTR("name", "ambapp0", 8),
	PROPA_PTR("systemid", &amba_systemid, 4),
	PROPA_PTR("ioarea", 0, 4),
	PROPA_PTR_END("ipi_num", &ipi_num, 4),
};
#define NAMEIDX   1
#define IOAREAIDX 3

struct node *ahb2ahb_create_bus_node(struct ambapp_bus *ainfo, unsigned long ioarea, int bus_idx) 
{
	struct node *n = 0;
#ifndef FLATOFTREE
	struct propa_ptr *pa;

	/* create a bus node */
	n = (struct node *)prom_calloc(sizeof(struct node));
	/* initialize properties from template */
	pa = (struct propa_ptr *)prom_calloc(sizeof(props_ambapp_templ));
	_memcpy((void*)pa, (void*)&props_ambapp_templ, sizeof(props_ambapp_templ));
	n->props = (struct prop *)pa;
	/* customize ioarea */
	pa[IOAREAIDX].value = prom_calloc(sizeof(int));
	*(int *)pa[IOAREAIDX].value = ioarea;
	/* customize name */
	pa[NAMEIDX].value = prom_calloc(8);
	_memcpy((void*)pa[NAMEIDX].value,"ambapp",6);
	((char *)pa[NAMEIDX].value)[6] = '0' + bus_idx;
	ainfo->ahbsn[bus_idx] = n;
	ainfo->ahbsnn[bus_idx] = &n->child;
#endif
	return n;
}

/* AHB-AHB IOMMU Creation */
void ahb2ahb_bus_create_iommu(struct ambapp_bus *ainfo, struct ahb_bus *parent,
			ambapp_ahbdev *bridge)
{
	struct ahb_bus *subbus;
	struct node *n;
	struct iommu_regs *regs = 0;
	ambapp_ahbdev *pdev;
	int j;
	
	/* Add new bus to AMBA Information */
	subbus = &ainfo->ahbs[ainfo->ahb_buses];
	subbus->bus_idx = ainfo->ahb_buses++;
	subbus->parent_bus_idx = parent->bus_idx;

	/* save the bus-idx the io-mmu should node-link */
	bridge->extra = subbus->bus_idx;
	
	/* Get info from Bridge's PnP custom information */
	subbus->ioarea = -2; /* mark as ioarea, already scanned */
	subbus->ffact = 1;
	subbus->dir = 1; /* up */
	subbus->isiommu = 1;
	
	/* get per bus node */
	n = ahb2ahb_create_bus_node(ainfo, 0, subbus->bus_idx);

	/* scan for reg address */
	for (j = 0; j < 4; j++) {
		if (bridge->address[j] && bridge->type[j] == AMBA_TYPE_AHBIO) {
			regs = (struct iommu_regs *)bridge->address[j];
			break;
		}
	}

	/* scan the pnp information */
	subbus->mst_cnt = 0;
	if (regs) {
		subbus->msts = (ambapp_ahbdev *)startup_malloc(16 * sizeof(ambapp_ahbdev));
		pdev = &subbus->msts[0];
		for (j = 0; j < 16; j++) {
			uint32_t mst;
			_memcpy4_pa((void *)&mst, (void *)&regs->mstconf[j], 4);
			if (mst) {
				struct ambapp_pnp_ahb ahb_buf;
				_memset((void*)&ahb_buf, 0, sizeof(ahb_buf));
				*(int *)&(ahb_buf.id) = mst;
				*(int *)&(ahb_buf.mbar[0]) = 0; /* mst */
				
				/* Build a AHB Master Device */
				pdev->dev_type = DEV_AHB_MST;
				ambapp_dev_init_ahb(subbus, &ahb_buf, pdev);
				
				if (ambapp_fixup_dev((ambapp_dev *)pdev) == 0) {
					subbus->mst_cnt++;
					pdev++;
				} 
			}
                }
	}
	
}

/* AHB-AHB Bridge Creation */
void ahb2ahb_bus_create(struct ambapp_bus *ainfo, struct ahb_bus *parent,
			ambapp_ahbdev *bridge)
{
	struct ahb_bus *subbus;
	unsigned long ioarea;
	int i;
	struct node *n;
	
	/* Is there a PnP area behind the bridge?*/
	ioarea = bridge->userdef[1];
	if (ioarea == 0)
		return;

	/* Check that bus hasn't been scanned already */
	for (i = 0; i < ainfo->ahb_buses; i++)
		if (ainfo->ahbs[i].ioarea == ioarea)
			return;

	
	/* Add new bus to AMBA Information */
	subbus = &ainfo->ahbs[ainfo->ahb_buses];
	subbus->bus_idx = ainfo->ahb_buses++;
	subbus->parent_bus_idx = parent->bus_idx;

	/* save the bus-idx the bridge should node-link */
	bridge->extra = subbus->bus_idx;

	/* Get info from Bridge's PnP custom information */
	subbus->ioarea = ioarea;
	subbus->ffact = (bridge->userdef[0] & AMBAPP_FLAG_FFACT) >> 4;
	subbus->dir = (bridge->userdef[0] & AMBAPP_FLAG_FFACT_DIR);

	/* get per bus node */
	n = ahb2ahb_create_bus_node(ainfo, ioarea, subbus->bus_idx);
	
}

/* AHB->APB Bridge */
void ahb2apb_bus_create(struct ambapp_bus *ainfo, struct ahb_bus *parent,
			ambapp_ahbdev *bridge)
{
	struct apb_bus *subbus;

	subbus = &ainfo->apbs[ainfo->apb_buses];
	subbus->bus_idx = ainfo->apb_buses++;
	subbus->parent_bus_idx = parent->bus_idx;
	subbus->ahb_bus_idx = bridge->bus_idx;
	
	/* Get info from Bridge's PnP custom information */
	subbus->ioarea = bridge->address[0];
}

struct ambapp_bridge_driver bridgedrvs[] = {
	{
		.vendor = VENDOR_GAISLER,
		.device = GAISLER_AHB2AHB,
		.ver_min = 1,
		.ver_max = 0xff,
		.type = 0,
		.create = ahb2ahb_bus_create
	},
	{
		.vendor = VENDOR_GAISLER,
		.device = GAISLER_APBMST,
		.ver_min = 0,
		.ver_max = 0xff,
		.type = 1,
		.create = ahb2apb_bus_create
	},
	{
		.vendor = VENDOR_GAISLER,
		.device = GAISLER_GRIOMMU,
		.ver_min = 0,
		.ver_max = 0xff,
		.type = 0,
		.create = ahb2ahb_bus_create_iommu
	},
	{
		.vendor = VENDOR_GAISLER,
		.device = GAISLER_L2CACHE,
		.ver_min = 0,
		.ver_max = 0xff,
		.type = 0,
		.create = ahb2ahb_bus_create
	},
};
#define AMBAPP_BRIDGE_DRV_NUM \
		(sizeof(bridgedrvs) / sizeof(struct ambapp_bridge_driver))


/************************/

void _memcpy4_pa(void *dst, void *src, int len)
{
	unsigned long *d, *s, *end;

	/* only request a even count words */
	if (len & 0x3)
		while (1) ;

	d = dst;
	s = src;
	end = (unsigned long *)(src + len);
	while (s < end) {
		*d = BYPASS_LOAD_PA(s);
		d++;
		s++;
	}
}

/* Detect a driver for AHB<->AHB, AHB->AHB or AHB->APB bridge */
static struct ambapp_bridge_driver *ambapp_bridge_detect(ambapp_ahbdev *dev)
{
	int i;
	struct ambapp_bridge_driver *drv;

	for (i = 0; i < AMBAPP_BRIDGE_DRV_NUM; i++) {
		drv = &bridgedrvs[i];

		if ((drv->device == dev->device) &&
		    (drv->vendor == dev->vendor) &&
		    ((dev->ver >= drv->ver_min) &&
		    (dev->ver <= drv->ver_max)))
			return drv;
	}

	return NULL;
}

void ambapp_dev_init_ahb(
	struct ahb_bus *bus,
	struct ambapp_pnp_ahb *ahb,
	ambapp_ahbdev *dev
	)
{
	int bar;
	unsigned int addr, mask, mbar, type;

	/* Setup device struct */
	dev->vendor = ambapp_pnp_vendor(ahb->id);
	dev->device = ambapp_pnp_device(ahb->id);
	dev->ver = ambapp_pnp_ver(ahb->id);
	_memset(&dev->irq[0], 0, sizeof(dev->irq));
	dev->extra = 0;
	dev->irq[0] = ambapp_pnp_irq(ahb->id);
	dev->bus_idx = bus->bus_idx;
	dev->userdef[0] = (unsigned int)ahb->custom[0];
	dev->userdef[1] = (unsigned int)ahb->custom[1];
	dev->userdef[2] = (unsigned int)ahb->custom[2];

	/* Memory BARs */
	for (bar = 0; bar < 4; bar++) {
		mbar = ahb->mbar[bar];
		if (mbar == 0) {
			addr = 0;
			mask = 0;
			type = 0;
		} else {
			addr = ambapp_pnp_mbar_start(mbar);
			type = ambapp_pnp_mbar_type(mbar);
			if (type == AMBA_TYPE_AHBIO) {
				/* AHB I/O area is releative IO_AREA */
				addr = ambapp_pnp_ahbio_adr(addr, bus->ioarea);
				mask = (((unsigned int)(ambapp_pnp_mbar_mask(~mbar)<<8)|0xff))+1;
			} else {
				/* AHB memory area, absolute address */
				mask = (~((unsigned int)(ambapp_pnp_mbar_mask(mbar)<<20)))+1;
			}
		}
		dev->address[bar] = addr;
		dev->mask[bar] = mask;
		dev->type[bar] = type;
	}
}

void ambapp_dev_init_apb(
	struct apb_bus *bus,
	struct ambapp_pnp_apb *apb,
	ambapp_apbdev *dev)
{
	/* Setup device struct */
	dev->bus_idx = bus->bus_idx;
	dev->vendor = ambapp_pnp_vendor(apb->id);
	dev->device = ambapp_pnp_device(apb->id);
	dev->ver = ambapp_pnp_ver(apb->id);
	_memset(&dev->irq[0], 0, sizeof(dev->irq));
	dev->extra = 0;
	dev->irq[0] = ambapp_pnp_irq(apb->id);
	dev->bus_idx = bus->bus_idx;
	dev->extra = bus->ahb_bus_idx;
	dev->address = ambapp_pnp_apb_start(apb->iobar, bus->ioarea);
	dev->mask = ambapp_pnp_apb_mask(apb->iobar);
}

/* Finds for AHBMST, AHBSLV and APBSLV devices, counts then (returns count)
 * puts the addresses into pnp_entries in order
 */
static int ambapp_scan_bus(
	unsigned long pnparea,
	int max,
	int entry_size,
	void **pnp_entries,
	int *tot_dev_size /* I/O */
	)
{
	void *entry;
	unsigned long id;
	int cnt, i, dev_size = *tot_dev_size;

	cnt = 0;
	*tot_dev_size = 0;
	entry = (void *)pnparea;
	for (i = 0; i < max; i++) {
		_memcpy4_pa(&id, entry, 4);
		if (id != 0) {
			/* A device present here */
			pnp_entries[cnt++] = entry;
			*tot_dev_size = *tot_dev_size + dev_size;
		}
		entry += entry_size;
	}
	return cnt;
}

/* Scan one AHB bus, both AHB Masters and AHB Slaves */
static void ambapp_scan_bus_ahb(
	struct ambapp_bus *ainfo,
	struct ahb_bus *bus
	)
{
	struct ambapp_pnp_ahb *ahbs[64], ahb_buf, **pahb;
	ambapp_ahbdev *pdev;
	int i, size, max_entries;
	unsigned long ioarea;
	struct ambapp_bridge_driver *busdrv;

	/* scan first bus for 64 devices, rest for 16 devices */
	if (bus->bus_idx == 0)
		max_entries = 64;
	else
		max_entries = 16;

	/* AHB MASTERS */
	ioarea = bus->ioarea | AMBA_CONF_AREA;
	size = sizeof(ambapp_ahbdev);
	bus->mst_cnt = ambapp_scan_bus(ioarea, max_entries,
				sizeof(struct ambapp_pnp_ahb), (void **)ahbs,
				&size);
	bus->msts = (ambapp_ahbdev *)startup_malloc(size);

	for (i = 0, pahb = &ahbs[0], pdev = &bus->msts[0];
		i < bus->mst_cnt; i++, pahb++, pdev++) {
		_memcpy4_pa(&ahb_buf, *pahb, sizeof(struct ambapp_pnp_ahb));

		/* Build a AHB Master Device */
		pdev->dev_type = DEV_AHB_MST;
		ambapp_dev_init_ahb(bus, &ahb_buf, pdev);

		if (ambapp_fixup_dev((ambapp_dev *)pdev) == 1) {
			/* skip device */
			bus->mst_cnt--;
			pdev--;
			i--;
			continue;
		}
	}

	ioarea |= AMBA_AHBSLV_AREA;
	size = sizeof(ambapp_ahbdev);
	bus->slv_cnt = ambapp_scan_bus(ioarea, max_entries,
				sizeof(struct ambapp_pnp_ahb), (void **)ahbs,
				&size);
	bus->slvs = (ambapp_ahbdev *)startup_malloc(size);

	for (i = 0, pahb = &ahbs[0], pdev = &bus->slvs[0];
		i < bus->slv_cnt; i++, pahb++, pdev++) {
		_memcpy4_pa(&ahb_buf, *pahb, sizeof(struct ambapp_pnp_ahb));

		/* Build a AHB Slave Device */
		pdev->dev_type = DEV_AHB_SLV;
		ambapp_dev_init_ahb(bus, &ahb_buf, pdev);

		if (ambapp_fixup_dev((ambapp_dev *)pdev) == 1) {
			/* skip device */
			bus->slv_cnt--;
			pdev--;
			i--;
			continue;
		}

		/* Detect if the device is a bridge. In that case just add the
		 * it, do not scan it. This is a one bus scan routine...
		 */
		busdrv = ambapp_bridge_detect(pdev);
		if (busdrv)
			busdrv->create(ainfo, bus, pdev);
	}
}

static void ambapp_scan_bus_apb(struct ambapp_bus *ainfo, struct apb_bus *bus)
{
	struct ambapp_pnp_apb *apbs[AMBA_APB_SLAVES], apb_buf, **papb;
	ambapp_apbdev *pdev;
	int i, size;

	/* AHB MASTERS */
	size = sizeof(ambapp_apbdev);
	bus->slv_cnt = ambapp_scan_bus(bus->ioarea | AMBA_CONF_AREA,
					AMBA_APB_SLAVES,
					sizeof(struct ambapp_pnp_apb),
					(void **)apbs,
					&size);
	bus->slvs = (ambapp_apbdev *)startup_malloc(size);

	for (i = 0, papb = &apbs[0], pdev = &bus->slvs[0];
		i < bus->slv_cnt; i++, papb++, pdev++) {
		_memcpy4_pa(&apb_buf, *papb, sizeof(struct ambapp_pnp_apb));

		/* Build a AHB Slave Device */
		pdev->dev_type = DEV_APB_SLV;
		ambapp_dev_init_apb(bus, &apb_buf, pdev);

		if (ambapp_fixup_dev((ambapp_dev *)pdev) == 1) {
			/* skip device */
			bus->slv_cnt--;
			pdev--;
			i--;
			continue;
		}

		/* APB never have bridges */
	}
}

/* Create Root bus and initialize AMBA information */
void ambapp_bus_init(
	struct ambapp_bus *ainfo,
	unsigned long ioarea)
{
	int i;

	_memset(ainfo, 0, sizeof(ainfo));
	ainfo->ahbs[0].parent_bus_idx = 0xff; /* No parent */
	ainfo->ahbs[0].ioarea = ioarea;
	ainfo->ahb_buses = 1;

	for (i = 0; i < AMBA_AHB_MAX; i++) {
		if (ainfo->ahbs[i].ioarea == -2) /* iommu */
			continue;
		if (ainfo->ahbs[i].ioarea == 0)
			break;
		ambapp_scan_bus_ahb(ainfo, &ainfo->ahbs[i]);
	}

	for (i = 0; i < AMBA_APB_MAX; i++) {
		if (ainfo->apbs[i].ioarea == 0)
			break;
		ambapp_scan_bus_apb(ainfo, &ainfo->apbs[i]);
	}

#ifdef CONFIG_BOOTLOADER_FREQ
	/* Register Hardwcoded Bus frequency for AHB Bus 0 */
	ambapp_freq_init(0, CONFIG_BOOTLOADER_FREQ);
#endif
}

/* AMBA Core Counter */
static int ambapp_core_cnt = 0;

int ambapp_for_each_dev(struct ambapp_bus *ainfo, int type,
				ambapp_dev_f func, void *arg)
{
	int ret, i, j;
	struct ahb_bus *ahbbus;
	struct apb_bus *apbbus;
	ambapp_ahbdev *ahb;
	ambapp_apbdev *apb;

	/* AMBA AHB Masters */
	if (type & 0x1) {
		for (i = 0, ahbbus = ainfo->ahbs;
		     i < ainfo->ahb_buses; i++, ahbbus++) {
			for (j = 0, ahb = ahbbus->msts;
			     j < ahbbus->mst_cnt; j++, ahb++) {
				ret = func(ainfo, (ambapp_dev *)ahb, arg);
				if (ret != 0)
					return ret;
			}
		}
	}

	/* AMBA AHB Slaves */
	if (type & 0x2) {
		for (i = 0, ahbbus = ainfo->ahbs;
		     i < ainfo->ahb_buses; i++, ahbbus++) {
			for (j = 0, ahb = ahbbus->slvs;
			     j < ahbbus->slv_cnt; j++, ahb++) {
				ret = func(ainfo, (ambapp_dev *)ahb, arg);
				if (ret != 0)
					return ret;
			}
		}
	}

	/* 3. Find all AMBA cores with only an AMBA APB Slave device */
	if (type & 0x4) {
		for (i = 0, apbbus = ainfo->apbs;
		     i < ainfo->apb_buses; i++, apbbus++) {
			for (j = 0, apb = apbbus->slvs;
			     j < apbbus->slv_cnt; j++, apb++) {
				ret = func(ainfo, (ambapp_dev *)apb, arg);
				if (ret != 0)
					return ret;
			}
		}
	}

	return 0; /* All Devices Processed */
}

ambapp_core *ambapp_core_create(
	struct ambapp_bus *ainfo,
	ambapp_dev *dev,
	int options)
{
	ambapp_core *core;
	struct ahb_bus *ahbbus;
	struct apb_bus *apbbus;
	ambapp_ahbdev *ahb;
	ambapp_apbdev *apb;
	int i, j;
	int vendor, device, ver;

	if (dev->irq[0] & 0x80) {
		/* This is the last time we will search this device,
		 * so we can unmark it as used.
		 */
		dev->irq[0] &= ~0x80;
		return NULL;
	}

	core = (ambapp_core *)startup_malloc(sizeof(ambapp_core));
	_memset(core, 0, sizeof(ambapp_core));

	vendor = dev->vendor;
	device = dev->device;
	ver = dev->ver;

	/* Find a AHB Slave for Device */
	if (options & 0x2) {
		for (i = 0, ahbbus = ainfo->ahbs;
		     i < ainfo->ahb_buses; i++, ahbbus++) {
			for (j = 0, ahb = ahbbus->slvs;
			     j < ahbbus->slv_cnt; j++, ahb++) {
				if ((vendor == ahb->vendor) &&
				    (device == ahb->device) &&
				    (ver == ahb->ver) &&
				    ((ahb->irq[0] & 0x80) == 0)) {
					/* AHB Slave Device matching and not
					 * marked taken.
					 */
					ahb->irq[0] |= 0x80;
					core->ahbslv = ahb;
					break;
				}
			}
		}
	}

	/* Find a APB Slave for Device */
	if (options & 0x4) {
		for (i = 0, apbbus = ainfo->apbs;
		     i < ainfo->apb_buses; i++, apbbus++) {
			for (j = 0, apb = apbbus->slvs;
			     j < apbbus->slv_cnt; j++, apb++) {
				if ((vendor == apb->vendor) &&
				    (device == apb->device) &&
				    (ver == apb->ver) &&
				    ((apb->irq[0] & 0x80) == 0)) {
					/* AHB Slave Device matching and not
					 * marked taken.
					 */
					apb->irq[0] |= 0x80;
					core->apbslv = apb;
					break;
				}
			}
		}
	}

	core->index = ambapp_core_cnt++;
	return core;
}

void ambapp_cores_init(struct ambapp_bus *ainfo)
{
	int i, j;
	ambapp_core *core, **pcore;
	struct ahb_bus *ahbbus;
	struct apb_bus *apbbus;
	ambapp_ahbdev *ahb;
	ambapp_apbdev *apb;

	/* Ee have all device information in RAM, next step is to pair
	 * AMBA devices together into AMBA "cores". Each core constist of
	 * a combination of max:
	 *   - one AHB Master
	 *   - one AHB Slave
	 *   - one APB Slave
	 *
	 * The devices are paired together by looking at the
	 * VENDOR:DEVICE:VERSION of devices. Each core is given an Index in
	 * which order they are found, starting at zero.
	 * To a avoid creating different cores associated with the same AMBA
	 * device each assigned device is marked as "taken" temporarily
	 * by setting bit 7 in IRQ.
	 */
	ambapp_core_cnt = 0;
	pcore = &ainfo->cores;

	/* 1. Find all AMBA cores with an AMBA AHB Master (an possibly an
	 *    AHB Slave and/or APB Slave)
	 */
	for (i = 0, ahbbus = ainfo->ahbs; i < ainfo->ahb_buses; i++, ahbbus++) {
		for (j = 0, ahb = ahbbus->msts;
		     j < ahbbus->mst_cnt; j++, ahb++) {
			core = ambapp_core_create(
				ainfo,
				(ambapp_dev *)ahb,
				0x6); /* AHB SLV and APB SLV */
			if (core == NULL)
				continue; /* device already used */
			core->ahbmst = ahb;
			if (ambapp_fixup_core(core) == 1) {
				_memset(core, 0, sizeof(core));
				continue;
			}

			*pcore = core;
			pcore = &core->next;
		}
	}

	/* 2. Find all AMBA cores with an AMBA AHB Slave (an possibly an
	 *    APB Slave) which does not have an AHB Master.
	 */
	for (i = 0, ahbbus = ainfo->ahbs; i < ainfo->ahb_buses; i++, ahbbus++) {
		for (j = 0, ahb = ahbbus->slvs;
		     j < ahbbus->slv_cnt; j++, ahb++) {
			core = ambapp_core_create(
				ainfo,
				(ambapp_dev *)ahb,
				0x4); /* APB SLV */
			if (core == NULL)
				continue; /* device already used */

			core->ahbslv = ahb;

			if (ambapp_fixup_core(core) == 1) {
				_memset(core, 0, sizeof(core));
				continue;
			}

			*pcore = core;
			pcore = &core->next;
		}
	}

	/* 3. Find all AMBA cores with only an AMBA APB Slave device */
	for (i = 0, apbbus = ainfo->apbs; i < ainfo->apb_buses; i++, apbbus++) {
		for (j = 0, apb = apbbus->slvs;
		     j < apbbus->slv_cnt; j++, apb++) {
			core = ambapp_core_create(
				ainfo,
				(ambapp_dev *)apb,
				0x0); /* Nothing */
			if (core == NULL)
				continue; /* device already used */

			core->apbslv = apb;

			if (ambapp_fixup_core(core) == 1) {
				_memset(core, 0, sizeof(core));
				continue;
			}

			*pcore = core;
			pcore = &core->next;
		}
	}

	*pcore = NULL;
	ainfo->core_cnt = ambapp_core_cnt;
}

struct ambapp_bus ambapp_plb;

/* Initialize AMBA Structures but not bus frequency, pair devices into cores */
void ambapp_init(void)
{
	ambapp_bus_init(&ambapp_plb, CONFIG_AMBA_IO_AREA);

	ambapp_cores_init(&ambapp_plb);
}

void ambapp_drivers_init(void)
{
	struct ambapp_bus *ainfo = &ambapp_plb;
	struct amba_driver *drv;
	int i, lvl;
	ambapp_core *core, **ppcore;
	ambapp_dev *dev;

	/* match cores with drivers */
	core = ainfo->cores;
	while (core) {
		/* Find a device so that we get a VENDOR:DEVICE:VERSION */
		dev = ambapp_get_first_dev(core);

		for (i = 0, drv = amba_drivers;
		     i < AMBAPP_DRV_NUM; i++, drv++) {
			if ((drv->device == dev->device) &&
			    drv->init &&
			    (drv->vendor == dev->vendor) &&
			    ((dev->ver >= drv->ver_min) &&
			    (dev->ver <= drv->ver_max))) {
				/* Add core to end of list */
				ppcore = &drv->cores;
				while (*ppcore)
					ppcore = &(*ppcore)->next_in_drv;
				*ppcore = core;
				core->next_in_drv = NULL;
				break;
			}
		}
		core = core->next;
	}

	/* All initialization functions 5 times */
	for (lvl = 0; lvl < 5; lvl++) {
		for (i = 0, drv = amba_drivers;
		     i < AMBAPP_DRV_NUM; i++, drv++) {
			core = drv->cores;
			while (core) {
				drv->init(core, lvl);
				core = core->next_in_drv;
			}
		}
	}
}

static unsigned int ambapp_freq_calc(int dir, int ffact, unsigned int freq)
{
	if (dir == 0)
		return int_mul(freq, ffact);
	else
		return int_div(freq, ffact);
}

static void ambapp_subbus_freq_calc(struct ahb_bus *bus)
{
	struct ambapp_bus *ainfo = &ambapp_plb;
	struct ahb_bus *subbus;
	int i;

	for (i = 0; i < ainfo->ahb_buses; i++) {
		subbus = &ainfo->ahbs[i];
		if (subbus == bus)
			continue;
		if (subbus->parent_bus_idx == bus->bus_idx) {
			/* Subbus is slave */
			if (subbus->freq == 0) {
				/* freq already calculated */
				subbus->freq = ambapp_freq_calc(
							subbus->dir,
							subbus->ffact,
							bus->freq);
			}
			ambapp_subbus_freq_calc(subbus);
		}
	}
}

/* Set the frequency of an AMBA bus, used to determine all other buses
 * frequencies as well.
 *
 * 1. Determine the frequency of the root bus
 * 2. Iterate over all buses and determine their frequency relative
 *    to the parent bus
 */
void ambapp_freq_init(int ahb_bus_idx, unsigned long freq)
{
	struct ambapp_bus *ainfo = &ambapp_plb;
	struct ahb_bus *bus;
	int dir;

	bus = &ainfo->ahbs[ahb_bus_idx];
	bus->freq = freq;

	while (bus->parent_bus_idx != 0xff) {
		/* upstreams towards root bus means we swap order of frequency
		 * factor direction.
		 */
		dir = bus->dir;
		if (dir == 0)
			dir = 1;
		else
			dir = 0;
		freq = ambapp_freq_calc(dir, bus->ffact, freq);
		bus = &ainfo->ahbs[bus->parent_bus_idx];
		bus->freq = freq;
	}

	ambapp_subbus_freq_calc(&ainfo->ahbs[0]);
}

/* Get bus frequency for a device */
unsigned long ambapp_dev_freq(ambapp_dev *dev)
{
	int ahb_idx;

	if (dev->dev_type == DEV_APB_SLV)
		ahb_idx = ambapp_plb.apbs[dev->bus_idx].parent_bus_idx;
	else
		ahb_idx = dev->bus_idx;

	return ambapp_plb.ahbs[ahb_idx].freq;
}

#ifndef CONFIG_CUSTOM_NODES

void writehex(char *n, int v)
{
	v = v & 0xf;
	*n = v > 9 ? v - 10 + 'a' : v + '0';
}

/* Create Name Property */
int ambapp_prop_name(ambapp_core *core, struct prop_std *buf)
{
	char *str;
	const char *device, *vendor;
	ambapp_dev *dev = ambapp_get_first_dev(core);

	str = (char *)&buf->v.data[0];

	/* Find Device and Vendor String from AMBA PnP Data Base */
	device = ambapp_device_id2str(dev->vendor, dev->device);
	vendor = ambapp_vendor_id2str(dev->vendor);
	if (device && vendor) {
		_strcpy(str, (char *)vendor);
		buf->length = _strnlen(vendor, 32);
		*(str + buf->length) = '_';
		_strcpy(str + buf->length + 1, (char *)device);
		buf->length += 2 + _strnlen(device, 32);
	} else {
		writehex(&str[0], dev->vendor >> 4);
		writehex(&str[1], dev->vendor);
		str[2] = '_';
		writehex(&str[3], dev->device >> 8);
		writehex(&str[4], dev->device >> 4);
		writehex(&str[5], dev->device);
		str[6] = '\0';
		str[7] = 0;
		buf->length = 7;
	}

	buf->name = prop_str_name;
	buf->next = NULL;
	buf->options = PO_DATA | PO_NEXT;

	return sizeof(struct prop_data) + buf->length;
}

int ambapp_prop_vendor(ambapp_core *core, struct prop_std *buf)
{
	struct prop_data *dp = (struct prop_data *)buf;
	ambapp_dev *dev = ambapp_get_first_dev(core);

	dp->name = prop_str_vendor;
	dp->next = NULL;
	dp->length = 4;
	dp->options = PO_DATA | PO_NEXT;
	dp->data[0] = dev->vendor;

	return sizeof(struct prop_data) + sizeof(int);
}

int ambapp_prop_device(ambapp_core *core, struct prop_std *buf)
{
	struct prop_data *dp = (struct prop_data *)buf;
	ambapp_dev *dev = ambapp_get_first_dev(core);

	dp->name = prop_str_device;
	dp->next = NULL;
	dp->length = 4;
	dp->options = PO_DATA | PO_NEXT;
	dp->data[0] = dev->device;

	return sizeof(struct prop_data) + sizeof(int);
}

int ambapp_prop_version(ambapp_core *core, struct prop_std *buf)
{
	struct prop_data *dp = (struct prop_data *)buf;
	ambapp_dev *dev = ambapp_get_first_dev(core);

	dp->name = prop_str_version;
	dp->next = NULL;
	dp->length = 4;
	dp->options = PO_DATA | PO_NEXT;
	dp->data[0] = dev->ver;

	return sizeof(struct prop_data) + sizeof(int);
}

int ambapp_prop_devmsk(ambapp_core *core, struct prop_std *buf)
{
	struct prop_data *dp = (struct prop_data *)buf;
	unsigned int devmsk;

	dp->name = prop_str_devmsk;
	dp->next = NULL;
	dp->length = 4;
	dp->options = PO_DATA | PO_NEXT;

	devmsk = 0;
	if (core->apbslv)
		devmsk |= 0x1;
	if (core->ahbslv)
		devmsk |= 0x2;
	if (core->ahbmst)
		devmsk |= 0x4;

	dp->data[0] = devmsk;

	return sizeof(struct prop_data) + sizeof(int);
}

int ambapp_prop_interrupts_add(ambapp_dev *dev, int *buf, int index)
{
	int i;

	if (!dev)
		return index;

	for (i = 0; i < 32 && dev->irq[i] != 0 && index < 32; i++)
		buf[index++] = dev->irq[i];
	return index;
}

int ambapp_prop_interrupts(ambapp_core *core, struct prop_std *buf)
{
	int i;
	struct prop_data *dp = (struct prop_data *)buf;

	i = 0;
	i = ambapp_prop_interrupts_add((ambapp_dev *)core->apbslv, dp->data, i);
	i = ambapp_prop_interrupts_add((ambapp_dev *)core->ahbslv, dp->data, i);
	i = ambapp_prop_interrupts_add((ambapp_dev *)core->ahbmst, dp->data, i);

	if (i == 0)
		return 0; /* no such property */

	dp->name = prop_str_interrupts;
	dp->next = NULL;
	dp->options = PO_DATA | PO_NEXT;
	dp->length = i << 2;

	return sizeof(struct prop_data) + dp->length;
}

int ambapp_prop_reg(ambapp_core *core, struct prop_std *buf)
{
	int i, j;
	struct prop_data *dp = (struct prop_data *)buf;

	i = 0;
	if (core->apbslv) {
		dp->data[i++] = core->apbslv->address;
		dp->data[i++] = core->apbslv->mask;
	}
	if (core->ahbslv) {
		for (j = 0; j < 4; j++) {
			if (core->ahbslv->mask[j] > 0) {
				dp->data[i++] = core->ahbslv->address[j];
				dp->data[i++] = core->ahbslv->mask[j];
			}
		}
	}

	if (i == 0)
		return 0; /* no reg property */

	dp->name = prop_str_reg;
	dp->next = NULL;
	dp->options = PO_DATA | PO_NEXT;
	dp->length = i << 2;

	return sizeof(struct prop_data) + dp->length;
}

int ambapp_prop_userdef(ambapp_core *core, struct prop_std *buf)
{
	int i;
	struct prop_data *dp = (struct prop_data *)buf;

	i = 0;
	if (core->ahbslv) {
		dp->data[i++] = core->ahbslv->userdef[0];
		dp->data[i++] = core->ahbslv->userdef[1];
		dp->data[i++] = core->ahbslv->userdef[2];
	}
	if (core->ahbmst) {
		dp->data[i++] = core->ahbmst->userdef[0];
		dp->data[i++] = core->ahbmst->userdef[1];
		dp->data[i++] = core->ahbmst->userdef[2];
	}

	if (i == 0)
		return 0; /* no reg property */

	dp->name = prop_str_userdef;
	dp->next = NULL;
	dp->options = PO_DATA | PO_NEXT;
	dp->length = i << 2;

	return sizeof(struct prop_data) + dp->length;
}

int ambapp_prop_freq(ambapp_core *core, struct prop_std *buf)
{
	int i;
	struct prop_data *dp = (struct prop_data *)buf;

	dp->name = prop_str_freq;
	dp->next = NULL;
	dp->options = PO_DATA | PO_NEXT;
	i = 0;
	if (core->apbslv)
		dp->data[i++] = ambapp_dev_freq((ambapp_dev *)core->apbslv);
	if (core->ahbslv)
		dp->data[i++] = ambapp_dev_freq((ambapp_dev *)core->ahbslv);
	if (core->ahbmst)
		dp->data[i++] = ambapp_dev_freq((ambapp_dev *)core->ahbmst);
	dp->length = i << 2;

	return sizeof(struct prop_data) + dp->length;
}

int ambapp_prop_index(ambapp_core *core, struct prop_std *buf)
{
	struct prop_data *dp = (struct prop_data *)buf;

	dp->name = prop_str_index;
	dp->next = NULL;
	dp->options = PO_DATA | PO_NEXT;
	dp->data[0] = core->index;
	dp->length = sizeof(int);

	return sizeof(struct prop_data) + sizeof(int);
}

int ambapp_prop_ampopts(ambapp_core *core, struct prop_std *buf)
{
#ifndef CONFIG_AMPOPTS
	return 0;
#else
	struct amp_opt *a;

	a = &ampopts[0];
	while (a->idx != -1) {
		if (a->idx == core->index) {
			/* Found Matching AMP String, create property */
			struct prop_data *dp = (struct prop_data *)buf;

			dp->name = prop_str_ampopts;
			dp->next = NULL;
			dp->options = PO_DATA | PO_NEXT;
			dp->data[0] = a->val;
			dp->length = sizeof(int);

			return sizeof(struct prop_data) + sizeof(int);
		}
		a++;
	}

	/* no AMP option for this core */
	return 0;
#endif
}

/* Create Description Property... very optional :) */
int ambapp_prop_description(ambapp_core *core, struct prop_std *buf)
{
	char *str;
	const char *desc;
	ambapp_dev *dev = ambapp_get_first_dev(core);

	str = (char *)&buf->v.data[0];

	/* Find Device description from AMBA PnP Data Base */
	desc = ambapp_device_id2desc(dev->vendor, dev->device);
	if (desc == NULL)
		return 0;

	_strcpy(str, (char *)desc);
	buf->length = 1 + _strnlen(desc, 64);

	buf->name = prop_str_description;
	buf->next = NULL;
	buf->options = PO_DATA | PO_NEXT;

	return sizeof(struct prop_data) + buf->length;
}

int (*ambapp_prop_creators[])(ambapp_core *core, struct prop_std *buf) = {
	ambapp_prop_name,
	ambapp_prop_vendor,
	ambapp_prop_device,
	ambapp_prop_version,
	ambapp_prop_devmsk,
	ambapp_prop_interrupts,
	ambapp_prop_reg,
	ambapp_prop_userdef,
	ambapp_prop_freq,
	ambapp_prop_index,
	ambapp_prop_ampopts,
	ambapp_prop_description,
};
#define PROP_CREATORS_NUM (sizeof(ambapp_prop_creators)/sizeof(void (*)(void)))

#ifdef HASXML
extern struct mksel_def sel[];
extern struct mknod_def nod[];
extern int selcnt;
int
match_node(ambapp_core *core, struct mksel_def **d, struct mksel_def *e, int *p) {
	int r = 0;
	struct mksel_def *s = *d;
	ambapp_dev *a = ambapp_get_first_dev(core);
    
	while (s < e) {
		if (s->ven && s->dev) {
			if ((s->idx == -1 || s->idx == core->scanidx)
			    && s->ven == a->vendor
			    && s->dev == a->device) {
				break;
			}
		} else if (core->index == s->pidx) {
			break;
		}
		s++; if (p) *p = *p + 1;
	}
	if (s < e)
		r = s->to - s->from;
	*d = s;
	return r;
}
#endif

/* Add a AMBA Core to the end of the node tree */
struct node *ambapp_node_create(ambapp_core *core)
{
	struct node *n;
	int i, size;
	unsigned int prop_buf[32];
	struct prop_std *prop, *buf = (struct prop_std *)prop_buf;

	n = (struct node *)prom_malloc(sizeof(struct node));
	n->child = NULL;
	n->sibling = NULL;
	n->props = NULL;
	prop = NULL;

	for (i = 0; i < PROP_CREATORS_NUM; i++) {
		size = ambapp_prop_creators[i](core, buf);
		if (size == 0)
			continue;
		prop = (struct prop_std *)prom_malloc(size);
		_memcpy((void *)prop, (void *)buf, size);
		prop_add(n, (struct prop *)prop);
	}

#ifdef HASXML
	{
		/* add extra properties|nodes from -xml <xmlfile> passed properties */
		int sc, k, pit;
		struct node *cn;
		struct mksel_def *s = sel, *e = &sel[selcnt];
		while(s < e && (sc = match_node(core, &s, e, 0))) {
			if (!(s->isaddnode)) { /* add property */
				struct mknod_def *nd = nod; struct propa_ptr *p;
				for (k = s->from; k < s->to; k++) {
					p = nd[k].p;
					for (pit = 0; pit < nd[k].len; pit++) {
						prop = (struct prop_std *)prom_malloc(sizeof(struct prop_std));
						prop->options = PO_PTR | PO_NEXT;
						prop->length = p[pit].length;
						prop->name = p[pit].name;
						prop->next = NULL;
						prop->v.value = p[pit].value;
						prop_add(n, (struct prop *)prop);
					}
				}
			} else { /* add node-tree */
				struct mknod_def *nd = nod;
				for (k = s->from; k < s->to; k++) {
					cn = (struct node *)prom_calloc(sizeof(struct node));
					struct node **np = &n->child;
					while (*np)
						np = &((*np)->sibling);
					*np = cn;
					cn->child = nd[k].n.child;
					cn->props = nd[k].n.props;
				}
			}
			s++;
		}
	}
#endif
	
	if (prop)
		prop->options |= PO_END;

	return n;
}

struct node *pnodes_amba = NULL;

void ambapp_nodes_init(struct node *b0)
{
	struct node *n; int busidx = 0;
	ambapp_core *core = ambapp_plb.cores;
	
#ifdef HASXML
	ambapp_core *cn;
	/* init ambapp_core.scanidx dublicates count */
	while (core) {
		ambapp_dev *a = ambapp_get_first_dev(core);
		cn = core->next;
		while (cn) {
			ambapp_dev *b = ambapp_get_first_dev(cn);
			if (a->device == b->device &&
			    a->vendor == b->vendor) {
				cn->scanidx++;
			}
			cn = cn->next;
		}
		core = core->next;
	}
#endif
	/* ambapp0 */
	ambapp_plb.ahbsn[0] = b0;
	ambapp_plb.ahbsnn[0] = &b0->child;

	core = ambapp_plb.cores;
	while (core) {
		n = ambapp_node_create(core);

		if (ambapp_node_fixup(core, n) == 0) {

#ifndef FLATOFTREE
			static struct ambapp_bridge_driver *b;
			ambapp_dev *a = ambapp_get_first_dev(core);
		
			/* add it to the bus it belongs to */
			busidx = a->bus_idx;
			if (!ambapp_plb.ahbsn[busidx]) {
				busidx = 0;
			}
			
			switch (a->dev_type) {
			case DEV_APB_SLV:
				if (a->dev_type == DEV_APB_SLV) {
					/* use the ahb-bus-idx that the AHB-APB bridge is connected to */
					if (!ambapp_plb.ahbsn[busidx = a->extra])
						busidx = 0;
				}
				break;
			case DEV_AHB_MST:
			case DEV_AHB_SLV:
				if (core->ahbslv
				    && (!(a->vendor == VENDOR_GAISLER && a->device == GAISLER_APBMST))
				    && (b = ambapp_bridge_detect((ambapp_ahbdev *)a))) {
					/* bridges connect with their slave */
					a = (ambapp_dev *)core->ahbslv;
					busidx = a->bus_idx;
					/* dev.extra is the bridge to be child */
					if (busidx < a->extra) {
						struct node **np = &n->child;
						while (*np)
							np = &((*np)->sibling);
						*np = ambapp_plb.ahbsn[a->extra];
					}
				} 
				break;
			}
#else
			busidx = 0;
#endif

			/* Insert Node last in chain */
			*ambapp_plb.ahbsnn[busidx] = n;
			ambapp_plb.ahbsnn[busidx] = &n->sibling;
		}

		core = core->next;
	}
}
#endif
