/* GRSPW Driver Access Help Library
 *
 * Copyright (c) 2016 Cobham Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#ifndef __GRSPWLIB_H__
#define __GRSPWLIB_H__

/* interface definitions and structures */
#include "spwlib.h"

struct grspw_dev {
	int fd;
	int index;
	struct grspw_hw_sup hwsup;

	/* Pointer to Array that is used by reclaim(), recv() to translate
	 * packet ID into packet structure. The user is responsible to intialize
	 * it before assigning it.
	 */
	struct spwlib_pkt **pktid2pkt;
};

/* Open GRSPW device by index, and get hardware description of GRSPW
 * device by calling grspw_hwsup_get(&dev->hwsup).
 *
 * Return 0 on success, -1 on failure.
 */
int grspw_open(struct grspw_dev *dev, int idx);

void grspw_close(struct grspw_dev *dev);

void grspw_assign_id2pkt(struct grspw_dev *dev, struct spwlib_pkt **tab);

int grspw_hwsup_get(struct grspw_dev *dev, struct grspw_hw_sup *hw);
int grspw_bufcfg_set(struct grspw_dev *dev, struct grspw_bufcfg *bufcfg);
int grspw_cfg_get(struct grspw_dev *dev, struct grspw_config *cfg);
int grspw_cfg_set(struct grspw_dev *dev, struct grspw_config *cfg);
void grspw_cfg_print(struct grspw_hw_sup *hw, struct grspw_config *cfg);
int grspw_stats_get(struct grspw_dev *dev, struct grspw_stats *stats);
int grspw_stats_clear(struct grspw_dev *dev);
void grspw_stats_print(struct grspw_dev *dev, struct grspw_stats *stats);
int grspw_qpktcnt_get(struct grspw_dev *dev, struct grspw_qpktcnt *qpktcnt);
int grspw_linkstate_get(struct grspw_dev *dev, struct grspw_link_state *ls);
void grspw_linkstate_print(struct grspw_link_state *ls);
int grspw_link_set(struct grspw_dev *dev, struct grspw_link_ctrl *ctrl);
/* 0=Port0, 1=Port1, Other=HW Select */
int grspw_port_set(struct grspw_dev *dev, int portctrl);
int grspw_start(struct grspw_dev *dev);
int grspw_stop(struct grspw_dev *dev);

/* Return GRSPW current value of the status register */
unsigned int grspw_link_status(struct grspw_dev *dev);

/* Clear one/more bits in GRSPW STATUS register */
int grspw_link_status_clr(struct grspw_dev *dev, unsigned int clearmask);

/* Set TCTRL and TIMECNT, and Send Time-code (Generate Tick-In).
 * Returns 0 on success and negative on failure.
 */
int grspw_tc_set(struct grspw_dev *dev, unsigned int options);

/* Read TCTRL and TIMECNT from GRSPW time register.
 * Returns 0 on success and negative on failure.
 */
int grspw_tc_read(struct grspw_dev *dev, int *time);

/* Return GRSPW current value of the status register.
 * Returns 0 on success and negative on failure.
 */
int grspw_status_read(struct grspw_dev *dev, unsigned int *status);

/* Put Packet Chain 'lst' into the GRSPW driver's SEND List. The packets on
 * the SEND list will at some point make it into a descriptor. What triggers
 * packets to be scheduled are:
 *  - calls to grspw_tx_send()
 *  - calls to grspw_tx_reclaim()
 *  - TX/RX DMA IRQ (Can be controlled from configure structure)
 *
 * dma_chan determines which DMA channel packet will be sent upon. Often only
 * one DMA Channel is available, then set the dma_chan to 0.
 *
 * Return Values:
 *  * positive   = Number of packets sent (lst list updated containing only
 *                 unsent packets). If all packets has been sent 'lst' does
 *                 not contain any packets.
 *  * -1         = Some error occured.
 */
int grspw_tx_send(struct grspw_dev *dev, int dma_chan, struct spwlib_list *lst);

/* Get as many sent/error TX Packets previously shceudled for transmission with
 * grspw_tx_send(). The reclaimed packet buffers will be put in the Packet Chain
 * pointed to be 'lst'. The reclaimed packets are taken from the GRSPW driver's
 * SENT List.
 *
 * dma_chan_mask is a bit mask indicating which DMA channels sent packet buffers
 * are reclaimed from. Often only one DMA Channels is available, then set the
 * bit-mask to 1.
 *
 * Return Values:
 *  * positive   = Number of packets reclaimed (lst list updated containing
 *                 reclaimed packets). If no packets has been sent 'lst' does
 *                 not contain any packets.
 *  * -1         = Some error occured.
 */
int grspw_tx_reclaim(struct grspw_dev *dev, int dma_chan_mask, struct spwlib_list *lst);

int grspw_rx_prepare(struct grspw_dev *dev, int dma_chan, struct spwlib_list *lst);

int grspw_rx_recv(struct grspw_dev *dev, int dma_chan_mask, struct spwlib_list *lst);

int grspw_tx_wait(struct grspw_dev *dev, int dma_chan, int send_cnt, int op, int sent_cnt, long timeout_ms);

int grspw_rx_wait(struct grspw_dev *dev, int dma_chan, int recv_cnt, int op, int ready_cnt, long timeout_ms);

#endif
