/*
 * Cobham Gaisler GRSPW2 SpaceWire Kernel Library Interface.
 *
 * Copyright (c) 2016-2022 Cobham Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <asm/uaccess.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/mman.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/workqueue.h>

#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/dma-mapping.h>

#ifdef CONFIG_SPARC_LEON
#include <asm/leon.h>
#endif
#include <linux/grlib/grspw.h>
#include <linux/version.h>

#define SPIN_DECLARE(name) spinlock_t name
#define SPIN_INIT(lock) spin_lock_init(lock)
#define SPIN_LOCK(lock, level) spin_lock(lock)
#define SPIN_LOCK_IRQ(lock, level) spin_lock_irqsave(lock, level)
#define SPIN_UNLOCK(lock, level) spin_unlock(lock)
#define SPIN_UNLOCK_IRQ(lock, level) spin_unlock_irqrestore(lock, level)
#define IRQFLAGS_TYPE unsigned long

/*#define STATIC*/
#define STATIC static

#ifndef init_MUTEX
#define init_MUTEX(a) sema_init(a, 1);
#endif
#ifndef init_MUTEX_LOCKED
#define init_MUTEX_LOCKED(a) sema_init(a, 0);
#endif

struct grspw_dma_regs {
	volatile u32 ctrl;	/* DMA Channel Control */
	volatile u32 rxmax;	/* RX Max Packet Length */
	volatile u32 txdesc;	/* TX Descriptor Base/Current */
	volatile u32 rxdesc;	/* RX Descriptor Base/Current */
	volatile u32 addr;	/* Address Register */
	volatile u32 resv[3];
};

struct grspw_regs {
	volatile u32 ctrl;
	volatile u32 status;
	volatile u32 nodeaddr;
	volatile u32 clkdiv;
	volatile u32 destkey;
	volatile u32 time;
	volatile u32 timer;	/* Used only in GRSPW1 */
	volatile u32 resv1;

	/* DMA Registers, ctrl.NCH determines number of ports, 
	 * up to 4 channels are supported
	 */
	struct grspw_dma_regs dma[4];

	volatile u32 icctrl;
	volatile u32 icrx;
	volatile u32 icack;
	volatile u32 ictimeout;
	volatile u32 ictickomask;
	volatile u32 icaamask;
	volatile u32 icrlpresc;
	volatile u32 icrlisr;
	volatile u32 icrlintack;
	volatile u32 resv2;
	volatile u32 icisr;
	volatile u32 resv3;
};

/* GRSPW - Control Register - 0x00 */
#define GRSPW_CTRL_RA_BIT	31
#define GRSPW_CTRL_RX_BIT	30
#define GRSPW_CTRL_RC_BIT	29
#define GRSPW_CTRL_NCH_BIT	27
#define GRSPW_CTRL_PO_BIT	26
#define GRSPW_CTRL_PS_BIT	21
#define GRSPW_CTRL_NP_BIT	20
#define GRSPW_CTRL_RD_BIT	17
#define GRSPW_CTRL_RE_BIT	16
#define GRSPW_CTRL_TR_BIT	11
#define GRSPW_CTRL_TT_BIT	10
#define GRSPW_CTRL_LI_BIT	9
#define GRSPW_CTRL_TQ_BIT	8
#define GRSPW_CTRL_RS_BIT	6
#define GRSPW_CTRL_PM_BIT	5
#define GRSPW_CTRL_TI_BIT	4
#define GRSPW_CTRL_IE_BIT	3
#define GRSPW_CTRL_AS_BIT	2
#define GRSPW_CTRL_LS_BIT	1
#define GRSPW_CTRL_LD_BIT	0

#define GRSPW_CTRL_RA	(1<<GRSPW_CTRL_RA_BIT)
#define GRSPW_CTRL_RX	(1<<GRSPW_CTRL_RX_BIT)
#define GRSPW_CTRL_RC	(1<<GRSPW_CTRL_RC_BIT)
#define GRSPW_CTRL_NCH	(0x3<<GRSPW_CTRL_NCH_BIT)
#define GRSPW_CTRL_PO	(1<<GRSPW_CTRL_PO_BIT)
#define GRSPW_CTRL_PS	(1<<GRSPW_CTRL_PS_BIT)
#define GRSPW_CTRL_NP	(1<<GRSPW_CTRL_NP_BIT)
#define GRSPW_CTRL_RD	(1<<GRSPW_CTRL_RD_BIT)
#define GRSPW_CTRL_RE	(1<<GRSPW_CTRL_RE_BIT)
#define GRSPW_CTRL_TR	(1<<GRSPW_CTRL_TR_BIT)
#define GRSPW_CTRL_TT	(1<<GRSPW_CTRL_TT_BIT)
#define GRSPW_CTRL_LI	(1<<GRSPW_CTRL_LI_BIT)
#define GRSPW_CTRL_TQ	(1<<GRSPW_CTRL_TQ_BIT)
#define GRSPW_CTRL_RS	(1<<GRSPW_CTRL_RS_BIT)
#define GRSPW_CTRL_PM	(1<<GRSPW_CTRL_PM_BIT)
#define GRSPW_CTRL_TI	(1<<GRSPW_CTRL_TI_BIT)
#define GRSPW_CTRL_IE	(1<<GRSPW_CTRL_IE_BIT)
#define GRSPW_CTRL_AS	(1<<GRSPW_CTRL_AS_BIT)
#define GRSPW_CTRL_LS	(1<<GRSPW_CTRL_LS_BIT)
#define GRSPW_CTRL_LD	(1<<GRSPW_CTRL_LD_BIT)

/* GRSPW - Status Register - 0x04 */
#define GRSPW_STS_LS_BIT	21
#define GRSPW_STS_AP_BIT	9
#define GRSPW_STS_EE_BIT	8
#define GRSPW_STS_IA_BIT	7
#define GRSPW_STS_WE_BIT	6	/* GRSPW1 */
#define GRSPW_STS_PE_BIT	4
#define GRSPW_STS_DE_BIT	3
#define GRSPW_STS_ER_BIT	2
#define GRSPW_STS_CE_BIT	1
#define GRSPW_STS_TO_BIT	0

#define GRSPW_STS_LS	(0x7<<GRSPW_STS_LS_BIT)
#define GRSPW_STS_AP	(1<<GRSPW_STS_AP_BIT)
#define GRSPW_STS_EE	(1<<GRSPW_STS_EE_BIT)
#define GRSPW_STS_IA	(1<<GRSPW_STS_IA_BIT)
#define GRSPW_STS_WE	(1<<GRSPW_STS_WE_BIT)	/* GRSPW1 */
#define GRSPW_STS_PE	(1<<GRSPW_STS_PE_BIT)
#define GRSPW_STS_DE	(1<<GRSPW_STS_DE_BIT)
#define GRSPW_STS_ER	(1<<GRSPW_STS_ER_BIT)
#define GRSPW_STS_CE	(1<<GRSPW_STS_CE_BIT)
#define GRSPW_STS_TO	(1<<GRSPW_STS_TO_BIT)

/* GRSPW - Default Address Register - 0x08 */
#define GRSPW_DEF_ADDR_BIT	0
#define GRSPW_DEF_MASK_BIT	8
#define GRSPW_DEF_ADDR	(0xff<<GRSPW_DEF_ADDR_BIT)
#define GRSPW_DEF_MASK	(0xff<<GRSPW_DEF_MASK_BIT)

/* GRSPW - Clock Divisor Register - 0x0C */
#define GRSPW_CLKDIV_START_BIT	8
#define GRSPW_CLKDIV_RUN_BIT	0
#define GRSPW_CLKDIV_START	(0xff<<GRSPW_CLKDIV_START_BIT)
#define GRSPW_CLKDIV_RUN	(0xff<<GRSPW_CLKDIV_RUN_BIT)
#define GRSPW_CLKDIV_MASK	(GRSPW_CLKDIV_START|GRSPW_CLKDIV_RUN)

/* GRSPW - Destination key Register - 0x10 */
#define GRSPW_DK_DESTKEY_BIT	0
#define GRSPW_DK_DESTKEY	(0xff<<GRSPW_DK_DESTKEY_BIT)

/* GRSPW - Time Register - 0x14 */
#define GRSPW_TIME_CTRL_BIT	6
#define GRSPW_TIME_CNT_BIT	0
#define GRSPW_TIME_CTRL		(0x3<<GRSPW_TIME_CTRL_BIT)
#define GRSPW_TIME_TCNT		(0x3f<<GRSPW_TIME_CNT_BIT)

/* GRSPW - DMA Control Register - 0x20*N */
#define GRSPW_DMACTRL_LE_BIT	16
#define GRSPW_DMACTRL_SP_BIT	15
#define GRSPW_DMACTRL_SA_BIT	14
#define GRSPW_DMACTRL_EN_BIT	13
#define GRSPW_DMACTRL_NS_BIT	12
#define GRSPW_DMACTRL_RD_BIT	11
#define GRSPW_DMACTRL_RX_BIT	10
#define GRSPW_DMACTRL_AT_BIT	9
#define GRSPW_DMACTRL_RA_BIT	8
#define GRSPW_DMACTRL_TA_BIT	7
#define GRSPW_DMACTRL_PR_BIT	6
#define GRSPW_DMACTRL_PS_BIT	5
#define GRSPW_DMACTRL_AI_BIT	4
#define GRSPW_DMACTRL_RI_BIT	3
#define GRSPW_DMACTRL_TI_BIT	2
#define GRSPW_DMACTRL_RE_BIT	1
#define GRSPW_DMACTRL_TE_BIT	0

#define GRSPW_DMACTRL_LE	(1<<GRSPW_DMACTRL_LE_BIT)
#define GRSPW_DMACTRL_SP	(1<<GRSPW_DMACTRL_SP_BIT)
#define GRSPW_DMACTRL_SA	(1<<GRSPW_DMACTRL_SA_BIT)
#define GRSPW_DMACTRL_EN	(1<<GRSPW_DMACTRL_EN_BIT)
#define GRSPW_DMACTRL_NS	(1<<GRSPW_DMACTRL_NS_BIT)
#define GRSPW_DMACTRL_RD	(1<<GRSPW_DMACTRL_RD_BIT)
#define GRSPW_DMACTRL_RX	(1<<GRSPW_DMACTRL_RX_BIT)
#define GRSPW_DMACTRL_AT	(1<<GRSPW_DMACTRL_AT_BIT)
#define GRSPW_DMACTRL_RA	(1<<GRSPW_DMACTRL_RA_BIT)
#define GRSPW_DMACTRL_TA	(1<<GRSPW_DMACTRL_TA_BIT)
#define GRSPW_DMACTRL_PR	(1<<GRSPW_DMACTRL_PR_BIT)
#define GRSPW_DMACTRL_PS	(1<<GRSPW_DMACTRL_PS_BIT)
#define GRSPW_DMACTRL_AI	(1<<GRSPW_DMACTRL_AI_BIT)
#define GRSPW_DMACTRL_RI	(1<<GRSPW_DMACTRL_RI_BIT)
#define GRSPW_DMACTRL_TI	(1<<GRSPW_DMACTRL_TI_BIT)
#define GRSPW_DMACTRL_RE	(1<<GRSPW_DMACTRL_RE_BIT)
#define GRSPW_DMACTRL_TE	(1<<GRSPW_DMACTRL_TE_BIT)

/* GRSPW - DMA Channel Max Packet Length Register - (0x20*N + 0x04) */
#define GRSPW_DMARXLEN_MAX_BIT	0
#define GRSPW_DMARXLEN_MAX	(0xffffff<<GRSPW_DMARXLEN_MAX_BIT)

/* GRSPW - DMA Channel Address Register - (0x20*N + 0x10) */
#define GRSPW_DMAADR_ADDR_BIT	0
#define GRSPW_DMAADR_MASK_BIT	8
#define GRSPW_DMAADR_ADDR	(0xff<<GRSPW_DMAADR_ADDR_BIT)
#define GRSPW_DMAADR_MASK	(0xff<<GRSPW_DMAADR_MASK_BIT)

/* GRSPW - Interrupt code receive register - 0xa4 */
#define GRSPW_ICCTRL_INUM_BIT	27
#define GRSPW_ICCTRL_IA_BIT	24
#define GRSPW_ICCTRL_LE_BIT	23
#define GRSPW_ICCTRL_PR_BIT	22
#define GRSPW_ICCTRL_DQ_BIT	21 /* never used */
#define GRSPW_ICCTRL_TQ_BIT	20
#define GRSPW_ICCTRL_AQ_BIT	19
#define GRSPW_ICCTRL_IQ_BIT	18
#define GRSPW_ICCTRL_IR_BIT	17
#define GRSPW_ICCTRL_IT_BIT	16
#define GRSPW_ICCTRL_NUMI_BIT	13
#define GRSPW_ICCTRL_BIRQ_BIT	8
#define GRSPW_ICCTRL_ID_BIT	7
#define GRSPW_ICCTRL_II_BIT	6
#define GRSPW_ICCTRL_TXIRQ_BIT	0
#define GRSPW_ICCTRL_INUM	(0x3f << GRSPW_ICCTRL_INUM_BIT)
#define GRSPW_ICCTRL_IA		(1 << GRSPW_ICCTRL_IA_BIT)
#define GRSPW_ICCTRL_LE		(1 << GRSPW_ICCTRL_LE_BIT)
#define GRSPW_ICCTRL_PR		(1 << GRSPW_ICCTRL_PR_BIT)
#define GRSPW_ICCTRL_DQ		(1 << GRSPW_ICCTRL_DQ_BIT)
#define GRSPW_ICCTRL_TQ		(1 << GRSPW_ICCTRL_TQ_BIT)
#define GRSPW_ICCTRL_AQ		(1 << GRSPW_ICCTRL_AQ_BIT)
#define GRSPW_ICCTRL_IQ		(1 << GRSPW_ICCTRL_IQ_BIT)
#define GRSPW_ICCTRL_IR		(1 << GRSPW_ICCTRL_IR_BIT)
#define GRSPW_ICCTRL_IT		(1 << GRSPW_ICCTRL_IT_BIT)
#define GRSPW_ICCTRL_NUMI	(0x7 << GRSPW_ICCTRL_NUMI_BIT)
#define GRSPW_ICCTRL_BIRQ	(0x1f << GRSPW_ICCTRL_BIRQ_BIT)
#define GRSPW_ICCTRL_ID		(1 << GRSPW_ICCTRL_ID_BIT)
#define GRSPW_ICCTRL_II		(1 << GRSPW_ICCTRL_II_BIT)
#define GRSPW_ICCTRL_TXIRQ	(0x3f << GRSPW_ICCTRL_TXIRQ_BIT)

/* RX Buffer Descriptor */
struct grspw_rxbd {
   volatile u32 ctrl;
   volatile u32 addr;
};

/* TX Buffer Descriptor */
struct grspw_txbd {
   volatile u32 ctrl;
   volatile u32 haddr;
   volatile u32 dlen;
   volatile u32 daddr;
};

/* GRSPW - DMA RXBD Ctrl */
#define GRSPW_RXBD_LEN_BIT 0
#define GRSPW_RXBD_LEN	(0x1ffffff<<GRSPW_RXBD_LEN_BIT)
#define GRSPW_RXBD_EN	(1<<25)
#define GRSPW_RXBD_WR	(1<<26)
#define GRSPW_RXBD_IE	(1<<27)
#define GRSPW_RXBD_EP	(1<<28)
#define GRSPW_RXBD_HC	(1<<29)
#define GRSPW_RXBD_DC	(1<<30)
#define GRSPW_RXBD_TR	(1<<31)

#define GRSPW_TXBD_HLEN	(0xff<<0)
#define GRSPW_TXBD_NCL	(0xf<<8)
#define GRSPW_TXBD_EN	(1<<12)
#define GRSPW_TXBD_WR	(1<<13)
#define GRSPW_TXBD_IE	(1<<14)
#define GRSPW_TXBD_LE	(1<<15)
#define GRSPW_TXBD_HC	(1<<16)
#define GRSPW_TXBD_DC	(1<<17)

#define GRSPW_DMAADR_MASK_BIT	8
#define GRSPW_DMAADR_ADDR	(0xff<<GRSPW_DMAADR_ADDR_BIT)
#define GRSPW_DMAADR_MASK	(0xff<<GRSPW_DMAADR_MASK_BIT)


/* GRSPW Error Condition */
#define GRSPW_STAT_ERROR	(GRSPW_STS_EE | GRSPW_STS_IA | GRSPW_STS_WE | GRSPW_STS_PE | GRSPW_STS_DE | GRSPW_STS_ER | GRSPW_STS_CE)
#define GRSPW_DMA_STATUS_ERROR	(GRSPW_DMACTRL_RA | GRSPW_DMACTRL_TA)
/* GRSPW Link configuration options */
#define GRSPW_LINK_CFG		(GRSPW_CTRL_LI | GRSPW_CTRL_LD | GRSPW_CTRL_LS | GRSPW_CTRL_AS)
#define GRSPW_LINKSTATE(status)	((status & GRSPW_CTRL_LS) >> GRSPW_CTRL_LS_BIT)

/* Software Defaults */
#define DEFAULT_RXMAX 1024	/* 1 KBytes Max RX Packet Size */

/* GRSPW Constants */
#define GRSPW_TXBD_NR 64	/* Maximum number of TX Descriptors */
#define GRSPW_RXBD_NR 128	/* Maximum number of RX Descriptors */
#define GRSPW_TXBD_SIZE 16	/* Size in bytes of one TX descriptor */
#define GRSPW_RXBD_SIZE 8	/* Size in bytes of one RX descriptor */
#define BDTAB_SIZE 0x400	/* BD Table Size (RX or TX) */
#define BDTAB_ALIGN 0x400	/* BD Table Alignment Requirement */

#if (defined(CONFIG_SPARC_LEON) || defined(CONFIG_RISCV))
/* Memory and HW Registers Access routines. All 32-bit access routines */
static inline void grspw_write_bd(volatile u32 *bd, u32 val)
{
#if 0
	__raw_writel(cpu_to_be32(val), bd);
#else
	__raw_writel(val, bd);
#endif
}
static inline u32 grspw_read_bd(volatile u32 *bd)
{
#if 0
	return be32_to_cpu(__raw_readl(bd));
#else
	return __raw_readl(bd);
#endif
}
#define BD_WRITE(addr, val) (grspw_write_bd((volatile u32 *)(addr), (u32)(val)))
#define BD_READ(addr) grspw_read_bd((volatile u32 *)(addr))
#define REG_WRITE(addr, val) (*(volatile u32 *)(addr) = (u32)(val))
#define REG_READ(addr) (*(volatile u32 *)(addr))
#else
#error UNSUPPORTED ARCHITECTURE
#endif

struct grspw_ring {
	struct grspw_ring *next;	/* Next Descriptor */
	union {
		struct grspw_txbd *tx;	/* Descriptor Address */
		struct grspw_rxbd *rx;	/* Descriptor Address */
	} bd;
	struct grspw_pkt *pkt;		/* Packet description associated.NULL if none*/	
};

/* An entry in the TX descriptor Ring */
struct grspw_txring {
	struct grspw_txring *next;	/* Next Descriptor */
	struct grspw_txbd *bd;		/* Descriptor Address */
	struct grspw_pkt *pkt;		/* Packet description associated.NULL if none*/
};

/* An entry in the RX descriptor Ring */
struct grspw_rxring {
	struct grspw_rxring *next;	/* Next Descriptor */
	struct grspw_rxbd *bd;		/* Descriptor Address */
	struct grspw_pkt *pkt;		/* Packet description associated.NULL if none*/
};


struct grspw_dma_priv {
	struct grspw_priv *core;	/* GRSPW Core */
	struct grspw_dma_regs __iomem *regs;	/* DMA Channel Registers */
	int index;			/* DMA Channel Index @ GRSPW core */
	int open;			/* DMA Channel opened by user */
	int started;			/* DMA Channel activity (start|stop) */
	struct semaphore sem_rxdma;	/* DMA Channel RX Semaphore */
	struct semaphore sem_txdma;	/* DMA Channel TX Semaphore */
	struct grspw_dma_stats stats;	/* DMA Channel Statistics */
	struct grspw_dma_config cfg;	/* DMA Channel Configuration */

	/* Scheduled Work on the Work queue for this particular DMA
	 * channel.
	 */
	struct work_struct work;


	/*** RX ***/

	/* RX Descriptor Ring */
	struct grspw_rxbd *rx_bds;		/* Descriptor Address */
	dma_addr_t rx_bds_pa;			/* Descriptor Physical Address */
	struct grspw_rxring *rx_ring_base;
	struct grspw_rxring *rx_ring_head;	/* Next descriptor to enable */
	struct grspw_rxring *rx_ring_tail;	/* Oldest enabled Descriptor */
	int rx_irq_en_cnt_curr;
	struct {
		int waiting;
		int ready_cnt;
		int op;
		int recv_cnt;
		struct semaphore sem_wait;	/* RX Semaphore used to implement RX blocking */
	} rx_wait;

	/* Queue of Packets READY to be scheduled */
	struct grspw_list ready;
	int ready_cnt;

	/* Scheduled RX Packets Queue */
	struct grspw_list rx_sched;
	int rx_sched_cnt;

	/* Queue of Packets that has been RECIEVED */
	struct grspw_list recv;
	int recv_cnt;


	/*** TX ***/

	/* TX Descriptor Ring */
	struct grspw_txbd *tx_bds;		/* Descriptor Address */
	dma_addr_t tx_bds_pa;			/* Descriptor Physical Address */
	struct grspw_txring *tx_ring_base;
	struct grspw_txring *tx_ring_head;
	struct grspw_txring *tx_ring_tail;
	int tx_irq_en_cnt_curr;
	struct {
		int waiting;
		int send_cnt;
		int op;
		int sent_cnt;
		struct semaphore sem_wait;	/* TX Semaphore used to implement TX blocking */
	} tx_wait;

	/* Queue of Packets ready to be scheduled for transmission */
	struct grspw_list send;
	int send_cnt;

	/* Scheduled TX Packets Queue */
	struct grspw_list tx_sched;
	int tx_sched_cnt;

	/* Queue of Packets that has been SENT */
	struct grspw_list sent;
	int sent_cnt;
};

struct grspw_priv {
	char devname[8];		/* Device name "grspw%d" */
	struct device *dev;		/* Linux platform device */
	struct grspw_regs *regs;	/* Virtual Address of APB Registers */
	int irq;			/* AMBA IRQ number of core */
	int index;			/* Index in order it was probed */
	int core_index;			/* Core Bus Index */
	int open;			/* If Device is alrady opened (=1) or not (=0) */
	void *data;			/* User private Data for this device instance, set by grspw_initialize_user */

	/* Features supported by Hardware */
	struct grspw_hw_sup hwsup;

	/* Pointer to an array of Maximally 4 DMA Channels */
	struct grspw_dma_priv *dma;

	/* Work global for the GRSPW core */
	struct work_struct work;

	/* Spin-lock ISR protection */
	SPIN_DECLARE(devlock);

	/* Descriptor Memory Area for TX & RX and all DMA channels */
	void *bd_mem;
	dma_addr_t bd_mem_pa;
	unsigned int bdtabsize;

	/*** Time Code Handling ***/
	void (*tcisr)(void *data, int timecode);
	void *tcisr_arg;

	/* Bit mask representing events which shall cause link disable. */
	unsigned int dis_link_on_err;

	/* Bit mask for link status bits to clear by ISR */
	unsigned int stscfg;

	/* "Core Global" Statistics gathered, not dependent on DMA channel */
	struct grspw_core_stats stats;
};

int grspw_initialized = 0;
int grspw_count = 0;
struct workqueue_struct *grspw_workq = NULL;
struct semaphore grspw_sem;
static struct grspw_priv *priv_tab[GRSPW_MAX];

/* callback to upper layer when devices are discovered/removed */
void *(*grspw_dev_add)(int) = NULL;
void (*grspw_dev_del)(int,void*) = NULL;

STATIC void grspw_hw_stop(struct grspw_priv *priv);
STATIC void grspw_hw_dma_stop(struct grspw_dma_priv *dma);
STATIC void grspw_dma_reset(struct grspw_dma_priv *dma);

void *grspw_open(int dev_no)
{
	struct grspw_priv *priv;
	int i;
	int semrc;

	if (grspw_initialized != 1 || (dev_no >= grspw_count))
		return NULL;

	priv = priv_tab[dev_no];

	/* Take GRSPW lock */
	while ( (semrc=down_interruptible(&grspw_sem)) )
		printk(KERN_WARNING "grspw_open: down_interruptible %d\n", semrc);

	if (priv->open) {
		up(&grspw_sem);
		return NULL;
	}

	/* Take the device */
	priv->open = 1;

	/* Return GRSPW Lock */
	up(&grspw_sem);

	/* Initialize Spin-lock for GRSPW Device. This is to protect
	 * CTRL and DMACTRL registers from ISR.
	 */
	SPIN_INIT(&priv->devlock);

	priv->tcisr = NULL;
	priv->tcisr_arg = NULL;
	priv->stscfg = LINKSTS_MASK;

	grspw_stats_clr(priv);

	/* Allocate TX & RX Descriptor memory area for all DMA
	 * channels. Max-size descriptor area is allocated:
	 *  - 128 RX descriptors per DMA Channel
	 *  - 64 TX descriptors per DMA Channel
 	 */
	priv->bdtabsize = 2 * BDTAB_SIZE * priv->hwsup.ndma_chans;
	priv->bd_mem = dma_alloc_coherent(priv->dev, priv->bdtabsize,
				          &priv->bd_mem_pa,
					  GFP_KERNEL | __GFP_ZERO);
	if (!priv->bd_mem) {
		priv->open = 0;
		return NULL;
		}
	if ((long)priv->bd_mem & (BDTAB_ALIGN-1)) {
		printk(KERN_INFO "GRSPW: Descriptors not aligned to boundary: %p\n",
					priv->bd_mem);
		dma_free_coherent(priv->dev, priv->bdtabsize, priv->bd_mem,
				  priv->bd_mem_pa);
		priv->open = 0;
		return NULL;
	}

	for (i=0; i<priv->hwsup.ndma_chans; i++) {
		/* Do DMA Channel Init, other variables etc. are inited
		 * when respective DMA channel is opened.
		 *
		 * index & core are initialized by probe function.
		 */
		priv->dma[i].open = 0;
		priv->dma[i].rx_bds = (struct grspw_rxbd *)
			(priv->bd_mem + i*BDTAB_SIZE*2);
		priv->dma[i].rx_bds_pa = priv->bd_mem_pa + BDTAB_SIZE*(2*i);
		priv->dma[i].tx_bds = (struct grspw_txbd *)
			(priv->bd_mem + BDTAB_SIZE*(2*i+1));
		priv->dma[i].tx_bds_pa = priv->bd_mem_pa + BDTAB_SIZE*(2*i+1);
	}

	/* Basic initialization of hardware, clear some registers but
	 * keep Link/RMAP/Node-Address registers intact.
	 */
	grspw_hw_stop(priv);

	return priv;
}
EXPORT_SYMBOL(grspw_open);

int grspw_close(void *d)
{
	struct grspw_priv *priv = d;
	int i;
	int semrc;

	/* Take GRSPW lock */
	while ( (semrc=down_interruptible(&grspw_sem)) )
		printk(KERN_WARNING "grspw_close: down_interruptible %d\n", semrc);

	/* Check that user has stopped and closed all DMA channels
	 * appropriately. At this point the Hardware shall not be doing DMA
	 * or generating Interrupts. We want HW in a "startup-state".
	 */
	for (i=0; i<priv->hwsup.ndma_chans; i++) {
		if (priv->dma[i].open) {
			up(&grspw_sem);
			return 1;
		}
	}
	grspw_hw_stop(priv);

	dma_free_coherent(priv->dev, priv->bdtabsize, priv->bd_mem,
			  priv->bd_mem_pa);

	/* Mark not open */
	priv->open = 0;
	up(&grspw_sem);
	return 0;
}
EXPORT_SYMBOL(grspw_close);

void grspw_hw_support(void *d, struct grspw_hw_sup *hw)
{
	struct grspw_priv *priv = d;

	*hw = priv->hwsup;
}
EXPORT_SYMBOL(grspw_hw_support);

void grspw_addr_ctrl(void *d, struct grspw_addr_config *cfg)
{
	struct grspw_priv *priv = d;
	struct grspw_regs *regs = priv->regs;
	unsigned int ctrl, nodeaddr;
	IRQFLAGS_TYPE irqflags;
	int i;

	if (!priv || !cfg)
		return;

	SPIN_LOCK_IRQ(&priv->devlock, irqflags);

	if (cfg->promiscuous != -1) {
		/* Set Configuration */
		ctrl = REG_READ(&regs->ctrl);
		if (cfg->promiscuous)
			ctrl |= GRSPW_CTRL_PM;
		else
			ctrl &= ~GRSPW_CTRL_PM;
		REG_WRITE(&regs->ctrl, ctrl);
		REG_WRITE(&regs->nodeaddr, (cfg->def_mask<<8) | cfg->def_addr);

		for (i=0; i<priv->hwsup.ndma_chans; i++) {
			ctrl = REG_READ(&regs->dma[i].ctrl);
			ctrl &= ~(GRSPW_DMACTRL_PS|GRSPW_DMACTRL_PR|GRSPW_DMA_STATUS_ERROR);
			if (cfg->dma_nacfg[i].node_en) {
				ctrl |= GRSPW_DMACTRL_EN;
				REG_WRITE(&regs->dma[i].addr,
				          (cfg->dma_nacfg[i].node_addr & 0xff) |
				          ((cfg->dma_nacfg[i].node_mask & 0xff)<<8));
			} else {
				ctrl &= ~GRSPW_DMACTRL_EN;
			}
			REG_WRITE(&regs->dma[i].ctrl, ctrl);
		}
	}

	/* Read Current Configuration */
	cfg->promiscuous = REG_READ(&regs->ctrl) & GRSPW_CTRL_PM;
	nodeaddr = REG_READ(&regs->nodeaddr);
	cfg->def_addr = (nodeaddr & GRSPW_DEF_ADDR) >> GRSPW_DEF_ADDR_BIT;
	cfg->def_mask = (nodeaddr & GRSPW_DEF_MASK) >> GRSPW_DEF_MASK_BIT;
	for (i=0; i<priv->hwsup.ndma_chans; i++) {
		cfg->dma_nacfg[i].node_en = REG_READ(&regs->dma[i].ctrl) &
						GRSPW_DMACTRL_EN;
		ctrl = REG_READ(&regs->dma[i].addr);
		cfg->dma_nacfg[i].node_addr = (ctrl & GRSPW_DMAADR_ADDR) >>
						GRSPW_DMAADR_ADDR_BIT;
		cfg->dma_nacfg[i].node_mask = (ctrl & GRSPW_DMAADR_MASK) >>
						GRSPW_DMAADR_MASK_BIT;
	}
	SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
	for (; i<4; i++) {
		cfg->dma_nacfg[i].node_en = 0;
		cfg->dma_nacfg[i].node_addr = 0;
		cfg->dma_nacfg[i].node_mask = 0;
	}
}
EXPORT_SYMBOL(grspw_addr_ctrl);

/* Return Current Status Register */
unsigned int grspw_link_status(void *d)
{
	struct grspw_priv *priv = d;

	return REG_READ(&priv->regs->status);
}
EXPORT_SYMBOL(grspw_link_status);

/* Clear Status Register bits */
void grspw_link_status_clr(void *d, unsigned int mask)
{
	struct grspw_priv *priv = d;

	REG_WRITE(&priv->regs->status, mask);
}
EXPORT_SYMBOL(grspw_link_status_clr);

/* Return Current Link State */
spw_link_state_t grspw_link_state(void *d)
{
	struct grspw_priv *priv = d;
	unsigned int status = REG_READ(&priv->regs->status);

	return (status & GRSPW_STS_LS) >> GRSPW_STS_LS_BIT;
}
EXPORT_SYMBOL(grspw_link_state);

/* options [in/out]: set to -1 to only read current config */
void grspw_link_ctrl(void *d, int *options, int *stscfg, int *clkdiv)
{
	struct grspw_priv *priv = d;
	struct grspw_regs *regs = priv->regs;
	unsigned int ctrl;
	IRQFLAGS_TYPE irqflags;

	/* Write? */
	if (clkdiv) {
		if (*clkdiv != -1)
			REG_WRITE(&regs->clkdiv, *clkdiv & GRSPW_CLKDIV_MASK);
		*clkdiv = REG_READ(&regs->clkdiv) & GRSPW_CLKDIV_MASK;
	}
	if (options) {
		SPIN_LOCK_IRQ(&priv->devlock, irqflags);
		ctrl = REG_READ(&regs->ctrl);
		if (*options != -1) {
			ctrl = (ctrl & ~GRSPW_LINK_CFG) |
				(*options & GRSPW_LINK_CFG);

			/* Enable Global IRQ only of LI or TQ is set */
			if (ctrl & (GRSPW_CTRL_LI|GRSPW_CTRL_TQ))
				ctrl |= GRSPW_CTRL_IE;
			else
				ctrl &= ~GRSPW_CTRL_IE;

			REG_WRITE(&regs->ctrl, ctrl);
			/* Store the link disable events for use in
			ISR. The LINKOPTS_DIS_ON_* options are actually the
			corresponding bits in the status register, shifted
			by 16. */
			priv->dis_link_on_err = *options &
				(LINKOPTS_MASK_DIS_ON | LINKOPTS_DIS_ONERR);
		}
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
		*options = (ctrl & GRSPW_LINK_CFG) | priv->dis_link_on_err;
	}
	if (stscfg) {
		if (*stscfg != -1) {
			priv->stscfg = *stscfg & LINKSTS_MASK;
		}
		*stscfg = priv->stscfg;
	}
}
EXPORT_SYMBOL(grspw_link_ctrl);

/* Generate Tick-In (increment Time Counter, Send Time Code) */
void grspw_tc_tx(void *d)
{
	struct grspw_priv *priv = d;
	struct grspw_regs *regs = priv->regs;
	IRQFLAGS_TYPE irqflags;

	SPIN_LOCK_IRQ(&priv->devlock, irqflags);
	REG_WRITE(&regs->ctrl, REG_READ(&regs->ctrl) | GRSPW_CTRL_TI);
	SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
}
EXPORT_SYMBOL(grspw_tc_tx);

void grspw_tc_ctrl(void *d, int *options)
{
	struct grspw_priv *priv = d;
	struct grspw_regs *regs = priv->regs;
	unsigned int ctrl;
	IRQFLAGS_TYPE irqflags;

	if (options == NULL)
		return;

	/* Write? */
	if (*options != -1) {
		SPIN_LOCK_IRQ(&priv->devlock, irqflags);
		ctrl = REG_READ(&regs->ctrl);
		ctrl &= ~(GRSPW_CTRL_TR|GRSPW_CTRL_TT|GRSPW_CTRL_TQ);
		ctrl |= (*options & 0xd) << GRSPW_CTRL_TQ_BIT;

		/* Enable Global IRQ only of LI or TQ is set */
		if (ctrl & (GRSPW_CTRL_LI|GRSPW_CTRL_TQ))
			ctrl |= GRSPW_CTRL_IE;
		else
			ctrl &= ~GRSPW_CTRL_IE;

		REG_WRITE(&regs->ctrl, ctrl);
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
	} else
		ctrl = REG_READ(&regs->ctrl);
	*options = (ctrl >> GRSPW_CTRL_TQ_BIT) & 0xd;
}
EXPORT_SYMBOL(grspw_tc_ctrl);

/* Assign ISR Function to TimeCode RX IRQ */
void grspw_tc_isr(void *d, void (*tcisr)(void *data, int tc), void *data)
{
	struct grspw_priv *priv = d;

	priv->tcisr_arg = data;
	priv->tcisr = tcisr;
}
EXPORT_SYMBOL(grspw_tc_isr);

/* Read/Write TCTRL and TIMECNT. Write if not -1, always read current value
 * TCTRL   = bits 7 and 6
 * TIMECNT = bits 5 to 0
 */
void grspw_tc_time(void *d, int *time)
{
	struct grspw_priv *priv = d;
	struct grspw_regs *regs = priv->regs;

	if (time == NULL)
		return;
	if (*time != -1)
		REG_WRITE(&regs->time, *time & (GRSPW_TIME_TCNT | GRSPW_TIME_CTRL));
	*time = REG_READ(&regs->time) & (GRSPW_TIME_TCNT | GRSPW_TIME_CTRL);
}
EXPORT_SYMBOL(grspw_tc_time);

/* Set (not -1) and/or read RMAP options. */
int grspw_rmap_ctrl(void *d, int *options, int *dstkey)
{
	struct grspw_priv *priv = d;
	struct grspw_regs *regs = priv->regs;
	unsigned int ctrl;
	IRQFLAGS_TYPE irqflags;

	if (dstkey) {
		if (*dstkey != -1)
			REG_WRITE(&regs->destkey, *dstkey & GRSPW_DK_DESTKEY);
		*dstkey = REG_READ(&regs->destkey) & GRSPW_DK_DESTKEY;
	}
	if (options) {
		if (*options != -1) {
			if ((*options & RMAPOPTS_EN_RMAP) && !priv->hwsup.rmap)
				return -1;


			SPIN_LOCK_IRQ(&priv->devlock, irqflags);
			ctrl = REG_READ(&regs->ctrl);
			ctrl &= ~(GRSPW_CTRL_RE|GRSPW_CTRL_RD);
			ctrl |= (*options & 0x3) << GRSPW_CTRL_RE_BIT;
			REG_WRITE(&regs->ctrl, ctrl);
			SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
		}
		*options = (REG_READ(&regs->ctrl) >> GRSPW_CTRL_RE_BIT) & 0x3;
	}

	return 0;
}
EXPORT_SYMBOL(grspw_rmap_ctrl);

void grspw_rmap_support(void *d, char *rmap, char *rmap_crc)
{
	struct grspw_priv *priv = d;

	if (rmap)
		*rmap = priv->hwsup.rmap;
	if (rmap_crc)
		*rmap_crc = priv->hwsup.rmap_crc;
}
EXPORT_SYMBOL(grspw_rmap_support);

/* Select port, if 
 * -1=The current selected port is returned
 * 0=Port 0
 * 1=Port 1
 * Others=Both Port0 and Port1
 */
int grspw_port_ctrl(void *d, int *port)
{
	struct grspw_priv *priv = d;
	struct grspw_regs *regs = priv->regs;
	unsigned int ctrl;
	IRQFLAGS_TYPE irqflags;

	if (port == NULL)
		return -1;

	if ((*port == 1) || (*port == 0)) {
		/* Select port user selected */
		if ((*port == 1) && (priv->hwsup.nports < 2))
			return -1; /* Changing to Port 1, but only one port available */
		SPIN_LOCK_IRQ(&priv->devlock, irqflags);
		ctrl = REG_READ(&regs->ctrl);
		ctrl &= ~(GRSPW_CTRL_NP | GRSPW_CTRL_PS);
		ctrl |= (*port & 1) << GRSPW_CTRL_PS_BIT;
		REG_WRITE(&regs->ctrl, ctrl);
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
	} else if (*port > 1) {
		/* Select both ports */
		SPIN_LOCK_IRQ(&priv->devlock, irqflags);
		REG_WRITE(&regs->ctrl, REG_READ(&regs->ctrl) | GRSPW_CTRL_NP);
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
	}

	/* Get current settings */
	ctrl = REG_READ(&regs->ctrl);
	if (ctrl & GRSPW_CTRL_NP) {
		/* Any port, selected by hardware */
		if (priv->hwsup.nports > 1)
			*port = 3;
		else
			*port = 0; /* Port0 the only port available */
	} else {
		*port = (ctrl & GRSPW_CTRL_PS) >> GRSPW_CTRL_PS_BIT;
	}

	return 0;
}
EXPORT_SYMBOL(grspw_port_ctrl);

/* Returns Number ports available in hardware */
int grspw_port_count(void *d)
{
	struct grspw_priv *priv = d;

	return priv->hwsup.nports;
}
EXPORT_SYMBOL(grspw_port_count);

/* Current active port: 0 or 1 */
int grspw_port_active(void *d)
{
	struct grspw_priv *priv = d;
	unsigned int status;

	status = REG_READ(&priv->regs->status);

	return (status & GRSPW_STS_AP) >> GRSPW_STS_AP_BIT;
}
EXPORT_SYMBOL(grspw_port_active);

void grspw_stats_read(void *d, struct grspw_core_stats *sts)
{
	struct grspw_priv *priv = d;

	if (sts == NULL)
		return;
	memcpy(sts, &priv->stats, sizeof(priv->stats));
}
EXPORT_SYMBOL(grspw_stats_read);

void grspw_stats_clr(void *d)
{
	struct grspw_priv *priv = d;

	/* Clear most of the statistics */	
	memset(&priv->stats, 0, sizeof(priv->stats));
}
EXPORT_SYMBOL(grspw_stats_clr);

/*** DMA Interface ***/

/* Initialize the RX and TX Descriptor Ring, empty of packets */
STATIC void grspw_bdrings_init(struct grspw_dma_priv *dma)
{
	struct grspw_ring *r;
	int i;

	/* Empty BD rings */
	dma->rx_ring_head = dma->rx_ring_base;
	dma->rx_ring_tail = dma->rx_ring_base;
	dma->tx_ring_head = dma->tx_ring_base;
	dma->tx_ring_tail = dma->tx_ring_base;

	/* Init RX Descriptors */
	r = (struct grspw_ring *)dma->rx_ring_base;
	for (i=0; i<GRSPW_RXBD_NR; i++) {

		/* Init Ring Entry */
		r[i].next = &r[i+1];
		r[i].bd.rx = &dma->rx_bds[i];
		r[i].pkt = NULL;

		/* Init HW Descriptor */
		BD_WRITE(&r[i].bd.rx->ctrl, 0);
		BD_WRITE(&r[i].bd.rx->addr, 0);
	}
	r[GRSPW_RXBD_NR-1].next = &r[0];

	/* Init TX Descriptors */
	r = (struct grspw_ring *)dma->tx_ring_base;
	for (i=0; i<GRSPW_TXBD_NR; i++) {

		/* Init Ring Entry */
		r[i].next = &r[i+1];
		r[i].bd.tx = &dma->tx_bds[i];
		r[i].pkt = NULL;

		/* Init HW Descriptor */
		BD_WRITE(&r[i].bd.tx->ctrl, 0);
		BD_WRITE(&r[i].bd.tx->haddr, 0);
		BD_WRITE(&r[i].bd.tx->dlen, 0);
		BD_WRITE(&r[i].bd.tx->daddr, 0);
	}
	r[GRSPW_TXBD_NR-1].next = &r[0];
}

/* Try to populate descriptor ring with as many as possible READY unused packet
 * buffers. The packets assigned with to a descriptor are put in the end of 
 * the scheduled list.
 *
 * The number of Packets scheduled is returned.
 *
 *  - READY List -> RX-SCHED List
 *  - Descriptors are initialized and enabled for reception
 */
STATIC int grspw_rx_schedule_ready(struct grspw_dma_priv *dma)
{
	int cnt;
	unsigned int ctrl, dmactrl;
	struct grspw_rxring *curr_bd;
	struct grspw_pkt *curr_pkt, *last_pkt;
	struct grspw_list lst;
	IRQFLAGS_TYPE irqflags;

	/* Is Ready Q empty? */
	if (grspw_list_is_empty(&dma->ready))
		return 0;

	cnt = 0;
	lst.head = curr_pkt = dma->ready.head;
	curr_bd = dma->rx_ring_head;
	while (!curr_bd->pkt) {

		/* Assign Packet to descriptor */
		curr_bd->pkt = curr_pkt;

		/* Prepare descriptor address. */
		BD_WRITE(&curr_bd->bd->addr, curr_pkt->data);

		ctrl = GRSPW_RXBD_EN;
		if (curr_bd->next == dma->rx_ring_base) {
			/* Wrap around (only needed when smaller descriptor
			 * table)
			 */
			ctrl |= GRSPW_RXBD_WR;
		}

		/* Is this Packet going to be an interrupt Packet? */
		if ((--dma->rx_irq_en_cnt_curr) <= 0) {
			if (dma->cfg.rx_irq_en_cnt == 0) {
				/* IRQ is disabled. A big number to avoid
				 * equal to zero too often
				 */
				dma->rx_irq_en_cnt_curr = 0x3fffffff;
			} else {
				dma->rx_irq_en_cnt_curr = dma->cfg.rx_irq_en_cnt;
				ctrl |= GRSPW_RXBD_IE;
			}
		}

		if (curr_pkt->flags & RXPKT_FLAG_IE)
			ctrl |= GRSPW_RXBD_IE;

		/* Enable descriptor */
		BD_WRITE(&curr_bd->bd->ctrl, ctrl);

		last_pkt = curr_pkt;
		curr_bd = curr_bd->next;
		cnt++;

		/* Get Next Packet from Ready Queue */
		if (curr_pkt == dma->ready.tail) {
			/* Handled all in ready queue. */
			curr_pkt = NULL;
			break;
		}
		curr_pkt = curr_pkt->next;
	}

	/* Has Packets been scheduled? */
	if (cnt > 0) {
		/* Prepare list for insertion/deleation */
		lst.tail = last_pkt;

		/* Remove scheduled packets from ready queue */
		grspw_list_remove_head_list(&dma->ready, &lst);
		dma->ready_cnt -= cnt;
		if (dma->stats.ready_cnt_min > dma->ready_cnt)
			dma->stats.ready_cnt_min = dma->ready_cnt;

		/* Insert scheduled packets into scheduled queue */
		grspw_list_append_list(&dma->rx_sched, &lst);
		dma->rx_sched_cnt += cnt;
		if (dma->stats.rx_sched_cnt_max < dma->rx_sched_cnt)
			dma->stats.rx_sched_cnt_max = dma->rx_sched_cnt;

		/* Update TX ring posistion */
		dma->rx_ring_head = curr_bd;

		/* Make hardware aware of the newly enabled descriptors 
		 * We must protect from ISR which writes RI|TI
		 */
		SPIN_LOCK_IRQ(&dma->core->devlock, irqflags);
		dmactrl = REG_READ(&dma->regs->ctrl);
		dmactrl &= ~(GRSPW_DMACTRL_PS|GRSPW_DMACTRL_PR|GRSPW_DMA_STATUS_ERROR);
		dmactrl |= GRSPW_DMACTRL_RE | GRSPW_DMACTRL_RD;
		REG_WRITE(&dma->regs->ctrl, dmactrl);
		SPIN_UNLOCK_IRQ(&dma->core->devlock, irqflags);
	}

	return cnt;
}

/* Scans the RX desciptor table for scheduled Packet that has been received,
 * and moves these Packet from the head of the scheduled queue to the
 * tail of the recv queue.
 *
 * Also, for all packets the status is updated.
 *
 *  - SCHED List -> SENT List
 *
 * Return Value
 * Number of packets moved
 */
STATIC int grspw_rx_process_scheduled(struct grspw_dma_priv *dma)
{
	struct grspw_rxring *curr;
	struct grspw_pkt *last_pkt;
	int recv_pkt_cnt = 0;
	unsigned int ctrl;
	struct grspw_list lst;

	curr = dma->rx_ring_tail;

	/* Step into RX ring to find if packets have been scheduled for 
	 * reception.
	 */
	if (!curr->pkt)
		return 0; /* No scheduled packets, thus no received, abort */

	/* There has been Packets scheduled ==> scheduled Packets may have been
	 * received and needs to be collected into RECV List.
	 *
	 * A temporary list "lst" with all received packets is created.
	 */
	lst.head = curr->pkt;

	/* Loop until first enabled "unrecveived" SpW Packet is found.
	 * An unused descriptor is indicated by an unassigned pkt field.
	 */
	while (curr->pkt && !((ctrl=BD_READ(&curr->bd->ctrl)) & GRSPW_RXBD_EN)) {
		/* Handle one received Packet */

		/* Remember last handled Packet so that insertion/removal from
		 * Packet lists go fast.
		 */
		last_pkt = curr->pkt;

		/* Get Length of Packet in bytes, and reception options */
		last_pkt->dlen = (ctrl & GRSPW_RXBD_LEN) >> GRSPW_RXBD_LEN_BIT;

		/* Set flags to indicate error(s) and CRC information,
		 * and Mark Received.
		 */
		last_pkt->flags = (last_pkt->flags & ~RXPKT_FLAG_OUTPUT_MASK) |
		                  ((ctrl >> 20) & RXPKT_FLAG_OUTPUT_MASK) |
		                  RXPKT_FLAG_RX;

		/* Packet was Truncated? */
		if (ctrl & GRSPW_RXBD_TR)
			dma->stats.rx_err_trunk++;

		/* Error End-Of-Packet? */
		if (ctrl & GRSPW_RXBD_EP)
			dma->stats.rx_err_endpkt++;
		curr->pkt = NULL; /* Mark descriptor unused */

		/* Increment */
		curr = curr->next;
		recv_pkt_cnt++;
	}

	/* 1. Remove all handled packets from scheduled queue
	 * 2. Put all handled packets into recv queue
	 */
	if (recv_pkt_cnt > 0) {

		/* Update Stats, Number of Received Packets */
		dma->stats.rx_pkts += recv_pkt_cnt;

		/* Save RX ring posistion */
		dma->rx_ring_tail = curr;

		/* Prepare list for insertion/deleation */
		lst.tail = last_pkt;

		/* Remove received Packets from RX-SCHED queue */
		grspw_list_remove_head_list(&dma->rx_sched, &lst);
		dma->rx_sched_cnt -= recv_pkt_cnt;
		if (dma->stats.rx_sched_cnt_min > dma->rx_sched_cnt)
			dma->stats.rx_sched_cnt_min = dma->rx_sched_cnt;

		/* Insert received Packets into RECV queue */
		grspw_list_append_list(&dma->recv, &lst);
		dma->recv_cnt += recv_pkt_cnt;
		if (dma->stats.recv_cnt_max < dma->recv_cnt)
			dma->stats.recv_cnt_max = dma->recv_cnt;
	}

	return recv_pkt_cnt;
}

/* Try to populate descriptor ring with as many SEND packets as possible. The
 * packets assigned with to a descriptor are put in the end of 
 * the scheduled list.
 *
 * The number of Packets scheduled is returned.
 *
 *  - SEND List -> TX-SCHED List
 *  - Descriptors are initialized and enabled for transmission
 */
STATIC int grspw_tx_schedule_send(struct grspw_dma_priv *dma)
{
	int cnt;
	unsigned int ctrl, dmactrl;
	struct grspw_txring *curr_bd;
	struct grspw_pkt *curr_pkt, *last_pkt;
	struct grspw_list lst;
	IRQFLAGS_TYPE irqflags;

	/* Is Ready Q empty? */
	if (grspw_list_is_empty(&dma->send))
		return 0;

	cnt = 0;
	lst.head = curr_pkt = dma->send.head;
	curr_bd = dma->tx_ring_head;
	while (!curr_bd->pkt) {

		/* Assign Packet to descriptor */
		curr_bd->pkt = curr_pkt;

		/* Prepare descriptor address. */
		if ( curr_pkt->data && curr_pkt->dlen ) {
			BD_WRITE(&curr_bd->bd->daddr, curr_pkt->data);
			BD_WRITE(&curr_bd->bd->dlen, curr_pkt->dlen);
		} else {
			BD_WRITE(&curr_bd->bd->daddr, 0);
			BD_WRITE(&curr_bd->bd->dlen, 0);
			}

		/* Set up header transmission */
		if (curr_pkt->hdr && curr_pkt->hlen) {
			BD_WRITE(&curr_bd->bd->haddr, curr_pkt->hdr);
			ctrl = GRSPW_TXBD_EN | curr_pkt->hlen;
		} else {
			ctrl = GRSPW_TXBD_EN;
		}
		/* Enable IRQ generation and CRC options as specified
		 * by user.
		 */
		ctrl |= (curr_pkt->flags & TXPKT_FLAG_INPUT_MASK) << 8;

		if (curr_bd->next == dma->tx_ring_base) {
			/* Wrap around (only needed when smaller descriptor table) */
			ctrl |= GRSPW_TXBD_WR;
		}

		/* Is this Packet going to be an interrupt Packet? */
		if ((--dma->tx_irq_en_cnt_curr) <= 0) {
			if (dma->cfg.tx_irq_en_cnt == 0) {
				/* IRQ is disabled.
				 * A big number to avoid equal to zero too often 
				 */
				dma->tx_irq_en_cnt_curr = 0x3fffffff;
			} else {
				dma->tx_irq_en_cnt_curr = dma->cfg.tx_irq_en_cnt;
				ctrl |= GRSPW_TXBD_IE;
			}
		}

		/* Enable descriptor */
		BD_WRITE(&curr_bd->bd->ctrl, ctrl);

		last_pkt = curr_pkt;
		curr_bd = curr_bd->next;
		cnt++;

		/* Get Next Packet from Ready Queue */
		if (curr_pkt == dma->send.tail) {
			/* Handled all in ready queue. */
			curr_pkt = NULL;
			break;
		}
		curr_pkt = curr_pkt->next;
	}

	/* Have Packets been scheduled? */
	if (cnt > 0) {
		/* Prepare list for insertion/deleation */
		lst.tail = last_pkt;

		/* Remove scheduled packets from ready queue */
		grspw_list_remove_head_list(&dma->send, &lst);
		dma->send_cnt -= cnt;
		if (dma->stats.send_cnt_min > dma->send_cnt)
			dma->stats.send_cnt_min = dma->send_cnt;

		/* Insert scheduled packets into scheduled queue */
		grspw_list_append_list(&dma->tx_sched, &lst);
		dma->tx_sched_cnt += cnt;
		if (dma->stats.tx_sched_cnt_max < dma->tx_sched_cnt)
			dma->stats.tx_sched_cnt_max = dma->tx_sched_cnt;

		/* Update TX ring posistion */
		dma->tx_ring_head = curr_bd;

		/* Make hardware aware of the newly enabled descriptors */
		SPIN_LOCK_IRQ(&dma->core->devlock, irqflags);
		dmactrl = REG_READ(&dma->regs->ctrl);
		dmactrl &= ~(GRSPW_DMACTRL_PS|GRSPW_DMACTRL_PR|GRSPW_DMA_STATUS_ERROR);
		dmactrl |= GRSPW_DMACTRL_TE;
		REG_WRITE(&dma->regs->ctrl, dmactrl);
		SPIN_UNLOCK_IRQ(&dma->core->devlock, irqflags);
	}
	return cnt;
}

/* Scans the TX desciptor table for transmitted packets, and moves these 
 * packets from the head of the scheduled queue to the tail of the sent queue.
 *
 * Also, for all packets the status is updated.
 *
 *  - SCHED List -> SENT List
 *
 * Return Value
 * Number of packet moved
 */
STATIC int grspw_tx_process_scheduled(struct grspw_dma_priv *dma)
{
	struct grspw_txring *curr;
	struct grspw_pkt *last_pkt;
	int sent_pkt_cnt = 0;
	unsigned int ctrl;
	struct grspw_list lst;

	curr = dma->tx_ring_tail;

	/* Step into TX ring to find if packets have been scheduled for 
	 * transmission.
	 */
	if (!curr->pkt)
		return 0; /* No scheduled packets, thus no sent, abort */

	/* There has been Packets scheduled ==> scheduled Packets may have been
	 * transmitted and needs to be collected into SENT List.
	 *
	 * A temporary list "lst" with all sent packets is created.
	 */
	lst.head = curr->pkt;

	/* Loop until first enabled "un-transmitted" SpW Packet is found.
	 * An unused descriptor is indicated by an unassigned pkt field.
	 */
	while (curr->pkt && !((ctrl=BD_READ(&curr->bd->ctrl)) & GRSPW_TXBD_EN)) {
		/* Handle one sent Packet */

		/* Remember last handled Packet so that insertion/removal from
		 * packet lists go fast.
		 */
		last_pkt = curr->pkt;

		/* Set flags to indicate error(s) and Mark Sent.
		 */
		last_pkt->flags = (last_pkt->flags & ~TXPKT_FLAG_OUTPUT_MASK) |
					(ctrl & TXPKT_FLAG_LINKERR) |
					TXPKT_FLAG_TX;

		/* Sent packet experienced link error? */
		if (ctrl & GRSPW_TXBD_LE)
			dma->stats.tx_err_link++;

		curr->pkt = NULL; /* Mark descriptor unused */

		/* Increment */
		curr = curr->next;
		sent_pkt_cnt++;
	}

	/* 1. Remove all handled packets from TX-SCHED queue
	 * 2. Put all handled packets into SENT queue
	 */
	if (sent_pkt_cnt > 0) {
		/* Update Stats, Number of Transmitted Packets */
		dma->stats.tx_pkts += sent_pkt_cnt;

		/* Save TX ring posistion */
		dma->tx_ring_tail = curr;

		/* Prepare list for insertion/deleation */
		lst.tail = last_pkt;

		/* Remove sent packets from TX-SCHED queue */
		grspw_list_remove_head_list(&dma->tx_sched, &lst);
		dma->tx_sched_cnt -= sent_pkt_cnt;
		if (dma->stats.tx_sched_cnt_min > dma->tx_sched_cnt)
			dma->stats.tx_sched_cnt_min = dma->tx_sched_cnt;

		/* Insert received packets into SENT queue */
		grspw_list_append_list(&dma->sent, &lst);
		dma->sent_cnt += sent_pkt_cnt;
		if (dma->stats.sent_cnt_max < dma->sent_cnt)
			dma->stats.sent_cnt_max = dma->sent_cnt;
	}

	return sent_pkt_cnt;
}

void *grspw_dma_open(void *d, int chan_no)
{
	struct grspw_priv *priv = d;
	struct grspw_dma_priv *dma;
	int size;
	int semrc;

	if ((chan_no < 0) || (priv->hwsup.ndma_chans <= chan_no))
		return NULL;

	dma = &priv->dma[chan_no];

	/* Take GRSPW lock */
	while ( (semrc=down_interruptible(&grspw_sem)) )
		printk(KERN_WARNING "grspw_dma_open: down_interruptible %d\n", semrc);

	if (dma->open) {
		dma = NULL;
		goto out;
	}

	dma->started = 0;

	/* Set Default Configuration:
	 *
	 *  - MAX RX Packet Length = 
	 *  - Disable IRQ generation
	 *  -
	 */
	dma->cfg.rxmaxlen = DEFAULT_RXMAX;
	dma->cfg.rx_irq_en_cnt = 0;
	dma->cfg.tx_irq_en_cnt = 0;
	dma->cfg.flags = DMAFLAG_NO_SPILL;

	init_MUTEX(&dma->sem_rxdma);
	init_MUTEX(&dma->sem_txdma);

	/* Allocate memory for the two descriptor rings */
	size = sizeof(struct grspw_ring) * (GRSPW_RXBD_NR + GRSPW_TXBD_NR);
	dma->rx_ring_base = (struct grspw_rxring *)kmalloc(size, GFP_KERNEL);
	if (!dma->rx_ring_base) {
		dma = NULL;
		goto out;
	}
	dma->tx_ring_base = (struct grspw_txring *)&dma->rx_ring_base[GRSPW_RXBD_NR];

	/* Reset software structures */
	grspw_dma_reset(dma);

	/* Take the device */
	dma->open = 1;
out:
	/* Return GRSPW Lock */
	up(&grspw_sem);

	return dma;
}
EXPORT_SYMBOL(grspw_dma_open);

/* Initialize Software Structures:
 *  - Clear all Queues 
 *  - init BD ring 
 *  - init IRQ counter
 *  - clear statistics counters
 *  - init wait structures and semaphores
 */
STATIC void grspw_dma_reset(struct grspw_dma_priv *dma)
{
	init_MUTEX_LOCKED(&dma->rx_wait.sem_wait);
	init_MUTEX_LOCKED(&dma->tx_wait.sem_wait);

	/* Empty RX and TX queues */
	grspw_list_clr(&dma->ready);
	grspw_list_clr(&dma->rx_sched);
	grspw_list_clr(&dma->recv);
	grspw_list_clr(&dma->send);
	grspw_list_clr(&dma->tx_sched);
	grspw_list_clr(&dma->sent);
	dma->ready_cnt = 0;
	dma->rx_sched_cnt = 0;
	dma->recv_cnt = 0;
	dma->send_cnt = 0;
	dma->tx_sched_cnt = 0;
	dma->sent_cnt = 0;

	dma->rx_irq_en_cnt_curr = 0;
	dma->tx_irq_en_cnt_curr = 0;

	grspw_bdrings_init(dma);

	dma->rx_wait.waiting = 0;
	dma->tx_wait.waiting = 0;

	grspw_dma_stats_clr(dma);
}

int grspw_dma_close(void *c)
{
	struct grspw_dma_priv *dma = c;
	int semrc;

	if (!dma->open)
		return 0;

	/* Take device lock - Wait until we get semaphore */
	while ( (semrc=down_interruptible(&dma->sem_rxdma)) )
		printk(KERN_WARNING "grspw_dma_close: down_interruptible %d\n", semrc);
	while ( (semrc=down_interruptible(&dma->sem_txdma)) )
		printk(KERN_WARNING "grspw_dma_close: down_interruptible %d\n", semrc);

	/* Can not close active DMA channel. User must stop DMA and make sure
	 * no threads are active/blocked within driver.
	 */
	if (dma->started || dma->rx_wait.waiting || dma->tx_wait.waiting) {
		up(&dma->sem_txdma);
		up(&dma->sem_rxdma);
		return 1;
	}

	/* Free memory */
	if (dma->rx_ring_base)
		kfree(dma->rx_ring_base);
	dma->rx_ring_base = NULL;
	dma->tx_ring_base = NULL;

	dma->open = 0;
	return 0;
}
EXPORT_SYMBOL(grspw_dma_close);

/* Schedule List of packets for transmission at some point in
 * future.
 *
 * 1. Move transmitted packets to SENT List (SCHED->SENT)
 * 2. Add the requested packets to the SEND List (USER->SEND)
 * 3. Schedule as many packets as possible (SEND->SCHED)
 */
int grspw_dma_tx_send(void *c, int opts, struct grspw_list *pkts, int count)
{
	struct grspw_dma_priv *dma = c;
	int ret;
	int semrc;

	/* Take device lock */
	while ( (semrc=down_interruptible(&dma->sem_txdma)) )
		printk(KERN_WARNING "grspw_dma_tx_send: down_interruptible %d\n", semrc);

	if (dma->started == 0) {
		ret = 1; /* signal DMA has been stopped */
		goto out;
	}
	ret = 0;

	/* 1. Move transmitted packets to SENT List (SCHED->SENT) */
	if ((opts & 1) == 0)
		grspw_tx_process_scheduled(dma);

	/* 2. Add the requested packets to the SEND List (USER->SEND) */
	if (pkts) {
		grspw_list_append_list(&dma->send, pkts);
		dma->send_cnt += count;
		if (dma->stats.send_cnt_max < dma->send_cnt)
			dma->stats.send_cnt_max = dma->send_cnt;
	}

	/* 3. Schedule as many packets as possible (SEND->SCHED) */
	if ((opts & 2) == 0)
		grspw_tx_schedule_send(dma);

out:
	/* Unlock Device */
	up(&dma->sem_txdma);

	return ret;
}
EXPORT_SYMBOL(grspw_dma_tx_send);

int grspw_dma_tx_reclaim(void *c, int opts, struct grspw_list *pkts, int *count)
{
	struct grspw_dma_priv *dma = c;
	struct grspw_pkt *pkt, *lastpkt;
	int cnt, started;
	int semrc;

	/* Take device lock */
	while ( (semrc=down_interruptible(&dma->sem_txdma)) )
		printk(KERN_WARNING "grspw_dma_tx_reclaim: down_interruptible %d\n", semrc);

	/* 1. Move transmitted packets to SENT List (SCHED->SENT) */
	started = dma->started;
	if ((started > 0) && ((opts & 1) == 0))
		grspw_tx_process_scheduled(dma);

	/* Move all/count SENT packet to the callers list (SENT->USER) */
	if (pkts) {
		if ((count == NULL) || (*count == -1) ||
		    (*count >= dma->sent_cnt)) {
			/* Move all SENT Packets */
			*pkts = dma->sent;
			grspw_list_clr(&dma->sent);
			if (count)
				*count = dma->sent_cnt;
			dma->sent_cnt = 0;
		} else {
			/* Move a number of SENT Packets */
			pkts->head = pkt = lastpkt = dma->sent.head;
			cnt = 0;
			while (cnt < *count) {
				lastpkt = pkt;
				pkt = pkt->next;
				cnt++;
			}
			if (cnt > 0) {
				pkts->tail = lastpkt;
				grspw_list_remove_head_list(&dma->sent, pkts);
				dma->sent_cnt -= cnt;
			} else {
				grspw_list_clr(pkts);
			}
		}
	} else if (count) {
		*count = 0;
	}

	/* 3. Schedule as many packets as possible (SEND->SCHED) */
	if ((started > 0) && ((opts & 2) == 0))
		grspw_tx_schedule_send(dma);

	/* Unlock Device */
	up(&dma->sem_txdma);

	return (~started) & 1; /* signal DMA has been stopped */
}
EXPORT_SYMBOL(grspw_dma_tx_reclaim);

void grspw_dma_tx_count(void *c, int *send, int *sched, int *sent, int *hw)
{
	struct grspw_dma_priv *dma = c;
	int sched_cnt, diff, semrc;
	unsigned int hwbd;
	struct grspw_txbd *tailbd;

	/* Take device lock - Wait until we get semaphore.
	 * The lock is taken so that the counters are in sync with each other
	 * and that DMA descriptor table and tx_ring_tail is not being updated
	 * during HW counter processing in this function.
	 */
	while ( (semrc=down_interruptible(&dma->sem_txdma)) )
		printk(KERN_WARNING "grspw_dma_tx_count: down_interruptible %d\n", semrc);

	if (send)
		*send = dma->send_cnt;
	sched_cnt = dma->tx_sched_cnt;
	if (sched)
		*sched = sched_cnt;
	if (sent)
		*sent = dma->sent_cnt;
	if (hw) {
		/* Calculate number of descriptors (processed by HW) between
		 * HW pointer and oldest SW pointer.
		 */
		hwbd = REG_READ(&dma->regs->txdesc);
		tailbd = dma->tx_ring_tail->bd;
		diff = ((hwbd - (unsigned long)tailbd) / GRSPW_TXBD_SIZE) &
			(GRSPW_TXBD_NR - 1);
		/* Handle special case when HW and SW pointers are equal
		 * because all TX descriptors have been processed by HW.
		 */
		if ((diff == 0) && (sched_cnt == GRSPW_TXBD_NR) &&
		    ((BD_READ(&tailbd->ctrl) & GRSPW_TXBD_EN) == 0)) {
			diff = GRSPW_TXBD_NR;
		}
		*hw = diff;
	}

	/* Unlock DMA channel */
	up(&dma->sem_txdma);
}
EXPORT_SYMBOL(grspw_dma_tx_count);

static inline int grspw_tx_wait_eval(struct grspw_dma_priv *dma)
{
	int send_val, sent_val;

	if (dma->tx_wait.send_cnt >= (dma->send_cnt + dma->tx_sched_cnt))
		send_val = 1;
	else
		send_val = 0;

	if (dma->tx_wait.sent_cnt <= dma->sent_cnt)
		sent_val = 1;
	else
		sent_val = 0;

	/* AND or OR ? */
	if (dma->tx_wait.op == 0)
		return send_val & sent_val; /* AND */
	else
		return send_val | sent_val; /* OR */
}

/* Block until send_cnt or fewer packets are Queued in "Send and Scheduled" Q,
 * op (AND or OR), sent_cnt or more packet "have been sent" (Sent Q) condition
 * is met.
 * If a link error occurs and the Stop on Link error is defined, this function
 * will also return to caller.
 */
int grspw_dma_tx_wait(void *c, int send_cnt, int op, int sent_cnt, long timeout)
{
	struct grspw_dma_priv *dma = c;
	int ret, rc, initialized = 0;
	int semrc;
	long long jiffies_64_end = 0;
	long left;

	if (timeout > 0)
		jiffies_64_end = jiffies_64 + timeout;

check_condition:

	/* Take DMA channel lock */
	while ( (semrc=down_interruptible(&dma->sem_txdma)) )
		printk(KERN_WARNING "grspw_dma_tx_wait: down_interruptible %d\n", semrc);

	/* Check so that no other thread is waiting, this driver only supports
	 * one waiter at a time.
	 */
	if (initialized == 0 && dma->tx_wait.waiting) {
		ret = 3;
		goto out_release;
	}

	/* Stop if link error or similar (DMA stopped), abort */
	if (dma->started == 0) {
		ret = 1;
		goto out_release;
	}

	/* Set up Condition */
	dma->tx_wait.send_cnt = send_cnt;
	dma->tx_wait.op = op;
	dma->tx_wait.sent_cnt = sent_cnt;

	if (grspw_tx_wait_eval(dma) == 0) {
		/* Prepare Wait */
		initialized = 1;
		dma->tx_wait.waiting = 1;

		/* Release device lock */
		up(&dma->sem_txdma);

		/* Try to take Wait lock */
		if (timeout > 0) {
			if (jiffies_64_end <= jiffies_64) {
				ret = 2;
				goto out;
			}
			left = jiffies_64_end - jiffies_64;
			rc = down_timeout(&dma->tx_wait.sem_wait, left);
			if (rc == -ETIME) {
				ret = 2;
				goto out;
			}
		} else {
			rc = down_interruptible(&dma->tx_wait.sem_wait);
		}
		if (rc != 0) {
			/* Unknown Error */
			printk(KERN_WARNING "grspw_dma_tx_wait: sem down failed %d\n", rc);
			ret = -1;
			goto out;
		} else if (dma->started == 0) {
			ret = 1;
			goto out;
		}

		/* Check condition once more */
		goto check_condition;
	}

	ret = 0;

out_release:
	/* Unlock DMA channel */
	up(&dma->sem_txdma);

out:
	if (initialized)
		dma->tx_wait.waiting = 0;
	return ret;
}
EXPORT_SYMBOL(grspw_dma_tx_wait);

int grspw_dma_rx_recv(void *c, int opts, struct grspw_list *pkts, int *count)
{
	struct grspw_dma_priv *dma = c;
	struct grspw_pkt *pkt, *lastpkt;
	int cnt, started;
	int semrc;

	/* Take device lock */
	while ( (semrc=down_interruptible(&dma->sem_rxdma)) )
		printk(KERN_WARNING "grspw_dma_rx_recv: down_interruptible %d\n", semrc);

	/* 1. Move Scheduled packets to RECV List (SCHED->RECV) */
	started = dma->started;
	if (((opts & 1) == 0) && (started > 0))
		grspw_rx_process_scheduled(dma);

	/* Move all RECV packet to the callers list */
	if (pkts) {
		if ((count == NULL) || (*count == -1) ||
		    (*count >= dma->recv_cnt)) {
			/* Move all Received packets */
			*pkts = dma->recv;
			grspw_list_clr(&dma->recv);
			if ( count )
				*count = dma->recv_cnt;
			dma->recv_cnt = 0;
		} else {
			/* Move a number of RECV Packets */
			pkts->head = pkt = lastpkt = dma->recv.head;
			cnt = 0;
			while (cnt < *count) {
				lastpkt = pkt;
				pkt = pkt->next;
				cnt++;
			}
			if (cnt > 0) {
				pkts->tail = lastpkt;
				grspw_list_remove_head_list(&dma->recv, pkts);
				dma->recv_cnt -= cnt;
			} else {
				grspw_list_clr(pkts);
			}
		}
	} else if (count) {
		*count = 0;
	}

	/* 3. Schedule as many free packet buffers as possible (READY->SCHED) */
	if (((opts & 2) == 0) && (started > 0))
		grspw_rx_schedule_ready(dma);

	/* Unlock Device */
	up(&dma->sem_rxdma);

	return (~started) & 1;
}
EXPORT_SYMBOL(grspw_dma_rx_recv);

int grspw_dma_rx_prepare(void *c, int opts, struct grspw_list *pkts, int count)
{
	struct grspw_dma_priv *dma = c;
	int ret;
	int semrc;

	/* Take device lock */
	while ( (semrc=down_interruptible(&dma->sem_rxdma)) )
		printk(KERN_WARNING "grspw_dma_rx_prepare: down_interruptible %d\n", semrc);

	if (dma->started == 0) {
		ret = 1;
		goto out;
	}

	/* 1. Move Received packets to RECV List (SCHED->RECV) */
	if ((opts & 1) == 0)
		grspw_rx_process_scheduled(dma);

	/* 2. Add the "free/ready" packet buffers to the READY List (USER->READY) */
	if (pkts && (count > 0)) {
		grspw_list_append_list(&dma->ready, pkts);
		dma->ready_cnt += count;
		if (dma->stats.ready_cnt_max < dma->ready_cnt)
			dma->stats.ready_cnt_max = dma->ready_cnt;
	}

	/* 3. Schedule as many packets as possible (READY->SCHED) */
	if ((opts & 2) == 0)
		grspw_rx_schedule_ready(dma);

	ret = 0;
out:
	/* Unlock Device */
	up(&dma->sem_rxdma);

	return ret;
}
EXPORT_SYMBOL(grspw_dma_rx_prepare);

void grspw_dma_rx_count(void *c, int *ready, int *sched, int *recv, int *hw)
{
	struct grspw_dma_priv *dma = c;
	int sched_cnt, diff, semrc;
	unsigned int hwbd;
	struct grspw_rxbd *tailbd;

	/* Take device lock - Wait until we get semaphore.
	 * The lock is taken so that the counters are in sync with each other
	 * and that DMA descriptor table and rx_ring_tail is not being updated
	 * during HW counter processing in this function.
	 */
	while ( (semrc=down_interruptible(&dma->sem_rxdma)) )
		printk(KERN_WARNING "grspw_dma_rx_count: down_interruptible %d\n", semrc);

	if (ready)
		*ready = dma->ready_cnt;
	sched_cnt = dma->rx_sched_cnt;
	if (sched)
		*sched = sched_cnt;
	if (recv)
		*recv = dma->recv_cnt;
	if (hw) {
		/* Calculate number of descriptors (processed by HW) between
		 * HW pointer and oldest SW pointer.
		 */
		hwbd = REG_READ(&dma->regs->rxdesc);
		tailbd = dma->rx_ring_tail->bd;
		diff = ((hwbd - (unsigned long)tailbd) / GRSPW_RXBD_SIZE) &
			(GRSPW_RXBD_NR - 1);
		/* Handle special case when HW and SW pointers are equal
		 * because all RX descriptors have been processed by HW.
		 */
		if ((diff == 0) && (sched_cnt == GRSPW_RXBD_NR) &&
		    ((BD_READ(&tailbd->ctrl) & GRSPW_RXBD_EN) == 0)) {
			diff = GRSPW_RXBD_NR;
		}
		*hw = diff;
	}

	/* Unlock DMA channel */
	up(&dma->sem_rxdma);
}
EXPORT_SYMBOL(grspw_dma_rx_count);

static inline int grspw_rx_wait_eval(struct grspw_dma_priv *dma)
{
	int ready_val, recv_val;

	if (dma->rx_wait.ready_cnt >= (dma->ready_cnt + dma->rx_sched_cnt))
		ready_val = 1;
	else
		ready_val = 0;

	if (dma->rx_wait.recv_cnt <= dma->recv_cnt)
		recv_val = 1;
	else
		recv_val = 0;

	/* AND or OR ? */
	if (dma->rx_wait.op == 0)
		return ready_val & recv_val; /* AND */
	else
		return ready_val | recv_val; /* OR */
}

/* Block until recv_cnt or more packets are Queued in RECV Q, op (AND or OR), 
 * ready_cnt or fewer packet buffers are available in the "READY and Scheduled" Q,
 * condition is met.
 * If a link error occurs and the Stop on Link error is defined, this function
 * will also return to caller, however with an error.
 */
int grspw_dma_rx_wait(void *c, int recv_cnt, int op, int ready_cnt, long timeout)
{
	struct grspw_dma_priv *dma = c;
	int ret, rc, initialized = 0;
	int semrc;
	long long jiffies_64_end = 0;
	long left;

	if (timeout > 0)
		jiffies_64_end = jiffies_64 + timeout;

check_condition:

	/* Take device lock */
	while ( (semrc=down_interruptible(&dma->sem_rxdma)) )
		printk(KERN_WARNING "grspw_dma_rx_wait: down_interruptible %d\n", semrc);

	/* Check so that no other thread is waiting, this driver only supports
	 * one waiter at a time.
	 */
	if (initialized == 0 && dma->rx_wait.waiting) {
		ret = 3;
		goto out_release;
	}

	/* Stop if link error or similar (DMA stopped), abort */
	if (dma->started == 0) {
		ret = 1;
		goto out_release;
	}

	/* Set up Condition */
	dma->rx_wait.recv_cnt = recv_cnt;
	dma->rx_wait.op = op;
	dma->rx_wait.ready_cnt = ready_cnt;

	if (grspw_rx_wait_eval(dma) == 0) {
		/* Prepare Wait */
		initialized = 1;
		dma->rx_wait.waiting = 1;

		/* Release device lock */
		up(&dma->sem_rxdma);

		/* Try to take Wait lock */
		if (timeout > 0) {
			if (jiffies_64_end <= jiffies_64) {
			    	ret = 2;
				goto out;
			}
			left = jiffies_64_end - jiffies_64;
			rc = down_timeout(&dma->rx_wait.sem_wait, left);
			if (rc == -ETIME) {
				ret = 2;
				goto out;
			}
		} else {
			rc = down_interruptible(&dma->rx_wait.sem_wait);
		}
		if (rc != 0) {
			/* Unknown Error */
			printk(KERN_WARNING "grspw_dma_rx_wait: sem down failed %d\n", rc);
			ret = -1;
			goto out;
		} else if (dma->started == 0) {
			ret = 1;
			goto out;
		}

		/* Check condition once more */
		goto check_condition;
	}

	ret = 0;

out_release:
	/* Unlock DMA channel */
	up(&dma->sem_rxdma);

out:
	if (initialized)
		dma->rx_wait.waiting = 0;
	return ret;
}
EXPORT_SYMBOL(grspw_dma_rx_wait);

int grspw_dma_config(void *c, struct grspw_dma_config *cfg)
{
	struct grspw_dma_priv *dma = c;

	if (dma->started || !cfg)
		return -1;

	if (cfg->flags & ~(DMAFLAG_MASK | DMAFLAG2_MASK))
		return -1;

	/* Update Configuration */
	memcpy(&dma->cfg, cfg, sizeof(*cfg));

	return 0;
}
EXPORT_SYMBOL(grspw_dma_config);

void grspw_dma_config_read(void *c, struct grspw_dma_config *cfg)
{
	struct grspw_dma_priv *dma = c;

	/* Copy Current Configuration */
	memcpy(cfg, &dma->cfg, sizeof(*cfg));
}
EXPORT_SYMBOL(grspw_dma_config_read);

void grspw_dma_stats_read(void *c, struct grspw_dma_stats *sts)
{
	struct grspw_dma_priv *dma = c;

	memcpy(sts, &dma->stats, sizeof(dma->stats));
}
EXPORT_SYMBOL(grspw_dma_stats_read);

void grspw_dma_stats_clr(void *c)
{
	struct grspw_dma_priv *dma = c;

	/* Clear most of the statistics */	
	memset(&dma->stats, 0, sizeof(dma->stats));

	/* Init proper default values so that comparisons will work the
	 * first time.
	 */
	dma->stats.send_cnt_min = 0x3fffffff;
	dma->stats.tx_sched_cnt_min = 0x3fffffff;
	dma->stats.ready_cnt_min = 0x3fffffff;
	dma->stats.rx_sched_cnt_min = 0x3fffffff;
}
EXPORT_SYMBOL(grspw_dma_stats_clr);

int grspw_dma_start(void *c)
{
	struct grspw_dma_priv *dma = c;
	struct grspw_dma_regs *dregs = dma->regs;
	unsigned int ctrl;
	IRQFLAGS_TYPE irqflags;

	if (dma->started)
		return 0;

	/* Initialize Software Structures:
	 *  - Clear all Queues
	 *  - init BD ring 
	 *  - init IRQ counter
	 *  - clear statistics counters
	 *  - init wait structures and semaphores
	 */
	grspw_dma_reset(dma);

	/* RX&RD and TX is not enabled until user fills SEND and READY Queue
	 * with SpaceWire Packet buffers. So we do not have to worry about
	 * IRQs for this channel just yet. However other DMA channels
	 * may be active.
	 *
	 * Some functionality that is not changed during started mode is set up
	 * once and for all here:
	 *
	 *   - RX MAX Packet length
	 *   - TX Descriptor base address to first BD in TX ring (not enabled)
	 *   - RX Descriptor base address to first BD in RX ring (not enabled)
	 *   - IRQs (TX DMA, RX DMA, DMA ERROR)
	 *   - Strip PID
	 *   - Strip Address
	 *   - No Spill
	 *   - Receiver Enable
	 *   - disable on link error (LE)
	 *
	 * Note that the address register and the address enable bit in DMACTRL
	 * register must be left untouched, they are configured on a GRSPW
	 * core level.
	 *
	 * Note that the receiver is enabled here, but since descriptors are
	 * not enabled the GRSPW core may stop/pause RX (if NS bit set) until
	 * descriptors are enabled or it may ignore RX packets (NS=0) until
	 * descriptors are enabled (writing RD bit).
	 */
	REG_WRITE(&dregs->txdesc, dma->tx_bds_pa);
	REG_WRITE(&dregs->rxdesc, dma->rx_bds_pa);

	/* MAX Packet length */
	REG_WRITE(&dma->regs->rxmax, dma->cfg.rxmaxlen);

	ctrl =  GRSPW_DMACTRL_AI | GRSPW_DMACTRL_PS | GRSPW_DMACTRL_PR |
		GRSPW_DMACTRL_TA | GRSPW_DMACTRL_RA | GRSPW_DMACTRL_RE |
		(dma->cfg.flags & DMAFLAG_MASK) << GRSPW_DMACTRL_NS_BIT;
	if (dma->core->dis_link_on_err & LINKOPTS_DIS_ONERR)
		ctrl |= GRSPW_DMACTRL_LE;
	if (dma->cfg.rx_irq_en_cnt != 0 || dma->cfg.flags & DMAFLAG2_RXIE)
		ctrl |= GRSPW_DMACTRL_RI;
	if (dma->cfg.tx_irq_en_cnt != 0 || dma->cfg.flags & DMAFLAG2_TXIE)
		ctrl |= GRSPW_DMACTRL_TI;
	SPIN_LOCK_IRQ(&dma->core->devlock, irqflags);
	ctrl |= REG_READ(&dregs->ctrl) & GRSPW_DMACTRL_EN;
	REG_WRITE(&dregs->ctrl, ctrl);
	SPIN_UNLOCK_IRQ(&dma->core->devlock, irqflags);

	dma->started = 1; /* open up other DMA interfaces */

	return 0;
}
EXPORT_SYMBOL(grspw_dma_start);

static void grspw_dma_stop_locked(struct grspw_dma_priv *dma)
{
	struct grspw_priv *priv = dma->core;
	IRQFLAGS_TYPE irqflags;

	if (dma->started == 0)
		return;
	dma->started = 0;

	SPIN_LOCK_IRQ(&priv->devlock, irqflags);
	grspw_hw_dma_stop(dma);
	SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);

	/* From here no more packets will be sent, however
	 * there may still exist scheduled packets that has been
	 * sent, and packets in the SEND Queue waiting for free
	 * descriptors. All packets are moved to the SENT Queue
	 * so that the user may get its buffers back, the user
	 * must look at the TXPKT_FLAG_TX in order to determine
	 * if the packet was sent or not.
	 */

	/* Retreive scheduled all sent packets */
	grspw_tx_process_scheduled(dma);

	/* Move un-sent packets in SEND and SCHED queue to the
	 * SENT Queue. (never marked sent)
	 */
	if (!grspw_list_is_empty(&dma->tx_sched)) {
		grspw_list_append_list(&dma->sent, &dma->tx_sched);
		grspw_list_clr(&dma->tx_sched);
		dma->sent_cnt += dma->tx_sched_cnt;
		dma->tx_sched_cnt = 0;
	}
	if (!grspw_list_is_empty(&dma->send)) {
		grspw_list_append_list(&dma->sent, &dma->send);
		grspw_list_clr(&dma->send);
		dma->sent_cnt += dma->send_cnt;
		dma->send_cnt = 0;
	}

	/* Similar for RX */
	grspw_rx_process_scheduled(dma);
	if (!grspw_list_is_empty(&dma->rx_sched)) {
		grspw_list_append_list(&dma->recv, &dma->rx_sched);
		grspw_list_clr(&dma->rx_sched);
		dma->recv_cnt += dma->rx_sched_cnt;
		dma->rx_sched_cnt = 0;
	}
	if (!grspw_list_is_empty(&dma->ready)) {
		grspw_list_append_list(&dma->recv, &dma->ready);
		grspw_list_clr(&dma->ready);
		dma->recv_cnt += dma->ready_cnt;
		dma->ready_cnt = 0;
	}

	/* Throw out blocked threads */
	if (dma->rx_wait.waiting)
		up(&dma->rx_wait.sem_wait);
	if (dma->tx_wait.waiting)
		up(&dma->tx_wait.sem_wait);
}

void grspw_dma_stop(void *c)
{
	struct grspw_dma_priv *dma = c;
	int semrc;

	/* If DMA channel is closed we should not access the semaphore */
	if (!dma->open)
		return;

	/* Take DMA Channel lock */
	while ( (semrc=down_interruptible(&dma->sem_rxdma)) )
			printk(KERN_WARNING "grspw_dma_stop: down_interruptible %d\n", semrc);
	while ( (semrc=down_interruptible(&dma->sem_txdma)) )
			printk(KERN_WARNING "grspw_dma_stop: down_interruptible %d\n", semrc);

	grspw_dma_stop_locked(dma);

	up(&dma->sem_txdma);
	up(&dma->sem_rxdma);
}
EXPORT_SYMBOL(grspw_dma_stop);

/* Do general work, invoked indirectly from ISR */
static void grspw_work_func(struct work_struct *work)
{
	struct grspw_priv *priv = container_of(work, struct grspw_priv, work);
	int i;

	/* Link is down for some reason, and the user has configured
	 * that we stop all (open) DMA channels and throw out all their
	 * blocked threads.
	 */
	for (i=0; i<priv->hwsup.ndma_chans; i++)
		grspw_dma_stop(&priv->dma[i]);
	grspw_hw_stop(priv);
}

/* Do DMA work on one channel, invoked indirectly from ISR */
static void grspw_work_dma_func(struct work_struct *work)
{
	struct grspw_dma_priv *dma = container_of(work, struct grspw_dma_priv, work);
	struct grspw_priv *priv = dma->core;
	int tx_cond_true, rx_cond_true;
	unsigned int ctrl;
	int semrc;
	IRQFLAGS_TYPE irqflags;

	/* If DMA channel is closed we should not access the semaphore */
	if (dma->open == 0)
		return;

	rx_cond_true = 0;
	tx_cond_true = 0;
	dma->stats.irq_cnt++;

	/* Look at cause we were woken up and clear source */
	SPIN_LOCK_IRQ(&priv->devlock, irqflags);
	if (dma->started == 0) {
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
		return;
	}
	ctrl = REG_READ(&dma->regs->ctrl);

	/* Read/Write DMA error ? */
	if (ctrl & GRSPW_DMA_STATUS_ERROR) {
		/* DMA error -> Stop DMA channel (both RX and TX) */
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
		grspw_dma_stop(dma);
	} else if (ctrl & (GRSPW_DMACTRL_PR | GRSPW_DMACTRL_PS)) {
		/* DMA has finished a TX/RX packet
		 * Clear Source.
		 */
		ctrl &= ~GRSPW_DMACTRL_AT;
		if (dma->cfg.rx_irq_en_cnt != 0 ||
		    (dma->cfg.flags & DMAFLAG2_RXIE))
			ctrl |= GRSPW_DMACTRL_RI;
		if (dma->cfg.tx_irq_en_cnt != 0 ||
		    (dma->cfg.flags & DMAFLAG2_TXIE))
			ctrl |= GRSPW_DMACTRL_TI;
		REG_WRITE(&dma->regs->ctrl, ctrl);
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
		if (ctrl & GRSPW_DMACTRL_PR) {
			/* Do RX Work */

			/* Take DMA Channel RX lock */
			while ( (semrc=down_interruptible(&dma->sem_rxdma)) )
				printk(KERN_WARNING "grspw_work_dma_func: down_interruptible %d\n", semrc);		

			dma->stats.rx_work_cnt++;
			grspw_rx_process_scheduled(dma);
			if (dma->started) {
				dma->stats.rx_work_enabled +=
					grspw_rx_schedule_ready(dma);
				/* Check to see if condition for waking blocked
			 	 * USER task is fullfilled.
				 */
				if (dma->rx_wait.waiting)
					rx_cond_true = grspw_rx_wait_eval(dma);
			}
			up(&dma->sem_rxdma);
		}
		if (ctrl & GRSPW_DMACTRL_PS) {
			/* Do TX Work */

			/* Take DMA Channel TX lock */
			while ( (semrc=down_interruptible(&dma->sem_txdma)) )
				printk(KERN_WARNING "grspw_work_dma_func: down_interruptible %d\n", semrc);

			dma->stats.tx_work_cnt++;
			grspw_tx_process_scheduled(dma);
			if (dma->started) {
				dma->stats.tx_work_enabled +=
					grspw_tx_schedule_send(dma);
				/* Check to see if condition for waking blocked
			 	 * USER task is fullfilled.
				 */
				if (dma->tx_wait.waiting)
					tx_cond_true = grspw_tx_wait_eval(dma);
			}
			up(&dma->sem_txdma);
		}
	} else
		SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);

	if (rx_cond_true)
		up(&dma->rx_wait.sem_wait);

	if (tx_cond_true)
		up(&dma->tx_wait.sem_wait);
}

static irqreturn_t grspw_isr(int irq, void *data)
{
	struct grspw_priv *priv = data;
	unsigned int dma_stat, stat, stat_clrmsk, ctrl, timecode;
	int i;
	irqreturn_t ret = 0;

	/* Get Status from Hardware */
	stat = REG_READ(&priv->regs->status);
	stat_clrmsk = stat & (GRSPW_STS_TO | GRSPW_STAT_ERROR) & priv->stscfg;

	/* Make sure to put the timecode handling first in order to get the
	 * smallest possible interrupt latency
	 */
	if ((stat & GRSPW_STS_TO) && (priv->tcisr != NULL)) {
		ctrl = REG_READ(&priv->regs->ctrl);
		if (ctrl & GRSPW_CTRL_TQ) {
			/* Timecode received. Let custom function handle this */
			timecode = REG_READ(&priv->regs->time) &
					(GRSPW_TIME_CTRL | GRSPW_TIME_TCNT);
			(priv->tcisr)(priv->tcisr_arg, timecode);
		}
	}

	/* An Error occured? */
	if (stat & GRSPW_STAT_ERROR) {
		/* Wake Global WorkQ */
		ret = IRQ_HANDLED;

		if (stat & GRSPW_STS_EE)
			priv->stats.err_eeop++;

		if (stat & GRSPW_STS_IA)
			priv->stats.err_addr++;

		if (stat & GRSPW_STS_PE)
			priv->stats.err_parity++;

		if (stat & GRSPW_STS_DE)
			priv->stats.err_disconnect++;

		if (stat & GRSPW_STS_ER)
			priv->stats.err_escape++;

		if (stat & GRSPW_STS_CE)
			priv->stats.err_credit++;

		if (stat & GRSPW_STS_WE)
			priv->stats.err_wsync++;

		if ((priv->dis_link_on_err >> 16) & stat) {
			/* Disable the link, no more transfers are expected
			 * on any DMA channel.
			 */
			SPIN_LOCK(&priv->devlock, irqflags);
			ctrl = REG_READ(&priv->regs->ctrl);
			REG_WRITE(&priv->regs->ctrl, GRSPW_CTRL_LD |
				(ctrl & ~(GRSPW_CTRL_IE|GRSPW_CTRL_LS)));
			SPIN_UNLOCK(&priv->devlock, irqflags);
			/* Start Work to clean up */
			queue_work(grspw_workq, &priv->work);
		}
	}

	/* Clear Status Flags */
	if (stat_clrmsk) {
		ret = IRQ_HANDLED;
		REG_WRITE(&priv->regs->status, stat_clrmsk);
	}

	/* A DMA transfer or Error occured? In that case disable more IRQs
	 * from the DMA channel, then invoke the workQ.
	 *
	 * The far most common HW setup is one DMA channel, so we optimize
	 * it. Also the GI interrupt flag may not be available for older
	 * designs where (was added together with mutiple DMA channels).
	 */
	SPIN_LOCK(&priv->devlock, irqflags);
	if ( priv->hwsup.ndma_chans == 1 ) {
		dma_stat = REG_READ(&priv->regs->dma[0].ctrl);
		/* Check for Errors and if Packets been sent or received if
		 * respective IRQ are enabled
		 */
		if ( (((dma_stat << 3) & (GRSPW_DMACTRL_PR|GRSPW_DMACTRL_PS))
		     | GRSPW_DMA_STATUS_ERROR) & dma_stat ) {
			/* Disable Further IRQs (until enabled again)
			 * from this DMA channel. Let the status
			 * bit remain so that they can be handled by
			 * work function.
			 */
			REG_WRITE(&priv->regs->dma[0].ctrl, dma_stat & 
				~(GRSPW_DMACTRL_RI|GRSPW_DMACTRL_TI|
				GRSPW_DMACTRL_PR|GRSPW_DMACTRL_PS|
				GRSPW_DMACTRL_RA|GRSPW_DMACTRL_TA|
				GRSPW_DMACTRL_AT));
			queue_work(grspw_workq, &priv->dma[0].work);
			ret = IRQ_HANDLED;
		}
	} else {
		for (i=0; i<priv->hwsup.ndma_chans; i++) {
			dma_stat = REG_READ(&priv->regs->dma[i].ctrl);
#ifdef HW_WITH_GI
		if ( dma_stat & (GRSPW_DMA_STATUS_ERROR | GRSPW_DMACTRL_GI) ) {
#else
		if ( (((dma_stat << 3) & (GRSPW_DMACTRL_PR | GRSPW_DMACTRL_PS))
		     | GRSPW_DMA_STATUS_ERROR) & dma_stat ) {
#endif
			/* Disable Further IRQs (until enabled again)
			 * from this DMA channel. Let the status
			 * bit remain so that they can be handled by
			 * work function.
			 */
			REG_WRITE(&priv->regs->dma[i].ctrl, dma_stat & 
				~(GRSPW_DMACTRL_RI|GRSPW_DMACTRL_TI|
				GRSPW_DMACTRL_PR|GRSPW_DMACTRL_PS|
				GRSPW_DMACTRL_RA|GRSPW_DMACTRL_TA|
				GRSPW_DMACTRL_AT));
				queue_work(grspw_workq, &priv->dma[i].work);
				ret = IRQ_HANDLED;
		}
		}
	}
	SPIN_UNLOCK(&priv->devlock, irqflags);

	if ( ret == IRQ_HANDLED )
		priv->stats.irq_cnt++;

	return ret;
}

STATIC void grspw_hw_dma_stop(struct grspw_dma_priv *dma)
{
	unsigned int ctrl;
	struct grspw_dma_regs *dregs = dma->regs;

	ctrl = REG_READ(&dregs->ctrl) & (GRSPW_DMACTRL_LE | GRSPW_DMACTRL_EN |
	       GRSPW_DMACTRL_SP | GRSPW_DMACTRL_SA | GRSPW_DMACTRL_NS);
	ctrl |=	GRSPW_DMACTRL_AT;
	REG_WRITE(&dregs->ctrl, ctrl);
}

STATIC void grspw_hw_dma_softreset(struct grspw_dma_priv *dma)
{
	unsigned int ctrl;
	struct grspw_dma_regs *dregs = dma->regs;

	ctrl = REG_READ(&dregs->ctrl) & (GRSPW_DMACTRL_LE | GRSPW_DMACTRL_EN);
	REG_WRITE(&dregs->ctrl, ctrl);

	REG_WRITE(&dregs->rxmax, DEFAULT_RXMAX);
	REG_WRITE(&dregs->txdesc, 0);
	REG_WRITE(&dregs->rxdesc, 0);
}

/* Hardware Action:
 *  - stop DMA
 *  - do not bring down the link (RMAP may be active)
 *  - RMAP settings untouched (RMAP may be active)
 *  - port select untouched (RMAP may be active)
 *  - timecodes are disabled
 *  - IRQ generation disabled
 *  - status not cleared (let user analyze it if requested later on)
 *  - Node address / First DMA channels Node address
 *    is untouched (RMAP may be active)
 */
STATIC void grspw_hw_stop(struct grspw_priv *priv)
{
	int i;
	unsigned int ctrl;
	IRQFLAGS_TYPE irqflags;

	SPIN_LOCK_IRQ(&priv->devlock, irqflags);

	for (i=0; i<priv->hwsup.ndma_chans; i++)
		grspw_hw_dma_stop(&priv->dma[i]);

	ctrl = REG_READ(&priv->regs->ctrl);
	REG_WRITE(&priv->regs->ctrl, ctrl & (
		GRSPW_CTRL_LD | GRSPW_CTRL_LS | GRSPW_CTRL_AS |
		GRSPW_CTRL_RE | GRSPW_CTRL_RD |
		GRSPW_CTRL_NP | GRSPW_CTRL_PS));

	SPIN_UNLOCK_IRQ(&priv->devlock, irqflags);
}

/* Soft reset of GRSPW core registers */
STATIC void grspw_hw_softreset(struct grspw_priv *priv)
{
	int i;
	unsigned int tmp;

	for (i=0; i<priv->hwsup.ndma_chans; i++)
		grspw_hw_dma_softreset(&priv->dma[i]);

	REG_WRITE(&priv->regs->status, 0xffffffff);
	REG_WRITE(&priv->regs->time, 0);
	/* Clear all but valuable reset values of ICCTRL */
	tmp = REG_READ(&priv->regs->icctrl);
	tmp &= GRSPW_ICCTRL_INUM | GRSPW_ICCTRL_BIRQ | GRSPW_ICCTRL_TXIRQ;
	tmp |= GRSPW_ICCTRL_ID;
	REG_WRITE(&priv->regs->icctrl, tmp);
	REG_WRITE(&priv->regs->icrx, 0xffffffff);
	REG_WRITE(&priv->regs->icack, 0xffffffff);
	REG_WRITE(&priv->regs->ictimeout, 0xffffffff);
}

int grspw_dev_count(void)
{
	return grspw_count;
}
EXPORT_SYMBOL(grspw_dev_count);

void grspw_initialize_user(void *(*devfound)(int), void (*devremove)(int,void*))
{
	int i;
	struct grspw_priv *priv;

	/* Set new Device Found Handler */
	grspw_dev_add = devfound;
	grspw_dev_del = devremove;

	if (grspw_initialized == 1 && grspw_dev_add) {
		/* Call callback for every previously found device */
		for (i=0; i<grspw_count; i++) {
			priv = priv_tab[i];
			if (priv)
				priv->data = grspw_dev_add(i);
		}
	}
}
EXPORT_SYMBOL(grspw_initialize_user);

static int grspw_of_probe(struct platform_device *of_dev)
{
	struct device_node *onp = of_dev->dev.of_node;
	struct grspw_priv *priv;
	int i, size, err, len;
	unsigned int ctrl;
	const int *pampopts, *pversion, *pindex, *pdevice;
	int devid = 0;
	char *devname;

	if ( grspw_count >= GRSPW_MAX )
		return -ENOMEM;

	if (!of_dev || !of_dev->resource) {
		printk(KERN_NOTICE "GRSPW: of_dev=%p has no resources.. ignoring\n",
			of_dev);
		return -EINVAL;
	}

	/* Skip this device if user has set ampopts to zero, the device may
	 * be used by another OS instance in AMP systems.
	 */
	pampopts = of_get_property(of_dev->dev.of_node, "ampopts", &len);
	if ( pampopts && (len == 4) && (*pampopts == 0) )
		return 0;

	priv = (struct grspw_priv *)kmalloc(sizeof(*priv), GFP_KERNEL);
	if ( !priv )
		return -ENOMEM;
	memset(priv, 0, sizeof(*priv));


	/* MAP Registers into Kernel Virtual Address Space */
	priv->regs = devm_platform_ioremap_resource(of_dev, 0);
	if (IS_ERR(priv->regs)) {
		err = PTR_ERR(priv->regs);
		goto error;
	}

	/* Get IRQ number of GRSPW device */
	priv->irq = irq_of_parse_and_map(onp, 0);
	if (!priv->irq) {
		dev_err(&of_dev->dev, "no irq found\n");
		err = -ENODEV;
		goto error;
	}
	priv->dev = &of_dev->dev;

	SPIN_INIT(&priv->devlock);

	/* Register character device in registered region */
	priv->index = grspw_count;
	priv_tab[priv->index] = priv;

	strcat(priv->devname, "GRSPW_N");
	priv->devname[6] = '0' + priv->index;

	/* Read Hardware Support from Control Register */
	ctrl = REG_READ(&priv->regs->ctrl);
	priv->hwsup.rmap = (ctrl & GRSPW_CTRL_RA) >> GRSPW_CTRL_RA_BIT;
	priv->hwsup.rmap_crc = (ctrl & GRSPW_CTRL_RC) >> GRSPW_CTRL_RC_BIT;
	priv->hwsup.rx_unalign = (ctrl & GRSPW_CTRL_RX) >> GRSPW_CTRL_RX_BIT;
	priv->hwsup.nports = 1 + ((ctrl & GRSPW_CTRL_PO) >> GRSPW_CTRL_PO_BIT);
	priv->hwsup.ndma_chans = 1 + ((ctrl & GRSPW_CTRL_NCH) >> GRSPW_CTRL_NCH_BIT);

	/* Find Device ID and Version */
	priv->hwsup.hw_version = 0;
	pversion = of_get_property(of_dev->dev.of_node, "version", &len);
	if ( pversion && (len == 4) )
		priv->hwsup.hw_version = *pversion;
	pdevice = 0;
	pdevice = of_get_property(of_dev->dev.of_node, "device", &len);
	if ( pdevice && (len == 4) )
		devid = *pdevice;
	priv->hwsup.hw_version |= devid << 16;
	if ( (devid == GAISLER_SPW2) || (devid == GAISLER_SPW2_DMA) ) {
		priv->hwsup.strip_adr = 1; /* All GRSPW2 can strip Address */
		priv->hwsup.strip_pid = 1; /* All GRSPW2 can strip PID */
		if ( devid == GAISLER_SPW2 )
			devname = "GRSPW2";
		else
			devname = "GRSPW2_ROUTER";
	} else {
		/* Autodetect GRSPW1 features? */
		priv->hwsup.strip_adr = 0;
		priv->hwsup.strip_pid = 0;
		devname = "GRSPW1";
	}

	/* Get device Index just for Information */
	priv->core_index = 0;
	pindex = of_get_property(of_dev->dev.of_node, "index", &len);
	if ( pindex && (len == 4) )
		priv->core_index = *pindex;

	printk(KERN_NOTICE "GRSPW[%d]: type=%s, core_index=%d, vadr_regs=%p\n",
		priv->index, devname, priv->core_index, priv->regs);

	/* Allocate Memory for all DMA channels */
	size = sizeof(struct grspw_dma_priv) * priv->hwsup.ndma_chans;
	priv->dma = (struct grspw_dma_priv *) kmalloc(size, GFP_KERNEL);
	memset(priv->dma, 0, size);

	/* Initializing Work for this particular GRSPW Core */
	INIT_WORK(&priv->work, grspw_work_func);
	for (i=0; i<priv->hwsup.ndma_chans; i++) {
		priv->dma[i].core = priv;
		priv->dma[i].index = i;
		priv->dma[i].regs = &priv->regs->dma[i];
		INIT_WORK(&priv->dma[i].work, grspw_work_dma_func);
	}

	/* Startup Action:
	 *  - stop DMA
	 *  - do not bring down the link (RMAP may be active)
	 *  - RMAP settings untouched (RMAP may be active)
	 *  - port select untouched (RMAP may be active)
	 *  - timecodes are diabled
	 *  - IRQ generation disabled
	 *  - status cleared
	 *  - Node address / First DMA channels Node address
	 *    is untouched (RMAP may be active)
	 */
	grspw_hw_stop(priv);
	grspw_hw_softreset(priv);

	/* Register IRQ Handler */
	err = request_irq(priv->irq, grspw_isr, IRQF_SHARED,
				priv->devname, priv);
	if ( err ) {
		printk(KERN_NOTICE "GRSPW[%d]: Failed installing IRQ %d\n",
			priv->index, priv->irq);
		goto error;
	}

	dev_set_drvdata(&of_dev->dev, priv);

	grspw_count++;

	/* Tell above layer about new device */
	if (grspw_dev_add)
		priv->data = grspw_dev_add(priv->index);

	return 0;

error:
	return err;
}

static int grspw_of_remove(struct platform_device *of_dev)
{
	struct grspw_priv *priv = dev_get_drvdata(&of_dev->dev);

	if ( grspw_dev_del )
		grspw_dev_del(priv->index, priv->data);

	/* Remove IRQ handler */
	free_irq(priv->irq, priv);

	priv_tab[priv->index] = NULL;

	kfree(priv);
	dev_set_drvdata(&of_dev->dev, NULL);

	return 0;
}

static const struct of_device_id grspw_of_match[] = {
#if 0 /* GRSPW1 not tested yet... */
	{ /* SpaceWire 1 Interface */
	 .name = "GAISLER_SPW",
	 .data = (void *)GAISLER_SPW,
	 },
	{ /* SpaceWire 1 Interface, in numbers */
	 .name = "01_01f",
	 .data = (void *)GAISLER_SPW,
	 },
#endif
	{ /* SpaceWire 2 Interface */
	 .name = "GAISLER_SPW2",
	 .data = (void *)GAISLER_SPW2,
	 },
	{ /* SpaceWire 2 Interface, in numbers */
	 .name = "01_029",
	 .data = (void *)GAISLER_SPW2,
	 },
	{ /* SpaceWire Router DMA Interface */
	 .name = "GAISLER_SPW2_DMA",
	 .data = (void *)GAISLER_SPW2_DMA,
	 },
	{
	 .name = "01_08a",
	 .data = (void *)GAISLER_SPW2_DMA,
	 },
	{ /* SpaceWire 2 Interface */
	 .compatible = "gaisler,grspw2",
	 .data = (void *)GAISLER_SPW2,
	 },
	{ /* SpaceWire Router DMA Interface */
	 .compatible = "gaisler,grspw2_dma",
	 .data = (void *)GAISLER_SPW2_DMA,
	 },
	{}, /* Mark end */
};

MODULE_DEVICE_TABLE(of, grspw_of_match);

static struct platform_driver grspw_of_driver = {
	.driver = {
		.name = "grlib-grspw",
		.owner = THIS_MODULE,
		.of_match_table = grspw_of_match,
	},
	.probe = grspw_of_probe,
	.remove = grspw_of_remove,
};

static int __init grspw_init(void)
{
	grspw_count = 0;
	memset(priv_tab, 0, sizeof(priv_tab));
	init_MUTEX(&grspw_sem);

	/* Create Work queue for all GRSPW driver instances */
	grspw_workq = create_workqueue("GRSPW_WORK");
	if ( grspw_workq == NULL ) {
		printk(KERN_WARNING "GRSPW: Failed to create Work Queue\n");
			return -1;
	}

	grspw_initialized = 1;

	/* Register Platform Driver */
	return platform_driver_register(&grspw_of_driver);
}

static void __exit grspw_exit (void)
{
	grspw_initialized = 0;
	if ( grspw_workq ) {
		destroy_workqueue(grspw_workq);
		grspw_workq = NULL;
	}
	grspw_count = 0;

	platform_driver_unregister(&grspw_of_driver);
}

module_init(grspw_init);
module_exit(grspw_exit);

MODULE_AUTHOR("Cobham Gaisler AB.");
MODULE_DESCRIPTION("Cobham Gaisler GRSPW SpaceWire Library");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:grlib-grspw");
