/*
 * Device Memory MAP Library help functions for GRLIB Device Drivers
 *
 * Copyright (c) 2016-2023 Frontgrade Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <asm/uaccess.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/grlib/maplib.h>

MODULE_LICENSE ("GPL");

/*#define MAPLIB_DEBUG*/

#ifdef MAPLIB_DEBUG
	#define MAPLIB_DBG(doCode) doCode
#else
	#define MAPLIB_DBG(doCode)
#endif

/* this interface was not ported because it was unused going from 3.10 to 5.10 kernel */
#define MAPLIB_KERN_TO_USR_LOOKUP 0

#define MAPLIB_MAX 1

#define BLK_SIZE   0x20000
#define BLK_SHIFT  17

struct maplib_blk {
	unsigned int length;			/* Length of Block, 0x20000 unless last*/
	void *start;				/* Kernel Address */
};

struct maplib_priv {
	/*** HARDWARE PARAMS ***/
	char devname[16];
	int index;

	/*** SOFWTARE PARAMS ***/
	struct cdev cdev;		/* Char device */
	int open;			/* If Device is Opened by user */

	/*** MEMORY MAP TO USERSPACE HANDLING ***/

	/* Buffer Blocks, usrblk_per_blk Buffers per block */
	struct maplib_blk *blks;	/* Array of TM Frame Buffer Blocks */
	int blk_cnt;			/* Number of Blocks */
	int usrblk_per_blk;
	int usrblk_size;
	int blk_tot_size;		/* Total Size of all blocks */

	unsigned long ustart;
	unsigned long uend;
	struct vm_area_struct *user_vma;

#if MAPLIB_KERN_TO_USR_LOOKUP
	/* Kernel Address to Block number Translation Table.
	 * MSB 8-bit.
	 */
	short *kern_tab[256];
#endif

	/* All drivers attached */
	struct maplib_drv *drvs;
};

struct maplib_priv *maplib_privs = NULL;
int maplib_count = 0;
static struct class *maplib_class = NULL;
static int dev_major = 0;

static int maplib_open(struct inode *inode, struct file *file);
static int maplib_release(struct inode *inode, struct file *file);
static long maplib_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);
static int maplib_mmap(struct file *filp, struct vm_area_struct *vma);
static void maplib_vma_close(struct vm_area_struct *area);

struct file_operations maplib_fops = {
	.owner          = THIS_MODULE,
	.open           = maplib_open,
	.release        = maplib_release,
	.unlocked_ioctl = maplib_ioctl,
	.mmap           = maplib_mmap,
};

struct vm_operations_struct maplib_vops = {
	.close = maplib_vma_close,
};

/* We go from all memory being mapped into not all memory being
 * mapped. This means that we must signal to all drivers to
 * stop using the memory.
 */
static void maplib_drv_unreg_all(struct maplib_priv *priv)
{
	struct maplib_drv *drv = priv->drvs;
	while ( drv ) {
		drv->unmap(priv->index, drv->priv);
		drv = drv->next;
	}
	priv->drvs = NULL;
}

/* Register Driver as active user of Memory mapped memory */
int maplib_drv_reg(int idx, struct maplib_drv *drv)
{
	struct maplib_priv *priv = &maplib_privs[idx];

	if ( priv->user_vma == NULL )
		return -1;

	drv->next = priv->drvs;
	priv->drvs = drv;
	return 0;
}
EXPORT_SYMBOL(maplib_drv_reg);

/* Unregister Driver */
void maplib_drv_unreg(int idx, struct maplib_drv *drv)
{
	struct maplib_priv *priv = &maplib_privs[idx];
	struct maplib_drv **prev, *d = priv->drvs;

	prev = &priv->drvs;
	while ( (d != drv) && d ) {
		prev = &d->next;
		d = d->next;
	}
	if ( d == drv ) {
		*prev = d->next;
	}
}
EXPORT_SYMBOL(maplib_drv_unreg);

/* Unregister Driver */
int maplib_drv_chkvalid(int idx)
{
	if (idx < 0 || idx >= maplib_count)
		return -1;
	if (maplib_privs == NULL)
		return -1;
	return 0; /* MAP index exists */
}
EXPORT_SYMBOL(maplib_drv_chkvalid);

#if 0
void maplib_tab_print(struct maplib_priv *priv, short **usrtab)
{
	struct maplib_blk *blk;
	short *blktab;
	int i,j;

	for(i=0; i<256; i++){
		if ( usrtab[i] ) {
			blktab = usrtab[i];
			for (j=0; j<128; j++) {
				if ( blktab[j] != -1 ) {
					blk = &priv->blks[blktab[j]];
					printk(KERN_DEBUG "  %02x:%02x:%d: %x-%x\n",
						i, j, blktab[j],
						blk->user_vma->vm_start,
						blk->user_vma->vm_end);
				}
			}
		}
	}
}
#endif

/* Translate previously MMAPed User Space Virtual Address into Kernel Address
 * Return -1 if invalid address
 */
int maplib_lookup_kern(int idx, void *usr, void **kern, phys_addr_t *hw, int len)
{
	struct maplib_priv *priv = &maplib_privs[idx];
	unsigned long uadr = (unsigned long)usr;
	int blkno, blkno_end;
	struct maplib_blk *blk;
	unsigned long ofs;

	if ( (uadr < priv->ustart) || (uadr >= priv->uend) )
		return -1;

	ofs = uadr - priv->ustart;
	blkno = ofs >> BLK_SHIFT;
	blkno_end = (ofs+len-1) >> BLK_SHIFT;
	if ( blkno != blkno_end ) /* Check that block boundary is not passed */
		return -1;
	ofs = ofs & (BLK_SIZE-1);
	blk = &priv->blks[blkno];
	if ( kern )
		*kern = (void *)((unsigned long)blk->start + ofs);
	if ( hw )
		*hw = virt_to_phys(blk->start + ofs);

	return 0;
}
EXPORT_SYMBOL(maplib_lookup_kern);

#if MAPLIB_KERN_TO_USR_LOOKUP
/* Translate previously MMAPed Kernel Virtual Address into UserSpace Address
 * Return -1 if invalid address
 */
int maplib_lookup_usr(int idx, void *kern, phys_addr_t hw, void **usr, int len)
{
	int msb = (unsigned int)kern >> 24;
	int blkidx = ((unsigned int)kern & 0x00FFFFFF) >> BLK_SHIFT;
	int blkno;
	struct maplib_blk *blk;
	struct maplib_priv *priv = &maplib_privs[idx];
	void *usrbase;

	if ( hw )
		kern = phys_to_virt(hw);

	if ( priv->kern_tab[msb] && (priv->kern_tab[msb][blkidx] != -1) ) {
		blkno = priv->kern_tab[msb][blkidx];
		blk = &priv->blks[blkno];
		usrbase = (void *)(priv->ustart + (blkno<<BLK_SHIFT));
		*usr = (void *)((unsigned int)usrbase +
			((unsigned int)kern & (BLK_SIZE-1)));
		/* Check Range */
		if ( ((unsigned int)*usr + len) > ((unsigned int)usrbase + blk->length) )
			return -1;
		return 0;
	}
	return -1;
}
EXPORT_SYMBOL(maplib_lookup_usr);

/* Build KernAdr->BlkNo Translation tree */
static void maplib_tabs_build(struct maplib_priv *priv)
{
	struct maplib_blk *blk;
	unsigned int kern;
	short **blktab;
	int i, j;

	for (i=0; i<priv->blk_cnt; i++) {
		blk = &priv->blks[i];

		/* Kernel VA Table entry */
		kern = (unsigned int)blk->start;
		blktab = &priv->kern_tab[kern >> 24];

		if ( *blktab == NULL ) {
			*blktab = kmalloc(128 * sizeof(unsigned short), GFP_KERNEL);
			for(j=0; j<128; j++)
				(*blktab)[j] = -1;
		}
		(*blktab)[(kern & 0x00ffffff) >> BLK_SHIFT] = i;
	}
}

static void maplib_tabs_free(struct maplib_priv *priv)
{
	int i;

	for (i=0; i<256; i++) {
		if ( priv->kern_tab[i] ) {
			kfree(priv->kern_tab[i]);
			priv->kern_tab[i] = NULL;
		}
	}
}
#endif

static void maplib_free_blks(struct maplib_priv *priv)
{
	struct maplib_blk *blk;
	int i;

	if ( priv->blks ) {
		blk = &priv->blks[0];
		for (i=0; i<priv->blk_cnt; i++) {
			if ( blk[i].start ) {
				free_pages((unsigned long)blk[i].start,
					get_order(blk[i].length));
				blk[i].start = NULL;
			}
		}
		kfree(priv->blks);
	}
	priv->blks = NULL;
	priv->blk_cnt = 0;
}

static int maplib_open(struct inode *inode, struct file *filp)
{
	struct maplib_priv *priv;

	priv = container_of(inode->i_cdev, struct maplib_priv, cdev);

	if (priv == NULL) {
		printk (KERN_WARNING "MAPLIB: device doesnt exist\n");
		return -ENODEV;
	}

	if (priv->open != 0) {
		printk (KERN_WARNING "MAPLIB: device %d already opened\n", priv->index);
		return -EBUSY;
	}

	/* Take Device */
	priv->open = 1;
	filp->private_data = priv;

	return 0;
}

static int maplib_release (struct inode *inode, struct file *file)
{
	struct maplib_priv *priv;

	priv = container_of(inode->i_cdev, struct maplib_priv, cdev);

	/* Free buffer memory */
	maplib_free_blks(priv);

	priv->open = 0;

	return 0;
}

static int maplib_mmap(struct file *filp, struct vm_area_struct *vma)
{
	struct maplib_priv *priv;
	int i, len, result;
	struct maplib_blk *blk;
	unsigned long ustart;

	priv = (struct maplib_priv *)filp->private_data;

#ifdef MAPLIB_DEBUG
	printk(KERN_DEBUG "MAPLIB: MMAP Request\n");
	printk(KERN_DEBUG "        VM_START:         0x%x\n", (unsigned int)vma->vm_start);
	printk(KERN_DEBUG "        VM_END:           0x%x\n", (unsigned int)vma->vm_end);
	printk(KERN_DEBUG "        VM_pgoff:         0x%x\n", (unsigned int)vma->vm_pgoff);
	printk(KERN_DEBUG "        VM_private_data:  %p\n", vma->vm_private_data);
	printk(KERN_DEBUG "        VM_page_prot:     0x%x (%x,%x,%x)\n",
		(unsigned int)vma->vm_page_prot, PROT_READ, PROT_WRITE, PROT_EXEC);
	printk(KERN_DEBUG "        VM_flags:         0x%x\n", (unsigned int)vma->vm_flags);
	printk(KERN_DEBUG "        VM_ops:           0x%x\n", (unsigned int)vma->vm_ops);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(6,3,0)
	vma->vm_flags |= VM_DONTEXPAND | VM_DONTDUMP;
#else
	vm_flags_set(vma, vma->vm_flags | VM_DONTEXPAND | VM_DONTDUMP);
#endif
	vma->vm_private_data = priv;
	vma->vm_ops = &maplib_vops;

	/* First 1Mb addresses are reserved of something else */
	if (vma->vm_pgoff < (0x100000 >> PAGE_SHIFT)) {
		return -EINVAL;
	}

	/*** 1Mb - CFG_MAX Mb is Buffer Blocks ***/

	if (  (vma->vm_end - vma->vm_start) != priv->blk_tot_size ) {
		printk(KERN_DEBUG "MAPLIB: map size %x, required %x\n",
			(unsigned int)(vma->vm_end - vma->vm_start),
			(unsigned int)priv->blk_tot_size);
		return -EINVAL;
	}

	/* MAP All blocks. */
	ustart = vma->vm_start;
	for (i=0; i<priv->blk_cnt; i++) {
		/* Do the Mapping */
		len = BLK_SIZE;
		blk = &priv->blks[i];
		if ( i == (priv->blk_cnt-1) )
			len = priv->blks[i-1].length;
		result = remap_pfn_range(vma, ustart, __pa(blk->start) >> PAGE_SHIFT,
			len, vma->vm_page_prot);
		if ( result ) {
			printk(KERN_INFO "MAPLIB: Block mapping failed: %d, %p\n",
				result, blk->start);
			return -EAGAIN;
		}
#ifdef MAPLIB_DEBUG
		printk(KERN_DEBUG "MAPLIB: MAPPED 0x%x-0x%x to 0x%x-0x%x\n",
			(unsigned int)ustart, (unsigned int)(ustart+len),
			(unsigned int)__pa(blk->start), (unsigned int)__pa(blk->start)+len);
#endif
		/* Next USpace Address */
		ustart += len;
	}
	priv->ustart = vma->vm_start;
	priv->uend = vma->vm_end;
	/* Remember the User VMA that we have mapped */
	priv->user_vma = vma;

#if MAPLIB_KERN_TO_USR_LOOKUP
	/* Now that we know the User-address and Kernel addresses,
	 * the Kernel->UserAdress Table can be built.
	 */
	maplib_tabs_build(priv);
#endif

#ifdef MAPLIB_DEBUG
	printk(KERN_DEBUG "MAPLIB: suceeded in mapping memory\n");
#endif

	return 0;
}

static void maplib_vma_close(struct vm_area_struct *area)
{
	struct maplib_priv *priv = area->vm_private_data;

	/* First block is beeing unmapped, lets inform
	 * all active drivers that the memory is not
	 * longer available. The need to stop using the
	 * memory to avoid data overwrite.
	 */
#ifdef MAPLIB_DEBUG
	printk(KERN_DEBUG "MAPLIB: unmapping, telling drivers\n");
#endif
	maplib_drv_unreg_all(priv);
#if MAPLIB_KERN_TO_USR_LOOKUP
	maplib_tabs_free(priv);
#endif

	/* Mark Block as not mapped */
	priv->user_vma = NULL;
}

static int maplib_buf_setup(struct maplib_priv *priv, struct maplib_setup *setup)
{
	int i;

	if ( priv->blks ) {
		/* All Buffers must be unmapped from previous configurations.
		 */
		if ( priv->user_vma ) {
			printk(KERN_DEBUG "MAPLIB: setup: not all unmapped\n");
			return -EINVAL;
		}

		/* All unmapped since last time. We free old buffers. */
		maplib_free_blks(priv);
	}

	/* Check if request was to free buffers only */
	if ( (setup->blk_cnt < 1) || (setup->blk_size < 1) ||
	     (setup->blk_size > BLK_SIZE))
		return 0;

	/*** Setup Buffers ***/

	/* Calculate number of MMAP-Blocks required to fit all
	 * user's blocks. A MMAP block is limited to a specific
	 * size limited by the arch port. 128Kb seems standard.
	 */
	priv->usrblk_per_blk = BLK_SIZE / setup->blk_size;
	priv->blk_cnt = (setup->blk_cnt + priv->usrblk_per_blk - 1) /
				priv->usrblk_per_blk;
	priv->usrblk_size = setup->blk_size;

	/* Allocate memory for Block structures */
	priv->blks = (struct maplib_blk *)kmalloc(
			priv->blk_cnt *	sizeof(struct maplib_blk),
			GFP_KERNEL);
	memset(priv->blks, 0, priv->blk_cnt * sizeof(struct maplib_blk));

	/* Init All block structures */
	for (i=0; i<priv->blk_cnt; i++) {
		if ( i == (priv->blk_cnt-1) ) {
			/* Last Block */
			priv->blks[i].length = (priv->blk_cnt - priv->usrblk_per_blk*i);
		} else {
			priv->blks[i].length = priv->usrblk_per_blk;
		}
		priv->blks[i].length = PAGE_ALIGN(priv->blks[i].length * priv->usrblk_size);

#ifdef MAPLIB_DEBUG
		printk(KERN_DEBUG "MAPLIB: allocating Buffer block: %d, %d (%x)\n",
				i,
				priv->blks[i].length,
				(unsigned int)PAGE_ALIGN(priv->blks[i].length));
#endif
		/* Allocate Block Buffer memory. Must be linear */
		priv->blks[i].start = (void *)__get_free_pages(GFP_KERNEL,
						get_order(priv->blks[i].length));
		if ( priv->blks[i].start == NULL ) {
			/* Failed to allocate memory */
			printk(KERN_INFO "MAPLIB: Failed alloc Buffer blk: %d\n", i);
			maplib_free_blks(priv);
			return -ENOMEM;
		}
	}

	priv->user_vma = NULL;
	priv->blk_tot_size = ((priv->blk_cnt-1) * priv->blks[i-1].length) +
				priv->blks[priv->blk_cnt-1].length;

	return 0;
}

static int maplib_mmapinfo_get(struct maplib_priv *priv, struct maplib_mmap_info *info)
{
	if ( priv->blks == NULL ) {
		printk(KERN_INFO "MAPLIB: MMAP Info not available until configured\n");
		return -EINVAL;
	}

	/* Get Frame Structure MMAP Info */
	info->buf_offset = 0x00100000;
	info->buf_length = ((priv->blk_cnt-1) * BLK_SIZE) +
				priv->blks[priv->blk_cnt-1].length;
	info->buf_blk_size = BLK_SIZE;

	return 0;
}

static long maplib_ioctl(
	struct file *filp,
	unsigned int cmd,
	unsigned long arg)
{
	struct maplib_priv *priv;
	void __user *argp = (void __user *)arg;

	priv = (struct maplib_priv *)filp->private_data;

	if (priv == NULL) {
		printk (KERN_WARNING "MAPLIB: device %d doesnt exist\n", priv->index);
		return -ENODEV;
	}

	/* Verify READ/WRITE Access to user provided buffer */
	if (_IOC_DIR(cmd) & (_IOC_WRITE | _IOC_READ))
		if (!access_ok((void __user *)arg, _IOC_SIZE(cmd)))
			return -EFAULT;

	switch (cmd) {
	case MAPLIB_IOCTL_SETUP:
	{
		struct maplib_setup setup;
		if(copy_from_user(&setup, argp, sizeof(struct maplib_setup)))
			return -EFAULT;
		return maplib_buf_setup(priv, &setup);
	}

	case MAPLIB_IOCTL_MMAPINFO:
	{
		struct maplib_mmap_info mmapinfo;
		int err = maplib_mmapinfo_get(priv, &mmapinfo);
		if ( err )
			return err;
		if(copy_to_user(argp, &mmapinfo, sizeof(struct maplib_mmap_info)))
			return -EFAULT;
		break;
	}

	default:
		return -ENOSYS;
	}

	return 0;
}

static int __init maplib_init (void)
{
	int result, i;
	struct maplib_priv *priv;
	dev_t dev_id;

	maplib_privs = NULL;
	maplib_count = MAPLIB_MAX;

	/* Get Major Number of CHAR driver */
	result = alloc_chrdev_region(&dev_id, 0, maplib_count, "maplib");
	if ( result < 0 ) {
		printk(KERN_WARNING "MAPLIB: Failed to register CHAR region\n");
		return 0;
	}

	dev_major = MAJOR(dev_id);

#if LINUX_VERSION_CODE < KERNEL_VERSION(6,4,0)
	maplib_class = class_create(THIS_MODULE, "maplib");
#else
	maplib_class = class_create("maplib");
#endif
	/* Allocate private memory for all MAPLIB cores */
	maplib_privs = (struct maplib_priv *)
		kmalloc(sizeof(struct maplib_priv) * maplib_count, GFP_KERNEL);
	memset(maplib_privs, 0, sizeof(struct maplib_priv) * maplib_count);

	/* Init all private structures */
	for (i=0; i<maplib_count; i++) {
		priv = &maplib_privs[i];
		priv->index = i;
		strcat(priv->devname, "MAPLIB_N");
		priv->devname[7] = '0' + i;

		/* Init and Register CHAR driver */
		dev_id = MKDEV(dev_major, i);
		cdev_init(&priv->cdev, &maplib_fops);
		result = cdev_add(&priv->cdev, dev_id, 1);
		if ( result ) {
			printk(KERN_NOTICE "MAPLIB: Failed adding CHAR dev\n");
			continue;
		}
		device_create(maplib_class, NULL, dev_id, NULL, "maplib%d", i);
	}

	printk(KERN_DEBUG "MAPLIB: max %d mmap-pools\n", maplib_count);

	return 0;
}

static void __exit maplib_exit (void)
{
	int i;
	struct maplib_priv *priv;

	for (i = 0; i<maplib_count; i++) {
		priv = &maplib_privs[i];
		cdev_del(&priv->cdev);
	}

	for (i = 0; i < maplib_count; i++) {
		device_destroy(maplib_class, MKDEV(dev_major, i));
	}

	class_unregister(maplib_class);
	class_destroy(maplib_class);

	unregister_chrdev_region(MKDEV(dev_major, 0), maplib_count);

	if ( maplib_privs )
		kfree(maplib_privs);
	maplib_privs = NULL;
	maplib_count = 0;
#ifdef MAPLIB_DEBUG
	printk (KERN_DEBUG "MAPLIB: char driver cleaned up\n");
#endif
}

module_init(maplib_init);
module_exit(maplib_exit);

MODULE_AUTHOR("Frontgrade Gaisler AB.");
MODULE_DESCRIPTION("GRLIB Help Memory MAP Library Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:grlib-maplib");
