/*
 * Library that provides drivers with a way to have common
 * Memory MAPed buffers. For example SpW0 buffers can be used
 * zero-copy to SpW1.
 *
 * Copyright (c) 2010 Aeroflex Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#ifndef __MMAPLIB_H__
#define __MMAPLIB_H__

/*** USER IOCTL INTERFACE ***/

/* MMAP Lib setup information. Actual memory taken will be at least
 * (blk_size * blk_cnt).
 */
struct maplib_setup {
	unsigned int	blk_cnt;	/* Number of Blocks */
	unsigned int	blk_size;	/* Size of each block */
};

/* Block Buffer MMAP information */
struct maplib_mmap_info {
	unsigned int	buf_offset;	/* Offset into device memory */
	int		buf_length;	/* Total Length */
	int		buf_blk_size;	/* Buffer Block Size */
};

/* Configure Memory MAP Library, and allocate need memory,
 * all previous (if any) memory mapped pages must be unmapped
 * otherwise and error will occur.
 */
#define MAPLIB_IOCTL_SETUP	_IOR('a', 0x01, struct maplib_setup)
/* Get Current MMAP Info from Driver, this tells the user how
 * to memory map the memory into user space.
 */
#define MAPLIB_IOCTL_MMAPINFO	_IOW('a', 0x02, struct maplib_mmap_info)


/*** KERNEL INTRFACE ***/

/* Driver that uses the library */
struct maplib_drv {
	struct maplib_drv *next;		/* Used internally */
	char *name;
	void (*unmap)(int idx, void *priv);	/* Called by library when memory is unmapped */
	void *priv;
};

/* Register Driver as active user of Memory mapped memory
 *
 * This will fail if not all memory blocks has been mapped to user memory.
 * 
 * The custom unmap() function is called if the user unmap memory, if this
 * call is made the user did not close the drivers in the correct order, but to
 * prevent a bug this must exist. After this the driver may not perform
 * any memory access to the memory range.
 */
extern int maplib_drv_reg(int idx, struct maplib_drv *drv);

/* Unregister Driver */
extern void maplib_drv_unreg(int idx, struct maplib_drv *drv);

/* Translate previously MMAPed User Space Virtual Address into Kernel Address
 * Return -1 if invalid address
 *
 * Check that the complete user/kernel address is mapped and linear
 * into kernel space (and hence user space). Set LEN=0 to avoid check.
 */
extern int maplib_lookup_kern(int idx, void *usr, void **kern, void **hw, int len);

/* Translate previously MMAPed Kernel Virtual Address into UserSpace Address
 * Return -1 if invalid address
 *
 * Check that the complete user/kernel address is mapped and linear
 * into kernel space (and hence user space). Set LEN=0 to avoid check.
 */
extern int maplib_lookup_usr(int idx, void *kern, void *hw, void **usr, int len);

#endif
