/*
 * GRSPW DMA SpaceWire packet tdemonstration software
 *
 * Copyright (c) 2010 Aeroflex Gaisler AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301 USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>
#include <string.h>

#include <linux/grlib/grspw.h>
#include <linux/grlib/grspw_user.h>

#include "spwlib.h"
#include "grspwlib.h"
#include "maplib_user.h"

#undef DEBUG_TEST

#ifdef DEBUG_TEST
#define PRINT(str...) printf(str); fflush(NULL)
#define PRINT2(str...) printf(str); fflush(NULL)
#define PRINT3(str...) printf(str); fflush(NULL)
#else
#define PRINT(str...)
#define PRINT2(str...)
#define PRINT3(str...)
#endif

#define DMACHAN 1
#define DMACHAN_MASK (1<<DMACHAN)

/******** MAP LIBRARY ********/
#define MMAPLIB_POOL 0
int map_alloc_pkt_idx = 0;
int map_alloc_pkt_ofs = 0;
int map_alloc_hdr_idx = 0;
int map_alloc_hdr_ofs = 0;

/******** SpaceWire  ********/
#define TXPKTS 5461 /* Max packet structures for 128Kb */
#define RXPKTS 5461 /* Max packet structures for 128Kb */
#define SPW_PROT_ID 155

/* Number of seconds to run Test */
/*#define TEST_LENGTH_SEC (60*60*12)*/
#define TEST_LENGTH_SEC (30)

/* Number of micro seconds to wait for all packets to be received */
#define TEST_END_WAIT_MSEC 10000

/* Skip RX check to save CPU and thereby get higher throughput figures
 * all packets are checked in the end any way...
 */
#define SKIP_CHECK_FOR_EVERY_RECEIVED_PACKET

/* SpaceWire Packet first sender (INITIATOR) device Index */
#define SPW_INITIATOR 0

/* SpW links on SpaceWire Router determine which PATH addess the AMBA
 * targets has
 */
#define ROUTER_SPWLINK_NO 8

/* Number of SpaceWire Links to share the packet buffers with, each
 * SpaceWire Link have 2 Packet Queues per DMA Unit.
 */
#define SPW_LINKS_USED 4
#define SPW_DMA_CHANS_PER_LINK_USED 1 /* only 1 supported at this time */
#define SPW_DIRECTIONS_USED (SPW_LINKS_USED*SPW_DMA_CHANS_PER_LINK_USED*2)

/*#define PKT_CNT 2016*/
#define SPW_BUF_MB_BLOCKS 32
#define SPW_BUF_SIZE (SPW_BUF_MB_BLOCKS*1024*1024)
#define HDR_SIZE 16
#ifndef PKT_SIZE
  #define PKT_SIZE 1024
#endif
#define PKT_DATA_SIZE (PKT_SIZE-1-1-1-1-4) /* - sizeof(pkt_hdr)... */
#define PKT_CNT  SPW_BUF_SIZE/PKT_SIZE/SPW_DIRECTIONS_USED
#define PKT_CNT_TOT PKT_CNT*SPW_DIRECTIONS_USED

/* Make PKT_CNT packets per RX/TX channel, each packet has a 1024 Bytes 
 * large buffer. Approx 8MBytes are used with PKT_CNT=2016.
 */
struct spwlib_pool_cfg poolcfgs[3] = 
{
	SPWLIB_POOL_CFG(PKT_SIZE, HDR_SIZE, PKT_CNT_TOT),
	SPWLIB_POOL_CFG_END,
};

struct spwlib_poolset *poolset;

struct pkt_hdr {
	unsigned char addr;
	unsigned char protid;
	unsigned char dstadr_next; /* 0 or -1, stops packet */
	unsigned char resv; /* Zero for now */
	unsigned int pkt_number;
	unsigned short data[PKT_DATA_SIZE/2];
};

struct route_entry {
	unsigned char dstadr_next;
	unsigned char dstadr[16];	/* 0 terminates array */
};

#if 0
struct route_entry initial_route = 
{
	.dstadr_next = 1,
	.dstadr = {2,0},
};

#define ROUTE_MAX 256
struct route_entry routetab[ROUTE_MAX] = 
{
	{0, {0}},
	{2, {1, 0}},
	{1, {2, 0}},
};
#endif

#if (SPW_LINKS_USED == 2)
struct route_entry initial_route = 
{
	.dstadr_next = 0x1,
	.dstadr = {1+ROUTER_SPWLINK_NO+1, 0x2, 0},
};

#define ROUTE_MAX 256
struct route_entry routetab[ROUTE_MAX] = 
{
	{0, {0, 0}},
	{0x2, {1+ROUTER_SPWLINK_NO, 0x1, 0}},
	{0x1, {1+ROUTER_SPWLINK_NO+1, 0x2, 0}},
};
#elif (SPW_LINKS_USED == 4)
struct route_entry initial_route = 
{
	.dstadr_next = 0x3,
	.dstadr = {1+ROUTER_SPWLINK_NO+1, 0x2, 0},
};

#define ROUTE_MAX 256
struct route_entry routetab[ROUTE_MAX] = 
{
	/* DST */  /* NEXT-ADR, {PATH/LOGICAL ADDRESS} */
	/* 0x00 */ {0, {0, 0}},
	/* 0x01 */ {0x2, {1+ROUTER_SPWLINK_NO, 0x1, 0}},
	/* 0x02 */ {0x3, {1+ROUTER_SPWLINK_NO+1, 0x2, 0}},
	/* 0x03 */ {0x4, {1+ROUTER_SPWLINK_NO+2, 0x3, 0}},
	/* 0x04 */ {0x1, {1+ROUTER_SPWLINK_NO+3, 0x4, 0}},
};
#endif

/* Packet ID -> Packet Pointer. All Devices share the Packet ID. */
struct spwlib_pkt *pktid2pkt[PKT_CNT_TOT];
int pkt_id_next = 0;

/* Table of start contents of packets, the table is indexed using the
 * Packet Number.
 */
unsigned short pkts_content_tab[PKT_CNT];

struct grspw_device {
	struct grspw_dev dev;
	struct grspw_config cfg;
	struct spwlib_list rx_list;
	struct spwlib_list rx_prep_list;
	struct spwlib_list tx_list;
	struct spwlib_list check_list;
	struct route_entry *routetab;
	int pkt_number_next;
	int check_no;
};
struct grspw_device devs[8];

struct grspw_config dev_configs[8] = 
{
	/*** GRSPW[0] ***/
	{
		.adrcfg =
		{
			.promiscuous = 0,
			.def_addr = 0x1,
			.def_mask = 0,
			.dma_nacfg =
			{
				/* Since only one DMA Channel is used, only
				 * the default Address|Mask is used.
				 */
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
			},
		},
		.rmap_cfg = 0,		/* Disable RMAP */
		.rmap_dstkey = 0,	/* No RMAP DESTKEY needed when disabled */
		.tc_cfg = TCOPTS_EN_TX,	/* Enable TimeCode Transmission */
		.enable_chan_mask = 1,	/* Enable only the first DMA Channel */
		.chan = 
		{
			{
				.flags = DMAFLAG_NO_SPILL,
				.rxmaxlen = PKT_SIZE, /* Max 1024 bytes per packet */
				.rx_irq_en_cnt = 0, /* Disable RX IRQ generation */
				.tx_irq_en_cnt = 0, /* Disable TX IRQ generation */
			},
			/* The other 3 DMA Channels are unused */
			{
				.flags = DMAFLAG_NO_SPILL,
				.rxmaxlen = PKT_SIZE, /* Max 1024 bytes per packet */
				.rx_irq_en_cnt = 0, /* Disable RX IRQ generation */
				.tx_irq_en_cnt = 0, /* Disable TX IRQ generation */
			},
			
		},
	},

	/*** GRSPW[1] ***/
	{
		.adrcfg =
		{
			.promiscuous = 0,
			.def_addr = 0x2,
			.def_mask = 0,
			.dma_nacfg =
			{
				/* Since only one DMA Channel is used, only
				 * the default Address|Mask is used.
				 */
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
			},
		},
		.rmap_cfg = 0,		/* Disable RMAP */
		.rmap_dstkey = 0,	/* No RMAP DESTKEY needed when disabled */
		.tc_cfg = TCOPTS_EN_RX,	/* Enable TimeCode Reception */
		.enable_chan_mask = 1,	/* Enable only the first DMA Channel */
		.chan = 
		{
			{
				.flags = DMAFLAG_NO_SPILL,
				.rxmaxlen = PKT_SIZE, /* Max 1024 bytes per packet */
				.rx_irq_en_cnt = 0, /* Disable RX IRQ generation */
				.tx_irq_en_cnt = 0, /* Disable TX IRQ generation */
			},
			/* The other 3 DMA Channels are unused */
			{
				.flags = DMAFLAG_NO_SPILL,
				.rxmaxlen = PKT_SIZE, /* Max 1024 bytes per packet */
				.rx_irq_en_cnt = 0, /* Disable RX IRQ generation */
				.tx_irq_en_cnt = 0, /* Disable TX IRQ generation */
			},
		},
	},

	/*** GRSPW[2] ***/
	{
		.adrcfg =
		{
			.promiscuous = 0,
			.def_addr = 0x3,
			.def_mask = 0,
			.dma_nacfg =
			{
				/* Since only one DMA Channel is used, only
				 * the default Address|Mask is used.
				 */
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
			},
		},
		.rmap_cfg = 0,		/* Disable RMAP */
		.rmap_dstkey = 0,	/* No RMAP DESTKEY needed when disabled */
		.tc_cfg = TCOPTS_EN_RX,	/* Enable TimeCode Reception */
		.enable_chan_mask = 1,	/* Enable only the first DMA Channel */
		.chan = 
		{
			{
				.flags = DMAFLAG_NO_SPILL,
				.rxmaxlen = PKT_SIZE, /* Max 1024 bytes per packet */
				.rx_irq_en_cnt = 0, /* Disable RX IRQ generation */
				.tx_irq_en_cnt = 0, /* Disable TX IRQ generation */
			},
			/* The other 3 DMA Channels are unused */
		},
	},

	/*** GRSPW[3] ***/
	{
		.adrcfg =
		{
			.promiscuous = 0,
			.def_addr = 0x4,
			.def_mask = 0,
			.dma_nacfg =
			{
				/* Since only one DMA Channel is used, only
				 * the default Address|Mask is used.
				 */
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
			},
		},
		.rmap_cfg = 0,		/* Disable RMAP */
		.rmap_dstkey = 0,	/* No RMAP DESTKEY needed when disabled */
		.tc_cfg = TCOPTS_EN_RX,	/* Enable TimeCode Reception */
		.enable_chan_mask = 1,	/* Enable only the first DMA Channel */
		.chan = 
		{
			{
				.flags = DMAFLAG_NO_SPILL,
				.rxmaxlen = PKT_SIZE, /* Max 1024 bytes per packet */
				.rx_irq_en_cnt = 0, /* Disable RX IRQ generation */
				.tx_irq_en_cnt = 0, /* Disable TX IRQ generation */
			},
			/* The other 3 DMA Channels are unused */
		},
	},
};

int grspw_init(int idx)
{
	struct grspw_bufcfg bufcfg;
	struct grspw_device *device = &devs[idx];
	struct grspw_dev *dev = &device->dev;
	struct grspw_link_state ls;
	struct grspw_link_ctrl lctrl;

	memset(device, 0, sizeof(struct grspw_device));
	if ( grspw_open(dev, idx) ) {
		printf("Failed to open GRSPW device %d: %d\n", idx, errno);
		return -1;
	}

	/* Buffer Configuration */
	bufcfg.rx_pkts = RXPKTS;
	bufcfg.tx_pkts = TXPKTS;
	if ( grspw_bufcfg_set(dev, &bufcfg) ) {
		grspw_close(dev);
		return -1;
	}

	/* Print out current settings */
	if ( grspw_cfg_get(dev, &device->cfg) ) {
		grspw_close(dev);
		return -1;
	}
	printf("\n\n---- DEFAULT CONFIGURATION FROM DRIVER/HARDWARE ----\n");
	grspw_cfg_print(&dev->hwsup, &device->cfg);
	if ( grspw_cfg_set(dev, &dev_configs[dev->index]) ) {
		grspw_close(dev);
		return -1;
	}
	printf("\n\n---- APPLICATION CONFIGURATION ----\n");
	device->cfg = dev_configs[dev->index];
	grspw_cfg_print(&dev->hwsup, &device->cfg);
	printf("\n\n");

	/* This will result in an error if only one port available */
	if ( dev->hwsup.nports < 2 ) {
		if ( grspw_port_set(dev, 1) == 0 ) {
			printf("Succeeded to select port1, however only one PORT on dev %d!\n", dev->index);
			return -1;
		} else {
			printf("PORT TEST SUCCEEDED, ERROR PRINTOUT IS EXPECTED\n");
		}
	}

	printf("\n\nBefore Link Start:\n");
	if ( grspw_linkstate_get(dev, &ls) ) {
		printf("Failed to get Link State for dev %d\n", dev->index);
		return -1;
	}
	grspw_linkstate_print(&ls);

	/* Try to bring Link Up */
	lctrl.ctrl = LINKOPTS_ENABLE | LINKOPTS_AUTOSTART | LINKOPTS_START;
	lctrl.clkdiv_start = 2;
	lctrl.clkdiv_run = 0;
	if ( grspw_link_set(dev, &lctrl) != 0 ) {
		printf("Succeeded to select port1, however only one PORT on dev %d!\n", dev->index);
		return -1;
	}

	printf("\n\nAfter Link Start:\n");
	if ( grspw_linkstate_get(dev, &ls) ) {
		printf("Failed to get Link State for dev %d\n", dev->index);
		return -1;
	}
	grspw_linkstate_print(&ls);

	if ( (dev->hwsup.hw_version >> 16) == GAISLER_SPW2_DMA ) {
		printf("\n\n###NOTE: running on SPW-ROUTER DMA SpaceWire link:\n");
		printf(    "         there is no Link-state available\n");
	}

	if ( grspw_stats_clear(dev) ) {
		return -1;
	}

	return 0;
}

void grspw_cleanup(int idx)
{
	struct grspw_device *device = &devs[idx];
	struct grspw_dev *dev = &device->dev;

	if ( dev->fd != -1 ) {
		grspw_stop(dev);
		grspw_close(dev);
	}
}

#if 0
/* Memory MAP memory to use for example */
int mmaplib_init(int idx, int blk_size, int blk_cnt)
{
	int i;
	void *adr;
	unsigned int start, end, len;
	struct maplib_mmap_info *mapi = &mmapinfo;
	char devname[32];
	struct maplib_setup setup;

	sprintf(devname, "/dev/maplib%d", idx);
	map_fd = open(devname, O_RDWR);
	if ( map_fd < 0 ) {
		printf("Failed to open MMAPLib\n");
		return -1;
	}

	/* Setup MAP */
	setup.blk_size = blk_size;
	setup.blk_cnt = blk_cnt;
	if ( ioctl(map_fd, MAPLIB_IOCTL_SETUP, &setup) ) {
		printf("Failed to Setup MMAPLib: %d\n", errno);
		return -2;
	}

	/* Get MMAP information calculated by driver */
	if ( ioctl(map_fd, MAPLIB_IOCTL_MMAPINFO, mapi) ) {
		printf("Failed to get MMAPINFO, errno: %d\n", errno);
		return -3;
	}

	printf("MMAP INFO: GRSPW Packet Buffers:\n");
	printf("    offset: 0x%x\n", mapi->buf_offset);
	printf("    length: %d (0x%x)\n", mapi->buf_length, mapi->buf_length);
	printf("    MMAP Block size: 0x%x\n", mapi->buf_blk_size);

	memset(maps, 0, sizeof(maps));

	/* Map all SpaceWire Packet Buffers */
	start = mapi->buf_offset;
	end = mapi->buf_offset + mapi->buf_length;

	/* Memory MAP driver's Buffers READ-and-WRITE  */
	adr = mmap(NULL, mapi->buf_length, PROT_READ|PROT_WRITE, MAP_SHARED,
			map_fd, start);
	if ( (unsigned int)adr == 0xffffffff ) {
		printf("MMAP Bufs Failed: %p, errno %d, %x\n", adr, errno, mapi->buf_length);
		return -4;
	}
	i=0;
	while ( start < end ) {
		/* Calc length of MMAP Block */
		if ( (end - start) >= mapi->buf_blk_size )
			len = mapi->buf_blk_size;
		else
			len = end - start;

		maps[i].start = adr;
		maps[i].length = len;
		maps[i].ofs = start;
		/*printf("MAPPED OFS=%p, BASE=%x, LEN=%p\n", maps[i].ofs, maps[i].start, maps[i].length);*/
		i++;

		/* Next Block */
		start += mapi->buf_blk_size;
		adr += len;
	}

	printf("M-Mapping successful\n");

	return 0;
}

int mmaplib_cleanup(void)
{
	/* Close MAP Lib, this will automatically unmap all memory for us. */
	close(map_fd);

	maps[0].length = 0;
	maps[0].start = 0;
	maps[0].ofs = 0;
}
#endif

/* It is important that block boundaries are not overlapped in memory alllcated
 * here.
 */
void *spw_alloc_pkt(void *data, int poolno, int hdr, int length)
{
	struct map *map;
	void *retval;
	int *map_alloc_idx;
	int *map_alloc_ofs;

	if ( hdr ) {
		/* It is assumed that the header will never fill a MAP 
		 * block, so the range check (map->length==0) is not
		 * completely correcy... however will work. */
		map_alloc_idx = &map_alloc_hdr_idx;
		map_alloc_ofs = &map_alloc_hdr_ofs;
	} else {
		map_alloc_idx = &map_alloc_pkt_idx;
		map_alloc_ofs = &map_alloc_pkt_ofs;
	}
try:
	map = &maps[*map_alloc_idx];
	if ( (length > map->length) || (map->length == 0) )
		return NULL;
	if ( map->length < (length + *map_alloc_ofs) ) {
		(*map_alloc_idx)++;
		*map_alloc_ofs = 0;
		goto try;
	} else {
		retval = (void *)((unsigned int)map->start + *map_alloc_ofs);
		(*map_alloc_ofs) += length;
	}
	/*printf("ALLOCED: %p %d\n", retval, length);*/
	return retval;
}

void spw_alloc_reset(void *data)
{
#if HDR_SIZE > 0
	/* Let headers have first MAP block */
	map_alloc_pkt_idx = 1;
	map_alloc_pkt_ofs = 0;
#else
	/* No headers: Let headers have NO MAP block */
	map_alloc_pkt_idx = 0;
	map_alloc_pkt_ofs = 0;
#endif

	/* Let Header start at index 0. */
	map_alloc_hdr_idx = 0;
	map_alloc_hdr_ofs = 0;
}

int pkt_pool_setup(void)
{
	struct spwlib_alloc_cfg memalgo;

	printf("\n\n---- SETTING UP POOLS ----\n");

	memalgo.data = NULL;
	memalgo.alloc = spw_alloc_pkt;
	memalgo.reset = spw_alloc_reset;

	poolset = spwlib_poolset_alloc(&poolcfgs[0]);
	if ( poolset == NULL ) {
		printf("Failed to allocate poolset\n");
		return -1;
	}

	if ( spwlib_poolset_pkt_alloc(poolset, &memalgo) ) {
		printf("Failed to allocate pool packets\n");
		return -2;
	}

	spwlib_poolset_print(poolset);

	return 0;
}

/* Set:
 *   - packet destination address
 *   - NEXT packet destination address
 *   - Header length
 */
void pkt_init_hdr(struct spwlib_pkt *pkt, struct route_entry *route)
{
	int i;
	struct pkt_hdr *pkt_hdr = (struct pkt_hdr *)pkt->data;
	unsigned char *hdr = pkt->hdr;

	pkt_hdr->dstadr_next = route->dstadr_next;

	/* If path addressing we put non-first Destination Addresses in 
	 * header. route->dstadr[0] is always non-zero.
	 */
	i = 0;
	while ( route->dstadr[i+1] != 0 ) {
		hdr[i] = route->dstadr[i];
		i++;
	}
	/* Put last address in pkthdr->addr */
	pkt->hlen = i;
	pkt_hdr->addr = route->dstadr[i];
}

/* Packets are used differently for different devices:
 * The Initiator:
 *   - Starts by preparing PKT_CNT packets with data according to a pattern
 *   - A receive queue is prepared with PKT_CNT packets for receiving data
 *   - All received packets are put last in the transmit queue for
 *     retransmission.
 *
 * The Responder:
 *   - Prepares PKT_CNT*2 packets for receiving
 *   - No TX packets are created
 *   - When a Packet is Received that packet is scheduled for transmission
 *     in the same order as received in.
 *
 */
int pkts_init(struct grspw_device *device, int initiator)
{
	int i, cnt, pkt_number;
	unsigned short pkt_content;
	struct spwlib_pkt *pkt;
	struct pkt_hdr *pkt_hdr;

	spwlib_list_clr(&device->rx_list);
	spwlib_list_clr(&device->rx_prep_list);
	spwlib_list_clr(&device->tx_list);
	spwlib_list_clr(&device->check_list);

	if ( initiator ) {
		/* Initiator */

		/* Allocate PKT_CNT 1k Packets for RX/TX buffers */
		cnt = spwlib_pkt_chain_take(poolset, PKT_SIZE, HDR_SIZE, PKT_CNT, &device->rx_prep_list);
		if ( cnt != PKT_CNT ) {
			printf("Failed allocating RX packets for %d\n", initiator);
			return -1;
		}
		cnt = spwlib_list_cnt(&device->rx_prep_list);
		if ( cnt != PKT_CNT ) {
			printf("pkts_init: ERROR1 rxlist: %d\n", cnt);
			return -2;
		}

		/* Allocate PKT_CNT Packets for TX buffers */
		cnt = spwlib_pkt_chain_take(poolset, PKT_SIZE, HDR_SIZE, PKT_CNT, &device->tx_list);
		if ( cnt != PKT_CNT ) {
			printf("Failed allocating TX packets for %d\n", initiator);
			return -1;
		}
		cnt = spwlib_list_cnt(&device->tx_list);
		if ( cnt != PKT_CNT ) {
			printf("pkts_init: ERROR2 txlist: %d\n", cnt);
		}

		/* Initialize TX Packets with some pattern */
		pkt_number = 0;
		pkt_content = 0;
		pkt = device->tx_list.head;
		while ( pkt ) {
			PRINT("pkt_init6a %d pkt=%p\n", pkt_number, pkt);
			if ( HDR_SIZE > 0 )
				memset(pkt->hdr, 0, HDR_SIZE);

			pkt_hdr = (struct pkt_hdr *)pkt->data;
			pkt_hdr->protid = SPW_PROT_ID;
			pkt_hdr->resv = 0;
			/* Set PKT-ID to index */
			pkt_hdr->pkt_number = pkt_number;
			pkts_content_tab[pkt_number] = pkt_content;

			/* Init Data */
			for (i=0; i<PKT_DATA_SIZE/2; i++) {
				pkt_hdr->data[i] = pkt_content++;
			}

			/* Init Header and DST address in data */
			pkt_init_hdr(pkt, &initial_route);

			pkt_number++;
			pkt->dlen = PKT_SIZE;
			pkt = pkt->next;
		}
	} else {
		/* Allocate PKT_CNT*2 Packets for RX/TX buffers */
		cnt = spwlib_pkt_chain_take(poolset, PKT_SIZE, HDR_SIZE, PKT_CNT*2, &device->rx_prep_list);
		if ( cnt != PKT_CNT*2 ) {
			printf("Failed allocating RX packets for %d\n", initiator);
			return -1;
		}
		cnt = spwlib_list_cnt(&device->rx_prep_list);
		if ( cnt != PKT_CNT*2 ) {
			printf("pkts_init: ERROR3 txlist: %d\n", cnt);
		}
	}

	/* Clear RX packets */
	pkt = device->rx_prep_list.head;
	while ( pkt ) {
		if ( HDR_SIZE > 0 )
			memset(pkt->hdr, 0, HDR_SIZE);
		memset(pkt->data, 0, PKT_SIZE);
		pkt = pkt->next;
	}

	/*** Assign all packets a Packet-ID used in GRSPW Driver. ***/
	pkt = device->tx_list.head;
	while ( pkt ) {
		pkt->user_data[0] = pkt_id_next;
		pktid2pkt[pkt_id_next] = pkt;
		pkt_id_next++;
		pkt = pkt->next;
	}

	pkt = device->rx_prep_list.head;
	while ( pkt ) {
		pkt->user_data[0] = pkt_id_next;
		pktid2pkt[pkt_id_next] = pkt;
		pkt_id_next++;
		pkt = pkt->next;
	}

	grspw_assign_id2pkt(&device->dev, pktid2pkt);

	return 0;
}

struct route_entry *get_next_route(struct route_entry *tab, struct spwlib_pkt *pkt)
{
	struct pkt_hdr *hdr = pkt->data;
	int dstadr = hdr->dstadr_next;
	struct route_entry *r;

	if ( dstadr >= ROUTE_MAX ) {
		printf(" ROUTE %d does not exist\n", dstadr);
		return NULL;
	}

	r = &tab[dstadr];
	if ( r->dstadr_next == 0 ) {
		printf(" dstadr_next = 0, %d, %p, %p\n", dstadr, pkt->data, &hdr->dstadr_next);
		return NULL;
	}

	return r;
}

void pktlist_set_flags(struct spwlib_list *lst, unsigned short flags)
{
	struct spwlib_pkt *pkt = lst->head;
	while ( pkt ) {
		pkt->flags = flags;
		pkt = pkt->next;
	}
}

int debug_test(int got, int exp, void *adr)
{
	static int i = 0;
	asm volatile("ta 0x01\n\t"::);
	return i++;
}

int pktlist_check_rx(struct grspw_device *device, struct spwlib_list *lst, int content)
{
	struct spwlib_pkt *pkt = lst->head;
	struct pkt_hdr *pkt_hdr;
	unsigned short exp, got;
	int i;

	device->check_no++;
	while ( pkt ) {
		/* Check Reception HW-Flags */
		if ( (pkt->flags & (RXPKT_FLAG_TRUNK|RXPKT_FLAG_EEOP|RXPKT_FLAG_RX))
		     != RXPKT_FLAG_RX ) {
			printf("PKT RX ERROR: 0x%08x\n", pkt->flags);
			return -1;
		}

		/* Check Packet number (Sequence check) */
		pkt_hdr = (struct pkt_hdr *)pkt->data;
		if ( pkt_hdr->pkt_number != device->pkt_number_next ) {
			printf("PKT RX SEQ ERROR: exp %d, got %d\n",
				device->pkt_number_next, pkt_hdr->pkt_number);
			return -2;
		}
		/* Check Packet Content */
		if ( content ) {
			exp = pkts_content_tab[pkt_hdr->pkt_number];
			for (i=0; i<PKT_DATA_SIZE/2; i++) {
				got = pkt_hdr->data[i];
				if ( exp != got ) {
					debug_test(got, exp, &pkt_hdr->data[i]);
					printf(" BAD DATA in PACKET: got 0x%04x expected 0x%04x (%p)\n",
						got, exp, &pkt_hdr->data[i]);
					return -1;
				}
				exp++;
			}
		}
		/* Prepare next Packet content */
		device->pkt_number_next++;
		if (device->pkt_number_next >= PKT_CNT)
			device->pkt_number_next = 0;
#if 0
		if ( device->dev.index == 1 ) {
			huge_bus_ofs += sprintf(&huge_buf[huge_bus_ofs], " %03d: FLG=%x,NUM=%d\n", device->check_no, pkt->flags, pkt_hdr->pkt_number);
		}
#endif
		pkt = pkt->next;
	}
	return 0;
}

/* Process:
 *   1. Take transmitted packets and prepare for recepion
 *   2. Get received packets
 *   2b. If STOP=1, Put Packets to check list, then return
 *   3. Prepare received packets for transmission by setting new header.
 *   4. Put prepared packets last in tx-queue
 *   5. Schedule packets in tx-queue
 */
int grspw_process(struct grspw_device *device, int options, int stop)
{
	struct spwlib_list lst;
	struct spwlib_pkt *pkt;
	struct route_entry *route;
	int result;
	struct grspw_dev *dev = &device->dev;

	/* 1. */
	spwlib_list_clr(&lst);
	result = grspw_tx_reclaim(dev, DMACHAN_MASK, &lst);
	if ( result < 0 ) {
		printf("Reclaim failed: %d (%d)\n", result, dev->index);
		return -1;
	}
	if ( spwlib_list_is_empty(&lst) == 0 ) {
		/* Check TX Flags... optimized, done in grspw_tx_reclaim() for
		 * us.
		 */

		/* Clear TX Packet Flags */
		pktlist_set_flags(&lst, 0);

		spwlib_list_append_list(&device->rx_prep_list, &lst);
	}
	PRINT("1E\n");

	if ( spwlib_list_is_empty(&device->rx_prep_list) == 0 ) {
		result = grspw_rx_prepare(dev, DMACHAN, &device->rx_prep_list);
		if ( result ) {
			printf("Prepare failed: %d (%d)\n", result, dev->index);
			return -5;
		}
		spwlib_list_clr(&device->rx_prep_list);
	}

	/* 2. */
	PRINT("2A\n");
	spwlib_list_clr(&lst);
	result = grspw_rx_recv(dev, DMACHAN_MASK, &lst);
	if ( result ) {
		printf("Receive failed: %d (%d)\n", result, dev->index);
		return -10;
	}
#ifndef SKIP_CHECK_FOR_EVERY_RECEIVED_PACKET
	/* Check Packet Sequence Number and HW-Transmission Flags */
	if ( spwlib_list_is_empty(&lst) == 0 ) {
		if ( pktlist_check_rx(device, &lst, 0) ) {
			printf(" ERROR In Packet\n");
			return -100;
		}
	}
#endif
	if ( stop ) {
		if ( spwlib_list_is_empty(&lst) == 0 ) {
			spwlib_list_append_list(&device->check_list, &lst);
		}
		return 0;
	}
	PRINT("2B\n");
	if ( spwlib_list_is_empty(&lst) == 0 ) {
		/* 3. */
		PRINT("3\n");
		pkt = lst.head;
		while ( pkt ) {
			/* Clear RX Flags */
			pkt->flags = 0;

			/* Find route entry */
			if ( (route=get_next_route(device->routetab, pkt)) == NULL ) {
				printf("Failed to get route for packet\n");
				return -15;
			}

			/* Init Packet header accordingly */
			pkt_init_hdr(pkt, route);
			pkt = pkt->next;
		}
		spwlib_list_append_list(&device->rx_list, &lst);

		/* 4. */
		PRINT("4\n");
		spwlib_list_append_list(&device->tx_list, &device->rx_list);
		spwlib_list_clr(&device->rx_list);
	}

	/* 5. */
	PRINT("5A\n");
	if ( (options & 2) && spwlib_list_is_empty(&device->tx_list) == 0 ) {
		PRINT("5B\n");
		result = grspw_tx_send(dev, DMACHAN, &device->tx_list);
		if ( result < 0 ) {
			printf("Send failed: %d (%d)\n", result, dev->index);
			return -20;
		}
		spwlib_list_clr(&device->tx_list);
	}

	return 0;
}

int main(void)
{
	int i, result, shutdown, stop, devno, options, tot;
	struct timeval t0, t1, t2;
	double time;
	int hdrblk = 0;
	int initiator, loopcnt, do_stop, cnt;
	struct grspw_stats stats;
	struct grspw_qpktcnt qpktcnt;

	for (i=0; i<SPW_LINKS_USED; i++)
		devs[i].dev.fd = -1;

	printf("Test built for SpW-router with %d SpW Ports and at least %d AMBA ports.\n"
	       "Note that the SpW-links will not be used.\n",
	       ROUTER_SPWLINK_NO, SPW_LINKS_USED);
	printf("\n\n");

	/* Allocate SPW_BUF_MB_BLOCKS Mbyte Memory mapped memory in blocks
	 * with maximal block size 128kB.
	 * One extra block is allocated for header Data (a of waste of memory..)
	 */
#if HDR_SIZE > 0
	hdrblk = 1;
#endif
	if ( mmaplib_init(MMAPLIB_POOL, 128*1024, 8*SPW_BUF_MB_BLOCKS+hdrblk) ) {
		return 1;
	}

	for ( i=0; i<SPW_LINKS_USED; i++) {
		if ( grspw_init(i) ) {
			printf("Failed to initialize GRSPW%d\n", i);
			return 2;
		}
		fflush(NULL);
	}

	result = pkt_pool_setup();
	if ( result < 0 ) {
		printf("Packet Pool initialization failed: %d\n", result);
		printf("  SPW_BUF_SIZE: %d\n", SPW_BUF_SIZE);
		printf("  HDR_SIZE: %d\n", HDR_SIZE);
		printf("  PKT_SIZE: %d\n", PKT_SIZE);
		printf("  PKT_DATA_SIZE: %d\n", PKT_DATA_SIZE);
		printf("  PKT_CNT_TOT: %d\n", PKT_CNT_TOT);
		printf("  PKT_CNT: %d\n", PKT_CNT);
		return 3;
	}
	printf("PKT_CNT:             %d\n", PKT_CNT);
	printf("PKT_CNT_TOT:         %d\n", PKT_CNT_TOT);
	printf("PKT_SIZE:            %d\n", PKT_SIZE);
	printf("SPW_LINKS_USED:      %d\n", SPW_LINKS_USED);
	printf("DMA_CHANS_PER_LINK:  %d\n", SPW_DMA_CHANS_PER_LINK_USED);
	printf("DIRECTIONS:          %d\n", SPW_DIRECTIONS_USED);

	/* Set up the simplest case:
	 *   - Two nodes
	 *   - initiator (SPW0) with address 1
	 *   - responder (SPW1) with address 2
	 */
	for (i=0; i<SPW_LINKS_USED; i++) {
		if ( i == SPW_INITIATOR )
			initiator = 1;
		else
			initiator = 0;

		/* Initialize Packets for SPW[N] */
		printf("Initializing Packets for SpW%d. (Initiator=%s)\n", i,
			initiator ? "YES" : "NO");

		result = pkts_init(&devs[i], initiator);
		if ( result < 0 ) {
			printf("Packet initialization failed SpW%d: %d\n", 
				i, result);
			return 4;
		}

		/* Assign route table for SPW devices. All use the same "routing
		 * table".
		 */
		devs[i].routetab = &routetab[0];

		/* Init Packet Check */
		devs[i].pkt_number_next = 0;
		devs[i].check_no = 0;
	}

	printf("\n\n");
	for (i=0; i<SPW_LINKS_USED; i++) {
		printf("SpW%d Number of packets:\n", i);
		cnt = spwlib_list_cnt(&devs[i].rx_list);
		printf( "      rx_list: %d\n", cnt);
		cnt = spwlib_list_cnt(&devs[i].rx_prep_list);
		printf( "      rx_prep_list: %d\n", cnt);
		cnt = spwlib_list_cnt(&devs[i].tx_list);
		printf( "      tx_list: %d\n", cnt);
		cnt = spwlib_list_cnt(&devs[i].check_list);
		printf( "      check_list: %d\n", cnt);		
	}

	printf("\n\nStarting SpW DMA channels\n");
	for ( i=0; i<SPW_LINKS_USED; i++) {
		printf("Starting SpW%d: ", i);
		fflush(NULL);
		if ( grspw_start(&devs[i].dev) ) {
			printf("Failed to initialize SpW%d\n", i);
			return 2;
		}
		printf("Started Successfully\n");
	}

	printf("Starting Packet processing loop, will take approx %d secs\n",
		TEST_LENGTH_SEC);

	/* Packet processing loop */
	options = 1; /* Only allow RX prepare first time */
	devno = -1;
	shutdown = 0;
	loopcnt = 0;
	stop = 0;

	gettimeofday(&t0, NULL);
	while ( shutdown == 0 ) {

		/* Check is 10s has gone now and then... */
		if ( (stop == 0) && ((loopcnt & 0x3f) == 0x3f) ) {
			gettimeofday(&t1, NULL);
			if ( t1.tv_sec > (t0.tv_sec + TEST_LENGTH_SEC) ) {
				stop = 1;
				/* Calculate when to stop */
				t1.tv_usec += ((TEST_END_WAIT_MSEC % 1000) * 1000);
				if ( t1.tv_usec > 1000000 ) {
					t1.tv_usec -= 1000000;
					t1.tv_sec++;
				}
				t1.tv_sec += TEST_END_WAIT_MSEC / 1000;
			}
		} else if ( stop && ((loopcnt & 0x1f) == 0x1f) ) {
			/* Check if INITIATOR:
			 *   - TX_SCHEDULE Count is Zero (nothing more to send)
			 *   - Number of Received == Number of Transmitted
			 */
			if ( stop == 1 ) {
				if ( grspw_qpktcnt_get(&devs[SPW_INITIATOR].dev, &qpktcnt) ) {
					printf("Failed to get Driver Packet Queue info\n");
					return -1;
				}
				if ( qpktcnt.chan[0].tx_sched == 0 )
					stop = 2;
			}
			if ( stop == 2 ) {
				/* Stop if all packets we sent has been collected
				 * back at the receiver.
				 */
				grspw_stats_get(&devs[SPW_INITIATOR].dev, &stats);
				if ( stats.chan[SPW_INITIATOR].rx_pkts == 
				     stats.chan[SPW_INITIATOR].tx_pkts )
					stop = 3;
			}
			if ( (stop > 2) && (stop++ >= 6) ) {
				/* Stop 3 loops after the conditions above
				 * are true.
				 */
				break;
			}

			/* Stop Collect Process by time, this is to avoid
			 * endless hang if not all packets are received
			 */
			gettimeofday(&t2, NULL);

			if ( (t2.tv_sec > t1.tv_sec) ||
			     ((t2.tv_sec == t1.tv_sec) && (t2.tv_usec > t1.tv_usec)))
				break;
		}

		for ( i=0; i<SPW_LINKS_USED; i++) {
			/*printf("PROCESS%d %d, %d\n", j, stop, i);*/

			/* Only Initiator collect the packets in the end. */
			do_stop = 0;
			if ( (i == SPW_INITIATOR) && stop )
				do_stop = 1;

			shutdown = grspw_process(&devs[i], options, do_stop);
			if ( shutdown ) {
				printf("Shutting down: \n");
				devno = i;
				break;
			}
		}
		options = 3;
		loopcnt++;
	}

	/* Get Stats of Channel 0, this is to determine how many packets have
	 * been received.
	 */
	grspw_stats_get(&devs[SPW_INITIATOR].dev, &stats);
	tot = stats.chan[SPW_INITIATOR].rx_pkts;

	/* Stop time */
	gettimeofday(&t1, NULL);

	printf("\n\nShutting down: %d (SpW %d), loops=%d, stop=%d\n\n", shutdown, devno, loopcnt, stop);
	for (i=0; i<SPW_LINKS_USED; i++) {
		printf("\n\n--- SpW%d Device ---\n", i);
		printf(" RX List count: %d\n", spwlib_list_cnt(&devs[i].rx_list));
		printf(" RX PREP List count: %d\n", spwlib_list_cnt(&devs[i].rx_prep_list));
		printf(" TX List count: %d\n", spwlib_list_cnt(&devs[i].tx_list));
		printf(" Check List count: %d\n", spwlib_list_cnt(&devs[i].check_list));
		if ( grspw_qpktcnt_get(&devs[i].dev, &qpktcnt) ) {
			printf("Failed to get Driver Packet Queue info\n");
			return -1;
		}
		printf(" DRVQ RX_READY: %d\n", qpktcnt.chan[0].rx_ready);
		printf(" DRVQ RX_SCHED: %d\n", qpktcnt.chan[0].rx_sched);
		printf(" DRVQ RX_RECV: %d\n", qpktcnt.chan[0].rx_recv);
		printf(" DRVQ TX_SEND: %d\n", qpktcnt.chan[0].tx_send);
		printf(" DRVQ TX_SCHED: %d\n", qpktcnt.chan[0].tx_sched);
		printf(" DRVQ TX_SENT: %d\n", qpktcnt.chan[0].tx_sent);

		grspw_stats_get(&devs[i].dev, &stats);
		grspw_stats_print(&devs[i].dev, &stats);
	}
	printf(" t0.tv_sec:     %u\n", (unsigned int)t0.tv_sec);
	printf(" t0.tv_usec:    %d\n", t0.tv_usec);
	printf(" t1.tv_sec:     %u\n", (unsigned int)t1.tv_sec);
	printf(" t1.tv_usec:    %d\n", t1.tv_usec);
	time = (float)t1.tv_sec + (float)t1.tv_usec/1000000;
	time = time - (float)t0.tv_sec - (float)t0.tv_usec/1000000;
	printf(" Time:          %f seconds\n", time);
	printf(" Packets sent:  %d packets\n", tot);
	printf(" Packet size:   %d bytes in total\n", PKT_SIZE);
	printf(" Throughput:    %f packets/sec\n", ((float)tot)/time);
	printf(" Throughput:    %f bytes/sec\n", ((((float)tot)*PKT_SIZE))/time);
	printf(" Throughput:    %f bits/sec\n", 8 * ((((float)tot)*PKT_SIZE)/time));

	/* Check Content and Packet number of resulting Received Packets
	 * Since we stopped transmitting abrupt, we look at the sequence number
	 * of first packet to init packet checker.
	 */
	if ( spwlib_list_is_empty(&devs[SPW_INITIATOR].check_list) == 0 ) {
		struct pkt_hdr *pkt_hdr = (struct pkt_hdr *)
			devs[SPW_INITIATOR].check_list.head->data;
		devs[SPW_INITIATOR].pkt_number_next = pkt_hdr->pkt_number;

		result = pktlist_check_rx(&devs[SPW_INITIATOR],
					&devs[SPW_INITIATOR].check_list,
					1);
		if ( result ) {
			printf("###PACKET CONTENT/PKT-SEQUENCE CHECK FAILED: %d\n", result);
		} else {
			printf("PACKET CONTENT/PKT-SEQUENCE CHECK SUCCESSFUL\n");
		}
	} else {
		printf("###NO RECEIVED PACKETS IN CHECKLIST: PKT CHECK FAILED\n");
	}

	for ( i=0; i<SPW_LINKS_USED; i++) {
		grspw_cleanup(i);
	}

	mmaplib_cleanup();

	return 0;
}

