#!/bin/bash

usage() {
    echo "Usage:"
    echo "$0 <linux-image> <out-file> [-freq <frequency>] [-base <baseaddr>] [-cmdline <string>] [-amp <string>] [-ethmac <string>] [-ipi irq_num]"
    echo "  <linux-image> linux image, normally <linux-dir>/arch/sparc/boot/image."
    echo "  <out-file>    output image that can be uploaded using GRMON or run in TSIM."
    echo ""
    echo " optional parameters:"
    echo "  -base <baseaddr>  optional baseaddress. The default is 0x40000000."
    echo "  -uart <baseaddr>  uart base addr"
    echo "  -cmdline <string> kernel parameter string. Default is \"console=ttyS0,38400\"."
    echo "  -freq <frequency> optional frequency parameter in case if it cannot be retrived from the Timer scalar."
    echo "  -amp <string>     optional strng of format <idx0>=<val0>:<idx1>=<val1>:... that sets for core index"
    echo "                    n property ampopts to value n. Example 0=1:1:0 will set core index 0's ampopts to 1"
    echo "		      and core index 1's ampopts to 0."
    echo "  -ethmac <string>  set the ethernet mac address as 12 dgt hex. Default: 00007ccc0145"
    echo "  -ipi <irq_num>    IRQ number used by Linux SMP for IPIs. May not be shared. Allowed values: 1..14"
    echo "  -allocinmem       Allocate openprom structures in memory instead of repeated recursive scanning"
    echo "  -xml <file>       Specify a xml file that adds extra properties and nodes to the device tree."
    echo "                    The -xml switch only functions if the -allocinmem switch is choosen."
    
    exit 1;
}

# The xml files format:
# tags:
#   add-property
#   add-node
#   property
#
# <add-property>:
#   Add properties to a choosen node(s).
#   There are 2 variants to:
#    <add-property vendor="0x1" device="0xc" idx="0" >
#     selects the node to add properties to by vendor:device:idx
#     where <idx> is the amba-scan index
#    <add-property parent="4" >
#     selects the node to add properties to by amba-scan index <parent>
#
# <add-node>:
#   Add node subtree to a choosen node(s) as child.
#   There are 2 variants to:
#    <add-node vendor="0x1" device="0xc" idx="0" >
#     selects the node to add node subtree to by vendor:device:idx
#     where <idx> is the amba-scan index
#    <add-node parent="4" >
#     selects the node to add node subtree to by amba-scan index <parent>
#
# <property name="[name]" type="[int|str]" value="val" >:
#   Add a property by name=value. 2 types are supported:
#   "int" : adds a integer, specified by "value"
#   "str" : adds a string, specified by "value"
#
# 
# example xml:
# --------------------- start ----------------------------
#<!-- add property by vendor:device:idx -->
#<add-property vendor="0x1" device="0xc" idx="0" >
#  <property name="add-prop-1.c.test1" type="int" value="1024"/>
#  <property name="add-prop-1.c.test2" type="str" >teststr</property>
#</add-property>
#
#<!-- add property by scan idx -->
#<add-property parent="4"  >
#  <property name="add-prop-p4.test1" type="int" value="1024"/>
#  <property name="add-prop-p4.test2" type="str" >teststr</property>
#</add-property>
#
#<!-- add subtree by vendor:device:idx -->
#<add-node name="newnode-add-node-1-c" vendor="0x1" device="0xc" idx="0" >
#  <property name="newnode-1-p1" type="int" value="5"/>
#  <property name="newnode-1-p2" type="int" value="6"/>
#  <add-node name="newnode-1-1" >
#    <property name="newnode-1-1-p1" type="int" value="7"/>
#    <property name="newnode-1-1-p2" type="int" value="8"/>
#  </add-node>
#  <add-node name="newnode-1-2" >
#    <property name="newnode-1-2-p1" type="int" value="9"/>
#    <property name="newnode-1-2-p2" type="int" value="10"/>
#  </add-node>
#</add-node>
#
#<!-- add subtree by scan idx -->
#<add-node name="newnode-add-p4" parent="4" >
#  <property name="newnode-p1" type="int" value="1"/>
#  <property name="newnode-p2" type="int" value="2"/>
#  <add-node name="newnode-2-1" >
#    <property name="newnode-2-1-p1" type="int" value="3"/>
#    <property name="newnode-2-1-p2" type="int" value="4"/>
#    <property name="newnode-2-1-p2-2" type="int" value="5"/>
#  </add-node>
#</add-node>
# ---------------------  end  ----------------------------

xml=;
isallocinmem=0;isshrink=0; isuart=0;
isdump=0;isdbg=0;istest=0; ofile=""; ldir=""; baddr="0x40000000"; freq="0x2625A00"; cmdline="console=ttyS0,38400"; amp=""; smp=""; ipi="0"
ethmac="00007ccc0145";
while [ $# -ne 0 ]
do
    case "$1" in
	#-o) shift; ofile="$1";
	-allocinmem) isallocinmem=1;;
	-dump) isdump=1;;
	-shrink) isshrink=1;;
	-d) isdbg=1;;
	-t) istest=1;;
	-freq) shift; freq="$1";;
	-cmdline) shift; cmdline="$1";;
	-base) shift; baddr="$1";;
	-uart) shift; isuart=1;uart="$1";;
	-ethmac) shift; ethmac="$1";;
	-amp) shift; amp="$1";;
	-xml) shift; xml="$1";;
	-smp) ;; # ignore deprecated SMP flag, is autodetected
	-ipi) shift; ipi="$1";;
	*) a[${#a[*]}]="$1";;
    esac
    shift
done

ampoptfile="amp_opts.c"
imageo="image.o"
imagep="image.piggy"
linuxwrap2S="linuxwrap.2.S"
linuxwrapld="linuxwrap.ld"
pgts="pgt.S"
pgto="pgt.o"
promstage2o="prom_stage2.o"
removepredefsh="remove_predefs.h"
xmlo="xml.c"
if [ "$isdump" == "0" ]; then
    tmpdir=`mktemp -d`
    tmpextra=-I${tmpdir}
    ampoptfile=$tmpdir/$ampoptfile
    imageo=$tmpdir/$imageo
    imagep=$tmpdir/$imagep
    linuxwrapld=$tmpdir/$linuxwrapld
    linuxwrap2S=$tmpdir/$linuxwrap2S
    pgts=$tmpdir/$pgts
    pgto=$tmpdir/$pgto
    promstage2o=$tmpdir/$promstage2o
    removepredefsh=$tmpdir/$removepredefsh
    xmlo=$tmpdir/$xmlo
fi    

ldir=${a[0]}
limg=${a[0]}
o=${a[1]}
prefix=`dirname $(readlink -f $0)`/

if [ "${#a[*]}" != "2" ]; then usage; fi

bss_addr=0x`sparc-linux-objdump -h $limg | grep \\\\.bss\\[\\[:space:\\]\\] | awk '{ print $4 }' `;
bss_size=0x`sparc-linux-objdump -h $limg | grep \\\\.bss\\[\\[:space:\\]\\] | awk '{ print $3 }' `;

# Autodetect if SMP image (if symbol trapbase_cpu0 exists)
smp_img="NO"
if sparc-linux-readelf -s $limg | grep trapbase_cpu0 > /dev/null ; then
	smp="-DCONFIG_SMP";
	smp_img="YES"
fi

# Autodetect Linux version from Linux header in head_32.S
hdr_base_adr=`sparc-linux-objdump -x $limg | grep root_flags | sed 's/[0-9a-fA-F][ ].*$//g'`
hdr_base_adr=`echo ${hdr_base_adr}0`
linux_ver=`sparc-linux-readelf -x 1 $limg | grep -E "^  0x${hdr_base_adr}" | cut -d' ' -f5`
linux_ver=`echo 0x${linux_ver}`

macrostr=-DETHMACDEF=0x${ethmac}ull
if [ "$isallocinmem" == "1" ]; then
macrostr="${macrostr} -DALLOC_IN_MEM "
else
if [ "${xml}" != "" ]; then
echo "The -xml switch only works together with -allocinmem"; exit 1;
fi
fi
if [ "$isshrink" == "1" ]; then
macrostr="${macrostr} -DISSHRINK "
fi
if [ "$isuart" == "1" ]; then
macrostr="${macrostr} -DUARTADDR=${uart} "
fi


#" -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE=\"\\\"${cmdline}\\\"\" "
#echo "Linux-dir     :" $ldir
echo "Linux version :" $linux_ver
echo "Linux-img     :" $limg "(bss:" $bss_addr ")"
echo "Linux-wrap    :" $o
echo "prefix        :" $prefix
echo "base          :" $baddr
echo "macrodef      :" $macrostr
echo "amp           :" $amp
echo "ethmac        :" $ethmac
echo "SMP detected  :" $smp_img
echo "ipi           :" $ipi "(zero = use Linux default)"
bss_clear_start=$((($bss_addr-0xf0000000+0x40000000)))   
bss_clear_end=$((($bss_clear_start+$bss_size)))   

# 1. create a static MMU page table pgt.S using pgt_gen.c
# 2. compile prom_stage2.c and pgt.S
# 3. convert Linux-img into a binary elf section
# 4. create a linkerscript linuxwrap.ld that merges Linux-img, pgt.S and prom_stage2.c
# 5. create the output image

echo "" > ${ampoptfile}
if [ "x$amp" != "x" ]; then
    echo $amp | awk '{split($0,a,":"); for (v in a) { print a[v]; }; }' | awk '{split($0,a,"="); print "{" a[1] "," a[2] "},"; }; ' >> ${ampoptfile}
fi

if [ "${xml}" != "" ]; then
    echo "processing xml: cat ${xml} | awk -f ${prefix}scanxml.awk -v of=${xmlo}"
    echo > ${xmlo}
    cat ${xml} | awk -f ${prefix}scanxml.awk -v of=${xmlo}
    macrostr="${macrostr} -DHASXML="\"${xmlo}\"" "
fi

# -I$ldir/include -I. -I$ldir/arch/sparc/include 
echo -e "########### \n" \
     "gcc -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE=\"\\\"${cmdline}\\\"\" -D__KERNEL__ -o pgt_gen ${prefix}pgt_gen.c && \
	./pgt_gen >${pgts} && rm pgt_gen"
      gcc -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE="\"${cmdline}\""  -D__KERNEL__ -o pgt_gen ${prefix}pgt_gen.c && \
	./pgt_gen >${pgts} && rm pgt_gen 

echo -e "########### \n" \
     "sparc-linux-gcc ${macrostr} -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE=\"\\\"${cmdline}\\\"\" -c ${pgts} -o ${pgto}"
      sparc-linux-gcc ${macrostr} -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE="\"${cmdline}\"" -c ${pgts} -o ${pgto}
      
echo -e "########### \n" \
     "sparc-linux-gcc ${macrostr} -DLINUX_VERSION_CODE=$linux_ver $smp -DIPI_NUM=$ipi $tmpextra -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE=\"\\\"${cmdline}\\\"\" -DFORCE_GRMON_LOAD_SYM=\"\\\"${limg}\\\"\" -I. ${prefix}prom_stage2.c -c -o ${promstage2o} -fno-builtin  -D__KERNEL__ -mno-fpu -nostdinc -iwithprefix include -g -Os "
      sparc-linux-gcc ${macrostr} -DLINUX_VERSION_CODE=$linux_ver $smp -DIPI_NUM=$ipi $tmpextra -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE="\"${cmdline}\""       -DFORCE_GRMON_LOAD_SYM="\"${limg}\""       -I. ${prefix}prom_stage2.c -c -o ${promstage2o} -fno-builtin  -D__KERNEL__ -mno-fpu -nostdinc -iwithprefix include -g -Os || exit

echo -e "########### \n" \
     "sparc-linux-objcopy -O binary -R .note -R .comment -S $limg ${imagep}"
      sparc-linux-objcopy -O binary -R .note -R .comment -S $limg ${imagep} || exit
		    
echo -e "########### \n" \
     "sparc-linux-ld -r -b binary ${imagep} -o ${imageo} -g "
      sparc-linux-ld -r -b binary ${imagep} -o ${imageo} -g  || exit

echo -e "########### \n" \
      "Prepare ${removepredefsh}"
echo "" > ${removepredefsh}

echo -e "########### \n" \
     "cat ${prefix}linuxwrap.S | sed -e 's/PREDEF/`echo ${removepredefsh} | sed -e 's/\//\\\\\//g'`/g' > ${linuxwrap2S}"
      cat ${prefix}linuxwrap.S | sed -e "s/PREDEF/`echo ${removepredefsh} | sed -e 's/\//\\\\\//g'`/g" > ${linuxwrap2S}

sparc-linux-gcc -I. -dM -C -E -P ${linuxwrap2S} | cut -d' ' -f -2 | sed 's/#define/#undef/g' | grep -v __STDC_HOSTED__ > ${removepredefsh}.2 || exit
mv ${removepredefsh}.2 ${removepredefsh}
sparc-linux-nm $limg | awk '{ if ((toupper($2) == "T" || toupper($2) == "B") && !match($3,"\\.") ) { print $3 " = 0x" $1 ";"; } }' >> ${removepredefsh}

echo -e "########### \n" \
     "sparc-linux-gcc ${macrostr} -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE=\"\\\"${cmdline}\\\"\" -E -C -P -I. -DIMAGE=\"${imageo}\" -DPROMSTAGE2=${promstage2o} -DPGTO=${pgto} ${linuxwrap2S} -o ${linuxwrapld} " 
      sparc-linux-gcc ${macrostr} -DLEONSETUP_MEM_BASEADDR=${baddr} -DBOOTLOADER_freq=${freq} -DCONFIG_KERNEL_COMMAND_LINE="\"${cmdline}\""       -E -C -P -I. -DIMAGE="${imageo}"   -DPROMSTAGE2=${promstage2o} -DPGTO=${pgto} ${linuxwrap2S} -o ${linuxwrapld} || exit

echo -e "########### \n" \
     "sparc-linux-ld -X -T ${linuxwrapld} -o $o --defsym bss_start=$bss_clear_start --defsym bss_end=$bss_clear_end -g "
      sparc-linux-ld -X -T ${linuxwrapld} -o $o --defsym bss_start=$bss_clear_start --defsym bss_end=$bss_clear_end -g || exit


if [ "$isallocinmem" == "0" ]; then
echo -e "########### \n" \
    "test for ctx_table location: sparc-linux-nm $o | grep ctx_table | awk '{ print \$1; exit 0}'"
ctxtableaddr=`sparc-linux-nm $o | grep ctx_table | awk '{ print $1; exit 0;}'`
echo " ctx_table at: 0x${ctxtableaddr}, expecting ${baddr} + 0x2000 ($(( 0x${ctxtableaddr} == (${baddr} + 0x2000) ))) "
if [ "$(( 0x${ctxtableaddr} == ${baddr} + 0x2000 ))" == "0" ]; then
  echo " expecting ctx_table to be at ${baddr} + 0x2000. Please reduce the image by undefining the DEBUG_PROMSTAGE macro."; exit 1;
fi
fi

if [ "$isdump" == "1" ]; then
echo -e "########### \n" \
     "sparc-linux-objdump -d $o > $o.dis"
      sparc-linux-objdump -d $o > $o.dis 
fi

if [ "$isdump" == "0" ]; then
    rm -rf ${tmpdir}
fi
