/*  This file contains the driver for the AGGA4 UART serial port.
 *  Interrupt/Polling/Task driven mode can be overriding these weak variables:
 *
 *  - agga4_console[0|1]_mode (0: Polling, 1: Interrupt, 2: Task-Driven-Interrupt Mode)
 *  - agga4_console_sys       (-1 : No System Console, >= 0 : Suggested system console no.)
 *
 *  COPYRIGHT (c) 2016.
 *  Cobham Gaisler.
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 *
 *  2016-06-30, Arvid Björkengren <arvid@gaisler.com>
 *   derived from console_uart_cons.c
 *
 *  2010-09-27, Daniel Hellstrom <daniel@gaisler.com>
 *   created
 */

/******************* Driver manager interface ***********************/
#include <bsp.h>
#include <rtems/libio.h>
#include <stdlib.h>
#include <assert.h>
#include <rtems/bspIo.h>
#include <string.h>
#include <stdio.h>
#include <stdint.h>

#include <agga4/uart.h>
#include <rtems/termiostypes.h>

/*#define DEBUG 1  */

/* Configuration variables */
int agga4_console_sys __attribute__((weak)) = 0;
int agga4_console_dbg __attribute__((weak)) = 0;
int agga4_console0_mode __attribute__((weak)) = 0;
int agga4_console1_mode __attribute__((weak)) = 0;

#ifdef DEBUG
#define DBG(x...) printk(x)
#else
#define DBG(x...)
#endif

#define AGGA4_REG_UART_CONTROL_RTD  0x000000FF /* RX/TX data */

static struct console_uart_priv {
    char devName[32];
    void *cookie;
    volatile int sending;
    int mode;
    int init;
    int rxpos;
    int rxmax;
    int refcnt;
    uint8_t rxbuf[8] __attribute__((aligned(4)));
    uint8_t txbuf[8] __attribute__((aligned(4)));
} privs[2];

static rtems_termios_callbacks console_callbacks[2];

/* UART driver functions */
extern rtems_status_code __uart_initialize(void);
extern rtems_status_code __uart_open(rtems_device_minor_number minor);
extern rtems_status_code __uart_close(rtems_device_minor_number minor);
extern rtems_status_code __uart_ioctl(rtems_device_minor_number minor, int ctl, void *arg);
extern rtems_status_code __uart_printk_init(rtems_device_minor_number minor);
extern rtems_status_code __uart_printk_inb(rtems_device_minor_number minor, int *c);
extern rtems_status_code __uart_printk_outb(rtems_device_minor_number minor, int c);

/* TERMIOS Layer Callback functions */
static int console_uart_set_attributes(int minor, const struct termios *t);
static ssize_t console_uart_write_polled(int minor, const char *buf, size_t len);
static int console_uart_pollRead(int minor);
static ssize_t console_uart_write_intr(int minor, const char *buf, size_t len);
static int console_uart_pollRead_task(int minor);

static void console_uart_rx_isr(void *arg);
static void console_uart_tx_isr(void *arg);

#if AGGA4_UART_MODE == TERMIOS_IRQ_DRIVEN
#elif AGGA4_UART_MODE == TERMIOS_TASK_DRIVEN
#elif AGGA4_UART_MODE == TERMIOS_POLLED
#else
#error "AGGA4_UART_MODE han an invalid value!"
#endif

int console_uart_init1(
        int minor,
        struct console_uart_priv *priv)
{
    DBG("UART[%d] on bus %s\n", dev->minor_drv, dev->parent->dev->name);

    __uart_initialize();

    /* Private data was allocated by BSP */
    if (!priv) {
        return RTEMS_NO_MEMORY;
    } else if (priv->init) {
        // Already inited (debug console)
        return RTEMS_SUCCESSFUL;
    }

    memset(priv, 0, sizeof(struct console_uart_priv));



    priv->init = 1;

    return 0;
}

static void console_uart_rx_arm(struct console_uart_priv *priv, int minor)
{
    struct uart_desc desc;

    priv->rxpos = 0;
    priv->rxpos = 0;
    if (priv->mode == TERMIOS_POLLED) {
        priv->rxmax = sizeof(priv->rxbuf);
    } else {
        priv->rxmax = 1;
    }
    desc.addr = &priv->rxbuf[0];
    desc.bytes = priv->rxmax;
    __uart_ioctl(minor, UART_IOCTL_RX, &desc);
}

/* This routine transmits a character, it will busy-wait until on character
 * fits in the UART Transmit FIFO
 */
void console_uart_outbyte_polled(
        struct console_uart_priv *priv,
        int minor,
        unsigned char ch,
        int do_cr_on_newline,
        int wait_sent)
{
    struct uart_desc desc;
    while (__uart_ioctl(minor, UART_IOCTL_TX_POLL, NULL) == RTEMS_TIMEOUT) {
        /* Lower bus utilization while waiting for UART */
        asm volatile ("nop"::);    asm volatile ("nop"::);
        asm volatile ("nop"::);    asm volatile ("nop"::);
        asm volatile ("nop"::);    asm volatile ("nop"::);
        asm volatile ("nop"::);    asm volatile ("nop"::);
    }

    *((volatile uint8_t *)&priv->txbuf[0]) = ch;
    desc.addr = &priv->txbuf[0];
    desc.bytes = 1;
    if ((ch == '\n') && do_cr_on_newline) {
        priv->txbuf[1] = '\r';
        desc.bytes = 2;
    }
    __uart_ioctl(minor, UART_IOCTL_TX, &desc);

    /* Wait until the character has been sent? */
    if (wait_sent) {
        while (__uart_ioctl(minor, UART_IOCTL_TX_POLL, NULL) == RTEMS_TIMEOUT) {
            /* Lower bus utilization while waiting for UART */
            asm volatile ("nop"::);    asm volatile ("nop"::);
            asm volatile ("nop"::);    asm volatile ("nop"::);
            asm volatile ("nop"::);    asm volatile ("nop"::);
            asm volatile ("nop"::);    asm volatile ("nop"::);
        }
    }
}

/* This routine polls for one character, return EOF if no character is available */
int console_uart_inbyte_nonblocking(
        struct console_uart_priv *priv, int minor)
{
    int data;
    int rxnum;
    rtems_status_code status;
    uint32_t uart_status = 0;

    __uart_ioctl(minor, UART_IOCTL_GET_STATUS, &uart_status);
    if (uart_status) {
        __uart_ioctl(minor, UART_IOCTL_CLR_STATUS, (void*)UART_STATUS_CLEAR);
    }

    status = __uart_ioctl(minor, UART_IOCTL_RX_POLL, &rxnum);

    if (priv->rxpos < rxnum) {
        data = priv->rxbuf[priv->rxpos++];
    } else {
        data = EOF;
    }

    if(status == RTEMS_SUCCESSFUL && priv->rxpos >= priv->rxmax) {
        console_uart_rx_arm(priv, minor);
    }

    return data;
}

int console_uart_open(
        int minor,
        void *arg)
{
    struct console_uart_priv *uart = &privs[minor];
    rtems_status_code status;
    rtems_libio_open_close_args_t *ioarg = arg;

    if (uart->refcnt++ == 0) {
        status = __uart_open(minor);
        if (status != RTEMS_SUCCESSFUL) {
            return status;
        }

        if ( ioarg && ioarg->iop )
            uart->cookie = ioarg->iop->data1;
        else
            uart->cookie = NULL;

        /* Enable TX/RX */
        __uart_ioctl(minor, UART_IOCTL_START, NULL);
        console_uart_rx_arm(uart, minor);

        if (uart->mode != TERMIOS_POLLED) {
            uart->sending = 0;

            /* Register interrupt and enable it */
            struct uart_isr isr;
            isr.isr = console_uart_rx_isr;
            isr.arg = (void*)minor;
            isr.irq = UART_ISR_RXDONE;
            __uart_ioctl(minor, UART_IOCTL_SET_ISR, &isr);

            /* Enable TX interrupt */
            isr.isr = console_uart_tx_isr;
            isr.arg = (void*)minor;
            isr.irq = UART_ISR_TXDONE;
            __uart_ioctl(minor, UART_IOCTL_SET_ISR, &isr);
        }
    }
    return RTEMS_SUCCESSFUL;
}

int console_uart_close(
        int minor,
        void *arg)
{
    struct console_uart_priv *uart = &privs[minor];

    if (--uart->refcnt == 0) {
        if (uart->mode != TERMIOS_POLLED) {
            struct uart_isr isr;

            isr.isr = 0;
            isr.arg = 0;
            isr.irq = UART_ISR_RXDONE;
            __uart_ioctl(minor, UART_IOCTL_SET_ISR, &isr);

            /**** Flush device ****/
            while (uart->sending) {
                /* Wait until all data has been sent */
            }

            isr.isr = 0;
            isr.arg = 0;
            isr.irq = UART_ISR_TXDONE;
            __uart_ioctl(minor, UART_IOCTL_SET_ISR, &isr);

            uart->rxpos = 0;
            uart->rxmax = 0;
        }
        return __uart_close(minor);
    }

    return RTEMS_SUCCESSFUL;
}

int console_uart_pollRead(
        int minor)
{
    struct console_uart_priv *uart = &privs[minor];

    return console_uart_inbyte_nonblocking(uart, minor);
}

int console_uart_pollRead_task(
        int minor)
{
    struct console_uart_priv *uart = &privs[minor];
    int c, tot;
    char buf[32];

    tot = 0;
    while ((c=console_uart_inbyte_nonblocking(uart, minor)) != EOF) {
        buf[tot++] = c;
        if (tot >= sizeof(buf)) {
            rtems_termios_enqueue_raw_characters(uart->cookie, buf, tot);
            tot = 0;
        }
    }
    if (tot > 0)
        rtems_termios_enqueue_raw_characters(uart->cookie, buf, tot);

    return EOF;
}

struct console_uart_baud {
    uint32_t num;
    uint32_t baud;
};

struct console_uart_baud console_uart_baud_table[] = {
        {B50, 50},
        {B75, 75},
        {B110, 110},
        {B134, 134},
        {B150, 150},
        {B200, 200},
        {B300, 300},
        {B600, 600},
        {B1200, 1200},
        {B1800, 1800},
        {B2400, 2400},
        {B4800, 4800},
        {B9600, 9600},
        {B19200, 19200},
        {B38400, 38400},
        {B57600, 57600},
        {B115200, 115200},
        {B230400, 230400},
        {B460800, 460800},
};
#define BAUD_NUM (sizeof(console_uart_baud_table)/sizeof(struct console_uart_baud))

int console_uart_baud_num2baud(uint32_t num)
{
    int i;

    for(i=0; i<BAUD_NUM; i++)
        if (console_uart_baud_table[i].num == num)
            return console_uart_baud_table[i].baud;
    return -1;
}

struct console_uart_baud *console_uart_get_baud_closest(
        struct console_uart_priv *uart, int minor)
{
    uint32_t baud = 0;
    int i, diff;

    __uart_ioctl(minor, UART_IOCTL_GET_RATE, &baud);

    for(i=0; i<BAUD_NUM-1; i++) {
        diff = console_uart_baud_table[i+1].baud -
                console_uart_baud_table[i].baud;
        if (baud < (console_uart_baud_table[i].baud + diff/2))
            return &console_uart_baud_table[i];
    }
    return &console_uart_baud_table[BAUD_NUM-1];
}

int console_uart_set_attributes(
        int minor,
        const struct termios *t)
{
    int baud;

    switch(t->c_cflag & CSIZE) {
    default:
    case CS5:
    case CS6:
    case CS7:
        /* Hardware doesn't support other than CS8 */
        return -1;
    case CS8:
        break;
    }


    switch(t->c_cflag & (PARENB|PARODD)){
    case (PARENB|PARODD):
            		        /* Odd parity */
		        __uart_ioctl(minor, UART_IOCTL_SET_PARITY, (void*)UART_PARITY_ODD);
    break;

    case PARENB:
        /* Even parity */
        __uart_ioctl(minor, UART_IOCTL_SET_PARITY, (void*)UART_PARITY_EVEN);
        break;

    default:
        /* No Parity */
        __uart_ioctl(minor, UART_IOCTL_SET_PARITY, (void*)UART_PARITY_NONE);
    }

    if (!(t->c_cflag & CLOCAL)) {
        __uart_ioctl(minor, UART_IOCTL_SET_FLOW, (void*)UART_FLOW_YES);
    } else {
        __uart_ioctl(minor, UART_IOCTL_SET_FLOW, (void*)UART_FLOW_NO);
    }

    /* Baud rate */
    baud = console_uart_baud_num2baud(t->c_cflag & CBAUD);
    if (baud > 0){
        __uart_ioctl(minor, UART_IOCTL_SET_RATE, (void*)baud);
    }

    return 0;
}

void console_uart_get_attributes(
        int minor,
        struct termios *t)
{
    struct console_uart_priv *uart = &privs[minor];
    int parity = 0;
    int flow = 0;
    struct console_uart_baud *baud;

    t->c_cflag = t->c_cflag & ~(CSIZE|PARENB|PARODD|CLOCAL|CBAUD);

    /* Hardware support only CS8 */
    t->c_cflag |= CS8;

    /* Read out current parity */
    __uart_ioctl(minor, UART_IOCTL_GET_PARITY, &parity);
    if (parity == UART_PARITY_ODD) {
        t->c_cflag |= PARENB|PARODD;
    } else if (parity == UART_PARITY_EVEN) {
        t->c_cflag |= PARENB;
    }

    __uart_ioctl(minor, UART_IOCTL_GET_FLOW, &flow);
    if (flow) {
        t->c_cflag |= CLOCAL;
    }

    baud = console_uart_get_baud_closest(uart, minor);
    t->c_cflag |= baud->num;
}

ssize_t console_uart_write_polled(
        int minor,
        const char *buf,
        size_t len)
{
    int nwrite = 0;
    struct console_uart_priv *uart = &privs[minor];

    while (nwrite < len) {
        console_uart_outbyte_polled(uart, minor, *buf++, 0, 0);
        nwrite++;
    }
    return nwrite;
}

ssize_t console_uart_write_intr(
        int minor,
        const char *buf,
        size_t len)
{
    struct console_uart_priv *uart = &privs[minor];
    uint32_t oldLevel;
    struct uart_desc desc;

    rtems_interrupt_disable(oldLevel);

    if (!uart->sending) {
        desc.addr = &uart->txbuf[0];
        desc.bytes = len < sizeof(uart->txbuf) ? len : sizeof(uart->txbuf);
        memcpy(desc.addr, &buf[0], desc.bytes);

        /* start UART TX, this will result in an interrupt when done */
        if (__uart_ioctl(minor, UART_IOCTL_TX, &desc) == RTEMS_SUCCESSFUL) {
            uart->sending = desc.bytes;
        }
    }

    rtems_interrupt_enable(oldLevel);

    return 0;
}

/* Handle UART interrupts */
void console_uart_rx_isr(void *arg)
{
    int minor = (int)arg;
    struct console_uart_priv *uart = &privs[minor];
    rtems_status_code status;
    int rxnum;

    /* Get all received characters */
    status = __uart_ioctl(minor, UART_IOCTL_RX_POLL, &rxnum);
        if (uart->rxpos < rxnum) {
            if (uart->mode == TERMIOS_TASK_DRIVEN) {
            rtems_termios_rxirq_occured(uart->cookie);
        } else {
            rtems_termios_enqueue_raw_characters(uart->cookie,
                    (char*)(uart->rxbuf+uart->rxpos), rxnum-uart->rxpos);
        }
    }
    if (status == RTEMS_SUCCESSFUL) {
        console_uart_rx_arm(uart, minor);
    }
}

void console_uart_tx_isr(void *arg)
{
    int minor = (int)arg;
    struct console_uart_priv *uart = &privs[minor];
    int txnum;

    if (uart->sending &&
            __uart_ioctl(minor, UART_IOCTL_TX_POLL, NULL) == RTEMS_SUCCESSFUL) {

        txnum = uart->sending;
        uart->sending = 0;

        /* console_uart_write_intr() will get called from this function */
        rtems_termios_dequeue_characters(uart->cookie, txnum);
    }
}

/*
 * RTEMS I/O interface
 * **************************************************************************/
rtems_device_driver console_initialize(
        rtems_device_major_number    major,
        rtems_device_minor_number    minor,
        void                *arg)
{
    char name[16];
    struct console_uart_priv *uart;
    rtems_termios_callbacks *callbacks;
    rtems_status_code status;
    rtems_device_minor_number i = 0;

    rtems_termios_initialize();

    for (i=0; i<2; i++) {
        uart = &privs[i];
        callbacks = &console_callbacks[i];

        console_uart_init1(minor, uart);

        if (i == 1) {
            uart->mode = agga4_console1_mode;
        } else {
            uart->mode = agga4_console0_mode;
        }

        if (uart->mode == TERMIOS_IRQ_DRIVEN) {
            callbacks->write = console_uart_write_intr;
            callbacks->setAttributes = console_uart_set_attributes;
            callbacks->outputUsesInterrupts = TERMIOS_IRQ_DRIVEN;
        } else if (uart->mode == TERMIOS_TASK_DRIVEN) {
            callbacks->pollRead = console_uart_pollRead_task;
            callbacks->write = console_uart_write_intr;
            callbacks->setAttributes = console_uart_set_attributes;
            callbacks->outputUsesInterrupts = TERMIOS_TASK_DRIVEN;
        } else {
            callbacks->pollRead = console_uart_pollRead;
            callbacks->write = console_uart_write_polled;
            callbacks->setAttributes = console_uart_set_attributes;
            callbacks->outputUsesInterrupts = TERMIOS_POLLED;
        }

        strcpy(name, "/dev/console_a");
        if (i == agga4_console_sys) {
            name[12] = '\0';
        } else {
            name[13] += i; /* when minor=0, this has no effect */
        }

        status = rtems_io_register_name(name, major, i);
        if (status != RTEMS_SUCCESSFUL)
            rtems_fatal_error_occurred(status);
    }

    return RTEMS_SUCCESSFUL;
}

rtems_device_driver console_open(
        rtems_device_major_number    major,
        rtems_device_minor_number    minor,
        void                *arg)
{
    rtems_status_code status;
    struct termios term;

    if (minor >= 2)
        return RTEMS_INVALID_NUMBER;

    status = rtems_termios_open(
            major,
            minor,
            arg,
            &console_callbacks[minor]);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    status = console_uart_open(minor, arg);
    if (status != RTEMS_SUCCESSFUL) {
        rtems_termios_close(arg);
        return status;
    }

    /* Inherit UART hardware parameters from bootloader on system console */
    if (minor == agga4_console_sys) {
        if (tcgetattr(STDIN_FILENO, &term) >= 0) {
            console_uart_get_attributes(minor, &term);
            term.c_oflag |= ONLCR;
            tcsetattr(STDIN_FILENO, TCSANOW, &term);
        }
    }

    return status;
}

rtems_device_driver console_close(
        rtems_device_major_number    major,
        rtems_device_minor_number    minor,
        void                *arg)
{
    console_uart_close(minor, arg);
    return rtems_termios_close(arg);
}

rtems_device_driver console_read(
        rtems_device_major_number    major,
        rtems_device_minor_number    minor,
        void                *arg)
{
    return rtems_termios_read(arg);
}

rtems_device_driver console_write(
        rtems_device_major_number    major,
        rtems_device_minor_number    minor,
        void                *arg)
{
    return rtems_termios_write(arg);
}

rtems_device_driver console_control(
        rtems_device_major_number    major,
        rtems_device_minor_number    minor,
        void                *arg)
{
    return rtems_termios_ioctl(arg);
}

/*
 * Printk support
 * **************************************************************************/

/* Before UART driver has registered (or when no UART is available), calls to
 * printk that gets to bsp_out_char() will be filling data into the
 * console_printk_buf[] buffer, hopefully the buffer can help debugging the
 * early BSP boot.. At least the last printk() will be caught.
 */
char console_printk_buf[32];
int console_printk_pos = 0;
static int console_printk_is_init = 0;

#define console_printk_buf_wrap(p) ((p) & (sizeof(console_printk_buf) - 1))

void console_printk_init(void)
{
    struct console_uart_priv *uart;

    /* Update agga4_console_dbg to index used as debug console.
     * Let user select Debug console by setting agga4_console_dbg.
     * If agga4_console_dbg==-1 no printk will be provided.
     */
    if (agga4_console_dbg < 0) {
        return;
    } else if (agga4_console_dbg >= 1) {
        agga4_console_dbg = 0; /* Default dbg-console */
    }

    uart = &privs[agga4_console_dbg];
    console_uart_init1(agga4_console_dbg, uart);

    __uart_printk_init(agga4_console_dbg);

    console_printk_is_init = 1;
}

static int console_printk_in(void)
{
    struct console_uart_priv *uart;
    int c;

    if (agga4_console_dbg < 0 || console_printk_is_init == 0) {
        return 0;
    }

    uart = &privs[agga4_console_dbg];

    if (__uart_printk_inb(agga4_console_dbg, &c) == RTEMS_RESOURCE_IN_USE) {
        /* Console is opened by user or system*/
        do {
            c = console_uart_inbyte_nonblocking(uart, agga4_console_dbg);
        } while (c == EOF);
    }

    return c;
}

static void console_printk_out(char c)
{
    struct console_uart_priv *uart;

    if (agga4_console_dbg < 0) {
        return;
    }

    if (console_printk_is_init == 0) {
        /* Local debug buffer when UART driver has not registered */
        console_printk_buf[console_printk_pos++] = c;
        console_printk_pos = console_printk_buf_wrap(console_printk_pos);
        console_printk_buf[console_printk_pos++] = '\0';
        console_printk_pos = console_printk_buf_wrap(console_printk_pos);
        return;
    }

    uart = &privs[agga4_console_dbg];

    if (__uart_printk_outb(agga4_console_dbg, c) == RTEMS_RESOURCE_IN_USE) {
        return console_uart_outbyte_polled(uart, agga4_console_dbg, c, 1, 1);
    }
}

BSP_polling_getchar_function_type BSP_poll_char = console_printk_in;
BSP_output_char_function_type BSP_output_char = console_printk_out;



