#include <string.h>
#include <stdlib.h>

#include <rtems.h>
#include <rtems/libio.h>

#include <bsp.h>
#include <agga4/crc.h>

struct crc_regs {
    volatile uint32_t LFSR;
    volatile uint32_t Polynom;
    volatile uint32_t Final_XOR;
    volatile uint32_t Ctrl;
    volatile uint32_t SAP;
    volatile uint32_t EAP;
    volatile uint32_t CAP;
};

static struct crc_priv {
    struct crc_regs *regs;
    struct crc_isr isr;
    uint32_t lfsr;
    char open;
} crc_priv;

void crc_isr(void *arg)
{
    struct crc_priv *priv = arg;

    if (CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_CRC)) {
        if (priv->isr.isr) {
            priv->isr.isr(priv->isr.arg);
        }
        CIC_Clear_interrupt(AGGA4_INTERRUPT_CIC_CRC);
    }
}

void crc_install_isr(struct crc_priv *priv)
{
    CIC_Clear_interrupt(AGGA4_INTERRUPT_CIC_CRC);
    BSP_shared_interrupt_register(AGGA4_INTERRUPT_CIC, "CRC", crc_isr, priv);
    CIC_Unmask_interrupt(AGGA4_INTERRUPT_CIC_CRC);
}

void crc_uninstall_isr(struct crc_priv *priv)
{
    BSP_shared_interrupt_unregister(AGGA4_INTERRUPT_CIC, crc_isr, priv);
    CIC_Mask_interrupt(AGGA4_INTERRUPT_CIC_CRC);
}

static rtems_status_code crc_ioctl_start(
        struct crc_priv *priv, struct crc_desc *desc, int cont)
{
    if (!desc) {
        return RTEMS_INVALID_NAME;
    }

    if (!cont) {
        priv->regs->LFSR = priv->lfsr;
    }

    if (!priv->isr.isr) {
        CIC_Clear_interrupt(AGGA4_INTERRUPT_CIC_CRC);
    }

    priv->regs->SAP = (uint32_t)desc->addr;
    priv->regs->EAP = (uint32_t)(desc->addr+desc->bytes-1);

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_poll(
        struct crc_priv *priv)
{
    if (priv->isr.isr) {
        return RTEMS_RESOURCE_IN_USE;
    }

    while (!CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_CRC)) {
        /* Wait a bit to lower bus utilization */
        asm volatile ("nop; nop; nop; nop; nop; nop; nop; nop;");
    }

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_get_crc(
        struct crc_priv *priv, uint32_t *reg)
{
    if (!reg) {
        return RTEMS_INVALID_NAME;
    }

    *reg = priv->regs->LFSR;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_set_lfsr(
        struct crc_priv *priv, uint32_t lfsr) {
    priv->lfsr = lfsr;
    priv->regs->LFSR = lfsr;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_get_polynom(
        struct crc_priv *priv, uint32_t *reg)
{
    if (!reg) {
        return RTEMS_INVALID_NAME;
    }

    *reg = priv->regs->Polynom;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_set_polynom(
        struct crc_priv *priv, uint32_t reg)
{
    priv->regs->Polynom = reg;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_get_final_xor(
        struct crc_priv *priv, uint32_t *reg)
{
    if (!reg) {
        return RTEMS_INVALID_NAME;
    }
    *reg = priv->regs->Final_XOR;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_set_final_xor(
        struct crc_priv *priv, uint32_t reg)
{
    priv->regs->Final_XOR = reg;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_get_ctrl(
        struct crc_priv *priv, uint32_t *reg)
{
    if (!reg) {
        return RTEMS_INVALID_NAME;
    }
    *reg = priv->regs->Ctrl;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_set_ctrl(
        struct crc_priv *priv, uint32_t reg)
{
    priv->regs->Ctrl = reg;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code crc_ioctl_isr(
        struct crc_priv *priv, struct crc_isr *isr)
{
    rtems_interrupt_level level;

    if (isr == NULL) {
        return RTEMS_INVALID_NAME;
    }

    rtems_interrupt_disable(level);
    priv->isr.isr = isr->isr;
    priv->isr.arg = isr->arg;
    if (priv->isr.isr) {
        crc_install_isr(priv);
    } else {
        crc_uninstall_isr(priv);
    }
    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

void crc_init(struct crc_priv *priv)
{
    memset(priv, 0, sizeof(crc_priv));
    priv->regs = (struct crc_regs *)&LEON_REG->CRC_LFSR;
}

rtems_status_code crc_open(
        struct crc_priv *priv, rtems_device_minor_number minor)
{
    if (minor >= 1) {
        return RTEMS_TOO_MANY;
    }

    if (priv->open) {
        return RTEMS_RESOURCE_IN_USE;
    }
    priv->open = 1;

    return RTEMS_SUCCESSFUL;
}

rtems_status_code crc_close(
        struct crc_priv *priv, rtems_device_minor_number minor)
{
    struct crc_isr isr = {0};
    crc_ioctl_isr(priv, &isr);

    priv->open = 0;

    return RTEMS_SUCCESSFUL;
}

/*****************************************************************************
 * RTEMS I/O
 */
rtems_device_driver crc_io_initialize(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    rtems_status_code status;
    struct crc_priv *priv = &crc_priv;

    crc_init(priv);

    status = rtems_io_register_name(AGGA4_CRC_DEVNAME, major, 0);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    return RTEMS_SUCCESSFUL;
}

rtems_device_driver crc_io_open(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    struct crc_priv *priv = &crc_priv;

    return crc_open(priv, minor);
}

rtems_device_driver crc_io_close(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    struct crc_priv *priv = &crc_priv;

    return crc_close(priv, minor);
}

rtems_device_driver crc_io_control(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    rtems_status_code status;
    struct crc_priv *priv = &crc_priv;
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;
    switch(ioarg->command) {
    case CRC_IOCTL_START: {
        status = crc_ioctl_start(priv, (struct crc_desc *)ioarg->buffer, 0);
        break;
    }
    case CRC_IOCTL_CONTINUE: {
        status = crc_ioctl_start(priv, (struct crc_desc *)ioarg->buffer, 1);
        break;
    }
    case CRC_IOCTL_GET_CRC: {
        status = crc_ioctl_get_crc(priv, (uint32_t*)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_POLL: {
        status = crc_ioctl_poll(priv);
        break;
    }
    case CRC_IOCTL_SET_LFSR: {
        status = crc_ioctl_set_lfsr(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_GET_POLYNOM: {
        status = crc_ioctl_get_polynom(priv, (uint32_t*)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_SET_POLYNOM: {
        status = crc_ioctl_set_polynom(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_GET_FINAL_XOR: {
        status = crc_ioctl_get_final_xor(priv, (uint32_t*)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_SET_FINAL_XOR: {
        status = crc_ioctl_set_final_xor(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_GET_CTRL: {
        status = crc_ioctl_get_ctrl(priv, (uint32_t*)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_SET_CTRL: {
        status = crc_ioctl_set_ctrl(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case CRC_IOCTL_SET_ISR: {
        status = crc_ioctl_isr(priv, (struct crc_isr *)ioarg->buffer);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}
