#include <string.h>
#include <stdlib.h>

#include <rtems.h>
#include <rtems/libio.h>

#include <bsp.h>
#include <agga4/fft.h>

struct fft_regs {
    struct {
        volatile uint32_t real;
        volatile uint32_t imag;
    } value[128];
    volatile uint32_t Ctrl;
};

static struct fft_priv {
    struct fft_regs *regs;
    struct fft_isr isr;
    uint32_t block;
    rtems_id sem;
    char open;
} fft_priv;

void fft_isr(void *arg)
{
    struct fft_priv *priv = arg;
    if (priv->block & FFT_BLOCKING) {
        rtems_semaphore_release(priv->sem);
    }

    if (priv->isr.isr) {
        priv->isr.isr(priv->isr.arg);
    }
}

static inline void fft_install_isr(struct fft_priv *priv)
{
    BSP_shared_interrupt_register(AGGA4_INTERRUPT_FFT_DONE, "FFT", fft_isr, priv);
}

static inline void fft_uninstall_isr(struct fft_priv *priv)
{
    BSP_shared_interrupt_unregister(AGGA4_INTERRUPT_FFT_DONE, fft_isr, priv);
}

static rtems_status_code fft_ioctl_blocking(
        struct fft_priv *priv, uint32_t block)
{
    rtems_status_code status = RTEMS_SUCCESSFUL;
    rtems_interrupt_level level;

    rtems_interrupt_disable(level);

    if (!(priv->block & FFT_BLOCKING) && (block & FFT_BLOCKING)) {
        status = rtems_semaphore_create(
                rtems_build_name('F', 'F', 'T', '0'),
                0,
                RTEMS_FIFO | RTEMS_SIMPLE_BINARY_SEMAPHORE | \
                RTEMS_NO_INHERIT_PRIORITY | RTEMS_NO_PRIORITY_CEILING,
                0,
                &(priv->sem));

        if (status == RTEMS_SUCCESSFUL && !priv->isr.isr) {
            fft_install_isr(priv);
        }

    } else if ((priv->block & FFT_BLOCKING) && !(block & FFT_BLOCKING)) {
        if (!priv->isr.isr) {
            fft_uninstall_isr(priv);
        }
        rtems_semaphore_delete(priv->sem);
    }
    priv->block = block;

    rtems_interrupt_enable(level);

    return status;
}

static rtems_status_code fft_ioctl_get_ctrl(
        struct fft_priv *priv, uint32_t *ctrl)
{
    if (ctrl) {
        *ctrl = priv->regs->Ctrl;
        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

static rtems_status_code fft_ioctl_set_ctrl(
        struct fft_priv *priv, uint32_t ctrl)
{
        if ((ctrl & FFT_CTRL_START) && (priv->block & FFT_BLOCKING)) {
            rtems_semaphore_obtain(priv->sem, RTEMS_NO_WAIT, RTEMS_NO_TIMEOUT);
        }
        priv->regs->Ctrl = ctrl;

        return RTEMS_SUCCESSFUL;

}

static rtems_status_code fft_ioctl_get_value(
        struct fft_priv *priv, struct fft_value *val)
{
    if (val) {
        val->real = priv->regs->value[val->index].real;
        val->imag = priv->regs->value[val->index].imag;
        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

static rtems_status_code fft_ioctl_set_value(
        struct fft_priv *priv, struct fft_value *val)
{
    if (val) {
        priv->regs->value[val->index].real = val->real;
        priv->regs->value[val->index].imag = val->imag;
        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

static rtems_status_code fft_ioctl_isr(
        struct fft_priv *priv, struct fft_isr *isr)
{
    rtems_interrupt_level level;

    if (isr == NULL) {
        return RTEMS_INVALID_NAME;
    }

    rtems_interrupt_disable(level);
    if (!priv->isr.isr && isr->isr) {
        priv->isr.isr = isr->isr;
        priv->isr.arg = isr->arg;
        if (!(priv->block & FFT_BLOCKING)) {
            fft_install_isr(priv);
        }
    } else if (priv->isr.isr && !isr->isr) {
        priv->isr.isr = NULL;
        priv->isr.arg = NULL;
        if (!(priv->block & FFT_BLOCKING)) {
            fft_uninstall_isr(priv);
        }
    } else {
        priv->isr.isr = isr->isr;
        priv->isr.arg = isr->arg;
    }
    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

static void fft_init(struct fft_priv *priv)
{
    memset(priv, 0, sizeof(fft_priv));
    priv->regs = (struct fft_regs *)0xB0000000;
}

static rtems_status_code fft_open(
        struct fft_priv *priv, rtems_device_minor_number minor)
{
    if (minor >= 1) {
        return RTEMS_TOO_MANY;
    }

    if (priv->open) {
        return RTEMS_RESOURCE_IN_USE;
    }
    priv->open = 1;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code fft_close(
        struct fft_priv *priv, rtems_device_minor_number minor)
{
    struct fft_isr isr = {0};
    fft_ioctl_isr(priv, &isr);
    fft_ioctl_blocking(priv, FFT_NONBLOCKING); // Destroy semaphore

    priv->open = 0;

    return RTEMS_SUCCESSFUL;
}


static rtems_status_code fft_read(
        struct fft_priv *priv, void *buf, size_t cnt)
{
    if ( (cnt != 1024) || !buf ) {
            return RTEMS_INVALID_NAME; /* EINVAL */
    }

    while (priv->regs->Ctrl & FFT_CTRL_START) {
        if (!(priv->block & (FFT_BLOCKING_READ|FFT_POLLING_READ))) {
            return RTEMS_TIMEOUT; /* EWOULDBLOCK / EAGAIN */
        } else if (priv->block & FFT_BLOCKING_READ) {
            rtems_semaphore_obtain(priv->sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
            break;
        }
    }

    memcpy(buf, (void*)&priv->regs->value[0].real, cnt);

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code fft_write(
        struct fft_priv *priv, void *buf, size_t cnt)
{
    if ( (cnt != 1024) || !buf ) {
            return RTEMS_INVALID_NAME; /* EINVAL */
    }

    while (priv->regs->Ctrl & FFT_CTRL_START) {
        if (!(priv->block & (FFT_BLOCKING_WRITE|FFT_POLLING_WRITE))) {
            return RTEMS_TIMEOUT; /* EWOULDBLOCK / EAGAIN */
        } else if (priv->block & FFT_BLOCKING_WRITE) {
            rtems_semaphore_obtain(priv->sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
            break;
        }
    }

    memcpy((void*)&priv->regs->value[0].real, buf, cnt);

    fft_ioctl_set_ctrl(priv, FFT_CTRL_START);

    return RTEMS_SUCCESSFUL;
}

/*****************************************************************************
 * RTEMS I/O
 */
rtems_status_code fft_io_initialize(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct fft_priv *priv = &fft_priv;

    fft_init(priv);

    status = rtems_io_register_name(AGGA4_FFT_DEVNAME, major, 0);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code fft_io_open(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct fft_priv *priv = &fft_priv;

    return fft_open(priv, minor);
}

rtems_status_code fft_io_close(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct fft_priv *priv = &fft_priv;

    return fft_close(priv, minor);
}

rtems_status_code fft_io_read(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct fft_priv *priv = &fft_priv;
    rtems_libio_rw_args_t *rw_args = (rtems_libio_rw_args_t*)argument;

    rw_args->bytes_moved = rw_args->count;
    return fft_read(priv, rw_args->buffer, rw_args->count);
}

rtems_status_code fft_io_write(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct fft_priv *priv = &fft_priv;
    rtems_libio_rw_args_t *rw_args = (rtems_libio_rw_args_t*)argument;

    rw_args->bytes_moved = rw_args->count;
    return fft_write(priv, rw_args->buffer, rw_args->count);
}

rtems_status_code fft_io_control(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct fft_priv *priv = &fft_priv;
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;
    switch(ioarg->command) {
    case FFT_IOCTL_BLOCKING: {
        status = fft_ioctl_blocking(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case FFT_IOCTL_GET_CTRL: {
        status = fft_ioctl_get_ctrl(priv, (uint32_t*)ioarg->buffer);
        break;
    }
    case FFT_IOCTL_SET_CTRL: {
        status = fft_ioctl_set_ctrl(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case FFT_IOCTL_GET_VALUE: {
        struct fft_value *val = (struct fft_value *)ioarg->buffer;
        status = fft_ioctl_get_value(priv, val);
        break;
    }
    case FFT_IOCTL_SET_VALUE: {
        struct fft_value *val = (struct fft_value *)ioarg->buffer;
        status = fft_ioctl_set_value(priv, val);
        break;
    }
    case FFT_IOCTL_SET_ISR: {
        status = fft_ioctl_isr(priv, (struct fft_isr *)ioarg->buffer);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}
