#include <string.h>
#include <stdlib.h>

#include <rtems.h>
#include <rtems/libio.h>

#include <bsp.h>
#include <agga4/gnss.h>

#define GNSS_INTERRUPT_NUM_IRQ (GNSS_INTERRUPT_GIC_DMA_ERR+1)

static struct gnss_priv {
    struct gnss_regs *regs;
    struct gnss_isr_priv {
        gnss_isr_func isr;
        void *arg;
    } isr[42];
    char open;
} gnss_priv;

static rtems_status_code gnss_ioctl_get_regs(
        struct gnss_priv *priv, struct gnss_regs **regs)
{
    if (regs) {
        *regs = priv->regs;
        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

static rtems_status_code gnss_ioctl_set_clk(
        struct gnss_priv *priv, int clk)
{
    if (clk) {
        LEON_REG->GNSS_Core_Clk_Ctrl |= GNSS_CLK_INTERNAL;
    } else {
        LEON_REG->GNSS_Core_Clk_Ctrl &= ~GNSS_CLK_INTERNAL;
    }

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code gnss_ioctl_reset(struct gnss_priv *priv)
{
	uint32_t mask;
	int irq;
	rtems_interrupt_level level;

	rtems_interrupt_disable(level);

	LEON_REG->SW_Reset_Enable = 0xDEADAFFE;
	LEON_REG->SW_Reset_Execute = 0xCAFEBEBE;

	while (priv->regs->gic.qlow != 0x3f) {
		;
	}
	while (priv->regs->gic.qhigh != 0x3f) {
		;
	}

	mask = 0;
	for (irq=0; irq<32; irq++) {
		if (priv->isr[0+irq].isr) {
			mask |= 1<irq;
		}
	}
	priv->regs->gic.mask[0] = mask;

	mask = 0;
	for (irq=0; irq<10; irq++) {
		if (priv->isr[32+irq].isr) {
			mask |= 1<irq;
		}
	}
	priv->regs->gic.mask[1] = mask;

	rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

void gnss_gicl_isr(void *arg)
{
    struct gnss_priv *priv = (struct gnss_priv *)arg;
    int irq;
    uint32_t pending;
    uint32_t clr;
    uint32_t mask;

    clr = 0;
    pending = priv->regs->gic.pend[0];
    for (irq=0; irq<32; irq++) {
    	mask = 1<<irq;
		if (pending & mask) {
			struct gnss_isr_priv *isr = &priv->isr[0+irq];
			isr->isr(isr->arg);
			clr |= mask;
		}
    }
    priv->regs->gic.clear[0] = clr;
}

void gnss_gich_isr(void *arg)
{
    struct gnss_priv *priv = (struct gnss_priv *)arg;
    int irq;
    uint32_t pending;
    uint32_t clr;
    uint32_t mask;

    clr = 0;
    pending = priv->regs->gic.pend[1];
    for (irq=0; irq<10; irq++) {
    	mask = 1<<irq;
		if (pending & mask) {
			struct gnss_isr_priv *isr = &priv->isr[32+irq];
			isr->isr(isr->arg);
			clr |= mask;
		}
    }
    priv->regs->gic.clear[1] = clr;
}

static rtems_status_code gnss_ioctl_isr(
        struct gnss_priv *priv, struct gnss_isr *isr)
{
    rtems_interrupt_level level;
    int gic;
    int lh;
    char *pic_name;
	int pic_irq;
	bsp_shared_isr pic_isr;

    if (isr == NULL || isr->irq < 0 || isr->irq >= GNSS_INTERRUPT_NUM_IRQ) {
        return RTEMS_INVALID_NAME;
    }

    rtems_interrupt_disable(level);

    if (isr->irq < 32) {
        lh  = 0;
        gic = isr->irq;
        pic_name = "GICL";
        pic_irq = AGGA4_INTERRUPT_GICL;
        pic_isr = gnss_gicl_isr;
    } else {
        lh  = 1;
        gic = isr->irq-32;
        pic_name = "GICH";
        pic_irq = AGGA4_INTERRUPT_GICH;
        pic_isr = gnss_gich_isr;
    }
    priv->isr[isr->irq].isr = isr->isr;
    priv->isr[isr->irq].arg = isr->arg;
    if (isr->isr) {
        // Install interrupt handler
        if (priv->regs->gic.mask[lh] == 0) {
            BSP_shared_interrupt_register(
            		pic_irq, pic_name, pic_isr, priv);
        }
        priv->regs->gic.mask[lh] |= 1<<gic;

    } else {
        // Uninstall interrupt handler
    	priv->regs->gic.mask[lh] &= ~(1<<gic);
        if (priv->regs->gic.mask[lh] == 0) {
            BSP_shared_interrupt_unregister(
            		pic_irq, pic_isr, priv);
        }
    }

    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

void gnss_init(struct gnss_priv *priv)
{
    memset(priv, 0, sizeof(gnss_priv));
    priv->regs = (struct gnss_regs *)0xA0000000;
}

rtems_status_code gnss_open(
        struct gnss_priv *priv, rtems_device_minor_number minor)
{
    if (minor >= 1) {
        return RTEMS_TOO_MANY;
    }

    if (priv->open) {
        return RTEMS_RESOURCE_IN_USE;
    }
    priv->open = 1;

    return RTEMS_SUCCESSFUL;
}

rtems_status_code gnss_close(
        struct gnss_priv *priv, rtems_device_minor_number minor)
{
    priv->open = 0;

    return RTEMS_SUCCESSFUL;
}

/*****************************************************************************
 * RTEMS I/O
 */
rtems_status_code gnss_io_initialize(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct gnss_priv *priv = &gnss_priv;

    gnss_init(priv);

    status = rtems_io_register_name(AGGA4_GNSS_DEVNAME, major, 0);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code gnss_io_open(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct gnss_priv *priv = &gnss_priv;

    return gnss_open(priv, minor);
}

rtems_status_code gnss_io_close(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct gnss_priv *priv = &gnss_priv;

    return gnss_close(priv, minor);
}

rtems_status_code gnss_io_control(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct gnss_priv *priv = &gnss_priv;
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;
    switch(ioarg->command) {
    case GNSS_IOCTL_GET_REGS: {
    	struct gnss_regs **regs = (struct gnss_regs **)ioarg->buffer;
        status = gnss_ioctl_get_regs(priv, regs);
        break;
    }
    case GNSS_IOCTL_SET_CLK: {
    	status = gnss_ioctl_set_clk(priv, (int)ioarg->buffer);
        break;
    }
    case GNSS_IOCTL_SET_ISR: {
        status = gnss_ioctl_isr(priv, (struct gnss_isr *)ioarg->buffer);
        break;
    }
    case GNSS_IOCTL_RESET: {
        status = gnss_ioctl_reset(priv);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}
