#include <string.h>
#include <stdlib.h>

#include <rtems.h>
#include <rtems/libio.h>

#include <bsp.h>
#include <agga4/ip1553b.h>

#define IP1553B_CF_DW16_EN_BIT     27
#define IP1553B_CF_MEM_AREA_BIT    16
#define IP1553B_CF_ERR_1553_BIT    15
#define IP1553B_CF_IT_SYNC_BIT     14
#define IP1553B_CF_IT_TROK_BIT     13
#define IP1553B_CF_EXT_SUB_AD_BIT  8
#define IP1553B_CF_WD_SIZE_BIT     7
#define IP1553B_CF_EXT_AREA_BIT    6
#define IP1553B_CF_BRCST_EN_BIT    4
#define IP1553B_CF_DBC_EN_BIT      3
#define IP1553B_CF_TIMEOUT_BIT     2
#define IP1553B_CF_MODE_BIT        0

#define IP1553B_CF_DW16_EN         (0x1   << IP1553B_CF_DW16_EN_BIT)
#define IP1553B_CF_MEM_AREA        (0x1FF << IP1553B_CF_MEM_AREA_BIT)
#define IP1553B_CF_ERR_1553        (0x1   << IP1553B_CF_ERR_1553_BIT)
#define IP1553B_CF_IT_SYNC         (0x1   << IP1553B_CF_IT_SYNC_BIT)
#define IP1553B_CF_IT_TROK         (0x1   << IP1553B_CF_IT_TROK_BIT)
#define IP1553B_CF_EXT_SUB_AD      (0x1F  << IP1553B_CF_EXT_SUB_AD_BIT)
#define IP1553B_CF_WD_SIZE         (0x1   << IP1553B_CF_WD_SIZE_BIT)
#define IP1553B_CF_EXT_AREA        (0x1   << IP1553B_CF_EXT_AREA_BIT)
#define IP1553B_CF_BRCST_EN        (0x1   << IP1553B_CF_BRCST_EN_BIT)
#define IP1553B_CF_DBC_EN          (0x1   << IP1553B_CF_DBC_EN_BIT)
#define IP1553B_CF_TIMEOUT         (0x1   << IP1553B_CF_TIMEOUT_BIT)
#define IP1553B_CF_MODE            (0x3   << IP1553B_CF_MODE_BIT)

#define IP1553B_CDST_LAST_BIT      15
#define IP1553B_CDST_EPAR_BIT      14
#define IP1553B_CDST_BUSY_BIT      6
#define IP1553B_CDST_RSTST_BIT     5
#define IP1553B_CDST_ST_BIT        5
#define IP1553B_CDST_GO_BIT        1
#define IP1553B_CDST_RST_BIT       0

#define IP1553B_CDST_LAST          (0x1FFFF << IP1553B_CDST_LAST_BIT)
#define IP1553B_CDST_EPAR          (0x1     << IP1553B_CDST_EPAR_BIT)
#define IP1553B_CDST_ST            (0x3     << IP1553B_CDST_ST_BIT)   /* See  IP1553B_STATE_... */
#define IP1553B_CDST_BUSY          (0x1     << IP1553B_CDST_BUSY_BIT)
#define IP1553B_CDST_RSTST         (0x1     << IP1553B_CDST_RSTST_BIT)
#define IP1553B_CDST_GO            (0x1     << IP1553B_CDST_GO_BIT)
#define IP1553B_CDST_RST           (0x1     << IP1553B_CDST_RST_BIT)

#define IP1553B_INDIR_ADUPD_BIT    31
#define IP1553B_INDIR_CURBLK_BIT   21
#define IP1553B_INDIR_MAXBS_BIT    11
#define IP1553B_INDIR_BRBLK_BIT    0

#define IP1553B_INDIR_ADUPD        (0x1   << IP1553B_INDIR_ADUPD_BIT)
#define IP1553B_INDIR_CURBLK       (0x3FF << IP1553B_INDIR_CURBLK_BIT)
#define IP1553B_INDIR_MAXBS        (0x3FF << IP1553B_INDIR_MAXBS_BIT)
#define IP1553B_INDIR_BRBLK        (0x7FF << IP1553B_INDIR_BRBLK_BIT)

#define IP1553B_NIT_SWITCH_BIT     3
#define IP1553B_NIT_DBC_BIT        2
#define IP1553B_NIT_SYNC_BIT       1
#define IP1553B_NIT_TROK_BIT       0

#define IP1553B_NIT_SWITCH         (0x1 << IP1553B_NIT_SWITCH_BIT)
#define IP1553B_NIT_DBC            (0x1 << IP1553B_NIT_DBC_BIT)
#define IP1553B_NIT_SYNC           (0x1 << IP1553B_NIT_SYNC_BIT)
#define IP1553B_NIT_TROK           (0x1 << IP1553B_NIT_TROK_BIT)

#define IP1553B_EIT_RTAD_BIT       2
#define IP1553B_EIT_1553_BIT       1
#define IP1553B_EIT_MEM_BIT        0

#define IP1553B_EIT_RTAD           (0x1 << IP1553B_EIT_RTAD_BIT)
#define IP1553B_EIT_1553           (0x1 << IP1553B_EIT_1553_BIT)
#define IP1553B_EIT_MEM            (0x1 << IP1553B_EIT_MEM_BIT)

#define IP1553B_RIT_RST_BIT        0

#define IP1553B_RIT_RST            (0x1 << IP1553B_RIT_RST_BIT)

#define IP1553B_STATE_ACTIVE       0x3
#define IP1553B_STATE_RESET2       0x2
#define IP1553B_STATE_STANDBY      0x1
#define IP1553B_STATE_RESET        0x0

#define IP1553B_INTERRUPT_NUM_IRQ  (IP1553B_INTERRUPT_ERR_RTAD+1)
#define IP1553B_INTERRUPT_NOM_MASK 0x0F
#define IP1553B_INTERRUPT_RST_MASK 0x10
#define IP1553B_INTERRUPT_ERR_MASK 0xE0


struct ip1553b_regs {
    volatile uint32_t CF;
    volatile uint32_t EMBA;
    volatile uint32_t CDST;
    volatile uint32_t NIT;
    volatile uint32_t EIT;
    volatile uint32_t RIT;
    volatile uint32_t _dummy;
    volatile uint32_t RTI;
    volatile uint32_t TTI;
};

struct ip1553b_memarea {           /* WORD ADRESS       (AMBA ADRESS)       */
    union {
        struct {
            struct {
                uint16_t d;
            } word[32];
        } block32[2*1024];
        struct {
            struct {
                uint16_t d;
                uint16_t _dummy;
            } word[32];
        } block16[1*1024];
    } rxdata;                  /* 0x00000 - 0x07FFF (0x00000 - 0x1FFFF) */
    union {
        struct {
            struct {
                uint16_t d;
            } word[32];
        } block32[2*1024];
        struct {
            struct {
                uint16_t d;
                uint16_t _dummy;
            } word[32];
        } block[1*1024];
    } txdata;                  /* 0x08000 - 0x0FFFF (0x20000 - 0x3FFFF) */
    struct {
        uint16_t cmd;
        uint16_t ctrl;
        uint32_t info;
    } cmd[1920];               /* 0x10000 - 0x10EFF (0x40000 - 0x43BFF) */
    struct {
        uint16_t d;
        uint16_t _dummy;
    } rxmode[32];              /* 0x10F00 - 0x10F1F (0x43C00 - 0x43C7F) */
    struct {
        uint16_t d;
        uint16_t _dummy;
    } characterize[32];        /* 0x10F20 - 0x10F3F (0x43C80 - 0x43CFF) */
    struct {
        uint16_t d;
        uint16_t _dummy;
    } txmode[32];              /* 0x10F40 - 0x10F5F (0x43D00 - 0x43D7F) */
    uint32_t _dummy0[32];      /* 0x10F60 - 0x10F7F (0x43D80 - 0x43DFF) */
    struct {
        uint32_t _dummy0;
        uint32_t sa[30];
        uint32_t _dummy31;
    } tbl[4];                   /* 0x10F80 - 0x10F9F (0x43E00 - 0x43E7F) */
                                /* 0x10FA0 - 0x10FBF (0x43E80 - 0x43EFF) */
                                /* 0x10FC0 - 0x10FDF (0x43F00 - 0x43F7F) */
                                /* 0x10FE0 - 0x10FFF (0x43F80 - 0x43FFF) */
    uint32_t _dummy1[60*1024];  /* 0x11000 - 0x1FFFF (0x43000 - 0x80000) */
};

static struct ip1553b_priv {
    struct ip1553b_regs *regs;
    struct ip1553b_memarea *mem;
    struct ip1553b_isr_priv {
        ip1553b_isr_func isr;
        void *arg;
    } isr[IP1553B_INTERRUPT_NUM_IRQ];
    uint32_t mask;
    uint32_t next;

    char open;
} ip1553b_priv;

static rtems_status_code ip1553b_ioctl_start(
        struct ip1553b_priv *priv)
{
    uint32_t last;

    if (priv->regs->CDST & IP1553B_CDST_RST) {
        return RTEMS_NOT_DEFINED;
    } else if (priv->regs->CDST & IP1553B_CDST_GO) {
        return RTEMS_RESOURCE_IN_USE;
    }

    priv->regs->CDST |= IP1553B_CDST_GO;

    last = (priv->regs->CDST & IP1553B_CDST_LAST) >> IP1553B_CDST_LAST_BIT;
    priv->next = last - 0x10000;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_set_mem(
        struct ip1553b_priv *priv, void *mem)
{
    if (mem) {
    	uint32_t area;

        area = ((((uint32_t)mem)>>19)<<IP1553B_CF_MEM_AREA_BIT) & IP1553B_CF_MEM_AREA;
        if (area) {
            uint32_t reg = priv->regs->CF;
            reg &= ~(IP1553B_CF_MEM_AREA);
            reg |= area;
            priv->regs->CF = reg;
            priv->mem = (struct ip1553b_memarea *)((uint32_t)mem & ~((1<<19)-1));
        } else {
            return RTEMS_INVALID_NAME;
        }
    } else {
        return RTEMS_INVALID_NAME;
    }
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_set_ext(
        struct ip1553b_priv *priv, uint32_t sa)
{
    if (sa >= 0 || sa < 32) {
        uint32_t reg = priv->regs->CF;

        reg &= ~(IP1553B_CF_EXT_SUB_AD);
        reg |= (sa<<IP1553B_CF_EXT_SUB_AD_BIT) & IP1553B_CF_EXT_SUB_AD;
        reg |= IP1553B_CF_EXT_AREA;
        priv->regs->CF = reg;

    } else {
        priv->regs->CF &= ~(IP1553B_CF_EXT_AREA|IP1553B_CF_EXT_SUB_AD);
    }

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_set_timeout(
        struct ip1553b_priv *priv, uint32_t timeout)
{
    if (timeout) {
        priv->regs->CF |= IP1553B_CF_TIMEOUT;
    } else {
        priv->regs->CF &= ~IP1553B_CF_TIMEOUT;
    }
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_set_clk(
        struct ip1553b_priv *priv, uint32_t clk)
{
    if (clk) {
        LEON_REG->MilBus_Clk_Ctrl |= IP1553B_CLK_PLL;
    } else {
        LEON_REG->MilBus_Clk_Ctrl &= ~IP1553B_CLK_PLL;
    }
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_set_rt_address(
        struct ip1553b_priv *priv, uint32_t rtad)
{
    uint32_t reg;

    reg = LEON_REG->MilBus_RT_Address;
    reg &= ~0x1f;
    reg |= rtad & 0x1F;
    LEON_REG->MilBus_RT_Address = reg;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_set_parity(
        struct ip1553b_priv *priv, uint32_t par)
{
    if (par) {
        LEON_REG->MilBus_RT_Address |= 0x20;
    } else {
        LEON_REG->MilBus_RT_Address &= ~0x20;
    }
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_set_char(
        struct ip1553b_priv *priv, struct ip1553b_char *ch)
{
    if (ch) {
        if (ch->idx > 31) {
            return RTEMS_INVALID_NAME;
        }
        priv->mem->characterize[ch->idx].d = ch->word;
        return RTEMS_SUCCESSFUL;
    }
    return RTEMS_INVALID_NAME;
}

static rtems_status_code ip1553b_ioctl_set_indirection(
        struct ip1553b_priv *priv, struct ip1553b_indirection *ind)
{
    if (ind) {
        uint32_t r;
        if (ind->sa == 0 || ind->sa > 30 || ind->table > 3) {
            return RTEMS_INVALID_NAME;
        }

#define IP1553B_INDIR_CURBLK (0x3FF << IP1553B_INDIR_CURBLK_BIT)
#define IP1553B_INDIR_MAXBS  (0x3FF << IP1553B_INDIR_MAXBS_BIT)
#define IP1553B_INDIR_BRBLK  (0x7FF << IP1553B_INDIR_BRBLK_BIT)

        r = 0;
        r |= (ind->adupd << IP1553B_INDIR_ADUPD_BIT)  & IP1553B_INDIR_ADUPD;
        r |= (0          << IP1553B_INDIR_CURBLK_BIT) & IP1553B_INDIR_CURBLK;
        r |= (ind->maxbs << IP1553B_INDIR_MAXBS_BIT)  & IP1553B_INDIR_MAXBS;
        r |= (ind->brblk << IP1553B_INDIR_BRBLK_BIT)  & IP1553B_INDIR_BRBLK;
        priv->mem->tbl[ind->table].sa[ind->sa - 1] = r;
        return RTEMS_SUCCESSFUL;
    }
    return RTEMS_INVALID_NAME;
}

static rtems_status_code ip1553b_ioctl_data(
        struct ip1553b_priv *priv, int tx, struct ip1553b_data *data)
{
    uint32_t indir;
    uint8_t *buf;
    int beg;
    int max;

    if (data == NULL) {
        return RTEMS_INVALID_NAME;
    }

    if (data->sa == 0 || data->sa > 30 || data->table > 3) {
        return RTEMS_INVALID_NAME;
    }

    indir = priv->mem->tbl[data->table].sa[data->sa];
    beg = (indir & IP1553B_INDIR_BRBLK) >> IP1553B_INDIR_BRBLK_BIT;
    max = (indir & IP1553B_INDIR_MAXBS) >> IP1553B_INDIR_MAXBS_BIT;

    if (data->cnt+data->ofs > max*16) {
        return RTEMS_INVALID_NAME;
    }


    if (tx) {
    	buf = (uint8_t *)&priv->mem->txdata.block32[beg].word[0].d;
        memcpy(data->buf, &buf[data->ofs], data->cnt);
    } else {
    	buf = (uint8_t *)&priv->mem->rxdata.block32[beg].word[0].d;
        memcpy(&buf[data->ofs], data->buf, data->cnt);
    }

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_block(
        struct ip1553b_priv *priv, int tx, struct ip1553b_block *block)
{
    struct ip1553b_data data;

    if (block == NULL) {
        return RTEMS_INVALID_NAME;
    }

    data.table = block->table;
    data.sa = block->sa;
    data.ofs = (block->blk*16+block->ofs)*2;
    data.cnt = block->cnt*2;
    data.buf = (uint8_t *)block->buf;

    return ip1553b_ioctl_data(priv, tx, &data);
}

static rtems_status_code ip1553b_ioctl_get_last(
        struct ip1553b_priv *priv, struct ip1553b_cmd *cmd)
{
    uint32_t last;
    int ofs;

    last = (priv->regs->CDST & IP1553B_CDST_LAST) >> (IP1553B_CDST_LAST_BIT);
    if (last < 0x10000 && last >= 0x10FFE) {
        return RTEMS_INVALID_NAME;
    }

    ofs = (last-0x10000)>>1;
    cmd->cmd  = priv->mem->cmd[ofs].cmd;
    cmd->ctrl = priv->mem->cmd[ofs].ctrl;
    cmd->info = priv->mem->cmd[ofs].info;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code ip1553b_ioctl_get_next(
        struct ip1553b_priv *priv, struct ip1553b_cmd *cmd)
{
    uint32_t last;
    int ofs;

    last = (priv->regs->CDST & IP1553B_CDST_LAST) >> (IP1553B_CDST_LAST_BIT);
    last = (last-0x10000);

    if (priv->next >= 0xF00 && priv->next > last) {
        return RTEMS_INVALID_NAME;
    }

    ofs = priv->next>>1;
    cmd->cmd  = priv->mem->cmd[ofs].cmd;
    cmd->ctrl = priv->mem->cmd[ofs].ctrl;
    cmd->info = priv->mem->cmd[ofs].info;
    priv->next += 2;

    if (priv->next >= 0xF00) {
        priv->next = 0;
    }

    return RTEMS_SUCCESSFUL;
}

void ip1553b_isr(void *arg)
{
    struct ip1553b_priv *priv = (struct ip1553b_priv *)arg;
    uint32_t pending;

    if (CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_MILBUS_NOM)) {
        pending = priv->regs->NIT;
        if (pending & IP1553B_NIT_TROK) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_NOM_TROK];
            isr->isr(isr->arg);
        }
        if (pending & IP1553B_NIT_SYNC) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_NOM_SYNC];
            isr->isr(isr->arg);
        }
        if (pending & IP1553B_NIT_DBC) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_NOM_DBC];
            isr->isr(isr->arg);
        }
        if (pending & IP1553B_NIT_SWITCH) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_NOM_SWITCH];
            isr->isr(isr->arg);
        }

        CIC_Clear_interrupt(AGGA4_INTERRUPT_CIC_MILBUS_NOM);
    }

    if (CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_MILBUS_RST)) {
		pending = priv->regs->RIT;
        if (pending & IP1553B_RIT_RST) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_RST];
            isr->isr(isr->arg);
        }

        CIC_Clear_interrupt(AGGA4_INTERRUPT_CIC_MILBUS_RST);
    }

    if (CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_MILBUS_ERR)) {
		pending = priv->regs->EIT;
        if (pending & IP1553B_EIT_MEM) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_ERR_MEM];
            isr->isr(isr->arg);
        }
        if (pending & IP1553B_EIT_1553) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_ERR_1553];
            isr->isr(isr->arg);
        }
        if (pending & IP1553B_EIT_RTAD) {
            struct ip1553b_isr_priv *isr = &priv->isr[IP1553B_INTERRUPT_ERR_RTAD];
            isr->isr(isr->arg);
        }

        CIC_Clear_interrupt(AGGA4_INTERRUPT_CIC_MILBUS_ERR);
    }
}

static rtems_status_code ip1553b_ioctl_isr(
        struct ip1553b_priv *priv, struct ip1553b_isr *isr)
{
    int cic;
    uint32_t mask;
    uint32_t irq_mask;
    uint32_t cf;
    rtems_interrupt_level level;

    if (isr == NULL || isr->irq < 0 || isr->irq >= IP1553B_INTERRUPT_NUM_IRQ) {
        return RTEMS_INVALID_NAME;
    }

    irq_mask = (1<<isr->irq);

    if (IP1553B_INTERRUPT_NOM_MASK & irq_mask) {
        cic = AGGA4_INTERRUPT_CIC_MILBUS_NOM;
        mask = IP1553B_INTERRUPT_NOM_MASK;

    } else if (IP1553B_INTERRUPT_RST_MASK & irq_mask) {
        cic = AGGA4_INTERRUPT_CIC_MILBUS_RST;
        mask = IP1553B_INTERRUPT_RST_MASK;

    } else if (IP1553B_INTERRUPT_ERR_MASK & irq_mask) {
        cic = AGGA4_INTERRUPT_CIC_MILBUS_ERR;
        mask = IP1553B_INTERRUPT_ERR_MASK;
    };

    cf = 0;
    if (isr->irq == IP1553B_INTERRUPT_NOM_TROK) {
        cf = IP1553B_CF_IT_TROK;
    } else if (isr->irq == IP1553B_INTERRUPT_NOM_SYNC) {
        cf = IP1553B_CF_IT_SYNC;
    } else if (isr->irq == IP1553B_INTERRUPT_ERR_1553) {
        cf = IP1553B_CF_ERR_1553;
    }

    rtems_interrupt_disable(level);

    priv->isr[isr->irq].isr = isr->isr;
    priv->isr[isr->irq].arg = isr->arg;
    if (isr->isr) {
        // Install interrupt handler
        if (cf) {
            priv->regs->CF |= cf;
        }
        if ((priv->mask & mask) == 0) {
            CIC_Unmask_interrupt(cic);
        }
        if (priv->mask == 0) {
            BSP_shared_interrupt_register(AGGA4_INTERRUPT_CIC,
                    "IP1553B", ip1553b_isr, priv);
        }
        priv->mask |= irq_mask;

    } else {
        // Unistall interrupt handler
        priv->mask &= ~irq_mask;
        if (priv->mask == 0) {
            BSP_shared_interrupt_unregister(AGGA4_INTERRUPT_CIC,
                    ip1553b_isr, priv);
        }
        if ((priv->mask & mask) == 0) {
            CIC_Mask_interrupt(cic);
        }
        if (cf) {
            priv->regs->CF &= ~cf;
        }
    }

    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

void ip1553b_init(struct ip1553b_priv *priv)
{
    memset(priv, 0, sizeof(ip1553b_priv));
    priv->regs = (struct ip1553b_regs *)&LEON_REG->C53_CF;
}

rtems_status_code ip1553b_open(
        struct ip1553b_priv *priv, rtems_device_minor_number minor)
{
    rtems_interval ticks_per_second;

    if (minor >= 1) {
        return RTEMS_TOO_MANY;
    }

    if (priv->open) {
        return RTEMS_RESOURCE_IN_USE;
    }

    priv->regs->CDST |= IP1553B_CDST_RST;
    rtems_clock_get( RTEMS_CLOCK_GET_TICKS_PER_SECOND, &ticks_per_second );
    rtems_task_wake_after( ticks_per_second / 1000 );
    priv->regs->CDST = 0;
    priv->regs->CF = 0;

    priv->open = 1;

    return RTEMS_SUCCESSFUL;
}

rtems_status_code ip1553b_close(
        struct ip1553b_priv *priv, rtems_device_minor_number minor)
{
    priv->open = 0;

    return RTEMS_SUCCESSFUL;
}

/*****************************************************************************
 * RTEMS I/O
 */
rtems_status_code ip1553b_io_initialize(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct ip1553b_priv *priv = &ip1553b_priv;

    ip1553b_init(priv);

    status = rtems_io_register_name(AGGA4_IP1553B_DEVNAME, major, 0);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code ip1553b_io_open(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct ip1553b_priv *priv = &ip1553b_priv;

    return ip1553b_open(priv, minor);
}

rtems_status_code ip1553b_io_close(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct ip1553b_priv *priv = &ip1553b_priv;

    return ip1553b_close(priv, minor);
}

rtems_status_code ip1553b_io_control(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct ip1553b_priv *priv = &ip1553b_priv;
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;
    switch(ioarg->command) {
    case IP1553B_IOCTL_START: {
        status = ip1553b_ioctl_start(priv);
        break;
    }
    case IP1553B_IOCTL_SET_MEM: {
        status = ip1553b_ioctl_set_mem(priv, ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_EXT: {
        status = ip1553b_ioctl_set_ext(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_TIMEOUT: {
        status = ip1553b_ioctl_set_timeout(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_CLK: {
        status = ip1553b_ioctl_set_clk(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_RTAD: {
        status = ip1553b_ioctl_set_rt_address(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_PARITY: {
        status = ip1553b_ioctl_set_parity(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_ISR: {
        status = ip1553b_ioctl_isr(priv, (struct ip1553b_isr *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_CHAR: {
        status = ip1553b_ioctl_set_char(priv, (struct ip1553b_char *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_SET_INDIRECTION: {
        status = ip1553b_ioctl_set_indirection(priv, (struct ip1553b_indirection *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_RXDATA: {
        status = ip1553b_ioctl_data(priv, 0, (struct ip1553b_data *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_RXBLOCK: {
        status = ip1553b_ioctl_block(priv, 0, (struct ip1553b_block *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_TXDATA: {
        status = ip1553b_ioctl_data(priv, 1, (struct ip1553b_data *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_TXBLOCK: {
        status = ip1553b_ioctl_block(priv, 1, (struct ip1553b_block *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_GET_LAST: {
        status = ip1553b_ioctl_get_last(priv, (struct ip1553b_cmd *)ioarg->buffer);
        break;
    }
    case IP1553B_IOCTL_GET_NEXT: {
        status = ip1553b_ioctl_get_next(priv, (struct ip1553b_cmd *)ioarg->buffer);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}
