#include <string.h>
#include <stdlib.h>

#include <rtems.h>
#include <rtems/libio.h>

#include <bsp.h>
#include <agga4/spi.h>

#define SPI_CFG_RST_BIT 1

#define SPI_CFG_RST (1<<SPI_CFG_RST_BIT) /* Module reset */

#define SPI_DMA_STS_XMT_DONE_BIT  10
#define SPI_DMA_STS_XMT_EMPTY_BIT  9
#define SPI_DMA_STS_RX_AVAIL_BIT   8

#define SPI_DMA_STS_XMT_DONE  (1<<SPI_DMA_STS_XMT_DONE_BIT)  /* Tx done*/
#define SPI_DMA_STS_XMT_EMPTY (1<<SPI_DMA_STS_XMT_EMPTY_BIT) /* Tx empty */
#define SPI_DMA_STS_RX_AVAIL  (1<<SPI_DMA_STS_RX_AVAIL_BIT)  /* Rx available */

#define SPI_DMA_STS_MASK 0x00000700

#define SPI_CFG_DL_BIT   16
#define SPI_CFG_SS_BIT   11
#define SPI_CFG_CS_BIT    7
#define SPI_CFG_TE_BIT    5
#define SPI_CFG_MSB_BIT   4
#define SPI_CFG_CSEL_BIT  3
#define SPI_CFG_CPHS_BIT  2
#define SPI_CFG_CPOL_BIT  1
#define SPI_CFG_RPOL_BIT  0

#define SPI_CFG_DL    (0x0F<<SPI_CFG_DL_BIT)   /* Data length */
#define SPI_CFG_SS    (0x1F<<SPI_CFG_SS_BIT)   /* Slave select */
#define SPI_CFG_CS    (   1<<SPI_CFG_CS_BIT)   /* Continuous */
#define SPI_CFG_TE    (   1<<SPI_CFG_TE_BIT)   /* Tx enable */
#define SPI_CFG_MSB   (   1<<SPI_CFG_MSB_BIT)  /* Tx enable */
#define SPI_CFG_CSEL  (   1<<SPI_CFG_CSEL_BIT) /* Clock select */
#define SPI_CFG_CPHS  (   1<<SPI_CFG_CPHS_BIT) /* Cloch phase*/
#define SPI_CFG_CPOL  (   1<<SPI_CFG_CPOL_BIT) /* Clock polarity */
#define SPI_CFG_RPOL  (   1<<SPI_CFG_RPOL_BIT) /* Receive clock polarity */

#define SPI_CLKDIV_BIT  0

#define SPI_CLKDIV    (0xFF<<SPI_CLKDIV_BIT) /* Clock divider */

struct spi_regs {
    volatile uint32_t Status_and_Ctrl;
    volatile uint32_t CLK_Divider;
    volatile uint32_t Tx;
    volatile uint32_t Rx;
};

static struct spi_priv {
    struct spi_regs *regs;
    struct spi_isr isr;
    int bytes;
    uint32_t block;
    rtems_id sem;
    char open;
} spi_priv;

void spi_isr(void *arg)
{
    struct spi_priv *priv = arg;

    if (CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_SPI)) {
        if (priv->block & SPI_BLOCKING) {
            rtems_semaphore_release(priv->sem);
        }

        if (priv->isr.isr) {
            priv->isr.isr(priv->isr.arg);
        }

        CIC_Clear_interrupt(AGGA4_INTERRUPT_CIC_SPI);
    }
}

void spi_install_isr(struct spi_priv *priv)
{
    BSP_shared_interrupt_register(AGGA4_INTERRUPT_CIC, "SPI", spi_isr, priv);
    CIC_Unmask_interrupt(AGGA4_INTERRUPT_CIC_SPI);
}

void spi_uninstall_isr(struct spi_priv *priv)
{
    BSP_shared_interrupt_unregister(AGGA4_INTERRUPT_CIC, spi_isr, priv);
    CIC_Mask_interrupt(AGGA4_INTERRUPT_CIC_SPI);
}

static rtems_status_code spi_ioctl_start(struct spi_priv *priv)
{
    if (priv->regs->Status_and_Ctrl & SPI_CFG_TE) {
        return RTEMS_RESOURCE_IN_USE;
    }
    priv->regs->Status_and_Ctrl |= SPI_CFG_TE;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spi_ioctl_stop(struct spi_priv *priv)
{
    while (!(priv->regs->Status_and_Ctrl & SPI_DMA_STS_XMT_DONE)) {
        ;
    }

    priv->regs->Status_and_Ctrl &= ~SPI_CFG_TE;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spi_ioctl_blocking(
        struct spi_priv *priv, uint32_t block)
{
    rtems_status_code status = RTEMS_SUCCESSFUL;
    rtems_interrupt_level level;

    rtems_interrupt_disable(level);

    if (!(priv->block & SPI_BLOCKING) && (block & SPI_BLOCKING)) {
        status = rtems_semaphore_create(
                rtems_build_name('S', 'P', 'I', '0'),
                0,
                RTEMS_FIFO | RTEMS_SIMPLE_BINARY_SEMAPHORE | \
                RTEMS_NO_INHERIT_PRIORITY | RTEMS_NO_PRIORITY_CEILING,
                0,
                &(priv->sem));

        if (status == RTEMS_SUCCESSFUL && !priv->isr.isr) {
            spi_install_isr(priv);
        }

    } else if ((priv->block & SPI_BLOCKING) && !(block & SPI_BLOCKING)) {
        if (!priv->isr.isr) {
            spi_uninstall_isr(priv);
        }
        rtems_semaphore_delete(priv->sem);
    }
    priv->block = block;

    rtems_interrupt_enable(level);

    return status;
}

static rtems_status_code spi_ioctl_get_status(
        struct spi_priv *priv, uint32_t *status)
{
    if (status) {
        *status = SPI_DMA_STS_MASK & priv->regs->Status_and_Ctrl;
        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

static rtems_status_code spi_ioctl_isr(
        struct spi_priv *priv, struct spi_isr *isr)
{
    rtems_interrupt_level level;

    if (isr == NULL) {
        return RTEMS_INVALID_NAME;
    }

    rtems_interrupt_disable(level);
    if (!priv->isr.isr && isr->isr) {
        priv->isr.isr = isr->isr;
        priv->isr.arg = isr->arg;
        if (!(priv->block & SPI_BLOCKING)) {
            spi_install_isr(priv);
        }
    } else if (priv->isr.isr && !isr->isr) {
        priv->isr.isr = NULL;
        priv->isr.arg = NULL;
        if (!(priv->block & SPI_BLOCKING)) {
            spi_uninstall_isr(priv);
        }
    } else {
        priv->isr.isr = isr->isr;
        priv->isr.arg = isr->arg;
    }
    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

rtems_status_code spi_ioctl_set_config(
        struct spi_priv *priv, struct spi_config *cfg)
{
    uint32_t config;
    uint32_t clkdiv;

    if (cfg) {
        if (priv->regs->Status_and_Ctrl & SPI_CFG_TE) {
            return RTEMS_RESOURCE_IN_USE;
        }

        if (cfg->length < 8 || cfg->length > 16) {
            return RTEMS_INVALID_NUMBER;
        }

        if (cfg->clk_div > 255) {
            return RTEMS_INVALID_NUMBER;
        }

        config = 0;
        config |= ((cfg->length-1)  << SPI_CFG_DL_BIT)   & SPI_CFG_DL;
        config |= (cfg->slave       << SPI_CFG_SS_BIT)   & SPI_CFG_SS;
        config |= (cfg->continous   << SPI_CFG_CS_BIT)   & SPI_CFG_CS;
        config |= (cfg->msb_first   << SPI_CFG_MSB_BIT)  & SPI_CFG_MSB;
        config |= (cfg->clk_sel     << SPI_CFG_CSEL_BIT) & SPI_CFG_CSEL;
        config |= (cfg->clk_phase   << SPI_CFG_CPHS_BIT) & SPI_CFG_CPHS;
        config |= (cfg->clk_pol     << SPI_CFG_CPOL_BIT) & SPI_CFG_CPOL;
        config |= (cfg->rcv_clk_pol << SPI_CFG_RPOL_BIT) & SPI_CFG_RPOL;
        priv->regs->Status_and_Ctrl = config;

        priv->bytes = (cfg->length+7)/8;

        clkdiv = 0;
        clkdiv |= ((cfg->clk_div) << SPI_CLKDIV_BIT) & SPI_CLKDIV;
        priv->regs->CLK_Divider = clkdiv;

        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}
rtems_status_code spi_ioctl_get_config(
        struct spi_priv *priv, struct spi_config *cfg)
{
    uint32_t config;
    uint32_t clkdiv;

    if (cfg) {
        config = priv->regs->Status_and_Ctrl;
        cfg->length      = ((config & SPI_CFG_DL ) >> SPI_CFG_DL_BIT) +1;
        cfg->slave       = (config & SPI_CFG_SS  ) >> SPI_CFG_SS_BIT;
        cfg->continous   = (config & SPI_CFG_CS  ) >> SPI_CFG_CS_BIT;
        cfg->started     = (config & SPI_CFG_TE  ) >> SPI_CFG_TE_BIT;
        cfg->msb_first   = (config & SPI_CFG_MSB ) >> SPI_CFG_MSB_BIT;
        cfg->clk_sel     = (config & SPI_CFG_CSEL) >> SPI_CFG_CSEL_BIT;
        cfg->clk_phase   = (config & SPI_CFG_CPHS) >> SPI_CFG_CPHS_BIT;
        cfg->clk_pol     = (config & SPI_CFG_CPOL) >> SPI_CFG_CPOL_BIT;
        cfg->rcv_clk_pol = (config & SPI_CFG_RPOL) >> SPI_CFG_RPOL_BIT;

        clkdiv = priv->regs->CLK_Divider;
        cfg->clk_div     = ((clkdiv & SPI_CLKDIV)   >> SPI_CLKDIV_BIT);

        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

void spi_init(struct spi_priv *priv)
{
    memset(priv, 0, sizeof(spi_priv));
    priv->regs = (struct spi_regs *)&LEON_REG->SPI_Status_and_Ctrl;
}

rtems_status_code spi_open(
        struct spi_priv *priv, rtems_device_minor_number minor)
{
    uint32_t dump;

    if (minor >= 1) {
        return RTEMS_TOO_MANY;
    }

    if (priv->open) {
        return RTEMS_RESOURCE_IN_USE;
    }

    priv->open = 1;
    priv->regs->Status_and_Ctrl &= ~SPI_CFG_TE;
    dump = priv->regs->Rx;

    return RTEMS_SUCCESSFUL;
}

rtems_status_code spi_close(
        struct spi_priv *priv, rtems_device_minor_number minor)
{
    struct spi_isr isr = {0};
    spi_ioctl_isr(priv, &isr);
    spi_ioctl_blocking(priv, SPI_NONBLOCKING); // Destroy semaphore

    priv->open = 0;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spi_read(
        struct spi_priv *priv, char *buf, size_t cnt, uint32_t *moved)
{
    uint32_t rx = 0;

    if ( (cnt < priv->bytes) || buf == NULL ) {
            return RTEMS_INVALID_NAME; /* EINVAL */
    }

    while (!(priv->regs->Status_and_Ctrl & SPI_DMA_STS_RX_AVAIL)) {
        if (!(priv->block & (SPI_BLOCKING_READ|SPI_POLLING_READ))) {
            return RTEMS_TIMEOUT; /* ETIMEDOUT */
        } else if (priv->block & SPI_BLOCKING_READ) {
            rtems_semaphore_obtain(priv->sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
            break;
        }
    }

    rx = priv->regs->Rx;
    buf[0] = rx & 0xFF;
    if (priv->bytes == 2) {
        buf[1] = (rx >> 8) & 0xFF;
    }
    *moved = priv->bytes;

    if (priv->block & SPI_BLOCKING) {
        rtems_semaphore_obtain(priv->sem, RTEMS_NO_WAIT, RTEMS_NO_TIMEOUT);
    }

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spi_write(
        struct spi_priv *priv, char *buf, size_t cnt, uint32_t *moved)
{
    uint32_t tx = 0;

    if ( (cnt < priv->bytes) || buf == NULL ) {
            return RTEMS_INVALID_NAME; /* EINVAL */
    }

    while (!(priv->regs->Status_and_Ctrl & SPI_DMA_STS_XMT_EMPTY)) {
        if (!(priv->block & (SPI_BLOCKING_WRITE|SPI_POLLING_WRITE))) {
            return RTEMS_TIMEOUT; /* ETIMEDOUT */
        } else if (priv->block & SPI_BLOCKING_WRITE) {
            rtems_semaphore_obtain(priv->sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
            break;
        }
    }

    if (priv->block & SPI_BLOCKING) {
        rtems_semaphore_obtain(priv->sem, RTEMS_NO_WAIT, RTEMS_NO_TIMEOUT);
    }

    if (priv->bytes == 2) {
        tx = ((uint8_t)buf[1]<<8)|(uint8_t)buf[0];
    } else {
        tx = (uint8_t)buf[0];
    }
    priv->regs->Tx = tx;
    *moved = priv->bytes;

    return RTEMS_SUCCESSFUL;
}

/*****************************************************************************
 * RTEMS I/O
 */
rtems_status_code spi_io_initialize(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;

    spi_init(&spi_priv);

    status = rtems_io_register_name(AGGA4_SPI_DEVNAME, major, 0);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code spi_io_open(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spi_priv *priv = &spi_priv;

    return spi_open(priv, minor);
}

rtems_status_code spi_io_close(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spi_priv *priv = &spi_priv;

    return spi_close(priv, minor);
}

rtems_status_code spi_io_read(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spi_priv *priv = &spi_priv;
    rtems_libio_rw_args_t *rw_args = (rtems_libio_rw_args_t*)argument;

    return spi_read(priv, rw_args->buffer,
            rw_args->count, &rw_args->bytes_moved);
}

rtems_status_code spi_io_write(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spi_priv *priv = &spi_priv;
    rtems_libio_rw_args_t *rw_args = (rtems_libio_rw_args_t*)argument;

    return spi_write(priv, rw_args->buffer,
            rw_args->count, &rw_args->bytes_moved);
}

rtems_status_code spi_io_control(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct spi_priv *priv = &spi_priv;
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;
    switch(ioarg->command) {
    case SPI_IOCTL_START: {
        status = spi_ioctl_start(priv);
        break;
    }
    case SPI_IOCTL_STOP: {
        status = spi_ioctl_stop(priv);
        break;
    }
    case SPI_IOCTL_BLOCKING: {
        status = spi_ioctl_blocking(priv, (uint32_t)ioarg->buffer);
        break;
    }
    case SPI_IOCTL_GET_STATUS: {
        status = spi_ioctl_get_status(priv, (uint32_t *)ioarg->buffer);
        break;
    }
    case SPI_IOCTL_SET_ISR: {
        struct spi_isr *isr = (struct spi_isr *)ioarg->buffer;
        status = spi_ioctl_isr(priv, isr);
        break;
    }
    case SPI_IOCTL_SET_CONFIG: {
        struct spi_config *cfg = (struct spi_config *)ioarg->buffer;
        status = spi_ioctl_set_config(priv, cfg);
        break;
    }
    case SPI_IOCTL_GET_CONFIG: {
        struct spi_config *cfg = (struct spi_config *)ioarg->buffer;
        status = spi_ioctl_get_config(priv, cfg);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}
