#include <string.h>
#include <stdlib.h>

#include <rtems.h>
#include <rtems/libio.h>

#include <bsp.h>
#include <agga4/spw.h>
#include <agga4/tc.h>


#define SPW_CFG_RST_BIT 1

#define SPW_CFG_RST (1<<SPW_CFG_RST_BIT) /* Module reset */

#define SPW_DMA_STS_FCT_BIT 12
#define SPW_DMA_STS_ESC_BIT 11
#define SPW_DMA_STS_PAR_BIT 10
#define SPW_DMA_STS_DIS_BIT  9
#define SPW_DMA_STS_LOK_BIT  8

#define SPW_DMA_STS_FCT (1<<SPW_DMA_STS_FCT_BIT) /* FCT error */
#define SPW_DMA_STS_ESC (1<<SPW_DMA_STS_ESC_BIT) /* ESC error */
#define SPW_DMA_STS_PAR (1<<SPW_DMA_STS_PAR_BIT) /* Parity error */
#define SPW_DMA_STS_DIS (1<<SPW_DMA_STS_DIS_BIT) /* Disconnect error */
#define SPW_DMA_STS_LOK (1<<SPW_DMA_STS_LOK_BIT) /* Link OK */

#define SPW_DMA_CTRL_BR_BIT 3
#define SPW_DMA_CTRL_ER_BIT 2
#define SPW_DMA_CTRL_AS_BIT 1
#define SPW_DMA_CTRL_LS_BIT 0

#define SPW_DMA_CTRL_BR (3<<SPW_DMA_CTRL_BR_BIT) /* TX Bit rate */
#define SPW_DMA_CTRL_ER (1<<SPW_DMA_CTRL_ER_BIT) /* Enable bit rate (else 10Mbit) */
#define SPW_DMA_CTRL_AS (1<<SPW_DMA_CTRL_AS_BIT) /* Auto start */
#define SPW_DMA_CTRL_LS (1<<SPW_DMA_CTRL_LS_BIT) /* Link start */

#define SPW_DMA_CFG_EEP_BIT 8
#define SPW_DMA_CFG_LEE_BIT 4
#define SPW_DMA_CFG_NSE_BIT 2
#define SPW_DMA_CFG_HFC_BIT 1

#define SPW_DMA_CFG_EEP (1<<SPW_DMA_CFG_EEP_BIT) /* Early end of packet  */
#define SPW_DMA_CFG_LEE (1<<SPW_DMA_CFG_LEE_BIT) /* Little endian enable */
#define SPW_DMA_CFG_NSE (1<<SPW_DMA_CFG_NSE_BIT) /* No stop on end of packet */
#define SPW_DMA_CFG_HFC (1<<SPW_DMA_CFG_HFC_BIT) /* Header field control bit */

#define SPW_TIME_CODE_FLG_BIT 6
#define SPW_TIME_CODE_VAL_BIT 0

#define SPW_TIME_CODE_FLG (0x3 <<SPW_TIME_CODE_FLG_BIT) /* Time code flags (always 0) */
#define SPW_TIME_CODE_VAL (0x1F<<SPW_TIME_CODE_VAL_BIT) /* Time code value */

#define SPW_TIME_CTRL_EA_BIT 1
#define SPW_TIME_CTRL_EV_BIT 0

#define SPW_TIME_CTRL_EA (1<<SPW_TIME_CTRL_IEA_BIT) /* TICK_INT Interrupt enable all */
#define SPW_TIME_CTRL_EV (1<<SPW_TIME_CTRL_IEV_BIT) /* TICK_INT Interrupt enable valid */

#define SPW_INT_TICK_IN_BIT  31
#define SPW_INT_SPW3_RXF_BIT 28
#define SPW_INT_SPW3_EOP_BIT 27
#define SPW_INT_SPW3_TXD_BIT 26
#define SPW_INT_SPW3_LC_BIT  25
#define SPW_INT_SPW3_LE_BIT  24
#define SPW_INT_SPW2_RXF_BIT 20
#define SPW_INT_SPW2_EOP_BIT 19
#define SPW_INT_SPW2_TXD_BIT 18
#define SPW_INT_SPW2_LC_BIT  17
#define SPW_INT_SPW2_LE_BIT  16
#define SPW_INT_SPW1_RXF_BIT 12
#define SPW_INT_SPW1_EOP_BIT 11
#define SPW_INT_SPW1_TXD_BIT 10
#define SPW_INT_SPW1_LC_BIT   9
#define SPW_INT_SPW1_LE_BIT   8
#define SPW_INT_SPW0_RXF_BIT  4
#define SPW_INT_SPW0_EOP_BIT  3
#define SPW_INT_SPW0_TXD_BIT  2
#define SPW_INT_SPW0_LC_BIT   1
#define SPW_INT_SPW0_LE_BIT   0

#define SPW_INT_TICK_IN (1<<SPW_INT_TICK_IN_BIT)
#define SPW_INT_SPW3_RXF    (1<<SPW_INT_SPW3_RXF_BIT)
#define SPW_INT_SPW3_EOP    (1<<SPW_INT_SPW3_EOP_BIT)
#define SPW_INT_SPW3_TXD    (1<<SPW_INT_SPW3_TXD_BIT)
#define SPW_INT_SPW3_LC     (1<<SPW_INT_SPW3_LC_BIT)
#define SPW_INT_SPW3_LE     (1<<SPW_INT_SPW3_LE_BIT)
#define SPW_INT_SPW2_RXF    (1<<SPW_INT_SPW2_RXF_BIT)
#define SPW_INT_SPW2_EOP    (1<<SPW_INT_SPW2_EOP_BIT)
#define SPW_INT_SPW2_TXD    (1<<SPW_INT_SPW2_TXD_BIT)
#define SPW_INT_SPW2_LC     (1<<SPW_INT_SPW2_LC_BIT)
#define SPW_INT_SPW2_LE     (1<<SPW_INT_SPW2_LE_BIT)
#define SPW_INT_SPW1_RXF    (1<<SPW_INT_SPW1_RXF_BIT)
#define SPW_INT_SPW1_EOP    (1<<SPW_INT_SPW1_EOP_BIT)
#define SPW_INT_SPW1_TXD    (1<<SPW_INT_SPW1_TXD_BIT)
#define SPW_INT_SPW1_LC     (1<<SPW_INT_SPW1_LC_BIT)
#define SPW_INT_SPW1_LE     (1<<SPW_INT_SPW1_LE_BIT)
#define SPW_INT_SPW0_RXF    (1<<SPW_INT_SPW0_RXF_BIT)
#define SPW_INT_SPW0_EOP    (1<<SPW_INT_SPW0_EOP_BIT)
#define SPW_INT_SPW0_TXD    (1<<SPW_INT_SPW0_TXD_BIT)
#define SPW_INT_SPW0_LC     (1<<SPW_INT_SPW0_LC_BIT)
#define SPW_INT_SPW0_LE     (1<<SPW_INT_SPW0_LE_BIT)

/* SPW Error Condition */
#define SPW_DMA_STATUS_MASK 0x1F00
#define SPW_DMA_CTRL_MASK   0x001F
#define SPW_DMA_RATE_MASK   0x001C

#define SPW_DMA_MINOR 4

#define SPW_ISR_TICK_IN 31

#define SPW_ISR_DISABLE_AFTER_IRQ (\
        (1<<(SPW_ISR_RXAREA+ 0)) | \
		(1<<(SPW_ISR_RXAREA+ 8)) | \
        (1<<(SPW_ISR_RXAREA+16)) | \
		(1<<(SPW_ISR_RXAREA+24)) | \
        (1<<(SPW_ISR_EOP+ 0)) |    \
        (1<<(SPW_ISR_EOP+ 8)) |    \
        (1<<(SPW_ISR_EOP+16)) |    \
        (1<<(SPW_ISR_EOP+24)) )
#define SPW_ISR_CLEAR_AFTER_IRQ (\
        (1<<(SPW_ISR_LINK_CON+ 0)) | \
        (1<<(SPW_ISR_LINK_CON+ 8)) | \
        (1<<(SPW_ISR_LINK_CON+16)) | \
		(1<<(SPW_ISR_LINK_CON+24)) | \
        (1<<(SPW_ISR_LINK_ERR+ 0)) | \
        (1<<(SPW_ISR_LINK_ERR+ 8)) | \
        (1<<(SPW_ISR_LINK_ERR+16)) | \
        (1<<(SPW_ISR_LINK_ERR+24)) | \
        (1<<(SPW_ISR_TXDONE+ 0)) |   \
        (1<<(SPW_ISR_TXDONE+ 8)) |   \
        (1<<(SPW_ISR_TXDONE+16)) |   \
        (1<<(SPW_ISR_TXDONE+24)) |   \
		(1<<SPW_ISR_TICK_IN))

struct spw_dma_regs {
    volatile uint32_t Status_and_Ctrl;
    volatile uint32_t _dummy0;
    volatile uint32_t Tx_SAP;
    volatile uint32_t Tx_EAP;
    volatile uint32_t Tx_CAP;
    volatile uint32_t Tx_Rx_Config;
    volatile uint32_t Rx_SAP;
    volatile uint32_t Rx_EAP;
    volatile uint32_t Rx_CAP;
    volatile uint32_t _dummy1;
    volatile uint32_t _dummy2;
    volatile uint32_t _dummy3;
    volatile uint32_t _dummy4;
    volatile uint32_t _dummy5;
    volatile uint32_t _dummy6;
    volatile uint32_t _dummy7;
};

struct spw_tc_regs {
    volatile uint32_t Time_Ctrl;
    volatile uint32_t Time_Code;
};

struct spw_regs {
    struct spw_dma_regs dma[SPW_DMA_MINOR];
    volatile uint32_t Config;
    struct spw_tc_regs tc;
    volatile uint32_t Int_Mask;
    volatile uint32_t Int_Status;
    volatile uint32_t Int_Clear;
};

struct spw_dma_priv {
    struct spw_dma_regs *regs;
    struct {
        spw_isr_func isr;
        void *arg;
    } isr[5];
    char open;
};

struct spw_tc_priv {
    struct spw_tc_regs *regs;
    struct {
        spw_isr_func isr;
        void *arg;
    } isr;
    char open;
};

static struct spw_priv {
    struct spw_regs *regs;
    struct spw_dma_priv dma[SPW_DMA_MINOR];
    struct spw_tc_priv tc;
    uint32_t irqmask;
} spw_priv;

/*****************************************************************************
 * SpaceWire module functions
 */
static void spw_isr(void *arg)
{
    int i;
    struct spw_priv *priv = &spw_priv;

    int irq;
    uint32_t mask;
    uint32_t pending = priv->regs->Int_Status;
    uint32_t enabled = priv->regs->Int_Mask;
    uint32_t clr = 0;

    mask = 1<<SPW_ISR_TICK_IN;
    if ((pending & mask) && priv->tc.isr.isr) {
        priv->tc.isr.isr(priv->tc.isr.arg);
        clr |= mask;
    }

    for (i=0; i<SPW_DMA_MINOR; i++) {
        struct spw_dma_priv *dma = &priv->dma[i];
        int shift = 8*i;

        for (irq=SPW_ISR_LINK_ERR; irq<=SPW_ISR_RXAREA; irq++) {
            mask = 1<<(irq+shift);
            if ((pending & mask) && (enabled & mask) && dma->isr[irq].isr) {
                dma->isr[irq].isr(dma->isr[irq].arg);
                clr |= mask;
            }
        }
    }

    priv->regs->Int_Clear = clr & SPW_ISR_CLEAR_AFTER_IRQ;
    priv->regs->Int_Mask &= ~(clr & SPW_ISR_DISABLE_AFTER_IRQ);
}

static void spw_install_isr(struct spw_priv *priv, uint32_t mask)
{
    priv->regs->Int_Clear = mask & SPW_ISR_CLEAR_AFTER_IRQ;
    if (priv->irqmask == 0) {
        CIC_Unmask_interrupt(AGGA4_INTERRUPT_CIC_SPW);
        BSP_shared_interrupt_register(AGGA4_INTERRUPT_CIC,
                "SpW", spw_isr, priv);
    }
    priv->irqmask |= mask;
    priv->regs->Int_Mask |= mask & SPW_ISR_CLEAR_AFTER_IRQ;
}

static void spw_uninstall_isr(struct spw_priv *priv, uint32_t mask)
{
    priv->regs->Int_Mask &= ~mask;
    priv->irqmask &= ~mask;
    if (priv->irqmask == 0) {
        CIC_Mask_interrupt(AGGA4_INTERRUPT_CIC_SPW);
        BSP_shared_interrupt_unregister(AGGA4_INTERRUPT_CIC,
                spw_isr, priv);
    }
}

void spw_init(struct spw_priv *priv)
{
    static int spw_is_init;
    if (spw_is_init) {
        return;
    }
    spw_is_init = 1;

    priv->regs = (struct spw_regs *)&LEON_REG->SpW0_Status_and_Ctrl;
    priv->regs->Config = SPW_CFG_RST;
    priv->irqmask = 0;
}

/*****************************************************************************
 * SpaceWire DMA functions
 */
static rtems_status_code spw_ioctl_start(
        struct spw_priv *priv, int minor)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    dma->regs->Status_and_Ctrl |= SPW_DMA_CTRL_LS;
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_stop(
        struct spw_priv *priv, int minor)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    dma->regs->Status_and_Ctrl = 0;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_autostart(
        struct spw_priv *priv, int minor)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    dma->regs->Status_and_Ctrl |= SPW_DMA_CTRL_AS;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_get_status(
        struct spw_priv *priv, int minor, uint32_t *status)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    if (status) {
        *status = SPW_DMA_STATUS_MASK & dma->regs->Status_and_Ctrl;
        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

static rtems_status_code spw_ioctl_set_rate(
        struct spw_priv *priv, int minor, int rate)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    uint32_t ctrl;
    ctrl = dma->regs->Status_and_Ctrl;
    ctrl &= ~SPW_DMA_RATE_MASK;
    ctrl |= SPW_DMA_RATE_MASK & rate;
    dma->regs->Status_and_Ctrl = ctrl;

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_set_endian(
        struct spw_priv *priv, int minor, int endian)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    if (endian) {
        dma->regs->Tx_Rx_Config |= SPW_DMA_CFG_LEE;
    } else {
        dma->regs->Tx_Rx_Config &= ~SPW_DMA_CFG_LEE;
    }

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_set_stop_eop(
        struct spw_priv *priv, int minor, int stop)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    if (stop) {
        dma->regs->Tx_Rx_Config &= ~SPW_DMA_CFG_NSE;
    } else {
        dma->regs->Tx_Rx_Config |= SPW_DMA_CFG_NSE;
    }

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_rx_poll(
        struct spw_priv *priv, int minor)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    uint32_t eap = dma->regs->Rx_EAP;
    uint32_t cap = dma->regs->Rx_CAP;
    uint32_t sts = priv->regs->Int_Status & (((1<<SPW_ISR_EOP)|(1<<SPW_ISR_RXAREA))<<(minor*8));

    if(cap <= eap && sts==0) {
        return RTEMS_TIMEOUT; /* ETIMEDOUT */
    }
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_tx_poll(
        struct spw_priv *priv, int minor)
{
    struct spw_dma_priv *dma = &priv->dma[minor];

    uint32_t eap = dma->regs->Tx_EAP;
    uint32_t cap = dma->regs->Tx_CAP;

    if(cap <= eap) {
        return RTEMS_TIMEOUT; /* ETIMEDOUT */
    }
    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_rx(
        struct spw_priv *priv, int minor, struct spw_desc *desc)
{
    struct spw_dma_priv *dma = &priv->dma[minor];
    uint32_t mask;

    dma->regs->Rx_SAP = (uint32_t)desc->addr;
    mask = ((1<<SPW_ISR_EOP)|(1<<SPW_ISR_RXAREA))<<(minor*8);
    priv->regs->Int_Clear = mask;
    priv->regs->Int_Mask |= priv->irqmask & mask;
    dma->regs->Rx_EAP = (uint32_t)(desc->addr+desc->bytes-1);

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_tx(
        struct spw_priv *priv, int minor, struct spw_desc *desc)
{
    struct spw_dma_priv *dma = &priv->dma[minor];
    uint32_t mask;

    dma->regs->Tx_SAP = (uint32_t)desc->addr;
    mask = (1<<SPW_ISR_TXDONE)<<(minor*8);
    priv->regs->Int_Clear = mask;
    dma->regs->Tx_EAP = (uint32_t)(desc->addr+desc->bytes-1);

    return RTEMS_SUCCESSFUL;
}

static rtems_status_code spw_ioctl_isr(
        struct spw_priv *priv, int minor, struct spw_isr *isr)
{
    struct spw_dma_priv *dma = &priv->dma[minor];
    rtems_interrupt_level level;
    uint32_t mask;

    if (isr == NULL) {
        return RTEMS_INVALID_NAME;
    }

    rtems_interrupt_disable(level);

    switch (isr->irq) {
    default:
        return RTEMS_INVALID_NAME;
    case SPW_ISR_RXAREA:
    case SPW_ISR_EOP:
    case SPW_ISR_TXDONE:
    case SPW_ISR_LINK_CON:
    case SPW_ISR_LINK_ERR:
        dma->isr[isr->irq].isr = isr->isr;
        dma->isr[isr->irq].arg = isr->arg;
        mask = 1<<(minor*8+isr->irq);
        break;
    }

    if (isr->isr) {
        spw_install_isr(priv, mask);
    } else {
        spw_uninstall_isr(priv, mask);
   }
    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

void spw_dma_init(struct spw_priv *priv)
{
    int i;
    struct spw_dma_priv *dma;

    spw_init(priv);

    for (i=0; i<SPW_DMA_MINOR; i++) {
        dma = &priv->dma[i];
        memset(dma, 0, sizeof(struct spw_dma_priv));

        dma->regs = (struct spw_dma_regs *)&priv->regs->dma[i];
    }
}

rtems_status_code spw_dma_open(
        struct spw_priv *priv, rtems_device_minor_number minor)
{
    struct spw_dma_priv *dma;

    if (minor >= SPW_DMA_MINOR) {
        return RTEMS_TOO_MANY;
    }

    dma = &priv->dma[minor];
    if (dma->open) {
        return RTEMS_RESOURCE_IN_USE;
    }
    dma->open = 1;

    return RTEMS_SUCCESSFUL;
}

rtems_status_code spw_dma_close(
        struct spw_priv *priv, rtems_device_minor_number minor)
{
    struct spw_dma_priv *dma;

    spw_uninstall_isr(priv, 0x1F<<(8*minor));
    dma = &priv->dma[minor];
    dma->open = 0;

    return RTEMS_SUCCESSFUL;
}


/*****************************************************************************
 * SpaceWire TimeCode functions
 */
static rtems_status_code tc_ioctl_isr(
        struct spw_priv *priv, struct tc_isr *isr)
{
    struct spw_tc_priv *tc = &priv->tc;
    rtems_interrupt_level level;
    uint32_t mask;

    if (isr == NULL) {
        return RTEMS_INVALID_NAME;
    }

    rtems_interrupt_disable(level);

    mask = 1<<SPW_ISR_TICK_IN;
    tc->isr.isr = isr->isr;
    tc->isr.arg = isr->arg;

    if (isr->isr) {
        spw_install_isr(priv, mask);
    } else {
        /* Uninstall ISR */
        spw_uninstall_isr(priv, mask);
    }
    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

rtems_status_code tc_ioctl_set_timectrl(
        struct spw_priv *priv, int ctrl)
{
    struct spw_tc_priv *tc = &priv->tc;

    tc->regs->Time_Ctrl = ctrl & 0x3;

    return RTEMS_SUCCESSFUL;
}
rtems_status_code tc_ioctl_set_timecode(
        struct spw_priv *priv, struct tc_timecode *code)
{
    struct spw_tc_priv *tc = &priv->tc;

    // Ignore flags, they must always be zero.
    if (code) {
        tc->regs->Time_Code = code->code & 0x3f;

        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}
rtems_status_code tc_ioctl_get_timecode(
        struct spw_priv *priv, struct tc_timecode *code)
{
    struct spw_tc_priv *tc = &priv->tc;

    if (code) {
        uint32_t time_code = tc->regs->Time_Code;
        code->code = time_code & 0x3f;
        code->flag = (time_code >> 6) & 3;

        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

void spw_tc_init(struct spw_priv *priv)
{
    struct spw_tc_priv *tc;

    spw_init(priv);

    tc = &priv->tc;
    memset(tc, 0, sizeof(struct spw_tc_priv));
    tc->regs = &priv->regs->tc;
}

rtems_status_code spw_tc_open(
        struct spw_priv *priv, rtems_device_minor_number minor)
{
    struct spw_tc_priv *tc;

    if (minor >= 1) {
        return RTEMS_TOO_MANY;
    }

    tc = &priv->tc;
    if (tc->open) {
        return RTEMS_RESOURCE_IN_USE;
    }
    tc->open = 1;

    return RTEMS_SUCCESSFUL;
}

rtems_status_code spw_tc_close(
        struct spw_priv *priv, rtems_device_minor_number minor)
{
    struct spw_tc_priv *tc;

    spw_uninstall_isr(priv, 1<<SPW_ISR_TICK_IN);
    tc = &priv->tc;
    tc->open = 0;

    return RTEMS_SUCCESSFUL;
}


/*****************************************************************************
 * RTEMS I/O SpaceWire DMA
 */
rtems_status_code spw_io_initialize(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct spw_priv *priv = &spw_priv;
    int i;
    char *name[SPW_DMA_MINOR] = {
    		AGGA4_SPW0_DEVNAME,
    		AGGA4_SPW1_DEVNAME,
    		AGGA4_SPW2_DEVNAME,
    		AGGA4_SPW3_DEVNAME,
    };

    spw_dma_init(priv);

    for (i=0; i<SPW_DMA_MINOR; i++) {
        status = rtems_io_register_name(name[i], major, i);
        if (status != RTEMS_SUCCESSFUL) {
            return status;
        }
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code spw_io_open(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spw_priv *priv = &spw_priv;

    return spw_dma_open(priv, minor);
}

rtems_status_code spw_io_close(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spw_priv *priv = &spw_priv;

    return spw_dma_close(priv, minor);
}

rtems_status_code spw_io_control(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct spw_priv *priv = &spw_priv;
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;
    switch(ioarg->command) {
    case SPW_IOCTL_START: {
        status = spw_ioctl_start(priv, minor);
        break;
    }
    case SPW_IOCTL_STOP: {
        status = spw_ioctl_stop(priv, minor);
        break;
    }
    case SPW_IOCTL_AUTOSTART: {
        status = spw_ioctl_autostart(priv, minor);
        break;
    }
    case SPW_IOCTL_RX: {
        struct spw_desc *desc = (struct spw_desc *)ioarg->buffer;
        status = spw_ioctl_rx(priv, minor, desc);
        break;
    }
    case SPW_IOCTL_TX: {
        struct spw_desc *desc = (struct spw_desc *)ioarg->buffer;
        status = spw_ioctl_tx(priv, minor, desc);
        break;
    }
    case SPW_IOCTL_RX_POLL: {
        status = spw_ioctl_rx_poll(priv, minor);
        break;
    }
    case SPW_IOCTL_TX_POLL: {
        status = spw_ioctl_tx_poll(priv, minor);
        break;
    }
    case SPW_IOCTL_GET_STATUS: {
        status = spw_ioctl_get_status(priv, minor, (uint32_t *)ioarg->buffer);
        break;
    }
    case SPW_IOCTL_SET_RATE: {
        status = spw_ioctl_set_rate(priv, minor, (int)ioarg->buffer);
        break;
    }
    case SPW_IOCTL_SET_ENDIAN: {
        status = spw_ioctl_set_endian(priv, minor, (int)ioarg->buffer);
        break;
    }
    case SPW_IOCTL_SET_STOP_EOP: {
        status = spw_ioctl_set_stop_eop(priv, minor, (int)ioarg->buffer);
        break;
    }
    case SPW_IOCTL_SET_ISR: {
        struct spw_isr *isr = (struct spw_isr *)ioarg->buffer;
        status = spw_ioctl_isr(priv, minor, isr);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}

/*****************************************************************************
 * RTEMS I/O SpaceWire TimeCode
 */
rtems_status_code tc_io_initialize(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct spw_priv *priv = &spw_priv;

    spw_tc_init(priv);

    status = rtems_io_register_name(AGGA4_TC_DEVNAME, major, 0);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code tc_io_open(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spw_priv *priv = &spw_priv;

    return spw_tc_open(priv, minor);
}

rtems_status_code tc_io_close(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    struct spw_priv *priv = &spw_priv;

    return spw_tc_close(priv, minor);
}

rtems_status_code tc_io_control(
    rtems_device_major_number major,
    rtems_device_minor_number minor,
    void *argument)
{
    rtems_status_code status;
    struct spw_priv *priv = &spw_priv;
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;
    switch(ioarg->command) {
    case TC_IOCTL_SET_ISR: {
        struct tc_isr *isr = (struct tc_isr *)ioarg->buffer;
        status = tc_ioctl_isr(priv, isr);
        break;
    }
    case TC_IOCTL_SET_TIMECTRL: {
        status = tc_ioctl_set_timectrl(priv, (int)ioarg->buffer);
        break;
    }
    case TC_IOCTL_SET_TIMECODE: {
        struct tc_timecode *tc = (struct tc_timecode *)ioarg->buffer;
        status = tc_ioctl_set_timecode(priv, tc);
        break;
    }
    case TC_IOCTL_GET_TIMECODE: {
        struct tc_timecode *tc = (struct tc_timecode *)ioarg->buffer;
        status = tc_ioctl_get_timecode(priv, tc);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}
