#include <string.h>
#include <stdlib.h>

#include <rtems.h>
#include <rtems/libio.h>

#include <bsp.h>
#include <agga4/uart.h>


#define UART_RST_BIT 0

#define UART_RST (1<<UART_RST_BIT) /* Module reset */

#define UART_STS_RTO_BIT 7
#define UART_STS_FER_BIT 6
#define UART_STS_PER_BIT 5
#define UART_STS_OVR_BIT 4
#define UART_STS_BR_BIT  3
#define UART_STS_TE_BIT  2
#define UART_STS_TSE_BIT 1
#define UART_STS_DR_BIT  0

#define UART_STS_RTO (1<<UART_STS_RTO_BIT) /* Rx timeout */
#define UART_STS_FER (1<<UART_STS_FER_BIT) /* Framing error */
#define UART_STS_PER (1<<UART_STS_PER_BIT) /* Parity error */
#define UART_STS_OVR (1<<UART_STS_OVR_BIT) /* Rx overrun */
#define UART_STS_BR  (1<<UART_STS_BR_BIT)  /* Break recieved */
#define UART_STS_TE  (1<<UART_STS_TE_BIT)  /* Tx buffer empty */
#define UART_STS_TSE (1<<UART_STS_TSE_BIT) /* Tx shift register empty */
#define UART_STS_DR  (1<<UART_STS_DR_BIT)  /* Data ready */

#define UART_CTRL_LEND_BIT 9
#define UART_CTRL_TOE_BIT  8
#define UART_CTRL_LB_BIT   7
#define UART_CTRL_FL_BIT   6
#define UART_CTRL_PE_BIT   5
#define UART_CTRL_PS_BIT   4
#define UART_CTRL_TXEN_BIT 1
#define UART_CTRL_RXEN_BIT 0

#define UART_CTRL_LEND (3<<UART_CTRL_LEND_BIT) /* Little or Big Endian */
#define UART_CTRL_TOE  (1<<UART_CTRL_TOE_BIT)  /* Timeout enable */
#define UART_CTRL_LB   (1<<UART_CTRL_LB_BIT)   /* Loopback */
#define UART_CTRL_FL   (1<<UART_CTRL_FL_BIT)   /* Flow control */
#define UART_CTRL_PE   (3<<UART_CTRL_PE_BIT)   /* Parity enable */
#define UART_CTRL_PS   (1<<UART_CTRL_PS_BIT)   /* Parity even or odd */
#define UART_CTRL_TXEN (1<<UART_CTRL_TXEN_BIT) /* Tx enable */
#define UART_CTRL_RXEN (1<<UART_CTRL_RXEN_BIT) /* Rx enable */

#define UART_SCAL_TCNT_BIT 16
#define UART_SCAL_SCAL_BIT 0

#define UART_SCAL_TCNT (0xFF<<UART_SCAL_TCNT_BIT)  /* Timeout count */
#define UART_SCAL_SCAL (0xFFF<<UART_SCAL_SCAL_BIT) /* Scaler value */

#define UART_INT_UART_ERR  2
#define UART_INT_UART_TX   1
#define UART_INT_UART_RX   0

#define UART_NUM_MINOR 2
#define UART_NUM_CIC 3

struct uart_dma_regs {
    volatile uint32_t Tx_SAP;
    volatile uint32_t Tx_EAP;
    volatile uint32_t Tx_CAP;
    volatile uint32_t Rx_SAP;
    volatile uint32_t Rx_EAP;
    volatile uint32_t Rx_CAP;
    volatile uint32_t Status;
    volatile uint32_t Ctrl;
    volatile uint32_t Scaler;
};

struct uart_rst_regs {
    volatile uint32_t reset;
};

struct uart_regs {
    struct uart_dma_regs dma[UART_NUM_MINOR];
    struct uart_rst_regs rst;
};

struct uart_dma_priv {
    struct uart_dma_regs *regs;
    int cic[UART_NUM_CIC];
    struct {
        uart_isr_func isr;
        void *arg;
    } isr[UART_NUM_CIC];
    char open;
};

static struct uart_priv {
    struct uart_regs *regs;
    struct uart_dma_priv dma[UART_NUM_MINOR];
} uart_priv;

/*****************************************************************************
 * UART module functions
 */
static void uart_isr(void *arg)
{
    struct uart_priv *priv = arg;
    struct uart_dma_priv *dma;
    int minor;
    int irq;

    for (minor = 0; minor < UART_NUM_MINOR; minor ++) {
        dma = &priv->dma[minor];
        if (dma->open) {
            for (irq = 0; irq < UART_NUM_CIC; irq++) {
                if (CIC_Is_interrupt_pending(dma->cic[irq])) {
                    if (dma->isr[irq].isr) {
                        dma->isr[irq].isr(dma->isr[irq].arg);
                    }

                    CIC_Clear_interrupt(dma->cic[irq]);
                }
            }
        }
    }
}

#define UART_CIC_MASK ( \
		(1<<AGGA4_INTERRUPT_CIC_UART0_RXDONE)| \
		(1<<AGGA4_INTERRUPT_CIC_UART0_TXDONE)| \
		(1<<AGGA4_INTERRUPT_CIC_UART0_ERR)   | \
		(1<<AGGA4_INTERRUPT_CIC_UART1_RXDONE)| \
		(1<<AGGA4_INTERRUPT_CIC_UART1_TXDONE)| \
		(1<<AGGA4_INTERRUPT_CIC_UART1_ERR)     \
)

static inline void uart_install_isr(struct uart_priv *priv, int cic)
{
    if ((LEON_REG->CIC_Mask & UART_CIC_MASK) == 0) {
        BSP_shared_interrupt_register(AGGA4_INTERRUPT_CIC, "UART", uart_isr, priv);
    }
    CIC_Unmask_interrupt(cic);
}

static inline void uart_uninstall_isr(struct uart_priv *priv, int cic)
{
    CIC_Mask_interrupt(cic);
    if ((LEON_REG->CIC_Mask & UART_CIC_MASK) == 0) {
        BSP_shared_interrupt_unregister(AGGA4_INTERRUPT_CIC, uart_isr, priv);
    }
}

/*****************************************************************************
 * UART DMA functions
 */
static inline rtems_status_code uart_ioctl_start(
        struct uart_priv *priv, int minor)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    dma->regs->Ctrl |= UART_CTRL_TXEN|UART_CTRL_RXEN;
    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_stop(
        struct uart_priv *priv, int minor)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    dma->regs->Ctrl &=  ~(UART_CTRL_TXEN|UART_CTRL_RXEN);

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_get_status(
        struct uart_priv *priv, int minor, uint32_t *status)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    if (status) {
        *status = dma->regs->Status;
        return RTEMS_SUCCESSFUL;
    }

    return RTEMS_INVALID_NAME;
}

static inline rtems_status_code uart_ioctl_clr_status(
        struct uart_priv *priv, int minor, uint32_t status)
{
    struct uart_dma_priv *dma = &priv->dma[minor];


    dma->regs->Status = ~status;
    return RTEMS_SUCCESSFUL;
}


static inline rtems_status_code uart_ioctl_set_rate(
        struct uart_priv *priv, int minor, uint32_t rate)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    uint32_t scaler;
    uint32_t freq = agga4_freq_get();

    scaler = dma->regs->Scaler;
    scaler &= ~UART_SCAL_SCAL;
    scaler |= UART_SCAL_SCAL & ((freq/(rate*8)) - 1);
    dma->regs->Scaler = scaler;

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_get_rate(
        struct uart_priv *priv, int minor, uint32_t *rate)
{
    struct uart_dma_priv *dma = &priv->dma[minor];
    uint32_t scaler = dma->regs->Scaler & UART_SCAL_SCAL;
    uint32_t freq = agga4_freq_get();

    *rate = freq / ((scaler + 1) * 8);

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_set_endian(
        struct uart_priv *priv, int minor, int endian)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    if (endian) {
        dma->regs->Ctrl |= UART_CTRL_LEND;
    } else {
        dma->regs->Ctrl &= ~UART_CTRL_LEND;
    }

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_set_parity(
        struct uart_priv *priv, int minor, int parity)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    if (parity == UART_PARITY_EVEN) {
        dma->regs->Ctrl &= ~UART_CTRL_PS;
        dma->regs->Ctrl |=  UART_CTRL_PE;
    } else if (parity == UART_PARITY_ODD) {
        dma->regs->Ctrl |= (UART_CTRL_PE|UART_CTRL_PS);
    } else {
        dma->regs->Ctrl &= ~(UART_CTRL_PE|UART_CTRL_PS);
    }

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_get_parity(
        struct uart_priv *priv, int minor, int *parity)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    switch (dma->regs->Ctrl & (UART_CTRL_PE|UART_CTRL_PS)) {
    case (UART_CTRL_PE|UART_CTRL_PS):
    			        *parity = UART_PARITY_ODD;
    break;
    case UART_CTRL_PE:
        *parity = UART_PARITY_EVEN;
        break;
    default:
        *parity = UART_PARITY_NONE;
        break;
    }
    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_set_flow(
        struct uart_priv *priv, int minor, int flow)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    if (flow) {
        dma->regs->Ctrl |= UART_CTRL_FL;
    } else {
        dma->regs->Ctrl &= ~UART_CTRL_FL;
    }

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_get_flow(
        struct uart_priv *priv, int minor, int *flow)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    *flow = (dma->regs->Ctrl & UART_CTRL_FL) >> UART_CTRL_FL_BIT;

    return RTEMS_SUCCESSFUL;
}


static inline rtems_status_code uart_ioctl_rx_poll(
        struct uart_priv *priv, int minor, int *nchar)

{
    struct uart_dma_priv *dma = &priv->dma[minor];

    uint32_t sap = dma->regs->Rx_SAP;
    uint32_t eap = dma->regs->Rx_EAP;
    uint32_t cap = dma->regs->Rx_CAP;

    if (nchar) {
        *nchar = cap - sap;
    }
    /* Assumption: if sap, eap and cap are all zero, then the dev is reset */
    if(cap <= eap && (sap|eap|cap) != 0) {
        return RTEMS_TIMEOUT; /* EWOULDBLOCK / EAGAIN */
    }
    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_tx_poll(
        struct uart_priv *priv, int minor)
{
    struct uart_dma_priv *dma = &priv->dma[minor];

    uint32_t sap = dma->regs->Tx_SAP;
    uint32_t eap = dma->regs->Tx_EAP;
    uint32_t cap = dma->regs->Tx_CAP;

    /* Assumption: if sap, eap and cap are all zero, then the dev is reset */
    if(cap <= eap && (sap|eap|cap) != 0) {
        return RTEMS_TIMEOUT; /* EWOULDBLOCK / EAGAIN */
    }
    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_rx_arm(
        struct uart_priv *priv, int minor, struct uart_desc *desc)

{
    struct uart_dma_priv *dma = &priv->dma[minor];

    dma->regs->Rx_SAP = (uint32_t)desc->addr;
    dma->regs->Rx_EAP = (uint32_t)(desc->addr+desc->bytes-1);

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_tx_arm(
        struct uart_priv *priv, int minor, struct uart_desc *desc)

{
    struct uart_dma_priv *dma = &priv->dma[minor];

    dma->regs->Tx_SAP = (uint32_t)desc->addr;
    dma->regs->Tx_EAP = (uint32_t)(desc->addr+desc->bytes-1);

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_ioctl_isr(
        struct uart_priv *priv, int minor, struct uart_isr *isr)
{
    struct uart_dma_priv *dma = &priv->dma[minor];
    rtems_interrupt_level level;

    if (isr == NULL) {
        return RTEMS_INVALID_NAME;
    } else if (isr->irq < 0 || isr->irq > 2) {
        return RTEMS_INVALID_NAME;
    }

    rtems_interrupt_disable(level);

    dma->isr[isr->irq].isr = isr->isr;
    dma->isr[isr->irq].arg = isr->arg;
    if (isr->isr) {
        uart_install_isr(priv, dma->cic[isr->irq]);
    } else {
        uart_uninstall_isr(priv, dma->cic[isr->irq]);
    }
    rtems_interrupt_enable(level);

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_dma_init(struct uart_priv *priv, int minor)
{
    struct uart_dma_priv *dma;
    uint32_t db;

    dma = &priv->dma[minor];
    memset(dma, 0, sizeof(struct uart_dma_priv));

    dma->regs = (struct uart_dma_regs *)&priv->regs->dma[minor];

    if (minor == 0) {
        dma->cic[UART_ISR_ERROR]  = AGGA4_INTERRUPT_CIC_UART0_ERR;
        dma->cic[UART_ISR_TXDONE] = AGGA4_INTERRUPT_CIC_UART0_TXDONE;
        dma->cic[UART_ISR_RXDONE] = AGGA4_INTERRUPT_CIC_UART0_RXDONE;
    } else {
        dma->cic[UART_ISR_ERROR]  = AGGA4_INTERRUPT_CIC_UART1_ERR;
        dma->cic[UART_ISR_TXDONE] = AGGA4_INTERRUPT_CIC_UART1_TXDONE;
        dma->cic[UART_ISR_RXDONE] = AGGA4_INTERRUPT_CIC_UART1_RXDONE;
    }

    /* Let UART debug tunnelling be untouched if Flow-control is set.
     *
     * With old UARTs debug is enabled by setting LB and FL, since LB or
     * DB are not reset we can not trust them. However since FL is reset we
     * guess that we are debugging if FL is already set, the debugger set
     * either LB or DB depending on UART capabilities.
     */
    db = 0;
    if (dma->regs->Ctrl & UART_CTRL_FL) {
        db |= dma->regs->Ctrl & (UART_CTRL_LB | UART_CTRL_FL);
    }
    dma->regs->Ctrl = db;

    dma->regs->Status = 0;

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_dma_open(
        struct uart_priv *priv, rtems_device_minor_number minor)
{
    struct uart_dma_priv *dma;


    if (minor >= UART_NUM_MINOR) {
        return RTEMS_TOO_MANY;
    }

    dma = &priv->dma[minor];
    if (dma->open) {
        return RTEMS_RESOURCE_IN_USE;
    }
    dma->open = 1;

    dma->regs->Status = 0;

    return RTEMS_SUCCESSFUL;
}

static inline rtems_status_code uart_dma_close(
        struct uart_priv *priv, rtems_device_minor_number minor)
{
    struct uart_dma_priv *dma;
    int irq;

    if (minor >= UART_NUM_MINOR) {
        return RTEMS_TOO_MANY;
    }
    dma = &priv->dma[minor];
    dma->open = 0;

    for (irq=0; irq<UART_NUM_CIC; irq++) {
        if (dma->isr[irq].isr) {
            uart_uninstall_isr(priv, dma->cic[irq]);
        }
    }

    return RTEMS_SUCCESSFUL;
}

/*****************************************************************************
 * External interface
 */
rtems_status_code __uart_initialize(void)
{
    struct uart_priv *priv = &uart_priv;
    int i;
    static int __uart_is_init;

    if (__uart_is_init) {
        return RTEMS_SUCCESSFUL;
    }
    __uart_is_init = 1;

    priv->regs = (struct uart_regs *)&LEON_REG->UART0_Tx_SAP;

    for (i=0; i<UART_NUM_MINOR; i++) {
        uart_dma_init(priv, i);
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code __uart_open(rtems_device_minor_number minor)
{
    struct uart_priv *priv = &uart_priv;

    return uart_dma_open(priv, minor);
}

rtems_status_code __uart_close(rtems_device_minor_number minor)
{
    struct uart_priv *priv = &uart_priv;

    return uart_dma_close(priv, minor);
}
rtems_status_code __uart_ioctl(rtems_device_minor_number minor, int ctl, void *arg)
{
    struct uart_priv *priv = &uart_priv;
    rtems_status_code status;

    if (minor >= UART_NUM_MINOR) {
        return RTEMS_TOO_MANY;
    }

    switch(ctl) {
    case UART_IOCTL_START: {
        status = uart_ioctl_start(priv, minor);
        break;
    }
    case UART_IOCTL_STOP: {
        status = uart_ioctl_stop(priv, minor);
        break;
    }
    case UART_IOCTL_RX: {
        struct uart_desc *desc = (struct uart_desc *)arg;
        status = uart_ioctl_rx_arm(priv, minor, desc);
        break;
    }
    case UART_IOCTL_TX: {
        struct uart_desc *desc = (struct uart_desc *)arg;
        status = uart_ioctl_tx_arm(priv, minor, desc);
        break;
    }
    case UART_IOCTL_RX_POLL: {
        status = uart_ioctl_rx_poll(priv, minor, (int *)arg);
        break;
    }
    case UART_IOCTL_TX_POLL: {
        status = uart_ioctl_tx_poll(priv, minor);
        break;
    }
    case UART_IOCTL_GET_STATUS: {
        status = uart_ioctl_get_status(priv, minor, (uint32_t *)arg);
        break;
    }
    case UART_IOCTL_CLR_STATUS: {
        status = uart_ioctl_clr_status(priv, minor, (uint32_t)arg);
        break;
    }
    case UART_IOCTL_SET_RATE: {
        status = uart_ioctl_set_rate(priv, minor, (uint32_t)arg);
        break;
    }
    case UART_IOCTL_GET_RATE: {
        status = uart_ioctl_get_rate(priv, minor, (uint32_t*)arg);
        break;
    }
    case UART_IOCTL_SET_ENDIAN: {
        status = uart_ioctl_set_endian(priv, minor, (int)arg);
        break;
    }
    case UART_IOCTL_SET_PARITY: {
        status = uart_ioctl_set_parity(priv, minor, (int)arg);
        break;
    }
    case UART_IOCTL_GET_PARITY: {
        status = uart_ioctl_get_parity(priv, minor, (int*)arg);
        break;
    }
    case UART_IOCTL_SET_FLOW: {
        status = uart_ioctl_set_flow(priv, minor, (int)arg);
        break;
    }
    case UART_IOCTL_GET_FLOW: {
        status = uart_ioctl_get_flow(priv, minor, (int*)arg);
        break;
    }
    case UART_IOCTL_SET_ISR: {
        struct uart_isr *isr = (struct uart_isr *)arg;
        status = uart_ioctl_isr(priv, minor, isr);
        break;
    }
    default:
        status = RTEMS_NOT_DEFINED;
        break;
    }

    return status;
}

rtems_status_code __uart_rx_arm(int minor, struct uart_desc *desc)
{
    struct uart_priv *priv = &uart_priv;

    return uart_ioctl_rx_arm(priv, minor, desc);
}

rtems_status_code __uart_tx_arm(int minor, struct uart_desc *desc)
{
    struct uart_priv *priv = &uart_priv;

    return uart_ioctl_tx_arm(priv, minor, desc);
}

rtems_status_code __uart_rx_poll(int minor, int *nchar)
{
    struct uart_priv *priv = &uart_priv;

    return uart_ioctl_rx_poll(priv, minor, nchar);
}

rtems_status_code __uart_tx_poll(int minor)
{
    struct uart_priv *priv = &uart_priv;

    return uart_ioctl_tx_poll(priv, minor);
}

/*****************************************************************************
 * RTEMS I/O UART DMA
 */
rtems_status_code uart_io_initialize(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    rtems_status_code status;

    __uart_initialize();

    status = rtems_io_register_name(AGGA4_UART0_DEVNAME, major, 0);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    status = rtems_io_register_name(AGGA4_UART1_DEVNAME, major, 1);
    if (status != RTEMS_SUCCESSFUL) {
        return status;
    }

    return RTEMS_SUCCESSFUL;
}

rtems_status_code uart_io_open(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    return __uart_open(minor);
}

rtems_status_code uart_io_close(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    return __uart_close(minor);
}

rtems_status_code uart_io_control(
        rtems_device_major_number major,
        rtems_device_minor_number minor,
        void *argument)
{
    rtems_libio_ioctl_args_t *ioarg = (rtems_libio_ioctl_args_t*)argument;

    ioarg->ioctl_return = 0;

    return __uart_ioctl(minor, ioarg->command, ioarg->buffer);

}

/*****************************************************************************
 * RTEMS I/O PRINTK - Simple low-level functions for debugging
 */
rtems_status_code __uart_printk_init(rtems_device_minor_number minor)
{
    struct uart_priv *priv = &uart_priv;
    struct uart_dma_priv *dma = &priv->dma[minor];

    if (!dma->open) {
        dma->regs->Ctrl |= (UART_CTRL_RXEN | UART_CTRL_TXEN);
        return RTEMS_SUCCESSFUL;
    }
    return RTEMS_RESOURCE_IN_USE;
}

rtems_status_code __uart_printk_inb(rtems_device_minor_number minor, int *c)
{
    struct uart_priv *priv = &uart_priv;
    struct uart_dma_priv *dma = &priv->dma[minor];

    if (!dma->open) {
        volatile uint8_t __attribute__((aligned(4))) buf[4];

        dma->regs->Rx_SAP = (uint32_t)&buf[0];
        dma->regs->Rx_EAP = (uint32_t)&buf[0];
        while (dma->regs->Rx_CAP <= (uint32_t)&buf[0]) {
            __asm__ volatile ("nop;nop;nop;");
        }
        *c  = buf[0];

        return RTEMS_SUCCESSFUL;
    }
    return RTEMS_RESOURCE_IN_USE;
}

rtems_status_code __uart_printk_outb(rtems_device_minor_number minor, int c)
{
    struct uart_priv *priv = &uart_priv;
    struct uart_dma_priv *dma = &priv->dma[minor];
    int b;

    if (!dma->open) {
        volatile uint8_t __attribute__((aligned(4))) buf[4];
        b = 0;

        buf[b] = c;
        if (c == '\n') {
            buf[++b] = '\r';
        }
        dma->regs->Tx_SAP = (uint32_t)&buf[0];
        dma->regs->Tx_EAP = (uint32_t)&buf[b];
        while (dma->regs->Tx_CAP <= (uint32_t)&buf[b]) {
            __asm__ volatile ("nop;nop;nop;");
        }

        return RTEMS_SUCCESSFUL;
    }
    return RTEMS_RESOURCE_IN_USE;
}
