/*
 * A RTEMS sample application using the CRC core of the AGGA4 cpu
 *
 * The example will perform some simple self-checks to verify the behaviour of the driver.
 *
 * Copyright (C),
 * Cobham Gaisler 2016
 *
 */

#include <rtems.h>
#include <bsp.h>
#include <agga4/crc.h>

/* functions */

rtems_task Init(rtems_task_argument argument);

/* configuration information */

#define CONFIGURE_INIT
#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_MAXIMUM_SEMAPHORES        1
#define CONFIGURE_MAXIMUM_TASKS             1
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 4
#define CONFIGURE_APPLICATION_EXTRA_DRIVERS \
        AGGA4_CRC_DRIVER_TABLE_ENTRY
#include <rtems/confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdint.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>

static volatile int isr_count = 0;
void isr_func(void *arg)
{
        rtems_id *sem = arg;
        isr_count++;
        rtems_semaphore_release(*sem);
}

void print_error(int err, const char *fmt, ...)
{
        if (fmt) {
                va_list ap;
                va_start(ap, fmt);
                vfprintf(stderr, fmt, ap);
                va_end(ap);
        }
        fprintf(stderr, "CRC FAIL! %d - %s\n", err, strerror(err));

}

rtems_task Init(rtems_task_argument argument)
{
        rtems_status_code status;
        int fd = -1;
        uint8_t __attribute__((aligned(4))) buf[256];
        struct crc_desc desc;
        struct crc_isr isr;
        uint32_t crc;
        uint32_t exp;
        rtems_id sem;

        desc.addr = &buf[0];
        desc.bytes = 256;

        fd = open(AGGA4_CRC_DEVNAME, O_RDWR, 0);
        if (fd < 0) {
                print_error(errno, "Failed to open "AGGA4_CRC_DEVNAME": ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_SET_POLYNOM, CRC_CCITT_POLY);
        if (status < 0) {
                print_error(errno, "CFG CRC_IOCTL_SET_POLYNOM: ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_SET_FINAL_XOR, CRC_CCITT_FXOR);
        if (status < 0) {
                print_error(errno, "CFG CRC_IOCTL_SET_FINAL_XOR: ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_SET_CTRL, CRC_CCITT_CTRL);
        if (status < 0) {
                fprintf(stderr, "CFG CRC_IOCTL_SET_CTRL: ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_SET_LFSR, CRC_CCITT_LFSR);
        if (status < 0) {
                print_error(errno, "E1 CRC_IOCTL_SET_LFSR: ");
                goto end;
        }

        /* Example 1
         * =========
         * Use polling to wait
         */
        memset(buf, 'A', 256);
        exp = 0xEA0B;

        status = ioctl(fd, CRC_IOCTL_START, desc);
        if (status < 0) {
                print_error(errno, "E1 CRC_IOCTL_START: ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_POLL, NULL);
        if (status < 0) {
                print_error(errno, "E1 CRC_IOCTL_POLL: ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_GET_CRC, &crc);
        if (status < 0) {
                print_error(errno, "E1 CRC_IOCTL_GET_CRC: ");
                goto end;
        }

        if (!CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_CRC)) {
                fprintf(stderr, "E1: CIC CRC should be set\n");
                goto end;
        }

        if (crc != exp) {
                fprintf(stderr, "E1: CRC != EXP (0x%04x != 0x%04x)\n",
                                (unsigned int)crc, (unsigned int)exp);
                goto end;
        }

        /* Example 2
         * =========
         * Use ISR to wait
         */
        status = rtems_semaphore_create(
                rtems_build_name('S', 'C', 'R', 'C'),
                0,
                RTEMS_FIFO | RTEMS_SIMPLE_BINARY_SEMAPHORE | \
                RTEMS_NO_INHERIT_PRIORITY | RTEMS_NO_PRIORITY_CEILING,
                0,
                &sem);
        if (status != RTEMS_SUCCESSFUL) {
                print_error(errno, "E2 RTEMS_SEMAPORE_CREATE: ");
                goto end;
        }

        memset(buf, 'B', 256);
        exp = 0x8A76;

        isr.isr = isr_func;
        isr.arg = &sem;
        status = ioctl(fd, CRC_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E2 CRC_IOCTL_SET_ISR: ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_START, desc);
        if (status < 0) {
                print_error(errno, "E2 CRC_IOCTL_START: ");
                goto end;
        }

        status = rtems_semaphore_obtain(sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
        if (status != RTEMS_SUCCESSFUL) {
                print_error(errno, "E2 RTEMS_SEMAPORE_OBTAIN: ");
                goto end;
        }

        status = ioctl(fd, CRC_IOCTL_GET_CRC, &crc);
        if (status < 0) {
                print_error(errno, "E2 CRC_IOCTL_GET_CRC: ");
                goto end;
        }

        if (CIC_Is_interrupt_pending(AGGA4_INTERRUPT_CIC_CRC)) {
                fprintf(stderr, "E2: CIC CRC should be cleared\n");
                goto end;
        }

        if (crc != exp) {
                fprintf(stderr, "E2: CRC != EXP (0x%04x != 0x%04x)\n",
                                (unsigned int)crc, (unsigned int)exp);
                goto end;
        }
        isr.isr = NULL;
        status = ioctl(fd, CRC_IOCTL_SET_ISR, &isr);
        rtems_semaphore_delete(sem);

        printf("CRC: PASS\n");
end:
        if (fd > 0) {
                close(fd);
        }

        exit(0);
}

