/*
 * A RTEMS sample application using the GNSS core of the AGGA4 cpu
 *
 * Copyright (C),
 * Cobham Gaisler 2016
 *
 */

#include <rtems.h>
#include <bsp.h>
#include <agga4/gnss.h>

/* functions */

rtems_task Init(rtems_task_argument argument);

/* configuration information */

#define CONFIGURE_INIT
#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_MAXIMUM_SEMAPHORES        1
#define CONFIGURE_MAXIMUM_TASKS             1
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 4
#define CONFIGURE_APPLICATION_EXTRA_DRIVERS \
        AGGA4_GNSS_DRIVER_TABLE_ENTRY
#include <rtems/confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>

void isr_func(void *arg)
{
}

void print_error(int err, const char *fmt, ...)
{
        if (fmt) {
                va_list ap;
                va_start(ap, fmt);
                vfprintf(stderr, fmt, ap);
                va_end(ap);
        }
        fprintf(stderr, "GNSS FAIL! %d - %s\n", err, strerror(err));

}

rtems_task Init(rtems_task_argument argument)
{
        rtems_status_code status;
        int fd = -1;
        struct gnss_regs *regs = NULL;
        int i;
        struct gnss_isr isr;
        uint32_t u32[2];

        fd = open(AGGA4_GNSS_DEVNAME, O_RDWR, 0);
        if (fd < 0) {
                print_error(errno, "Failed to open "AGGA4_GNSS_DEVNAME": ");
                goto end;
        }

        /* Example 1
         * =========
         */
        status = ioctl(fd, GNSS_IOCTL_GET_REGS, &regs);
        if (status < 0) {
                print_error(errno, "E1 GNSS_IOCTL_GET_REGS: ");
                goto end;
        }

        if (((uint32_t)regs) != 0xA0000000) {
                fprintf(stderr, "E1 Wrong address\n");
                goto end;
        }


        u32[0] = 0;
        u32[1] = 0;
        isr.isr = isr_func;
        isr.arg = NULL;
        for (i=0; i<32; i++) {
                isr.irq = i;
                u32[0] |= 1<<isr.irq;
                status = ioctl(fd, GNSS_IOCTL_SET_ISR, &isr);
                if (status < 0) {
                        print_error(errno, "E1 GNSS_IOCTL_SET_ISR: ");
                        goto end;
                }
                /* Self-check */
                if (memcmp(&u32[0], (void*)&regs->gic.mask[0], 4)) {
                        fprintf(stderr, "E1 COMPARE: %d not enabled %08x %08x\n",
                                        i, (unsigned int)u32[0], (unsigned int)regs->gic.mask[0]);
                        goto end;
                }
        }
        for (i=32; i<35; i++) {
                isr.irq = i;
                u32[1] |= 1<<(isr.irq-32);
                status = ioctl(fd, GNSS_IOCTL_SET_ISR, &isr);
                if (status < 0) {
                        print_error(errno, "E1 GNSS_IOCTL_SET_ISR: ");
                        goto end;
                }
                /* Self-check */
                if (memcmp(&u32[1], (void*)&regs->gic.mask[1], 4)) {
                        fprintf(stderr, "E1 COMPARE: %d not enabled %08x %08x\n",
                                        i, (unsigned int)u32[1],(unsigned int)regs->gic.mask[1]);
                        goto end;
                }
        }

        isr.isr = NULL;
        for (i=0; i<32; i++) {
                isr.irq = i;
                u32[0]  &= ~(1<<isr.irq);
                status = ioctl(fd, GNSS_IOCTL_SET_ISR, &isr);
                if (status < 0) {
                        print_error(errno, "E1 GNSS_IOCTL_SET_ISR: ");
                        goto end;
                }
                /* Self-check */
                if (memcmp(&u32[0], (void*)&regs->gic.mask[0], 4)) {
                        fprintf(stderr, "E1 COMPARE: %d not disabled %08x %08x\n",
                                        i, (unsigned int)u32[0], (unsigned int)regs->gic.mask[0]);
                        goto end;
                }
        }
        for (i=32; i<35; i++) {
                isr.irq = i;
                u32[1]  &= ~(1<<(isr.irq-32));
                status = ioctl(fd, GNSS_IOCTL_SET_ISR, &isr);
                if (status < 0) {
                        print_error(errno, "E1 GNSS_IOCTL_SET_ISR: ");
                        goto end;
                }
                /* Self-check */
                if (memcmp(&u32[1], (void*)&regs->gic.mask[1], 4)) {
                        fprintf(stderr, "E1 COMPARE: %d not disabled %08x %08x\n",
                                        i, (unsigned int)u32[1], (unsigned int)regs->gic.mask[1]);
                        goto end;
                }
        }

        printf("GNSS: PASS\n");
end:
        if (fd > 0) {
                close(fd);
        }

        exit(0);
}

