/*  leon.h
 *
 *  This include file contains information pertaining to the AGGA4.
 *  The AGGA4 is a LEON2 SPARC V8 implementation.
 *
 *  NOTE:  Other than where absolutely required, this version currently
 *         supports only the peripherals and bits used by the basic board
 *         support package. This includes at least significant pieces of
 *         the following items:
 *
 *           + UART Channels A and B
 *           + Real Time Clock
 *           + Memory Control Register
 *           + Interrupt Control
 *
 *  COPYRIGHT (c) 1989-1998.
 *  On-Line Applications Research Corporation (OAR).
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 *
 *  Ported to LEON implementation of the SPARC by On-Line Applications
 *  Research Corporation (OAR) under contract to the European Space
 *  Agency (ESA).
 *
 *  LEON modifications of respective RTEMS file: COPYRIGHT (c) 1995.
 *  European Space Agency.
 *
 *  $Id$
 */

#ifndef _INCLUDE_LEON_h
#define _INCLUDE_LEON_h

#include <rtems/score/sparc.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 *  Interrupt Sources
 *
 *  The interrupt source numbers directly map to the trap type and to
 *  the bits used in the Interrupt Clear, Interrupt Force, Interrupt Mask,
 *  and the Interrupt Pending Registers.
 */

#define LEON_INTERRUPT_AHB_ERROR         1
#define AGGA4_INTERRUPT_PIO0             2
#define LEON_INTERRUPT_TIMER1            3
#define AGGA4_INTERRUPT_FFT_DONE         4
#define AGGA4_INTERRUPT_GICL             5  // Shared (GNSS low)
#define AGGA4_INTERRUPT_PIO1             6
#define AGGA4_INTERRUPT_TIMER2           7
#define AGGA4_INTERRUPT_CIC              8  // Shared (MilBus, Spw, UART, ...)
#define AGGA4_INTERRUPT_PIO2             9
#define AGGA4_INTERRUPT_TIMER3           10
#define AGGA4_INTERRUPT_GICH             11 // Shared (GNSS high)
#define AGGA4_INTERRUPT_PIO3             12
#define AGGA4_INTERRUPT_TIMER4           13
#define AGGA4_INTERRUPT_DSU_TB           14
#define AGGA4_INTERRUPT_NMI              15

#define AGGA4_INTERRUPT_CIC_SGPO_OV      0
#define AGGA4_INTERRUPT_CIC_CRC          1
#define AGGA4_INTERRUPT_CIC_SPI          2
#define AGGA4_INTERRUPT_CIC_UART0_RXDONE 3
#define AGGA4_INTERRUPT_CIC_UART0_TXDONE 4
#define AGGA4_INTERRUPT_CIC_UART0_ERR    5
#define AGGA4_INTERRUPT_CIC_UART1_RXDONE 6
#define AGGA4_INTERRUPT_CIC_UART1_TXDONE 7
#define AGGA4_INTERRUPT_CIC_UART1_ERR    8
#define AGGA4_INTERRUPT_CIC_SPW          9
#define AGGA4_INTERRUPT_CIC_MILBUS_ERR   10
#define AGGA4_INTERRUPT_CIC_MILBUS_RST   11
#define AGGA4_INTERRUPT_CIC_MILBUS_NOM   12

#ifndef ASM

/*
 *  Trap Types for on-chip peripherals
 *
 *  Source: Table 8 - Interrupt Trap Type and Default Priority Assignments
 *
 *  NOTE: The priority level for each source corresponds to the least
 *        significant nibble of the trap type.
 */

#define LEON_TRAP_TYPE( _source ) SPARC_ASYNCHRONOUS_TRAP((_source) + 0x10)

#define LEON_TRAP_SOURCE( _trap ) ((_trap) - 0x10)

#define LEON_INT_TRAP( _trap ) \
  ( (_trap) >= LEON_TRAP_TYPE( 1 ) && \
    (_trap) <= LEON_TRAP_TYPE( 15 ) )

/*
 *  Structure for LEON memory mapped registers.
 *
 *  Source: Section 6.1 - On-chip registers
 *
 *  NOTE:  There is only one of these structures per CPU, its base address
 *         is 0x80000000, and the variable LEON_REG is placed there by the
 *         linkcmds file.
 */

typedef struct {
	volatile uint32_t Memory_Config_1;
	volatile uint32_t Memory_Config_2;
	volatile uint32_t Edac_Control;
	volatile uint32_t Failed_Address;
	volatile uint32_t Memory_Status;
	volatile uint32_t Cache_Control;
	volatile uint32_t Power_Down;
	volatile uint32_t Write_Protection_1;
	volatile uint32_t Write_Protection_2;
	volatile uint32_t Leon_Configuration;
	volatile uint32_t _dummy1;
	volatile uint32_t _dummy2;
	volatile uint32_t _dummy3;
	volatile uint32_t _dummy4;
	volatile uint32_t _dummy5;
	volatile uint32_t _dummy6;
	volatile uint32_t Timer_Counter_1;
	volatile uint32_t Timer_Reload_1;
	volatile uint32_t Timer_Control_1;
	volatile uint32_t _dummy7;
	volatile uint32_t Timer_Counter_2;
	volatile uint32_t Timer_Reload_2;
	volatile uint32_t Timer_Control_2;
	volatile uint32_t _dummy8;
	volatile uint32_t Scaler_Counter;
	volatile uint32_t Scaler_Reload;
	volatile uint32_t _dummy9;
	volatile uint32_t _dummy10;
	volatile uint32_t Timer_Counter_3;
	volatile uint32_t Timer_Reload_3;
	volatile uint32_t Timer_Control_3;
	volatile uint32_t _dummy11;
	volatile uint32_t Timer_Counter_4;
	volatile uint32_t Timer_Reload_4;
	volatile uint32_t Timer_Control_5;
	volatile uint32_t _dummy12;
	volatile uint32_t Interrupt_Mask;
	volatile uint32_t Interrupt_Pending;
	volatile uint32_t Interrupt_Force;
	volatile uint32_t Interrupt_Clear;
	volatile uint32_t PIO_Data;
	volatile uint32_t PIO_Direction;
	volatile uint32_t PIO_Interrupt;
	volatile uint32_t _dummy13;
	volatile uint32_t Watchdog_Prescale;
	volatile uint32_t Watchdog_Reload;
	volatile uint32_t Watchdog_Select;
	volatile uint32_t Watchdog_Write_Enable;
	volatile uint32_t _dummy14;
	volatile uint32_t DUART_Status;
	volatile uint32_t DUART_Ctrl;
	volatile uint32_t DUART_Scalter;
	volatile uint32_t Write_Protection_3_SAP;
	volatile uint32_t Write_Protection_3_EAP;
	volatile uint32_t Write_Protection_4_SAP;
	volatile uint32_t Write_Protection_4_EAP;
	volatile uint32_t GPIO_Status;
	volatile uint32_t GPIO_Output;
	volatile uint32_t GPIO_Direction;
	volatile uint32_t Reset_Status;
	volatile uint32_t SW_Reset_Enable;
	volatile uint32_t SW_Reset_Execute;
	volatile uint32_t MilBus_RT_Address;
	volatile uint32_t Version;
	volatile uint32_t CIC_Mask;
	volatile uint32_t CIC_Pending;
	volatile uint32_t CIC_Clear;
	volatile uint32_t _dummy15;
	volatile uint32_t SPI_Status_and_Ctrl;
	volatile uint32_t SPI_CLK_Divider;
	volatile uint32_t SPI_Tx;
	volatile uint32_t SPI_Rx;
	volatile uint32_t CRC_LFSR;
	volatile uint32_t CRC_Polynom;
	volatile uint32_t CRC_Final_XOR;
	volatile uint32_t CRC_Ctrl;
	volatile uint32_t CRC_SAP;
	volatile uint32_t CRC_EAP;
	volatile uint32_t CRC_CAP;
	volatile uint32_t _dummy16;
	volatile uint32_t C53_CF;
	volatile uint32_t C53_EMBA;
	volatile uint32_t C53_CDST;
	volatile uint32_t C53_NIT;
	volatile uint32_t C53_EIT;
	volatile uint32_t C53_RIT;
	volatile uint32_t _dummy17;
	volatile uint32_t C53_RTI;
	volatile uint32_t C53_TTI;
	volatile uint32_t _dummy18;
	volatile uint32_t _dummy19;
	volatile uint32_t _dummy20;
	volatile uint32_t SGPO_Tx;
	volatile uint32_t SGPO_Status;
	volatile uint32_t SGPO_Ctrl;
	volatile uint32_t SGPO_Scaler;
	volatile uint32_t UART0_Tx_SAP;
	volatile uint32_t UART0_Tx_EAP;
	volatile uint32_t UART0_Tx_CAP;
	volatile uint32_t UART0_Rx_SAP;
	volatile uint32_t UART0_Rx_EAP;
	volatile uint32_t UART0_Rx_CAP;
	volatile uint32_t UART0_Status;
	volatile uint32_t UART0_Ctrl;
	volatile uint32_t UART0_Scaler;
	volatile uint32_t UART1_Tx_SAP;
	volatile uint32_t UART1_Tx_EAP;
	volatile uint32_t UART1_Tx_CAP;
	volatile uint32_t UART1_Rx_SAP;
	volatile uint32_t UART1_Rx_EAP;
	volatile uint32_t UART1_Rx_CAP;
	volatile uint32_t UART1_Status;
	volatile uint32_t UART1_Ctrl;
	volatile uint32_t UART1_Scaler;
	volatile uint32_t UART_Reset;
	volatile uint32_t _dummy21;
	volatile uint32_t _dummy22;
	volatile uint32_t _dummy23;
	volatile uint32_t _dummy24;
	volatile uint32_t _dummy25;
	volatile uint32_t GNSS_Core_Clk_Ctrl;
	volatile uint32_t _dummy26;
	volatile uint32_t MilBus_Clk_Ctrl;
	volatile uint32_t PLL_Status;
	volatile uint32_t _dummy27;
	volatile uint32_t _dummy28;
	volatile uint32_t EEPROM_Status_and_Ctrl;
	volatile uint32_t _dummy29;
	volatile uint32_t SpW0_Status_and_Ctrl;
	volatile uint32_t _dummy30;
	volatile uint32_t SpW0_Tx_SAP;
	volatile uint32_t SpW0_Tx_EAP;
	volatile uint32_t SpW0_Tx_CAP;
	volatile uint32_t SpW0_Tx_Rx_Config;
	volatile uint32_t SpW0_Rx_SAP;
	volatile uint32_t SpW0_Rx_EAP;
	volatile uint32_t SpW0_Rx_CAP;
	volatile uint32_t _dummy31;
	volatile uint32_t _dummy32;
	volatile uint32_t _dummy33;
	volatile uint32_t _dummy34;
	volatile uint32_t _dummy35;
	volatile uint32_t _dummy36;
	volatile uint32_t _dummy37;
	volatile uint32_t SpW1_Status_and_Ctrl;
	volatile uint32_t _dummy38;
	volatile uint32_t SpW1_Tx_SAP;
	volatile uint32_t SpW1_Tx_EAP;
	volatile uint32_t SpW1_Tx_CAP;
	volatile uint32_t SpW1_Tx_Rx_Config;
	volatile uint32_t SpW1_Rx_SAP;
	volatile uint32_t SpW1_Rx_EAP;
	volatile uint32_t SpW1_Rx_CAP;
	volatile uint32_t _dummy39;
	volatile uint32_t _dummy40;
	volatile uint32_t _dummy41;
	volatile uint32_t _dummy42;
	volatile uint32_t _dummy43;
	volatile uint32_t _dummy44;
	volatile uint32_t _dummy45;
	volatile uint32_t SpW2_Status_and_Ctrl;
	volatile uint32_t _dummy46;
	volatile uint32_t SpW2_Tx_SAP;
	volatile uint32_t SpW2_Tx_EAP;
	volatile uint32_t SpW2_Tx_CAP;
	volatile uint32_t SpW2_Tx_Rx_Config;
	volatile uint32_t SpW2_Rx_SAP;
	volatile uint32_t SpW2_Rx_EAP;
	volatile uint32_t SpW2_Rx_CAP;
	volatile uint32_t _dummy47;
	volatile uint32_t _dummy48;
	volatile uint32_t _dummy49;
	volatile uint32_t _dummy50;
	volatile uint32_t _dummy51;
	volatile uint32_t _dummy52;
	volatile uint32_t _dummy53;
	volatile uint32_t SpW_Module_Config;
	volatile uint32_t SpW_Module_Time_Ctrl;
	volatile uint32_t SpW_Module_Time_Code;
	volatile uint32_t SpW_Module_Int_Mask;
	volatile uint32_t SpW_Module_Int_Status;
	volatile uint32_t SpW_Module_Int_Clear;
} LEON_Register_Map;

#endif

/*
 *  The following constants are intended to be used ONLY in assembly
 *  language files.
 *
 *  NOTE:  The intended style of usage is to load the address of LEON REGS
 *         into a register and then use these as displacements from
 *         that register.
 */

#ifdef ASM

#define  LEON_REG_MEMCFG1_OFFSET                                  0x00
#define  LEON_REG_MEMCFG2_OFFSET                                  0x04
#define  LEON_REG_EDACCTRL_OFFSET                                 0x08
#define  LEON_REG_FAILADDR_OFFSET                                 0x0C
#define  LEON_REG_MEMSTATUS_OFFSET                                0x10
#define  LEON_REG_CACHECTRL_OFFSET                                0x14
#define  LEON_REG_POWERDOWN_OFFSET                                0x18
#define  LEON_REG_WRITEPROT1_OFFSET                               0x1C
#define  LEON_REG_WRITEPROT2_OFFSET                               0x20
#define  LEON_REG_LEONCONF_OFFSET                                 0x24
#define  LEON_REG_UNIMPLEMENTED_2_OFFSET                          0x28
#define  LEON_REG_UNIMPLEMENTED_3_OFFSET                          0x2C
#define  LEON_REG_UNIMPLEMENTED_4_OFFSET                          0x30
#define  LEON_REG_UNIMPLEMENTED_5_OFFSET                          0x34
#define  LEON_REG_UNIMPLEMENTED_6_OFFSET                          0x38
#define  LEON_REG_UNIMPLEMENTED_7_OFFSET                          0x3C
#define  LEON_REG_TIMERCNT1_OFFSET                                0x40
#define  LEON_REG_TIMERLOAD1_OFFSET                               0x44
#define  LEON_REG_TIMERCTRL1_OFFSET                               0x48
#define  LEON_REG_WDOG_OFFSET                                     0x4C
#define  LEON_REG_TIMERCNT2_OFFSET                                0x50
#define  LEON_REG_TIMERLOAD2_OFFSET                               0x54
#define  LEON_REG_TIMERCTRL2_OFFSET                               0x58
#define  LEON_REG_UNIMPLEMENTED_8_OFFSET                          0x5C
#define  LEON_REG_SCALERCNT_OFFSET                                0x60
#define  LEON_REG_SCALER_LOAD_OFFSET                              0x64
#define  LEON_REG_UNIMPLEMENTED_9_OFFSET                          0x68
#define  LEON_REG_UNIMPLEMENTED_10_OFFSET                         0x6C
#define  LEON_REG_TIMERCNT3_OFFSET                                0x70
#define  LEON_REG_TIMERLOAD3_OFFSET                               0x74
#define  LEON_REG_TIMERCTRL3_OFFSET                               0x78
#define  LEON_REG_UNIMPLEMENTED_11_OFFSET                         0x7C
#define  LEON_REG_TIMERCNT4_OFFSET                                0x80
#define  LEON_REG_TIMERLOAD4_OFFSET                               0x84
#define  LEON_REG_TIMERCTRL4_OFFSET                               0x88
#define  LEON_REG_UNIMPLEMENTED_12_OFFSET                         0x8C
#define  LEON_REG_IRQMASK_OFFSET                                  0x90
#define  LEON_REG_IRQPEND_OFFSET                                  0x94
#define  LEON_REG_IRQFORCE_OFFSET                                 0x98
#define  LEON_REG_IRQCLEAR_OFFSET                                 0x9C
#define  LEON_REG_PIODATA_OFFSET                                  0xA0
#define  LEON_REG_PIODIR_OFFSET                                   0xA4
#define  LEON_REG_PIOIRQ_OFFSET                                   0xA8
#define  LEON_REG_UNIMPLEMENTED_13_OFFSET                         0xAC
#define  LEON_REG_WDOG_PRESCLE_OFFSET                             0xB0
#define  LEON_REG_WDOG_RELOAD_OFFSET                              0xB4
#define  LEON_REG_WDOG_SELECT_OFFSET                              0xB8
#define  LEON_REG_WDOG_WRITE_ENABLE_OFFSET                        0xBC

#define  LEON_REG_SIM_RAM_SIZE_OFFSET                             0xF4
#define  LEON_REG_SIM_ROM_SIZE_OFFSET                             0xF8

#endif

/*
 *  The following defines the bits in Memory Configuration Register 1.
 */

#define LEON_MEMORY_CONFIGURATION_PROM_SIZE_MASK  0x0003C000

/*
 *  The following defines the bits in Memory Configuration Register 1.
 */

#define LEON_MEMORY_CONFIGURATION_RAM_SIZE_MASK  0x00001E00


/*
 *  The following defines the bits in the Timer Control Register.
 */

#define LEON_REG_TIMER_CONTROL_EN    0x00000001  /* 1 = enable counting */
                                                 /* 0 = hold scalar and counter */
#define LEON_REG_TIMER_CONTROL_RL    0x00000002  /* 1 = reload at 0 */
                                                 /* 0 = stop at 0 */
#define LEON_REG_TIMER_CONTROL_LD    0x00000004  /* 1 = load counter */
                                                 /* 0 = no function */
#ifndef ASM

/*
 *  This is used to manipulate the on-chip registers.
 *
 *  The following symbol must be defined in the linkcmds file and point
 *  to the correct location.
 */

/* extern LEON_Register_Map LEON_REG; */
static LEON_Register_Map * const LEON_REG = (LEON_Register_Map *)0x80000000;

static __inline__ int leon_irq_fixup(int irq)
{
	return irq;
}

/*
 *  Macros to manipulate the Interrupt Clear, Interrupt Force, Interrupt Mask,
 *  and the Interrupt Pending Registers.
 *
 *  NOTE: For operations which are not atomic, this code disables interrupts
 *        to guarantee there are no intervening accesses to the same register.
 *        The operations which read the register, modify the value and then
 *        store the result back are vulnerable.
 */

#define LEON_Clear_interrupt( _source ) \
  do { \
    LEON_REG->Interrupt_Clear = (1 << (_source)); \
  } while (0)

#define LEON_Force_interrupt( _source ) \
  do { \
    LEON_REG->Interrupt_Force = (1 << (_source)); \
  } while (0)

#define LEON_Is_interrupt_pending( _source ) \
  (LEON_REG->Interrupt_Pending & (1 << (_source)))

#define LEON_Is_interrupt_masked( _source ) \
  (LEON_REG->Interrupt_Masked & (1 << (_source)))

#define LEON_Mask_interrupt( _source ) \
  do { \
    uint32_t _level; \
    \
    _level = sparc_disable_interrupts(); \
      LEON_REG->Interrupt_Mask &= ~(1 << (_source)); \
    sparc_enable_interrupts( _level ); \
  } while (0)

#define LEON_Unmask_interrupt( _source ) \
  do { \
    uint32_t _level; \
    \
    _level = sparc_disable_interrupts(); \
      LEON_REG->Interrupt_Mask |= (1 << (_source)); \
    sparc_enable_interrupts( _level ); \
  } while (0)

#define LEON_Disable_interrupt( _source, _previous ) \
  do { \
    uint32_t _level; \
    uint32_t _mask = 1 << (_source); \
    \
    _level = sparc_disable_interrupts(); \
      (_previous) = LEON_REG->Interrupt_Mask; \
      LEON_REG->Interrupt_Mask = _previous & ~_mask; \
    sparc_enable_interrupts( _level ); \
    (_previous) &= _mask; \
  } while (0)

#define LEON_Restore_interrupt( _source, _previous ) \
  do { \
    uint32_t _level; \
    uint32_t _mask = 1 << (_source); \
    \
    _level = sparc_disable_interrupts(); \
      LEON_REG->Interrupt_Mask = \
        (LEON_REG->Interrupt_Mask & ~_mask) | (_previous); \
    sparc_enable_interrupts( _level ); \
  } while (0)

/* CIC
 * 	volatile uint32_t CIC_Mask;
 *	volatile uint32_t CIC_Pending;
 *	volatile uint32_t CIC_Clear;
 */
#define CIC_Clear_interrupt( _source ) \
  do { \
    LEON_REG->CIC_Clear = (1 << (_source)); \
  } while (0)

#define CIC_Is_interrupt_pending( _source ) \
  (LEON_REG->CIC_Pending & (1 << (_source)))

#define CIC_Is_interrupt_enabled( _source ) \
  (LEON_REG->CIC_Mask & (1 << (_source)))

#define CIC_Mask_interrupt( _source ) \
  do { \
    uint32_t _level; \
    \
    _level = sparc_disable_interrupts(); \
      LEON_REG->CIC_Mask &= ~(1 << (_source)); \
    sparc_enable_interrupts( _level ); \
  } while (0)

#define CIC_Unmask_interrupt( _source ) \
  do { \
    uint32_t _level; \
    \
    _level = sparc_disable_interrupts(); \
      LEON_REG->CIC_Mask |= (1 << (_source)); \
    sparc_enable_interrupts( _level ); \
  } while (0)

#define CIC_Disable_interrupt( _source, _previous ) \
  do { \
    uint32_t _level; \
    uint32_t _mask = 1 << (_source); \
    \
    _level = sparc_disable_interrupts(); \
      (_previous) = LEON_REG->CIC_Mask; \
      LEON_REG->CIC_Mask = _previous & ~_mask; \
    sparc_enable_interrupts( _level ); \
    (_previous) &= _mask; \
  } while (0)

#define CIC_Restore_interrupt( _source, _previous ) \
  do { \
    uint32_t _level; \
    uint32_t _mask = 1 << (_source); \
    \
    _level = sparc_disable_interrupts(); \
      LEON_REG->CIC_Mask = \
        (LEON_REG->CIC_Mask & ~_mask) | (_previous); \
    sparc_enable_interrupts( _level ); \
  } while (0)


/*
 *  Each timer control register is organized as follows:
 *
 *    D0 - Enable
 *          1 = enable counting
 *          0 = hold scaler and counter
 *
 *    D1 - Counter Reload
 *          1 = reload counter at zero and restart
 *          0 = stop counter at zero
 *
 *    D2 - Counter Load
 *          1 = load counter with preset value
 *          0 = no function
 *
 */

#define LEON_REG_TIMER_COUNTER_RELOAD_AT_ZERO     0x00000002
#define LEON_REG_TIMER_COUNTER_STOP_AT_ZERO       0x00000000

#define LEON_REG_TIMER_COUNTER_LOAD_COUNTER       0x00000004

#define LEON_REG_TIMER_COUNTER_ENABLE_COUNTING    0x00000001
#define LEON_REG_TIMER_COUNTER_DISABLE_COUNTING   0x00000000

#define LEON_REG_TIMER_COUNTER_RELOAD_MASK        0x00000002
#define LEON_REG_TIMER_COUNTER_ENABLE_MASK        0x00000001

#define LEON_REG_TIMER_COUNTER_DEFINED_MASK       0x00000003
#define LEON_REG_TIMER_COUNTER_CURRENT_MODE_MASK  0x00000003

static inline uint32_t agga4_freq_get()
{
	return (LEON_REG->Scaler_Reload+1) * 1000 * 1000;
}

/* Load 32-bit word by forcing a cache-miss */
static inline uint32_t leon_r32_no_cache(uint32_t addr)
{
	uint32_t tmp;
	asm volatile (" lda [%1] 1, %0\n" : "=r"(tmp) : "r"(addr) );
	return tmp;
}

#endif /* !ASM */

#ifdef __cplusplus
}
#endif

#endif /* !_INCLUDE_LEON_h */

