/*
 * A RTEMS sample application using the FFT core of the AGGA4 cpu
 *
 * The example will perform some simple self-checks to verify the behaviour of the driver.
 *
 * Copyright (C),
 * Cobham Gaisler 2016
 *
 */

#include <rtems.h>
#include <bsp.h>
#include <agga4/fft.h>

/* functions */

rtems_task Init(rtems_task_argument argument);

/* configuration information */

#define CONFIGURE_INIT
#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_MAXIMUM_SEMAPHORES        1
#define CONFIGURE_MAXIMUM_TASKS             1
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 4
#define CONFIGURE_APPLICATION_EXTRA_DRIVERS \
        AGGA4_FFT_DRIVER_TABLE_ENTRY
#include <rtems/confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>

static volatile int isr_count = 0;
void isr_func(void *arg)
{
        isr_count++;
}

void print_error(int err, const char *fmt, ...)
{
        if (fmt) {
                va_list ap;
                va_start(ap, fmt);
                vfprintf(stderr, fmt, ap);
                va_end(ap);
        }
        fprintf(stderr, "FFT FAIL! %d - %s\n", err, strerror(err));

}

uint32_t in[256];
uint32_t out[256];
uint32_t zero[256];

rtems_task Init(rtems_task_argument argument)
{
        rtems_status_code status;
        int fd = -1;
        int i;
        struct fft_value val;
        struct fft_isr isr;
        const uint32_t *fft = (void*)0xB0000000;

        for (i=0; i<sizeof(in)/sizeof(uint32_t); i+=2) {
                in[0]=i*i;
                in[1]=i/2;
        }
        memset(zero, 0, sizeof(zero));

        fd = open(AGGA4_FFT_DEVNAME, O_RDWR, 0);
        if (fd < 0) {
                print_error(errno, "Failed to open "AGGA4_FFT_DEVNAME": ");
                goto end;
        }

        /* Example 1
         * =========
         * Use read/write interface and blocking I/O to wait,
         * driver will use semaphores and interrupts.
         */
        memset(out, 0, sizeof(out));

        status = ioctl(fd, FFT_IOCTL_BLOCKING, FFT_BLOCKING);
        if (status < 0) {
                print_error(errno, "E1 FFT_IOCTL_BLOCKING: ");
                goto end;
        }

        status = write(fd, in, sizeof(in));
        if (status != sizeof(in)) {
                print_error(errno, "E1 write: ");
                goto end;
        }

        status = read(fd, out, sizeof(out));
        if (status != sizeof(out)) {
                print_error(errno, "E1 read: ");
                goto end;
        }


        if (memcmp(&zero[0], &out[0], 1024) == 0) {
                print_error(errno, "E1 Output not read (out == zero): ");
                goto end;
        }
        if (memcmp(&in[0], &out[0], 1024) == 0) {
                print_error(errno, "E1 Output was not calculated (out == in): ");
                goto end;
        }
        if (memcmp(&out[0], fft, 1024)) {
                print_error(errno, "E1 Output differs (out != fft): ");
                goto end;
        }

        if (memcmp(&in[0], &out[0], 1024) == 0) {
                print_error(errno, "E1 In == Out: ");
                goto end;
        }
        if (memcmp(&zero[0], &out[0], 1024) == 0) {
                print_error(errno, "E1 Zero == Out: ");
                goto end;
        }

        /* Example 2
         * =========
         * Use register interface, use custome ISR
         */
        memset(out, 0, sizeof(out));

        status = ioctl(fd, FFT_IOCTL_BLOCKING, FFT_NONBLOCKING);
        if (status < 0) {
                print_error(errno, "E2 FFT_IOCTL_BLOCKING: ");
                goto end;
        }

        isr.isr = isr_func;
        isr.arg = NULL;
        status = ioctl(fd, FFT_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E2 FFT_IOCTL_BLOCKING: ");
                goto end;
        }

        for (i=0; i<sizeof(in)/(2*sizeof(uint32_t)); i++) {
                val.index = i;
                val.real = in[2*i+0];
                val.imag = in[2*i+1];

                status = ioctl(fd, FFT_IOCTL_SET_VALUE, &val);
                if (status < 0) {
                        print_error(errno, "E2 FFT_IOCTL_SET_VALUE %d: ", i);
                        goto end;
                }
        }

        if (memcmp(&in[0], fft, 1024)) {
                print_error(errno, "E2 Input differs (out != fft): ");
                goto end;
        }

        status = ioctl(fd, FFT_IOCTL_SET_CTRL, FFT_CTRL_START);
        if (status < 0) {
                print_error(errno, "E2 FFT_IOCTL_SET_CTRL: ");
                goto end;
        }

        while (!isr_count) {
                ;
        }

        for (i=0; i<sizeof(out)/(2*sizeof(uint32_t)); i++) {
                val.index = i;
                status = ioctl(fd, FFT_IOCTL_GET_VALUE, &val);
                if (status < 0) {
                        print_error(errno, "E2 FFT_IOCTL_SET_VALUE %d: ", i);
                        goto end;
                }
                out[2*i+0] = val.real;
                out[2*i+1] = val.imag;
        }

        if (memcmp(&zero[0], &out[0], 1024) == 0) {
                print_error(errno, "E2 Output not read (out == zero): ");
                goto end;
        }
        if (memcmp(&in[0], &out[0], 1024) == 0) {
                print_error(errno, "E2 Output was not calculated (out == in): ");
                goto end;
        }
        if (memcmp(&out[0], fft, 1024)) {
                print_error(errno, "E2 Output differs (out != fft): ");
                goto end;
        }

        printf("FFT: PASS\n");

end:
        if (fd > 0) {
                close(fd);
        }

        exit(0);
}

