/*
 * A RTEMS sample application using the IP1553B core of the AGGA4 cpu
 *
 * Copyright (C),
 * Cobham Gaisler 2016
 *
 */

#include <rtems.h>
#include <bsp.h>
#include <agga4/ip1553b.h>

/* functions */

rtems_task Init(rtems_task_argument argument);

/* configuration information */

#define CONFIGURE_INIT
#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_MAXIMUM_SEMAPHORES        1
#define CONFIGURE_MAXIMUM_TASKS             1
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 4
#define CONFIGURE_APPLICATION_EXTRA_DRIVERS \
        AGGA4_IP1553B_DRIVER_TABLE_ENTRY
#include <rtems/confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>

void isr_trok(void *arg)
{
        struct ip1553b_cmd cmd;
        struct ip1553b_block blk;
        int fd = (int)arg;
        int status;
        uint32_t sa;
        uint16_t buf[32];

        status = ioctl(fd, IP1553B_IOCTL_GET_LAST, &cmd);
        if (status < 0) {
                goto end;
        }

        sa = cmd.cmd & IP1553B_CMD_WC >> IP1553B_CMD_WC_BIT;
        if (sa == 30) {
                /* Loopback data */
                blk.table = IP1553B_INDIRECTION_RX0;
                blk.sa = 30;
                blk.blk = 0;
                blk.ofs = 0;
                blk.cnt = cmd.cmd & IP1553B_CMD_SA >> IP1553B_CMD_SA_BIT;;
                blk.buf = &buf[0];
                status = ioctl(fd, IP1553B_IOCTL_RXBLOCK, &cmd);
                if (status < 0) {
                        goto end;
                }

                blk.table = IP1553B_INDIRECTION_TX0;
                status = ioctl(fd, IP1553B_IOCTL_TXBLOCK, &cmd);
                if (status < 0) {
                        goto end;
                }
        }

end:
        return;


}

volatile static int interrupt_err;
void isr_err(void *arg)
{
        interrupt_err++;
}


void print_error(int err, const char *fmt, ...)
{
        if (fmt) {
                va_list ap;
                va_start(ap, fmt);
                vfprintf(stderr, fmt, ap);
                va_end(ap);
        }
        fprintf(stderr, "IP1553B FAIL! %d - %s\n", err, strerror(err));

}

char memarea[512*1024] __attribute__((aligned(512*1024))) ;

rtems_task Init(rtems_task_argument argument)
{
        rtems_status_code status;
        struct ip1553b_indirection ind;
        struct ip1553b_char chr;
        struct ip1553b_isr trok;
        struct ip1553b_isr err;
        int fd = -1;

        fd = open(AGGA4_IP1553B_DEVNAME, O_RDWR, 0);
        if (fd < 0) {
                print_error(errno, "Failed to open "AGGA4_IP1553B_DEVNAME": ");
                goto end;
        }

        /* Example 1
         * =========
         * Setup
         */
        status = ioctl(fd, IP1553B_IOCTL_SET_MEM, &memarea[0]);
        if (status < 0) {
                print_error(errno, "E1 IP1553B_IOCTL_SET_MEM: ");
                goto end;
        }

        status = ioctl(fd, IP1553B_IOCTL_SET_CLK, IP1553B_CLK_PLL);
        if (status < 0) {
                print_error(errno, "E1 IP1553B_IOCTL_SET_TIMEOUT: ");
                goto end;
        }

        status = ioctl(fd, IP1553B_IOCTL_SET_RTAD, 1);
        if (status < 0) {
                print_error(errno, "E1 IP1553B_IOCTL_SET_RTAD: ");
                goto end;
        }

        status = ioctl(fd, IP1553B_IOCTL_SET_PARITY, IP1553B_PARITY_ODD);
        if (status < 0) {
                print_error(errno, "E1 IP1553B_IOCTL_SET_RTAD: ");
                goto end;
        }

        /* Example 2
         * =========
         * Loopback on subaddress 30
         */
        chr.idx = 30;
        chr.word = IP1553B_CHAR_NOM_RX_TROK| IP1553B_CHAR_NOM_RX_LEG  |
                   IP1553B_CHAR_NOM_RX_SSF | IP1553B_CHAR_NOM_RX_SREQ |
                                         IP1553B_CHAR_NOM_TX_LEG  |
                   IP1553B_CHAR_NOM_TX_SSF | IP1553B_CHAR_NOM_TX_SREQ;
        status = ioctl(fd, IP1553B_IOCTL_SET_CHAR, &chr);
        if (status < 0) {
                print_error(errno, "E2 IP1553B_IOCTL_SET_CHAR: ");
                goto end;
        }

        trok.irq = IP1553B_INTERRUPT_NOM_TROK;
        trok.isr = isr_trok;
        trok.arg = (void*)fd;
        if (status < 0) {
                print_error(errno, "E2 IP1553B_IOCTL_SET_ISR TROK: ");
                goto end;
        }

        err.irq = IP1553B_INTERRUPT_ERR_MEM;
        err.isr = isr_err;
        err.arg = NULL;
        if (status < 0) {
                print_error(errno, "E2 IP1553B_IOCTL_SET_ISR ERR: ");
                goto end;
        }

        ind.table = IP1553B_INDIRECTION_RX0;
        ind.sa = 30;
        ind.adupd = 0;
        ind.maxbs = 1;
        ind.brblk = 0;
        status = ioctl(fd, IP1553B_IOCTL_SET_INDIRECTION, &ind);
        if (status < 0) {
                print_error(errno, "E2 IP1553B_IOCTL_SET_INDIRECTION Rx0: ");
                goto end;
        }

        ind.table = IP1553B_INDIRECTION_TX0;
        ind.sa = 30;
        ind.adupd = 0;
        ind.maxbs = 1;
        ind.brblk = 0;
        status = ioctl(fd, IP1553B_IOCTL_SET_INDIRECTION, &ind);
        if (status < 0) {
                print_error(errno, "E2 IP1553B_IOCTL_SET_INDIRECTION Tx0: ");
                goto end;
        }

        status = ioctl(fd, IP1553B_IOCTL_START, NULL);
        if (status < 0) {
                print_error(errno, "E2 IP1553B_IOCTL_START ERR: ");
                goto end;
        }

        printf("IP1553B: PASS\n");
end:
        if (fd > 0) {
                close(fd);
        }

        exit(0);
}

