/*
 * A RTEMS sample application using the SPI core of the AGGA4 cpu.
 *
 * The example will perform some simple self-checks to verify the behaviour of the driver.
 *
 * The MOSI/MISO singals are required to be connected externally as loopback.
 *
 * Copyright (C),
 * Cobham Gaisler 2016
 *
 */

#include <rtems.h>
#include <bsp.h>
#include <agga4/spi.h>

/* functions */

rtems_task Init(rtems_task_argument argument);

/* configuration information */

#define CONFIGURE_INIT
#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_MAXIMUM_SEMAPHORES        1
#define CONFIGURE_MAXIMUM_TASKS             1
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 4
#define CONFIGURE_APPLICATION_EXTRA_DRIVERS \
        AGGA4_SPI_DRIVER_TABLE_ENTRY
#include <rtems/confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>

void print_error(int err, const char *fmt, ...)
{
        if (fmt) {
                va_list ap;
                va_start(ap, fmt);
                vfprintf(stderr, fmt, ap);
                va_end(ap);
        }
        fprintf(stderr, "SPI FAIL! %d - %s\n", err, strerror(err));

}

rtems_task Init(rtems_task_argument argument)
{
        rtems_status_code status;
        int num;
        struct spi_config cfg;
        int fd = -1;
        int i;
        char byte[2];
        short hword[2];

        fd = open(AGGA4_SPI_DEVNAME, O_RDWR, 0);
        if (fd < 0) {
                print_error(errno, "Failed to open "AGGA4_SPI_DEVNAME": ");
                goto end;
        }

        /* Example 1
         * =========
         */
        status = ioctl(fd, SPI_IOCTL_BLOCKING, SPI_BLOCKING);
        if (status < 0) {
                print_error(errno, "E1 SPI_IOCTL_BLOCKING: ");
                goto end;
        }

        cfg.length     = 8;
        cfg.slave      = 1;
        cfg.continous  = 0;
        cfg.msb_first  = 1;
        cfg.clk_sel    = 0;
        cfg.clk_phase  = 0;
        cfg.clk_pol    = 0;
        cfg.rcv_clk_pol= 0;
        cfg.clk_div    = 9; /* 1 MHz */
        status = ioctl(fd, SPI_IOCTL_SET_CONFIG, &cfg);
        if (status < 0) {
                print_error(errno, "E1 SPI_IOCTL_SET_CONFIG: ");
                goto end;
        }

        status = ioctl(fd, SPI_IOCTL_START, NULL);
        if (status < 0) {
                print_error(errno, "E1 SPI_IOCTL_START: ");
                goto end;
        }

        for (i=0; i<10; i++) {
                byte[0] = i;
                byte[1] = -1;
                num = write(fd, &byte[0], sizeof(byte[0]));
                if (num != 1) {
                        print_error(errno, "E1 Write: ");
                        goto end;
                }
                num = read(fd, &byte[1], sizeof(byte[1]));
                if (num != 1) {
                        print_error(errno, "E1 Read: ");
                        goto end;
                }
                if (byte[0] != byte[1]) {
                        fprintf(stderr, "E1 Failed to recive data: exp=%02x, act=%02x\n", byte[0], byte[1]);
                        goto end;
                }
        }


        /* Example 2
         * =========
         * Switch to continuous and non-blocking
         */
        status = ioctl(fd, SPI_IOCTL_STOP, NULL);
        if (status < 0) {
                print_error(errno, "E2 SPI_IOCTL_STOP: ");
                goto end;
        }

        status = ioctl(fd, SPI_IOCTL_BLOCKING, SPI_NONBLOCKING);
        if (status < 0) {
                print_error(errno, "E2 SPI_IOCTL_BLOCKING: ");
                goto end;
        }

        status = ioctl(fd, SPI_IOCTL_GET_CONFIG, &cfg);
        if (status < 0) {
                print_error(errno, "E2 SPI_IOCTL_SET_CONFIG: ");
                goto end;
        }

        cfg.continous = 1;
        cfg.length = 16;
        status = ioctl(fd, SPI_IOCTL_SET_CONFIG, &cfg);
        if (status < 0) {
                print_error(errno, "E2 SPI_IOCTL_SET_CONFIG: ");
                goto end;
        }

        status = ioctl(fd, SPI_IOCTL_START, NULL);
        if (status < 0) {
                print_error(errno, "E2 SPI_IOCTL_START: ");
                goto end;
        }
        for (i=0; i<10; i++) {
                hword[0] = ((-i) & 0xFF)<<8|(i & 0xFF);
                hword[1] = -1;
                while ((num = write(fd, &hword[0], sizeof(hword[0]))) == -1) {
                        if (errno != ETIMEDOUT) {
                                print_error(errno, "E2 Write: ");
                                goto end;
                        }
                }
                if (num != 2) {
                        fprintf(stderr, "E2 Write: Wrong number of bytes (%d)\n", num);
                        goto end;
                }

                while ((num = read(fd, &hword[1], sizeof(hword[1]))) == -1) {
                        if (errno != ETIMEDOUT) {
                                print_error(errno, "E2 Read: ");
                                goto end;
                        }
                }
                if (num != 2) {
                        fprintf(stderr, "E2 Read: Wrong number of bytes (%d)\n", num);
                        goto end;
                }
                if (hword[0] != hword[1]) {
                        fprintf(stderr, "E2 Failed to receive data: exp=%04x, act=%04x\n", hword[0], hword[1]);
                        goto end;
                }
        }
        printf("SPI: PASS\n");
end:
        if (fd > 0) {
                close(fd);
        }

        exit(0);
}

