/*
 * A RTEMS sample application using the SpaceWire and TimeCode core of the AGGA4 cpu
 *
 * The example will perform some simple self-checks to verify the behavior of the driver.
 *
 * The defines TX_DEVNAME and RX_DEVNAME selects which devices will be used.
 * Both can be defined to the same device if it is connected as loopback,.
 *
 * Copyright (C),
 * Cobham Gaisler 2016
 *
 */

#include <rtems.h>
#include <bsp.h>
#include <agga4/spw.h>
#include <agga4/tc.h>

/* functions */
rtems_task Init(rtems_task_argument argument);

/* configuration information */

#define CONFIGURE_INIT
#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_MAXIMUM_SEMAPHORES        1
#define CONFIGURE_MAXIMUM_TASKS             1
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 6
#define CONFIGURE_APPLICATION_EXTRA_DRIVERS \
        AGGA4_SPW_DRIVER_TABLE_ENTRY,\
        AGGA4_TC_DRIVER_TABLE_ENTRY
#include <rtems/confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>

#ifndef TX_DEVNAME
#define TX_DEVNAME AGGA4_SPW0_DEVNAME
#endif
#ifndef RX_DEVNAME
#define RX_DEVNAME AGGA4_SPW1_DEVNAME
#endif

static volatile int isr_count = 0;
void isr_func(void *arg)
{
        isr_count++;
}

void print_error(int err, const char *fmt, ...)
{
        if (fmt) {
                va_list ap;
                va_start(ap, fmt);
                vfprintf(stderr, fmt, ap);
                va_end(ap);
        }
        fprintf(stderr, "SPW FAIL! %d - %s\n", err, strerror(err));

}

#define TX 0
#define RX 1
#define TC 2
rtems_task Init(rtems_task_argument argument)
{
        int status;
        uint32_t reg;
        int fd[3] = {-1,-1,-1};
        int devc = 2;
        int i;
        struct tc_timecode timecode;
        struct spw_desc pkt;
        struct spw_isr isr;

        static uint8_t __attribute__((aligned(256))) rx[256];
        static uint8_t __attribute__((aligned(256))) tx[256];

	fd[TX] = open(TX_DEVNAME, O_RDWR, 0);
        if (fd[TX] < 0) {
                print_error(errno, "Failed to open "TX_DEVNAME": ");
                goto end;
        }
        if (strcmp(TX_DEVNAME, RX_DEVNAME) != 0) {
                fd[RX] = open(RX_DEVNAME, O_RDWR, 0);
                if (fd[RX] < 0) {
                        print_error(errno, "Failed to open "RX_DEVNAME": ");
                        goto end;
                }
        } else {
                devc = 1;
                fd[RX] = fd[TX];
        }

        for (i=0; i<sizeof(tx); i++) {
                tx[i] = i;
        }

        /* Example 1
         * =========
         * Setup connection
         */
        for (i=0; i<devc; i++) {
                status = ioctl(fd[i], SPW_IOCTL_SET_STOP_EOP, SPW_STOP_EOP_YES);
                if (status < 0) {
                        print_error(errno, "E1 (%d) SPW_IOCTL_SET_STOP_EOP: ", i);
                        goto end;
                }

                status = ioctl(fd[i], SPW_IOCTL_SET_RATE, SPW_RATE_10MBIT);
                if (status < 0) {
                        print_error(errno, "E1 (%d) SPW_IOCTL_SET_RATE: ", i);
                        goto end;
                }
                status = ioctl(fd[i], SPW_IOCTL_AUTOSTART, NULL);
                if (status < 0) {
                        print_error(errno, "E1 (%d) SPW_IOCTL_AUTOSTART: ", i);
                        goto end;
                }
        }
        status = ioctl(fd[TX], SPW_IOCTL_START, NULL);
        if (status < 0) {
                print_error(errno, "E1 (%d) SPW_IOCTL_START: ", TX);
                goto end;
        }
        for (i=0; i<devc; i++) {
                status = ioctl(fd[i], SPW_IOCTL_GET_STATUS, &reg);
                if (status < 0) {
                        print_error(errno, "E1 (%d) SPW_IOCTL_GET_STATUS: ", i);
                        goto end;
                }
                if (!(reg & SPW_STATUS_OK)) {
                        fprintf(stderr, "E1 (%d) Failed to connect: %s%s%s%s", i,
                                        status&SPW_STATUS_FCT ? "FCT" :"",
                                        status&SPW_STATUS_ESC ? "ECS" :"",
                                        status&SPW_STATUS_PAR ? "PAR" :"",
                                        status&SPW_STATUS_DIS ? "DIS" :""
                        );
                        goto end;
                }
        }

        printf("SPW: Test 1 OK\n");

        /*
         * Example 2
         * =========
         * Get current timecode
         */
        fd[TC] = open(AGGA4_TC_DEVNAME, O_RDWR, 0);
        if (fd[TC] < 0) {
                print_error(errno, "Failed to open "AGGA4_TC_DEVNAME": ");
                goto end;
        }

        status = ioctl(fd[TC], TC_IOCTL_GET_TIMECODE, &timecode);
        if (status < 0) {
                print_error(errno, "E2 TC_IOCTL_SET_TIMECODE: ");
                goto end;
        }

        printf("SPW: Test 2 OK\n");

        /* Example 3
         * =========
         * Send and recieve a packet (loopback), polling result.
         */
        pkt.addr = &rx[0];
        pkt.bytes = sizeof(rx);
        status = ioctl(fd[RX], SPW_IOCTL_RX, &pkt);
        if (status < 0) {
                print_error(errno, "E3 (%d) SPW_IOCTL_RX: ", RX);
                goto end;
        }

        pkt.addr = &tx[0];
        pkt.bytes = sizeof(tx);
        status = ioctl(fd[TX], SPW_IOCTL_TX, &pkt);
        if (status < 0) {
                print_error(errno, "E3 (%d) SPW_IOCTL_TX: ", TX);
                goto end;
        }
        do {
                status = ioctl(fd[TX], SPW_IOCTL_TX_POLL, NULL);
        } while (status < 0 && errno == ETIMEDOUT);
        if (status < 0) {
                print_error(errno, "E3 (%d) SPW_IOCTL_TX_POLL: ", TX);
                goto end;
        }

        do {
                status = ioctl(fd[RX], SPW_IOCTL_RX_POLL, NULL);
        } while (status < 0 && errno == ETIMEDOUT);
        if (status < 0) {
                print_error(errno, "E3 (%d) SPW_IOCTL_RX_POLL: ", RX);
                goto end;
        }

	printf("SPW: Test 3 OK\n");

        /* Example 4
         * =========
         * Send and receive a packet using interrupts TXDONE and EOP
         */
        isr.irq = SPW_ISR_TXDONE;
        isr.isr = isr_func;
        isr.arg = NULL;
        status = ioctl(fd[TX], SPW_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E4 (%d) SPW_IOCTL_SET_ISR: ", TX);
                goto end;
        }

	isr.irq = SPW_ISR_EOP;
        isr.isr = isr_func;
        isr.arg = NULL;
        status = ioctl(fd[RX], SPW_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E4 (%d) SPW_IOCTL_SET_ISR: ", RX);
                goto end;
        }
	isr_count = 0;

	pkt.addr = &rx[0];
        pkt.bytes = sizeof(rx);
        status = ioctl(fd[RX], SPW_IOCTL_RX, &pkt);
        if (status < 0) {
                print_error(errno, "E4 (%d) SPW_IOCTL_RX: ", RX);
                goto end;
        }

        pkt.addr = &tx[0];
        pkt.bytes = sizeof(tx)/2;
        status = ioctl(fd[TX], SPW_IOCTL_TX, &pkt);
        if (status < 0) {
                print_error(errno, "E4 (%d) SPW_IOCTL_TX: ", TX);
                goto end;
        }
        while (isr_count < 2) {
                ;
        }

	isr.irq = SPW_ISR_TXDONE;
        isr.isr = NULL;
        isr.arg = NULL;
        status = ioctl(fd[TX], SPW_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E4 (%d) SPW_IOCTL_SET_ISR (unset): ", TX);
                goto end;
        }
	isr.irq = SPW_ISR_EOP;
        isr.isr = NULL;
        isr.arg = NULL;
        status = ioctl(fd[RX], SPW_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E4 (%d) SPW_IOCTL_SET_ISR (unset): ", RX);
                goto end;
        }

	if (isr_count != 2) {
		fprintf(stderr, "E4 Wrong irq count Act %d Exp 2\n", isr_count);
		goto end;
	}

	printf("SPW: Test 4 OK\n");

        /* Example 5
         * =========
         * Send and receive a packet using interrupts RXFULL
         */
	isr.irq = SPW_ISR_RXAREA;
        isr.isr = isr_func;
        isr.arg = NULL;
        status = ioctl(fd[RX], SPW_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E5 (%d) SPW_IOCTL_SET_ISR: ", RX);
                goto end;
        }
	isr_count = 0;

	pkt.addr = &rx[0];
        pkt.bytes = sizeof(rx)/2;
        status = ioctl(fd[RX], SPW_IOCTL_RX, &pkt);
        if (status < 0) {
                print_error(errno, "E5 (%d) SPW_IOCTL_RX: ", RX);
                goto end;
        }

        pkt.addr = &tx[0];
        pkt.bytes = sizeof(tx);
        status = ioctl(fd[TX], SPW_IOCTL_TX, &pkt);
        if (status < 0) {
                print_error(errno, "E5 (%d) SPW_IOCTL_TX: ", TX);
                goto end;
        }

        while (isr_count < 1) {
                ;
        }

	isr.irq = SPW_ISR_RXAREA;
        isr.isr = NULL;
        isr.arg = NULL;
        status = ioctl(fd[RX], SPW_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E5 (%d) SPW_IOCTL_SET_ISR: ", RX);
                goto end;
        }

	if (isr_count != 1) {
		fprintf(stderr, "E4 Wrong irq count: Act %d Exp 1\n", isr_count);
		goto end;
	}

        printf("SPW: Test 5 OK\n");

	printf("SPW: PASS\n");

end:
        if (fd[TC] > 0) {
                close(fd[TC]);
        }
        if (fd[TX] > 0) {
                close(fd[TX]);
        }
        if (fd[RX] > 0 && devc >= 1) {
                close(fd[RX]);
        }

        exit(0);
}

