/*
 * A RTEMS sample application using the SPW core of the AGGA4 cpu
 *
 * The example will perform some simple self-checks to verify the behaviour of the driver.
 *
 * UART1 is used by the example and it is put in internal loopback, no external connection is needed.
 *
 * Copyright (C),
 * Cobham Gaisler 2016
 *
 */

#include <rtems.h>
#include <bsp.h>
#include <agga4/uart.h>

/* functions */

rtems_task Init(rtems_task_argument argument);

/* configuration information */

#define CONFIGURE_INIT
#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
#define CONFIGURE_MAXIMUM_SEMAPHORES        1
#define CONFIGURE_MAXIMUM_TASKS             1
#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_EXTRA_TASK_STACKS         (3 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 4
#define CONFIGURE_APPLICATION_EXTRA_DRIVERS \
        AGGA4_UART_DRIVER_TABLE_ENTRY
#include <rtems/confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>

#define BUF_SIZE 256

static volatile int isr_rx_count = 0;
struct isr_rx_arg {
        int fd;
        int current;
        uint8_t *bufs[2];
};
void isr_rx_func(void *arg)
{
        struct isr_rx_arg *a = arg;
        struct uart_desc desc;

        a->current = !a->current;
        desc.addr = a->bufs[a->current];
        desc.bytes = BUF_SIZE;
        ioctl(a->fd, UART_IOCTL_RX, &desc);

        isr_rx_count++;
}

static volatile int isr_tx_count = 0;
void isr_tx_func(void *arg)
{
        isr_tx_count++;
}

void print_error(int err, const char *fmt, ...)
{
        if (fmt) {
                va_list ap;
                va_start(ap, fmt);
                vfprintf(stderr, fmt, ap);
                va_end(ap);
        }
        fprintf(stderr, "UART FAIL! %d - %s\n", err, strerror(err));

}

rtems_task Init(rtems_task_argument argument)
{
        int status;
        uint32_t reg;
        int fd = -1;
        uint8_t __attribute__((aligned(4))) rx[2][BUF_SIZE];
        uint8_t __attribute__((aligned(4))) tx[2][BUF_SIZE];
        struct uart_isr isr;
        struct isr_rx_arg rx_arg;
        struct uart_desc desc;

        fd = open(AGGA4_UART1_DEVNAME, O_RDWR, 0);
        if (fd < 0) {
                print_error(errno, "Failed to open "AGGA4_UART1_DEVNAME": ");
                goto end;
        }

        /* Example 1
         * =========
         * Setup connection
         */
        *((volatile uint32_t*)0x800001C0) |= 1<<7; // Enable loopback for uart1

        status = ioctl(fd, UART_IOCTL_SET_RATE, (void*)38400);
        if (status < 0) {
                print_error(errno, "E1 UART_IOCTL_SET_RATE: ");
                goto end;
        }

        status = ioctl(fd, UART_IOCTL_SET_ENDIAN, (void*)UART_ENDIAN_BIG);
        if (status < 0) {
                print_error(errno, "E1 UART_IOCTL_SET_ENDIAN: ");
                goto end;
        }

        status = ioctl(fd, UART_IOCTL_SET_PARITY, (void*)UART_PARITY_NONE);
        if (status < 0) {
                print_error(errno, "E1 UART_IOCTL_SET_PARITY: ");
                goto end;
        }

        status = ioctl(fd, UART_IOCTL_SET_FLOW, (void*)UART_FLOW_NO);
        if (status < 0) {
                print_error(errno, "E1 UART_IOCTL_SET_FLOW: ");
                goto end;
        }

        status = ioctl(fd, UART_IOCTL_START, NULL);
        if (status < 0) {
                print_error(errno, "E1 UART_IOCTL_START: ");
                goto end;
        }

        status = ioctl(fd, UART_IOCTL_GET_STATUS, &reg);
        if (status < 0) {
                print_error(errno, "E1 UART_IOCTL_GET_STATUS: ");
                goto end;
        }
        if (reg != (UART_STATUS_TSE|UART_STATUS_THE)) {
                fprintf(stderr, "Status: %08x %s%s%s%s%s%s%s%s\n",
                        (unsigned int)reg,
                        reg&UART_STATUS_DR  ? "DR "  :"",
                        reg&UART_STATUS_TSE ? "TSE " :"",
                        reg&UART_STATUS_THE ? "THE " :"",
                        reg&UART_STATUS_BR  ? "BR "  :"",
                        reg&UART_STATUS_OVR ? "OVR " :"",
                        reg&UART_STATUS_PER ? "PER " :"",
                        reg&UART_STATUS_FER ? "FER " :"",
                        reg&UART_STATUS_RTO ? "RTO " :""
                );
                goto end;
        }

        /* Example 2
         * =========
         * Setup interrupt handler to alternate between buffers
         */
        rx_arg.bufs[0] = &rx[0][0];
        rx_arg.bufs[1] = &rx[1][0];
        rx_arg.current = 0;
        rx_arg.fd = fd;

        isr.isr = isr_rx_func;
        isr.arg = &rx_arg;
        isr.irq = UART_ISR_RXDONE;
        status = ioctl(fd, UART_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E2 UART_IOCTL_SET_ISR: ");
                goto end;
        }

        desc.addr = &rx[0][0];
        desc.bytes = BUF_SIZE;
        status = ioctl(fd, UART_IOCTL_RX, &desc);
        if (status < 0) {
                print_error(errno, "E2 UART_IOCTL_RX: ");
                goto end;
        }

        /* Example 3
         * =========
         * Send a packet and poll until ready
         */
        strncpy((char*)tx[0], "My First Packet (polling)!\n", BUF_SIZE);
        desc.addr = tx[0];
        desc.bytes = BUF_SIZE;
        status = ioctl(fd, UART_IOCTL_TX, &desc);
        if (status < 0) {
                print_error(errno, "E3 UART_IOCTL_TX: ");
                goto end;
        }

        while(ioctl(fd, UART_IOCTL_TX_POLL, NULL) < 0 && errno == ETIMEDOUT) {
                ;
        }

        while(isr_rx_count <= 0) {
                ;
        }
        if (strcmp((char*)tx[0], (char*)rx[0]) != 0) {
                print_error(errno, "E3 Failed to receive packet!");
                goto end;
        }


        /* Example 4
         * =========
         * Send a packet and poll until ready
         */
        isr.isr = isr_tx_func;
        isr.arg = NULL;
        isr.irq = UART_ISR_TXDONE;
        status = ioctl(fd, UART_IOCTL_SET_ISR, &isr);
        if (status < 0) {
                print_error(errno, "E4 UART_IOCTL_SET_ISR: ");
                goto end;
        }

        strncpy((char*)tx[1], "My Second Packet! (interrupt)\n", BUF_SIZE);
        desc.addr = tx[1];
        desc.bytes = BUF_SIZE;
        status = ioctl(fd, UART_IOCTL_TX, &desc);
        if (status < 0) {
                print_error(errno, "E4 UART_IOCTL_TX: ");
                goto end;
        }

        while (isr_tx_count == 0) {
                ;
        }

        while (isr_rx_count <= 1) {
                ;
        }
        if (strcmp((char*)tx[1], (char*)rx[1]) != 0) {
                print_error(errno, "E4 Failed to receive packet!");
                goto end;
        }

        printf("UART: PASS\n");
end:
        if (fd > 0) {
                close(fd);
        }

        exit(0);
}

