/*
 *  Interface for GRFIFO driver
 *
 *  COPYRIGHT (c) 2007.
 *  Gaisler Research
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 *
 */

#ifndef __GRFIFO_H__
#define __GRFIFO_H__

#include <amba.h>

struct grfifo_regs {
  volatile unsigned int cfg;              /* 0x00 */
  int space0[1];
  volatile unsigned int ctrl;             /* 0x08 */
  int space1[5];
  volatile unsigned int txchan_ctrl;      /* 0x20 */
  volatile unsigned int txchan_status;    /* 0x24 */
  volatile unsigned int txchan_addr;      /* 0x28 */
  volatile unsigned int txchan_size;      /* 0x2c */
  volatile unsigned int txchan_write;     /* 0x30 */
  volatile unsigned int txchan_read;      /* 0x34 */
  volatile unsigned int txchan_irq;       /* 0x38 */
  int space2[1];
  volatile unsigned int rxchan_ctrl;      /* 0x40 */
  volatile unsigned int rxchan_status;    /* 0x44 */
  volatile unsigned int rxchan_addr;      /* 0x48 */
  volatile unsigned int rxchan_size;      /* 0x4c */
  volatile unsigned int rxchan_write;     /* 0x50 */
  volatile unsigned int rxchan_read;      /* 0x54 */
  volatile unsigned int rxchan_irq;       /* 0x58 */
  int space3[1];
  volatile unsigned int dinput;           /* 0x60 */
  volatile unsigned int doutput;          /* 0x64 */
  volatile unsigned int ddir;             /* 0x68 */
};

struct grfifo_stats {
	unsigned int ahb_rx_errors;
	unsigned int rxparity_errors;
	unsigned int ahb_tx_errors;
};

struct grfifo_config {
	unsigned int abort_on_ahb_err; /* 0=don't abort on AHB error, oethers abort */
	unsigned int data_width;       /* 1=8-bit, 2=16-bit, others invalid. */
	unsigned int parity;           /* 0=even, 1=odd, others invalid */
	unsigned int wait_state_count; /* Number of wait states: 0..7, others invalid. */
};

struct grfifo_bufinfo {
  void *rx_buf;
  int  rx_length;
  void *tx_buf;
  int  tx_length;
};

struct grfifo_block_head;

struct grfifo_block_head {
  struct grfifo_block_head *next;
  void *buffer;
  unsigned int dlen;
};


/* User space usage example: 

struct block;

struct block {
  struct grfifo_block_head drv_data;
  
  // The real buffer data
  void *buf;
  int buf_len;
  struct block *next;
  struct block *prev;
}

*/

#define GRFIFO_CFG_ABORT  0x40
#define GRFIFO_CFG_DW     0x30
#define GRFIFO_CFG_PARITY 0x08
#define GRFIFO_CFG_WS     0x07

#define GRFIFO_CFG_ABORT_BIT  6
#define GRFIFO_CFG_DW_BIT     4
#define GRFIFO_CFG_PARITY_BIT 3
#define GRFIFO_CFG_WS_BIT     0

#define GRFIFO_CTRL_RESET 0x2

#define GRFIFO_RXSTAT_BYTECNT 0x700
#define GRFIFO_RXSTAT_ONGOING 0x40
#define GRFIFO_RXSTAT_PARITY  0x20
#define GRFIFO_RXSTAT_IRQ     0x10
#define GRFIFO_RXSTAT_FULL    0x08
#define GRFIFO_RXSTAT_ERROR   0x04
#define GRFIFO_RXSTAT_EF      0x02
#define GRFIFO_RXSTAT_HF      0x01

#define GRFIFO_RXSTAT_BYTECNT_BIT 8

#define GRFIFO_TXSTAT_ONGOING 0x40
#define GRFIFO_TXSTAT_IRQ     0x10
#define GRFIFO_TXSTAT_EMPTY   0x08
#define GRFIFO_TXSTAT_ERROR   0x04
#define GRFIFO_TXSTAT_FF      0x02
#define GRFIFO_TXSTAT_HF      0x01

#define GRFIFO_IRQ_TXIRQ 0
#define GRFIFO_IRQ_TXEMPTY 1
#define GRFIFO_IRQ_TXERROR 2
#define GRFIFO_IRQ_RXIRQ 3
#define GRFIFO_IRQ_RXFULL 4
#define GRFIFO_IRQ_RXERROR 5
#define GRFIFO_IRQ_RXPARITY 6

/* IOCTRL definitions */

#define GRFIFO_IOC_NONE 0 /* results in error response */

/*** I/O CONTROL operational commands */
#define GRFIFO_IOC_START 1       /* start transmitter & receiver (enables read/write) */
#define GRFIFO_IOC_STOP 2        /* stop transmitter & receiver */
#define GRFIFO_IOC_FLUSH 3       /* tx - block util all data has been sent */

#define GRFIFO_IOC_PREPBLKS 4    /* rx - prepare receiver with new blocks */
#define GRFIFO_IOC_GETBLKS 5     /* rx - receive blocks */

#define GRFIFO_IOC_PUTBLKS 6     /* tx - send blocks */
#define GRFIFO_IOC_REAPBLKS 7    /* tx - retrive sent blocks */

/*** I/O CONTROL commands valid only when grfifo is stopped */
#define GRFIFO_IOC_SET_MODE 16
#define GRFIFO_IOC_SET_ABORT 17
#define GRFIFO_IOC_SET_DATA_WIDTH 18
#define GRFIFO_IOC_SET_PARITY 19
#define GRFIFO_IOC_SET_WS_CNT 20
#define GRFIFO_IOC_SET_RXBUFLEN 21
#define GRFIFO_IOC_SET_TXBUFLEN 22

/* I/O CONTROL command valid during running and stopped state */
#define GRFIFO_IOC_SET_RXBLOCK 32
#define GRFIFO_IOC_SET_TXBLOCK 33
#define GRFIFO_IOC_SET_TXCOMPLETE 34
#define GRFIFO_IOC_SET_RXCOMPLETE 35
#define GRFIFO_IOC_GET_STATS 36
#define GRFIFO_IOC_CLR_STATS 37
#define GRFIFO_IOC_GET_BUFINFO 38


#define IOCTL_CONFIG_DW_8 1
#define IOCTL_CONFIG_DW_16 2

struct grfifo_device_info {
  unsigned int base_address;
  int irq;
};

/* Use hard coded addresses and IRQs to find hardware */
int grfifo_register_abs(struct grfifo_device_info *devices, int dev_cnt);

/* Use prescanned AMBA Plug&Play information to find all GRFIFO cores */
int grfifo_register(amba_confarea_type *abus);

#endif
