/*  General purpose input/output (GRPULSE) interface
 *
 *  GRPULSE is similar to GRGPIO but it has the ability
 *  to make short pulses (0..255 clks).
 * 
 *  COPYRIGHT (c) 2007.
 *  Gaisler Research
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 */

#ifndef __GRPULSE_H__
#define __GRPULSE_H__

struct grpulse {
	volatile unsigned int input;
	volatile unsigned int output;
	volatile unsigned int dir;
	volatile unsigned int imask;
	volatile unsigned int ipol;
	volatile unsigned int iedge;
	volatile unsigned int pulse;
	volatile unsigned int pulsecnt;
};

/* Initializes the grpulse help functions and resets the
 * grpulse core to initial state. (input, irq disabled)
 *
 * \param base_addr Base address of grpulse
 * \return number of available GPIO ports
 */
int grpulse_init(unsigned int base_addr, int base_irq);

/* Returns number of GPIO ports available */
int grpulse_get_port_count(void);

/* Returns number of GPIO ports with IRQ feature available */
int grpulse_get_irqport_count(void);

/* Returns number of GPIO ports with pulse feature available */
int grpulse_get_pulseport_count(void);

/********* Operate on all ports *********/

unsigned int grpulse_all_get_output(void);
void grpulse_all_set_output(unsigned int mask);

unsigned int grpulse_all_get_input(void);
void grpulse_all_set_input(unsigned int mask);

unsigned int grpulse_all_get_dir(void);
void grpulse_all_set_dir(unsigned int mask);

unsigned int grpulse_all_get_pulse(void);
void grpulse_all_set_pulse(unsigned int mask);

void grpulse_set_pulse_width(int clks);
int grpulse_get_pulse_width(void);

unsigned int grpulse_all_get_imask(void);
void grpulse_all_set_imask(unsigned int mask);

unsigned int grpulse_all_get_iedge(void);
void grpulse_all_set_iedge(unsigned int mask);

unsigned int grpulse_all_get_ipol(void);
void grpulse_all_set_ipol(unsigned int mask);

/********* Operate on a single port *********/

int grpulse_get_input(int port);

void grpulse_set_output(int port, int output);
int  grpulse_get_output(int port);

void grpulse_set_dir(int port, int dir);
int grpulse_get_dir(int port);

void grpulse_set_pulse(int port, int enable_pulse);
int grpulse_get_pulse(int port);

void grpulse_irq_enable(int port);
void grpulse_irq_disable(int port);

void grpulse_irq_pol(int port, int high);
void grpulse_irq_edge(int port, int edge);

/* installs an Interrupt Service Routine for the specified GPIO port.
 * 
 * \param  timer  what timer to install ISR to. [0..n]
 * \param  isr    handler function to be called when the IRQ fires.
 * \param  arg    custom argument passed along when the ISR is called.
 * \return zero   on success, non-zero on failure. 
 */
int grpulse_install_irq_handler(int port, void (*isr)(int irq, void *arg), void *arg);

#endif
