/* Modular timer (GRTIMER) interface
 * 
 *  COPYRIGHT (c) 2007.
 *  Gaisler Research
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 */

#ifndef __GRTIMER_H__
#define __GRTIMER_H__

#ifndef GRTIMER_MAX_TIMERS
 #define GRTIMER_MAX_TIMERS 8
#endif

struct grtimer_timer {
	volatile unsigned int counter;
	volatile unsigned int reload;
	volatile unsigned int control;
	volatile unsigned int latch;
};

struct grtimer {
	volatile unsigned int scaler;
	volatile unsigned int reload;
	volatile unsigned int config;
	volatile unsigned int latch;
	struct grtimer_timer timers[GRTIMER_MAX_TIMERS];
};

#define GRTIMER_CFG_TIMERS 0x007
#define GRTIMER_CFG_IRQ    0x0f8
#define GRTIMER_CFG_SI     0x100
#define GRTIMER_CFG_DF     0x200
#define GRTIMER_CFG_EE     0x400
#define GRTIMER_CFG_EL     0x800

#define GRTIMER_CFG_TIMERS_BIT 0
#define GRTIMER_CFG_IRQ_BIT    3
#define GRTIMER_CFG_SI_BIT     8
#define GRTIMER_CFG_DF_BIT     9
#define GRTIMER_CFG_EE_BIT     10
#define GRTIMER_CFG_EL_BIT     11

#define GRTIMER_TIM_CTRL_EN    0x01
#define GRTIMER_TIM_CTRL_RS    0x02
#define GRTIMER_TIM_CTRL_LD    0x04
#define GRTIMER_TIM_CTRL_IE    0x08
#define GRTIMER_TIM_CTRL_IP    0x10
#define GRTIMER_TIM_CTRL_CH    0x20
#define GRTIMER_TIM_CTRL_DH    0x40

#define GRTIMER_TIM_CTRL_EN_BIT 0
#define GRTIMER_TIM_CTRL_RS_BIT 1
#define GRTIMER_TIM_CTRL_LD_BIT 2
#define GRTIMER_TIM_CTRL_IE_BIT 3
#define GRTIMER_TIM_CTRL_IP_BIT 4
#define GRTIMER_TIM_CTRL_CH_BIT 5
#define GRTIMER_TIM_CTRL_DH_BIT 6

/* Initializes the grtimer help functions and resets all timers
 * to initial state.
 *
 * \param base_addr Base address of Modular timer
 * \return number of available timers
 */
int grtimer_init(unsigned int base_addr);

/* Returns number of timers available */
int grtimer_get_timer_count(void);

/* Enable/disable timer freeze when DSU(Debug support Unit) freezes 
 * the CPU during debugging.
 */
void grtimer_freeze(int enable_freeze);

/* Enables/disables the Prescaler to be clocked from an external 
 * clock source.
 */
void grtimer_extclk(int enable_external_clock_source);

/* Enables/disables the timer counter value to be latched into 
 * each timer's latch register when an IRQ (which matches the
 * latch configuration register) is fired
 */
void grtimer_latching(int enable_latching);

/* Sets the latch configuration register,
 * irqmask is a bit mask where one bit represents an
 * IRQ from the Second Interrupt controller. Setting
 * a bit in the latch configuration register makes the
 * timer's counter value be loaded into their respective
 * timer latch register.
 */
void grtimer_set_latchcfg(unsigned int irqmask);

/* Returns the latch IRQ selection register */
unsigned int grtimer_get_latchcfg(void);

/* Set Prescaler value */
void grtimer_set_scaler(unsigned int scaler);

/* Get Prescaler value */
unsigned int grtimer_get_scaler(void);

/* Set Prescaler reload value */
void grtimer_set_screload(unsigned int scaler_reload);

/* Get Prescaler reload value */
unsigned int grtimer_get_screload(void);



/**** Per timer functions ****/

/* Resets one timer (ISR will still be connected to IRQ source):
 *  - IRQ disabled
 *  - IRQ pending cleared
 *  - Disable Restart
 *  - Disables timers
 *  - Disable Chaining
 *  - Reload value is set to max (0xffffffff)
 *  - counter is loaded with reload value
 *  - latch is cleared
 */
void grtimer_reset(int timer);

/* installs an Interrupt Service Routine for the specified timer.
 * The handler must call grtimer_irq_clrpend(timer) to clear pending
 * interrupts, othervise multiple entires to isr will be experienced
 * 
 * \param  timer  what timer to install ISR to. [0..n]
 * \param  isr    handler function to be called when the IRQ fires.
 * \param  arg    custom argument passed along when the ISR is called.
 * \return zero   on success, non-zero on failure. 
 */
int grtimer_install_irq_handler(int timer, void (*isr)(int irq, void *arg), void *arg);

/* Clear any pending interrupts to the given timer 
 * \param timer what timer to clear pending IRQ for. [0..n]
 */
void grtimer_irq_clrpend(int timer);

/* Enable the ISR previously installed by grtimer_irq_install_handler.
 */
void grtimer_irq_enable(int timer);

/* Disable the ISR previously installed by grtimer_irq_install_handler.
 */
void grtimer_irq_disable(int timer);

/* Enables timerN to be chained to timerN-1. 
 * When timerN-1 underflow timerN will decrement once.
 * \param timer
 * \param enable_chaining zero=disbale chaining, non-zero=enable chaining
 */
void grtimer_chain(int timer, int enable_chaining);

/* Load timer with the timer reload value.
 * \param timer
 */
void grtimer_load(int timer);

/* Make timer reload with the timer reload value and continue decrementing 
 * when underflowing.
 *
 * \param timer
 * \param enable_restart zero=disable restart, non-zero=enable restart
 */
void grtimer_restart(int timer, int enable_restart);

/* Enable/disable the timer, start/stop the countdown */
void grtimer_enable(int timer, int enable);

/* Set a timer's latch value */
void grtimer_set_latch(int timer, unsigned int latch_value);

/* get a timer's current latch value */
unsigned int grtimer_get_latch(int timer);

/* Set a timer's reload register value */
void grtimer_set_reload(int timer, unsigned int reload_value);

/* get a timer's reload register value */
unsigned int grtimer_get_reload(int timer);

/* Set a timer's counter register value */
void grtimer_set_counter(int timer, unsigned int counter_value);

/* get a timer's counter register value */
unsigned int grtimer_get_counter(int timer);

#endif
