/************************************************************************/
/*   This file is a part of the mkprom3 boot-prom utility               */
/*   Copyright (C) 2004 Gaisler Research                                */
/*                                                                      */
/*   This library is free software; you can redistribute it and/or      */
/*   modify it under the terms of the GNU General Public                */
/*   License as published by the Free Software Foundation; either       */
/*   version 2 of the License, or (at your option) any later version.   */
/*                                                                      */
/*   See the file COPYING.GPL for the full details of the license.      */
/************************************************************************/

/*
* This file is part of MKPROM.
* 
* MKPROM3, LEON boot-prom utility. 
* Copyright (C) 2004 Gaisler Research - all rights reserved.
* 
*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#define VAL(x)  strtoul(x,(char **)NULL,0)
#define SECMAX  32
#define SECNAME 16

typedef struct sectype
{
    unsigned int paddr;
    unsigned int raddr;
    unsigned int len;
    unsigned int comp;
    unsigned char name[SECNAME];
}
tt;

struct sectype secarr[SECMAX];
char filename[128] = "a.out";

#define TOOLBASE "/opt/rtems-4.6"
const char version[] = "v1.0.13a";
int secnum = 0;
FILE *dumpfile;

int verbose = 0;
int leon = 1;
double freq = 5E7;
int comp = 1;
int flash = 0;
int entry = 0;
int starta = 0;

main (argc, argv)
     int argc;
     char **argv;

{

    char cmd[512];
    char msg[128];
    int baud = 19200;
    int dsubaud = 0;
    int i;
    int mctrl = 0x00880017;
    int memcfg = 0;
    int wrp = 0;
    int ramcs = 1;
    int rambanks = 1;
    int ramsize = 0x200000;
    int romsize = 0x80000;
    int ramws = 0;
    int ramrws = 0;
    int ramwws = 0;
    int ramwidth = 32;
    int romwidth = 32;
    int rmw = 0;
    int romws = 2;
    int romrws = 2;
    int romwws = 2;
    int stack = 0;
    int stat = 1;
    int bmsg = 1;
    int bdinit = 0;
    int dump = 0;
    char ofile[128] = "prom.out";
    int tmp, iows, ioarea, tmp2;
    int sdramsz = 0;
    int nosram = 0;
    int noinit = 0;
    int sdrambanks = 1;
    int sdcas = 0;
    int trp = 20;
    int trfc = 66;
    int colsz = 1;
    double refresh = 7.8;
    double ftmp;
    int refr;
    unsigned int startaddr = 0;
    unsigned int entry = 0;
    unsigned int uaddr = 0x80000100;
    unsigned int memcaddr = 0x80000000;
    unsigned int dsuuart_addr = 0x80000700;
    unsigned int gptaddr = 0x80000300;

    printf ("\nLEON MKPROM boot-prom builder for RCC %s\n", version);
    printf ("Copyright Gaisler Research 2004, all rights reserved.\n\n");
    if ((dumpfile = fopen ("dump.s", "w+")) == NULL)
      {
	  printf ("Failed to open temporary file\n");
	  exit (1);
      }
    while (stat < argc)
      {
	  if (argv[stat][0] == '-')
	    {
		if (strcmp (argv[stat], "-v") == 0)
		  {
		      verbose = 1;
		  }
		else if (strcmp (argv[stat], "-dsubaud") == 0)
		  {
		      if ((stat + 1) < argc)
			  dsubaud = VAL (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-baud") == 0)
		  {
		      if ((stat + 1) < argc)
			  baud = VAL (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-dump") == 0)
		  {
		      dump = 1;
		  }
		else if (strcmp (argv[stat], "-nocomp") == 0)
		  {
		      comp = 0;
		  }
		else if (strcmp (argv[stat], "-wrp") == 0)
		  {
		      wrp = 1;
		  }
		else if (strcmp (argv[stat], "-nomsg") == 0)
		  {
		      bmsg = 0;
		  }
		else if (strcmp (argv[stat], "-bdinit") == 0)
		  {
		      bdinit = 1;
		  }
		else if (strcmp (argv[stat], "-freq") == 0)
		  {
		      if ((stat + 1) < argc)
			  freq = atof (argv[++stat]);
		      freq *= 1E6;
		  }
		else if (strcmp (argv[stat], "-memc") == 0)
		  {
		      if ((stat + 1) < argc)
			  memcaddr = VAL (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-gpt") == 0)
		  {
		      if ((stat + 1) < argc)
			  gptaddr = VAL (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-duart") == 0)
		  {
		      if ((stat + 1) < argc)
			  dsuuart_addr = VAL (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-col") == 0)
		  {
		      if ((stat + 1) < argc)
			  colsz = VAL (argv[++stat]) - 8;
		      if ((colsz < 0) || (colsz > 3))
			  colsz = 1;
		  }
		else if (strcmp (argv[stat], "-start") == 0)
		  {
		      if ((stat + 1) < argc)
			  starta = VAL (argv[++stat]) & ~3;
		  }
		else if (strcmp (argv[stat], "-cas") == 0)
		  {
		      if ((stat + 1) < argc)
			  sdcas = VAL (argv[++stat]) - 2;
		      if ((sdcas < 0) || (sdcas > 1))
			  sdcas = 1;
		  }
		else if (strcmp (argv[stat], "-sdrambanks") == 0)
		  {
		      if ((stat + 1) < argc)
			  sdrambanks = VAL (argv[++stat]);
		      if ((sdrambanks < 1) || (sdrambanks > 4))
			  sdrambanks = 1;
		  }
		else if (strcmp (argv[stat], "-nosram") == 0)
		  {
		      nosram = 1;
		  }
		else if (strcmp (argv[stat], "-noinit") == 0)
		  {
		      noinit = 1;
		  }
		else if (strcmp (argv[stat], "-sdram") == 0)
		  {
		      if ((stat + 1) < argc)
			  sdramsz = VAL (argv[++stat]);
		      sdramsz *= 1024 * 1024;
		  }
		else if (strcmp (argv[stat], "-trfc") == 0)
		  {
		      if ((stat + 1) < argc)
			  trfc = VAL (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-trp") == 0)
		  {
		      if ((stat + 1) < argc)
			  trp = VAL (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-refresh") == 0)
		  {
		      if ((stat + 1) < argc)
			  refresh = atof (argv[++stat]);
		  }
		else if (strcmp (argv[stat], "-o") == 0)
		  {
		      strncpy (ofile, argv[++stat], 127);
		      ofile[127] = 0;
		  }
		else if (strcmp (argv[stat], "-ramsize") == 0)
		  {
		      if ((stat + 1) < argc)
			{
			    ramsize = (VAL (argv[++stat])) & 0x03ffff;
			    ramsize *= 1024;
			}
		  }
		else if (strcmp (argv[stat], "-romws") == 0)
		  {
		      if ((stat + 1) < argc)
			{
			    romws = (VAL (argv[++stat])) & 0xf;
			    romrws = romwws = romws;
			}
		  }
		else if (strcmp (argv[stat], "-romsize") == 0)
		  {
		      if ((stat + 1) < argc)
			{
			    romsize = (VAL (argv[++stat])) & 0x01ffff;
			    romsize *= 1024;
			}
		  }
		else if (strcmp (argv[stat], "-romwidth") == 0)
		  {
		      if ((stat + 1) < argc)
			  romwidth = (VAL (argv[++stat]));
		  }
		else if (strcmp (argv[stat], "-ramcs") == 0)
		  {
		      if ((stat + 1) < argc)
			{
			    rambanks = (VAL (argv[++stat])) & 0x0f;
			    if ((rambanks > 0) || (rambanks < 9))
				ramcs = rambanks;
			}
		  }
		else if (strcmp (argv[stat], "-entry") == 0)
		  {
		      if ((stat + 1) < argc)
			  entry = (VAL (argv[++stat])) & ~0x03;
		  }
		else if (strcmp (argv[stat], "-start") == 0)
		  {
		      if ((stat + 1) < argc)
			  startaddr = (VAL (argv[++stat])) & ~0x03;
		  }
		else if (strcmp (argv[stat], "-stack") == 0)
		  {
		      if ((stat + 1) < argc)
			  stack = (VAL (argv[++stat])) & ~0x01f;
		  }
		else if (strcmp (argv[stat], "-ramws") == 0)
		  {
		      if ((stat + 1) < argc)
			{
			    ramws = (VAL (argv[++stat])) & 0x3;
			    ramrws = ramwws = ramws;
			}
		  }
		else if (strcmp (argv[stat], "-ramrws") == 0)
		  {
		      if ((stat + 1) < argc)
			  ramrws = (VAL (argv[++stat])) & 0x3;
		  }
		else if (strcmp (argv[stat], "-ramwws") == 0)
		  {
		      if ((stat + 1) < argc)
			  ramwws = (VAL (argv[++stat])) & 0x3;
		  }
		else if (strcmp (argv[stat], "-ramwidth") == 0)
		  {
		      if ((stat + 1) < argc)
			  ramwidth = (VAL (argv[++stat]));
		  }
		else if (strcmp (argv[stat], "-rmw") == 0)
		  {
		      rmw = 1;
		  }
		else if (strcmp (argv[stat], "-uart") == 0)
		  {
		      if ((stat + 1) < argc)
			  uaddr = VAL (argv[++stat]);
		  }
		else
		  {
		      printf ("unknown option %s\n", argv[stat]);
		      usage ();
		      exit (1);
		  }
	    }
	  else
	    {
		if (secnum == 0)
		    strcpy (filename, argv[stat]);
		entry = elf_load (argv[stat]);
	    }
	  stat++;
      }
    printf ("creating LEON boot prom: %s\n", ofile);
    fprintf (dumpfile, "\n\t .text\n");
    fprintf (dumpfile, "\n\t.global filename\n");
    fprintf (dumpfile, "filename:\n");
    fprintf (dumpfile, "\t.string\t\"%s\"\n", filename);
    fprintf (dumpfile, "\n\t.align 32\n");
    fprintf (dumpfile, "\t.global sections\n");
    fprintf (dumpfile, "sections:\n");
    for (i = 0; i < secnum; i++)
      {
	  if (entry && (i == 0))
	      fprintf (dumpfile, "\t.word\t0x%x\n", entry);
	  else
	      fprintf (dumpfile, "\t.word\t0x%x\n", secarr[i].paddr);
	  fprintf (dumpfile, "\t.word\t_section%d\n", i);
	  fprintf (dumpfile, "\t.word\t0x%x\n", secarr[i].len);
	  fprintf (dumpfile, "\t.word\t0x%x\n", secarr[i].comp);
	  fprintf (dumpfile, "\t.string\t\"%s\"\n", secarr[i].name);
	  fprintf (dumpfile, "\n\t.align 32\n");
      }
    fprintf (dumpfile, "\t.word\t0\n");

    if (leon)
      {
	  fprintf (dumpfile,
		   "\n\t.global _memcfg1, _memcfg2, _memcfg3, _memcaddr, _uart, _dsuuart, _scaler,  _uaddr, _gptaddr, _dsuuart_addr \n");
      }
    fprintf (dumpfile, "\n\t.global ramsize, _stack, _entry\n");
    fprintf (dumpfile, "\t.global freq, configmsg, bmsg, noinit\n");
    fprintf (dumpfile, "freq:\n");
    fprintf (dumpfile, "\t.word\t%d\n", (int) (freq / 1000000));
    fprintf (dumpfile, "bmsg:\n");
    fprintf (dumpfile, "\t.word\t%d\n", bmsg);
    if (leon)
      {
	  switch (romwidth)
	    {
	    case 8:
		romwidth = 0;
		break;
	    case 16:
		romwidth = 1;
		break;
	    case 39:
		romwidth = 3;
		break;
	    default:
		romwidth = 2;
	    }
	  tmp = romsize;
	  tmp >>= 14;
	  i = 0;
	  while (tmp)
	    {
		i++;
		tmp >>= 1;
	    }
	  tmp = (i << 14) | romrws | (romwws << 4) | (romwidth << 8);
	  fprintf (dumpfile, "_memcfg1:\n");
	  fprintf (dumpfile, "\t.word\t0x%x\n", tmp);
	  tmp = ramsize / ramcs;
	  tmp >>= 14;
	  i = 0;
	  while (tmp)
	    {
		i++;
		tmp >>= 1;
	    }
	  tmp = (i << 9) | ramrws | (ramwws << 2);
	  switch (ramwidth)
	    {
	    case 8:
		ramwidth = 0;
		break;
	    case 16:
		ramwidth = 1;
		break;
	    case 39:
		ramwidth = 3;
		break;
	    default:
		ramwidth = 2;
	    }
	  tmp |= ramwidth << 4;
	  tmp |= rmw << 6;

	  i = 0;
	  if (sdramsz)
	    {
		tmp2 = sdramsz;
		tmp2 /= (sdrambanks * 8 * 1024 * 1024);
		while (tmp2)
		  {
		      tmp2 >>= 1;
		      i++;
		  }
		tmp |= 0x80184000;
	    }
	  tmp = (tmp & ~(3 << 21)) | (colsz << 21);
	  tmp = (tmp & ~(7 << 23)) | (i << 23);
	  tmp = (tmp & ~(1 << 26)) | (sdcas << 26);
	  if ((2.0E9 / freq) < (double) trp)
	      trp = 1;
	  else
	      trp = 0;
	  ftmp = ((double) trfc) - (3E9 / freq);
	  if (ftmp > 0)
	      trfc = 1 + (ftmp * freq) / 1E9;
	  else
	      trfc = 0;
	  if (trfc > 7)
	      trfc = 7;
	  tmp = (tmp & ~(7 << 27)) | (trfc << 27);
	  tmp = (tmp & ~(1 << 30)) | (trp << 30);
	  refr = (freq * refresh) / 1E6;
	  if (refr > 0x7fff)
	      refr = 0x7fff;
	  if (nosram)
	    {
		ramsize = sdramsz;
		tmp |= 1 << 13;
	    }
	  fprintf (dumpfile, "_memcfg2:\n");
	  fprintf (dumpfile, "\t.word\t0x%x\n", tmp);
	  tmp = (((10 * (int) freq) / (8 * baud)) - 5) / 10;
	  baud = freq / (8 * (tmp + 1));
	  fprintf (dumpfile, "_uart:\n");
	  fprintf (dumpfile, "\t.word\t0x%08x\n", tmp);
	  fprintf (dumpfile, "_uaddr:\n");
	  fprintf (dumpfile, "\t.word\t0x%08x\n", uaddr);
	  tmp = 0;
	  if (dsubaud)
	    {
		tmp = (((10 * (int) freq) / (8 * dsubaud)) + 5) / 10 - 1;
		dsubaud = freq / (8 * (tmp + 1));
	    }
	  fprintf (dumpfile, "_dsuuart:\n");
	  fprintf (dumpfile, "\t.word\t%d\n", tmp);
	  if (!stack)
	      stack = 0x40000000 + ramsize - 32;
	  tmp = 0;
	  if (sdramsz)
	      tmp = (refr << 12);
	  fprintf (dumpfile, "_memcfg3:\n");
	  fprintf (dumpfile, "\t.word\t0x%x\n", tmp);
	  fprintf (dumpfile, "noinit:\n");
	  fprintf (dumpfile, "\t.word\t%d\n", noinit);
      }

    if (starta)
	entry = starta;
    fprintf (dumpfile, "_entry:\n");
    fprintf (dumpfile, "\t.word\t0x%x\n", entry);
    fprintf (dumpfile, "ramsize:\n");
    fprintf (dumpfile, "\t.word\t0x%x\n", ramsize);
    fprintf (dumpfile, "_stack:\n");
    fprintf (dumpfile, "\t.word\t0x%x\n", stack);
    fprintf (dumpfile, "_memcaddr:\n");
    fprintf (dumpfile, "\t.word\t0x%x\n", memcaddr);
    fprintf (dumpfile, "_gptaddr:\n");
    fprintf (dumpfile, "\t.word\t0x%x\n", gptaddr);
    fprintf (dumpfile, "_dsuuart_addr:\n");
    fprintf (dumpfile, "\t.word\t0x%x\n", dsuuart_addr);

    sprintf (cmd, "  system clock   : %3.1f MHz\\n\\r", freq / 1E6);
    sprintf (msg, "  baud rate      : %d baud\\n\\r", baud);
    strcat (cmd, msg);
    sprintf (msg, "  prom           : %d K, (%d/%d) ws (r/w)\\n\\r",
	     romsize >> 10, romrws, romwws);
    strcat (cmd, msg);
    if (!nosram)
      {
	  sprintf (msg, "  sram           : %d K, %d bank(s),",
		   ramsize >> 10, rambanks);
	  strcat (cmd, msg);
	  sprintf (msg, " %d/", ramrws);
	  strcat (cmd, msg);
	  sprintf (msg, "%d ws (r/w)\\n\\r", ramwws);
	  strcat (cmd, msg);
      }
    else
      {
	  sprintf (msg,
		   "  sdram          : %d M, %d bank(s), %d-bit column\\n\\r",
		   sdramsz >> 10, sdrambanks, colsz + 8);
	  strcat (cmd, msg);
	  sprintf (msg, "  sdram          : ");
	  strcat (cmd, msg);
	  sprintf (msg,
		   "cas: %d, trp: %2.0f ns, trfc: %2.0f ns, refresh %3.1f us\\n\\r",
		   sdcas + 2, (double) (trp + 2) * 1E9 / freq,
		   (double) (trfc + 3) * 1E9 / freq,
		   (double) (refr + 1) * 1E6 / freq);
	  strcat (cmd, msg);
      }

    if (!flash)
      {
	  fprintf (dumpfile, "configmsg:\n");
	  fprintf (dumpfile, "\t.string\t\"%s\"\n\n\t.align 32\n", cmd);
      }
    fclose (dumpfile);
    sprintf (cmd,
	     "%s/bin/sparc-rtems-gcc -msoft-float -O2 -g -N -qprom -nostartfiles -Xlinker -Ttext -Xlinker 0x%x ",
	     TOOLBASE, startaddr);
    strcat (cmd, TOOLBASE);
    strcat (cmd, "/sparc-rtems/lib/promcore3.o dump.s ");
    if (bdinit)
	strcat (cmd, "bdinit.o ");
    strcat (cmd, "-lmkprom3 -o ");
    strcat (cmd, ofile);
    if (verbose)
	printf ("\n%s\n", cmd);
    system (cmd);
    if (!dump)
	system ("rm -f dump.s");
    exit (0);
}

usage ()
{
}

#define N   4096
#define F   18
#define THRESHOLD  2
#define NIL  N
#define MAGIC_NUMBER '\xaa'
#define EOP '\x55'
#ifndef SEEK_SET
#define SEEK_SET 0
#endif
#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif
#ifndef SEEK_END
#define SEEK_END 2
#endif

unsigned char text_buf[N + F - 1];
int match_position, match_length, lson[N + 1], rson[N + 257], dad[N + 1];
unsigned long textsize = 0, codesize = 0, printcount = 0;
unsigned char CHECKSUM;

typedef struct
{
    char MAGIC;
    unsigned char PARAMS;
    unsigned char CHECKSUM;
    unsigned char dummy;
    unsigned char ENCODED_SIZE[4];
    unsigned char DECODED_SIZE[4];
}
packet_header;

#define PH_SIZE 12

int
PutPacketInfo (buf)
     char *buf;
{
    packet_header PH;

    PH.MAGIC = MAGIC_NUMBER;
    PH.PARAMS = (unsigned char) (((N >> 6) & 0xf0) |
				 ((((F / 18) % 3) << 2) & 0x0c) | (THRESHOLD -
								   1));
    PH.CHECKSUM = CHECKSUM;
    PH.ENCODED_SIZE[0] = (codesize >> 24);
    PH.ENCODED_SIZE[1] = (codesize >> 16);
    PH.ENCODED_SIZE[2] = (codesize >> 8);
    PH.ENCODED_SIZE[3] = codesize;
    PH.DECODED_SIZE[0] = textsize >> 24;
    PH.DECODED_SIZE[1] = textsize >> 16;
    PH.DECODED_SIZE[2] = textsize >> 8;
    PH.DECODED_SIZE[3] = textsize;
    memcpy (buf, &PH, sizeof (packet_header));
    return 0;
}

void
InitTree (void)
{
    int i;

    for (i = N + 1; i <= N + 256; i++)
	rson[i] = NIL;
    for (i = 0; i < N; i++)
	dad[i] = NIL;
}

void
InsertNode (int r)
{
    int i, p, cmp;
    unsigned char *key;

    cmp = 1;
    key = &text_buf[r];
    p = N + 1 + key[0];
    rson[r] = lson[r] = NIL;
    match_length = 0;
    for (;;)
      {
	  if (cmp >= 0)
	    {
		if (rson[p] != NIL)
		    p = rson[p];
		else
		  {
		      rson[p] = r;
		      dad[r] = p;
		      return;
		  }
	    }
	  else
	    {
		if (lson[p] != NIL)
		    p = lson[p];
		else
		  {
		      lson[p] = r;
		      dad[r] = p;
		      return;
		  }
	    }
	  for (i = 1; i < F; i++)
	      if ((cmp = key[i] - text_buf[p + i]) != 0)
		  break;
	  if (i > match_length)
	    {
		match_position = p;
		if ((match_length = i) >= F)
		    break;
	    }
      }
    dad[r] = dad[p];
    lson[r] = lson[p];
    rson[r] = rson[p];
    dad[lson[p]] = r;
    dad[rson[p]] = r;
    if (rson[dad[p]] == p)
	rson[dad[p]] = r;
    else
	lson[dad[p]] = r;
    dad[p] = NIL;
}

void
DeleteNode (int p)
{
    int q;

    if (dad[p] == NIL)
	return;
    if (rson[p] == NIL)
	q = lson[p];
    else if (lson[p] == NIL)
	q = rson[p];
    else
      {
	  q = lson[p];
	  if (rson[q] != NIL)
	    {
		do
		  {
		      q = rson[q];
		  }
		while (rson[q] != NIL);
		rson[dad[q]] = lson[q];
		dad[lson[q]] = dad[q];
		lson[q] = lson[p];
		dad[lson[p]] = q;
	    }
	  rson[q] = rson[p];
	  dad[rson[p]] = q;
      }
    dad[q] = dad[p];
    if (rson[dad[p]] == p)
	rson[dad[p]] = q;
    else
	lson[dad[p]] = q;
    dad[p] = NIL;
}

int
Encode (inbuf, outbuf, buflen, oindex)
     unsigned char *inbuf;
     unsigned char *outbuf;
     int buflen, oindex;
{
    int i, c, len, r, s, last_match_length, code_buf_ptr;
    unsigned char code_buf[17], mask;

    int lindex = 0;

    CHECKSUM = 0xff;
    InitTree ();
    code_buf[0] = 0;
    code_buf_ptr = mask = 1;
    s = 0;
    r = N - F;
    for (i = s; i < r; i++)
	text_buf[i] = ' ';
    for (len = 0; len < F && (lindex < buflen); len++)
      {
	  c = inbuf[lindex++];
	  CHECKSUM ^= c;
	  text_buf[r + len] = c;
      }
    if ((textsize = len) == 0)
	return;
    for (i = 1; i <= F; i++)
	InsertNode (r - i);
    InsertNode (r);
    do
      {
	  if (match_length > len)
	      match_length = len;
	  if (match_length <= THRESHOLD)
	    {
		match_length = 1;
		code_buf[0] |= mask;
		code_buf[code_buf_ptr++] = text_buf[r];
	    }
	  else
	    {
		code_buf[code_buf_ptr++] = (unsigned char) match_position;
		code_buf[code_buf_ptr++] = (unsigned char)
		    (((match_position >> 4) & 0xf0)
		     | (match_length - (THRESHOLD + 1)));
	    }
	  if ((mask <<= 1) == 0)
	    {
		memcpy (&outbuf[oindex], code_buf, code_buf_ptr);
		oindex += code_buf_ptr;
		codesize += code_buf_ptr;
		code_buf[0] = 0;
		code_buf_ptr = mask = 1;
	    }
	  last_match_length = match_length;
	  for (i = 0; i < last_match_length && (lindex < buflen); i++)
	    {
		c = inbuf[lindex++];
		CHECKSUM ^= c;
		DeleteNode (s);
		text_buf[s] = c;
		if (s < F - 1)
		    text_buf[s + N] = c;
		s = (s + 1) & (N - 1);
		r = (r + 1) & (N - 1);
		InsertNode (r);
	    }
	  if ((textsize += i) > printcount)
	    {
		printcount += 1024;
	    }
	  while (i++ < last_match_length)
	    {
		DeleteNode (s);
		s = (s + 1) & (N - 1);
		r = (r + 1) & (N - 1);
		if (--len)
		    InsertNode (r);
	    }
      }
    while (len > 0);
    if (code_buf_ptr > 1)
      {
	  memcpy (&outbuf[oindex], code_buf, code_buf_ptr);
	  oindex += code_buf_ptr;
	  codesize += code_buf_ptr;
      }
    outbuf[oindex++] = EOP;
    if (verbose)
      {
	  printf ("Uncoded stream length: %ld bytes\n", textsize);
	  printf ("Coded stream length: %ld bytes\n", codesize);
	  printf ("Compression Ratio: %.3f\n", (double) textsize / codesize);
      }
}

int
lzss (inbuf, outbuf, len, comp)
     char *inbuf;
     char *outbuf;
     int len;
     int comp;
{
    int index;

    textsize = 0;
    codesize = 0;
    printcount = 0;

    if (comp)
      {
	  index = sizeof (packet_header);
	  Encode (inbuf, outbuf, len, index);
	  if (PutPacketInfo (outbuf))
	    {
		printf ("Error:couldn't write packet header\n");
	    }
      }
    return (codesize);
}

#include "ansidecl.h"

#ifdef ANSI_PROTOTYPES
#include <stdarg.h>
#else
#include <varargs.h>
#endif

dump (section_address, buffer, count)
     int section_address;
     unsigned char *buffer;
     int count;
{
    int i;

    for (i = 0; i < count; i += 4)
      {
	  fprintf (dumpfile, "\t.word\t0x%02x%02x%02x%02x\n",
		   buffer[i], buffer[i + 1], buffer[i + 2], buffer[i + 3]);
      }
}

int
elf_load (fname)
     char *fname;
{
    int cc, c, tmp;
    unsigned char buf[10];
    unsigned long entry;
    char *lzss_buf;
    FILE *xfile;

    xfile = fopen (fname, "r");

    if (xfile == NULL)
      {
	  printf ("open of %s failed\n", fname);
	  return (-1);
      }
    if (verbose)
	printf ("loading %s:\n", fname);
    tmp = ldelf (xfile, dumpfile);
    printf ("\n");
    return (tmp);
}

dumpsec (char *buf, int section_address, int section_size,
	 char *section_name, FILE * dumpfile)
{
    char *lzss_buf;
    if ((secnum == 0) && (section_address == 0x660))
      {
	  comp = 0;
	  flash = 1;
      }
    if (flash)
	fprintf (dumpfile, "\t .section .shdata2, \"ax\", @progbits\n");
    else
	fprintf (dumpfile, "\t .text\n");
    secarr[secnum].paddr = section_address;
    secarr[secnum].len = section_size;
    secarr[secnum].comp = comp;
    strcpy (secarr[secnum].name, section_name);

    fprintf (dumpfile, "\n\t.global _section%1d\n", secnum);
    fprintf (dumpfile, "_section%1d:\n", secnum);

    if (comp)
      {
	  lzss_buf = (char *) malloc (section_size + section_size / 2 + 256);
      }
    secnum++;
    if (comp)
      {
	  section_size = lzss (buf, lzss_buf, section_size, 1);
	  dump (section_address, lzss_buf, section_size + 13);
	  free (buf);
	  free (lzss_buf);
      }
    else
      {
	  dump (section_address, buf, section_size);
	  free (buf);
      }

    return (0);
}

#include <netinet/in.h>

#define EI_NIDENT       16

typedef unsigned int Elf32_Addr;
typedef unsigned int Elf32_Word;
typedef unsigned int Elf32_Off;
typedef unsigned short Elf32_Half;

typedef struct
{
    unsigned char e_ident[EI_NIDENT];
    Elf32_Half e_type;
    Elf32_Half e_machine;
    Elf32_Word e_version;
    Elf32_Addr e_entry;
    Elf32_Off e_phoff;
    Elf32_Off e_shoff;
    Elf32_Word e_flags;
    Elf32_Half e_ehsize;
    Elf32_Half e_phentsize;
    Elf32_Half e_phnum;
    Elf32_Half e_shentsize;
    Elf32_Half e_shnum;
    Elf32_Half e_shstrndx;
} Elf32_Ehdr;

#define EI_MAG0	0
#define EI_MAG1	1
#define EI_MAG2	2
#define EI_MAG3	3
#define EM_SPARC 2

typedef struct
{
    Elf32_Word sh_name;
    Elf32_Word sh_type;
    Elf32_Word sh_flags;
    Elf32_Addr sh_addr;
    Elf32_Off sh_offset;
    Elf32_Word sh_size;
    Elf32_Word sh_link;
    Elf32_Word sh_info;
    Elf32_Word sh_addralign;
    Elf32_Word sh_entsize;
} Elf32_Shdr;

typedef struct
{
    Elf32_Word p_type;
    Elf32_Off p_offset;
    Elf32_Addr p_vaddr;
    Elf32_Addr p_paddr;
    Elf32_Word p_filesz;
    Elf32_Word p_memsz;
    Elf32_Word p_flags;
    Elf32_Word p_align;
} Elf32_Phdr;

ldelf (FILE * fp, FILE * dumpfile)
{
    Elf32_Ehdr fh;
    Elf32_Shdr sh, ssh;
    Elf32_Phdr ph;
    char *strtab;
    char *mem;
    unsigned int *memw, i, j;

    fseek (fp, 0, SEEK_SET);
    if (fread (&fh, sizeof (fh), 1, fp) != 1)
      {
	  return (-2);
      }

    if ((fh.e_ident[EI_MAG0] != 0x7f)
	|| (fh.e_ident[EI_MAG1] != 'E')
	|| (fh.e_ident[EI_MAG2] != 'L') || (fh.e_ident[EI_MAG3] != 'F'))
      {
	  return (-2);
      }
    fh.e_machine = ntohs (fh.e_machine);
    if (fh.e_machine != EM_SPARC)
      {
	  printf ("not a SPARC executable (%d)\n", fh.e_machine);
	  return (-2);
      }

    fh.e_entry = ntohl (fh.e_entry);
    fh.e_shoff = ntohl (fh.e_shoff);
    fh.e_phoff = ntohl (fh.e_phoff);
    fh.e_phnum = ntohs (fh.e_phnum);
    fh.e_shnum = ntohs (fh.e_shnum);
    fh.e_phentsize = ntohs (fh.e_phentsize);
    fh.e_shentsize = ntohs (fh.e_shentsize);
    fh.e_shstrndx = ntohs (fh.e_shstrndx);
    fseek (fp, fh.e_shoff + ((fh.e_shstrndx) * fh.e_shentsize), SEEK_SET);
    if (fread (&ssh, sizeof (ssh), 1, fp) != 1)
      {
	  printf ("file read error\n");
	  return (-1);
      }
    ssh.sh_name = ntohl (ssh.sh_name);
    ssh.sh_type = ntohl (ssh.sh_type);
    ssh.sh_offset = ntohl (ssh.sh_offset);
    ssh.sh_size = ntohl (ssh.sh_size);
    strtab = (char *) malloc (ssh.sh_size);
    fseek (fp, ssh.sh_offset, SEEK_SET);
    if (fread (strtab, ssh.sh_size, 1, fp) != 1)
      {
	  printf ("file read error\n");
	  return (-1);
      }

    for (i = 1; i < fh.e_shnum; i++)
      {
	  fseek (fp, fh.e_shoff + (i * fh.e_shentsize), SEEK_SET);
	  if (fread (&sh, sizeof (sh), 1, fp) != 1)
	    {
		printf ("file read error\n");
		return (-1);
	    }
	  sh.sh_name = ntohl (sh.sh_name);
	  sh.sh_addr = ntohl (sh.sh_addr);
	  sh.sh_size = ntohl (sh.sh_size);
	  sh.sh_type = ntohl (sh.sh_type);
	  sh.sh_offset = ntohl (sh.sh_offset);
	  sh.sh_flags = ntohl (sh.sh_flags);
	  if ((sh.sh_type == 1) && (sh.sh_size > 0) && (sh.sh_flags & 2))
	    {
		printf ("section: %s at 0x%x, size %d bytes\n",
			&strtab[sh.sh_name], sh.sh_addr, sh.sh_size);
		mem = (char *) malloc (sh.sh_size);
		if (mem != (char *) -1)
		  {
		      if (sh.sh_type == 1)
			{
			    fseek (fp, sh.sh_offset, SEEK_SET);
			    fread (mem, sh.sh_size, 1, fp);
			    memw = (unsigned int *) mem;
			    dumpsec (mem, sh.sh_addr, sh.sh_size,
				     &strtab[sh.sh_name], dumpfile);
			}
		  }
		else
		  {
		      printf ("load address outside physical memory\n");
		      printf ("load aborted\n");
		      return (-1);
		  }
	    }
      }

    free (strtab);
    return (fh.e_entry);
}
