/*
 *  AMBA Plag & Play Bus Driver Macros
 *
 *  Macros used for AMBA Plug & Play bus scanning
 *
 *  COPYRIGHT (c) 2004.
 *  Gaisler Research
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 *
 *  $Id: amba.h,v 1.1.2.2 2006/04/24 16:58:41 joel Exp $
 */

#ifndef __AMBA_H__
#define __AMBA_H__

#include <ambappdevs.h>

#ifdef __cplusplus
extern "C" {
#endif

#define LEON3_IO_AREA 0xfff00000
#define LEON3_CONF_AREA 0xff000
#define LEON3_AHB_SLAVE_CONF_AREA (1 << 11)

#define LEON3_AHB_CONF_WORDS 8
#define LEON3_APB_CONF_WORDS 2
#define LEON3_AHB_MASTERS 64
#define LEON3_AHB_SLAVES 64
#define LEON3_APB_SLAVES 16
#define LEON3_APBUARTS 8

/* 
 *
 * Macros for manipulating Configuration registers  
 *
 */

#define amba_get_confword(tab, index, word) (*((tab).addr[(index)]+(word)))

#define amba_vendor(x) (((x) >> 24) & 0xff)

#define amba_device(x) (((x) >> 12) & 0xfff)

#define amba_ahb_get_membar(tab, index, nr) (*((tab).addr[(index)]+4+(nr)))

#define amba_apb_get_membar(tab, index) (*((tab).addr[(index)]+1))

#define amba_membar_start(mbar) (((mbar) & 0xfff00000) & (((mbar) & 0xfff0) << 16))

#define amba_iobar_start(base, iobar) ((base) | ((((iobar) & 0xfff00000)>>12) & (((iobar) & 0xfff0)<<4)) )

#define amba_irq(conf) ((conf) & 0x1f)

#define amba_ver(conf) (((conf)>>5) & 0x1f)

#define amba_membar_type(mbar) ((mbar) & 0xf)

#define AMBA_TYPE_AHBIO_ADDR(addr,base_ioarea) ((unsigned int)(base_ioarea) | ((addr) >> 12))

#define AMBA_TYPE_APBIO 0x1
#define AMBA_TYPE_MEM   0x2
#define AMBA_TYPE_AHBIO 0x3

/*
 *  Types and structure used for AMBA Plug & Play bus scanning 
 *
 */

typedef struct {
  int            devnr;           /* numbrer of devices on AHB or APB bus */
  unsigned int   *addr[16];       /* addresses to the devices configuration tables */
} amba_device_table;

typedef struct amba_confarea_type {
  amba_device_table ahbmst;
  amba_device_table ahbslv;
  amba_device_table apbslv;
  unsigned int      apbmst;
	unsigned int      ioarea;
} amba_confarea_type;

extern amba_confarea_type amba_conf;
  /* extern amba_device_table amba_ahb_masters; 
extern amba_device_table amba_ahb_slaves;  
extern amba_device_table amba_apb_slaves;  */
extern unsigned int amba_apbmst_base;    /* APB master base address */

typedef struct {
	unsigned int start, irq, bus_id;
} amba_apb_device;

typedef struct {
	unsigned int start[4], irq, ver;
} amba_ahb_device;

/***** APB SLAVES *****/

/* Return number of APB Slave devices which has given vendor and device */
int amba_get_number_apbslv_devices(amba_confarea_type *amba_conf, int vendor, int device);

/* Get First APB Slave device of this vendor&device id */
int amba_find_apbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_apb_device *dev);

/* Get APB Slave device of this vendor&device id. (setting nr to 0 is eqivalent to calling amba_find_apbslv() ) */
int amba_find_next_apbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_apb_device *dev, int index);

/* Get first nr APB Slave devices, put them into dev (which is an array of nr length) */
int amba_find_apbslvs(amba_confarea_type *amba_conf, int vendor, int device, amba_apb_device *devs, int maxno);

/***** AHB SLAVES *****/

/* Return number of AHB Slave devices which has given vendor and device */
int amba_get_number_ahbslv_devices(amba_confarea_type *amba_conf, int vendor, int device);

/* Get First AHB Slave device of this vendor&device id */
int amba_find_ahbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_ahb_device *dev);

/* Get AHB Slave device of this vendor&device id. (setting nr to 0 is eqivalent to calling amba_find_ahbslv() ) */
int amba_find_next_ahbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_ahb_device *dev, int index);

/* Get first nr AHB Slave devices, put them into dev (which is an array of nr length) */
int amba_find_ahbslvs(amba_confarea_type *amba_conf, int vendor, int device, amba_ahb_device *devs, int maxno);

#ifdef __cplusplus
}
#endif

#endif /* __AMBA_H__ */

