/*
 * RTEMS SpaceWire Time Distribution Protocol (SPWTDP) core example. The
 * example communicates with a GRLIB target including a GRSPW2 and SPWTDP
 * core using RMAP, TimeCodes and SpW Interrupts. The local SPWTDP is operated
 * in initiator mode and the remote SPWTDP in target mode. This software
 * demonstrates to following:
 *  - Initiator SPWTDP initialization using SPWTDP driver
 *  - Initiator GRSPW2 initialization using GRSPW_PKT driver
 *  - Initiator GRSPW2 link control and DMA handling using GRSPW_PKT driver
 *  - RMAP communication to Target using GRSPW_PKT driver and RMAP-ASYNC stack
 *  - Target SPWTDP initialization over RMAP
 *  - Target GRSPW2 initialization over RMAP
 *  - Basic Elapsed Time format handling
 *  - Rough startup synchonization between Initiator RTEMS System Clock
 *    and SPWTDP
 *  - Latency Calculation
 *
 * The local and remote hardware required:
 *  - GRSPW2 with Distributed Interrupt Support
 *  - SPWTDP connected to GRSPW2
 *  - Initiator SpW0 connector must be connected to SpW1 connector of Target
 *  - SPWTDP of Initiator and Target use the same configuration
 *  - Maximum 64-bits fine time resolution
 *  - Latency value calculated is less than two octet of fine time
 *
 * The application consists of four threads:
 *
 *  INIT. Initialization and start task. Make basic initialization and
 *        creates the below tasks:
 *
 *  TA01. Example Task
 *
 *  TA02. Link monitor task. Prints out whenever a SpaceWire link switch
 *        from run-state to any other state or vice versa.
 *
 *  TA03. SpaceWire DMA task. Handles reception and transmission of SpaceWire
 *        packets on all SpaceWire devices.
 *
 * Other RTEMS objects:
 *
 *  SEM0. DMA semaphore to protect DMA RX and TX packet lists. The DMA task
 *        responds to changes and handles the DMA RX/TX operations that the
 *        example task TA01 makes. The TA01 task generates RMAP requests into
 *        DMA TX queue and processes RX queue for incomming RMAP responses.
 *
 */

/****** CONFIGURATION OF SPWTDP EXAMPLE *****
 *
 * Remote SpW-TDP Settings:
 */
#define R_SPWTDP_DSTADR 0xfe		/* SpW Destination address */
#define R_SPWTDP_DSTKEY 0x00		/* SpW Destination Key */
#define R_SPWTDP_BASEADR 0x80100700	/* Base address of SPWTDP on Remote */
#define R_SPWTDP_SPWDEV  1		/* SpaceWire device connector */
#define R_SPWTDP_MAPPING 6		/* Mapping of Time-Code to T-Field */
#define R_SPW_BASEADR    0x80000700	/* SpaceWire device 1 base address */

/* GRSPW device used to communicate with target. Note only SpW0 tested. */
#define DEV_IDX 0
#define RMAP_SPW_DEV_IDX DEV_IDX

/* Number of latency calculations */
#define LATENCY_COUNT 128

/* READ/WRITE Target SPWTDP over RMAP end demonstration */
#undef TOGGLE_TGT_ME

/* Enable to debug SpW Traffic */
#ifdef DEBUG_SPW
 #define DEBUG_SPW_TX
 #define DEBUG_SPW_RX
#endif

/* Add print outs for latency calculation. Warning! this might make the
 * calculation invlaid since printf() may take too long time. Works using a
 * Ethernet debuglink and doing -u UART tunneling with GRMON2.
 */
#undef DEBUG_LATENCY

/* Debug option to see when the Datation ET are sampled localy for Latency
 * calculation. A GPTIMER timer incatnce is programmed and used for the
 * sampling.
 */
#undef DEBUG_ET_SAMPLING_TIMES
/* Base address of GPTIMER timer used to sample time. Could use
 * GPTIMER->timer[2] to be more Plug&Play able */
#define TIMER_BASE 0x80000320

/* Define INCLUDE_PCI_DRIVERS to include PCI host and PCI board drivers with
 * SpaceWire interfaces
 */
/*#define INCLUDE_PCI_DRIVERS*/

#include <rtems.h>

/* configuration information */

#define CONFIGURE_INIT

#include <bsp.h> /* for device driver prototypes */

rtems_task Init( rtems_task_argument argument);	/* forward declaration needed */

/* configuration information */

#define CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
#define CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER

/* Set some values if someone should modify the example. The shared IRQ layer
 * need one semaphore.
 */
#define CONFIGURE_MAXIMUM_TASKS             8
#define CONFIGURE_MAXIMUM_SEMAPHORES        20
#define CONFIGURE_MAXIMUM_MESSAGE_QUEUES    20
#define CONFIGURE_LIBIO_MAXIMUM_FILE_DESCRIPTORS 32
#define CONFIGURE_MAXIMUM_DRIVERS 32
#define CONFIGURE_MAXIMUM_PERIODS             1

#define CONFIGURE_RTEMS_INIT_TASKS_TABLE
#define CONFIGURE_INIT_TASK_ATTRIBUTES    RTEMS_DEFAULT_ATTRIBUTES | RTEMS_FLOATING_POINT
#define CONFIGURE_EXTRA_TASK_STACKS         (40 * RTEMS_MINIMUM_STACK_SIZE)
#define CONFIGURE_MICROSECONDS_PER_TICK     RTEMS_MILLISECONDS_TO_MICROSECONDS(2)


/* Configure Driver manager */
#if defined(RTEMS_DRVMGR_STARTUP) && defined(LEON3) /* if --drvmgr was given to configure */
 /* Add Timer and UART Driver for this example */
 #ifdef CONFIGURE_APPLICATION_NEEDS_CLOCK_DRIVER
  #define CONFIGURE_DRIVER_AMBAPP_GAISLER_GPTIMER
 #endif
 #ifdef CONFIGURE_APPLICATION_NEEDS_CONSOLE_DRIVER
  #define CONFIGURE_DRIVER_AMBAPP_GAISLER_APBUART
 #endif
#endif

#define CONFIGURE_DRIVER_AMBAPP_GAISLER_GRSPW2     /* SpaceWire Packet driver */
#define CONFIGURE_DRIVER_AMBAPP_GAISLER_SPWTDP     /* SpaceWire TDP driver */

#ifdef INCLUDE_PCI_DRIVERS
/* Configure PCI Library to auto configuration. This can be substituted with
 * a static configuration by setting PCI_LIB_STATIC, see pci/. Static
 * configuration can be generated automatically by print routines in PCI
 * library.
 */
#define RTEMS_PCI_CONFIG_LIB
/*#define CONFIGURE_PCI_LIB PCI_LIB_STATIC*/
#define CONFIGURE_PCI_LIB PCI_LIB_AUTO

/*#define CONFIGURE_DRIVER_AMBAPP_GAISLER_PCIF*//* GRLIB PCIF Host driver  */
#define CONFIGURE_DRIVER_AMBAPP_GAISLER_GRPCI   /* GRPCI Host driver */
#define CONFIGURE_DRIVER_AMBAPP_GAISLER_GRPCI2  /* GRPCI2 Host Driver */
#define CONFIGURE_DRIVER_PCI_GR_RASTA_IO        /* GR-RASTA-IO PCI Target Driver */
#define CONFIGURE_DRIVER_PCI_GR_RASTA_TMTC      /* GR-RASTA-TMTC PCI Target Driver */
#endif

/*******************************************/

#ifdef LEON2
 /* PCI support for AT697 */
 #define CONFIGURE_DRIVER_LEON2_AT697PCI
 /* AMBA PnP Support for GRLIB-LEON2 */
 /*#define CONFIGURE_DRIVER_LEON2_AMBAPP*/
#endif

#include <rtems/confdefs.h>
#include <drvmgr/drvmgr_confdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

/* no network */
#undef ENABLE_NETWORK
#undef ENABLE_NETWORK_SMC_LEON3

#include "../../config.c"

rtems_task test_app(rtems_task_argument ignored);
rtems_id tid, tid_link, tid_dma;
rtems_id dma_sem;

rtems_task Init(
  rtems_task_argument ignored
)
{
	/* Initialize Driver manager and Networking, in config.c */
	system_init();

	/* Print device topology */
	/*drvmgr_print_topo();*/
	rtems_task_wake_after(4);

	/* Create Tasks for example application */
	rtems_task_create(
			rtems_build_name( 'T', 'A', '0', '1' ),
			10, RTEMS_MINIMUM_STACK_SIZE * 10, RTEMS_DEFAULT_MODES,
			RTEMS_FLOATING_POINT, &tid);
	rtems_task_create(
			rtems_build_name( 'T', 'A', '0', '2' ),
			10, RTEMS_MINIMUM_STACK_SIZE * 10, RTEMS_DEFAULT_MODES,
			RTEMS_FLOATING_POINT, &tid_link);
	rtems_task_create(
			rtems_build_name( 'T', 'A', '0', '3' ),
			10, RTEMS_MINIMUM_STACK_SIZE * 10, RTEMS_DEFAULT_MODES,
			RTEMS_FLOATING_POINT, &tid_dma);

	/* Device Semaphore created with count = 1 */
	if (rtems_semaphore_create(rtems_build_name('S', 'E', 'M', '0'), 1,
	  RTEMS_FIFO | RTEMS_SIMPLE_BINARY_SEMAPHORE | \
	  RTEMS_NO_INHERIT_PRIORITY | RTEMS_LOCAL | \
	  RTEMS_NO_PRIORITY_CEILING, 0, &dma_sem) != RTEMS_SUCCESSFUL) {
	  printf("Failed creating Semaphore\n");
	  exit(0);
	}

	rtems_task_start(tid, test_app, 0);
	rtems_task_suspend(RTEMS_SELF);
}

#include <grspw_pkt.h>
#include "grspw_pkt_lib.h"

#include <rmap.h>


int nospw = 0;
int tasks_stop = 0;
rtems_task link_ctrl_task(rtems_task_argument unused);
rtems_task dma_task(rtems_task_argument unused);

#define PKT_SIZE 256

struct grspw_device {
	/* GRSPW Device layout - must be the same as 'struct grspw_dev' */
	void *dh;
	void *dma[4];
	int index;
	struct grspw_hw_sup hwsup;

	/* Test structures */
	struct grspw_config cfg;
	int run;

	/* RX and TX lists with packet buffers */
	struct grspw_list rx_list, rx_buf_list, tx_list, tx_buf_list;
	int rx_list_cnt, rx_buf_list_cnt, tx_list_cnt, tx_buf_list_cnt;
};
#define DEV(device) ((struct grspw_dev *)(device))

#define DEVS_MAX 32
static struct grspw_device devs[DEVS_MAX];

struct grspw_config dev_def_cfg =
{
		.adrcfg =
		{
			.promiscuous = 1, /* Detect all packets */
			.def_addr = 32, /* updated bu dev_init() */
			.def_mask = 0,
			.dma_nacfg =
			{
				/* Since only one DMA Channel is used, only
				 * the default Address|Mask is used.
				 */
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
				{
					.node_en = 0,
					.node_addr = 0,
					.node_mask = 0,
				},
			},
		},
		.rmap_cfg = 0,		/* Disable RMAP */
		.rmap_dstkey = 0,	/* No RMAP DESTKEY needed when disabled */
		.tc_cfg = TCOPTS_EN_TX,/* Enable TimeCode */
		.tc_isr_callback = NULL,/* TimeCode ISR */
		.tc_isr_arg = NULL,	/* No TimeCode ISR Argument */
		.enable_chan_mask = 1,	/* Enable only the first DMA Channel */
		.chan =
		{
			{
				.flags = DMAFLAG_NO_SPILL,
				.rxmaxlen = PKT_SIZE,
				.rx_irq_en_cnt = 0, /* Disable RX IRQ generation */
				.tx_irq_en_cnt = 0, /* Disable TX IRQ generation */
			},
			/* The other 3 DMA Channels are unused */

		},
		.iccfg =
		{
			.tomask = 0,
			.aamask = 0,
			.scaler = 0,
			.isr_reload = (1 << 31),
			.ack_reload = (1 << 31),
		},
};

/* SpaceWire Routing table entry */
struct route_entry {
	unsigned char dstadr[16];	/* 0 terminates array */
};

/* SpaceWire packet payload (data) content layout */
struct pkt_hdr {
	unsigned char addr;
	unsigned char protid;
	unsigned int data[(PKT_SIZE-2)/4];
};

struct spwpkt {
	struct grspw_pkt p;
	unsigned long long data[PKT_SIZE/8+1]; /* 32 bytes of data - 2byte data-header (8 extra bytes to avoid truncated bad packets)*/
	unsigned long long hdr[256/8+2]; /* Max header size and extra room for RMAP stack */
};

/* All packet buffers used by application */
struct spwpkt pkts[DEVS_MAX][16];

void init_pkts(void)
{
	struct spwpkt *pkt;
	int i, j;

	memset(&pkts[0][0], 0, sizeof(pkts));

	for (i = 0; i < DEVS_MAX; i++) {
		grspw_list_clr(&devs[i].rx_list);
		grspw_list_clr(&devs[i].tx_list);
		grspw_list_clr(&devs[i].tx_buf_list);
		devs[i].rx_list_cnt = 0;
		devs[i].tx_list_cnt = 0;
		devs[i].rx_buf_list_cnt = 0;
		devs[i].tx_buf_list_cnt = 0;
		for (j = 0, pkt = &pkts[i][0]; j < 16; j++, pkt = &pkts[i][j]) {
			pkt->p.pkt_id = (i << 8)+ j; /* unused */
			pkt->p.data = &pkt->data[0];
			pkt->p.hdr = &pkt->hdr[0];
			if (j < 8) {
				/* RX buffer */

				/* Add to device RX list */
				grspw_list_append(&devs[i].rx_buf_list, &pkt->p);
				devs[i].rx_buf_list_cnt++;
			} else {
				/* TX buffer */
				pkt->p.dlen = 0;
				pkt->p.hlen = 0;

				/* Add to device TX list */
				grspw_list_append(&devs[i].tx_buf_list, &pkt->p);
				devs[i].tx_buf_list_cnt++;
			}
		}
	}
}

int dev_init(int idx)
{
	struct grspw_device *dev = &devs[idx];
	int i, ctrl, clkdiv, tc, stscfg;
	unsigned int icctrl;

	printf(" Initializing SpaceWire device %d\n", idx);

	memset(dev, 0, sizeof(struct grspw_device));

	dev->index = idx;
	dev->dh = grspw_open(idx);
	if (dev->dh == NULL) {
		printf("Failed to open GRSPW device %d\n", idx);
		return -1;
	}
	grspw_hw_support(dev->dh, &dev->hwsup);
#ifdef PRINT_GRSPW_RESET_CFG
	grspw_config_read(DEV(dev), &dev->cfg);
	printf("\n\n---- DEFAULT CONFIGURATION FROM DRIVER/HARDWARE ----\n");
	grspw_cfg_print(&dev->hwsup, &dev->cfg);
#endif
	dev->cfg = dev_def_cfg;
	dev->cfg.adrcfg.def_addr = 32 + idx;
	dev->cfg.tc_isr_arg = dev;
	tc = TCOPTS_EN_TX | TCOPTS_EN_RX | TCOPTS_EN_RXIRQ;
	grspw_tc_ctrl(dev->dh, &tc);

	if (grspw_cfg_set(DEV(dev), &dev->cfg)) {
		grspw_close(dev->dh);
		return -1;
	}
#ifdef PRINT_GRSPW_RESET_CFG
	printf("\n\n---- APPLICATION CONFIGURATION ----\n");
	grspw_cfg_print(&dev->hwsup, &dev->cfg);
	printf("\n\n");
#endif

	/* Make sure the required SpaceWire Interrupt support is available */
	if (dev->hwsup.irq == 0) {
		printf(" dev(%d) does not have SPW-IRQ support!\n", dev->index);
		return -1;
	}

	/* Enable CTRL.TF to make sure that GRSPW separate Time Codes from
	 * SpaceWire interrupt-codes.*/

	icctrl = -1;
	//grspw_ic_ctrl(DEV(dev), &icctrl);
	icctrl |= ICOPTS_EN_FLAGFILTER;
	//grspw_ic_ctrl(DEV(dev), &icctrl);

	/* This will result in an error if only one port available */
	if (dev->hwsup.nports < 2) {
		int port = 1;
		if ( grspw_port_ctrl(dev->dh, &port) == 0 ) {
			printf("Succeeded to select port1, however only one PORT on dev %d!\n", dev->index);
			return -1;
		}
	}

	/* Try to bring link up at fastest clockdiv but do not touch
	 * start-up clockdivisor */
	clkdiv = -1;
	grspw_link_ctrl(dev->dh, NULL, NULL, &clkdiv);
	ctrl = LINKOPTS_ENABLE | LINKOPTS_AUTOSTART | LINKOPTS_START;
	clkdiv &= 0xff00;
	clkdiv |= 0x13;
	stscfg = LINKSTS_MASK;
	grspw_link_ctrl(dev->dh, &ctrl, &stscfg, &clkdiv);

	if ( (dev->hwsup.hw_version >> 16) == GAISLER_SPW2_DMA )
		printf(" NOTE: running on SPW-ROUTER DMA SpaceWire link (no link-state available)\n");
	else
		printf(" After Link Start: %d\n", (int)grspw_link_state(dev->dh));
	dev->run = 0;

	grspw_stats_clr(dev->dh);

	for (i=0; i<dev->hwsup.ndma_chans; i++) {
		if (dev->dma[i])
			grspw_dma_stats_clr(dev->dma[i]);
	}

	grspw_list_clr(&dev->rx_list);
	grspw_list_clr(&dev->rx_buf_list);
	grspw_list_clr(&dev->tx_list);
	grspw_list_clr(&dev->tx_buf_list);
	dev->rx_list_cnt = dev->rx_buf_list_cnt = 0;
	dev->tx_list_cnt = dev->tx_buf_list_cnt = 0;

	return 0;
}

int dev_check_started(int idx)
{
	struct grspw_device *dev = &devs[idx];
	if (grspw_link_state(dev->dh) != SPW_LS_RUN)
		return 0;
	return 1;
}

void dev_enable_spwirq(int idx)
{
	struct grspw_device *dev = &devs[idx];
	unsigned int icctrl;
	icctrl = -1;
	grspw_ic_ctrl(dev->dh, &icctrl);
	icctrl |= ICOPTS_EN_RX | ICOPTS_EN_TX;
	grspw_ic_ctrl(dev->dh, &icctrl);
}

int dev_dma_close_all(int idx)
{
	struct grspw_device *dev = &devs[idx];
	int i, rc;
	for (i=0; i<dev->hwsup.ndma_chans; i++) {
		if (dev->dma[i]) {
			rc = grspw_dma_close(dev->dma[i]);
			if (rc)
				return rc;
			dev->dma[i] = NULL;
		}
	}
	return 0;
}

void dev_cleanup(int idx)
{
	struct grspw_device *dev = &devs[idx];

	if (dev->dh == NULL)
		return;

	/* Stop all DMA activity first */
	grspw_stop(DEV(dev));
	/* wait for other tasks to be thrown out from driver */
	rtems_task_wake_after(4);

	/* close all DMA channels */
	if (dev_dma_close_all(idx)) {
		printf("FAILED to close GRSPW%d DMA\n", idx);
	}

	if (grspw_close(dev->dh)) {
		printf("FAILED to close GRSPW%d\n", idx);
	}
	dev->dh = NULL;
}

void pktlist_set_flags(struct grspw_list *lst, unsigned short flags)
{
	struct grspw_pkt *pkt = lst->head;
	while (pkt) {
		pkt->flags = flags;
		pkt = pkt->next;
	}
}

int dma_process(struct grspw_device *dev);

struct grspw_device *rmap_dev;
void *rmap_stack;

#include <spwtdp.h>
struct spwtdp_regs spwtdp;
int remote_fine, remote_coarse;

int remote_spwtdp_init(void);
int remote_spwtdp_reset(void);
int remote_spwtdp_read(unsigned int ofs, int len);
int remote_spwtdp_write(unsigned int ofs, int len);


int parse_precision(unsigned short preamble, int *coarse, int *fine)
{
	int coarse_precision, fine_precision;

	if (preamble & 0x80) {
		puts("Pfield second extension set: unknown format");
		return -1;
	}
	if (!((preamble & 0x7000) == 0x2000 || (preamble & 0x7000) == 0x1000)) {
		puts(" PField indicates not unsegmented code: unknown format");
		return -1;
	}
	/*
	coarse_precision = 32;
	fine_precision = 24;
	*/
	coarse_precision = ((preamble >> 10) & 0x3) + 1;
	if (preamble & 0x80)
		coarse_precision += (preamble >> 5) & 0x3;
	fine_precision = (preamble >> 8) & 0x3;
	if (preamble & 0x80)
		fine_precision += (preamble >> 2) & 0x7;
	if (coarse)
		*coarse = coarse_precision;
	if (fine)
		*fine = fine_precision;
	return 0;
}

int remote_spwtdp_init(void)
{
	unsigned short preamble;

	memset(&spwtdp, 0, sizeof(spwtdp));

	if (remote_spwtdp_reset())
		return -1;
	if (remote_spwtdp_read(0, sizeof(spwtdp)))
		return -2;

	preamble = spwtdp.dat_ctrl & 0xffff;
	if (parse_precision(preamble, &remote_coarse, &remote_fine)) {
		puts("failed parsing PField of remote SPWTDP");
		return -3;
	}
	printf("Target Coarse precision: %d bits\n", remote_coarse*8);
	printf("Target Fine precision:   %d bits\n", remote_fine*8);

	/* Initialize basic initialization */
	spwtdp.conf[0] =
		(1 << 24) |	                /* Jitter Correction */
		(R_SPWTDP_MAPPING << 8) |	/* Mapping */
		(1 << 7) |			/* TD=1 */
		(R_SPWTDP_SPWDEV << 4) |	/* SpW device */
		(1 << 3) |			/* Mitigation Enable */
		(1 << 2) |			/* Enable RX for Target */
		(0 << 0);			/* Out of Reset */

	if (remote_spwtdp_write(0, sizeof(spwtdp.conf[0])))
		return -4;

	return 0;
}

/* Reset core and get out of reset */
int remote_spwtdp_reset()
{
	spwtdp.conf[0] = 1; /* Reset */
	if (remote_spwtdp_write(0, sizeof(spwtdp.conf[0])))
		return -1;

	spwtdp.conf[0] = 0; /* Stop Reset */
	if (remote_spwtdp_write(0, sizeof(spwtdp.conf[0])))
		return -2;

	return 0;
}

struct grspw_pkt *dma_alloc_tx_buf(struct grspw_device *dev)
{
	struct grspw_pkt *pkt;

	/* Allocate SpW packet buffer to hold RMAP request */
	rtems_semaphore_obtain(dma_sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
	pkt = dev->tx_buf_list.head;
	if (pkt == NULL) /* no free packets left */
		goto out;
	dev->tx_buf_list.head = pkt->next;
	if (pkt->next == NULL)
		dev->tx_buf_list.tail = NULL;
	dev->tx_buf_list_cnt--;

out:
	rtems_semaphore_release(dma_sem);
	return pkt;
}

int dma_sched_tx_pkt(struct grspw_device *dev, struct grspw_pkt *pkt)
{
	/* Put TX packet into transmission queue */
	rtems_semaphore_obtain(dma_sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
	grspw_list_append(&dev->tx_list, pkt);
	dev->tx_list_cnt++;
	rtems_semaphore_release(dma_sem);

	return 0;
}

/* GRMON triggering point, set breakpoint here */
int grmon_clk_bp1(int v0, int v1)
{
	return v0+v1; /* just do something */
}

/* Sync local cache copy of remote registers by reading from target */
int rmap_tgt_read(unsigned int address, void *data, int len, int dstadr)
{
	struct rmap_command_read cmd;
	struct rmap_command *pcmd;
	struct grspw_pkt *txpkt, *rxpkt, *prevpkt;
	struct rmap_spw_pkt rmappkt;
	int i, j, found, rc;
	static int cnt = 0;

	cnt++;

	/* Allocate SpW packet buffer to hold RMAP request */
	txpkt = dma_alloc_tx_buf(rmap_dev);
	if (txpkt == NULL) {
		printf(" No free transmit buffers available %d\n", cnt);
		rtems_semaphore_release(dma_sem);
		return -2;
	}

	/* Describe RMAP request command */
	cmd.type = RMAP_CMD_RI;
	cmd.dstadr = dstadr;
	cmd.dstkey = R_SPWTDP_DSTKEY;
	cmd.address = address;
	cmd.length = len;
	cmd.datalength = 0;
	cmd.data = data; /* let RMAP stack copy data */

	/* Let RMAP stack generate RMAP SpaceWire packet into TX buffer */
	rmappkt.options = 0;
	rmappkt.hdr = txpkt->hdr;
	rmappkt.data = txpkt->data;
	rc = rmap_send_async(rmap_stack, (struct rmap_command *)&cmd, &rmappkt);
	if (rc < 0) {
		printf("rmap_send_async failed %d\n", rc);
		return -3;
	}
	if (rc == 0) {
		/* No wait is expected to return directly */
		puts("rmap_send_async indicates no response");
		return -4;
	}
	/* update TXpkt with RMAP stack output */
	txpkt->dlen = rmappkt.dlen;
	txpkt->hlen = rmappkt.hlen;
	txpkt->flags = 0;
	if (rmappkt.options & PKT_OPTION_HDR_CRC)
		txpkt->flags |= TXPKT_FLAG_HCRC;
	if (rmappkt.options & PKT_OPTION_DATA_CRC)
		txpkt->flags |= TXPKT_FLAG_DCRC;

	/* Put TX packet into transmission queue */
	if (dma_sched_tx_pkt(rmap_dev, txpkt)) {
		printf("write(): Failed scheduling TX packet\n");
		return -5;
	}

	/* Wait for response */
	found = 0;
	for (i = 0; i<20 && !found; i++) {
		/* sleep one tick */
		rtems_task_wake_after(1);

		rtems_semaphore_obtain(dma_sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
		rxpkt = rmap_dev->rx_list.head;
		prevpkt = NULL;
		found = 0;
		for (j=0; j<rmap_dev->rx_list_cnt; j++) {
			/* search for our specific RMAP response
			 * Tell the RMAP stack about the Data and Header
			 * CRC results.
			 */
			rmappkt.options = 0;
			if ((rxpkt->flags & RXPKT_FLAG_DCRC) == 0)
				rmappkt.options |= PKT_OPTION_DATA_CRC;
			if ((rxpkt->flags & RXPKT_FLAG_HCRC) == 0)
				rmappkt.options |= PKT_OPTION_HDR_CRC;
			rmappkt.hdr = rxpkt->hdr;
			rmappkt.data = rxpkt->data;
			pcmd = (struct rmap_command *)&cmd;
			if (rmap_recv_async(rmap_stack, &rmappkt, &pcmd) == 0) {
				found = 1;
				break;
			}
			prevpkt = rxpkt;
			rxpkt = rxpkt->next;
		}
		if (found) {
			/* We found our packet.
			 *
			 * 1. Remove it from rx list
			 * 2. put it back into rx_buf list
			 */
			if (prevpkt == NULL)
				rmap_dev->rx_list.head = rxpkt->next;
			else
				prevpkt->next = rxpkt->next;
			if (rxpkt->next == NULL)
				rmap_dev->rx_list.tail = prevpkt;
			rmap_dev->rx_list_cnt--;

			grspw_list_append(&rmap_dev->rx_buf_list, rxpkt);
			rmap_dev->rx_buf_list_cnt++;

			/* Process the RMAP command */

			if (cmd.datalength != len)
				printf("Request and response length differs\n");
		}

		rtems_semaphore_release(dma_sem);
	}

	if (!found) {
		/* Remove command from RMAP async wait state */
		rmap_cancel_async(rmap_stack, (struct rmap_command *)&cmd);

		printf("Timeout on RMAP response\n");
		/* handle error here ... */

		return -6;
	}

	if (cmd.status != 0)
		return -7;

	/* The callers data buffer has now been updated */

	return 0;
}

int rmap_tgt_write(unsigned int address, void *data, int len, int dstadr)
{
	struct rmap_command_write cmd;
	struct rmap_command *pcmd;
	struct grspw_pkt *txpkt, *rxpkt, *prevpkt;
	struct rmap_spw_pkt rmappkt;
	int i, j, found, rc;
	static int cnt = 0;

	cnt++;

	/* Allocate SpW packet buffer to hold RMAP request */
	txpkt = dma_alloc_tx_buf(rmap_dev);
	if (txpkt == NULL) {
		printf("write(): No free transmit buffers available %d\n", cnt);
		rtems_semaphore_release(dma_sem);
		return -2;
	}

	/* Describe RMAP request command */
	cmd.type = RMAP_CMD_WIA;
	cmd.dstadr = dstadr;
	cmd.dstkey = R_SPWTDP_DSTKEY;
	cmd.address = address;
	cmd.length = len;
	cmd.data = txpkt->data; /* Data to be analysed (not copied) */

	/* Let RMAP stack generate RMAP SpaceWire packet into TX buffer */
	rmappkt.options = 0;
	rmappkt.hdr = txpkt->hdr;
	/* Copy data to SpW buffer (we could implement zero-copy here...) */
	memcpy(txpkt->data, data, len);

	rc = rmap_send_async(rmap_stack, (struct rmap_command *)&cmd, &rmappkt);
	if (rc < 0) {
		printf("write(): rmap_send_async failed %d\n", rc);
		return -3;
	}
	if (rc == 0) {
		/* No wait is expected to return directly */
		puts("write(): rmap_send_async indicates no response");
		return -4;
	}
	/* update TXpkt with RMAP stack output */
	txpkt->dlen = rmappkt.dlen;
	txpkt->hlen = rmappkt.hlen;
	txpkt->flags = 0;
	if (rmappkt.options & PKT_OPTION_HDR_CRC)
		txpkt->flags |= TXPKT_FLAG_HCRC;
	if (rmappkt.options & PKT_OPTION_DATA_CRC)
		txpkt->flags |= TXPKT_FLAG_DCRC;

	/* Put TX packet into transmission queue */
	if (dma_sched_tx_pkt(rmap_dev, txpkt)) {
		printf("write(): Failed scheduling TX packet\n");
		return -5;
	}

	/* Wait for response */
	found = 0;
	for (i = 0; i<20 && !found; i++) {
		/* sleep one tick */
		rtems_task_wake_after(1);

		rtems_semaphore_obtain(dma_sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);
		rxpkt = rmap_dev->rx_list.head;
		prevpkt = NULL;
		found = 0;
		for (j=0; j<rmap_dev->rx_list_cnt; j++) {
			/* search for our specific RMAP response
			 * Tell the RMAP stack about the Data and Header
			 * CRC results.
			 */
			rmappkt.options = 0;
			if ((rxpkt->flags & RXPKT_FLAG_DCRC) == 0)
				rmappkt.options |= PKT_OPTION_DATA_CRC;
			if ((rxpkt->flags & RXPKT_FLAG_HCRC) == 0)
				rmappkt.options |= PKT_OPTION_HDR_CRC;
			rmappkt.hdr = rxpkt->hdr;
			rmappkt.data = rxpkt->data;
			pcmd = (struct rmap_command *)&cmd;
			if (rmap_recv_async(rmap_stack, &rmappkt, &pcmd) == 0) {
				found = 1;
				break;
			}
			prevpkt = rxpkt;
			rxpkt = rxpkt->next;
		}
		if (found) {
			/* We found our packet.
			 *
			 * 1. Remove it from rx list
			 * 2. put it back into rx_buf list
			 */
			if (prevpkt == NULL)
				rmap_dev->rx_list.head = rxpkt->next;
			else
				prevpkt->next = rxpkt->next;
			if (rxpkt->next == NULL)
				rmap_dev->rx_list.tail = prevpkt;
			rmap_dev->rx_list_cnt--;

			grspw_list_append(&rmap_dev->rx_buf_list, rxpkt);
			rmap_dev->rx_buf_list_cnt++;
		}

		rtems_semaphore_release(dma_sem);
	}

	if (!found) {
		/* Remove command from RMAP async wait state */
		rmap_cancel_async(rmap_stack, (struct rmap_command *)&cmd);

		printf("write(): Timeout on RMAP response\n");
		/* handle error here ... */

		return -5;
	}

	/* Process the RMAP command */
	if (cmd.status != 0) {
		printf("write(): RMAP write failed with %d\n", cmd.status);
		return -7;
	}

	/* The callers data buffer has now been updated */


	return 0;
}

/* Sync local cache copy of remote registers by reading from target */
int remote_spwtdp_read(unsigned int ofs, int len)
{
	int status;

	if ((ofs+len) > sizeof(spwtdp))
		return -1;

	/* Sync Read */
	status = rmap_tgt_read((unsigned int)R_SPWTDP_BASEADR + ofs,
				((void *)&spwtdp) + ofs, len, R_SPWTDP_DSTADR);
	if (status != 0) {
		printf("Failed executing RMAP read from SPWTDP Target: %d\n",
			status);
	}
	return status;
}

/* Sync local cache copy of remote registers by writing to target */
int remote_spwtdp_write(unsigned int ofs, int len)
{
	int status;

	if ((ofs+len) > sizeof(spwtdp))
		return -1;

	status = rmap_tgt_write((unsigned int)R_SPWTDP_BASEADR + ofs,
				((void *)&spwtdp) + ofs,
				len,
				R_SPWTDP_DSTADR);
	if (status != 0) {
		printf("Failed executing RMAP write to SPWTDP Target: %d\n",
			status);
	}
	return status;
}

void *tdp;
struct spwtdp_regs *tdp_regs;

void tdp_isr(unsigned int ists, void *data)
{
	printk("TDP_ISR: 0x%02x\n", ists);
}

#ifdef DEBUG_ET_SAMPLING_TIMES
unsigned int tmr_log[1024];
int tmr_cnt;

void log_tmr()
{
	if (tmr_cnt == 0) {
		*(volatile unsigned int *)TIMER_BASE = 0xffffffff;
		*(volatile unsigned int *)(TIMER_BASE + 0x4) = 0xffffffff;
		*(volatile unsigned int *)(TIMER_BASE + 0x8) = 0x55;
	}
	tmr_log[tmr_cnt] = 0xffffffff - (*(volatile unsigned int *)TIMER_BASE);
	tmr_cnt++;
	if (tmr_cnt >= 1024)
		tmr_cnt = 0;
}
#endif

/* Read and parse ET information from the PREAMBLE field.
 * ctrl = pointer to control register (cmd_ctrl, dat_ctrl, ...)
 */
int read_et(unsigned int *ctrl, unsigned char *coarse, unsigned char *fine)
{
	unsigned int sample[6];
	int i, ccnt, fcnt;

	for (i = 0; i < 6; i++) {
		sample[i] = ctrl[i];
	}
#ifdef DEBUG_ET_SAMPLING_TIMES
	log_tmr();
#endif

	if (parse_precision(sample[0] & 0xffff, &ccnt, &fcnt))
		return -1;

	if (coarse) {
		for (i = 0; i < ccnt; i++)
			coarse[i] = ((unsigned char *)sample)[4+i];
	}
	if (fine) {
		for (i = ccnt; i < ccnt+fcnt; i++)
			fine[i-ccnt] = ((unsigned char *)sample)[4+i];
	}
	return 0;
}

/* Read the ET and parse the preamble pointed to by ctrl, and then put
 * only the time (not the preamble) into memory pointed to by time argument
 */
int read_et_all(unsigned int *ctrl, unsigned char *time)
{
	unsigned int sample[6];
	int i, ccnt, fcnt;

	for (i = 0; i < 6; i++) {
		sample[i] = ctrl[i];
	}

	if (parse_precision(sample[0] & 0xffff, &ccnt, &fcnt))
		return -1;

	for (i=0; i < ccnt+fcnt; i++)
		time[i] = ((unsigned char *)sample)[4+i];
	return 0;
}

/* Return only the fine time of the ET, parse PREAMBLE field to know format. */
int split_et(unsigned short preamble, unsigned char *et, unsigned char *coarse, unsigned char *fine)
{
	int i, ccnt, fcnt;

	if (parse_precision(preamble, &ccnt, &fcnt))
		return -1;

	if (coarse) {
		for (i = 0; i < ccnt; i++)
			coarse[i] = et[i];
	}
	if (fine) {
		for (i = ccnt; i < ccnt+fcnt; i++)
			fine[i-ccnt] = et[i];
	}

	return 0;
}

/* Convert the Elapse Time into two unsigned 64-bit integers, couse and
 * fine time according to the PREAMBLE field format.
 * Retuns zero on success and negative on failure.
 */
int convert_uint_et(unsigned short preamble, unsigned char *et, unsigned long long *coarse, unsigned long long *fine)
{
	int i, ccnt, fcnt;

	if (parse_precision(preamble, &ccnt, &fcnt))
		return -1;

	if (ccnt > 64)
		return -1;
	if (fcnt > 64)
		return -1;

	if (coarse) {
		*coarse = 0;
		for (i = 0; i < ccnt; i++)
			*coarse |= et[i] << ((ccnt-1) - i)*8;
	}
	if (fine) {
		*fine = 0;
		for (i = ccnt; i < ccnt+fcnt; i++)
			*fine |= et[i] << ((fcnt-1) - (i-ccnt))*8;
	}

	return 0;
}

/* Subtract two Elapsed Time codes:
 * this is implemented under the assumption that a>b
 * r = a - b
 */
int sub_et(
	unsigned short preamble,
	unsigned char *a,
	unsigned char *b,
	unsigned char *r)
{
	int i, oldrem, rem, ccnt, fcnt;

	if (parse_precision(preamble, &ccnt, &fcnt))
		return -1;

	rem = 0;
	oldrem=0;
	for (i = ccnt + fcnt - 1; i > fcnt; i--) {
		oldrem = rem;

		if ((a[i] - oldrem) < b[i])
			rem = 1;
		else
			rem = 0;

		r[i] = (a[i] - oldrem + (rem << 8)) - b[i];
	}

	if ((a[i] - oldrem + (rem << 8)) < b[i]) {
		printf("sub_et: error input: %d\n",
		       (a[i] - oldrem + (rem << 8)) < b[i]);
		return -1; /* input error: a>b was not true */
	}

	return 0;
}

/* Add two Elapsed Time codes:
 * this is implemented under the assumption: a+b < "what preamble can hold"
 */
int add_et(
	unsigned short preamble,
	unsigned char *a,
	unsigned char *b,
	unsigned char *r)
{
	int i, oldrem, rem, ccnt, fcnt;
	unsigned short tmp;

	if (parse_precision(preamble, &ccnt, &fcnt))
		return -1;

	rem = 0;
	for (i = ccnt + fcnt - 1; i > fcnt; i--) {
		oldrem = rem;

		tmp = (unsigned short)a[i] + (unsigned short)b[i] + oldrem;

		if (tmp > 0xff)
			rem = 1;
		else
			rem = 0;

		r[i] = tmp & 0xff;
	}

	return 0;
}

/* Compare two Elapsed Time codes:
 * returns -1 if b < a
 * returns 1 if b > a
 * returns 0 if b == a
 */
int cmp_et(
	unsigned short preamble,
	unsigned char *a,
	unsigned char *b)
{
	int i, ccnt, fcnt;

	if (parse_precision(preamble, &ccnt, &fcnt))
		return -1;

	for (i = 0; i < ccnt+fcnt; i++) {
		if (b[i] < a[i])
			return -1;
		else if (b[i] > a[i])
			return 1;
	}

	return 0;
}

/* For Latency calculation. Moved to a global context so that they can be
 * inpected from GRMON mem commands.
 */
unsigned char i_tsrx[10], i_tstx[10], t_tsrx[10], t_tstx[10];
unsigned char i_diff[10], t_diff[10];
unsigned char total[10], min[10], max[10];
unsigned char result_diff[LATENCY_COUNT][10];
unsigned char fnow[10], flast;
unsigned long long total_coarse, total_fine, latency_fine;

rtems_task test_app(rtems_task_argument ignored)
{
	int i;
	struct rmap_config rmap_cfg;
	int status;

	unsigned short preamble;
	int coarse, fine;
	rtems_interrupt_level level;
	int num_words;
	unsigned int tmp;

	/* Initialize two GRSPW AMBA ports */
	nospw = grspw_dev_count();
	if (nospw < 1) {
		printf("Found no SpaceWire cores, aborting\n");
		exit(0);
	}
	if (nospw > DEVS_MAX) {
		printf("Limiting to %d SpaceWire devices\n", DEVS_MAX);
		nospw = DEVS_MAX;
	}

	memset(devs, 0, sizeof(devs));
	for (i=0; i<nospw; i++) {
		if (dev_init(i)) {
			printf("Failed to initialize GRSPW%d\n", i);
			exit(0);
		}
		fflush(NULL);
	}

	/* Initialize GRSPW packets */
	init_pkts();

	printf("\n\nStarting SpW DMA channels\n");
	for (i = 0; i < nospw; i++) {
		printf("Starting GRSPW%d: ", i);
		fflush(NULL);
		if (grspw_start(DEV(&devs[i]))) {
			printf("Failed to initialize GRSPW%d\n", i);
			exit(0);
		}
		printf("DMA Started Successfully\n");
	}

	/* Make sure the SpaceWire link is in run-state before proceeding */
	if (dev_check_started(DEV_IDX) == 0) {
		printf("Link to SPWTDP target is not in run-state\n");
		exit(0);
	}



	/* Get on-chip SPWTDP */
	tdp = spwtdp_open(0);
	if (!tdp) {
		puts("No on-chip SPWTDP device present. Aborting");
		exit(0);
	}
	tdp_regs = spwtdp_get_regs(tdp);
	if (!tdp_regs) {
		puts("Failed getting SPW-TDP registers");
		exit(0);
	}
	printf("using SPWTDP at 0x%08x\n", (unsigned int)tdp_regs);
	if (spwtdp_reset(tdp)) {
		puts("on-chip SPWTDP device present. Aborting");
		exit(0);
	}
	spwtdp_clr_stats(tdp);
	/* Install ISR but do not enable IRQ until later */
	spwtdp_int_register(tdp, tdp_isr, tdp_regs);

	/* Start SpW DMA and SpW link-status tasks and wait */
	fflush(NULL);
	rtems_task_start(tid_link, link_ctrl_task, 0);
	rtems_task_start(tid_dma, dma_task, 0);
	rtems_task_wake_after(12);

	/* Set up asynchronos RMAP stack */
	rmap_dev = &devs[RMAP_SPW_DEV_IDX];
	rmap_cfg.route_func = NULL;
	rmap_cfg.spw_adr = rmap_dev->cfg.adrcfg.def_addr;
	rmap_cfg.tid_msb = 0xff; /* only communication with one SpW RMAP target */
	if (rmap_dev->hwsup.rmap_crc)
		rmap_cfg.drv_cap = DRV_CAP_HDR_CRC | DRV_CAP_DATA_CRC;
	else
		rmap_cfg.drv_cap = 0;
	rmap_cfg.thread_safe = 0; /* only one sender */
	rmap_cfg.drv = NULL; /* not needed in async mode */
	rmap_cfg.max_rx_len = rmap_dev->cfg.chan[0].rxmaxlen - 32; /* adjust for RMAP header */
	rmap_cfg.max_tx_len = rmap_dev->cfg.chan[0].rxmaxlen;
	rmap_stack = rmap_async_init(&rmap_cfg, 16);
	if (!rmap_stack) {
		printf("RMAP stack initialization failed\n");
		exit(0);
	}

	/*** Init SpW-TDPs ***/

	/* CFG0 register setup:
	 *  - Core has already been resetted
	 *  - Select TDP over CUC
	 *  - Select SpW0 for SpaceWire transmissions (IRQ and TC)
	 *  - Mapping generate 64 TimeCodes per Second (2^6)
	 *  - Enable Trasmitter
	 */
	tdp_regs->conf[0] = (6<<8) | (1<<7) | (0<<4) | (1<<1);

	/* Leave CFG1 and CFG2 to reset values */

	/* CFG3 register setup:
	 *  - INRX=4, INTX=5 - this is to match Target
	 *  - Disable Acknowledge on Interrupts
	 *  - STM Mask to do timeStamping at every 4th TC, i.e. 16 times/second
	 */
	tdp_regs->conf[3] = (0x03<<16) | (0<<10) | (4<<5) | (5<<0);

	preamble = tdp_regs->dat_ctrl & 0xffff;
	if (parse_precision(preamble, &coarse, &fine)) {
		puts("failed parsing PField of on-chip SPWTDP");
		exit(0);
	}

	printf("Coarse precision: %d bits\n", coarse*8);
	printf("Fine precision:   %d bits\n", fine*8);

	/* Command Control register:
	 *  - NC/IS initialization (will set datation when ET written)
	 *  - init Preamble as CCSDS 301.0-B-4:
	 *    * 32bit coarse time, 64-bit fractional time
	 *    * 1958 Jan 1 - epoch
	 *
	 * examples:
	 *   - 32+32 bits: 0x9e08 == 0x9c10
	 *   - 32+48 bits: 0x9e10 == 0x9c18
	 *   - 32+64 bits: 0x9d1c == 0x9f14
	 */
	tdp_regs->cmd_ctrl = (1<<31) | preamble;

	/* Sync datation with RTEMS System Timer overflow. This approach
	 * is dangerous but is an example what basically could be needed. We
	 * wait for the next tick and then we write the ET so the datation is
	 * initialized.
	 * Instead of writing all zeros to the datation unit we shuold get
	 * the time from RTEMS and convert that into the correct time format...
	 * but this is just an example.
	 */
	rtems_interrupt_disable(level);
	/* init as much as possible of ET */
	num_words = (((coarse + fine) + 0x3) & ~0x3) / 4;
	for (i = 0; i<(num_words-1); i++) {
		tdp_regs->dat_et[i] = 0;
	}
	/* Wait for last 1us before tick */
	while (LEON3_Timer_Regs->timer[0].value > 0) {
		asm volatile ("nop\n\t"::); /* wait 8 clocks */
		asm volatile ("nop\n\t"::);
		asm volatile ("nop\n\t"::);
		asm volatile ("nop\n\t"::);
		asm volatile ("nop\n\t"::);
		asm volatile ("nop\n\t"::);
		asm volatile ("nop\n\t"::);
		asm volatile ("nop\n\t"::);
	}
	asm volatile ("nop\n\t"::); /* add an additional small delay */
	asm volatile ("nop\n\t"::);
	asm volatile ("nop\n\t"::);
	asm volatile ("nop\n\t"::);
	tdp_regs->dat_et[i] = 0; /* this should write ET to datation */
	for (i++; i<5; i++) {
		tdp_regs->dat_et[i] = 0;
	}

	/* use GRMON here to set a breakpoint on grmon_clk_bp() to see the
	 * initialization accuracy... used to debug.
	 */
	/*grmon_clk_bp1(0, LEON3_Timer_Regs->timer[0].value);*/

	rtems_interrupt_enable(level); /* let the tick be handled */

	/* At this point the Initiator sends 64 SpW timecodes per second.
	 * the targets will be initialized within one second can wait for
	 * the next SpW Time Code of 0 (fine time = 0)
	 */

	/*** Target Initialization ***/

	/* Enable TimeCode and Interrupt on remote SpW device end */
	tmp = 0;
	if (rmap_tgt_read(R_SPW_BASEADR, &tmp, 4, R_SPWTDP_DSTADR)) {
		printf("Failed reading Target SpW Ctrl register\n");
		exit(0);
	}
	tmp |= 0x01802;
	if (rmap_tgt_write(R_SPW_BASEADR, &tmp, 4, R_SPWTDP_DSTADR)) {
		printf("Failed writing Target SpW Ctrl register\n");
		exit(0);
	}
	/* clr status register */
	tmp = 0xffffffff;
	if (rmap_tgt_write(R_SPW_BASEADR+0x04, &tmp, 4, R_SPWTDP_DSTADR)) {
		printf("Failed clearing Target SpW Status register\n");
		exit(0);
	}

	/* Init SPWTDP */

	status = remote_spwtdp_init();
	if (status) {
		printf("Remote initialization failed: %d\n", status);
		exit(0);
	}

	/* Check if Initiator and Target have same hardware setup */
	if (remote_coarse != coarse || remote_fine != fine) {
		puts("This example is made for equal precision of counters");
		exit(0);
	}

	/* CFG3 register setup:
	 *  - INRX=5, INTX=4 - this is to match Initiator
	 *  - Disable Acknowledge on Interrupts
	 *  - STM Mask to do timeStamping at every 4th TC, i.e. 16 times/second
	 */
	spwtdp.conf[3] = (0x03<<16) | (0<<10) | (5<<5) | (4<<0);

	/* Set up next time for coarse time increment */
	spwtdp.cmd_ctrl = (1<<31) | (1<<30) | (0<<16) | preamble;
	spwtdp.cmd_et[0] = 0;
	spwtdp.cmd_et[1] = 0;
	spwtdp.cmd_et[2] = 0;
	spwtdp.cmd_et[3] = 0;
	spwtdp.cmd_et[4] = 0;
	/* next coarse time in the relevant octet depending on the length of
	 * the coarse time
	 */
	((unsigned char *)spwtdp.cmd_et)[coarse - 1] = 1;

	/* Sync (changed parts of) local cached copy to Target register using
	 * RMAP
	 */
	remote_spwtdp_write(0x0c, 0x38 - 0xc);

	/* Resync to see that written registers have an effect */
	remote_spwtdp_read(0, sizeof(spwtdp));
	if (((spwtdp.cmd_ctrl >> 31) & 0x1) != 0x1) {
		printf("Target ET init failed1: 0x%08x\n", spwtdp.cmd_ctrl);
		exit(0);
	}

	/* Wait for next tick for coarse time, then check that target has been
	 * initialized
	 */
	while (((volatile unsigned char *)tdp_regs->dat_et)[coarse - 1] == 0) {
		rtems_task_wake_after(1);
	}

	/* Resync to see that Target ET has been initialized */
	remote_spwtdp_read(0, sizeof(spwtdp));

	if (((spwtdp.cmd_ctrl >> 31) & 0x1) != 0x0) {
		printf("Target ET init failed2: 0x%08x\n", spwtdp.cmd_ctrl);
		exit(0);
	}
	if (((unsigned char *)spwtdp.dat_et)[coarse - 1] != 1) {
		printf("Target Datation init failed: 0x%08x 0x%08x 0x%08x 0x%08x 0x%04x 0x%08x\n",
			spwtdp.cmd_ctrl, spwtdp.dat_et[0],
			spwtdp.dat_et[1], spwtdp.dat_et[2],
			spwtdp.dat_et[3], spwtdp.dat_et[4]);
		exit(0);
	}

	puts("successful init");
	printf("Start latency calculation for %d samples\n", LATENCY_COUNT);

	/*** Make latency calculations ***/

	/* Enable RX/TX Distributed Interrupts locally */
	dev_enable_spwirq(DEV_IDX);

	/* Enable distributed interrupts on SpW device used by SpWTDP */
	tmp = (0x3 << 16);
	if (rmap_tgt_write(R_SPW_BASEADR+0xa0, &tmp, 4, R_SPWTDP_DSTADR)) {
		printf("Failed writing Target SpW 0xa0 register\n");
		exit(0);
	}
	/* Set up every TimeCode of value=0bxxxx10 a Interrupts is generated */
	tdp_regs->ts_tx_ctrl = (2<<16) | preamble;

	/* Enable Latency Calculation of the Target before initiator */
	spwtdp.conf[0] |= 1<<16;
	remote_spwtdp_write(0, 4);

	read_et((unsigned int *)&tdp_regs->dat_ctrl, NULL, fnow);

	/* Start latency calculation by setting LE bit */
	tdp_regs->conf[0] |= (1<<16);
	tdp_regs->ien =0x00000030;
	/* Make calculations*/
	for (i=0; i<LATENCY_COUNT; i++) {
	        while ( ( (volatile unsigned char *)tdp_regs->ists) != (volatile unsigned char *) 0x00000030 )
	        {}
	        tdp_regs->ists =0x00000030;
		/* Read the Time Stamp TX and RX values of initiator */
		memset(i_tsrx, 0, sizeof(i_tsrx));
		memset(i_tstx, 0, sizeof(i_tstx));
		read_et_all((unsigned int *)&tdp_regs->ts_rx_ctrl, i_tsrx);
		read_et_all((unsigned int *)&tdp_regs->ts_tx_ctrl, i_tstx);

		remote_spwtdp_read(0x60, 0x98-0x60);

		/* Read the Time Stamp TX and RX values of target*/
		memset(t_tsrx, 0, sizeof(t_tsrx));
		memset(t_tstx, 0, sizeof(t_tstx));
		read_et_all((unsigned int *)&spwtdp.ts_rx_ctrl, t_tsrx);
		read_et_all((unsigned int *)&spwtdp.ts_tx_ctrl, t_tstx);

		/* Do latency calculations:
		 * ((i.TsRX-i.TsTX) - (t.TsTX-t.TsRX)) / 2
		 */

		sub_et(preamble, i_tsrx, i_tstx, i_diff);
		sub_et(preamble, t_tstx, t_tsrx, t_diff);
		sub_et(preamble, i_diff, t_diff, &result_diff[i][0]);

#ifdef DEBUG_LATENCY
		/*tdp_regs->conf[0] &= ~(1<<16);*/
		printf("%d: R 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n",i,
			result_diff[i][0], result_diff[i][1],
			result_diff[i][2], result_diff[i][3],
			result_diff[i][4], result_diff[i][5],
			result_diff[i][6], result_diff[i][7],
			result_diff[i][8]);
		/*exit(0);*/
#endif
	}

	/* Disable latency calculation by clearing LE bit */
	tdp_regs->conf[0] &= ~(1<<16);
	spwtdp.conf[0] &= ~(1<<16);
	remote_spwtdp_write(0, 4);

	memset(total, 0, 10);
	memset(max, 0, 10);
	memset(min, 0xff, fine+coarse);
	for (i=0; i<LATENCY_COUNT; i++) {
		add_et(preamble, &result_diff[i][0], total, total);

		if (cmp_et(preamble, max, &result_diff[i][0]) > 0)
			memcpy(max, &result_diff[i][0], 10);

		if (cmp_et(preamble, min, &result_diff[i][0]) < 0)
			memcpy(min, &result_diff[i][0], 10);

	}

	printf("TOTAL:      0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n",
			total[0], total[1],
			total[2], total[3],
			total[4], total[5],
			total[6], total[7],
			total[8]);

	printf("MIN SAMPLE: 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n",
			min[0], min[1],
			min[2], min[3],
			min[4], min[5],
			min[6], min[7],
			min[8]);

	printf("MAX SAMPLE: 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n",
			max[0], max[1],
			max[2], max[3],
			max[4], max[5],
			max[6], max[7],
			max[8]);

	total_coarse = total_fine = 0;
	if (convert_uint_et(preamble, total, &total_coarse, &total_fine)) {
		puts("fine time must be limited to 64 bits for this example");
		exit(0);
	}
	if (total_coarse > 0) {
		puts("Example assumes average to be covered by fine time only. Aborting");
		exit(0);
	}
	latency_fine = total_fine / (LATENCY_COUNT);
	printf("AVERAGE FINE TIME LATENCY: %llu\n", latency_fine);
	latency_fine = latency_fine / 2;
	printf("AVERAGE FINE TIME LATENCY / 2: %llu\n", latency_fine);

	/* Update Target with latency detected values
	 *
	 * For now we assume that latency is convered by the 2 least
	 * significant byte of fine time.
	 */
	printf("Writing the latency %llu to target registers\n", latency_fine);
	if (latency_fine > 65025) {
		printf("Too big a latency value for this example to work\n");
		exit(0);
	}
	for (i = 0; i < fine + coarse; i++) {
	  ((unsigned char *)spwtdp.lat_et)[i] = 0;
	  if (latency_fine < 256) {
	    if (i == (fine + coarse - 1))
	        ((unsigned char *)spwtdp.lat_et)[i] = latency_fine & 0xff;
	  }
	  if (latency_fine > 255) {
              if (i == (fine + coarse - 1))
		  ((unsigned char *)spwtdp.lat_et)[i] = latency_fine % 0x100;
	      if (i == (fine + coarse - 2))
		  ((unsigned char *)spwtdp.lat_et)[i] = latency_fine / 0x100;
	  }
	}
	remote_spwtdp_write(0xa4, 5*4);

	/* Resync the register from target to see the last values */
	remote_spwtdp_read(0, sizeof(spwtdp));

	printf("Example successfully completed\n");


#ifdef TOGGLE_TGT_ME
	/* Enable this to sync Remote Taget SPWTDP registers over to local
	 * initiator memory every 10 RTEMS system clock ticks. The Mitigation
	 * bit of the target is toggled every 100 ticks just as an example
	 * how to write the CONF[0] Register over RMAP.
	 */
	memset(&spwtdp, 0, sizeof(spwtdp));
	i = 0;
	while (1) {
		status = remote_spwtdp_read(0, sizeof(spwtdp));
		if (status != 0)
			printf("Error in spwtdp_read() Status=%d\n", status);
		#if 0 /* example how to toggle Mitigation Enable bit */
		if (i == 10) {
			printf("CONF[0] before: 0x%x\n", spwtdp.conf[0]);
			i = 0;
			if (spwtdp.conf[0] & 0x8)
				spwtdp.conf[0] &= ~0x8;
			else
				spwtdp.conf[0] |= 0x8;
			status = remote_spwtdp_write(0,4);/* Sync only CONF[0]*/
			if (status != 0)
				printf("Error in spwtdp_write() Status=%d\n", status);
		}
		#endif
		rtems_task_wake_after(10);
		i++;
	}
#endif

/*	for ( i=0; i<nospw; i++)
		dev_cleanup(i);
	tasks_stop = 1;
	rtems_task_wake_after(8);*/

	printf("\n\nEXAMPLE COMPLETED.\n\n");
	exit(0);
}

rtems_task link_ctrl_task(rtems_task_argument unused)
{
	int i, run;
	struct grspw_device *dev;
	spw_link_state_t state;

	printf("Started link control task\n");

	while (tasks_stop == 0) {
		for (i = 0; i < nospw; i++) {
			dev = &devs[i];
			if (dev->dh == NULL)
				continue;

			/* Check if link status has changed */
			state = grspw_link_state(dev->dh);
			run = 0;
			if (state == SPW_LS_RUN)
				run = 1;
			if ((run && dev->run == 0) || (run == 0 && dev->run)) {
				if (run)
					printf("GRSPW%d: link state entering run-state\n", i);
				else
					printf("GRSPW%d: link state leaving run-state\n", i);
				dev->run = run;
			}
		}
		rtems_task_wake_after(4);
	}

	printf(" Link control task shutdown\n");

	rtems_task_delete(RTEMS_SELF);
}

rtems_task dma_task(rtems_task_argument unused)
{
	int i;
	struct grspw_device *dev;

	printf("Started DMA control task\n");

	while (tasks_stop == 0) {
		rtems_semaphore_obtain(dma_sem, RTEMS_WAIT, RTEMS_NO_TIMEOUT);

		for (i = 0; i < nospw; i++) {
			dev = &devs[i];
			if (dev->dh == NULL)
				continue;

			dma_process(dev);
		}
		rtems_semaphore_release(dma_sem);
		rtems_task_wake_after(1);
	}

	printf(" DMA task shutdown\n");

	rtems_task_delete(RTEMS_SELF);
}

int dma_process(struct grspw_device *dev)
{
	int cnt, rc, prnt_pkt;
	struct grspw_list lst;
	struct grspw_pkt *pkt;
	unsigned char *c;
#ifdef DEBUG_SPW_TX
	int i
#endif

	/* Prepare receiver with packet buffers */
	if (dev->rx_buf_list_cnt > 0) {
		rc = grspw_dma_rx_prepare(dev->dma[0], 0, &dev->rx_buf_list,
							dev->rx_buf_list_cnt);
		if (rc != 0) {
			printf("rx_prep failed %d\n", rc);
			return -1;
		}
		/*printf("GRSPW%d: Prepared %d RX packet buffers for future "
		 "reception\n", dev->index, dev->rx_list_cnt);*/
		grspw_list_clr(&dev->rx_buf_list);
		dev->rx_buf_list_cnt = 0;
	}

	/* Try to receive packets on receiver interface */
	grspw_list_clr(&lst);
	cnt = -1; /* as many packets as possible */
	rc = grspw_dma_rx_recv(dev->dma[0], 0, &lst, &cnt);
	if (rc != 0) {
		printf("rx_recv failed %d\n", rc);
		return -1;
	}
	if (cnt > 0) {
#ifdef DEBUG_SPW_RX
		printf("GRSPW%d: Recevied %d packets\n", dev->index, cnt);
#endif
		for (pkt = lst.head; pkt; pkt = pkt->next) {
			prnt_pkt = 0;
			if ((pkt->flags & RXPKT_FLAG_RX) == 0) {
				printf(" PKT not received.. buf ret\n");
				continue;
			} else if (pkt->flags &
					(RXPKT_FLAG_EEOP | RXPKT_FLAG_TRUNK)) {
				printf(" PKT RX errors:");
				if (pkt->flags & RXPKT_FLAG_TRUNK)
					printf(" truncated");
				if (pkt->flags & RXPKT_FLAG_EEOP)
					printf(" EEP");
				printf(" (0x%x)", pkt->flags);
				prnt_pkt = 1;
			}
#ifdef DEBUG_SPW_RX
			 else {
				printf(" PKT");
				prnt_pkt = 1;
			}
#endif
			if (prnt_pkt == 1) {
				c = (unsigned char *)pkt->data;
				printf(" of length %d bytes, 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x...\n",
					pkt->dlen, c[0], c[1], c[2], c[3], c[4], c[5], c[6], c[7]);
			}
		}

		/* Reuse packet buffers by moving packets to rx_list */
		grspw_list_append_list(&dev->rx_list, &lst);
		dev->rx_list_cnt += cnt;
	}

	/*** TX PART ***/

	/* Reclaim already transmitted buffers */
	cnt = -1; /* as many packets as possible */
	grspw_list_clr(&lst);
	rc = grspw_dma_tx_reclaim(dev->dma[0], 0, &lst, &cnt);
	if (rc != 0) {
		printf("tx_reclaim failed %d\n", rc);
		exit(0);
	}
	/* put sent packets in end of send queue for retransmission */
	if (cnt > 0) {
		/*printf("GRSPW%d: Reclaimed %d TX packet buffers\n",
			dev->index, cnt);*/
		/* Clear transmission flags */
		pkt = lst.head;
		while (pkt) {
			if ((pkt->flags & TXPKT_FLAG_TX) == 0)
				printf(" One Packet TX failed\n");
			else if (pkt->flags & TXPKT_FLAG_LINKERR)
				printf(" One Packet with TX errors\n");
			pkt = pkt->next;
		}

		grspw_list_append_list(&dev->tx_buf_list, &lst);
		dev->tx_buf_list_cnt += cnt;
	}

	/* Send packets in the tx_list queue */
	if (dev->tx_list_cnt > 0) {
#ifdef DEBUG_SPW_TX
			printf("GRSPW%d: Sending %d packets\n", dev->index,
				dev->tx_list_cnt);
			for (pkt = dev->tx_list.head; pkt; pkt = pkt->next) {
				printf(" PKT of length %d bytes,", pkt->hlen+pkt->dlen);
				for (i = 0; i < pkt->hlen+pkt->dlen && i < 8; i++) {
					if (i < pkt->hlen)
						c = i + (unsigned char *)pkt->hdr;
					else
						c = i - pkt->hlen + (unsigned char *)pkt->data;
					printf(" 0x%02x", *c);
				}
				printf("\n");
			}
#endif
			rc = grspw_dma_tx_send(dev->dma[0], 0, &dev->tx_list,
							dev->tx_list_cnt);
			if (rc != 0) {
				printf("tx_send failed %d\n", rc);
				exit(0);
			}
			dev->tx_list_cnt = 0;
			grspw_list_clr(&dev->tx_list);
	}
	return 0;
}
