/*
 *******************************************
 *   Definitions from Motorola/FreeScale   *
 *******************************************
 */

/*
 * File:		mcf5235.h
 * Purpose:		MCF5235 definitions
 *
 * Notes:
 */

#ifndef _CPU_MCF5235_H
#define _CPU_MCF5235_H

/********************************************************************/

/*
 * File:		mcf5xxx.h
 * Purpose:		Definitions common to all ColdFire processors
 *
 * Notes:
 */

#ifndef _CPU_MCF5XXX_H
#define _CPU_MCF5XXX_H

/***********************************************************************/
/*
 * The basic data types
 */

typedef unsigned char		uint8;  /*  8 bits */
typedef unsigned short int	uint16; /* 16 bits */
typedef unsigned long int	uint32; /* 32 bits */

typedef signed char			int8;   /*  8 bits */
typedef signed short int	int16;  /* 16 bits */
typedef signed long int		int32;  /* 32 bits */

typedef volatile uint8		vuint8;  /*  8 bits */
typedef volatile uint16		vuint16; /* 16 bits */
typedef volatile uint32		vuint32; /* 32 bits */

/***********************************************************************/
/*
 * Common M68K & ColdFire definitions
 *
 ***********************************************************************/

#define ADDRESS			uint32
#define INSTRUCTION		uint16
#define ILLEGAL			0x4AFC
#define CPU_WORD_SIZE	16

#define MCF5XXX_SR_T		(0x8000)
#define MCF5XXX_SR_S		(0x2000)
#define MCF5XXX_SR_M		(0x1000)
#define MCF5XXX_SR_IPL		(0x0700)
#define MCF5XXX_SR_IPL_0	(0x0000)
#define MCF5XXX_SR_IPL_1	(0x0100)
#define MCF5XXX_SR_IPL_2	(0x0200)
#define MCF5XXX_SR_IPL_3	(0x0300)
#define MCF5XXX_SR_IPL_4	(0x0400)
#define MCF5XXX_SR_IPL_5	(0x0500)
#define MCF5XXX_SR_IPL_6	(0x0600)
#define MCF5XXX_SR_IPL_7	(0x0700)
#define MCF5XXX_SR_X		(0x0010)
#define MCF5XXX_SR_N		(0x0008)
#define MCF5XXX_SR_Z		(0x0004)
#define MCF5XXX_SR_V		(0x0002)
#define MCF5XXX_SR_C		(0x0001)

#define MCF5XXX_CACR_CENB		(0x80000000)
#define MCF5XXX_CACR_CPDI		(0x10000000)
#define MCF5XXX_CACR_CPD		(0x10000000)
#define MCF5XXX_CACR_CFRZ		(0x08000000)
#define MCF5XXX_CACR_CINV		(0x01000000)
#define MCF5XXX_CACR_DIDI		(0x00800000)
#define MCF5XXX_CACR_DISD		(0x00400000)
#define MCF5XXX_CACR_INVI		(0x00200000)
#define MCF5XXX_CACR_INVD		(0x00100000)
#define MCF5XXX_CACR_CEIB		(0x00000400)
#define MCF5XXX_CACR_DCM_WR		(0x00000000)
#define MCF5XXX_CACR_DCM_CB		(0x00000100)
#define MCF5XXX_CACR_DCM_IP		(0x00000200)
#define MCF5XXX_CACR_DCM		(0x00000200)
#define MCF5XXX_CACR_DCM_II		(0x00000300)
#define MCF5XXX_CACR_DBWE		(0x00000100)
#define MCF5XXX_CACR_DWP		(0x00000020)
#define MCF5XXX_CACR_EUST		(0x00000010)
#define MCF5XXX_CACR_CLNF_00	(0x00000000)
#define MCF5XXX_CACR_CLNF_01	(0x00000002)
#define MCF5XXX_CACR_CLNF_10	(0x00000004)
#define MCF5XXX_CACR_CLNF_11	(0x00000006)

#define MCF5XXX_ACR_AB(a)		((a)&0xFF000000)
#define MCF5XXX_ACR_AM(a)		(((a)&0xFF000000) >> 8)
#define MCF5XXX_ACR_EN			(0x00008000)
#define MCF5XXX_ACR_SM_USER		(0x00000000)
#define MCF5XXX_ACR_SM_SUPER	(0x00002000)
#define MCF5XXX_ACR_SM_IGNORE	(0x00006000)
#define MCF5XXX_ACR_ENIB		(0x00000080)
#define MCF5XXX_ACR_CM			(0x00000040)
#define MCF5XXX_ACR_DCM_WR		(0x00000000)
#define MCF5XXX_ACR_DCM_CB		(0x00000020)
#define MCF5XXX_ACR_DCM_IP		(0x00000040)
#define MCF5XXX_ACR_DCM_II		(0x00000060)
#define MCF5XXX_ACR_CM			(0x00000040)
#define MCF5XXX_ACR_BWE			(0x00000020)
#define MCF5XXX_ACR_WP			(0x00000004)

#define MCF5XXX_RAMBAR_BA(a)	((a)&0xFFFFC000)
#define MCF5XXX_RAMBAR_PRI_00	(0x00000000)
#define MCF5XXX_RAMBAR_PRI_01	(0x00004000)
#define MCF5XXX_RAMBAR_PRI_10	(0x00008000)
#define MCF5XXX_RAMBAR_PRI_11	(0x0000C000)
#define MCF5XXX_RAMBAR_WP		(0x00000100)
#define MCF5XXX_RAMBAR_CI		(0x00000020)
#define MCF5XXX_RAMBAR_SC		(0x00000010)
#define MCF5XXX_RAMBAR_SD		(0x00000008)
#define MCF5XXX_RAMBAR_UC		(0x00000004)
#define MCF5XXX_RAMBAR_UD		(0x00000002)
#define MCF5XXX_RAMBAR_V		(0x00000001)

/***********************************************************************/
/*
 * The ColdFire family of processors has a simplified exception stack
 * frame that looks like the following:
 *
 *              3322222222221111 111111
 *              1098765432109876 5432109876543210
 *           8 +----------------+----------------+
 *             |         Program Counter         |
 *           4 +----------------+----------------+
 *             |FS/Fmt/Vector/FS|      SR        |
 *   SP -->  0 +----------------+----------------+
 *
 * The stack self-aligns to a 4-byte boundary at an exception, with
 * the FS/Fmt/Vector/FS field indicating the size of the adjustment
 * (SP += 0,1,2,3 bytes).
 */

#define MCF5XXX_RD_SF_FORMAT(PTR)	\
	((*((uint16 *)(PTR)) >> 12) & 0x00FF)

#define MCF5XXX_RD_SF_VECTOR(PTR)	\
	((*((uint16 *)(PTR)) >>  2) & 0x00FF)

#define MCF5XXX_RD_SF_FS(PTR)		\
	( ((*((uint16 *)(PTR)) & 0x0C00) >> 8) | (*((uint16 *)(PTR)) & 0x0003) )

#define MCF5XXX_SF_SR(PTR)	*((uint16 *)(PTR)+1)
#define MCF5XXX_SF_PC(PTR)	*((uint32 *)(PTR)+1)

/********************************************************************/
/*
 * Functions provided by mcf5xxx.s
 */
 
int 	asm_set_ipl (uint32);
void	mcf5xxx_wr_cacr (uint32);
void	mcf5xxx_wr_acr0 (uint32);
void	mcf5xxx_wr_acr1 (uint32);
void	mcf5xxx_wr_acr2 (uint32);
void	mcf5xxx_wr_acr3 (uint32);
void	mcf5xxx_wr_other_a7 (uint32);
void	mcf5xxx_wr_other_sp (uint32);
void	mcf5xxx_wr_vbr (uint32);
void	mcf5xxx_wr_macsr (uint32);
void	mcf5xxx_wr_mask (uint32);
void	mcf5xxx_wr_acc0 (uint32);
void	mcf5xxx_wr_accext01 (uint32);
void	mcf5xxx_wr_accext23 (uint32);
void	mcf5xxx_wr_acc1 (uint32);
void	mcf5xxx_wr_acc2 (uint32);
void	mcf5xxx_wr_acc3 (uint32);
void	mcf5xxx_wr_sr (uint32);
void	mcf5xxx_wr_rambar0 (uint32);
void	mcf5xxx_wr_rambar1 (uint32);
void	mcf5xxx_wr_mbar (uint32);
void	mcf5xxx_wr_mbar0 (uint32);
void	mcf5xxx_wr_mbar1 (uint32);

/********************************************************************/

#endif	/* _CPU_MCF5XXX_H */


/********************************************************************/
/*
 * Memory map definitions from linker command files
 */
extern uint8 __IPSBAR[];

/*********************************************************************
*
* Watchdog Timer Modules (WTM)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_WCR      (*(vuint16*)(void*)(&__IPSBAR[0x140000]))
#define MCF5235_WMR      (*(vuint16*)(void*)(&__IPSBAR[0x140002]))
#define MCF5235_WCNTR    (*(vuint16*)(void*)(&__IPSBAR[0x140004]))
#define MCF5235_WSR      (*(vuint16*)(void*)(&__IPSBAR[0x140006]))

/* Bit definitions and macros for MCF5235_WTM_WCR */
#define MCF5235_WCR_EN        (0x0001)
#define MCF5235_WCR_HALTED    (0x0002)
#define MCF5235_WCR_DOZE      (0x0004)
#define MCF5235_WCR_WAIT      (0x0008)

/* Bit definitions and macros for MCF5235_WTM_WMR */
#define MCF5235_WMR_WM0       (0x0001)
#define MCF5235_WMR_WM1       (0x0002)
#define MCF5235_WMR_WM2       (0x0004)
#define MCF5235_WMR_WM3       (0x0008)
#define MCF5235_WMR_WM4       (0x0010)
#define MCF5235_WMR_WM5       (0x0020)
#define MCF5235_WMR_WM6       (0x0040)
#define MCF5235_WMR_WM7       (0x0080)
#define MCF5235_WMR_WM8       (0x0100)
#define MCF5235_WMR_WM9       (0x0200)
#define MCF5235_WMR_WM10      (0x0400)
#define MCF5235_WMR_WM11      (0x0800)
#define MCF5235_WMR_WM12      (0x1000)
#define MCF5235_WMR_WM13      (0x2000)
#define MCF5235_WMR_WM14      (0x4000)
#define MCF5235_WMR_WM15      (0x8000)

/* Bit definitions and macros for MCF5235_WTM_WCNTR */
#define MCF5235_WCNTR_WC0     (0x0001)
#define MCF5235_WCNTR_WC1     (0x0002)
#define MCF5235_WCNTR_WC2     (0x0004)
#define MCF5235_WCNTR_WC3     (0x0008)
#define MCF5235_WCNTR_WC4     (0x0010)
#define MCF5235_WCNTR_WC5     (0x0020)
#define MCF5235_WCNTR_WC6     (0x0040)
#define MCF5235_WCNTR_WC7     (0x0080)
#define MCF5235_WCNTR_WC8     (0x0100)
#define MCF5235_WCNTR_WC9     (0x0200)
#define MCF5235_WCNTR_WC10    (0x0400)
#define MCF5235_WCNTR_WC11    (0x0800)
#define MCF5235_WCNTR_WC12    (0x1000)
#define MCF5235_WCNTR_WC13    (0x2000)
#define MCF5235_WCNTR_WC14    (0x4000)
#define MCF5235_WCNTR_WC15    (0x8000)
#define MCF5235_WSR_WS0       (0x0001)
#define MCF5235_WSR_WS1       (0x0002)
#define MCF5235_WSR_WS2       (0x0004)
#define MCF5235_WSR_WS3       (0x0008)
#define MCF5235_WSR_WS4       (0x0010)
#define MCF5235_WSR_WS5       (0x0020)
#define MCF5235_WSR_WS6       (0x0040)
#define MCF5235_WSR_WS7       (0x0080)
#define MCF5235_WSR_WS8       (0x0100)
#define MCF5235_WSR_WS9       (0x0200)
#define MCF5235_WSR_WS10      (0x0400)
#define MCF5235_WSR_WS11      (0x0800)
#define MCF5235_WSR_WS12      (0x1000)
#define MCF5235_WSR_WS13      (0x2000)
#define MCF5235_WSR_WS14      (0x4000)
#define MCF5235_WSR_WS15      (0x8000)

/********************************************************************/

/*********************************************************************
*
* Universal Asynchronous Receiver Transmitter (UART)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_UART_UMR0        (*(vuint8 *)(void*)(&__IPSBAR[0x000200]))
#define MCF5235_UART_USR0        (*(vuint8 *)(void*)(&__IPSBAR[0x000204]))
#define MCF5235_UART_UCSR0       (*(vuint8 *)(void*)(&__IPSBAR[0x000204]))
#define MCF5235_UART_UCR0        (*(vuint8 *)(void*)(&__IPSBAR[0x000208]))
#define MCF5235_UART_URB0        (*(vuint8 *)(void*)(&__IPSBAR[0x00020C]))
#define MCF5235_UART_UTB0        (*(vuint8 *)(void*)(&__IPSBAR[0x00020C]))
#define MCF5235_UART_UIPCR0      (*(vuint8 *)(void*)(&__IPSBAR[0x000210]))
#define MCF5235_UART_UACR0       (*(vuint8 *)(void*)(&__IPSBAR[0x000210]))
#define MCF5235_UART_UISR0       (*(vuint8 *)(void*)(&__IPSBAR[0x000214]))
#define MCF5235_UART_UIMR0       (*(vuint8 *)(void*)(&__IPSBAR[0x000214]))
#define MCF5235_UART_UBG10       (*(vuint8 *)(void*)(&__IPSBAR[0x000218]))
#define MCF5235_UART_UBG20       (*(vuint8 *)(void*)(&__IPSBAR[0x00021C]))
#define MCF5235_UART_UIP0        (*(vuint8 *)(void*)(&__IPSBAR[0x000234]))
#define MCF5235_UART_UOP10       (*(vuint8 *)(void*)(&__IPSBAR[0x000238]))
#define MCF5235_UART_UOP00       (*(vuint8 *)(void*)(&__IPSBAR[0x00023C]))
#define MCF5235_UART_UMR1        (*(vuint8 *)(void*)(&__IPSBAR[0x000240]))
#define MCF5235_UART_USR1        (*(vuint8 *)(void*)(&__IPSBAR[0x000244]))
#define MCF5235_UART_UCSR1       (*(vuint8 *)(void*)(&__IPSBAR[0x000244]))
#define MCF5235_UART_UCR1        (*(vuint8 *)(void*)(&__IPSBAR[0x000248]))
#define MCF5235_UART_URB1        (*(vuint8 *)(void*)(&__IPSBAR[0x00024C]))
#define MCF5235_UART_UTB1        (*(vuint8 *)(void*)(&__IPSBAR[0x00024C]))
#define MCF5235_UART_UIPCR1      (*(vuint8 *)(void*)(&__IPSBAR[0x000250]))
#define MCF5235_UART_UACR1       (*(vuint8 *)(void*)(&__IPSBAR[0x000250]))
#define MCF5235_UART_UISR1       (*(vuint8 *)(void*)(&__IPSBAR[0x000254]))
#define MCF5235_UART_UIMR1       (*(vuint8 *)(void*)(&__IPSBAR[0x000254]))
#define MCF5235_UART_UBG11       (*(vuint8 *)(void*)(&__IPSBAR[0x000258]))
#define MCF5235_UART_UBG21       (*(vuint8 *)(void*)(&__IPSBAR[0x00025C]))
#define MCF5235_UART_UIP1        (*(vuint8 *)(void*)(&__IPSBAR[0x000274]))
#define MCF5235_UART_UOP11       (*(vuint8 *)(void*)(&__IPSBAR[0x000278]))
#define MCF5235_UART_UOP01       (*(vuint8 *)(void*)(&__IPSBAR[0x00027C]))
#define MCF5235_UART_UMR2        (*(vuint8 *)(void*)(&__IPSBAR[0x000280]))
#define MCF5235_UART_USR2        (*(vuint8 *)(void*)(&__IPSBAR[0x000284]))
#define MCF5235_UART_UCSR2       (*(vuint8 *)(void*)(&__IPSBAR[0x000284]))
#define MCF5235_UART_UCR2        (*(vuint8 *)(void*)(&__IPSBAR[0x000288]))
#define MCF5235_UART_URB2        (*(vuint8 *)(void*)(&__IPSBAR[0x00028C]))
#define MCF5235_UART_UTB2        (*(vuint8 *)(void*)(&__IPSBAR[0x00028C]))
#define MCF5235_UART_UIPCR2      (*(vuint8 *)(void*)(&__IPSBAR[0x000290]))
#define MCF5235_UART_UACR2       (*(vuint8 *)(void*)(&__IPSBAR[0x000290]))
#define MCF5235_UART_UISR2       (*(vuint8 *)(void*)(&__IPSBAR[0x000294]))
#define MCF5235_UART_UIMR2       (*(vuint8 *)(void*)(&__IPSBAR[0x000294]))
#define MCF5235_UART_UBG12       (*(vuint8 *)(void*)(&__IPSBAR[0x000298]))
#define MCF5235_UART_UBG22       (*(vuint8 *)(void*)(&__IPSBAR[0x00029C]))
#define MCF5235_UART_UIP2        (*(vuint8 *)(void*)(&__IPSBAR[0x0002B4]))
#define MCF5235_UART_UOP12       (*(vuint8 *)(void*)(&__IPSBAR[0x0002B8]))
#define MCF5235_UART_UOP02       (*(vuint8 *)(void*)(&__IPSBAR[0x0002BC]))
#define MCF5235_UART_UMR(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x000200+((x)*0x040)]))
#define MCF5235_UART_USR(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x000204+((x)*0x040)]))
#define MCF5235_UART_UCSR(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x000204+((x)*0x040)]))
#define MCF5235_UART_UCR(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x000208+((x)*0x040)]))
#define MCF5235_UART_URB(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x00020C+((x)*0x040)]))
#define MCF5235_UART_UTB(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x00020C+((x)*0x040)]))
#define MCF5235_UART_UIPCR(x)    (*(vuint8 *)(void*)(&__IPSBAR[0x000210+((x)*0x040)]))
#define MCF5235_UART_UACR(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x000210+((x)*0x040)]))
#define MCF5235_UART_UISR(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x000214+((x)*0x040)]))
#define MCF5235_UART_UIMR(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x000214+((x)*0x040)]))
#define MCF5235_UART_UBG1(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x000218+((x)*0x040)]))
#define MCF5235_UART_UBG2(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x00021C+((x)*0x040)]))
#define MCF5235_UART_UIP(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x000234+((x)*0x040)]))
#define MCF5235_UART_UOP1(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x000238+((x)*0x040)]))
#define MCF5235_UART_UOP0(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x00023C+((x)*0x040)]))

/* Bit definitions and macros for MCF5235_UART_UMR */
#define MCF5235_UART_UMR_BC(x)              (((x)&0x03)<<0)
#define MCF5235_UART_UMR_PT                 (0x04)
#define MCF5235_UART_UMR_PM(x)              (((x)&0x03)<<3)
#define MCF5235_UART_UMR_ERR                (0x20)
#define MCF5235_UART_UMR_RXIRQ              (0x40)
#define MCF5235_UART_UMR_RXRTS              (0x80)
#define MCF5235_UART_UMR_SB(x)              (((x)&0x0F)<<0)
#define MCF5235_UART_UMR_TXCTS              (0x10)
#define MCF5235_UART_UMR_TXRTS              (0x20)
#define MCF5235_UART_UMR_CM(x)              (((x)&0x03)<<6)
#define MCF5235_UART_UMR_PM_MULTI_ADDR      (0x1C)
#define MCF5235_UART_UMR_PM_MULTI_DATA      (0x18)
#define MCF5235_UART_UMR_PM_NONE            (0x10)
#define MCF5235_UART_UMR_PM_FORCE_HI        (0x0C)
#define MCF5235_UART_UMR_PM_FORCE_LO        (0x08)
#define MCF5235_UART_UMR_PM_ODD             (0x04)
#define MCF5235_UART_UMR_PM_EVEN            (0x00)
#define MCF5235_UART_UMR_BC_5               (0x00)
#define MCF5235_UART_UMR_BC_6               (0x01)
#define MCF5235_UART_UMR_BC_7               (0x02)
#define MCF5235_UART_UMR_BC_8               (0x03)
#define MCF5235_UART_UMR_CM_NORMAL          (0x00)
#define MCF5235_UART_UMR_CM_ECHO            (0x40)
#define MCF5235_UART_UMR_CM_LOCAL_LOOP      (0x80)
#define MCF5235_UART_UMR_CM_REMOTE_LOOP     (0xC0)
#define MCF5235_UART_UMR_STOP_BITS_1     (0x07)
#define MCF5235_UART_UMR_STOP_BITS_15    (0x08)
#define MCF5235_UART_UMR_STOP_BITS_2     (0x0F)
#define MCF5235_UART_USR_RXRDY              (0x01)
#define MCF5235_UART_USR_FFULL              (0x02)
#define MCF5235_UART_USR_TXRDY              (0x04)
#define MCF5235_UART_USR_TXEMP              (0x08)
#define MCF5235_UART_USR_OE                 (0x10)
#define MCF5235_UART_USR_PE                 (0x20)
#define MCF5235_UART_USR_FE                 (0x40)
#define MCF5235_UART_USR_RB                 (0x80)
#define MCF5235_UART_UCSR_TCS(x)            (((x)&0x0F)<<0)
#define MCF5235_UART_UCSR_RCS(x)            (((x)&0x0F)<<4)
#define MCF5235_UART_UCSR_RCS_SYS_CLK       (0xD0)
#define MCF5235_UART_UCSR_RCS_CTM16         (0xE0)
#define MCF5235_UART_UCSR_RCS_CTM           (0xF0)
#define MCF5235_UART_UCSR_TCS_SYS_CLK       (0x0D)
#define MCF5235_UART_UCSR_TCS_CTM16         (0x0E)
#define MCF5235_UART_UCSR_TCS_CTM           (0x0F)
#define MCF5235_UART_UCR_RXC(x)             (((x)&0x03)<<0)
#define MCF5235_UART_UCR_TXC(x)             (((x)&0x03)<<2)
#define MCF5235_UART_UCR_MISC(x)            (((x)&0x07)<<4)
#define MCF5235_UART_UCR_NONE               (0x00)
#define MCF5235_UART_UCR_STOP_BREAK         (0x70)
#define MCF5235_UART_UCR_START_BREAK        (0x60)
#define MCF5235_UART_UCR_BKCHGINT           (0x50)
#define MCF5235_UART_UCR_RESET_ERROR        (0x40)
#define MCF5235_UART_UCR_RESET_TX           (0x30)
#define MCF5235_UART_UCR_RESET_RX           (0x20)
#define MCF5235_UART_UCR_RESET_MR           (0x10)
#define MCF5235_UART_UCR_TX_DISABLED        (0x08)
#define MCF5235_UART_UCR_TX_ENABLED         (0x04)
#define MCF5235_UART_UCR_RX_DISABLED        (0x02)
#define MCF5235_UART_UCR_RX_ENABLED         (0x01)
#define MCF5235_UART_UIPCR_CTS              (0x01)
#define MCF5235_UART_UIPCR_COS              (0x10)
#define MCF5235_UART_UACR_IEC               (0x01)
#define MCF5235_UART_UISR_TXRDY             (0x01)
#define MCF5235_UART_UISR_RXRDY             (0x02)
#define MCF5235_UART_UISR_DB                (0x04)
#define MCF5235_UART_UISR_RXFTO             (0x08)
#define MCF5235_UART_UISR_TXFIFO            (0x10)
#define MCF5235_UART_UISR_RXFIFO            (0x20)
#define MCF5235_UART_UISR_COS               (0x80)
#define MCF5235_UART_UIMR_TXRDY             (0x01)
#define MCF5235_UART_UIMR_FFULL             (0x02)
#define MCF5235_UART_UIMR_DB                (0x04)
#define MCF5235_UART_UIMR_COS               (0x80)
#define MCF5235_UART_UIP_CTS                (0x01)
#define MCF5235_UART_UOP1_RTS               (0x01)
#define MCF5235_UART_UOP0_RTS               (0x01)


/*********************************************************************
*
* SDRAM Controller (SDRAMC)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_SDRAMC_DCR      (*(vuint16*)(void*)(&__IPSBAR[0x000040]))
#define MCF5235_SDRAMC_DACR0    (*(vuint32*)(void*)(&__IPSBAR[0x000048]))
#define MCF5235_SDRAMC_DMR0     (*(vuint32*)(void*)(&__IPSBAR[0x00004C]))
#define MCF5235_SDRAMC_DACR1    (*(vuint32*)(void*)(&__IPSBAR[0x000050]))
#define MCF5235_SDRAMC_DMR1     (*(vuint32*)(void*)(&__IPSBAR[0x000054]))

/* Bit definitions and macros for MCF5235_SDRAMC_DCR */
#define MCF5235_SDRAMC_DCR_RC(x)        (((x)&0x01FF)<<0)
#define MCF5235_SDRAMC_DCR_RTIM(x)      (((x)&0x0003)<<9)
#define MCF5235_SDRAMC_DCR_IS           (0x0800)
#define MCF5235_SDRAMC_DCR_COC          (0x1000)
#define MCF5235_SDRAMC_DCR_NAM          (0x2000)
#define MCF5235_SDRAMC_DACR0_IP         (0x00000008)
#define MCF5235_SDRAMC_DACR0_PS(x)      (((x)&0x00000003)<<4)
#define MCF5235_SDRAMC_DACR0_MRS        (0x00000040)
#define MCF5235_SDRAMC_DACR0_CBM(x)     (((x)&0x00000007)<<8)
#define MCF5235_SDRAMC_DACR0_CASL(x)    (((x)&0x00000003)<<12)
#define MCF5235_SDRAMC_DACR0_RE         (0x00008000)
#define MCF5235_SDRAMC_DACR0_BA(x)      (((x)&0x00003FFF)<<18)
#define MCF5235_SDRAMC_DMR0_V           (0x00000001)
#define MCF5235_SDRAMC_DMR0_WP          (0x00000100)
#define MCF5235_SDRAMC_DMR0_BAM(x)      (((x)&0x00003FFF)<<18)
#define MCF5235_SDRAMC_DACR1_IP         (0x00000008)
#define MCF5235_SDRAMC_DACR1_PS(x)      (((x)&0x00000003)<<4)
#define MCF5235_SDRAMC_DACR1_MRS        (0x00000040)
#define MCF5235_SDRAMC_DACR1_CBM(x)     (((x)&0x00000007)<<8)
#define MCF5235_SDRAMC_DACR1_CASL(x)    (((x)&0x00000003)<<12)
#define MCF5235_SDRAMC_DACR1_RE         (0x00008000)
#define MCF5235_SDRAMC_DACR1_BA(x)      (((x)&0x00003FFF)<<18)
#define MCF5235_SDRAMC_DMR1_V           (0x00000001)
#define MCF5235_SDRAMC_DMR1_WP          (0x00000100)
#define MCF5235_SDRAMC_DMR1_BAM(x)      (((x)&0x00003FFF)<<18)
#define MCF5235_SDRAMC_DMR_BAM_4G			(0xFFFC0000)
#define MCF5235_SDRAMC_DMR_BAM_2G			(0x7FFC0000)
#define MCF5235_SDRAMC_DMR_BAM_1G			(0x3FFC0000)
#define MCF5235_SDRAMC_DMR_BAM_1024M		(0x3FFC0000)
#define MCF5235_SDRAMC_DMR_BAM_512M			(0x1FFC0000)
#define MCF5235_SDRAMC_DMR_BAM_256M			(0x0FFC0000)
#define MCF5235_SDRAMC_DMR_BAM_128M			(0x07FC0000)
#define MCF5235_SDRAMC_DMR_BAM_64M			(0x03FC0000)
#define MCF5235_SDRAMC_DMR_BAM_32M			(0x01FC0000)
#define MCF5235_SDRAMC_DMR_BAM_16M			(0x00FC0000)
#define MCF5235_SDRAMC_DMR_BAM_8M			(0x007C0000)
#define MCF5235_SDRAMC_DMR_BAM_4M			(0x003C0000)
#define MCF5235_SDRAMC_DMR_BAM_2M			(0x001C0000)
#define MCF5235_SDRAMC_DMR_BAM_1M			(0x000C0000)
#define MCF5235_SDRAMC_DMR_BAM_1024K		(0x000C0000)
#define MCF5235_SDRAMC_DMR_BAM_512K			(0x00040000)
#define MCF5235_SDRAMC_DMR_BAM_256K			(0x00000000)
#define MCF5235_SDRAMC_DMR_WP				(0x00000100)
#define MCF5235_SDRAMC_DMR_CI				(0x00000040)
#define MCF5235_SDRAMC_DMR_AM				(0x00000020)
#define MCF5235_SDRAMC_DMR_SC				(0x00000010)
#define MCF5235_SDRAMC_DMR_SD				(0x00000008)
#define MCF5235_SDRAMC_DMR_UC				(0x00000004)
#define MCF5235_SDRAMC_DMR_UD				(0x00000002)
#define MCF5235_SDRAMC_DMR_V				(0x00000001)

/*********************************************************************
*
* DMA Timers (TIMER)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_TIMER_DTMR0       (*(vuint16*)(void*)(&__IPSBAR[0x000400]))
#define MCF5235_TIMER_DTXMR0      (*(vuint8 *)(void*)(&__IPSBAR[0x000402]))
#define MCF5235_TIMER_DTER0       (*(vuint8 *)(void*)(&__IPSBAR[0x000403]))
#define MCF5235_TIMER_DTRR0       (*(vuint32*)(void*)(&__IPSBAR[0x000404]))
#define MCF5235_TIMER_DTCR0       (*(vuint32*)(void*)(&__IPSBAR[0x000408]))
#define MCF5235_TIMER_DTCN0       (*(vuint32*)(void*)(&__IPSBAR[0x00040C]))
#define MCF5235_TIMER_DTMR1       (*(vuint16*)(void*)(&__IPSBAR[0x000440]))
#define MCF5235_TIMER_DTXMR1      (*(vuint8 *)(void*)(&__IPSBAR[0x000442]))
#define MCF5235_TIMER_DTER1       (*(vuint8 *)(void*)(&__IPSBAR[0x000443]))
#define MCF5235_TIMER_DTRR1       (*(vuint32*)(void*)(&__IPSBAR[0x000444]))
#define MCF5235_TIMER_DTCR1       (*(vuint32*)(void*)(&__IPSBAR[0x000448]))
#define MCF5235_TIMER_DTCN1       (*(vuint32*)(void*)(&__IPSBAR[0x00044C]))
#define MCF5235_TIMER_DTMR2       (*(vuint16*)(void*)(&__IPSBAR[0x000480]))
#define MCF5235_TIMER_DTXMR2      (*(vuint8 *)(void*)(&__IPSBAR[0x000482]))
#define MCF5235_TIMER_DTER2       (*(vuint8 *)(void*)(&__IPSBAR[0x000483]))
#define MCF5235_TIMER_DTRR2       (*(vuint32*)(void*)(&__IPSBAR[0x000484]))
#define MCF5235_TIMER_DTCR2       (*(vuint32*)(void*)(&__IPSBAR[0x000488]))
#define MCF5235_TIMER_DTCN2       (*(vuint32*)(void*)(&__IPSBAR[0x00048C]))
#define MCF5235_TIMER3_DTMR       (*(vuint16*)(void*)(&__IPSBAR[0x0004C0]))
#define MCF5235_TIMER_DTXMR3      (*(vuint8 *)(void*)(&__IPSBAR[0x0004C2]))
#define MCF5235_TIMER_DTER3       (*(vuint8 *)(void*)(&__IPSBAR[0x0004C3]))
#define MCF5235_TIMER_DTRR3       (*(vuint32*)(void*)(&__IPSBAR[0x0004C4]))
#define MCF5235_TIMER_DTCR3       (*(vuint32*)(void*)(&__IPSBAR[0x0004C8]))
#define MCF5235_TIMER3_DTCN       (*(vuint32*)(void*)(&__IPSBAR[0x0004CC]))
#define MCF5235_TIMER_DTMR(x)     (*(vuint16*)(void*)(&__IPSBAR[0x000400+((x)*0x040)]))
#define MCF5235_TIMER_DTXMR(x)    (*(vuint8 *)(void*)(&__IPSBAR[0x000402+((x)*0x040)]))
#define MCF5235_TIMER_DTER(x)     (*(vuint8 *)(void*)(&__IPSBAR[0x000403+((x)*0x040)]))
#define MCF5235_TIMER_DTRR(x)     (*(vuint32*)(void*)(&__IPSBAR[0x000404+((x)*0x040)]))
#define MCF5235_TIMER_DTCR(x)     (*(vuint32*)(void*)(&__IPSBAR[0x000408+((x)*0x040)]))
#define MCF5235_TIMER_DTCN(x)     (*(vuint32*)(void*)(&__IPSBAR[0x00040C+((x)*0x040)]))

/* Bit definitions and macros for MCF5235_TIMER_DTMR */
#define MCF5235_TIMER_DTMR_RST          (0x0001)
#define MCF5235_TIMER_DTMR_CLK(x)       (((x)&0x0003)<<1)
#define MCF5235_TIMER_DTMR_FRR          (0x0008)
#define MCF5235_TIMER_DTMR_ORRI         (0x0010)
#define MCF5235_TIMER_DTMR_OM           (0x0020)
#define MCF5235_TIMER_DTMR_CE(x)        (((x)&0x0003)<<6)
#define MCF5235_TIMER_DTMR_PS(x)        (((x)&0x00FF)<<8)
#define MCF5235_TIMER_DTMR_CE_ANY       (0x00C0)
#define MCF5235_TIMER_DTMR_CE_FALL      (0x0080)
#define MCF5235_TIMER_DTMR_CE_RISE      (0x0040)
#define MCF5235_TIMER_DTMR_CE_NONE      (0x0000)
#define MCF5235_TIMER_DTMR_CLK_DTIN     (0x0006)
#define MCF5235_TIMER_DTMR_CLK_DIV16    (0x0004)
#define MCF5235_TIMER_DTMR_CLK_DIV1     (0x0002)
#define MCF5235_TIMER_DTMR_CLK_STOP     (0x0000)
#define MCF5235_TIMER_DTXMR_MODE16      (0x01)
#define MCF5235_TIMER_DTXMR_DMAEN       (0x80)
#define MCF5235_TIMER_DTER_CAP          (0x01)
#define MCF5235_TIMER_DTER_REF          (0x02)

/*********************************************************************
*
* System SRAM (SRAM)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_SRAM_RAMBAR    (*(vuint32*)(void*)(&__IPSBAR[0x20000000]))

/* Bit definitions and macros for MCF5235_SRAM_RAMBAR */
#define MCF5235_SRAM_RAMBAR_V        (0x00000001)
#define MCF5235_SRAM_RAMBAR_UD       (0x00000002)
#define MCF5235_SRAM_RAMBAR_UC       (0x00000004)
#define MCF5235_SRAM_RAMBAR_SD       (0x00000008)
#define MCF5235_SRAM_RAMBAR_SC       (0x00000010)
#define MCF5235_SRAM_RAMBAR_C/I      (0x00000020)
#define MCF5235_SRAM_RAMBAR_WP       (0x00000100)
#define MCF5235_SRAM_RAMBAR_SPV      (0x00000200)
#define MCF5235_SRAM_RAMBAR_PRI2     (0x00000400)
#define MCF5235_SRAM_RAMBAR_PRI1     (0x00000800)
#define MCF5235_SRAM_RAMBAR_BA(x)    (((x)&0x0000FFFF)<<16)

/*********************************************************************
*
* System Control Module (SCM)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_SCM_IPSBAR     (*(vuint32*)(void*)(&__IPSBAR[0x000000]))
#define MCF5235_SCM_RAMBAR     (*(vuint32*)(void*)(&__IPSBAR[0x000008]))
#define MCF5235_SCM_CRSR       (*(vuint8 *)(void*)(&__IPSBAR[0x000010]))
#define MCF5235_SCM_CWCR       (*(vuint8 *)(void*)(&__IPSBAR[0x000011]))
#define MCF5235_SCM_LPICR      (*(vuint8 *)(void*)(&__IPSBAR[0x000012]))
#define MCF5235_SCM_CWSR       (*(vuint8 *)(void*)(&__IPSBAR[0x000013]))
#define MCF5235_SCM_DMAREQC    (*(vuint32*)(void*)(&__IPSBAR[0x000014]))
#define MCF5235_SCM_MPARK      (*(vuint32*)(void*)(&__IPSBAR[0x00001C]))
#define MCF5235_SCM_MPR        (*(vuint8 *)(void*)(&__IPSBAR[0x000020]))
#define MCF5235_SCM_PACR0      (*(vuint8 *)(void*)(&__IPSBAR[0x000024]))
#define MCF5235_SCM_PACR1      (*(vuint8 *)(void*)(&__IPSBAR[0x000025]))
#define MCF5235_SCM_PACR2      (*(vuint8 *)(void*)(&__IPSBAR[0x000026]))
#define MCF5235_SCM_PACR3      (*(vuint8 *)(void*)(&__IPSBAR[0x000027]))
#define MCF5235_SCM_PACR4      (*(vuint8 *)(void*)(&__IPSBAR[0x000028]))
#define MCF5235_SCM_PACR5      (*(vuint8 *)(void*)(&__IPSBAR[0x00002A]))
#define MCF5235_SCM_PACR6      (*(vuint8 *)(void*)(&__IPSBAR[0x00002B]))
#define MCF5235_SCM_PACR7      (*(vuint8 *)(void*)(&__IPSBAR[0x00002C]))
#define MCF5235_SCM_PACR8      (*(vuint8 *)(void*)(&__IPSBAR[0x00002E]))
#define MCF5235_SCM_GPACR0     (*(vuint8 *)(void*)(&__IPSBAR[0x000030]))

/* Bit definitions */
#define MCF5235_SCM_IPSBAR_V                 (0x00000001)
#define MCF5235_SCM_IPSBAR_BA(x)             (((x)&0x00000003)<<30)
#define MCF5235_SCM_RAMBAR_BDE               (0x00000200)
#define MCF5235_SCM_RAMBAR_BA(x)             (((x)&0x0000FFFF)<<16)
#define MCF5235_SCM_CRSR_CWDR                (0x20)
#define MCF5235_SCM_CRSR_EXT                 (0x80)
#define MCF5235_SCM_CWCR_CWTIC               (0x01)
#define MCF5235_SCM_CWCR_CWTAVAL             (0x02)
#define MCF5235_SCM_CWCR_CWTA                (0x04)
#define MCF5235_SCM_CWCR_CWT(x)              (((x)&0x07)<<3)
#define MCF5235_SCM_CWCR_CWRI                (0x40)
#define MCF5235_SCM_CWCR_CWE                 (0x80)
#define MCF5235_SCM_LPICR_XLPM_IPL(x)        (((x)&0x07)<<4)
#define MCF5235_SCM_LPICR_ENBSTOP            (0x80)
#define MCF5235_SCM_DMAREQC_DMAC0(x)         (((x)&0x0000000F)<<0)
#define MCF5235_SCM_DMAREQC_DMAC1(x)         (((x)&0x0000000F)<<4)
#define MCF5235_SCM_DMAREQC_DMAC2(x)         (((x)&0x0000000F)<<8)
#define MCF5235_SCM_DMAREQC_DMAC3(x)         (((x)&0x0000000F)<<12)
#define MCF5235_SCM_MPARK_LCKOUT_TIME(x)     (((x)&0x0000000F)<<8)
#define MCF5235_SCM_MPARK_PRKLAST            (0x00001000)
#define MCF5235_SCM_MPARK_TIMEOUT            (0x00002000)
#define MCF5235_SCM_MPARK_FIXED              (0x00004000)
#define MCF5235_SCM_MPARK_M1_PRTY(x)         (((x)&0x00000003)<<16)
#define MCF5235_SCM_MPARK_M0_PRTY(x)         (((x)&0x00000003)<<18)
#define MCF5235_SCM_MPARK_M2_PRTY(x)         (((x)&0x00000003)<<20)
#define MCF5235_SCM_MPARK_M3_PRTY(x)         (((x)&0x00000003)<<22)
#define MCF5235_SCM_MPARK_BCR24BIT           (0x01000000)
#define MCF5235_SCM_MPARK_M2_P_EN            (0x02000000)
#define MCF5235_SCM_MPR_MPR(x)               (((x)&0x0F)<<0)
#define MCF5235_SCM_PACR0_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR0_LOCK0              (0x08)
#define MCF5235_SCM_PACR0_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR0_LOCK1              (0x80)
#define MCF5235_SCM_PACR1_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR1_LOCK0              (0x08)
#define MCF5235_SCM_PACR1_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR1_LOCK1              (0x80)
#define MCF5235_SCM_PACR2_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR2_LOCK0              (0x08)
#define MCF5235_SCM_PACR2_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR2_LOCK1              (0x80)
#define MCF5235_SCM_PACR3_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR3_LOCK0              (0x08)
#define MCF5235_SCM_PACR3_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR3_LOCK1              (0x80)
#define MCF5235_SCM_PACR4_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR4_LOCK0              (0x08)
#define MCF5235_SCM_PACR4_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR4_LOCK1              (0x80)
#define MCF5235_SCM_PACR5_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR5_LOCK0              (0x08)
#define MCF5235_SCM_PACR5_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR5_LOCK1              (0x80)
#define MCF5235_SCM_PACR6_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR6_LOCK0              (0x08)
#define MCF5235_SCM_PACR6_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR6_LOCK1              (0x80)
#define MCF5235_SCM_PACR7_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR7_LOCK0              (0x08)
#define MCF5235_SCM_PACR7_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR7_LOCK1              (0x80)
#define MCF5235_SCM_PACR8_ACCESS_CTRL0(x)    (((x)&0x07)<<0)
#define MCF5235_SCM_PACR8_LOCK0              (0x08)
#define MCF5235_SCM_PACR8_ACCESS_CTRL1(x)    (((x)&0x07)<<4)
#define MCF5235_SCM_PACR8_LOCK1              (0x80)
#define MCF5235_SCM_GPACR0_ACCESS_CTRL(x)    (((x)&0x0F)<<0)
#define MCF5235_SCM_GPACR0_LOCK              (0x80)


/*********************************************************************
*
* FlexCAN Module (CAN)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_CAN_CANMCR0        (*(vuint32*)(void*)(&__IPSBAR[0x1C0000]))
#define MCF5235_CAN_CANCTRL0       (*(vuint32*)(void*)(&__IPSBAR[0x1C0004]))
#define MCF5235_CAN_TIMER0         (*(vuint32*)(void*)(&__IPSBAR[0x1C0008]))
#define MCF5235_CAN_RXGMASK0       (*(vuint32*)(void*)(&__IPSBAR[0x1C0010]))
#define MCF5235_CAN_RX14MASK0      (*(vuint32*)(void*)(&__IPSBAR[0x1C0014]))
#define MCF5235_CAN_RX15MASK0      (*(vuint32*)(void*)(&__IPSBAR[0x1C0018]))
#define MCF5235_CAN_ERRCNT0        (*(vuint32*)(void*)(&__IPSBAR[0x1C001C]))
#define MCF5235_CAN_ERRSTAT0       (*(vuint32*)(void*)(&__IPSBAR[0x1C0020]))
#define MCF5235_CAN_IMASK0         (*(vuint16*)(void*)(&__IPSBAR[0x1C002A]))
#define MCF5235_CAN_IFLAG0         (*(vuint16*)(void*)(&__IPSBAR[0x1C0032]))
#define MCF5235_CAN_CANMCR1        (*(vuint32*)(void*)(&__IPSBAR[0x1F0000]))
#define MCF5235_CAN_CANCTRL1       (*(vuint32*)(void*)(&__IPSBAR[0x1F0004]))
#define MCF5235_CAN_TIMER1         (*(vuint32*)(void*)(&__IPSBAR[0x1F0008]))
#define MCF5235_CAN_RXGMASK1       (*(vuint32*)(void*)(&__IPSBAR[0x1F0010]))
#define MCF5235_CAN_RX14MASK1      (*(vuint32*)(void*)(&__IPSBAR[0x1F0014]))
#define MCF5235_CAN_RX15MASK1      (*(vuint32*)(void*)(&__IPSBAR[0x1F0018]))
#define MCF5235_CAN_ERRCNT1        (*(vuint32*)(void*)(&__IPSBAR[0x1F001C]))
#define MCF5235_CAN_ERRSTAT1       (*(vuint32*)(void*)(&__IPSBAR[0x1F0020]))
#define MCF5235_CAN_IMASK1         (*(vuint16*)(void*)(&__IPSBAR[0x1F002A]))
#define MCF5235_CAN_IFLAG1         (*(vuint16*)(void*)(&__IPSBAR[0x1F0032]))
#define MCF5235_CAN_CANMCR(x)      (*(vuint32*)(void*)(&__IPSBAR[0x1C0000+((x)*0x30000)]))
#define MCF5235_CAN_CANCTRL(x)     (*(vuint32*)(void*)(&__IPSBAR[0x1C0004+((x)*0x30000)]))
#define MCF5235_CAN_TIMER(x)       (*(vuint32*)(void*)(&__IPSBAR[0x1C0008+((x)*0x30000)]))
#define MCF5235_CAN_RXGMASK(x)     (*(vuint32*)(void*)(&__IPSBAR[0x1C0010+((x)*0x30000)]))
#define MCF5235_CAN_RX14MASK(x)    (*(vuint32*)(void*)(&__IPSBAR[0x1C0014+((x)*0x30000)]))
#define MCF5235_CAN_RX15MASK(x)    (*(vuint32*)(void*)(&__IPSBAR[0x1C0018+((x)*0x30000)]))
#define MCF5235_CAN_ERRCNT(x)      (*(vuint32*)(void*)(&__IPSBAR[0x1C001C+((x)*0x30000)]))
#define MCF5235_CAN_ERRSTAT(x)     (*(vuint32*)(void*)(&__IPSBAR[0x1C0020+((x)*0x30000)]))
#define MCF5235_CAN_IMASK(x)       (*(vuint16*)(void*)(&__IPSBAR[0x1C002A+((x)*0x30000)]))
#define MCF5235_CAN_IFLAG(x)       (*(vuint16*)(void*)(&__IPSBAR[0x1C0032+((x)*0x30000)]))

#define MCF5235_CAN_MBUF0_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0080+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_TMSTP(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0082+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C0084+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0088+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0089+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C008A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C008B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C008C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C008D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C008E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_BYTE7(x)      (*(vuint8  *)(void *)(&__IPSBAR[0x1C008F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0090+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_TMSTP(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0092+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_ID(x)			(*(vuint32 *)(void *)(&__IPSBAR[0x1C0094+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0098+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0099+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C009A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C009B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C009C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C009D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C009E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C009F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C00A0+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C00A4+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00A8+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00A9+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00AA+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00AB+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00AC+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00AD+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00AE+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_BYTE7(x)      (*(vuint8  *)(void *)(&__IPSBAR[0x1C00AF+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C00B0+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_ID(x)			(*(vuint32 *)(void *)(&__IPSBAR[0x1C00B4+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00B8+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00B9+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00BA+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00BB+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00BC+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00BD+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00BE+((x)*0x30000)]))
#define MCF5235_CAN_MBUF3_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00BF+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C00C0+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C00C4+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00C8+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00C9+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00CA+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00CB+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00CC+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00CD+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00CE+((x)*0x30000)]))
#define MCF5235_CAN_MBUF4_BYTE7(x)      (*(vuint8  *)(void *)(&__IPSBAR[0x1C00CF+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C00D0+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_ID(x)			(*(vuint32 *)(void *)(&__IPSBAR[0x1C00D4+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00D8+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00D9+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00DA+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00DB+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00DC+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00DD+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00DE+((x)*0x30000)]))
#define MCF5235_CAN_MBUF5_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00DF+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C00E0+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C00E4+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00E8+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00E9+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00EA+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00EB+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00EC+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00ED+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C00EE+((x)*0x30000)]))
#define MCF5235_CAN_MBUF6_BYTE7(x)      (*(vuint8  *)(void *)(&__IPSBAR[0x1C00EF+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C00F0+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_ID(x)			(*(vuint32 *)(void *)(&__IPSBAR[0x1C00F4+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00F8+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00F9+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00FA+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00FB+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00FC+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00FD+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00FE+((x)*0x30000)]))
#define MCF5235_CAN_MBUF7_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C00FF+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0100+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C0104+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0108+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0109+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C010A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C010B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C010C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C010D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C010E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF8_BYTE7(x)      (*(vuint8  *)(void *)(&__IPSBAR[0x1C010F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0100+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_ID(x)			(*(vuint32 *)(void *)(&__IPSBAR[0x1C0114+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0118+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0119+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C011A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C011B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C011C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C011D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C011E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF9_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C011F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0120+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C0124+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0128+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0129+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C012A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C012B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C012C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C012D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C012E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF10_BYTE7(x)     (*(vuint8  *)(void *)(&__IPSBAR[0x1C012F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0130+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_ID(x)		(*(vuint32 *)(void *)(&__IPSBAR[0x1C0134+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0138+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0139+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C013A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C013B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C013C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C013D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C013E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF11_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C013F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0140+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C0144+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0148+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0149+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C014A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C014B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C014C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C014D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C014E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF12_BYTE7(x)     (*(vuint8  *)(void *)(&__IPSBAR[0x1C014F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0150+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_ID(x)		(*(vuint32 *)(void *)(&__IPSBAR[0x1C0154+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0158+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0159+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C015A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C015B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C015C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C015D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C015E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF13_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C015F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0160+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_ID(x) 	    (*(vuint32 *)(void *)(&__IPSBAR[0x1C0164+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE0(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0168+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE1(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C0169+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE2(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C016A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE3(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C016B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE4(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C016C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE5(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C016D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE6(x)	    (*(vuint8  *)(void *)(&__IPSBAR[0x1C016E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF14_BYTE7(x)     (*(vuint8  *)(void *)(&__IPSBAR[0x1C016F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_CTRL(x)		(*(vuint16 *)(void *)(&__IPSBAR[0x1C0170+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_ID(x)		(*(vuint32 *)(void *)(&__IPSBAR[0x1C0174+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE0(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0178+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE1(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C0179+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE2(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C017A+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE3(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C017B+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE4(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C017C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE5(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C017D+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE6(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C017E+((x)*0x30000)]))
#define MCF5235_CAN_MBUF15_BYTE7(x)		(*(vuint8  *)(void *)(&__IPSBAR[0x1C017F+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_DATAL(x)	    (*(vuint32  *)(void *)(&__IPSBAR[0x1C0088+((x)*0x30000)]))
#define MCF5235_CAN_MBUF0_DATAH(x)	    (*(vuint32  *)(void *)(&__IPSBAR[0x1C008C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_DATAL(x)		(*(vuint32  *)(void *)(&__IPSBAR[0x1C0098+((x)*0x30000)]))
#define MCF5235_CAN_MBUF1_DATAH(x)		(*(vuint32  *)(void *)(&__IPSBAR[0x1C009C+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_DATAL(x)	    (*(vuint32  *)(void *)(&__IPSBAR[0x1C00A8+((x)*0x30000)]))
#define MCF5235_CAN_MBUF2_DATAH(x)	    (*(vuint32  *)(void *)(&__IPSBAR[0x1C00AC+((x)*0x30000)]))
																				  
																				  
/* Bit definitions and macros for MCF5235_CAN_CANMCR */								  
#define MCF5235_CAN_CANMCR_MAXMB(x)            (((x)&0x0000000F)<<0)				  
#define MCF5235_CAN_CANMCR_SUPV                (0x00800000)							  
#define MCF5235_CAN_CANMCR_FRZACK              (0x01000000)							  
#define MCF5235_CAN_CANMCR_SOFTRST             (0x02000000)							  
#define MCF5235_CAN_CANMCR_HALT                (0x10000000)							  
#define MCF5235_CAN_CANMCR_FRZ                 (0x40000000)							  
#define MCF5235_CAN_CANMCR_MDIS                (0x80000000)							  
#define MCF5235_CAN_CANCTRL_PROPSEG(x)         (((x)&0x00000007)<<0)
#define MCF5235_CAN_CANCTRL_LOM                (0x00000008)
#define MCF5235_CAN_CANCTRL_LBUF               (0x00000010)
#define MCF5235_CAN_CANCTRL_TSYNC              (0x00000020)
#define MCF5235_CAN_CANCTRL_BOFFREC            (0x00000040)
#define MCF5235_CAN_CANCTRL_SAMP               (0x00000080)
#define MCF5235_CAN_CANCTRL_LPB                (0x00001000)
#define MCF5235_CAN_CANCTRL_CLKSRC             (0x00002000)
#define MCF5235_CAN_CANCTRL_ERRMSK             (0x00004000)
#define MCF5235_CAN_CANCTRL_BOFFMSK            (0x00008000)
#define MCF5235_CAN_CANCTRL_PSEG2(x)           (((x)&0x00000007)<<16)
#define MCF5235_CAN_CANCTRL_PSEG1(x)           (((x)&0x00000007)<<19)
#define MCF5235_CAN_CANCTRL_RJW(x)             (((x)&0x00000003)<<22)
#define MCF5235_CAN_CANCTRL_PRESDIV(x)         (((x)&0x000000FF)<<24)
#define MCF5235_CAN_TIMER_TIMER(x)             (((x)&0x0000FFFF)<<0)
#define MCF5235_CAN_RXGMASK_MI(x)              (((x)&0x1FFFFFFF)<<0)
#define MCF5235_CAN_RX14MASK_MI(x)             (((x)&0x1FFFFFFF)<<0)
#define MCF5235_CAN_RX15MASK_MI(x)             (((x)&0x1FFFFFFF)<<0)
#define MCF5235_CAN_ERRCNT_TXECTR(x)           (((x)&0x000000FF)<<0)
#define MCF5235_CAN_ERRCNT_RXECTR(x)           (((x)&0x000000FF)<<8)
#define MCF5235_CAN_ERRSTAT_WAKINT             (0x00000001)
#define MCF5235_CAN_ERRSTAT_ERRINT             (0x00000002)
#define MCF5235_CAN_ERRSTAT_BOFFINT            (0x00000004)
#define MCF5235_CAN_ERRSTAT_FLTCONF(x)         (((x)&0x00000003)<<4)
#define MCF5235_CAN_ERRSTAT_TXRX               (0x00000040)
#define MCF5235_CAN_ERRSTAT_IDLE               (0x00000080)
#define MCF5235_CAN_ERRSTAT_RXWRN              (0x00000100)
#define MCF5235_CAN_ERRSTAT_TXWRN              (0x00000200)
#define MCF5235_CAN_ERRSTAT_STFERR             (0x00000400)
#define MCF5235_CAN_ERRSTAT_FRMERR             (0x00000800)
#define MCF5235_CAN_ERRSTAT_CRCERR             (0x00001000)
#define MCF5235_CAN_ERRSTAT_ACKERR             (0x00002000)
#define MCF5235_CAN_ERRSTAT_BITERR(x)          (((x)&0x00000003)<<14)
#define MCF5235_CAN_ERRSTAT_FLTCONF_ACTIVE     (0x00000000)
#define MCF5235_CAN_ERRSTAT_FLTCONF_PASSIVE    (0x00000010)
#define MCF5235_CAN_ERRSTAT_FLTCONF_BUSOFF     (0x00000020)
#define MCF5235_CAN_IMASK_BUF0M                (0x0001)
#define MCF5235_CAN_IMASK_BUF1M                (0x0002)
#define MCF5235_CAN_IMASK_BUF2M                (0x0004)
#define MCF5235_CAN_IMASK_BUF3M                (0x0008)
#define MCF5235_CAN_IMASK_BUF4M                (0x0010)
#define MCF5235_CAN_IMASK_BUF5M                (0x0020)
#define MCF5235_CAN_IMASK_BUF6M                (0x0040)
#define MCF5235_CAN_IMASK_BUF7M                (0x0080)
#define MCF5235_CAN_IMASK_BUF8M                (0x0100)
#define MCF5235_CAN_IMASK_BUF9M                (0x0200)
#define MCF5235_CAN_IMASK_BUF10M               (0x0400)
#define MCF5235_CAN_IMASK_BUF11M               (0x0800)
#define MCF5235_CAN_IMASK_BUF12M               (0x1000)
#define MCF5235_CAN_IMASK_BUF13M               (0x2000)
#define MCF5235_CAN_IMASK_BUF14M               (0x4000)
#define MCF5235_CAN_IMASK_BUF15M               (0x8000)

/* Bit definitions and macros for MCF5235_CAN_IFLAG */
#define MCF5235_CAN_IFLAG_BUF0I                (0x0001)
#define MCF5235_CAN_IFLAG_BUF1I                (0x0002)
#define MCF5235_CAN_IFLAG_BUF2I                (0x0004)
#define MCF5235_CAN_IFLAG_BUF3I                (0x0008)
#define MCF5235_CAN_IFLAG_BUF4I                (0x0010)
#define MCF5235_CAN_IFLAG_BUF5I                (0x0020)
#define MCF5235_CAN_IFLAG_BUF6I                (0x0040)
#define MCF5235_CAN_IFLAG_BUF7I                (0x0080)
#define MCF5235_CAN_IFLAG_BUF8I                (0x0100)
#define MCF5235_CAN_IFLAG_BUF9I                (0x0200)
#define MCF5235_CAN_IFLAG_BUF10I               (0x0400)
#define MCF5235_CAN_IFLAG_BUF11I               (0x0800)
#define MCF5235_CAN_IFLAG_BUF12I               (0x1000)
#define MCF5235_CAN_IFLAG_BUF13I               (0x2000)
#define MCF5235_CAN_IFLAG_BUF14I               (0x4000)
#define MCF5235_CAN_IFLAG_BUF15I               (0x8000)


/*********************************************************************
*
* Chip Configuration Module (CCM)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_CCM_CCR     (*(vuint16*)(void*)(&__IPSBAR[0x110004]))
#define MCF5235_CCM_LPCR    (*(vuint8 *)(void*)(&__IPSBAR[0x110007]))
#define MCF5235_CCM_CIR     (*(vuint16*)(void*)(&__IPSBAR[0x11000A]))
#define MCF5235_CCM_RCON    (*(vuint16*)(void*)(&__IPSBAR[0x110008]))

/* Bit definitions and macros for MCF5235_CCM_CCR */
#define MCF5235_CCM_CCR_BMT(x)        (((x)&0x0007)<<0)
#define MCF5235_CCM_CCR_BME           (0x0008)
#define MCF5235_CCM_CCR_SZEN          (0x0040)
#define MCF5235_CCM_CCR_MODE(x)       (((x)&0x0007)<<8)
#define MCF5235_CCM_LPCR_STPMD(x)     (((x)&0x03)<<3)
#define MCF5235_CCM_LPCR_LPMD(x)      (((x)&0x03)<<6)
#define MCF5235_CCM_LPCR_LPMD_STOP    (0xC0)
#define MCF5235_CCM_LPCR_LPMD_WAIT    (0x80)
#define MCF5235_CCM_LPCR_LPMD_DOZE    (0x40)
#define MCF5235_CCM_LPCR_LPMD_RUN     (0x00)
#define MCF5235_CCM_CIR_PRN(x)        (((x)&0x003F)<<0)
#define MCF5235_CCM_CIR_PIN(x)        (((x)&0x03FF)<<6)
#define MCF5235_CCM_RCON_MODE         (0x0001)
#define MCF5235_CCM_RCON_BOOTPS(x)    (((x)&0x0003)<<3)
#define MCF5235_CCM_RCON_RLOAD        (0x0020)
#define MCF5235_CCM_RCON_RCSC(x)      (((x)&0x0003)<<8)

/*********************************************************************
*
* Chip Selects (CS)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_CS_CSAR0      (*(vuint16*)(void*)(&__IPSBAR[0x000080]))
#define MCF5235_CS_CSMR0      (*(vuint32*)(void*)(&__IPSBAR[0x000084]))
#define MCF5235_CS_CSCR0      (*(vuint16*)(void*)(&__IPSBAR[0x00008A]))
#define MCF5235_CS_CSAR1      (*(vuint16*)(void*)(&__IPSBAR[0x00008C]))
#define MCF5235_CS_CSMR1      (*(vuint32*)(void*)(&__IPSBAR[0x000090]))
#define MCF5235_CS_CSCR1      (*(vuint16*)(void*)(&__IPSBAR[0x000096]))
#define MCF5235_CS_CSAR2      (*(vuint16*)(void*)(&__IPSBAR[0x000098]))
#define MCF5235_CS_CSMR2      (*(vuint32*)(void*)(&__IPSBAR[0x00009C]))
#define MCF5235_CS_CSCR2      (*(vuint16*)(void*)(&__IPSBAR[0x0000A2]))
#define MCF5235_CS_CSAR3      (*(vuint16*)(void*)(&__IPSBAR[0x0000A4]))
#define MCF5235_CS_CSMR3      (*(vuint32*)(void*)(&__IPSBAR[0x0000A8]))
#define MCF5235_CS_CSCR3      (*(vuint16*)(void*)(&__IPSBAR[0x0000AE]))
#define MCF5235_CS_CSAR4      (*(vuint16*)(void*)(&__IPSBAR[0x0000B0]))
#define MCF5235_CS_CSMR4      (*(vuint32*)(void*)(&__IPSBAR[0x0000B4]))
#define MCF5235_CS_CSCR4      (*(vuint16*)(void*)(&__IPSBAR[0x0000BA]))
#define MCF5235_CS_CSAR5      (*(vuint16*)(void*)(&__IPSBAR[0x0000BC]))
#define MCF5235_CS_CSMR5      (*(vuint32*)(void*)(&__IPSBAR[0x0000C0]))
#define MCF5235_CS_CSCR5      (*(vuint16*)(void*)(&__IPSBAR[0x0000C6]))
#define MCF5235_CS_CSAR6      (*(vuint16*)(void*)(&__IPSBAR[0x0000C8]))
#define MCF5235_CS_CSMR6      (*(vuint32*)(void*)(&__IPSBAR[0x0000CC]))
#define MCF5235_CS_CSCR6      (*(vuint16*)(void*)(&__IPSBAR[0x0000D2]))
#define MCF5235_CS_CSAR7      (*(vuint16*)(void*)(&__IPSBAR[0x0000D4]))
#define MCF5235_CS_CSMR7      (*(vuint32*)(void*)(&__IPSBAR[0x0000D8]))
#define MCF5235_CS_CSCR7      (*(vuint16*)(void*)(&__IPSBAR[0x0000DE]))
#define MCF5235_CS_CSAR(x)    (*(vuint16*)(void*)(&__IPSBAR[0x000080+((x)*0x00C)]))
#define MCF5235_CS_CSMR(x)    (*(vuint32*)(void*)(&__IPSBAR[0x000084+((x)*0x00C)]))
#define MCF5235_CS_CSCR(x)    (*(vuint16*)(void*)(&__IPSBAR[0x00008A+((x)*0x00C)]))

/* Bit definitions and macros for MCF5235_CS_CSAR */
#define MCF5235_CS_CSAR_BA(x)        ((uint16)(((x)&0xFFFF0000)>>16))
#define MCF5235_CS_CSMR_V            (0x00000001)
#define MCF5235_CS_CSMR_UD           (0x00000002)
#define MCF5235_CS_CSMR_UC           (0x00000004)
#define MCF5235_CS_CSMR_SD           (0x00000008)
#define MCF5235_CS_CSMR_SC           (0x00000010)
#define MCF5235_CS_CSMR_CI           (0x00000020)
#define MCF5235_CS_CSMR_AM           (0x00000040)
#define MCF5235_CS_CSMR_WP           (0x00000100)
#define MCF5235_CS_CSMR_BAM(x)       (((x)&0x0000FFFF)<<16)
#define MCF5235_CS_CSMR_BAM_4G       (0xFFFF0000)
#define MCF5235_CS_CSMR_BAM_2G       (0x7FFF0000)
#define MCF5235_CS_CSMR_BAM_1G       (0x3FFF0000)
#define MCF5235_CS_CSMR_BAM_1024M    (0x3FFF0000)
#define MCF5235_CS_CSMR_BAM_512M     (0x1FFF0000)
#define MCF5235_CS_CSMR_BAM_256M     (0x0FFF0000)
#define MCF5235_CS_CSMR_BAM_128M     (0x07FF0000)
#define MCF5235_CS_CSMR_BAM_64M      (0x03FF0000)
#define MCF5235_CS_CSMR_BAM_32M      (0x01FF0000)
#define MCF5235_CS_CSMR_BAM_16M      (0x00FF0000)
#define MCF5235_CS_CSMR_BAM_8M       (0x007F0000)
#define MCF5235_CS_CSMR_BAM_4M       (0x003F0000)
#define MCF5235_CS_CSMR_BAM_2M       (0x001F0000)
#define MCF5235_CS_CSMR_BAM_1M       (0x000F0000)
#define MCF5235_CS_CSMR_BAM_1024K    (0x000F0000)
#define MCF5235_CS_CSMR_BAM_512K     (0x00070000)
#define MCF5235_CS_CSMR_BAM_256K     (0x00030000)
#define MCF5235_CS_CSMR_BAM_128K     (0x00010000)
#define MCF5235_CS_CSMR_BAM_64K      (0x00000000)
#define MCF5235_CS_CSCR_SWWS(x)      (((x)&0x0007)<<0)
#define MCF5235_CS_CSCR_BSTW         (0x0008)
#define MCF5235_CS_CSCR_BSTR         (0x0010)
#define MCF5235_CS_CSCR_BEM          (0x0020)
#define MCF5235_CS_CSCR_PS(x)        (((x)&0x0003)<<6)
#define MCF5235_CS_CSCR_AA           (0x0100)
#define MCF5235_CS_CSCR_IWS(x)       (((x)&0x000F)<<10)
#define MCF5235_CS_CSCR_SRWS(x)      (((x)&0x0003)<<14)
#define MCF5235_CS_CSCR_PS_8         (0x0040)
#define MCF5235_CS_CSCR_PS_16        (0x0080)
#define MCF5235_CS_CSCR_PS_32        (0x0000)

/*********************************************************************
*
* Edge Port Module (EPORT)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_EPORT_EPPAR    (*(vuint16*)(void*)(&__IPSBAR[0x130000]))
#define MCF5235_EPORT_EPDDR    (*(vuint8 *)(void*)(&__IPSBAR[0x130002]))
#define MCF5235_EPORT_EPIER    (*(vuint8 *)(void*)(&__IPSBAR[0x130003]))
#define MCF5235_EPORT_EPDR     (*(vuint8 *)(void*)(&__IPSBAR[0x130004]))
#define MCF5235_EPORT_EPPDR    (*(vuint8 *)(void*)(&__IPSBAR[0x130005]))
#define MCF5235_EPORT_EPFR     (*(vuint8 *)(void*)(&__IPSBAR[0x130006]))

/* Bit definitions and macros for MCF5235_EPORT_EPPAR */
#define MCF5235_EPORT_EPPAR_EPPA1(x)         (((x)&0x0003)<<2)
#define MCF5235_EPORT_EPPAR_EPPA2(x)         (((x)&0x0003)<<4)
#define MCF5235_EPORT_EPPAR_EPPA3(x)         (((x)&0x0003)<<6)
#define MCF5235_EPORT_EPPAR_EPPA4(x)         (((x)&0x0003)<<8)
#define MCF5235_EPORT_EPPAR_EPPA5(x)         (((x)&0x0003)<<10)
#define MCF5235_EPORT_EPPAR_EPPA6(x)         (((x)&0x0003)<<12)
#define MCF5235_EPORT_EPPAR_EPPA7(x)         (((x)&0x0003)<<14)
#define MCF5235_EPORT_EPPAR_EPPAx_LEVEL      (0)
#define MCF5235_EPORT_EPPAR_EPPAx_RISING     (1)
#define MCF5235_EPORT_EPPAR_EPPAx_FALLING    (2)
#define MCF5235_EPORT_EPPAR_EPPAx_BOTH       (3)
#define MCF5235_EPORT_EPDDR_EPDD1            (0x02)
#define MCF5235_EPORT_EPDDR_EPDD2            (0x04)
#define MCF5235_EPORT_EPDDR_EPDD3            (0x08)
#define MCF5235_EPORT_EPDDR_EPDD4            (0x10)
#define MCF5235_EPORT_EPDDR_EPDD5            (0x20)
#define MCF5235_EPORT_EPDDR_EPDD6            (0x40)
#define MCF5235_EPORT_EPDDR_EPDD7            (0x80)
#define MCF5235_EPORT_EPIER_EPIE1            (0x02)
#define MCF5235_EPORT_EPIER_EPIE2            (0x04)
#define MCF5235_EPORT_EPIER_EPIE3            (0x08)
#define MCF5235_EPORT_EPIER_EPIE4            (0x10)
#define MCF5235_EPORT_EPIER_EPIE5            (0x20)
#define MCF5235_EPORT_EPIER_EPIE6            (0x40)
#define MCF5235_EPORT_EPIER_EPIE7            (0x80)
#define MCF5235_EPORT_EPDR_EPD1              (0x02)
#define MCF5235_EPORT_EPDR_EPD2              (0x04)
#define MCF5235_EPORT_EPDR_EPD3              (0x08)
#define MCF5235_EPORT_EPDR_EPD4              (0x10)
#define MCF5235_EPORT_EPDR_EPD5              (0x20)
#define MCF5235_EPORT_EPDR_EPD6              (0x40)
#define MCF5235_EPORT_EPDR_EPD7              (0x80)
#define MCF5235_EPORT_EPPDR_EPPD1            (0x02)
#define MCF5235_EPORT_EPPDR_EPPD2            (0x04)
#define MCF5235_EPORT_EPPDR_EPPD3            (0x08)
#define MCF5235_EPORT_EPPDR_EPPD4            (0x10)
#define MCF5235_EPORT_EPPDR_EPPD5            (0x20)
#define MCF5235_EPORT_EPPDR_EPPD6            (0x40)
#define MCF5235_EPORT_EPPDR_EPPD7            (0x80)
#define MCF5235_EPORT_EPFR_EPF1              (0x02)
#define MCF5235_EPORT_EPFR_EPF2              (0x04)
#define MCF5235_EPORT_EPFR_EPF3              (0x08)
#define MCF5235_EPORT_EPFR_EPF4              (0x10)
#define MCF5235_EPORT_EPFR_EPF5              (0x20)
#define MCF5235_EPORT_EPFR_EPF6              (0x40)
#define MCF5235_EPORT_EPFR_EPF7              (0x80)


/*********************************************************************
*
* enhanced Time Processor Unit (ETPU)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_ETPU_EMCR          (*(vuint32*)(void*)(&__IPSBAR[0x1D0000]))
#define MCF5235_ETPU_ECDCR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0004]))
#define MCF5235_ETPU_EMISCCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D000C]))
#define MCF5235_ETPU_ESCMODR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0010]))
#define MCF5235_ETPU_EECR          (*(vuint32*)(void*)(&__IPSBAR[0x1D0014]))
#define MCF5235_ETPU_ETBCR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0020]))
#define MCF5235_ETPU_ETB1R         (*(vuint32*)(void*)(&__IPSBAR[0x1D0024]))
#define MCF5235_ETPU_ETB2R         (*(vuint32*)(void*)(&__IPSBAR[0x1D0028]))
#define MCF5235_ETPU_EREDCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D002C]))
#define MCF5235_ETPU_ECISR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0200]))
#define MCF5235_ETPU_ECDTRSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0210]))
#define MCF5235_ETPU_ECIOSR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0220]))
#define MCF5235_ETPU_ECDTROSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0230]))
#define MCF5235_ETPU_ECIER         (*(vuint32*)(void*)(&__IPSBAR[0x1D0240]))
#define MCF5235_ETPU_ECDTRER       (*(vuint32*)(void*)(&__IPSBAR[0x1D0250]))
#define MCF5235_ETPU_ECPSSR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0280]))
#define MCF5235_ETPU_ECSSR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0290]))
#define MCF5235_ETPU_EC0SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0404]))
#define MCF5235_ETPU_EC1SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0414]))
#define MCF5235_ETPU_EC2SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0424]))
#define MCF5235_ETPU_EC3SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0434]))
#define MCF5235_ETPU_EC4SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0444]))
#define MCF5235_ETPU_EC5SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0454]))
#define MCF5235_ETPU_EC6SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0464]))
#define MCF5235_ETPU_EC7SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0474]))
#define MCF5235_ETPU_EC8SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0484]))
#define MCF5235_ETPU_EC9SCR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0494]))
#define MCF5235_ETPU_EC10SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D04A4]))
#define MCF5235_ETPU_EC11SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D04B4]))
#define MCF5235_ETPU_EC12SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D04C4]))
#define MCF5235_ETPU_EC13SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D04D4]))
#define MCF5235_ETPU_EC14SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D04E4]))
#define MCF5235_ETPU_EC15SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D04F4]))
#define MCF5235_ETPU_EC16SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0504]))
#define MCF5235_ETPU_EC17SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0514]))
#define MCF5235_ETPU_EC18SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0524]))
#define MCF5235_ETPU_EC19SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0534]))
#define MCF5235_ETPU_EC20SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0544]))
#define MCF5235_ETPU_EC21SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0554]))
#define MCF5235_ETPU_EC22SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0564]))
#define MCF5235_ETPU_EC23SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0574]))
#define MCF5235_ETPU_EC24SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0584]))
#define MCF5235_ETPU_EC25SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0594]))
#define MCF5235_ETPU_EC26SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D05A4]))
#define MCF5235_ETPU_EC27SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D05B4]))
#define MCF5235_ETPU_EC28SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D05C4]))
#define MCF5235_ETPU_EC29SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D05D4]))
#define MCF5235_ETPU_EC30SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D05E4]))
#define MCF5235_ETPU_EC31SCR       (*(vuint32*)(void*)(&__IPSBAR[0x1D05F4]))
#define MCF5235_ETPU_ECnSCR(x)     (*(vuint32*)(void*)(&__IPSBAR[0x1D0404+((x)*0x010)]))
#define MCF5235_ETPU_EC0CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0400]))
#define MCF5235_ETPU_EC1CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0410]))
#define MCF5235_ETPU_EC2CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0420]))
#define MCF5235_ETPU_EC3CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0430]))
#define MCF5235_ETPU_EC4CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0440]))
#define MCF5235_ETPU_EC5CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0450]))
#define MCF5235_ETPU_EC6CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0460]))
#define MCF5235_ETPU_EC7CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0470]))
#define MCF5235_ETPU_EC8CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0480]))
#define MCF5235_ETPU_EC9CR         (*(vuint32*)(void*)(&__IPSBAR[0x1D0490]))
#define MCF5235_ETPU_EC10CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D04A0]))
#define MCF5235_ETPU_EC11CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D04B0]))
#define MCF5235_ETPU_EC12CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D04C0]))
#define MCF5235_ETPU_EC13CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D04D0]))
#define MCF5235_ETPU_EC14CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D04E0]))
#define MCF5235_ETPU_EC15CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D04F0]))
#define MCF5235_ETPU_EC16CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0500]))
#define MCF5235_ETPU_EC17CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0510]))
#define MCF5235_ETPU_EC18CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0520]))
#define MCF5235_ETPU_EC19CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0530]))
#define MCF5235_ETPU_EC20CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0540]))
#define MCF5235_ETPU_EC21CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0550]))
#define MCF5235_ETPU_EC22CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0560]))
#define MCF5235_ETPU_EC23CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0570]))
#define MCF5235_ETPU_EC24CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0580]))
#define MCF5235_ETPU_EC25CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D0590]))
#define MCF5235_ETPU_EC26CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D05A0]))
#define MCF5235_ETPU_EC27CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D05B0]))
#define MCF5235_ETPU_EC28CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D05C0]))
#define MCF5235_ETPU_EC29CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D05D0]))
#define MCF5235_ETPU_EC30CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D05E0]))
#define MCF5235_ETPU_EC31CR        (*(vuint32*)(void*)(&__IPSBAR[0x1D05F0]))
#define MCF5235_ETPU_ECnCR(x)      (*(vuint32*)(void*)(&__IPSBAR[0x1D0400+((x)*0x010)]))
#define MCF5235_ETPU_EC0HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0408]))
#define MCF5235_ETPU_EC1HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0418]))
#define MCF5235_ETPU_EC2HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0428]))
#define MCF5235_ETPU_EC3HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0438]))
#define MCF5235_ETPU_EC4HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0448]))
#define MCF5235_ETPU_EC5HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0458]))
#define MCF5235_ETPU_EC6HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0468]))
#define MCF5235_ETPU_EC7HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0478]))
#define MCF5235_ETPU_EC8HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0488]))
#define MCF5235_ETPU_EC9HSSR       (*(vuint32*)(void*)(&__IPSBAR[0x1D0498]))
#define MCF5235_ETPU_EC10HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D04A8]))
#define MCF5235_ETPU_EC11HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D04B8]))
#define MCF5235_ETPU_EC12HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D04C8]))
#define MCF5235_ETPU_EC13HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D04D8]))
#define MCF5235_ETPU_EC14HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D04E8]))
#define MCF5235_ETPU_EC15HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D04F8]))
#define MCF5235_ETPU_EC16HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0508]))
#define MCF5235_ETPU_EC17HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0518]))
#define MCF5235_ETPU_EC18HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0528]))
#define MCF5235_ETPU_EC19HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0538]))
#define MCF5235_ETPU_EC20HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0548]))
#define MCF5235_ETPU_EC21HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0558]))
#define MCF5235_ETPU_EC22HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0568]))
#define MCF5235_ETPU_EC23HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0578]))
#define MCF5235_ETPU_EC24HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0588]))
#define MCF5235_ETPU_EC25HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D0598]))
#define MCF5235_ETPU_EC26HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D05A8]))
#define MCF5235_ETPU_EC27HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D05B8]))
#define MCF5235_ETPU_EC28HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D05C8]))
#define MCF5235_ETPU_EC29HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D05D8]))
#define MCF5235_ETPU_EC30HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D05E8]))
#define MCF5235_ETPU_EC31HSSR      (*(vuint32*)(void*)(&__IPSBAR[0x1D05F8]))
#define MCF5235_ETPU_ECnHSSR(x)    (*(vuint32*)(void*)(&__IPSBAR[0x1D0408+((x)*0x010)]))

/* Bit definitions and macros for MCF5235_ETPU_EMCR */
#define MCF5235_ETPU_EMCR_GTBE               (0x00000001)
#define MCF5235_ETPU_EMCR_VIS                (0x00000040)
#define MCF5235_ETPU_EMCR_SCMMISEN           (0x00000200)
#define MCF5235_ETPU_EMCR_SCMMISF            (0x00000400)
#define MCF5235_ETPU_EMCR_SCMSIZE(x)         (((x)&0x0000001F)<<16)
#define MCF5235_ETPU_EMCR_ILF2               (0x01000000)
#define MCF5235_ETPU_EMCR_ILF1               (0x02000000)
#define MCF5235_ETPU_EMCR_MGE2               (0x04000000)
#define MCF5235_ETPU_EMCR_MGE1               (0x08000000)
#define MCF5235_ETPU_EMCR_GEC                (0x80000000)
#define MCF5235_ETPU_ECDCR_PARM1(x)          (((x)&0x0000007F)<<0)
#define MCF5235_ETPU_ECDCR_WR                (0x00000080)
#define MCF5235_ETPU_ECDCR_PARM0(x)          (((x)&0x0000007F)<<8)
#define MCF5235_ETPU_ECDCR_PWIDTH            (0x00008000)
#define MCF5235_ETPU_ECDCR_PBASE(x)          (((x)&0x000003FF)<<16)
#define MCF5235_ETPU_ECDCR_CTBASE(x)         (((x)&0x0000001F)<<26)
#define MCF5235_ETPU_ECDCR_STS               (0x80000000)
#define MCF5235_ETPU_EECR_ETB(x)             (((x)&0x0000001F)<<0)
#define MCF5235_ETPU_EECR_CDFC(x)            (((x)&0x00000003)<<14)
#define MCF5235_ETPU_EECR_FPSK(x)            (((x)&0x00000007)<<16)
#define MCF5235_ETPU_EECR_HLTF               (0x00800000)
#define MCF5235_ETPU_EECR_STF                (0x10000000)
#define MCF5235_ETPU_EECR_MDIS               (0x40000000)
#define MCF5235_ETPU_EECR_FEND               (0x80000000)
#define MCF5235_ETPU_ETBCR_TCR1P(x)          (((x)&0x000000FF)<<0)
#define MCF5235_ETPU_ETBCR_TCR1CTL(x)        (((x)&0x00000003)<<14)
#define MCF5235_ETPU_ETBCR_TCR2P(x)          (((x)&0x0000003F)<<16)
#define MCF5235_ETPU_ETBCR_AM                (0x02000000)
#define MCF5235_ETPU_ETBCR_TCRCF(x)          (((x)&0x00000003)<<27)
#define MCF5235_ETPU_ETBCR_TCR2CTL(x)        (((x)&0x00000007)<<29)
#define MCF5235_ETPU_ETB1R_TCR1(x)           (((x)&0x00FFFFFF)<<0)
#define MCF5235_ETPU_ETB2R_TCR2(x)           (((x)&0x00FFFFFF)<<0)
#define MCF5235_ETPU_EREDCR_SRV2(x)          (((x)&0x0000000F)<<0)
#define MCF5235_ETPU_EREDCR_SERVER_ID2(x)    (((x)&0x0000000F)<<8)
#define MCF5235_ETPU_EREDCR_RSC2             (0x00004000)
#define MCF5235_ETPU_EREDCR_REN2             (0x00008000)
#define MCF5235_ETPU_EREDCR_SRV1(x)          (((x)&0x0000000F)<<16)
#define MCF5235_ETPU_EREDCR_SERVER_ID1(x)    (((x)&0x0000000F)<<24)
#define MCF5235_ETPU_EREDCR_RSC1             (0x40000000)
#define MCF5235_ETPU_EREDCR_REN1             (0x80000000)
#define MCF5235_ETPU_ECISR_CIS0              (0x00000001)
#define MCF5235_ETPU_ECISR_CIS1              (0x00000002)
#define MCF5235_ETPU_ECISR_CIS2              (0x00000004)
#define MCF5235_ETPU_ECISR_CIS3              (0x00000008)
#define MCF5235_ETPU_ECISR_CIS4              (0x00000010)
#define MCF5235_ETPU_ECISR_CIS5              (0x00000020)
#define MCF5235_ETPU_ECISR_CIS6              (0x00000040)
#define MCF5235_ETPU_ECISR_CIS7              (0x00000080)
#define MCF5235_ETPU_ECISR_CIS8              (0x00000100)
#define MCF5235_ETPU_ECISR_CIS9              (0x00000200)
#define MCF5235_ETPU_ECISR_CIS10             (0x00000400)
#define MCF5235_ETPU_ECISR_CIS11             (0x00000800)
#define MCF5235_ETPU_ECISR_CIS12             (0x00001000)
#define MCF5235_ETPU_ECISR_CIS13             (0x00002000)
#define MCF5235_ETPU_ECISR_CIS14             (0x00004000)
#define MCF5235_ETPU_ECISR_CIS15             (0x00008000)
#define MCF5235_ETPU_ECISR_CIS16             (0x00010000)
#define MCF5235_ETPU_ECISR_CIS17             (0x00020000)
#define MCF5235_ETPU_ECISR_CIS18             (0x00040000)
#define MCF5235_ETPU_ECISR_CIS19             (0x00080000)
#define MCF5235_ETPU_ECISR_CIS20             (0x00100000)
#define MCF5235_ETPU_ECISR_CIS21             (0x00200000)
#define MCF5235_ETPU_ECISR_CIS22             (0x00400000)
#define MCF5235_ETPU_ECISR_CIS23             (0x00800000)
#define MCF5235_ETPU_ECISR_CIS24             (0x01000000)
#define MCF5235_ETPU_ECISR_CIS25             (0x02000000)
#define MCF5235_ETPU_ECISR_CIS26             (0x04000000)
#define MCF5235_ETPU_ECISR_CIS27             (0x08000000)
#define MCF5235_ETPU_ECISR_CIS28             (0x10000000)
#define MCF5235_ETPU_ECISR_CIS29             (0x20000000)
#define MCF5235_ETPU_ECISR_CIS30             (0x40000000)
#define MCF5235_ETPU_ECISR_CIS31             (0x80000000)
#define MCF5235_ETPU_ECDTRSR_DTRS0           (0x00000001)
#define MCF5235_ETPU_ECDTRSR_DTRS1           (0x00000002)
#define MCF5235_ETPU_ECDTRSR_DTRS2           (0x00000004)
#define MCF5235_ETPU_ECDTRSR_DTRS3           (0x00000008)
#define MCF5235_ETPU_ECDTRSR_DTRS4           (0x00000010)
#define MCF5235_ETPU_ECDTRSR_DTRS5           (0x00000020)
#define MCF5235_ETPU_ECDTRSR_DTRS6           (0x00000040)
#define MCF5235_ETPU_ECDTRSR_DTRS7           (0x00000080)
#define MCF5235_ETPU_ECDTRSR_DTRS8           (0x00000100)
#define MCF5235_ETPU_ECDTRSR_DTRS9           (0x00000200)
#define MCF5235_ETPU_ECDTRSR_DTRS10          (0x00000400)
#define MCF5235_ETPU_ECDTRSR_DTRS11          (0x00000800)
#define MCF5235_ETPU_ECDTRSR_DTRS12          (0x00001000)
#define MCF5235_ETPU_ECDTRSR_DTRS13          (0x00002000)
#define MCF5235_ETPU_ECDTRSR_DTRS14          (0x00004000)
#define MCF5235_ETPU_ECDTRSR_DTRS15          (0x00008000)
#define MCF5235_ETPU_ECDTRSR_DTRS16          (0x00010000)
#define MCF5235_ETPU_ECDTRSR_DTRS17          (0x00020000)
#define MCF5235_ETPU_ECDTRSR_DTRS18          (0x00040000)
#define MCF5235_ETPU_ECDTRSR_DTRS19          (0x00080000)
#define MCF5235_ETPU_ECDTRSR_DTRS20          (0x00100000)
#define MCF5235_ETPU_ECDTRSR_DTRS21          (0x00200000)
#define MCF5235_ETPU_ECDTRSR_DTRS22          (0x00400000)
#define MCF5235_ETPU_ECDTRSR_DTRS23          (0x00800000)
#define MCF5235_ETPU_ECDTRSR_DTRS24          (0x01000000)
#define MCF5235_ETPU_ECDTRSR_DTRS25          (0x02000000)
#define MCF5235_ETPU_ECDTRSR_DTRS26          (0x04000000)
#define MCF5235_ETPU_ECDTRSR_DTRS27          (0x08000000)
#define MCF5235_ETPU_ECDTRSR_DTRS28          (0x10000000)
#define MCF5235_ETPU_ECDTRSR_DTRS29          (0x20000000)
#define MCF5235_ETPU_ECDTRSR_DTRS30          (0x40000000)
#define MCF5235_ETPU_ECDTRSR_DTRS31          (0x80000000)
#define MCF5235_ETPU_ECIOSR_CIOS0            (0x00000001)
#define MCF5235_ETPU_ECIOSR_CIOS1            (0x00000002)
#define MCF5235_ETPU_ECIOSR_CIOS2            (0x00000004)
#define MCF5235_ETPU_ECIOSR_CIOS3            (0x00000008)
#define MCF5235_ETPU_ECIOSR_CIOS4            (0x00000010)
#define MCF5235_ETPU_ECIOSR_CIOS5            (0x00000020)
#define MCF5235_ETPU_ECIOSR_CIOS6            (0x00000040)
#define MCF5235_ETPU_ECIOSR_CIOS7            (0x00000080)
#define MCF5235_ETPU_ECIOSR_CIOS8            (0x00000100)
#define MCF5235_ETPU_ECIOSR_CIOS9            (0x00000200)
#define MCF5235_ETPU_ECIOSR_CIOS10           (0x00000400)
#define MCF5235_ETPU_ECIOSR_CIOS11           (0x00000800)
#define MCF5235_ETPU_ECIOSR_CIOS12           (0x00001000)
#define MCF5235_ETPU_ECIOSR_CIOS13           (0x00002000)
#define MCF5235_ETPU_ECIOSR_CIOS14           (0x00004000)
#define MCF5235_ETPU_ECIOSR_CIOS15           (0x00008000)
#define MCF5235_ETPU_ECIOSR_CIOS16           (0x00010000)
#define MCF5235_ETPU_ECIOSR_CIOS17           (0x00020000)
#define MCF5235_ETPU_ECIOSR_CIOS18           (0x00040000)
#define MCF5235_ETPU_ECIOSR_CIOS19           (0x00080000)
#define MCF5235_ETPU_ECIOSR_CIOS20           (0x00100000)
#define MCF5235_ETPU_ECIOSR_CIOS21           (0x00200000)
#define MCF5235_ETPU_ECIOSR_CIOS22           (0x00400000)
#define MCF5235_ETPU_ECIOSR_CIOS23           (0x00800000)
#define MCF5235_ETPU_ECIOSR_CIOS24           (0x01000000)
#define MCF5235_ETPU_ECIOSR_CIOS25           (0x02000000)
#define MCF5235_ETPU_ECIOSR_CIOS26           (0x04000000)
#define MCF5235_ETPU_ECIOSR_CIOS27           (0x08000000)
#define MCF5235_ETPU_ECIOSR_CIOS28           (0x10000000)
#define MCF5235_ETPU_ECIOSR_CIOS29           (0x20000000)
#define MCF5235_ETPU_ECIOSR_CIOS30           (0x40000000)
#define MCF5235_ETPU_ECIOSR_CIOS31           (0x80000000)
#define MCF5235_ETPU_ECDTROSR_DTROS0         (0x00000001)
#define MCF5235_ETPU_ECDTROSR_DTROS1         (0x00000002)
#define MCF5235_ETPU_ECDTROSR_DTROS2         (0x00000004)
#define MCF5235_ETPU_ECDTROSR_DTROS3         (0x00000008)
#define MCF5235_ETPU_ECDTROSR_DTROS4         (0x00000010)
#define MCF5235_ETPU_ECDTROSR_DTROS5         (0x00000020)
#define MCF5235_ETPU_ECDTROSR_DTROS6         (0x00000040)
#define MCF5235_ETPU_ECDTROSR_DTROS7         (0x00000080)
#define MCF5235_ETPU_ECDTROSR_DTROS8         (0x00000100)
#define MCF5235_ETPU_ECDTROSR_DTROS9         (0x00000200)
#define MCF5235_ETPU_ECDTROSR_DTROS10        (0x00000400)
#define MCF5235_ETPU_ECDTROSR_DTROS11        (0x00000800)
#define MCF5235_ETPU_ECDTROSR_DTROS12        (0x00001000)
#define MCF5235_ETPU_ECDTROSR_DTROS13        (0x00002000)
#define MCF5235_ETPU_ECDTROSR_DTROS14        (0x00004000)
#define MCF5235_ETPU_ECDTROSR_DTROS15        (0x00008000)
#define MCF5235_ETPU_ECDTROSR_DTROS16        (0x00010000)
#define MCF5235_ETPU_ECDTROSR_DTROS17        (0x00020000)
#define MCF5235_ETPU_ECDTROSR_DTROS18        (0x00040000)
#define MCF5235_ETPU_ECDTROSR_DTROS19        (0x00080000)
#define MCF5235_ETPU_ECDTROSR_DTROS20        (0x00100000)
#define MCF5235_ETPU_ECDTROSR_DTROS21        (0x00200000)
#define MCF5235_ETPU_ECDTROSR_DTROS22        (0x00400000)
#define MCF5235_ETPU_ECDTROSR_DTROS23        (0x00800000)
#define MCF5235_ETPU_ECDTROSR_DTROS24        (0x01000000)
#define MCF5235_ETPU_ECDTROSR_DTROS25        (0x02000000)
#define MCF5235_ETPU_ECDTROSR_DTROS26        (0x04000000)
#define MCF5235_ETPU_ECDTROSR_DTROS27        (0x08000000)
#define MCF5235_ETPU_ECDTROSR_DTROS28        (0x10000000)
#define MCF5235_ETPU_ECDTROSR_DTROS29        (0x20000000)
#define MCF5235_ETPU_ECDTROSR_DTROS30        (0x40000000)
#define MCF5235_ETPU_ECDTROSR_DTROS31        (0x80000000)
#define MCF5235_ETPU_ECIER_CIE0              (0x00000001)
#define MCF5235_ETPU_ECIER_CIE1              (0x00000002)
#define MCF5235_ETPU_ECIER_CIE2              (0x00000004)
#define MCF5235_ETPU_ECIER_CIE3              (0x00000008)
#define MCF5235_ETPU_ECIER_CIE4              (0x00000010)
#define MCF5235_ETPU_ECIER_CIE5              (0x00000020)
#define MCF5235_ETPU_ECIER_CIE6              (0x00000040)
#define MCF5235_ETPU_ECIER_CIE7              (0x00000080)
#define MCF5235_ETPU_ECIER_CIE8              (0x00000100)
#define MCF5235_ETPU_ECIER_CIE9              (0x00000200)
#define MCF5235_ETPU_ECIER_CIE10             (0x00000400)
#define MCF5235_ETPU_ECIER_CIE11             (0x00000800)
#define MCF5235_ETPU_ECIER_CIE12             (0x00001000)
#define MCF5235_ETPU_ECIER_CIE13             (0x00002000)
#define MCF5235_ETPU_ECIER_CIE14             (0x00004000)
#define MCF5235_ETPU_ECIER_CIE15             (0x00008000)
#define MCF5235_ETPU_ECIER_CIE16             (0x00010000)
#define MCF5235_ETPU_ECIER_CIE17             (0x00020000)
#define MCF5235_ETPU_ECIER_CIE18             (0x00040000)
#define MCF5235_ETPU_ECIER_CIE19             (0x00080000)
#define MCF5235_ETPU_ECIER_CIE20             (0x00100000)
#define MCF5235_ETPU_ECIER_CIE21             (0x00200000)
#define MCF5235_ETPU_ECIER_CIE22             (0x00400000)
#define MCF5235_ETPU_ECIER_CIE23             (0x00800000)
#define MCF5235_ETPU_ECIER_CIE24             (0x01000000)
#define MCF5235_ETPU_ECIER_CIE25             (0x02000000)
#define MCF5235_ETPU_ECIER_CIE26             (0x04000000)
#define MCF5235_ETPU_ECIER_CIE27             (0x08000000)
#define MCF5235_ETPU_ECIER_CIE28             (0x10000000)
#define MCF5235_ETPU_ECIER_CIE29             (0x20000000)
#define MCF5235_ETPU_ECIER_CIE30             (0x40000000)
#define MCF5235_ETPU_ECIER_CIE31             (0x80000000)
#define MCF5235_ETPU_ECDTRER_DTRE0           (0x00000001)
#define MCF5235_ETPU_ECDTRER_DTRE1           (0x00000002)
#define MCF5235_ETPU_ECDTRER_DTRE2           (0x00000004)
#define MCF5235_ETPU_ECDTRER_DTRE3           (0x00000008)
#define MCF5235_ETPU_ECDTRER_DTRE4           (0x00000010)
#define MCF5235_ETPU_ECDTRER_DTRE5           (0x00000020)
#define MCF5235_ETPU_ECDTRER_DTRE6           (0x00000040)
#define MCF5235_ETPU_ECDTRER_DTRE7           (0x00000080)
#define MCF5235_ETPU_ECDTRER_DTRE8           (0x00000100)
#define MCF5235_ETPU_ECDTRER_DTRE9           (0x00000200)
#define MCF5235_ETPU_ECDTRER_DTRE10          (0x00000400)
#define MCF5235_ETPU_ECDTRER_DTRE11          (0x00000800)
#define MCF5235_ETPU_ECDTRER_DTRE12          (0x00001000)
#define MCF5235_ETPU_ECDTRER_DTRE13          (0x00002000)
#define MCF5235_ETPU_ECDTRER_DTRE14          (0x00004000)
#define MCF5235_ETPU_ECDTRER_DTRE15          (0x00008000)
#define MCF5235_ETPU_ECDTRER_DTRE16          (0x00010000)
#define MCF5235_ETPU_ECDTRER_DTRE17          (0x00020000)
#define MCF5235_ETPU_ECDTRER_DTRE18          (0x00040000)
#define MCF5235_ETPU_ECDTRER_DTRE19          (0x00080000)
#define MCF5235_ETPU_ECDTRER_DTRE20          (0x00100000)
#define MCF5235_ETPU_ECDTRER_DTRE21          (0x00200000)
#define MCF5235_ETPU_ECDTRER_DTRE22          (0x00400000)
#define MCF5235_ETPU_ECDTRER_DTRE23          (0x00800000)
#define MCF5235_ETPU_ECDTRER_DTRE24          (0x01000000)
#define MCF5235_ETPU_ECDTRER_DTRE25          (0x02000000)
#define MCF5235_ETPU_ECDTRER_DTRE26          (0x04000000)
#define MCF5235_ETPU_ECDTRER_DTRE27          (0x08000000)
#define MCF5235_ETPU_ECDTRER_DTRE28          (0x10000000)
#define MCF5235_ETPU_ECDTRER_DTRE29          (0x20000000)
#define MCF5235_ETPU_ECDTRER_DTRE30          (0x40000000)
#define MCF5235_ETPU_ECDTRER_DTRE31          (0x80000000)
#define MCF5235_ETPU_ECPSSR_SR0              (0x00000001)
#define MCF5235_ETPU_ECPSSR_SR1              (0x00000002)
#define MCF5235_ETPU_ECPSSR_SR2              (0x00000004)
#define MCF5235_ETPU_ECPSSR_SR3              (0x00000008)
#define MCF5235_ETPU_ECPSSR_SR4              (0x00000010)
#define MCF5235_ETPU_ECPSSR_SR5              (0x00000020)
#define MCF5235_ETPU_ECPSSR_SR6              (0x00000040)
#define MCF5235_ETPU_ECPSSR_SR7              (0x00000080)
#define MCF5235_ETPU_ECPSSR_SR8              (0x00000100)
#define MCF5235_ETPU_ECPSSR_SR9              (0x00000200)
#define MCF5235_ETPU_ECPSSR_SR10             (0x00000400)
#define MCF5235_ETPU_ECPSSR_SR11             (0x00000800)
#define MCF5235_ETPU_ECPSSR_SR12             (0x00001000)
#define MCF5235_ETPU_ECPSSR_SR13             (0x00002000)
#define MCF5235_ETPU_ECPSSR_SR14             (0x00004000)
#define MCF5235_ETPU_ECPSSR_SR15             (0x00008000)
#define MCF5235_ETPU_ECPSSR_SR16             (0x00010000)
#define MCF5235_ETPU_ECPSSR_SR17             (0x00020000)
#define MCF5235_ETPU_ECPSSR_SR18             (0x00040000)
#define MCF5235_ETPU_ECPSSR_SR19             (0x00080000)
#define MCF5235_ETPU_ECPSSR_SR20             (0x00100000)
#define MCF5235_ETPU_ECPSSR_SR21             (0x00200000)
#define MCF5235_ETPU_ECPSSR_SR22             (0x00400000)
#define MCF5235_ETPU_ECPSSR_SR23             (0x00800000)
#define MCF5235_ETPU_ECPSSR_SR24             (0x01000000)
#define MCF5235_ETPU_ECPSSR_SR25             (0x02000000)
#define MCF5235_ETPU_ECPSSR_SR26             (0x04000000)
#define MCF5235_ETPU_ECPSSR_SR27             (0x08000000)
#define MCF5235_ETPU_ECPSSR_SR28             (0x10000000)
#define MCF5235_ETPU_ECPSSR_SR29             (0x20000000)
#define MCF5235_ETPU_ECPSSR_SR30             (0x40000000)
#define MCF5235_ETPU_ECPSSR_SR31             (0x80000000)
#define MCF5235_ETPU_ECSSR_SS0               (0x00000001)
#define MCF5235_ETPU_ECSSR_SS1               (0x00000002)
#define MCF5235_ETPU_ECSSR_SS2               (0x00000004)
#define MCF5235_ETPU_ECSSR_SS3               (0x00000008)
#define MCF5235_ETPU_ECSSR_SS4               (0x00000010)
#define MCF5235_ETPU_ECSSR_SS5               (0x00000020)
#define MCF5235_ETPU_ECSSR_SS6               (0x00000040)
#define MCF5235_ETPU_ECSSR_SS7               (0x00000080)
#define MCF5235_ETPU_ECSSR_SS8               (0x00000100)
#define MCF5235_ETPU_ECSSR_SS9               (0x00000200)
#define MCF5235_ETPU_ECSSR_SS10              (0x00000400)
#define MCF5235_ETPU_ECSSR_SS11              (0x00000800)
#define MCF5235_ETPU_ECSSR_SS12              (0x00001000)
#define MCF5235_ETPU_ECSSR_SS13              (0x00002000)
#define MCF5235_ETPU_ECSSR_SS14              (0x00004000)
#define MCF5235_ETPU_ECSSR_SS15              (0x00008000)
#define MCF5235_ETPU_ECSSR_SS16              (0x00010000)
#define MCF5235_ETPU_ECSSR_SS17              (0x00020000)
#define MCF5235_ETPU_ECSSR_SS18              (0x00040000)
#define MCF5235_ETPU_ECSSR_SS19              (0x00080000)
#define MCF5235_ETPU_ECSSR_SS20              (0x00100000)
#define MCF5235_ETPU_ECSSR_SS21              (0x00200000)
#define MCF5235_ETPU_ECSSR_SS22              (0x00400000)
#define MCF5235_ETPU_ECSSR_SS23              (0x00800000)
#define MCF5235_ETPU_ECSSR_SS24              (0x01000000)
#define MCF5235_ETPU_ECSSR_SS25              (0x02000000)
#define MCF5235_ETPU_ECSSR_SS26              (0x04000000)
#define MCF5235_ETPU_ECSSR_SS27              (0x08000000)
#define MCF5235_ETPU_ECSSR_SS28              (0x10000000)
#define MCF5235_ETPU_ECSSR_SS29              (0x20000000)
#define MCF5235_ETPU_ECSSR_SS30              (0x40000000)
#define MCF5235_ETPU_ECSSR_SS31              (0x80000000)
#define MCF5235_ETPU_ECnSCR_FM(x)            (((x)&0x00000003)<<0)
#define MCF5235_ETPU_ECnSCR_OBE              (0x00002000)
#define MCF5235_ETPU_ECnSCR_OPS              (0x00004000)
#define MCF5235_ETPU_ECnSCR_IPS              (0x00008000)
#define MCF5235_ETPU_ECnSCR_DTROS            (0x00400000)
#define MCF5235_ETPU_ECnSCR_DTRS             (0x00800000)
#define MCF5235_ETPU_ECnSCR_CIOS             (0x40000000)
#define MCF5235_ETPU_ECnSCR_CIS              (0x80000000)
#define MCF5235_ETPU_ECnCR_CPBA(x)           (((x)&0x000007FF)<<0)
#define MCF5235_ETPU_ECnCR_OPOL              (0x00004000)
#define MCF5235_ETPU_ECnCR_ODIS              (0x00008000)
#define MCF5235_ETPU_ECnCR_CFS(x)            (((x)&0x0000001F)<<16)
#define MCF5235_ETPU_ECnCR_ETCS              (0x01000000)
#define MCF5235_ETPU_ECnCR_CPR(x)            (((x)&0x00000003)<<28)
#define MCF5235_ETPU_ECnCR_DTRE              (0x40000000)
#define MCF5235_ETPU_ECnCR_CIE               (0x80000000)
#define MCF5235_ETPU_ECnHSSR_HSR(x)          (((x)&0x00000007)<<0)


/*********************************************************************
*
* Fast Ethernet Controller (FEC)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_FEC_EIR                   (*(vuint32*)(void*)(&__IPSBAR[0x001004]))
#define MCF5235_FEC_EIMR                  (*(vuint32*)(void*)(&__IPSBAR[0x001008]))
#define MCF5235_FEC_RDAR                  (*(vuint32*)(void*)(&__IPSBAR[0x001010]))
#define MCF5235_FEC_TDAR                  (*(vuint32*)(void*)(&__IPSBAR[0x001014]))
#define MCF5235_FEC_ECR                   (*(vuint32*)(void*)(&__IPSBAR[0x001024]))
#define MCF5235_FEC_MMFR                  (*(vuint32*)(void*)(&__IPSBAR[0x001040]))
#define MCF5235_FEC_MSCR                  (*(vuint32*)(void*)(&__IPSBAR[0x001044]))
#define MCF5235_FEC_MIBC                  (*(vuint32*)(void*)(&__IPSBAR[0x001064]))
#define MCF5235_FEC_RCR                   (*(vuint32*)(void*)(&__IPSBAR[0x001084]))
#define MCF5235_FEC_TCR                   (*(vuint32*)(void*)(&__IPSBAR[0x0010C4]))
#define MCF5235_FEC_PALR                  (*(vuint32*)(void*)(&__IPSBAR[0x0010E4]))
#define MCF5235_FEC_PAUR                  (*(vuint32*)(void*)(&__IPSBAR[0x0010E8]))
#define MCF5235_FEC_OPD                   (*(vuint32*)(void*)(&__IPSBAR[0x0010EC]))
#define MCF5235_FEC_IAUR                  (*(vuint32*)(void*)(&__IPSBAR[0x001118]))
#define MCF5235_FEC_IALR                  (*(vuint32*)(void*)(&__IPSBAR[0x00111C]))
#define MCF5235_FEC_GAUR                  (*(vuint32*)(void*)(&__IPSBAR[0x001120]))
#define MCF5235_FEC_GALR                  (*(vuint32*)(void*)(&__IPSBAR[0x001124]))
#define MCF5235_FEC_TFWR                  (*(vuint32*)(void*)(&__IPSBAR[0x001144]))
#define MCF5235_FEC_FRBR                  (*(vuint32*)(void*)(&__IPSBAR[0x00114C]))
#define MCF5235_FEC_FRSR                  (*(vuint32*)(void*)(&__IPSBAR[0x001150]))
#define MCF5235_FEC_ERDSR                 (*(vuint32*)(void*)(&__IPSBAR[0x001180]))
#define MCF5235_FEC_ETDSR                 (*(vuint32*)(void*)(&__IPSBAR[0x001184]))
#define MCF5235_FEC_EMRBR                 (*(vuint32*)(void*)(&__IPSBAR[0x001188]))
#define MCF5235_FEC_RMON_T_DROP           (*(vuint32*)(void*)(&__IPSBAR[0x001200]))
#define MCF5235_FEC_RMON_T_PACKETS        (*(vuint32*)(void*)(&__IPSBAR[0x001204]))
#define MCF5235_FEC_RMON_T_BC_PKT         (*(vuint32*)(void*)(&__IPSBAR[0x001208]))
#define MCF5235_FEC_RMON_T_MC_PKT         (*(vuint32*)(void*)(&__IPSBAR[0x00120C]))
#define MCF5235_FEC_RMON_T_CRC_ALIGN      (*(vuint32*)(void*)(&__IPSBAR[0x001210]))
#define MCF5235_FEC_RMON_T_UNDERSIZE      (*(vuint32*)(void*)(&__IPSBAR[0x001214]))
#define MCF5235_FEC_RMON_T_OVERSIZE       (*(vuint32*)(void*)(&__IPSBAR[0x001218]))
#define MCF5235_FEC_RMON_T_FRAG           (*(vuint32*)(void*)(&__IPSBAR[0x00121C]))
#define MCF5235_FEC_RMON_T_JAB            (*(vuint32*)(void*)(&__IPSBAR[0x001220]))
#define MCF5235_FEC_RMON_T_COL            (*(vuint32*)(void*)(&__IPSBAR[0x001224]))
#define MCF5235_FEC_RMON_T_P64            (*(vuint32*)(void*)(&__IPSBAR[0x001228]))
#define MCF5235_FEC_RMON_T_P65TO127       (*(vuint32*)(void*)(&__IPSBAR[0x00122C]))
#define MCF5235_FEC_RMON_T_P128TO255      (*(vuint32*)(void*)(&__IPSBAR[0x001230]))
#define MCF5235_FEC_RMON_T_P256TO511      (*(vuint32*)(void*)(&__IPSBAR[0x001234]))
#define MCF5235_FEC_RMON_T_P512TO1023     (*(vuint32*)(void*)(&__IPSBAR[0x001238]))
#define MCF5235_FEC_RMON_T_P1024TO2047    (*(vuint32*)(void*)(&__IPSBAR[0x00123C]))
#define MCF5235_FEC_RMON_T_P_GTE2048      (*(vuint32*)(void*)(&__IPSBAR[0x001240]))
#define MCF5235_FEC_RMON_T_OCTETS         (*(vuint32*)(void*)(&__IPSBAR[0x001244]))
#define MCF5235_FEC_IEEE_T_DROP           (*(vuint32*)(void*)(&__IPSBAR[0x001248]))
#define MCF5235_FEC_IEEE_T_FRAME_OK       (*(vuint32*)(void*)(&__IPSBAR[0x00124C]))
#define MCF5235_FEC_IEEE_T_1COL           (*(vuint32*)(void*)(&__IPSBAR[0x001250]))
#define MCF5235_FEC_IEEE_T_MCOL           (*(vuint32*)(void*)(&__IPSBAR[0x001254]))
#define MCF5235_FEC_IEEE_T_DEF            (*(vuint32*)(void*)(&__IPSBAR[0x001258]))
#define MCF5235_FEC_IEEE_T_LCOL           (*(vuint32*)(void*)(&__IPSBAR[0x00125C]))
#define MCF5235_FEC_IEEE_T_EXCOL          (*(vuint32*)(void*)(&__IPSBAR[0x001260]))
#define MCF5235_FEC_IEEE_T_MACERR         (*(vuint32*)(void*)(&__IPSBAR[0x001264]))
#define MCF5235_FEC_IEEE_T_CSERR          (*(vuint32*)(void*)(&__IPSBAR[0x001268]))
#define MCF5235_FEC_IEEE_T_SQE            (*(vuint32*)(void*)(&__IPSBAR[0x00126C]))
#define MCF5235_FEC_IEEE_T_FDXFC          (*(vuint32*)(void*)(&__IPSBAR[0x001270]))
#define MCF5235_FEC_IEEE_T_OCTETS_OK      (*(vuint32*)(void*)(&__IPSBAR[0x001274]))
#define MCF5235_FEC_RMON_R_PACKETS        (*(vuint32*)(void*)(&__IPSBAR[0x001284]))
#define MCF5235_FEC_RMON_R_BC_PKT         (*(vuint32*)(void*)(&__IPSBAR[0x001288]))
#define MCF5235_FEC_RMON_R_MC_PKT         (*(vuint32*)(void*)(&__IPSBAR[0x00128C]))
#define MCF5235_FEC_RMON_R_CRC_ALIGN      (*(vuint32*)(void*)(&__IPSBAR[0x001290]))
#define MCF5235_FEC_RMON_R_UNDERSIZE      (*(vuint32*)(void*)(&__IPSBAR[0x001294]))
#define MCF5235_FEC_RMON_R_OVERSIZE       (*(vuint32*)(void*)(&__IPSBAR[0x001298]))
#define MCF5235_FEC_RMON_R_FRAG           (*(vuint32*)(void*)(&__IPSBAR[0x00129C]))
#define MCF5235_FEC_RMON_R_JAB            (*(vuint32*)(void*)(&__IPSBAR[0x0012A0]))
#define MCF5235_FEC_RMON_R_RESVD_0        (*(vuint32*)(void*)(&__IPSBAR[0x0012A4]))
#define MCF5235_FEC_RMON_R_P64            (*(vuint32*)(void*)(&__IPSBAR[0x0012A8]))
#define MCF5235_FEC_RMON_R_P65T0127       (*(vuint32*)(void*)(&__IPSBAR[0x0012AC]))
#define MCF5235_FEC_RMON_R_P128TO255      (*(vuint32*)(void*)(&__IPSBAR[0x0012B0]))
#define MCF5235_FEC_RMON_R_P256TO511      (*(vuint32*)(void*)(&__IPSBAR[0x0012B4]))
#define MCF5235_FEC_RMON_R_P512TO1023      (*(vuint32*)(void*)(&__IPSBAR[0x0012B8]))
#define MCF5235_FEC_RMON_R_GTE2048        (*(vuint32*)(void*)(&__IPSBAR[0x0012C0]))
#define MCF5235_FEC_RMON_R_P1024TO2047     (*(vuint32*)(void*)(&__IPSBAR[0x0012BC]))
#define MCF5235_FEC_RMON_R_OCTETS         (*(vuint32*)(void*)(&__IPSBAR[0x0012C4]))
#define MCF5235_FEC_IEEE_R_DROP           (*(vuint32*)(void*)(&__IPSBAR[0x0012C8]))
#define MCF5235_FEC_IEEE_R_FRAME_OK       (*(vuint32*)(void*)(&__IPSBAR[0x0012CC]))
#define MCF5235_FEC_IEEE_R_CRC            (*(vuint32*)(void*)(&__IPSBAR[0x0012D0]))
#define MCF5235_FEC_IEEE_R_ALIGN          (*(vuint32*)(void*)(&__IPSBAR[0x0012D4]))
#define MCF5235_FEC_IEEE_R_MACERR         (*(vuint32*)(void*)(&__IPSBAR[0x0012D8]))
#define MCF5235_FEC_IEEE_R_FDXFC          (*(vuint32*)(void*)(&__IPSBAR[0x0012DC]))
#define MCF5235_FEC_IEEE_R_OCTETS_OK      (*(vuint32*)(void*)(&__IPSBAR[0x0012E0]))

/* Bit definitions and macros for MCF5235_FEC_EIR */
#define MCF5235_FEC_EIR_UN                  (0x00080000)
#define MCF5235_FEC_EIR_RL                  (0x00100000)
#define MCF5235_FEC_EIR_LC                  (0x00200000)
#define MCF5235_FEC_EIR_EBERR               (0x00400000)
#define MCF5235_FEC_EIR_MII                 (0x00800000)
#define MCF5235_FEC_EIR_RXB                 (0x01000000)
#define MCF5235_FEC_EIR_RXF                 (0x02000000)
#define MCF5235_FEC_EIR_TXB                 (0x04000000)
#define MCF5235_FEC_EIR_TXF                 (0x08000000)
#define MCF5235_FEC_EIR_GRA                 (0x10000000)
#define MCF5235_FEC_EIR_BABT                (0x20000000)
#define MCF5235_FEC_EIR_BABR                (0x40000000)
#define MCF5235_FEC_EIR_HBERR               (0x80000000)
#define MCF5235_FEC_EIMR_UN                 (0x00080000)
#define MCF5235_FEC_EIMR_RL                 (0x00100000)
#define MCF5235_FEC_EIMR_LC                 (0x00200000)
#define MCF5235_FEC_EIMR_EBERR              (0x00400000)
#define MCF5235_FEC_EIMR_MII                (0x00800000)
#define MCF5235_FEC_EIMR_RXB                (0x01000000)
#define MCF5235_FEC_EIMR_RXF                (0x02000000)
#define MCF5235_FEC_EIMR_TXB                (0x04000000)
#define MCF5235_FEC_EIMR_TXF                (0x08000000)
#define MCF5235_FEC_EIMR_GRA                (0x10000000)
#define MCF5235_FEC_EIMR_BABT               (0x20000000)
#define MCF5235_FEC_EIMR_BABR               (0x40000000)
#define MCF5235_FEC_EIMR_HBERR              (0x80000000)
#define MCF5235_FEC_RDAR_R_DES_ACTIVE       (0x01000000)
#define MCF5235_FEC_TDAR_X_DES_ACTIVE       (0x01000000)
#define MCF5235_FEC_ECR_RESET               (0x00000001)
#define MCF5235_FEC_ECR_ETHER_EN            (0x00000002)
#define MCF5235_FEC_MMFR_DATA(x)            (((x)&0x0000FFFF)<<0)
#define MCF5235_FEC_MMFR_TA(x)              (((x)&0x00000003)<<16)
#define MCF5235_FEC_MMFR_RA(x)              (((x)&0x0000001F)<<18)
#define MCF5235_FEC_MMFR_PA(x)              (((x)&0x0000001F)<<23)
#define MCF5235_FEC_MMFR_OP(x)              (((x)&0x00000003)<<28)
#define MCF5235_FEC_MMFR_ST(x)              (((x)&0x00000003)<<30)
#define MCF5235_FEC_MMFR_ST_01             (0x40000000)
#define MCF5235_FEC_MMFR_OP_READ           (0x20000000)
#define MCF5235_FEC_MMFR_OP_WRITE          (0x10000000)
#define MCF5235_FEC_MMFR_TA_10             (0x00020000)
#define MCF5235_FEC_MSCR_MII_SPEED(x)       (((x)&0x0000003F)<<1)
#define MCF5235_FEC_MSCR_DIS_PREAMBLE       (0x00000080)
#define MCF5235_FEC_MIBC_MIB_IDLE           (0x40000000)
#define MCF5235_FEC_MIBC_MIB_DISABLE        (0x80000000)
#define MCF5235_FEC_RCR_LOOP                (0x00000001)
#define MCF5235_FEC_RCR_DRT                 (0x00000002)
#define MCF5235_FEC_RCR_MII_MODE            (0x00000004)
#define MCF5235_FEC_RCR_PROM                (0x00000008)
#define MCF5235_FEC_RCR_BC_REJ              (0x00000010)
#define MCF5235_FEC_RCR_FCE                 (0x00000020)
#define MCF5235_FEC_RCR_MAX_FL(x)           (((x)&0x000007FF)<<16)
#define MCF5235_FEC_TCR_GTS                 (0x00000001)
#define MCF5235_FEC_TCR_HBC                 (0x00000002)
#define MCF5235_FEC_TCR_FDEN                (0x00000004)
#define MCF5235_FEC_TCR_TFC_PAUSE           (0x00000008)
#define MCF5235_FEC_TCR_RFC_PAUSE           (0x00000010)
#define MCF5235_FEC_PAUR_TYPE(x)            (((x)&0x0000FFFF)<<0)
#define MCF5235_FEC_PAUR_PADDR2(x)          (((x)&0x0000FFFF)<<16)
#define MCF5235_FEC_OPD_PAUSE_DUR(x)        (((x)&0x0000FFFF)<<0)
#define MCF5235_FEC_OPD_OPCODE(x)           (((x)&0x0000FFFF)<<16)
#define MCF5235_FEC_TFWR_X_WMRK(x)          (((x)&0x00000003)<<0)
#define MCF5235_FEC_FRBR_R_BOUND(x)         (((x)&0x000000FF)<<2)
#define MCF5235_FEC_FRSR_R_FSTART(x)        (((x)&0x000000FF)<<2)
#define MCF5235_FEC_ERDSR_R_DES_START(x)    (((x)&0x3FFFFFFF)<<2)
#define MCF5235_FEC_ETDSR_X_DES_START(x)    (((x)&0x3FFFFFFF)<<2)
#define MCF5235_FEC_EMRBR_R_BUF_SIZE(x)     (((x)&0x0000007F)<<4)
#define MCF5235_FEC_TxBD_R					0x8000
#define MCF5235_FEC_TxBD_BUSY				0x4000
#define MCF5235_FEC_TxBD_TO1				0x4000
#define MCF5235_FEC_TxBD_W					0x2000
#define MCF5235_FEC_TxBD_TO2				0x1000
#define MCF5235_FEC_TxBD_FIRST				0x1000
#define MCF5235_FEC_TxBD_L					0x0800
#define MCF5235_FEC_TxBD_TC					0x0400
#define MCF5235_FEC_TxBD_DEF				0x0200
#define MCF5235_FEC_TxBD_HB					0x0100
#define MCF5235_FEC_TxBD_LC					0x0080
#define MCF5235_FEC_TxBD_RL					0x0040
#define MCF5235_FEC_TxBD_UN					0x0002
#define MCF5235_FEC_TxBD_CSL				0x0001
#define MCF5235_FEC_RxBD_E					0x8000
#define MCF5235_FEC_RxBD_INUSE				0x4000
#define MCF5235_FEC_RxBD_R01				0x4000
#define MCF5235_FEC_RxBD_W					0x2000
#define MCF5235_FEC_RxBD_R02				0x1000
#define MCF5235_FEC_RxBD_L					0x0800
#define MCF5235_FEC_RxBD_M					0x0100
#define MCF5235_FEC_RxBD_BC					0x0080
#define MCF5235_FEC_RxBD_MC					0x0040
#define MCF5235_FEC_RxBD_LG					0x0020
#define MCF5235_FEC_RxBD_NO					0x0010
#define MCF5235_FEC_RxBD_CR					0x0004
#define MCF5235_FEC_RxBD_OV					0x0002
#define MCF5235_FEC_RxBD_TR					0x0001

/************************************************************
*
*	Clock 
*************************************************************/
/* Register read/write macros */
#define MCF5235_FMPLL_SYNCR    (*(vuint32*)(void*)(&__IPSBAR[0x120000]))
#define MCF5235_FMPLL_SYNSR    (*(vuint32*)(void*)(&__IPSBAR[0x120004]))

/* Bit definitions and macros for MCF5235_FMPLL_SYNCR */
#define MCF5235_FMPLL_SYNCR_EXP(x)       (((x)&0x000003FF)<<0)
#define MCF5235_FMPLL_SYNCR_DEPTH(x)     (((x)&0x00000003)<<10)
#define MCF5235_FMPLL_SYNCR_RATE         (0x00001000)
#define MCF5235_FMPLL_SYNCR_LOCIRQ       (0x00002000)
#define MCF5235_FMPLL_SYNCR_LOLIRQ       (0x00004000)
#define MCF5235_FMPLL_SYNCR_DISCLK       (0x00008000)
#define MCF5235_FMPLL_SYNCR_LOCRE        (0x00010000)
#define MCF5235_FMPLL_SYNCR_LOLRE        (0x00020000)
#define MCF5235_FMPLL_SYNCR_LOCEN        (0x00040000)
#define MCF5235_FMPLL_SYNCR_RFD(x)       (((x)&0x00000007)<<19)
#define MCF5235_FMPLL_SYNCR_MFD(x)       (((x)&0x00000007)<<24)
#define MCF5235_FMPLL_SYNSR_CALPASS      (0x00000001)
#define MCF5235_FMPLL_SYNSR_CALDONE      (0x00000002)
#define MCF5235_FMPLL_SYNSR_LOCF         (0x00000004)
#define MCF5235_FMPLL_SYNSR_LOCK         (0x00000008)
#define MCF5235_FMPLL_SYNSR_LOCKS        (0x00000010)
#define MCF5235_FMPLL_SYNSR_PLLREF       (0x00000020)
#define MCF5235_FMPLL_SYNSR_PLLSEL       (0x00000040)
#define MCF5235_FMPLL_SYNSR_MODE         (0x00000080)
#define MCF5235_FMPLL_SYNSR_LOC          (0x00000100)
#define MCF5235_FMPLL_SYNSR_LOLF         (0x00000200)


/*********************************************************************
*
* General Purpose I/O (GPIO)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_GPIO_PODR_ADDR        (*(vuint8 *)(void*)(&__IPSBAR[0x100000]))
#define MCF5235_GPIO_PODR_DATAH       (*(vuint8 *)(void*)(&__IPSBAR[0x100001]))
#define MCF5235_GPIO_PODR_DATAL       (*(vuint8 *)(void*)(&__IPSBAR[0x100002]))
#define MCF5235_GPIO_PODR_BUSCTL      (*(vuint8 *)(void*)(&__IPSBAR[0x100003]))
#define MCF5235_GPIO_PODR_BS          (*(vuint8 *)(void*)(&__IPSBAR[0x100004]))
#define MCF5235_GPIO_PODR_CS          (*(vuint8 *)(void*)(&__IPSBAR[0x100005]))
#define MCF5235_GPIO_PODR_SDRAM       (*(vuint8 *)(void*)(&__IPSBAR[0x100006]))
#define MCF5235_GPIO_PODR_FECI2C      (*(vuint8 *)(void*)(&__IPSBAR[0x100007]))
#define MCF5235_GPIO_PODR_UARTH       (*(vuint8 *)(void*)(&__IPSBAR[0x100008]))
#define MCF5235_GPIO_PODR_UARTL       (*(vuint8 *)(void*)(&__IPSBAR[0x100009]))
#define MCF5235_GPIO_PODR_QSPI        (*(vuint8 *)(void*)(&__IPSBAR[0x10000A]))
#define MCF5235_GPIO_PODR_TIMER       (*(vuint8 *)(void*)(&__IPSBAR[0x10000B]))
#define MCF5235_GPIO_PODR_ETPU        (*(vuint8 *)(void*)(&__IPSBAR[0x10000C]))
#define MCF5235_GPIO_PDDR_APDDR       (*(vuint8 *)(void*)(&__IPSBAR[0x100010]))
#define MCF5235_GPIO_PDDR_DATAH       (*(vuint8 *)(void*)(&__IPSBAR[0x100011]))
#define MCF5235_GPIO_PDDR_DATAL       (*(vuint8 *)(void*)(&__IPSBAR[0x100012]))
#define MCF5235_GPIO_PDDR_BUSCTL      (*(vuint8 *)(void*)(&__IPSBAR[0x100013]))
#define MCF5235_GPIO_PDDR_BS          (*(vuint8 *)(void*)(&__IPSBAR[0x100014]))
#define MCF5235_GPIO_PDDR_CS          (*(vuint8 *)(void*)(&__IPSBAR[0x100015]))
#define MCF5235_GPIO_PDDR_SDRAM       (*(vuint8 *)(void*)(&__IPSBAR[0x100016]))
#define MCF5235_GPIO_PDDR_FECI2C      (*(vuint8 *)(void*)(&__IPSBAR[0x100017]))
#define MCF5235_GPIO_PDDR_UARTH       (*(vuint8 *)(void*)(&__IPSBAR[0x100018]))
#define MCF5235_GPIO_PDDR_UARTL       (*(vuint8 *)(void*)(&__IPSBAR[0x100019]))
#define MCF5235_GPIO_PDDR_QSPI        (*(vuint8 *)(void*)(&__IPSBAR[0x10001A]))
#define MCF5235_GPIO_PDDR_TIMER       (*(vuint8 *)(void*)(&__IPSBAR[0x10001B]))
#define MCF5235_GPIO_PDDR_ETPU        (*(vuint8 *)(void*)(&__IPSBAR[0x10001C]))
#define MCF5235_GPIO_PPDSDR_ADDR      (*(vuint8 *)(void*)(&__IPSBAR[0x100020]))
#define MCF5235_GPIO_PPDSDR_DATAH     (*(vuint8 *)(void*)(&__IPSBAR[0x100021]))
#define MCF5235_GPIO_PPDSDR_DATAL     (*(vuint8 *)(void*)(&__IPSBAR[0x100022]))
#define MCF5235_GPIO_PPDSDR_BUSCTL    (*(vuint8 *)(void*)(&__IPSBAR[0x100023]))
#define MCF5235_GPIO_PPDSDR_BS        (*(vuint8 *)(void*)(&__IPSBAR[0x100024]))
#define MCF5235_GPIO_PPDSDR_FECI2C    (*(vuint8 *)(void*)(&__IPSBAR[0x100027]))
#define MCF5235_GPIO_PPDSDR_CS        (*(vuint8 *)(void*)(&__IPSBAR[0x100025]))
#define MCF5235_GPIO_PPDSDR_SDRAM     (*(vuint8 *)(void*)(&__IPSBAR[0x100026]))
#define MCF5235_GPIO_PPDSDR_UARTH     (*(vuint8 *)(void*)(&__IPSBAR[0x100028]))
#define MCF5235_GPIO_PPDSDR_UARTL     (*(vuint8 *)(void*)(&__IPSBAR[0x100029]))
#define MCF5235_GPIO_PPDSDR_QSPI      (*(vuint8 *)(void*)(&__IPSBAR[0x10002A]))
#define MCF5235_GPIO_PPDSDR_TIMER     (*(vuint8 *)(void*)(&__IPSBAR[0x10002B]))
#define MCF5235_GPIO_PPDSDR_ETPU      (*(vuint8 *)(void*)(&__IPSBAR[0x10002C]))
#define MCF5235_GPIO_PCLRR_ADDR       (*(vuint8 *)(void*)(&__IPSBAR[0x100030]))
#define MCF5235_GPIO_PCLRR_DATAH      (*(vuint8 *)(void*)(&__IPSBAR[0x100031]))
#define MCF5235_GPIO_PCLRR_DATAL      (*(vuint8 *)(void*)(&__IPSBAR[0x100032]))
#define MCF5235_GPIO_PCLRR_BUSCTL     (*(vuint8 *)(void*)(&__IPSBAR[0x100033]))
#define MCF5235_GPIO_PCLRR_BS         (*(vuint8 *)(void*)(&__IPSBAR[0x100034]))
#define MCF5235_GPIO_PCLRR_CS         (*(vuint8 *)(void*)(&__IPSBAR[0x100035]))
#define MCF5235_GPIO_PCLRR_SDRAM      (*(vuint8 *)(void*)(&__IPSBAR[0x100036]))
#define MCF5235_GPIO_PCLRR_FECI2C     (*(vuint8 *)(void*)(&__IPSBAR[0x100037]))
#define MCF5235_GPIO_PCLRR_UARTH      (*(vuint8 *)(void*)(&__IPSBAR[0x100038]))
#define MCF5235_GPIO_PCLRR_UARTL      (*(vuint8 *)(void*)(&__IPSBAR[0x100039]))
#define MCF5235_GPIO_PCLRR_QSPI       (*(vuint8 *)(void*)(&__IPSBAR[0x10003A]))
#define MCF5235_GPIO_PCLRR_TIMER      (*(vuint8 *)(void*)(&__IPSBAR[0x10003B]))
#define MCF5235_GPIO_PCLRR_ETPU       (*(vuint8 *)(void*)(&__IPSBAR[0x10003C]))
#define MCF5235_GPIO_PAR_AD           (*(vuint8 *)(void*)(&__IPSBAR[0x100040]))
#define MCF5235_GPIO_PAR_BUSCTL       (*(vuint16*)(void*)(&__IPSBAR[0x100042]))
#define MCF5235_GPIO_PAR_BS           (*(vuint8 *)(void*)(&__IPSBAR[0x100044]))
#define MCF5235_GPIO_PAR_CS           (*(vuint8 *)(void*)(&__IPSBAR[0x100045]))
#define MCF5235_GPIO_PAR_SDRAM        (*(vuint8 *)(void*)(&__IPSBAR[0x100046]))
#define MCF5235_GPIO_PAR_FECI2C       (*(vuint8 *)(void*)(&__IPSBAR[0x100047]))
#define MCF5235_GPIO_UART         (*(vuint16*)(void*)(&__IPSBAR[0x100048]))
#define MCF5235_GPIO_PAR_QSPI         (*(vuint8 *)(void*)(&__IPSBAR[0x10004A]))
#define MCF5235_GPIO_PAR_TIMER        (*(vuint16*)(void*)(&__IPSBAR[0x10004C]))
#define MCF5235_GPIO_PAR_ETPU         (*(vuint8 *)(void*)(&__IPSBAR[0x10004E]))
#define MCF5235_GPIO_DSCR_EIM         (*(vuint8 *)(void*)(&__IPSBAR[0x100050]))
#define MCF5235_GPIO_DSCR_ETPU        (*(vuint8 *)(void*)(&__IPSBAR[0x100051]))
#define MCF5235_GPIO_DSCR_FECI2C      (*(vuint8 *)(void*)(&__IPSBAR[0x100052]))
#define MCF5235_GPIO_DSCR_UART        (*(vuint8 *)(void*)(&__IPSBAR[0x100053]))
#define MCF5235_GPIO_DSCR_QSPI        (*(vuint8 *)(void*)(&__IPSBAR[0x100054]))
#define MCF5235_GPIO_DSCR_TIMER       (*(vuint8 *)(void*)(&__IPSBAR[0x100055]))

/* Bit definitions and macros for MCF5235_GPIO_PODR_ADDR */
#define MCF5235_GPIO_PODR_ADDR_PODR_ADDR5            (0x20)
#define MCF5235_GPIO_PODR_ADDR_PODR_ADDR6            (0x40)
#define MCF5235_GPIO_PODR_ADDR_PODR_ADDR7            (0x80)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH0          (0x01)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH1          (0x02)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH2          (0x04)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH3          (0x08)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH4          (0x10)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH5          (0x20)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH6          (0x40)
#define MCF5235_GPIO_PODR_DATAH_PODR_DATAH7          (0x80)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL0          (0x01)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL1          (0x02)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL2          (0x04)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL3          (0x08)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL4          (0x10)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL5          (0x20)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL6          (0x40)
#define MCF5235_GPIO_PODR_DATAL_PODR_DATAL7          (0x80)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL0        (0x01)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL1        (0x02)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL2        (0x04)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL3        (0x08)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL4        (0x10)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL5        (0x20)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL6        (0x40)
#define MCF5235_GPIO_PODR_BUSCTL_PODR_BUSCTL7        (0x80)
#define MCF5235_GPIO_PODR_BS_PODR_BS0                (0x01)
#define MCF5235_GPIO_PODR_BS_PODR_BS1                (0x02)
#define MCF5235_GPIO_PODR_BS_PODR_BS2                (0x04)
#define MCF5235_GPIO_PODR_BS_PODR_BS3                (0x08)
#define MCF5235_GPIO_PODR_CS_PODR_CS1                (0x02)
#define MCF5235_GPIO_PODR_CS_PODR_CS2                (0x04)
#define MCF5235_GPIO_PODR_CS_PODR_CS3                (0x08)
#define MCF5235_GPIO_PODR_CS_PODR_CS4                (0x10)
#define MCF5235_GPIO_PODR_CS_PODR_CS5                (0x20)
#define MCF5235_GPIO_PODR_CS_PODR_CS6                (0x40)
#define MCF5235_GPIO_PODR_CS_PODR_CS7                (0x80)
#define MCF5235_GPIO_PODR_SDRAM_PODR_SDRAM0          (0x01)
#define MCF5235_GPIO_PODR_SDRAM_PODR_SDRAM1          (0x02)
#define MCF5235_GPIO_PODR_SDRAM_PODR_SDRAM2          (0x04)
#define MCF5235_GPIO_PODR_SDRAM_PODR_SDRAM3          (0x08)
#define MCF5235_GPIO_PODR_SDRAM_PODR_SDRAM4          (0x10)
#define MCF5235_GPIO_PODR_SDRAM_PODR_SDRAM5          (0x20)
#define MCF5235_GPIO_PODR_FECI2C_PODR_FECI2C0        (0x01)
#define MCF5235_GPIO_PODR_FECI2C_PODR_FECI2C1        (0x02)
#define MCF5235_GPIO_PODR_FECI2C_PODR_FECI2C2        (0x04)
#define MCF5235_GPIO_PODR_FECI2C_PODR_FECI2C3        (0x08)
#define MCF5235_GPIO_PODR_UARTH_PODR_UARTH0          (0x01)
#define MCF5235_GPIO_PODR_UARTH_PODR_UARTH1          (0x02)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL0          (0x01)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL1          (0x02)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL2          (0x04)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL3          (0x08)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL4          (0x10)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL5          (0x20)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL6          (0x40)
#define MCF5235_GPIO_PODR_UARTL_PODR_UARTL7          (0x80)
#define MCF5235_GPIO_PODR_QSPI_PODR_QSPI0            (0x01)
#define MCF5235_GPIO_PODR_QSPI_PODR_QSPI1            (0x02)
#define MCF5235_GPIO_PODR_QSPI_PODR_QSPI2            (0x04)
#define MCF5235_GPIO_PODR_QSPI_PODR_QSPI3            (0x08)
#define MCF5235_GPIO_PODR_QSPI_PODR_QSPI4            (0x10)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER0          (0x01)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER1          (0x02)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER2          (0x04)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER3          (0x08)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER4          (0x10)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER5          (0x20)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER6          (0x40)
#define MCF5235_GPIO_PODR_TIMER_PODR_TIMER7          (0x80)
#define MCF5235_GPIO_PODR_ETPU_PODR_ETPU0            (0x01)
#define MCF5235_GPIO_PODR_ETPU_PODR_ETPU1            (0x02)
#define MCF5235_GPIO_PODR_ETPU_PODR_ETPU2            (0x04)
#define MCF5235_GPIO_PDDR_APDDR_PDDR_APDDR5          (0x20)
#define MCF5235_GPIO_PDDR_APDDR_PDDR_APDDR6          (0x40)
#define MCF5235_GPIO_PDDR_APDDR_PDDR_APDDR7          (0x80)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH0          (0x01)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH1          (0x02)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH2          (0x04)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH3          (0x08)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH4          (0x10)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH5          (0x20)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH6          (0x40)
#define MCF5235_GPIO_PDDR_DATAH_PDDR_DATAH7          (0x80)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL0          (0x01)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL1          (0x02)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL2          (0x04)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL3          (0x08)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL4          (0x10)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL5          (0x20)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL6          (0x40)
#define MCF5235_GPIO_PDDR_DATAL_PDDR_DATAL7          (0x80)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL0        (0x01)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL1        (0x02)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL2        (0x04)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL3        (0x08)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL4        (0x10)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL5        (0x20)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL6        (0x40)
#define MCF5235_GPIO_PDDR_BUSCTL_PDDR_BUSCTL7        (0x80)
#define MCF5235_GPIO_PDDR_BS_PDDR_BS0                (0x01)
#define MCF5235_GPIO_PDDR_BS_PDDR_BS3(x)             (((x)&0x07)<<1)
#define MCF5235_GPIO_PDDR_CS_PDDR_CS1                (0x02)
#define MCF5235_GPIO_PDDR_CS_PDDR_CS2                (0x04)
#define MCF5235_GPIO_PDDR_CS_PDDR_CS3                (0x08)
#define MCF5235_GPIO_PDDR_CS_PDDR_CS4                (0x10)
#define MCF5235_GPIO_PDDR_CS_PDDR_CS5                (0x20)
#define MCF5235_GPIO_PDDR_CS_PDDR_CS6                (0x40)
#define MCF5235_GPIO_PDDR_CS_PDDR_CS7                (0x80)
#define MCF5235_GPIO_PDDR_SDRAM_PDDR_SDRAM0          (0x01)
#define MCF5235_GPIO_PDDR_SDRAM_PDDR_SDRAM1          (0x02)
#define MCF5235_GPIO_PDDR_SDRAM_PDDR_SDRAM2          (0x04)
#define MCF5235_GPIO_PDDR_SDRAM_PDDR_SDRAM3          (0x08)
#define MCF5235_GPIO_PDDR_SDRAM_PDDR_SDRAM4          (0x10)
#define MCF5235_GPIO_PDDR_SDRAM_PDDR_SDRAM5          (0x20)
#define MCF5235_GPIO_PDDR_FECI2C_PDDR_FECI2C0        (0x01)
#define MCF5235_GPIO_PDDR_FECI2C_PDDR_FECI2C1        (0x02)
#define MCF5235_GPIO_PDDR_FECI2C_PDDR_FECI2C2        (0x04)
#define MCF5235_GPIO_PDDR_FECI2C_PDDR_FECI2C3        (0x08)
#define MCF5235_GPIO_PDDR_UARTH_PDDR_UARTH0          (0x01)
#define MCF5235_GPIO_PDDR_UARTH_PDDR_UARTH1          (0x02)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL0          (0x01)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL1          (0x02)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL2          (0x04)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL3          (0x08)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL4          (0x10)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL5          (0x20)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL6          (0x40)
#define MCF5235_GPIO_PDDR_UARTL_PDDR_UARTL7          (0x80)
#define MCF5235_GPIO_PDDR_QSPI_PDDR_QSPI0            (0x01)
#define MCF5235_GPIO_PDDR_QSPI_PDDR_QSPI1            (0x02)
#define MCF5235_GPIO_PDDR_QSPI_PDDR_QSPI2            (0x04)
#define MCF5235_GPIO_PDDR_QSPI_PDDR_QSPI3            (0x08)
#define MCF5235_GPIO_PDDR_QSPI_PDDR_QSPI4            (0x10)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER0          (0x01)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER1          (0x02)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER2          (0x04)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER3          (0x08)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER4          (0x10)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER5          (0x20)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER6          (0x40)
#define MCF5235_GPIO_PDDR_TIMER_PDDR_TIMER7          (0x80)
#define MCF5235_GPIO_PDDR_ETPU_PDDR_ETPU0            (0x01)
#define MCF5235_GPIO_PDDR_ETPU_PDDR_ETPU1            (0x02)
#define MCF5235_GPIO_PDDR_ETPU_PDDR_ETPU2            (0x04)
#define MCF5235_GPIO_PPDSDR_ADDR_PPDSDR_ADDR5        (0x20)
#define MCF5235_GPIO_PPDSDR_ADDR_PPDSDR_ADDR6        (0x40)
#define MCF5235_GPIO_PPDSDR_ADDR_PPDSDR_ADDR7        (0x80)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH0      (0x01)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH1      (0x02)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH2      (0x04)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH3      (0x08)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH4      (0x10)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH5      (0x20)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH6      (0x40)
#define MCF5235_GPIO_PPDSDR_DATAH_PPDSDR_DATAH7      (0x80)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL0      (0x01)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL1      (0x02)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL2      (0x04)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL3      (0x08)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL4      (0x10)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL5      (0x20)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL6      (0x40)
#define MCF5235_GPIO_PPDSDR_DATAL_PPDSDR_DATAL7      (0x80)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL0    (0x01)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL1    (0x02)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL2    (0x04)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL3    (0x08)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL4    (0x10)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL5    (0x20)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL6    (0x40)
#define MCF5235_GPIO_PPDSDR_BUSCTL_PPDSDR_BUSCTL7    (0x80)
#define MCF5235_GPIO_PPDSDR_BS_PPDSDR_BS0            (0x01)
#define MCF5235_GPIO_PPDSDR_BS_PPDSDR_BS1            (0x02)
#define MCF5235_GPIO_PPDSDR_BS_PPDSDR_BS2            (0x04)
#define MCF5235_GPIO_PPDSDR_BS_PPDSDR_BS3            (0x08)
#define MCF5235_GPIO_PPDSDR_FECI2C_PPDSDR_FECI2C0    (0x01)
#define MCF5235_GPIO_PPDSDR_FECI2C_PPDSDR_FECI2C1    (0x02)
#define MCF5235_GPIO_PPDSDR_FECI2C_PPDSDR_FECI2C2    (0x04)
#define MCF5235_GPIO_PPDSDR_FECI2C_PPDSDR_FECI2C3    (0x08)
#define MCF5235_GPIO_PPDSDR_CS_PPDSDR_CS1            (0x02)
#define MCF5235_GPIO_PPDSDR_CS_PPDSDR_CS2            (0x04)
#define MCF5235_GPIO_PPDSDR_CS_PPDSDR_CS3            (0x08)
#define MCF5235_GPIO_PPDSDR_CS_PPDSDR_CS4            (0x10)
#define MCF5235_GPIO_PPDSDR_CS_PPDSDR_CS5            (0x20)
#define MCF5235_GPIO_PPDSDR_CS_PPDSDR_CS6            (0x40)
#define MCF5235_GPIO_PPDSDR_CS_PPDSDR_CS7            (0x80)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM0      (0x01)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM1      (0x02)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM2      (0x04)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM3      (0x08)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM4      (0x10)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM5      (0x20)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM6      (0x40)
#define MCF5235_GPIO_PPDSDR_SDRAM_PPDSDR_SDRAM7      (0x80)
#define MCF5235_GPIO_PPDSDR_UARTH_PPDSDR_UARTH0      (0x01)
#define MCF5235_GPIO_PPDSDR_UARTH_PPDSDR_UARTH1      (0x02)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL0      (0x01)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL1      (0x02)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL2      (0x04)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL3      (0x08)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL4      (0x10)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL5      (0x20)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL6      (0x40)
#define MCF5235_GPIO_PPDSDR_UARTL_PPDSDR_UARTL7      (0x80)
#define MCF5235_GPIO_PPDSDR_QSPI_PPDSDR_QSPI0        (0x01)
#define MCF5235_GPIO_PPDSDR_QSPI_PPDSDR_QSPI1        (0x02)
#define MCF5235_GPIO_PPDSDR_QSPI_PPDSDR_QSPI2        (0x04)
#define MCF5235_GPIO_PPDSDR_QSPI_PPDSDR_QSPI3        (0x08)
#define MCF5235_GPIO_PPDSDR_QSPI_PPDSDR_QSPI4        (0x10)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER0      (0x01)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER1      (0x02)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER2      (0x04)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER3      (0x08)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER4      (0x10)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER5      (0x20)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER6      (0x40)
#define MCF5235_GPIO_PPDSDR_TIMER_PPDSDR_TIMER7      (0x80)
#define MCF5235_GPIO_PPDSDR_ETPU_PPDSDR_ETPU0        (0x01)
#define MCF5235_GPIO_PPDSDR_ETPU_PPDSDR_ETPU1        (0x02)
#define MCF5235_GPIO_PPDSDR_ETPU_PPDSDR_ETPU2        (0x04)
#define MCF5235_GPIO_PCLRR_ADDR_PCLRR_ADDR5          (0x20)
#define MCF5235_GPIO_PCLRR_ADDR_PCLRR_ADDR6          (0x40)
#define MCF5235_GPIO_PCLRR_ADDR_PCLRR_ADDR7          (0x80)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH0        (0x01)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH1        (0x02)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH2        (0x04)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH3        (0x08)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH4        (0x10)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH5        (0x20)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH6        (0x40)
#define MCF5235_GPIO_PCLRR_DATAH_PCLRR_DATAH7        (0x80)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL0        (0x01)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL1        (0x02)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL2        (0x04)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL3        (0x08)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL4        (0x10)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL5        (0x20)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL6        (0x40)
#define MCF5235_GPIO_PCLRR_DATAL_PCLRR_DATAL7        (0x80)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL0      (0x01)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL1      (0x02)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL2      (0x04)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL3      (0x08)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL4      (0x10)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL5      (0x20)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL6      (0x40)
#define MCF5235_GPIO_PCLRR_BUSCTL_PCLRR_BUSCTL7      (0x80)
#define MCF5235_GPIO_PCLRR_BS_PCLRR_BS0              (0x01)
#define MCF5235_GPIO_PCLRR_BS_PCLRR_BS1              (0x02)
#define MCF5235_GPIO_PCLRR_BS_PCLRR_BS2              (0x04)
#define MCF5235_GPIO_PCLRR_BS_PCLRR_BS3              (0x08)
#define MCF5235_GPIO_PCLRR_CS_PCLRR_CS1              (0x02)
#define MCF5235_GPIO_PCLRR_CS_PCLRR_CS2              (0x04)
#define MCF5235_GPIO_PCLRR_CS_PCLRR_CS3              (0x08)
#define MCF5235_GPIO_PCLRR_CS_PCLRR_CS4              (0x10)
#define MCF5235_GPIO_PCLRR_CS_PCLRR_CS5              (0x20)
#define MCF5235_GPIO_PCLRR_CS_PCLRR_CS6              (0x40)
#define MCF5235_GPIO_PCLRR_CS_PCLRR_CS7              (0x80)*/
#define MCF5235_GPIO_PCLRR_SDRAM_PCLRR_SDRAM0        (0x01)
#define MCF5235_GPIO_PCLRR_SDRAM_PCLRR_SDRAM1        (0x02)
#define MCF5235_GPIO_PCLRR_SDRAM_PCLRR_SDRAM2        (0x04)
#define MCF5235_GPIO_PCLRR_SDRAM_PCLRR_SDRAM3        (0x08)
#define MCF5235_GPIO_PCLRR_SDRAM_PCLRR_SDRAM4        (0x10)
#define MCF5235_GPIO_PCLRR_SDRAM_PCLRR_SDRAM5        (0x20)
#define MCF5235_GPIO_PCLRR_FECI2C_PCLRR_FECI2C0      (0x01)
#define MCF5235_GPIO_PCLRR_FECI2C_PCLRR_FECI2C1      (0x02)
#define MCF5235_GPIO_PCLRR_FECI2C_PCLRR_FECI2C2      (0x04)
#define MCF5235_GPIO_PCLRR_FECI2C_PCLRR_FECI2C3      (0x08)
#define MCF5235_GPIO_PCLRR_UARTH_PCLRR_UARTH0        (0x01)
#define MCF5235_GPIO_PCLRR_UARTH_PCLRR_UARTH1        (0x02)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL0        (0x01)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL1        (0x02)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL2        (0x04)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL3        (0x08)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL4        (0x10)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL5        (0x20)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL6        (0x40)
#define MCF5235_GPIO_PCLRR_UARTL_PCLRR_UARTL7        (0x80)
#define MCF5235_GPIO_PCLRR_QSPI_PCLRR_QSPI0          (0x01)
#define MCF5235_GPIO_PCLRR_QSPI_PCLRR_QSPI1          (0x02)
#define MCF5235_GPIO_PCLRR_QSPI_PCLRR_QSPI2          (0x04)
#define MCF5235_GPIO_PCLRR_QSPI_PCLRR_QSPI3          (0x08)
#define MCF5235_GPIO_PCLRR_QSPI_PCLRR_QSPI4          (0x10)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER0        (0x01)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER1        (0x02)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER2        (0x04)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER3        (0x08)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER4        (0x10)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER5        (0x20)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER6        (0x40)
#define MCF5235_GPIO_PCLRR_TIMER_PCLRR_TIMER7        (0x80)
#define MCF5235_GPIO_PCLRR_ETPU_PCLRR_ETPU0          (0x01)
#define MCF5235_GPIO_PCLRR_ETPU_PCLRR_ETPU1          (0x02)
#define MCF5235_GPIO_PCLRR_ETPU_PCLRR_ETPU2          (0x04)
#define MCF5235_GPIO_PAR_AD_PAR_DATAL                (0x01)
#define MCF5235_GPIO_PAR_AD_PAR_ADDR21               (0x20)
#define MCF5235_GPIO_PAR_AD_PAR_ADDR22               (0x40)
#define MCF5235_GPIO_PAR_AD_PAR_ADDR23               (0x80)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TIP(x)           (((x)&0x0003)<<0)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TS(x)            (((x)&0x0003)<<2)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TSIZ0            (0x0010)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TSIZ1            (0x0040)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_RWB              (0x0100)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TEA(x)           (((x)&0x0003)<<10)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TA               (0x1000)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_OE               (0x4000)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TEA_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TEA_DMA          (0x0800)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TEA_TEA          (0x0C00)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TS_GPIO          (0x0000)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TS_DMA           (0x0080)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TS_TS            (0x00C0)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TIP_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TIP_DMA          (0x0002)
#define MCF5235_GPIO_PAR_BUSCTL_PAR_TIP_TEA          (0x0003)
#define MCF5235_GPIO_PAR_BS_PAR_BS0                  (0x01)
#define MCF5235_GPIO_PAR_BS_PAR_BS1                  (0x02)
#define MCF5235_GPIO_PAR_BS_PAR_BS2                  (0x04)
#define MCF5235_GPIO_PAR_BS_PAR_BS3                  (0x08)
#define MCF5235_GPIO_PAR_CS_PAR_CS1                  (0x02)
#define MCF5235_GPIO_PAR_CS_PAR_CS2                  (0x04)
#define MCF5235_GPIO_PAR_CS_PAR_CS3                  (0x08)
#define MCF5235_GPIO_PAR_CS_PAR_CS4                  (0x10)
#define MCF5235_GPIO_PAR_CS_PAR_CS5                  (0x20)
#define MCF5235_GPIO_PAR_CS_PAR_CS6                  (0x40)
#define MCF5235_GPIO_PAR_CS_PAR_CS7                  (0x80)
#define MCF5235_GPIO_PAR_SDRAM_PAR_SDCS0             (0x01)
#define MCF5235_GPIO_PAR_SDRAM_PAR_SDCS1             (0x02)
#define MCF5235_GPIO_PAR_SDRAM_PAR_SCKE              (0x04)
#define MCF5235_GPIO_PAR_SDRAM_PAR_SRAS              (0x08)
#define MCF5235_GPIO_PAR_SDRAM_PAR_SCAS              (0x10)
#define MCF5235_GPIO_PAR_SDRAM_PAR_SDWE              (0x20)
#define MCF5235_GPIO_PAR_SDRAM_PAR_CSSDCS(x)         (((x)&0x03)<<6)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SDA(x)           (((x)&0x03)<<0)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SCL(x)           (((x)&0x03)<<2)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDIO(x)         (((x)&0x03)<<4)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDC(x)          (((x)&0x03)<<6)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDC_GPIO        (0x00)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDC_UART2       (0x40)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDC_I2C         (0x80)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDC_FEC         (0xC0)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDIO_GPIO       (0x00)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDIO_UART2      (0x10)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDIO_I2C        (0x20)
#define MCF5235_GPIO_PAR_FECI2C_PAR_EMDIO_FEC        (0x30)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SCL_GPIO         (0x00)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SCL_FLEX         (0x08)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SCL_I2C          (0x0C)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SDA_GPIO         (0x00)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SDA_FLEX         (0x02)
#define MCF5235_GPIO_PAR_FECI2C_PAR_SDA_I2C          (0x03)
#define MCF5235_GPIO_PAR_UART_PAR_U0RTS              (0x0001)
#define MCF5235_GPIO_PAR_UART_PAR_U0CTS              (0x0002)
#define MCF5235_GPIO_PAR_UART_PAR_U0TXD              (0x0004)
#define MCF5235_GPIO_PAR_UART_PAR_U0RXD              (0x0008)
#define MCF5235_GPIO_PAR_UART_PAR_U1RTS(x)           (((x)&0x0003)<<4)
#define MCF5235_GPIO_PAR_UART_PAR_U1CTS(x)           (((x)&0x0003)<<6)
#define MCF5235_GPIO_PAR_UART_PAR_U1TXD(x)           (((x)&0x0003)<<8)
#define MCF5235_GPIO_PAR_UART_PAR_U1RXD(x)           (((x)&0x0003)<<10)
#define MCF5235_GPIO_PAR_UART_PAR_U2TXD              (0x1000)
#define MCF5235_GPIO_PAR_UART_PAR_U2RXD              (0x2000)
#define MCF5235_GPIO_PAR_UART_PAR_CAN1EN             (0x4000)
#define MCF5235_GPIO_PAR_UART_PAR_DREQ2              (0x8000)
#define MCF5235_GPIO_PAR_UART_PAR_U1RXD_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_UART_PAR_U1RXD_FLEX         (0x0800)
#define MCF5235_GPIO_PAR_UART_PAR_U1RXD_UART1        (0x0C00)
#define MCF5235_GPIO_PAR_UART_PAR_U1TXD_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_UART_PAR_U1TXD_FLEX         (0x0200)
#define MCF5235_GPIO_PAR_UART_PAR_U1TXD_UART1        (0x0300)
#define MCF5235_GPIO_PAR_UART_PAR_U1CTS_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_UART_PAR_U1CTS_UART2        (0x0080)
#define MCF5235_GPIO_PAR_UART_PAR_U1CTS_UART1        (0x00C0)
#define MCF5235_GPIO_PAR_UART_PAR_U1RTS_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_UART_PAR_U1RTS_UART2        (0x0020)
#define MCF5235_GPIO_PAR_UART_PAR_U1RTS_UART1        (0x0030)
#define MCF5235_GPIO_PAR_QSPI_PAR_SCK(x)             (((x)&0x03)<<0)
#define MCF5235_GPIO_PAR_QSPI_PAR_DOUT               (0x04)
#define MCF5235_GPIO_PAR_QSPI_PAR_DIN(x)             (((x)&0x03)<<3)
#define MCF5235_GPIO_PAR_QSPI_PAR_PCS0               (0x20)
#define MCF5235_GPIO_PAR_QSPI_PAR_PCS1(x)            (((x)&0x03)<<6)
#define MCF5235_GPIO_PAR_QSPI_PAR_PCS1_GPIO          (0x00)
#define MCF5235_GPIO_PAR_QSPI_PAR_PCS1_SDRAMC        (0x80)
#define MCF5235_GPIO_PAR_QSPI_PAR_PCS1_QSPI          (0xC0)
#define MCF5235_GPIO_PAR_QSPI_PAR_DIN_GPIO           (0x00)
#define MCF5235_GPIO_PAR_QSPI_PAR_DIN_I2C            (0x10)
#define MCF5235_GPIO_PAR_QSPI_PAR_DIN_QSPI           (0x1C)
#define MCF5235_GPIO_PAR_QSPI_PAR_SCK_GPIO           (0x00)
#define MCF5235_GPIO_PAR_QSPI_PAR_SCK_I2C            (0x02)
#define MCF5235_GPIO_PAR_QSPI_PAR_SCK_QSPI           (0x03)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0OUT(x)          (((x)&0x0003)<<0)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1OUT(x)          (((x)&0x0003)<<2)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2OUT(x)          (((x)&0x0003)<<4)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3OUT(x)          (((x)&0x0003)<<6)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0IN(x)           (((x)&0x0003)<<8)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1IN(x)           (((x)&0x0003)<<10)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2IN(x)           (((x)&0x0003)<<12)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3IN(x)           (((x)&0x0003)<<14)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3IN_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3IN_QSPI         (0x4000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3IN_UART2        (0x8000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3IN_T3IN         (0xC000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2IN_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2IN_T2OUT        (0x1000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2IN_DMA          (0x2000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2IN_T2IN         (0x3000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1IN_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1IN_T1OUT        (0x0400)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1IN_DMA          (0x0800)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1IN_T1IN         (0x0C00)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0IN_GPIO         (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0IN_DMA          (0x0200)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0IN_T0IN         (0x0300)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3OUT_GPIO        (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3OUT_QSPI        (0x0040)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3OUT_UART2       (0x0080)
#define MCF5235_GPIO_PAR_TIMER_PAR_T3OUT_T3OUT       (0x00C0)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2OUT_GPIO        (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2OUT_DMA         (0x0020)
#define MCF5235_GPIO_PAR_TIMER_PAR_T2OUT_T2OUT       (0x0030)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1OUT_GPIO        (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1OUT_DMA         (0x0008)
#define MCF5235_GPIO_PAR_TIMER_PAR_T1OUT_T1OUT       (0x000C)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0OUT_GPIO        (0x0000)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0OUT_DMA         (0x0002)
#define MCF5235_GPIO_PAR_TIMER_PAR_T0OUT_T0OUT       (0x0003)
#define MCF5235_GPIO_PAR_ETPU_PAR_LTPU_ODIS          (0x01)
#define MCF5235_GPIO_PAR_ETPU_PAR_UTPU_ODIS          (0x02)
#define MCF5235_GPIO_PAR_ETPU_PAR_TCRCLK             (0x04)
#define MCF5235_GPIO_DSCR_EIM_DSCR_EIM0              (0x01)
#define MCF5235_GPIO_DSCR_EIM_DSCR_EIM1              (0x10)
#define MCF5235_GPIO_DSCR_ETPU_DSCR_ETPU_7_0         (0x01)
#define MCF5235_GPIO_DSCR_ETPU_DSCR_ETPU_15_8        (0x04)
#define MCF5235_GPIO_DSCR_ETPU_DSCR_ETPU_23_16       (0x10)
#define MCF5235_GPIO_DSCR_ETPU_DSCR_ETPU_31_24       (0x40)
#define MCF5235_GPIO_DSCR_FECI2C_DSCR_I2C            (0x01)
#define MCF5235_GPIO_DSCR_FECI2C_DSCR_FEC            (0x10)
#define MCF5235_GPIO_DSCR_UART_DSCR_UART0            (0x01)
#define MCF5235_GPIO_DSCR_UART_DSCR_UART1            (0x04)
#define MCF5235_GPIO_DSCR_UART_DSCR_UART2            (0x10)
#define MCF5235_GPIO_DSCR_UART_DSCR_IRQ              (0x40)
#define MCF5235_GPIO_DSCR_QSPI_DSCR_QSPI             (0x01)*/
#define MCF5235_GPIO_DSCR_TIMER_DSCR_TIMER           (0x01)

/*********************************************************************
*
* I2C Module (I2C)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_I2C_I2AR     (*(vuint8 *)(void*)(&__IPSBAR[0x000300]))
#define MCF5235_I2C_I2FDR    (*(vuint8 *)(void*)(&__IPSBAR[0x000304]))
#define MCF5235_I2C_I2CR     (*(vuint8 *)(void*)(&__IPSBAR[0x000308]))
#define MCF5235_I2C_I2SR     (*(vuint8 *)(void*)(&__IPSBAR[0x00030C]))
#define MCF5235_I2C_I2DR     (*(vuint8 *)(void*)(&__IPSBAR[0x000310]))
#define MCF5235_I2C_I2ICR    (*(vuint8 *)(void*)(&__IPSBAR[0x000320]))

/* Bit definitions and macros for MCF5235_I2C_I2AR */
#define MCF5235_I2C_I2AR_ADR(x)    (((x)&0x7F)<<1)
#define MCF5235_I2C_I2FDR_IC(x)    (((x)&0x3F)<<0)
#define MCF5235_I2C_I2CR_RSTA      (0x04)
#define MCF5235_I2C_I2CR_TXAK      (0x08)
#define MCF5235_I2C_I2CR_MTX       (0x10)
#define MCF5235_I2C_I2CR_MSTA      (0x20)
#define MCF5235_I2C_I2CR_IIEN      (0x40)
#define MCF5235_I2C_I2CR_IEN       (0x80)
#define MCF5235_I2C_I2SR_RXAK      (0x01)
#define MCF5235_I2C_I2SR_IIF       (0x02)
#define MCF5235_I2C_I2SR_SRW       (0x04)
#define MCF5235_I2C_I2SR_IAL       (0x10)
#define MCF5235_I2C_I2SR_IBB       (0x20)
#define MCF5235_I2C_I2SR_IAAS      (0x40)
#define MCF5235_I2C_I2SR_ICF       (0x80)
#define MCF5235_I2C_I2ICR_IE       (0x01)
#define MCF5235_I2C_I2ICR_RE       (0x02)
#define MCF5235_I2C_I2ICR_TE       (0x04)
#define MCF5235_I2C_I2ICR_BNBE     (0x08)

/*********************************************************************
*
* Interrupt Controller 0 (INTC0)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_INTC0_IPRH         (*(vuint32*)(void*)(&__IPSBAR[0x000C00]))
#define MCF5235_INTC0_IPRL         (*(vuint32*)(void*)(&__IPSBAR[0x000C04]))
#define MCF5235_INTC0_IMRH         (*(vuint32*)(void*)(&__IPSBAR[0x000C08]))
#define MCF5235_INTC0_IMRL         (*(vuint32*)(void*)(&__IPSBAR[0x000C0C]))
#define MCF5235_INTC0_INTFRCH      (*(vuint32*)(void*)(&__IPSBAR[0x000C10]))
#define MCF5235_INTC0_INTFRCL      (*(vuint32*)(void*)(&__IPSBAR[0x000C14]))
#define MCF5235_INTC0_IRLR         (*(vuint8 *)(void*)(&__IPSBAR[0x000C18]))
#define MCF5235_INTC0_IACKLPR      (*(vuint8 *)(void*)(&__IPSBAR[0x000C19]))
#define MCF5235_INTC0_ICR0         (*(vuint8 *)(void*)(&__IPSBAR[0x000C40]))
#define MCF5235_INTC0_ICR1         (*(vuint8 *)(void*)(&__IPSBAR[0x000C41]))
#define MCF5235_INTC0_ICR2         (*(vuint8 *)(void*)(&__IPSBAR[0x000C42]))
#define MCF5235_INTC0_ICR3         (*(vuint8 *)(void*)(&__IPSBAR[0x000C43]))
#define MCF5235_INTC0_ICR4         (*(vuint8 *)(void*)(&__IPSBAR[0x000C44]))
#define MCF5235_INTC0_ICR5         (*(vuint8 *)(void*)(&__IPSBAR[0x000C45]))
#define MCF5235_INTC0_ICR6         (*(vuint8 *)(void*)(&__IPSBAR[0x000C46]))
#define MCF5235_INTC0_ICR7         (*(vuint8 *)(void*)(&__IPSBAR[0x000C47]))
#define MCF5235_INTC0_ICR8         (*(vuint8 *)(void*)(&__IPSBAR[0x000C48]))
#define MCF5235_INTC0_ICR9         (*(vuint8 *)(void*)(&__IPSBAR[0x000C49]))
#define MCF5235_INTC0_ICR10        (*(vuint8 *)(void*)(&__IPSBAR[0x000C4A]))
#define MCF5235_INTC0_ICR11        (*(vuint8 *)(void*)(&__IPSBAR[0x000C4B]))
#define MCF5235_INTC0_ICR12        (*(vuint8 *)(void*)(&__IPSBAR[0x000C4C]))
#define MCF5235_INTC0_ICR13        (*(vuint8 *)(void*)(&__IPSBAR[0x000C4D]))
#define MCF5235_INTC0_ICR14        (*(vuint8 *)(void*)(&__IPSBAR[0x000C4E]))
#define MCF5235_INTC0_ICR15        (*(vuint8 *)(void*)(&__IPSBAR[0x000C4F]))
#define MCF5235_INTC0_ICR16        (*(vuint8 *)(void*)(&__IPSBAR[0x000C50]))
#define MCF5235_INTC0_ICR17        (*(vuint8 *)(void*)(&__IPSBAR[0x000C51]))
#define MCF5235_INTC0_ICR18        (*(vuint8 *)(void*)(&__IPSBAR[0x000C52]))
#define MCF5235_INTC0_ICR19        (*(vuint8 *)(void*)(&__IPSBAR[0x000C53]))
#define MCF5235_INTC0_ICR20        (*(vuint8 *)(void*)(&__IPSBAR[0x000C54]))
#define MCF5235_INTC0_ICR21        (*(vuint8 *)(void*)(&__IPSBAR[0x000C55]))
#define MCF5235_INTC0_ICR22        (*(vuint8 *)(void*)(&__IPSBAR[0x000C56]))
#define MCF5235_INTC0_ICR23        (*(vuint8 *)(void*)(&__IPSBAR[0x000C57]))
#define MCF5235_INTC0_ICR24        (*(vuint8 *)(void*)(&__IPSBAR[0x000C58]))
#define MCF5235_INTC0_ICR25        (*(vuint8 *)(void*)(&__IPSBAR[0x000C59]))
#define MCF5235_INTC0_ICR26        (*(vuint8 *)(void*)(&__IPSBAR[0x000C5A]))
#define MCF5235_INTC0_ICR27        (*(vuint8 *)(void*)(&__IPSBAR[0x000C5B]))
#define MCF5235_INTC0_ICR28        (*(vuint8 *)(void*)(&__IPSBAR[0x000C5C]))
#define MCF5235_INTC0_ICR29        (*(vuint8 *)(void*)(&__IPSBAR[0x000C5D]))
#define MCF5235_INTC0_ICR30        (*(vuint8 *)(void*)(&__IPSBAR[0x000C5E]))
#define MCF5235_INTC0_ICR31        (*(vuint8 *)(void*)(&__IPSBAR[0x000C5F]))
#define MCF5235_INTC0_ICR32        (*(vuint8 *)(void*)(&__IPSBAR[0x000C60]))
#define MCF5235_INTC0_ICR33        (*(vuint8 *)(void*)(&__IPSBAR[0x000C61]))
#define MCF5235_INTC0_ICR34        (*(vuint8 *)(void*)(&__IPSBAR[0x000C62]))
#define MCF5235_INTC0_ICR35        (*(vuint8 *)(void*)(&__IPSBAR[0x000C63]))
#define MCF5235_INTC0_ICR36        (*(vuint8 *)(void*)(&__IPSBAR[0x000C64]))
#define MCF5235_INTC0_ICR37        (*(vuint8 *)(void*)(&__IPSBAR[0x000C65]))
#define MCF5235_INTC0_ICR38        (*(vuint8 *)(void*)(&__IPSBAR[0x000C66]))
#define MCF5235_INTC0_ICR39        (*(vuint8 *)(void*)(&__IPSBAR[0x000C67]))
#define MCF5235_INTC0_ICR40        (*(vuint8 *)(void*)(&__IPSBAR[0x000C68]))
#define MCF5235_INTC0_ICR41        (*(vuint8 *)(void*)(&__IPSBAR[0x000C69]))
#define MCF5235_INTC0_ICR42        (*(vuint8 *)(void*)(&__IPSBAR[0x000C6A]))
#define MCF5235_INTC0_ICR43        (*(vuint8 *)(void*)(&__IPSBAR[0x000C6B]))
#define MCF5235_INTC0_ICR44        (*(vuint8 *)(void*)(&__IPSBAR[0x000C6C]))
#define MCF5235_INTC0_ICR45        (*(vuint8 *)(void*)(&__IPSBAR[0x000C6D]))
#define MCF5235_INTC0_ICR46        (*(vuint8 *)(void*)(&__IPSBAR[0x000C6E]))
#define MCF5235_INTC0_ICR47        (*(vuint8 *)(void*)(&__IPSBAR[0x000C6F]))
#define MCF5235_INTC0_ICR48        (*(vuint8 *)(void*)(&__IPSBAR[0x000C70]))
#define MCF5235_INTC0_ICR49        (*(vuint8 *)(void*)(&__IPSBAR[0x000C71]))
#define MCF5235_INTC0_ICR50        (*(vuint8 *)(void*)(&__IPSBAR[0x000C72]))
#define MCF5235_INTC0_ICR51        (*(vuint8 *)(void*)(&__IPSBAR[0x000C73]))
#define MCF5235_INTC0_ICR52        (*(vuint8 *)(void*)(&__IPSBAR[0x000C74]))
#define MCF5235_INTC0_ICR53        (*(vuint8 *)(void*)(&__IPSBAR[0x000C75]))
#define MCF5235_INTC0_ICR54        (*(vuint8 *)(void*)(&__IPSBAR[0x000C76]))
#define MCF5235_INTC0_ICR55        (*(vuint8 *)(void*)(&__IPSBAR[0x000C77]))
#define MCF5235_INTC0_ICR56        (*(vuint8 *)(void*)(&__IPSBAR[0x000C78]))
#define MCF5235_INTC0_ICR57        (*(vuint8 *)(void*)(&__IPSBAR[0x000C79]))
#define MCF5235_INTC0_ICR58        (*(vuint8 *)(void*)(&__IPSBAR[0x000C7A]))
#define MCF5235_INTC0_ICR59        (*(vuint8 *)(void*)(&__IPSBAR[0x000C7B]))
#define MCF5235_INTC0_ICR60        (*(vuint8 *)(void*)(&__IPSBAR[0x000C7C]))
#define MCF5235_INTC0_ICR61        (*(vuint8 *)(void*)(&__IPSBAR[0x000C7D]))
#define MCF5235_INTC0_ICR62        (*(vuint8 *)(void*)(&__IPSBAR[0x000C7E]))
#define MCF5235_INTC0_ICR63        (*(vuint8 *)(void*)(&__IPSBAR[0x000C7F]))
#define MCF5235_INTC0_ICRn(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x000C40+((x)*0x001)]))
#define MCF5235_INTC0_SWIACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CE0]))
#define MCF5235_INTC0_L1IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CE4]))
#define MCF5235_INTC0_L2IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CE8]))
#define MCF5235_INTC0_L3IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CEC]))
#define MCF5235_INTC0_L4IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CF0]))
#define MCF5235_INTC0_L5IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CF4]))
#define MCF5235_INTC0_L6IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CF8]))
#define MCF5235_INTC0_L7IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000CFC]))
#define MCF5235_INTC0_LnIACK(x)    (*(vuint8 *)(void*)(&__IPSBAR[0x000CE4+((x)*0x004)]))
#define MCF5235_INTC1_IPRH         (*(vuint32*)(void*)(&__IPSBAR[0x000D00]))
#define MCF5235_INTC1_IPRL         (*(vuint32*)(void*)(&__IPSBAR[0x000D04]))
#define MCF5235_INTC1_IMRH         (*(vuint32*)(void*)(&__IPSBAR[0x000D08]))
#define MCF5235_INTC1_IMRL         (*(vuint32*)(void*)(&__IPSBAR[0x000D0C]))
#define MCF5235_INTC1_INTFRCH      (*(vuint32*)(void*)(&__IPSBAR[0x000D10]))
#define MCF5235_INTC1_INTFRCL      (*(vuint32*)(void*)(&__IPSBAR[0x000D14]))
#define MCF5235_INTC1_IRLR         (*(vuint8 *)(void*)(&__IPSBAR[0x000D18]))
#define MCF5235_INTC1_IACKLPR      (*(vuint8 *)(void*)(&__IPSBAR[0x000D19]))
#define MCF5235_INTC1_ICR0         (*(vuint8 *)(void*)(&__IPSBAR[0x000D40]))
#define MCF5235_INTC1_ICR1         (*(vuint8 *)(void*)(&__IPSBAR[0x000D41]))
#define MCF5235_INTC1_ICR2         (*(vuint8 *)(void*)(&__IPSBAR[0x000D42]))
#define MCF5235_INTC1_ICR3         (*(vuint8 *)(void*)(&__IPSBAR[0x000D43]))
#define MCF5235_INTC1_ICR4         (*(vuint8 *)(void*)(&__IPSBAR[0x000D44]))
#define MCF5235_INTC1_ICR5         (*(vuint8 *)(void*)(&__IPSBAR[0x000D45]))
#define MCF5235_INTC1_ICR6         (*(vuint8 *)(void*)(&__IPSBAR[0x000D46]))
#define MCF5235_INTC1_ICR7         (*(vuint8 *)(void*)(&__IPSBAR[0x000D47]))
#define MCF5235_INTC1_ICR8         (*(vuint8 *)(void*)(&__IPSBAR[0x000D48]))
#define MCF5235_INTC1_ICR9         (*(vuint8 *)(void*)(&__IPSBAR[0x000D49]))
#define MCF5235_INTC1_ICR10        (*(vuint8 *)(void*)(&__IPSBAR[0x000D4A]))
#define MCF5235_INTC1_ICR11        (*(vuint8 *)(void*)(&__IPSBAR[0x000D4B]))
#define MCF5235_INTC1_ICR12        (*(vuint8 *)(void*)(&__IPSBAR[0x000D4C]))
#define MCF5235_INTC1_ICR13        (*(vuint8 *)(void*)(&__IPSBAR[0x000D4D]))
#define MCF5235_INTC1_ICR14        (*(vuint8 *)(void*)(&__IPSBAR[0x000D4E]))
#define MCF5235_INTC1_ICR15        (*(vuint8 *)(void*)(&__IPSBAR[0x000D4F]))
#define MCF5235_INTC1_ICR16        (*(vuint8 *)(void*)(&__IPSBAR[0x000D50]))
#define MCF5235_INTC1_ICR17        (*(vuint8 *)(void*)(&__IPSBAR[0x000D51]))
#define MCF5235_INTC1_ICR18        (*(vuint8 *)(void*)(&__IPSBAR[0x000D52]))
#define MCF5235_INTC1_ICR19        (*(vuint8 *)(void*)(&__IPSBAR[0x000D53]))
#define MCF5235_INTC1_ICR20        (*(vuint8 *)(void*)(&__IPSBAR[0x000D54]))
#define MCF5235_INTC1_ICR21        (*(vuint8 *)(void*)(&__IPSBAR[0x000D55]))
#define MCF5235_INTC1_ICR22        (*(vuint8 *)(void*)(&__IPSBAR[0x000D56]))
#define MCF5235_INTC1_ICR23        (*(vuint8 *)(void*)(&__IPSBAR[0x000D57]))
#define MCF5235_INTC1_ICR24        (*(vuint8 *)(void*)(&__IPSBAR[0x000D58]))
#define MCF5235_INTC1_ICR25        (*(vuint8 *)(void*)(&__IPSBAR[0x000D59]))
#define MCF5235_INTC1_ICR26        (*(vuint8 *)(void*)(&__IPSBAR[0x000D5A]))
#define MCF5235_INTC1_ICR27        (*(vuint8 *)(void*)(&__IPSBAR[0x000D5B]))
#define MCF5235_INTC1_ICR28        (*(vuint8 *)(void*)(&__IPSBAR[0x000D5C]))
#define MCF5235_INTC1_ICR29        (*(vuint8 *)(void*)(&__IPSBAR[0x000D5D]))
#define MCF5235_INTC1_ICR30        (*(vuint8 *)(void*)(&__IPSBAR[0x000D5E]))
#define MCF5235_INTC1_ICR31        (*(vuint8 *)(void*)(&__IPSBAR[0x000D5F]))
#define MCF5235_INTC1_ICR32        (*(vuint8 *)(void*)(&__IPSBAR[0x000D60]))
#define MCF5235_INTC1_ICR33        (*(vuint8 *)(void*)(&__IPSBAR[0x000D61]))
#define MCF5235_INTC1_ICR34        (*(vuint8 *)(void*)(&__IPSBAR[0x000D62]))
#define MCF5235_INTC1_ICR35        (*(vuint8 *)(void*)(&__IPSBAR[0x000D63]))
#define MCF5235_INTC1_ICR36        (*(vuint8 *)(void*)(&__IPSBAR[0x000D64]))
#define MCF5235_INTC1_ICR37        (*(vuint8 *)(void*)(&__IPSBAR[0x000D65]))
#define MCF5235_INTC1_ICR38        (*(vuint8 *)(void*)(&__IPSBAR[0x000D66]))
#define MCF5235_INTC1_ICR39        (*(vuint8 *)(void*)(&__IPSBAR[0x000D67]))
#define MCF5235_INTC1_ICR40        (*(vuint8 *)(void*)(&__IPSBAR[0x000D68]))
#define MCF5235_INTC1_ICR41        (*(vuint8 *)(void*)(&__IPSBAR[0x000D69]))
#define MCF5235_INTC1_ICR42        (*(vuint8 *)(void*)(&__IPSBAR[0x000D6A]))
#define MCF5235_INTC1_ICR43        (*(vuint8 *)(void*)(&__IPSBAR[0x000D6B]))
#define MCF5235_INTC1_ICR44        (*(vuint8 *)(void*)(&__IPSBAR[0x000D6C]))
#define MCF5235_INTC1_ICR45        (*(vuint8 *)(void*)(&__IPSBAR[0x000D6D]))
#define MCF5235_INTC1_ICR46        (*(vuint8 *)(void*)(&__IPSBAR[0x000D6E]))
#define MCF5235_INTC1_ICR47        (*(vuint8 *)(void*)(&__IPSBAR[0x000D6F]))
#define MCF5235_INTC1_ICR48        (*(vuint8 *)(void*)(&__IPSBAR[0x000D70]))
#define MCF5235_INTC1_ICR49        (*(vuint8 *)(void*)(&__IPSBAR[0x000D71]))
#define MCF5235_INTC1_ICR50        (*(vuint8 *)(void*)(&__IPSBAR[0x000D72]))
#define MCF5235_INTC1_ICR51        (*(vuint8 *)(void*)(&__IPSBAR[0x000D73]))
#define MCF5235_INTC1_ICR52        (*(vuint8 *)(void*)(&__IPSBAR[0x000D74]))
#define MCF5235_INTC1_ICR53        (*(vuint8 *)(void*)(&__IPSBAR[0x000D75]))
#define MCF5235_INTC1_ICR54        (*(vuint8 *)(void*)(&__IPSBAR[0x000D76]))
#define MCF5235_INTC1_ICR55        (*(vuint8 *)(void*)(&__IPSBAR[0x000D77]))
#define MCF5235_INTC1_ICR56        (*(vuint8 *)(void*)(&__IPSBAR[0x000D78]))
#define MCF5235_INTC1_ICR57        (*(vuint8 *)(void*)(&__IPSBAR[0x000D79]))
#define MCF5235_INTC1_ICR58        (*(vuint8 *)(void*)(&__IPSBAR[0x000D7A]))
#define MCF5235_INTC1_ICR59        (*(vuint8 *)(void*)(&__IPSBAR[0x000D7B]))
#define MCF5235_INTC1_ICR60        (*(vuint8 *)(void*)(&__IPSBAR[0x000D7C]))
#define MCF5235_INTC1_ICR61        (*(vuint8 *)(void*)(&__IPSBAR[0x000D7D]))
#define MCF5235_INTC1_ICR62        (*(vuint8 *)(void*)(&__IPSBAR[0x000D7E]))
#define MCF5235_INTC1_ICR63        (*(vuint8 *)(void*)(&__IPSBAR[0x000D7F]))
#define MCF5235_INTC1_ICRn(x)      (*(vuint8 *)(void*)(&__IPSBAR[0x000D40+((x)*0x001)]))
#define MCF5235_INTC1_SWIACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DE0]))
#define MCF5235_INTC1_L1IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DE4]))
#define MCF5235_INTC1_L2IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DE8]))
#define MCF5235_INTC1_L3IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DEC]))
#define MCF5235_INTC1_L4IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DF0]))
#define MCF5235_INTC1_L5IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DF4]))
#define MCF5235_INTC1_L6IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DF8]))
#define MCF5235_INTC1_L7IACK       (*(vuint8 *)(void*)(&__IPSBAR[0x000DFC]))
#define MCF5235_INTC1_LnIACK(x)    (*(vuint8 *)(void*)(&__IPSBAR[0x000DE4+((x)*0x004)]))

/* Bit definitions and macros for MCF5235_INTC0_IPRH */
#define MCF5235_INTC0_IPRH_INT32          (0x00000001)
#define MCF5235_INTC0_IPRH_INT33          (0x00000002)
#define MCF5235_INTC0_IPRH_INT34          (0x00000004)
#define MCF5235_INTC0_IPRH_INT35          (0x00000008)
#define MCF5235_INTC0_IPRH_INT36          (0x00000010)
#define MCF5235_INTC0_IPRH_INT37          (0x00000020)
#define MCF5235_INTC0_IPRH_INT38          (0x00000040)
#define MCF5235_INTC0_IPRH_INT39          (0x00000080)
#define MCF5235_INTC0_IPRH_INT40          (0x00000100)
#define MCF5235_INTC0_IPRH_INT41          (0x00000200)
#define MCF5235_INTC0_IPRH_INT42          (0x00000400)
#define MCF5235_INTC0_IPRH_INT43          (0x00000800)
#define MCF5235_INTC0_IPRH_INT44          (0x00001000)
#define MCF5235_INTC0_IPRH_INT45          (0x00002000)
#define MCF5235_INTC0_IPRH_INT46          (0x00004000)
#define MCF5235_INTC0_IPRH_INT47          (0x00008000)
#define MCF5235_INTC0_IPRH_INT48          (0x00010000)
#define MCF5235_INTC0_IPRH_INT49          (0x00020000)
#define MCF5235_INTC0_IPRH_INT50          (0x00040000)
#define MCF5235_INTC0_IPRH_INT51          (0x00080000)
#define MCF5235_INTC0_IPRH_INT52          (0x00100000)
#define MCF5235_INTC0_IPRH_INT53          (0x00200000)
#define MCF5235_INTC0_IPRH_INT54          (0x00400000)
#define MCF5235_INTC0_IPRH_INT55          (0x00800000)
#define MCF5235_INTC0_IPRH_INT56          (0x01000000)
#define MCF5235_INTC0_IPRH_INT57          (0x02000000)
#define MCF5235_INTC0_IPRH_INT58          (0x04000000)
#define MCF5235_INTC0_IPRH_INT59          (0x08000000)
#define MCF5235_INTC0_IPRH_INT60          (0x10000000)
#define MCF5235_INTC0_IPRH_INT61          (0x20000000)
#define MCF5235_INTC0_IPRH_INT62          (0x40000000)
#define MCF5235_INTC0_IPRH_INT63          (0x80000000)
#define MCF5235_INTC0_IPRL_INT1           (0x00000002)
#define MCF5235_INTC0_IPRL_INT2           (0x00000004)
#define MCF5235_INTC0_IPRL_INT3           (0x00000008)
#define MCF5235_INTC0_IPRL_INT4           (0x00000010)
#define MCF5235_INTC0_IPRL_INT5           (0x00000020)
#define MCF5235_INTC0_IPRL_INT6           (0x00000040)
#define MCF5235_INTC0_IPRL_INT7           (0x00000080)
#define MCF5235_INTC0_IPRL_INT8           (0x00000100)
#define MCF5235_INTC0_IPRL_INT9           (0x00000200)
#define MCF5235_INTC0_IPRL_INT10          (0x00000400)
#define MCF5235_INTC0_IPRL_INT11          (0x00000800)
#define MCF5235_INTC0_IPRL_INT12          (0x00001000)
#define MCF5235_INTC0_IPRL_INT13          (0x00002000)
#define MCF5235_INTC0_IPRL_INT14          (0x00004000)
#define MCF5235_INTC0_IPRL_INT15          (0x00008000)
#define MCF5235_INTC0_IPRL_INT16          (0x00010000)
#define MCF5235_INTC0_IPRL_INT17          (0x00020000)
#define MCF5235_INTC0_IPRL_INT18          (0x00040000)
#define MCF5235_INTC0_IPRL_INT19          (0x00080000)
#define MCF5235_INTC0_IPRL_INT20          (0x00100000)
#define MCF5235_INTC0_IPRL_INT21          (0x00200000)
#define MCF5235_INTC0_IPRL_INT22          (0x00400000)
#define MCF5235_INTC0_IPRL_INT23          (0x00800000)
#define MCF5235_INTC0_IPRL_INT24          (0x01000000)
#define MCF5235_INTC0_IPRL_INT25          (0x02000000)
#define MCF5235_INTC0_IPRL_INT26          (0x04000000)
#define MCF5235_INTC0_IPRL_INT27          (0x08000000)
#define MCF5235_INTC0_IPRL_INT28          (0x10000000)
#define MCF5235_INTC0_IPRL_INT29          (0x20000000)
#define MCF5235_INTC0_IPRL_INT30          (0x40000000)
#define MCF5235_INTC0_IPRL_INT31          (0x80000000)
#define MCF5235_INTC0_IMRH_INT32     (0x00000001)
#define MCF5235_INTC0_IMRH_INT33     (0x00000002)
#define MCF5235_INTC0_IMRH_INT34     (0x00000004)
#define MCF5235_INTC0_IMRH_INT35     (0x00000008)
#define MCF5235_INTC0_IMRH_INT36     (0x00000010)
#define MCF5235_INTC0_IMRH_INT37     (0x00000020)
#define MCF5235_INTC0_IMRH_INT38     (0x00000040)
#define MCF5235_INTC0_IMRH_INT39     (0x00000080)
#define MCF5235_INTC0_IMRH_INT40     (0x00000100)
#define MCF5235_INTC0_IMRH_INT41     (0x00000200)
#define MCF5235_INTC0_IMRH_INT42     (0x00000400)
#define MCF5235_INTC0_IMRH_INT43     (0x00000800)
#define MCF5235_INTC0_IMRH_INT44     (0x00001000)
#define MCF5235_INTC0_IMRH_INT45     (0x00002000)
#define MCF5235_INTC0_IMRH_INT46     (0x00004000)
#define MCF5235_INTC0_IMRH_INT47     (0x00008000)
#define MCF5235_INTC0_IMRH_INT48     (0x00010000)
#define MCF5235_INTC0_IMRH_INT49     (0x00020000)
#define MCF5235_INTC0_IMRH_INT50     (0x00040000)
#define MCF5235_INTC0_IMRH_INT51     (0x00080000)
#define MCF5235_INTC0_IMRH_INT52     (0x00100000)
#define MCF5235_INTC0_IMRH_INT53     (0x00200000)
#define MCF5235_INTC0_IMRH_INT54     (0x00400000)
#define MCF5235_INTC0_IMRH_INT55     (0x00800000)
#define MCF5235_INTC0_IMRH_INT56     (0x01000000)
#define MCF5235_INTC0_IMRH_INT57     (0x02000000)
#define MCF5235_INTC0_IMRH_INT58     (0x04000000)
#define MCF5235_INTC0_IMRH_INT59     (0x08000000)
#define MCF5235_INTC0_IMRH_INT60     (0x10000000)
#define MCF5235_INTC0_IMRH_INT61     (0x20000000)
#define MCF5235_INTC0_IMRH_INT62     (0x40000000)
#define MCF5235_INTC0_IMRH_INT63     (0x80000000)
#define MCF5235_INTC0_IMRL_MASKALL   (0x00000001)
#define MCF5235_INTC0_IMRL_INT1      (0x00000002)
#define MCF5235_INTC0_IMRL_INT2      (0x00000004)
#define MCF5235_INTC0_IMRL_INT3      (0x00000008)
#define MCF5235_INTC0_IMRL_INT4      (0x00000010)
#define MCF5235_INTC0_IMRL_INT5      (0x00000020)
#define MCF5235_INTC0_IMRL_INT6      (0x00000040)
#define MCF5235_INTC0_IMRL_INT7      (0x00000080)
#define MCF5235_INTC0_IMRL_INT8      (0x00000100)
#define MCF5235_INTC0_IMRL_INT9      (0x00000200)
#define MCF5235_INTC0_IMRL_INT10     (0x00000400)
#define MCF5235_INTC0_IMRL_INT11     (0x00000800)
#define MCF5235_INTC0_IMRL_INT12     (0x00001000)
#define MCF5235_INTC0_IMRL_INT13     (0x00002000)
#define MCF5235_INTC0_IMRL_INT14     (0x00004000)
#define MCF5235_INTC0_IMRL_INT15     (0x00008000)
#define MCF5235_INTC0_IMRL_INT16     (0x00010000)
#define MCF5235_INTC0_IMRL_INT17     (0x00020000)
#define MCF5235_INTC0_IMRL_INT18     (0x00040000)
#define MCF5235_INTC0_IMRL_INT19     (0x00080000)
#define MCF5235_INTC0_IMRL_INT20     (0x00100000)
#define MCF5235_INTC0_IMRL_INT21     (0x00200000)
#define MCF5235_INTC0_IMRL_INT22     (0x00400000)
#define MCF5235_INTC0_IMRL_INT23     (0x00800000)
#define MCF5235_INTC0_IMRL_INT24     (0x01000000)
#define MCF5235_INTC0_IMRL_INT25     (0x02000000)
#define MCF5235_INTC0_IMRL_INT26     (0x04000000)
#define MCF5235_INTC0_IMRL_INT27     (0x08000000)
#define MCF5235_INTC0_IMRL_INT28     (0x10000000)
#define MCF5235_INTC0_IMRL_INT29     (0x20000000)
#define MCF5235_INTC0_IMRL_INT30     (0x40000000)
#define MCF5235_INTC0_IMRL_INT31     (0x80000000)
#define MCF5235_INTC0_INTFRCH_INTFRC32    (0x00000001)
#define MCF5235_INTC0_INTFRCH_INTFRC33    (0x00000002)
#define MCF5235_INTC0_INTFRCH_INTFRC34    (0x00000004)
#define MCF5235_INTC0_INTFRCH_INTFRC35    (0x00000008)
#define MCF5235_INTC0_INTFRCH_INTFRC36    (0x00000010)
#define MCF5235_INTC0_INTFRCH_INTFRC37    (0x00000020)
#define MCF5235_INTC0_INTFRCH_INTFRC38    (0x00000040)
#define MCF5235_INTC0_INTFRCH_INTFRC39    (0x00000080)
#define MCF5235_INTC0_INTFRCH_INTFRC40    (0x00000100)
#define MCF5235_INTC0_INTFRCH_INTFRC41    (0x00000200)
#define MCF5235_INTC0_INTFRCH_INTFRC42    (0x00000400)
#define MCF5235_INTC0_INTFRCH_INTFRC43    (0x00000800)
#define MCF5235_INTC0_INTFRCH_INTFRC44    (0x00001000)
#define MCF5235_INTC0_INTFRCH_INTFRC45    (0x00002000)
#define MCF5235_INTC0_INTFRCH_INTFRC46    (0x00004000)
#define MCF5235_INTC0_INTFRCH_INTFRC47    (0x00008000)
#define MCF5235_INTC0_INTFRCH_INTFRC48    (0x00010000)
#define MCF5235_INTC0_INTFRCH_INTFRC49    (0x00020000)
#define MCF5235_INTC0_INTFRCH_INTFRC50    (0x00040000)
#define MCF5235_INTC0_INTFRCH_INTFRC51    (0x00080000)
#define MCF5235_INTC0_INTFRCH_INTFRC52    (0x00100000)
#define MCF5235_INTC0_INTFRCH_INTFRC53    (0x00200000)
#define MCF5235_INTC0_INTFRCH_INTFRC54    (0x00400000)
#define MCF5235_INTC0_INTFRCH_INTFRC55    (0x00800000)
#define MCF5235_INTC0_INTFRCH_INTFRC56    (0x01000000)
#define MCF5235_INTC0_INTFRCH_INTFRC57    (0x02000000)
#define MCF5235_INTC0_INTFRCH_INTFRC58    (0x04000000)
#define MCF5235_INTC0_INTFRCH_INTFRC59    (0x08000000)
#define MCF5235_INTC0_INTFRCH_INTFRC60    (0x10000000)
#define MCF5235_INTC0_INTFRCH_INTFRC61    (0x20000000)
#define MCF5235_INTC0_INTFRCH_INTFRC62    (0x40000000)
#define MCF5235_INTC0_INTFRCH_INTFRC63    (0x80000000)
#define MCF5235_INTC0_INTFRCL_INTFRC1     (0x00000002)
#define MCF5235_INTC0_INTFRCL_INTFRC2     (0x00000004)
#define MCF5235_INTC0_INTFRCL_INTFRC3     (0x00000008)
#define MCF5235_INTC0_INTFRCL_INTFRC4     (0x00000010)
#define MCF5235_INTC0_INTFRCL_INTFRC5     (0x00000020)
#define MCF5235_INTC0_INTFRCL_INT6        (0x00000040)
#define MCF5235_INTC0_INTFRCL_INT7        (0x00000080)
#define MCF5235_INTC0_INTFRCL_INT8        (0x00000100)
#define MCF5235_INTC0_INTFRCL_INT9        (0x00000200)
#define MCF5235_INTC0_INTFRCL_INT10       (0x00000400)
#define MCF5235_INTC0_INTFRCL_INTFRC11    (0x00000800)
#define MCF5235_INTC0_INTFRCL_INTFRC12    (0x00001000)
#define MCF5235_INTC0_INTFRCL_INTFRC13    (0x00002000)
#define MCF5235_INTC0_INTFRCL_INTFRC14    (0x00004000)
#define MCF5235_INTC0_INTFRCL_INT15       (0x00008000)
#define MCF5235_INTC0_INTFRCL_INTFRC16    (0x00010000)
#define MCF5235_INTC0_INTFRCL_INTFRC17    (0x00020000)
#define MCF5235_INTC0_INTFRCL_INTFRC18    (0x00040000)
#define MCF5235_INTC0_INTFRCL_INTFRC19    (0x00080000)
#define MCF5235_INTC0_INTFRCL_INTFRC20    (0x00100000)
#define MCF5235_INTC0_INTFRCL_INTFRC21    (0x00200000)
#define MCF5235_INTC0_INTFRCL_INTFRC22    (0x00400000)
#define MCF5235_INTC0_INTFRCL_INTFRC23    (0x00800000)
#define MCF5235_INTC0_INTFRCL_INTFRC24    (0x01000000)
#define MCF5235_INTC0_INTFRCL_INTFRC25    (0x02000000)
#define MCF5235_INTC0_INTFRCL_INTFRC26    (0x04000000)
#define MCF5235_INTC0_INTFRCL_INTFRC27    (0x08000000)
#define MCF5235_INTC0_INTFRCL_INTFRC28    (0x10000000)
#define MCF5235_INTC0_INTFRCL_INTFRC29    (0x20000000)
#define MCF5235_INTC0_INTFRCL_INTFRC30    (0x40000000)
#define MCF5235_INTC0_INTFRCL_INTFRC31    (0x80000000)
#define MCF5235_INTC0_IRLR_IRQ(x)         (((x)&0x7F)<<1)
#define MCF5235_INTC0_IACKLPR_PRI(x)      (((x)&0x0F)<<0)
#define MCF5235_INTC0_IACKLPR_LEVEL(x)    (((x)&0x07)<<4)
#define MCF5235_INTC_ICR_IP(x)          (((x)&0x07)<<0)
#define MCF5235_INTC_ICR_IL(x)          (((x)&0x07)<<3)
#define MCF5235_INTC1_IPRH_INT32          (0x00000001)
#define MCF5235_INTC1_IPRH_INT33          (0x00000002)
#define MCF5235_INTC1_IPRH_INT34          (0x00000004)
#define MCF5235_INTC1_IPRH_INT35          (0x00000008)
#define MCF5235_INTC1_IPRH_INT36          (0x00000010)
#define MCF5235_INTC1_IPRH_INT37          (0x00000020)
#define MCF5235_INTC1_IPRH_INT38          (0x00000040)
#define MCF5235_INTC1_IPRH_INT39          (0x00000080)
#define MCF5235_INTC1_IPRH_INT40          (0x00000100)
#define MCF5235_INTC1_IPRH_INT41          (0x00000200)
#define MCF5235_INTC1_IPRH_INT42          (0x00000400)
#define MCF5235_INTC1_IPRH_INT43          (0x00000800)
#define MCF5235_INTC1_IPRH_INT44          (0x00001000)
#define MCF5235_INTC1_IPRH_INT45          (0x00002000)
#define MCF5235_INTC1_IPRH_INT46          (0x00004000)
#define MCF5235_INTC1_IPRH_INT47          (0x00008000)
#define MCF5235_INTC1_IPRH_INT48          (0x00010000)
#define MCF5235_INTC1_IPRH_INT49          (0x00020000)
#define MCF5235_INTC1_IPRH_INT50          (0x00040000)
#define MCF5235_INTC1_IPRH_INT51          (0x00080000)
#define MCF5235_INTC1_IPRH_INT52          (0x00100000)
#define MCF5235_INTC1_IPRH_INT53          (0x00200000)
#define MCF5235_INTC1_IPRH_INT54          (0x00400000)
#define MCF5235_INTC1_IPRH_INT55          (0x00800000)
#define MCF5235_INTC1_IPRH_INT56          (0x01000000)
#define MCF5235_INTC1_IPRH_INT57          (0x02000000)
#define MCF5235_INTC1_IPRH_INT58          (0x04000000)
#define MCF5235_INTC1_IPRH_INT59          (0x08000000)
#define MCF5235_INTC1_IPRH_INT60          (0x10000000)
#define MCF5235_INTC1_IPRH_INT61          (0x20000000)
#define MCF5235_INTC1_IPRH_INT62          (0x40000000)
#define MCF5235_INTC1_IPRH_INT63          (0x80000000)
#define MCF5235_INTC1_IPRL_INT1           (0x00000002)
#define MCF5235_INTC1_IPRL_INT2           (0x00000004)
#define MCF5235_INTC1_IPRL_INT3           (0x00000008)
#define MCF5235_INTC1_IPRL_INT4           (0x00000010)
#define MCF5235_INTC1_IPRL_INT5           (0x00000020)
#define MCF5235_INTC1_IPRL_INT6           (0x00000040)
#define MCF5235_INTC1_IPRL_INT7           (0x00000080)
#define MCF5235_INTC1_IPRL_INT8           (0x00000100)
#define MCF5235_INTC1_IPRL_INT9           (0x00000200)
#define MCF5235_INTC1_IPRL_INT10          (0x00000400)
#define MCF5235_INTC1_IPRL_INT11          (0x00000800)
#define MCF5235_INTC1_IPRL_INT12          (0x00001000)
#define MCF5235_INTC1_IPRL_INT13          (0x00002000)
#define MCF5235_INTC1_IPRL_INT14          (0x00004000)
#define MCF5235_INTC1_IPRL_INT15          (0x00008000)
#define MCF5235_INTC1_IPRL_INT16          (0x00010000)
#define MCF5235_INTC1_IPRL_INT17          (0x00020000)
#define MCF5235_INTC1_IPRL_INT18          (0x00040000)
#define MCF5235_INTC1_IPRL_INT19          (0x00080000)
#define MCF5235_INTC1_IPRL_INT20          (0x00100000)
#define MCF5235_INTC1_IPRL_INT21          (0x00200000)
#define MCF5235_INTC1_IPRL_INT22          (0x00400000)
#define MCF5235_INTC1_IPRL_INT23          (0x00800000)
#define MCF5235_INTC1_IPRL_INT24          (0x01000000)
#define MCF5235_INTC1_IPRL_INT25          (0x02000000)
#define MCF5235_INTC1_IPRL_INT26          (0x04000000)
#define MCF5235_INTC1_IPRL_INT27          (0x08000000)
#define MCF5235_INTC1_IPRL_INT28          (0x10000000)
#define MCF5235_INTC1_IPRL_INT29          (0x20000000)
#define MCF5235_INTC1_IPRL_INT30          (0x40000000)
#define MCF5235_INTC1_IPRL_INT31          (0x80000000)
#define MCF5235_INTC1_IMRH_INT_MASK32     (0x00000001)
#define MCF5235_INTC1_IMRH_INT_MASK33     (0x00000002)
#define MCF5235_INTC1_IMRH_INT_MASK34     (0x00000004)
#define MCF5235_INTC1_IMRH_INT_MASK35     (0x00000008)
#define MCF5235_INTC1_IMRH_INT_MASK36     (0x00000010)
#define MCF5235_INTC1_IMRH_INT_MASK37     (0x00000020)
#define MCF5235_INTC1_IMRH_INT_MASK38     (0x00000040)
#define MCF5235_INTC1_IMRH_INT_MASK39     (0x00000080)
#define MCF5235_INTC1_IMRH_INT_MASK40     (0x00000100)
#define MCF5235_INTC1_IMRH_INT_MASK41     (0x00000200)
#define MCF5235_INTC1_IMRH_INT_MASK42     (0x00000400)
#define MCF5235_INTC1_IMRH_INT_MASK43     (0x00000800)
#define MCF5235_INTC1_IMRH_INT_MASK44     (0x00001000)
#define MCF5235_INTC1_IMRH_INT_MASK45     (0x00002000)
#define MCF5235_INTC1_IMRH_INT_MASK46     (0x00004000)
#define MCF5235_INTC1_IMRH_INT_MASK47     (0x00008000)
#define MCF5235_INTC1_IMRH_INT_MASK48     (0x00010000)
#define MCF5235_INTC1_IMRH_INT_MASK49     (0x00020000)
#define MCF5235_INTC1_IMRH_INT_MASK50     (0x00040000)
#define MCF5235_INTC1_IMRH_INT_MASK51     (0x00080000)
#define MCF5235_INTC1_IMRH_INT_MASK52     (0x00100000)
#define MCF5235_INTC1_IMRH_INT_MASK53     (0x00200000)
#define MCF5235_INTC1_IMRH_INT_MASK54     (0x00400000)
#define MCF5235_INTC1_IMRH_INT_MASK55     (0x00800000)
#define MCF5235_INTC1_IMRH_INT_MASK56     (0x01000000)
#define MCF5235_INTC1_IMRH_INT_MASK57     (0x02000000)
#define MCF5235_INTC1_IMRH_INT_MASK58     (0x04000000)
#define MCF5235_INTC1_IMRH_INT_MASK59     (0x08000000)
#define MCF5235_INTC1_IMRH_INT_MASK60     (0x10000000)
#define MCF5235_INTC1_IMRH_INT_MASK61     (0x20000000)
#define MCF5235_INTC1_IMRH_INT_MASK62     (0x40000000)
#define MCF5235_INTC1_IMRH_INT_MASK63     (0x80000000)
#define MCF5235_INTC1_IMRL_MASKALL        (0x00000001)
#define MCF5235_INTC1_IMRL_INT_MASK1      (0x00000002)
#define MCF5235_INTC1_IMRL_INT_MASK2      (0x00000004)
#define MCF5235_INTC1_IMRL_INT_MASK3      (0x00000008)
#define MCF5235_INTC1_IMRL_INT_MASK4      (0x00000010)
#define MCF5235_INTC1_IMRL_INT_MASK5      (0x00000020)
#define MCF5235_INTC1_IMRL_INT_MASK6      (0x00000040)
#define MCF5235_INTC1_IMRL_INT_MASK7      (0x00000080)
#define MCF5235_INTC1_IMRL_INT_MASK8      (0x00000100)
#define MCF5235_INTC1_IMRL_INT_MASK9      (0x00000200)
#define MCF5235_INTC1_IMRL_INT_MASK10     (0x00000400)
#define MCF5235_INTC1_IMRL_INT_MASK11     (0x00000800)
#define MCF5235_INTC1_IMRL_INT_MASK12     (0x00001000)
#define MCF5235_INTC1_IMRL_INT_MASK13     (0x00002000)
#define MCF5235_INTC1_IMRL_INT_MASK14     (0x00004000)
#define MCF5235_INTC1_IMRL_INT_MASK15     (0x00008000)
#define MCF5235_INTC1_IMRL_INT_MASK16     (0x00010000)
#define MCF5235_INTC1_IMRL_INT_MASK17     (0x00020000)
#define MCF5235_INTC1_IMRL_INT_MASK18     (0x00040000)
#define MCF5235_INTC1_IMRL_INT_MASK19     (0x00080000)
#define MCF5235_INTC1_IMRL_INT_MASK20     (0x00100000)
#define MCF5235_INTC1_IMRL_INT_MASK21     (0x00200000)
#define MCF5235_INTC1_IMRL_INT_MASK22     (0x00400000)
#define MCF5235_INTC1_IMRL_INT_MASK23     (0x00800000)
#define MCF5235_INTC1_IMRL_INT_MASK24     (0x01000000)
#define MCF5235_INTC1_IMRL_INT_MASK25     (0x02000000)
#define MCF5235_INTC1_IMRL_INT_MASK26     (0x04000000)
#define MCF5235_INTC1_IMRL_INT_MASK27     (0x08000000)
#define MCF5235_INTC1_IMRL_INT_MASK28     (0x10000000)
#define MCF5235_INTC1_IMRL_INT_MASK29     (0x20000000)
#define MCF5235_INTC1_IMRL_INT_MASK30     (0x40000000)
#define MCF5235_INTC1_IMRL_INT_MASK31     (0x80000000)
#define MCF5235_INTC1_INTFRCH_INTFRC32    (0x00000001)
#define MCF5235_INTC1_INTFRCH_INTFRC33    (0x00000002)
#define MCF5235_INTC1_INTFRCH_INTFRC34    (0x00000004)
#define MCF5235_INTC1_INTFRCH_INTFRC35    (0x00000008)
#define MCF5235_INTC1_INTFRCH_INTFRC36    (0x00000010)
#define MCF5235_INTC1_INTFRCH_INTFRC37    (0x00000020)
#define MCF5235_INTC1_INTFRCH_INTFRC38    (0x00000040)
#define MCF5235_INTC1_INTFRCH_INTFRC39    (0x00000080)
#define MCF5235_INTC1_INTFRCH_INTFRC40    (0x00000100)
#define MCF5235_INTC1_INTFRCH_INTFRC41    (0x00000200)
#define MCF5235_INTC1_INTFRCH_INTFRC42    (0x00000400)
#define MCF5235_INTC1_INTFRCH_INTFRC43    (0x00000800)
#define MCF5235_INTC1_INTFRCH_INTFRC44    (0x00001000)
#define MCF5235_INTC1_INTFRCH_INTFRC45    (0x00002000)
#define MCF5235_INTC1_INTFRCH_INTFRC46    (0x00004000)
#define MCF5235_INTC1_INTFRCH_INTFRC47    (0x00008000)
#define MCF5235_INTC1_INTFRCH_INTFRC48    (0x00010000)
#define MCF5235_INTC1_INTFRCH_INTFRC49    (0x00020000)
#define MCF5235_INTC1_INTFRCH_INTFRC50    (0x00040000)
#define MCF5235_INTC1_INTFRCH_INTFRC51    (0x00080000)
#define MCF5235_INTC1_INTFRCH_INTFRC52    (0x00100000)
#define MCF5235_INTC1_INTFRCH_INTFRC53    (0x00200000)
#define MCF5235_INTC1_INTFRCH_INTFRC54    (0x00400000)
#define MCF5235_INTC1_INTFRCH_INTFRC55    (0x00800000)
#define MCF5235_INTC1_INTFRCH_INTFRC56    (0x01000000)
#define MCF5235_INTC1_INTFRCH_INTFRC57    (0x02000000)
#define MCF5235_INTC1_INTFRCH_INTFRC58    (0x04000000)
#define MCF5235_INTC1_INTFRCH_INTFRC59    (0x08000000)
#define MCF5235_INTC1_INTFRCH_INTFRC60    (0x10000000)
#define MCF5235_INTC1_INTFRCH_INTFRC61    (0x20000000)
#define MCF5235_INTC1_INTFRCH_INTFRC62    (0x40000000)
#define MCF5235_INTC1_INTFRCH_INTFRC63    (0x80000000)
#define MCF5235_INTC1_INTFRCL_INTFRC1     (0x00000002)
#define MCF5235_INTC1_INTFRCL_INTFRC2     (0x00000004)
#define MCF5235_INTC1_INTFRCL_INTFRC3     (0x00000008)
#define MCF5235_INTC1_INTFRCL_INTFRC4     (0x00000010)
#define MCF5235_INTC1_INTFRCL_INTFRC5     (0x00000020)
#define MCF5235_INTC1_INTFRCL_INT6        (0x00000040)
#define MCF5235_INTC1_INTFRCL_INT7        (0x00000080)
#define MCF5235_INTC1_INTFRCL_INT8        (0x00000100)
#define MCF5235_INTC1_INTFRCL_INT9        (0x00000200)
#define MCF5235_INTC1_INTFRCL_INT10       (0x00000400)
#define MCF5235_INTC1_INTFRCL_INTFRC11    (0x00000800)
#define MCF5235_INTC1_INTFRCL_INTFRC12    (0x00001000)
#define MCF5235_INTC1_INTFRCL_INTFRC13    (0x00002000)
#define MCF5235_INTC1_INTFRCL_INTFRC14    (0x00004000)
#define MCF5235_INTC1_INTFRCL_INT15       (0x00008000)
#define MCF5235_INTC1_INTFRCL_INTFRC16    (0x00010000)
#define MCF5235_INTC1_INTFRCL_INTFRC17    (0x00020000)
#define MCF5235_INTC1_INTFRCL_INTFRC18    (0x00040000)
#define MCF5235_INTC1_INTFRCL_INTFRC19    (0x00080000)
#define MCF5235_INTC1_INTFRCL_INTFRC20    (0x00100000)
#define MCF5235_INTC1_INTFRCL_INTFRC21    (0x00200000)
#define MCF5235_INTC1_INTFRCL_INTFRC22    (0x00400000)
#define MCF5235_INTC1_INTFRCL_INTFRC23    (0x00800000)
#define MCF5235_INTC1_INTFRCL_INTFRC24    (0x01000000)
#define MCF5235_INTC1_INTFRCL_INTFRC25    (0x02000000)
#define MCF5235_INTC1_INTFRCL_INTFRC26    (0x04000000)
#define MCF5235_INTC1_INTFRCL_INTFRC27    (0x08000000)
#define MCF5235_INTC1_INTFRCL_INTFRC28    (0x10000000)
#define MCF5235_INTC1_INTFRCL_INTFRC29    (0x20000000)
#define MCF5235_INTC1_INTFRCL_INTFRC30    (0x40000000)
#define MCF5235_INTC1_INTFRCL_INTFRC31    (0x80000000)
#define MCF5235_INTC1_IRLR_IRQ(x)         (((x)&0x7F)<<1)
#define MCF5235_INTC1_IACKLPR_PRI(x)      (((x)&0x0F)<<0)
#define MCF5235_INTC1_IACKLPR_LEVEL(x)    (((x)&0x07)<<4)

/*********************************************************************
*
* Programmable Interrupt Timer Modules (PIT)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_PIT_PCSR0       (*(vuint16*)(void*)(&__IPSBAR[0x150000]))
#define MCF5235_PIT_PMR0        (*(vuint16*)(void*)(&__IPSBAR[0x150002]))
#define MCF5235_PIT_PCNTR0      (*(vuint16*)(void*)(&__IPSBAR[0x150004]))
#define MCF5235_PIT_PCSR1       (*(vuint16*)(void*)(&__IPSBAR[0x160000]))
#define MCF5235_PIT_PMR1        (*(vuint16*)(void*)(&__IPSBAR[0x160002]))
#define MCF5235_PIT_PCNTR1      (*(vuint16*)(void*)(&__IPSBAR[0x160004]))
#define MCF5235_PIT_PCSR2       (*(vuint16*)(void*)(&__IPSBAR[0x170000]))
#define MCF5235_PIT_PMR2        (*(vuint16*)(void*)(&__IPSBAR[0x170002]))
#define MCF5235_PIT_PCNTR2      (*(vuint16*)(void*)(&__IPSBAR[0x170004]))
#define MCF5235_PIT_PCSR3       (*(vuint16*)(void*)(&__IPSBAR[0x180000]))
#define MCF5235_PIT_PMR3        (*(vuint16*)(void*)(&__IPSBAR[0x180002]))
#define MCF5235_PIT_PCNTR3      (*(vuint16*)(void*)(&__IPSBAR[0x180004]))
#define MCF5235_PIT_PCSR(x)     (*(vuint16*)(void*)(&__IPSBAR[0x150000+((x)*0x10000)]))
#define MCF5235_PIT_PMR(x)      (*(vuint16*)(void*)(&__IPSBAR[0x150002+((x)*0x10000)]))
#define MCF5235_PIT_PCNTR(x)    (*(vuint16*)(void*)(&__IPSBAR[0x150004+((x)*0x10000)]))
#define MCF5235_PIT_PCSR_EN        (0x0001)
#define MCF5235_PIT_PCSR_RLD       (0x0002)
#define MCF5235_PIT_PCSR_PIF       (0x0004)
#define MCF5235_PIT_PCSR_PIE       (0x0008)
#define MCF5235_PIT_PCSR_OVW       (0x0010)
#define MCF5235_PIT_PCSR_HALTED    (0x0020)
#define MCF5235_PIT_PCSR_DOZE      (0x0040)
#define MCF5235_PIT_PCSR_PRE(x)    (((x)&0x000F)<<8)
#define MCF5235_PIT_PMR_PM0        (0x0001)
#define MCF5235_PIT_PMR_PM1        (0x0002)
#define MCF5235_PIT_PMR_PM2        (0x0004)
#define MCF5235_PIT_PMR_PM3        (0x0008)
#define MCF5235_PIT_PMR_PM4        (0x0010)
#define MCF5235_PIT_PMR_PM5        (0x0020)
#define MCF5235_PIT_PMR_PM6        (0x0040)
#define MCF5235_PIT_PMR_PM7        (0x0080)
#define MCF5235_PIT_PMR_PM8        (0x0100)
#define MCF5235_PIT_PMR_PM9        (0x0200)
#define MCF5235_PIT_PMR_PM10       (0x0400)
#define MCF5235_PIT_PMR_PM11       (0x0800)
#define MCF5235_PIT_PMR_PM12       (0x1000)
#define MCF5235_PIT_PMR_PM13       (0x2000)
#define MCF5235_PIT_PMR_PM14       (0x4000)
#define MCF5235_PIT_PMR_PM15       (0x8000)
#define MCF5235_PIT_PCNTR_PC0      (0x0001)
#define MCF5235_PIT_PCNTR_PC1      (0x0002)
#define MCF5235_PIT_PCNTR_PC2      (0x0004)
#define MCF5235_PIT_PCNTR_PC3      (0x0008)
#define MCF5235_PIT_PCNTR_PC4      (0x0010)
#define MCF5235_PIT_PCNTR_PC5      (0x0020)
#define MCF5235_PIT_PCNTR_PC6      (0x0040)
#define MCF5235_PIT_PCNTR_PC7      (0x0080)
#define MCF5235_PIT_PCNTR_PC8      (0x0100)
#define MCF5235_PIT_PCNTR_PC9      (0x0200)
#define MCF5235_PIT_PCNTR_PC10     (0x0400)
#define MCF5235_PIT_PCNTR_PC11     (0x0800)
#define MCF5235_PIT_PCNTR_PC12     (0x1000)
#define MCF5235_PIT_PCNTR_PC13     (0x2000)
#define MCF5235_PIT_PCNTR_PC14     (0x4000)
#define MCF5235_PIT_PCNTR_PC15     (0x8000)

/*********************************************************************
*
* Queued Serial Peripheral Interface (QSPI)
*
*********************************************************************/

/* Register read/write macros */
#define MCF5235_QSPI_QMR      (*(vuint16*)(void*)(&__IPSBAR[0x000340]))
#define MCF5235_QSPI_QDLYR    (*(vuint16*)(void*)(&__IPSBAR[0x000344]))
#define MCF5235_QSPI_QWR      (*(vuint16*)(void*)(&__IPSBAR[0x000348]))
#define MCF5235_QSPI_QIR      (*(vuint16*)(void*)(&__IPSBAR[0x00034C]))
#define MCF5235_QSPI_QAR      (*(vuint16*)(void*)(&__IPSBAR[0x000350]))
#define MCF5235_QSPI_QDR      (*(vuint16*)(void*)(&__IPSBAR[0x000354]))

/* Bit definitions and macros for MCF5235_QSPI_QMR */
#define MCF5235_QSPI_QMR_BAUD(x)     (((x)&0x00FF)<<0)
#define MCF5235_QSPI_QMR_CPHA        (0x0100)
#define MCF5235_QSPI_QMR_CPOL        (0x0200)
#define MCF5235_QSPI_QMR_BITS(x)     (((x)&0x000F)<<10)
#define MCF5235_QSPI_QMR_DOHIE       (0x4000)
#define MCF5235_QSPI_QMR_MSTR        (0x8000)
#define MCF5235_QSPI_QDLYR_DTL(x)    (((x)&0x00FF)<<0)
#define MCF5235_QSPI_QDLYR_QCD(x)    (((x)&0x007F)<<8)
#define MCF5235_QSPI_QDLYR_SPE       (0x8000)
#define MCF5235_QSPI_QWR_NEWQP(x)    (((x)&0x000F)<<0)
#define MCF5235_QSPI_QWR_ENDQP(x)    (((x)&0x000F)<<8)
#define MCF5235_QSPI_QWR_CSIV        (0x1000)
#define MCF5235_QSPI_QWR_WRTO        (0x2000)
#define MCF5235_QSPI_QWR_WREN        (0x4000)
#define MCF5235_QSPI_QWR_HALT        (0x8000)
#define MCF5235_QSPI_QIR_SPIF        (0x0001)
#define MCF5235_QSPI_QIR_ABRT        (0x0004)
#define MCF5235_QSPI_QIR_WCEF        (0x0008)
#define MCF5235_QSPI_QIR_SPIFE       (0x0100)
#define MCF5235_QSPI_QIR_ABRTE       (0x0400)
#define MCF5235_QSPI_QIR_WCEFE       (0x0800)
#define MCF5235_QSPI_QIR_ABRTL       (0x1000)
#define MCF5235_QSPI_QIR_ABRTB       (0x4000)
#define MCF5235_QSPI_QIR_WCEFB       (0x8000)
#define MCF5235_QSPI_QAR_ADDR(x)     (((x)&0x003F)<<0)

/********************************************************************/


#endif	/* _CPU_MCF5235_H */
