/* Driver Manager Interface Implementation.
 *
 *  COPYRIGHT (c) 2009.
 *  Aeroflex Gaisler AB
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <drvmgr/drvmgr.h>
#include <drvmgr/drvmgr_confdefs.h>

/* Driver Manaer Internal State */
#define DRV_MGR_STATE_NONE		0	/* Not started */
#define DRV_MGR_STATE_INIT		1	/* Initializing */
#define DRV_MGR_STATE_INIT1_DONE	2	/* Init Stage 1 Done, Doing stage 2 */
#define DRV_MGR_STATE_INIT2_DONE	3	/* Init Stage 2 Done */

/* Enable debugging */
/*#define DEBUG 1*/

#ifdef DEBUG
#define DBG(x...) printk(x)
#else
#define DBG(x...) 
#endif

struct rtems_driver_manager drv_mgr =
{
	/* state */			DRV_MGR_STATE_NONE,

	/* drivers */			LIST_INITIALIZER(struct rtems_drvmgr_drv_info, next),
	/* buses */			LIST_INITIALIZER(struct rtems_drvmgr_bus_info, next),

	/* devices */			LIST_INITIALIZER(struct rtems_drvmgr_dev_info, next),
	/* devices_registered */	LIST_INITIALIZER(struct rtems_drvmgr_dev_info, next),
	/* devices_inactive */		LIST_INITIALIZER(struct rtems_drvmgr_dev_info, next),

	/* root_drv */			NULL,
	/* root_device */		NULL
};

static int do_init1(struct rtems_driver_manager *mgr, struct rtems_drvmgr_dev_info *dev);
static int do_init2(struct rtems_driver_manager *mgr, struct rtems_drvmgr_dev_info *dev);

/* DRIVER MANAGER */

/* 1. Init driver manager
 * 2. Register Drivers
 * 3. Register "master bus" device
 * 4. Do init1 (Iterate until all devices have been initialized)
 * 5. Do init2
 * 6. Do Init3
 */
int rtems_drvmgr_init(void)
{
	struct rtems_driver_manager *mgr = &drv_mgr;
	struct rtems_drvmgr_drv_reg_func *drvreg;
	struct rtems_drvmgr_dev_info *root, *dev, *tmp;

	/* 1. */

	/* Fail if no root bus driver has been registered */
	if ( (mgr->root_drv == NULL) || (mgr->state != DRV_MGR_STATE_NONE) ) {
		return -1;
	}
	mgr->state = DRV_MGR_STATE_INIT;

	/* 2. */
	drvreg = &rtems_drvmgr_drivers[0];
	while ( drvreg->drv_reg ) {
		/* Make driver register */
		drvreg->drv_reg();
		drvreg++;
	}

	/* Insert Root bus driver */
	rtems_drvmgr_drv_register(mgr->root_drv);

	/* 3. */
	rtems_drvmgr_alloc_dev(&root);
	mgr->root_device = root;

	root->next = NULL;
	root->next_in_bus = NULL;
	root->parent = NULL;
	root->minor_drv = 0;
	root->minor_bus = 0;
	root->businfo = NULL;
	root->priv = NULL;
	root->name = "root bus";
	/* CUstom Driver association */
	root->drv = mgr->root_drv;

	/* This registers a bus */
	rtems_drvmgr_dev_register(root);

	/* 4. Init1 */

	/* The root bus device is always first in list at this point.
	 *
	 * Loop until list is empty. Until all found devices have reached 
	 * initialization level 1.
	 * New devices may be inserted into the list during looping, init1()
	 * functions may call rtems_drmgr_dev_register().
	 *
	 * Depending on how dev_register() inserts into the list is will be
	 * depth first or breth first initialization.
	 */
	while ( DEV_LIST_HEAD(&mgr->devices_registered) ) {

		/* Always process first in list */
		dev = DEV_LIST_HEAD(&mgr->devices_registered);

		/* Remove first in the list (will be inserted in appropriate list by do_init1()) */
		rtems_drvmgr_list_remove_head(&mgr->devices_registered);

		/* Initialize level 1 (init1) 
		 * This will put the device into the 
		 *  - device list if successful, or,
		 *  - inactive devices list if unsucessful (init 2 is skipped).
		 */
		do_init1(mgr, dev);
	}
	mgr->state = DRV_MGR_STATE_INIT1_DONE;

	/* 5. Init2 */
	dev = DEV_LIST_HEAD(&mgr->devices);
	while ( dev ) {

		/* Remember the next device since the device may be moved to another list */
		tmp = dev->next;

		/* Initialize level 2 (init2) */
		do_init2(mgr, dev);

		dev = tmp;
	}
	mgr->state = DRV_MGR_STATE_INIT2_DONE;

	return 0;
}

/* Take device to initialization level 1 */
static int do_init1(struct rtems_driver_manager *mgr, struct rtems_drvmgr_dev_info *dev)
{
	if ( dev->drv && dev->drv->ops && dev->drv->ops->init1 ) {
		/* Note: This init1 function may register new devices */
		if ( (dev->error=dev->drv->ops->init1(dev)) != RTEMS_SUCCESSFUL ) {
			/* An error of some kind has occured in the driver/device, the
			 * failed device it put into a separate list, this way Init2 will
			 * not be called for this device.
			 *
			 * The device is added to the failed device list instead of beeing 
			 * freed, this is for system overview and debugging.
			 */
			rtems_drvmgr_list_add_head(&mgr->devices_inactive, dev);
			dev->state |= (DEV_STATE_INIT1_DONE|DEV_STATE_INIT1_FAILED);

			DBG("do_init1: DRV: %s (DEV: %s) failed\n", dev->drv->name, dev->name);
			return 1;	/* Failed to take into  */
		}
	}
	dev->state |= DEV_STATE_INIT1_DONE;

	/* Put at end of device list so that init2() calls comes in the same order as init1() */
	rtems_drvmgr_list_add_tail(&mgr->devices, dev);

	return 0;
}

/* Take device to initialization level 2 */
static int do_init2(struct rtems_driver_manager *mgr, struct rtems_drvmgr_dev_info *dev)
{
	if ( dev->drv && dev->drv->ops && dev->drv->ops->init2 ) {
		if ( (dev->error=dev->drv->ops->init2(dev)) != RTEMS_SUCCESSFUL ) {
			/* Init2 failed */
			dev->state |= (DEV_STATE_INIT2_FAILED|DEV_STATE_INIT2_DONE);
			/* Move to Failed list */
			rtems_drvmgr_list_remove(&mgr->devices, dev);
			rtems_drvmgr_list_add_head(&mgr->devices_inactive, dev);

			DBG("do_init2: DRV: %s (DEV: %s) failed\n", dev->drv->name, dev->name);
			return -1;
		}
		dev->state |= DEV_STATE_INIT2_DONE;
	}
	return 0;
}

/* Register Root device driver */
int rtems_drvmgr_root_drv_register(struct rtems_drvmgr_drv_info *drv)
{
	struct rtems_driver_manager *mgr = &drv_mgr;

	if ( mgr->root_drv ) {
		/* Only possible to register root device once */
		return RTEMS_TOO_MANY;
	}

	/* Set root device driver */
	drv->next = NULL;
	mgr->root_drv = drv;
	return RTEMS_SUCCESSFUL;
}

/* Register a driver */
int rtems_drvmgr_drv_register(struct rtems_drvmgr_drv_info *drv)
{
	struct rtems_driver_manager *mgr = &drv_mgr;

	/* Put driver into list of registered drivers */
	rtems_drvmgr_list_add_head(&mgr->drivers, drv);

	return 0;
}

/* Get a unique number for a hardware device. Devices of the same type and on the same bus
 * may not have the same number, however device of different type may. This number can be
 * used later to separate devices on a particular bus.
 *
 * This function simply returns the number of devices of the same type currently 
 * registered. If no devices of the same type has been registered 0 is returned...
 * 
 */
static int find_bus_minor(
	struct rtems_drvmgr_dev_info *dev,
	struct rtems_drvmgr_dev_info *busdev,
	void *businfo)
{
	int cnt = 0;
	int (*func)(void *a, void *b);

	if ( !busdev || !busdev->parent || !busdev->parent->ops || !busdev->parent->ops->dev_id_compare )
		return 0;
	/* Get compare function from Bus driver that the device belongs to */
	func = busdev->parent->ops->dev_id_compare;

	while ( busdev ) {
		/* Compare */
		if ( (dev != busdev) && (func(businfo, busdev->businfo) == 0) ) {
			cnt++;
		}
		busdev = busdev->next_in_bus;
	}
	return cnt;
}

/* Find next minor number for a driver, when the devices are ordered by their
 * minor number (sorted linked list of devices) the minor number is found by
 * looking for a gap or at the end.
 * 
 */
void rtems_drvmgr_insert_dev_into_drv(
	struct rtems_drvmgr_drv_info *drv,
	struct rtems_drvmgr_dev_info *dev)
{
	struct rtems_drvmgr_dev_info *curr, *prev;
	int minor;

	minor = 0;
	prev = NULL;
	curr = drv->dev;

	while (curr) {
		if ( minor < curr->minor_drv ) {
			/* Found a gap. Insert new device between prev and curr. */
			break;
		}
		minor++;
		prev = curr;
		curr = curr->next_in_drv;
	}

	if ( prev == NULL ) {
		/* First in list */
		dev->next_in_drv = curr;
		drv->dev = dev;
	} else {
		/* End of list or in middle */
		dev->next_in_drv = prev->next_in_drv;
		prev->next_in_drv = dev;
	}

	/* Set minor */
	dev->minor_drv = minor;
}

/* Register a device */
int rtems_drvmgr_dev_register(struct rtems_drvmgr_dev_info *dev)
{
	struct rtems_driver_manager *mgr = &drv_mgr;
	struct rtems_drvmgr_drv_info *drv;
	struct rtems_drvmgr_bus_info *busdev = dev->parent;
	struct rtems_drvmgr_key *keys;

	/* Init dev */
	if ( busdev ) {
		/* Only root device has no parent. */
		dev->minor_bus = find_bus_minor(dev, busdev->children, dev->businfo);
		busdev->dev_cnt++;
	}

	/* Custom driver assocation? */
	if ( dev->drv ) {
		drv = dev->drv;
		DBG("CUSTOM ASSOCIATION (%s to %s)\n", dev->name, drv->name);
		goto united;
	}

	/* Try to find a driver that can handle this device */
	drv = DRV_LIST_HEAD(&mgr->drivers);
	while ( drv ) {
		if ( busdev->ops->unite(drv, dev) == 1) {
united:
			/* United device with driver 
			 * Put the device on the registered device list 
			 */
			dev->state |= DEV_STATE_UNITED;
			dev->drv = drv;

			/* Check if user want to skip this core. This is not a normal request,
			 * however in a multi-processor system the two RTEMS instances must
			 * not use the same devices in a system, not reporting a device to
			 * it's driver will effectively accomplish this. In a non Plug & Play
			 * system one can easily avoid this problem by not report the core, but
			 * in a Plug & Play system the bus driver will report all found cores.
			 *
			 * To stop the two RTEMS instances from using the same device the user
			 * can simply device a resource entry for a certain device but set the
			 * keys field to NULL.
			 */
			if ( rtems_drvmgr_keys_get(dev, &keys) == 0 ) {
				if ( keys == NULL ) {
					/* Found Driver resource entry point for this device, 
					 * it was NULL, this indicates to skip the core.
					 * We put it into the inactive list marking it as 
					 * ignored.
					 */
					 dev->state |= DEV_STATE_IGNORED;
					 break;
				}
			}

			/* Assign Driver Minor number and put into driver's device list */
			rtems_drvmgr_insert_dev_into_drv(drv, dev);
			drv->dev_cnt++;

			if ( mgr->state >= DRV_MGR_STATE_INIT1_DONE ) {
				/* Hot Plug device, all other devices have already be intialized to at 
				 * least level 1. This device is taken into level 1 directly.
				 */
				if ( (do_init1(mgr, dev) == 0) && (mgr->state >= DRV_MGR_STATE_INIT2_DONE) ) {
					/* Hot Plug device, all other devices have already been intialized to at 
					 * least level 2, and this device is currently at level 1.
					 * This device is taken into level 2 directly.
					 */
					do_init2(mgr, dev);
				}
				DBG("Registered Hot-Plug %s (DRV: %s) on %s\n",
					dev->name, drv->name, dev->parent ? dev->parent->dev->name: "NO PARENT" );
			} else {
#ifdef DRV_MGR_BREADTH_FIRST
				/* At the end of the list (breadth first search) */
				rtems_drvmgr_list_add_tail(&mgr->devices_registered, dev);
#else
				/* First in list (depth first search) */
				rtems_drvmgr_list_add_head(&mgr->devices_registered, dev);
#endif
				DBG("Registered %s (DRV: %s) on %s\n",
					dev->name, drv->name, dev->parent ? dev->parent->dev->name: "NO PARENT" );
			}
			return 0;
		}
		drv = drv->next;
	}

	/* No driver found that can handle this device, put into inactive list */
	dev->minor_drv = -1;
	rtems_drvmgr_list_add_head(&mgr->devices_inactive, dev);

	return 0;
}

/* Unregister device, 
 *  - let assigned driver handle deletion
 *  - remove from device list 
 *  - remove from driver list
 *  - remove from bus list
 *  - add to deleted list for debugging
 */
int rtems_drvmgr_dev_unregister(struct rtems_drvmgr_dev_info *dev, int delete)
{
	struct rtems_drvmgr_dev_info *subdev, **pprev;
	struct rtems_driver_manager *mgr = &drv_mgr;
	struct rtems_drvmgr_list *list;
	int err, removal_iterations;

	if ( dev->state & DEV_STATE_DELETED )
		return -1; /* Already deleted */

	/* Remove children if this device exports a bus of devices. All children must be removed first
	 * as they depend upon the bus services this device provide.
	 *
	 * This procedure is done twice if a driver depends on another driver. The removal will fail
	 * the first time hopefully not the second time since the other driver (the one requiring 
	 * service from the first) has been removed. This will not work for drivers on a remote
	 * bus however.
	 */
	if ( dev->bus ) {
		removal_iterations = 2;
remove_all_children:
		err = 0;
		subdev = dev->bus->children;
		while ( subdev ) {
			if ( rtems_drvmgr_dev_unregister(subdev, delete) ) {
				/* An error occured */
				err++;
			}
			subdev = subdev->next_in_bus;
		}
		if ( err > 0 && (--removal_iterations > 0) ) {
			/* Try to remove devices once more */
			goto remove_all_children;
		} else if ( err > 0 ) {
			/* Failed to remove all children on the bus. */
			return -1;
		}
	}

	/* Remove device by letting assigned driver take care of hardware issues */
	list = NULL;
	if ( dev->state & DEV_STATE_UNITED ) {
		/* Driver assigned to device */
		if ( dev->state & DEV_STATE_INIT1_DONE ) {
			/* Check if already removed */
			if ( !(dev->state & (DEV_STATE_INIT1_FAILED|DEV_STATE_INIT2_FAILED)) ){
				/* sucessfully installed device, now we remove it */
				if ( !dev->drv->ops->delete ) {
					/* No delete function must be considered severe */
					return -1;
				}
				if ( (dev->error=dev->drv->ops->delete(dev)) != RTEMS_SUCCESSFUL ) {
					/* Failed to delete device */
					return -1;
				}
				list = &mgr->devices;
			} else {
				list = &mgr->devices_inactive;
			}
		} else {
			/* Driver assigned but still havn't been moved from registered list by init1 stage */
			list = &mgr->devices_registered;
		}

		/* Delete from driver list */
		pprev = &dev->drv->dev;
		subdev = dev->drv->dev;
		while ( subdev != dev ) {
			pprev = &subdev->next_in_drv;
			subdev = subdev->next_in_drv;
		}
		/* Device must be in list */
		*pprev = subdev->next_in_drv;
		dev->drv->dev_cnt--;
	} else {
		/* The device havn't been assigned to a driver. Do nothing. */
		list = &mgr->devices_inactive;
	}

	dev->state |= DEV_STATE_DELETED;

	if ( (list != &mgr->devices_inactive) || delete ) {
		rtems_drvmgr_list_remove(list, dev);

		/* Insert into inactive list if device is not going to be deleted */
		if ( !delete ) {
			rtems_drvmgr_list_add_head(&mgr->devices_inactive, dev);
		}
	}

	/* Remove device from parent bus list */
	pprev = &dev->parent->children;
	subdev = dev->parent->children;
	while ( subdev != dev ) {
		pprev = &subdev->next_in_bus;
		subdev = subdev->next_in_bus;
	}
	/* Device must be in list */
	*pprev = subdev->next_in_bus;
	dev->parent->dev_cnt--;

	if ( dev->bus ) {
		/* Remove from bus list if this device provides a bus */
		rtems_drvmgr_list_remove(&mgr->buses, dev->bus);
	}

	/* All references to this device has been removed at this point 
	 * if delete is non-zero.
	 */
	if ( delete ) {
		free(dev);
	}

	return 0;
}

/* Register a bus */
int rtems_drvmgr_bus_register(struct rtems_drvmgr_bus_info *bus)
{
	struct rtems_driver_manager *mgr = &drv_mgr;

	/* Put driver into list of found buses */
	rtems_drvmgr_list_add_head(&mgr->buses, bus);

	return 0;
}

/* Allocate memory for a Device structure */
int rtems_drvmgr_alloc_dev(struct rtems_drvmgr_dev_info **pdev)
{
	struct rtems_drvmgr_dev_info *dev;

	dev = (struct rtems_drvmgr_dev_info *)malloc(sizeof(struct rtems_drvmgr_dev_info));
	if ( !dev ) {
		/* Failed to allocate device structure - critical error */
		rtems_fatal_error_occurred(RTEMS_NO_MEMORY);
	}
	*pdev = dev;
	memset(dev, 0, sizeof(struct rtems_drvmgr_dev_info));

	return 0;
}

/* Allocate memory for a Bus structure */
int rtems_drvmgr_alloc_bus(struct rtems_drvmgr_bus_info **pbus)
{
	struct rtems_drvmgr_bus_info *bus;

	bus = (struct rtems_drvmgr_bus_info *)malloc(sizeof(struct rtems_drvmgr_bus_info));
	if ( !bus ) {
		/* Failed to allocate device structure - critical error */
		rtems_fatal_error_occurred(RTEMS_NO_MEMORY);
	}
	*pbus = bus;
	memset(bus, 0, sizeof(struct rtems_drvmgr_bus_info));

	return 0;
}

/* Add driver resources to a bus instance */
int rtems_drvmgr_bus_res_add(struct rtems_drvmgr_bus_info *bus, struct rtems_drvmgr_drv_res *res)
{
	struct rtems_drvmgr_bus_res *node;

	node = malloc(sizeof(struct rtems_drvmgr_bus_res));
	if ( !node )
		return -1;

	node->next = bus->reslist;
	node->resource = res;
	bus->reslist = node;

	return 0;
}

int rtems_drvmgr_for_each_dev(
	struct rtems_drvmgr_list *devlist,
	unsigned int state_set_mask,
	unsigned int state_clr_mask,
	int (*func)(struct rtems_drvmgr_dev_info *dev, void *arg),
	void *arg
	)
{
	struct rtems_drvmgr_dev_info *dev;
	int ret;

	/* Get First Device */
	dev = DEV_LIST_HEAD(devlist);
	while ( dev ) {
		if ( ((state_set_mask != 0) && ((dev->state & state_set_mask) == state_set_mask)) ||
		     ((state_clr_mask != 0) && ((dev->state & state_clr_mask) == 0)) ||
		     ((state_set_mask == 0) && (state_clr_mask == 0)) )
			if ( (ret=func(dev, arg)) != 0 )
				return ret;
		dev = dev->next;
	}
	return 0;
}
