/**
 * @file
 *
 * This include file contains the static inline implementation of the private 
 * inlined routines for POSIX condition variables.
 */

/*
 *  COPYRIGHT (c) 1989-2013.
 *  On-Line Applications Research Corporation (OAR).
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.org/license/LICENSE.
 */

#ifndef _RTEMS_POSIX_CONDIMPL_H
#define _RTEMS_POSIX_CONDIMPL_H
 
#include <rtems/posix/cond.h>
#include <rtems/score/objectimpl.h>
#include <rtems/score/threadqimpl.h>

#include <errno.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 *  Constant to indicate condition variable does not currently have
 *  a mutex assigned to it.
 */
#define POSIX_CONDITION_VARIABLES_NO_MUTEX 0

#define POSIX_CONDITION_VARIABLES_TQ_OPERATIONS &_Thread_queue_Operations_FIFO

#define POSIX_CONDITION_VARIABLE_OF_THREAD_QUEUE_QUEUE( queue ) \
  RTEMS_CONTAINER_OF( \
    queue, POSIX_Condition_variables_Control, Wait_queue.Queue )

/**
 *  The following defines the information control block used to manage
 *  this class of objects.
 */
extern Objects_Information _POSIX_Condition_variables_Information;

/**
 *  The default condition variable attributes structure.
 */
extern const pthread_condattr_t _POSIX_Condition_variables_Default_attributes;

RTEMS_INLINE_ROUTINE void _POSIX_Condition_variables_Initialize(
  POSIX_Condition_variables_Control *the_cond,
  const pthread_condattr_t          *the_attr
)
{
  _Thread_queue_Object_initialize( &the_cond->Wait_queue );
  the_cond->mutex = POSIX_CONDITION_VARIABLES_NO_MUTEX;
  the_cond->clock = the_attr->clock;
}

RTEMS_INLINE_ROUTINE void _POSIX_Condition_variables_Destroy(
  POSIX_Condition_variables_Control *the_cond
)
{
  _Thread_queue_Destroy( &the_cond->Wait_queue );
}

RTEMS_INLINE_ROUTINE void _POSIX_Condition_variables_Acquire_critical(
  POSIX_Condition_variables_Control *the_cond,
  Thread_queue_Context              *queue_context
)
{
  _Thread_queue_Acquire_critical( &the_cond->Wait_queue, queue_context );
}

RTEMS_INLINE_ROUTINE void _POSIX_Condition_variables_Release(
  POSIX_Condition_variables_Control *the_cond,
  Thread_queue_Context              *queue_context
)
{
  _Thread_queue_Release( &the_cond->Wait_queue, queue_context );
}

/**
 *  @brief POSIX Condition Variable Allocate
 *
 *  This function allocates a condition variable control block from
 *  the inactive chain of free condition variable control blocks.
 */
RTEMS_INLINE_ROUTINE POSIX_Condition_variables_Control *
  _POSIX_Condition_variables_Allocate( void )
{
  return (POSIX_Condition_variables_Control *)
    _Objects_Allocate( &_POSIX_Condition_variables_Information );
}

/**
 *  @brief POSIX Condition Variable Free
 *
 *  This routine frees a condition variable control block to the
 *  inactive chain of free condition variable control blocks.
 */
RTEMS_INLINE_ROUTINE void _POSIX_Condition_variables_Free (
  POSIX_Condition_variables_Control *the_condition_variable
)
{
  _Objects_Free(
    &_POSIX_Condition_variables_Information,
    &the_condition_variable->Object
  );
}

POSIX_Condition_variables_Control *_POSIX_Condition_variables_Get(
  pthread_cond_t       *cond,
  Thread_queue_Context *queue_context
);

/**
 * @brief Implements wake up version of the "signal" operation.
 * 
 * A support routine which implements guts of the broadcast and single task
 * wake up version of the "signal" operation.
 */
int _POSIX_Condition_variables_Signal_support(
  pthread_cond_t            *cond,
  bool                       is_broadcast
);

/**
 * @brief POSIX condition variables wait support.
 *
 * A support routine which implements guts of the blocking, non-blocking, and
 * timed wait version of condition variable wait routines.
 */
int _POSIX_Condition_variables_Wait_support(
  pthread_cond_t            *cond,
  pthread_mutex_t           *mutex,
  const struct timespec     *abstime
);

#ifdef __cplusplus
}
#endif

#endif
/*  end of include file */
