/*
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (C) 2018, 2019 embedded brains GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * This file must be compatible to general purpose POSIX system, e.g. Linux,
 * FreeBSD.  It may be used for utility programs.
 */

#ifndef _RTEMS_RECORDDATA_H
#define _RTEMS_RECORDDATA_H

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * @defgroup RTEMSRecord Event Recording
 *
 * @ingroup RTEMSAPITracing
 *
 * @brief Low-level event recording support.
 *
 * @{
 */

/**
 * @brief The record version.
 *
 * The record version reflects the record event definitions.  It is reported by
 * the RTEMS_RECORD_VERSION event.
 */
#define RTEMS_RECORD_THE_VERSION 9

/**
 * @brief The items are in 32-bit little-endian format.
 */
#define RTEMS_RECORD_FORMAT_LE_32 0x11111111

/**
 * @brief The items are in 64-bit little-endian format.
 */
#define RTEMS_RECORD_FORMAT_LE_64 0x22222222

/**
 * @brief The items are in 32-bit big-endian format.
 */
#define RTEMS_RECORD_FORMAT_BE_32 0x33333333

/**
 * @brief The items are in 64-bit big-endian format.
 */
#define RTEMS_RECORD_FORMAT_BE_64 0x44444444

/**
 * @brief Magic number to identify a record item stream.
 *
 * This is a random number.
 */
#define RTEMS_RECORD_MAGIC 0x82e14ec1

/**
 * @brief The record events.
 */
typedef enum {
  /* There are 512 events reserved for the system */
  RTEMS_RECORD_EMPTY,
  RTEMS_RECORD_VERSION,

  /*
   * Keep the following system events in lexicographical order, increment
   * RTEMS_RECORD_THE_VERSION after each change.
   */
  RTEMS_RECORD_ACCEPT_ENTRY,
  RTEMS_RECORD_ACCEPT_EXIT,
  RTEMS_RECORD_ADDRESS,
  RTEMS_RECORD_ALIGNED_ALLOC_ENTRY,
  RTEMS_RECORD_ALIGNED_ALLOC_EXIT,
  RTEMS_RECORD_ARCH,
  RTEMS_RECORD_ARG_0,
  RTEMS_RECORD_ARG_1,
  RTEMS_RECORD_ARG_2,
  RTEMS_RECORD_ARG_3,
  RTEMS_RECORD_ARG_4,
  RTEMS_RECORD_ARG_5,
  RTEMS_RECORD_ARG_6,
  RTEMS_RECORD_ARG_7,
  RTEMS_RECORD_ARG_8,
  RTEMS_RECORD_ARG_9,
  RTEMS_RECORD_BIND_ENTRY,
  RTEMS_RECORD_BIND_EXIT,
  RTEMS_RECORD_BSP,
  RTEMS_RECORD_BUFFER,
  RTEMS_RECORD_CALLER,
  RTEMS_RECORD_CALLOC_ENTRY,
  RTEMS_RECORD_CALLOC_EXIT,
  RTEMS_RECORD_CHOWN_ENTRY,
  RTEMS_RECORD_CHOWN_EXIT,
  RTEMS_RECORD_CLOSE_ENTRY,
  RTEMS_RECORD_CLOSE_EXIT,
  RTEMS_RECORD_CONNECT_ENTRY,
  RTEMS_RECORD_CONNECT_EXIT,
  RTEMS_RECORD_ETHER_INPUT,
  RTEMS_RECORD_ETHER_OUTPUT,
  RTEMS_RECORD_ERRNO,
  RTEMS_RECORD_FATAL_CODE,
  RTEMS_RECORD_FATAL_SOURCE,
  RTEMS_RECORD_FCHMOD_ENTRY,
  RTEMS_RECORD_FCHMOD_EXIT,
  RTEMS_RECORD_FCNTL_ENTRY,
  RTEMS_RECORD_FCNTL_EXIT,
  RTEMS_RECORD_FDATASYNC_ENTRY,
  RTEMS_RECORD_FDATASYNC_EXIT,
  RTEMS_RECORD_FREE_ENTRY,
  RTEMS_RECORD_FREE_EXIT,
  RTEMS_RECORD_FREQUENCY,
  RTEMS_RECORD_FSTAT_ENTRY,
  RTEMS_RECORD_FSTAT_EXIT,
  RTEMS_RECORD_FSYNC_ENTRY,
  RTEMS_RECORD_FSYNC_EXIT,
  RTEMS_RECORD_FTRUNCATE_ENTRY,
  RTEMS_RECORD_FTRUNCATE_EXIT,
  RTEMS_RECORD_FUNCTION_ENTRY,
  RTEMS_RECORD_FUNCTION_EXIT,
  RTEMS_RECORD_GETSOCKOPT_ENTRY,
  RTEMS_RECORD_GETSOCKOPT_EXIT,
  RTEMS_RECORD_HEAP_ALLOC,
  RTEMS_RECORD_HEAP_FREE,
  RTEMS_RECORD_HEAP_SIZE,
  RTEMS_RECORD_HEAP_USAGE,
  RTEMS_RECORD_INTERRUPT_ENTRY,
  RTEMS_RECORD_INTERRUPT_EXIT,
  RTEMS_RECORD_INTERRUPT_INSTALL,
  RTEMS_RECORD_INTERRUPT_REMOVE,
  RTEMS_RECORD_INTERRUPT_SERVER_ENTRY,
  RTEMS_RECORD_INTERRUPT_SERVER_EXIT,
  RTEMS_RECORD_INTERRUPT_SERVER_INSTALL,
  RTEMS_RECORD_INTERRUPT_SERVER_MOVE,
  RTEMS_RECORD_INTERRUPT_SERVER_REMOVE,
  RTEMS_RECORD_INTERRUPT_SERVER_TRIGGER,
  RTEMS_RECORD_IOCTL_ENTRY,
  RTEMS_RECORD_IOCTL_EXIT,
  RTEMS_RECORD_IP6_INPUT,
  RTEMS_RECORD_IP6_OUTPUT,
  RTEMS_RECORD_IP_INPUT,
  RTEMS_RECORD_IP_OUTPUT,
  RTEMS_RECORD_ISR_DISABLE,
  RTEMS_RECORD_ISR_ENABLE,
  RTEMS_RECORD_ISR_LOCK_ACQUIRE_ENTRY,
  RTEMS_RECORD_ISR_LOCK_ACQUIRE_EXIT,
  RTEMS_RECORD_ISR_LOCK_ADDRESS,
  RTEMS_RECORD_ISR_LOCK_DESTROY,
  RTEMS_RECORD_ISR_LOCK_INITIALIZE,
  RTEMS_RECORD_ISR_LOCK_NAME,
  RTEMS_RECORD_ISR_LOCK_RELEASE,
  RTEMS_RECORD_KEVENT_ENTRY,
  RTEMS_RECORD_KEVENT_EXIT,
  RTEMS_RECORD_KQUEUE_ENTRY,
  RTEMS_RECORD_KQUEUE_EXIT,
  RTEMS_RECORD_LENGTH,
  RTEMS_RECORD_LINE,
  RTEMS_RECORD_LINK_ENTRY,
  RTEMS_RECORD_LINK_EXIT,
  RTEMS_RECORD_LISTEN_ENTRY,
  RTEMS_RECORD_LISTEN_EXIT,
  RTEMS_RECORD_LSEEK_ENTRY,
  RTEMS_RECORD_LSEEK_EXIT,
  RTEMS_RECORD_MALLOC_ENTRY,
  RTEMS_RECORD_MALLOC_EXIT,
  RTEMS_RECORD_MEMORY,
  RTEMS_RECORD_MKNOD_ENTRY,
  RTEMS_RECORD_MKNOD_EXIT,
  RTEMS_RECORD_MMAP_ENTRY,
  RTEMS_RECORD_MMAP_EXIT,
  RTEMS_RECORD_MOUNT_ENTRY,
  RTEMS_RECORD_MOUNT_EXIT,
  RTEMS_RECORD_MULTILIB,
  RTEMS_RECORD_OPEN_ENTRY,
  RTEMS_RECORD_OPEN_EXIT,
  RTEMS_RECORD_PAGE_ALLOC,
  RTEMS_RECORD_PAGE_FREE,
  RTEMS_RECORD_PER_CPU_COUNT,
  RTEMS_RECORD_PER_CPU_HEAD,
  RTEMS_RECORD_PER_CPU_OVERFLOW,
  RTEMS_RECORD_PER_CPU_TAIL,
  RTEMS_RECORD_POLL_ENTRY,
  RTEMS_RECORD_POLL_EXIT,
  RTEMS_RECORD_POSIX_MEMALIGN_ENTRY,
  RTEMS_RECORD_POSIX_MEMALIGN_EXIT,
  RTEMS_RECORD_PROCESSOR,
  RTEMS_RECORD_PROCESSOR_MAXIMUM,
  RTEMS_RECORD_READ_ENTRY,
  RTEMS_RECORD_READ_EXIT,
  RTEMS_RECORD_READLINK_ENTRY,
  RTEMS_RECORD_READLINK_EXIT,
  RTEMS_RECORD_READV_ENTRY,
  RTEMS_RECORD_READV_EXIT,
  RTEMS_RECORD_REALLOC_ENTRY,
  RTEMS_RECORD_REALLOC_EXIT,
  RTEMS_RECORD_RECV_ENTRY,
  RTEMS_RECORD_RECV_EXIT,
  RTEMS_RECORD_RECVFROM_ENTRY,
  RTEMS_RECORD_RECVFROM_EXIT,
  RTEMS_RECORD_RECVMSG_ENTRY,
  RTEMS_RECORD_RECVMSG_EXIT,
  RTEMS_RECORD_REGISTERS,
  RTEMS_RECORD_RENAME_ENTRY,
  RTEMS_RECORD_RENAME_EXIT,
  RTEMS_RECORD_RETURN_0,
  RTEMS_RECORD_RETURN_1,
  RTEMS_RECORD_RETURN_2,
  RTEMS_RECORD_RETURN_3,
  RTEMS_RECORD_RETURN_4,
  RTEMS_RECORD_RETURN_5,
  RTEMS_RECORD_RETURN_6,
  RTEMS_RECORD_RETURN_7,
  RTEMS_RECORD_RETURN_8,
  RTEMS_RECORD_RETURN_9,
  RTEMS_RECORD_RTEMS_BARRIER_CREATE,
  RTEMS_RECORD_RTEMS_BARRIER_DELETE,
  RTEMS_RECORD_RTEMS_BARRIER_RELEASE,
  RTEMS_RECORD_RTEMS_BARRIER_WAIT,
  RTEMS_RECORD_RTEMS_CALLOC_ENTRY,
  RTEMS_RECORD_RTEMS_CALLOC_EXIT,
  RTEMS_RECORD_RTEMS_EVENT_RECEIVE,
  RTEMS_RECORD_RTEMS_EVENT_SEND,
  RTEMS_RECORD_RTEMS_EVENT_SYSTEM_RECEIVE,
  RTEMS_RECORD_RTEMS_EVENT_SYSTEM_SEND,
  RTEMS_RECORD_RTEMS_MALLOC_ENTRY,
  RTEMS_RECORD_RTEMS_MALLOC_EXIT,
  RTEMS_RECORD_RTEMS_MESSAGE_QUEUE_BROADCAST,
  RTEMS_RECORD_RTEMS_MESSAGE_QUEUE_CREATE,
  RTEMS_RECORD_RTEMS_MESSAGE_QUEUE_DELETE,
  RTEMS_RECORD_RTEMS_MESSAGE_QUEUE_FLUSH,
  RTEMS_RECORD_RTEMS_MESSAGE_QUEUE_RECEIVE,
  RTEMS_RECORD_RTEMS_MESSAGE_QUEUE_SEND,
  RTEMS_RECORD_RTEMS_MESSAGE_QUEUE_URGENT,
  RTEMS_RECORD_RTEMS_PARTITION_CREATE,
  RTEMS_RECORD_RTEMS_PARTITION_DELETE,
  RTEMS_RECORD_RTEMS_PARTITION_GET_BUFFER,
  RTEMS_RECORD_RTEMS_PARTITION_RETURN_BUFFER,
  RTEMS_RECORD_RTEMS_RATE_MONOTONIC_CANCEL,
  RTEMS_RECORD_RTEMS_RATE_MONOTONIC_CREATE,
  RTEMS_RECORD_RTEMS_RATE_MONOTONIC_DELETE,
  RTEMS_RECORD_RTEMS_RATE_MONOTONIC_PERIOD,
  RTEMS_RECORD_RTEMS_SEMAPHORE_CREATE,
  RTEMS_RECORD_RTEMS_SEMAPHORE_DELETE,
  RTEMS_RECORD_RTEMS_SEMAPHORE_FLUSH,
  RTEMS_RECORD_RTEMS_SEMAPHORE_OBTAIN,
  RTEMS_RECORD_RTEMS_SEMAPHORE_RELEASE,
  RTEMS_RECORD_RTEMS_TIMER_CANCEL,
  RTEMS_RECORD_RTEMS_TIMER_CREATE,
  RTEMS_RECORD_RTEMS_TIMER_DELETE,
  RTEMS_RECORD_RTEMS_TIMER_FIRE_AFTER,
  RTEMS_RECORD_RTEMS_TIMER_FIRE_WHEN,
  RTEMS_RECORD_RTEMS_TIMER_RESET,
  RTEMS_RECORD_RTEMS_TIMER_SERVER_FIRE_AFTER,
  RTEMS_RECORD_RTEMS_TIMER_SERVER_FIRE_WHEN,
  RTEMS_RECORD_SBWAIT_ENTRY,
  RTEMS_RECORD_SBWAIT_EXIT,
  RTEMS_RECORD_SBWAKEUP_ENTRY,
  RTEMS_RECORD_SBWAKEUP_EXIT,
  RTEMS_RECORD_SCHEDULER_ADD_PROCESSOR,
  RTEMS_RECORD_SCHEDULER_ASK_FOR_HELP,
  RTEMS_RECORD_SCHEDULER_BLOCK,
  RTEMS_RECORD_SCHEDULER_CANCEL_JOB,
  RTEMS_RECORD_SCHEDULER_ID,
  RTEMS_RECORD_SCHEDULER_MAP_PRIORITY,
  RTEMS_RECORD_SCHEDULER_NAME,
  RTEMS_RECORD_SCHEDULER_PIN,
  RTEMS_RECORD_SCHEDULER_RECONSIDER_HELP_REQUEST,
  RTEMS_RECORD_SCHEDULER_RELEASE_JOB,
  RTEMS_RECORD_SCHEDULER_REMOVE_PROCESSOR,
  RTEMS_RECORD_SCHEDULER_SCHEDULE,
  RTEMS_RECORD_SCHEDULER_SET_AFFINITY,
  RTEMS_RECORD_SCHEDULER_TICK,
  RTEMS_RECORD_SCHEDULER_UNBLOCK,
  RTEMS_RECORD_SCHEDULER_UNMAP_PRIORITY,
  RTEMS_RECORD_SCHEDULER_UNPIN,
  RTEMS_RECORD_SCHEDULER_UPDATE_PRIORITY,
  RTEMS_RECORD_SCHEDULER_WITHDRAW_NODE,
  RTEMS_RECORD_SCHEDULER_YIELD,
  RTEMS_RECORD_SELECT_ENTRY,
  RTEMS_RECORD_SELECT_EXIT,
  RTEMS_RECORD_SEND_ENTRY,
  RTEMS_RECORD_SEND_EXIT,
  RTEMS_RECORD_SENDMSG_ENTRY,
  RTEMS_RECORD_SENDMSG_EXIT,
  RTEMS_RECORD_SENDTO_ENTRY,
  RTEMS_RECORD_SENDTO_EXIT,
  RTEMS_RECORD_SETSOCKOPT_ENTRY,
  RTEMS_RECORD_SETSOCKOPT_EXIT,
  RTEMS_RECORD_SHUTDOWN_ENTRY,
  RTEMS_RECORD_SHUTDOWN_EXIT,
  RTEMS_RECORD_SOABORT_ENTRY,
  RTEMS_RECORD_SOABORT_EXIT,
  RTEMS_RECORD_SOACCEPT_ENTRY,
  RTEMS_RECORD_SOACCEPT_EXIT,
  RTEMS_RECORD_SOALLOC_ENTRY,
  RTEMS_RECORD_SOALLOC_EXIT,
  RTEMS_RECORD_SOBINDAT_ENTRY,
  RTEMS_RECORD_SOBINDAT_EXIT,
  RTEMS_RECORD_SOBIND_ENTRY,
  RTEMS_RECORD_SOBIND_EXIT,
  RTEMS_RECORD_SOCKET_ENTRY,
  RTEMS_RECORD_SOCKET_EXIT,
  RTEMS_RECORD_SOCLOSE_ENTRY,
  RTEMS_RECORD_SOCLOSE_EXIT,
  RTEMS_RECORD_SOCONNECT2_ENTRY,
  RTEMS_RECORD_SOCONNECT2_EXIT,
  RTEMS_RECORD_SOCONNECTAT_ENTRY,
  RTEMS_RECORD_SOCONNECTAT_EXIT,
  RTEMS_RECORD_SOCREATE_ENTRY,
  RTEMS_RECORD_SOCREATE_EXIT,
  RTEMS_RECORD_SODEALLOC_ENTRY,
  RTEMS_RECORD_SODEALLOC_EXIT,
  RTEMS_RECORD_SODISCONNECT_ENTRY,
  RTEMS_RECORD_SODISCONNECT_EXIT,
  RTEMS_RECORD_SOFREE_ENTRY,
  RTEMS_RECORD_SOFREE_EXIT,
  RTEMS_RECORD_SOLISTEN_ENTRY,
  RTEMS_RECORD_SOLISTEN_EXIT,
  RTEMS_RECORD_SONEWCONN_ENTRY,
  RTEMS_RECORD_SONEWCONN_EXIT,
  RTEMS_RECORD_SORECEIVE_ENTRY,
  RTEMS_RECORD_SORECEIVE_EXIT,
  RTEMS_RECORD_SORFLUSH_ENTRY,
  RTEMS_RECORD_SORFLUSH_EXIT,
  RTEMS_RECORD_SOSEND_ENTRY,
  RTEMS_RECORD_SOSEND_EXIT,
  RTEMS_RECORD_SOSHUTDOWN_ENTRY,
  RTEMS_RECORD_SOSHUTDOWN_EXIT,
  RTEMS_RECORD_STATVFS_ENTRY,
  RTEMS_RECORD_STATVFS_EXIT,
  RTEMS_RECORD_SYMLINK_ENTRY,
  RTEMS_RECORD_SYMLINK_EXIT,
  RTEMS_RECORD_TCP_CLOSE,
  RTEMS_RECORD_TCP_INPUT,
  RTEMS_RECORD_TCP_OUTPUT,
  RTEMS_RECORD_THREAD_BEGIN,
  RTEMS_RECORD_THREAD_CONTINUE_ENTRY,
  RTEMS_RECORD_THREAD_CONTINUE_EXIT,
  RTEMS_RECORD_THREAD_CREATE,
  RTEMS_RECORD_THREAD_DELETE,
  RTEMS_RECORD_THREAD_DISPATCH_DISABLE,
  RTEMS_RECORD_THREAD_DISPATCH_ENABLE,
  RTEMS_RECORD_THREAD_EXIT,
  RTEMS_RECORD_THREAD_EXITTED,
  RTEMS_RECORD_THREAD_ID,
  RTEMS_RECORD_THREAD_NAME,
  RTEMS_RECORD_THREAD_PRIO_CURRENT_HIGH,
  RTEMS_RECORD_THREAD_PRIO_CURRENT_LOW,
  RTEMS_RECORD_THREAD_PRIO_REAL_HIGH,
  RTEMS_RECORD_THREAD_PRIO_REAL_LOW,
  RTEMS_RECORD_THREAD_QUEUE_ADDRESS,
  RTEMS_RECORD_THREAD_QUEUE_DESTROY,
  RTEMS_RECORD_THREAD_QUEUE_ENQUEUE,
  RTEMS_RECORD_THREAD_QUEUE_ENQUEUE_STICKY,
  RTEMS_RECORD_THREAD_QUEUE_EXTRACT,
  RTEMS_RECORD_THREAD_QUEUE_ID,
  RTEMS_RECORD_THREAD_QUEUE_INITIALIZE,
  RTEMS_RECORD_THREAD_QUEUE_NAME,
  RTEMS_RECORD_THREAD_QUEUE_SURRENDER,
  RTEMS_RECORD_THREAD_QUEUE_SURRENDER_STICKY,
  RTEMS_RECORD_THREAD_RESOURCE_OBTAIN,
  RTEMS_RECORD_THREAD_RESOURCE_RELEASE,
  RTEMS_RECORD_THREAD_RESTART,
  RTEMS_RECORD_THREAD_STACK_CURRENT,
  RTEMS_RECORD_THREAD_STACK_SIZE,
  RTEMS_RECORD_THREAD_STACK_USAGE,
  RTEMS_RECORD_THREAD_START,
  RTEMS_RECORD_THREAD_STATE_CLEAR,
  RTEMS_RECORD_THREAD_STATE_SET,
  RTEMS_RECORD_THREAD_SWITCH_IN,
  RTEMS_RECORD_THREAD_SWITCH_OUT,
  RTEMS_RECORD_THREAD_TERMINATE,
  RTEMS_RECORD_THREAD_TIMER_INSERT_MONOTONIC,
  RTEMS_RECORD_THREAD_TIMER_INSERT_REALTIME,
  RTEMS_RECORD_THREAD_TIMER_INSERT_TICKS,
  RTEMS_RECORD_THREAD_TIMER_REMOVE,
  RTEMS_RECORD_TOOLS,
  RTEMS_RECORD_UDP_INPUT,
  RTEMS_RECORD_UDP_OUTPUT,
  RTEMS_RECORD_UMA_ALLOC_PTR,
  RTEMS_RECORD_UMA_ALLOC_ZONE,
  RTEMS_RECORD_UMA_FREE_PTR,
  RTEMS_RECORD_UMA_FREE_ZONE,
  RTEMS_RECORD_UNLINK_ENTRY,
  RTEMS_RECORD_UNLINK_EXIT,
  RTEMS_RECORD_UNMOUNT_ENTRY,
  RTEMS_RECORD_UNMOUNT_EXIT,
  RTEMS_RECORD_UPTIME_HIGH,
  RTEMS_RECORD_UPTIME_LOW,
  RTEMS_RECORD_VERSION_CONTROL_KEY,
  RTEMS_RECORD_WATCHDOG_ADDRESS,
  RTEMS_RECORD_WATCHDOG_CPU,
  RTEMS_RECORD_WATCHDOG_INITIALIZE,
  RTEMS_RECORD_WATCHDOG_INSERT,
  RTEMS_RECORD_WATCHDOG_PREINITIALIZE,
  RTEMS_RECORD_WATCHDOG_REMOVE,
  RTEMS_RECORD_WATCHDOG_ROUTINE,
  RTEMS_RECORD_WATCHDOG_STATE,
  RTEMS_RECORD_WORKSPACE_ALLOC_ENTRY,
  RTEMS_RECORD_WORKSPACE_ALLOC_EXIT,
  RTEMS_RECORD_WORKSPACE_FREE_ENTY,
  RTEMS_RECORD_WORKSPACE_FREE_EXIT,
  RTEMS_RECORD_WORKSPACE_SIZE,
  RTEMS_RECORD_WORKSPACE_USAGE,
  RTEMS_RECORD_WRITE_ENTRY,
  RTEMS_RECORD_WRITE_EXIT,
  RTEMS_RECORD_WRITEV_ENTRY,
  RTEMS_RECORD_WRITEV_EXIT,

  /* Unused system events */
  RTEMS_RECORD_SYSTEM_341,
  RTEMS_RECORD_SYSTEM_342,
  RTEMS_RECORD_SYSTEM_343,
  RTEMS_RECORD_SYSTEM_344,
  RTEMS_RECORD_SYSTEM_345,
  RTEMS_RECORD_SYSTEM_346,
  RTEMS_RECORD_SYSTEM_347,
  RTEMS_RECORD_SYSTEM_348,
  RTEMS_RECORD_SYSTEM_349,
  RTEMS_RECORD_SYSTEM_350,
  RTEMS_RECORD_SYSTEM_351,
  RTEMS_RECORD_SYSTEM_352,
  RTEMS_RECORD_SYSTEM_353,
  RTEMS_RECORD_SYSTEM_354,
  RTEMS_RECORD_SYSTEM_355,
  RTEMS_RECORD_SYSTEM_356,
  RTEMS_RECORD_SYSTEM_357,
  RTEMS_RECORD_SYSTEM_358,
  RTEMS_RECORD_SYSTEM_359,
  RTEMS_RECORD_SYSTEM_360,
  RTEMS_RECORD_SYSTEM_361,
  RTEMS_RECORD_SYSTEM_362,
  RTEMS_RECORD_SYSTEM_363,
  RTEMS_RECORD_SYSTEM_364,
  RTEMS_RECORD_SYSTEM_365,
  RTEMS_RECORD_SYSTEM_366,
  RTEMS_RECORD_SYSTEM_367,
  RTEMS_RECORD_SYSTEM_368,
  RTEMS_RECORD_SYSTEM_369,
  RTEMS_RECORD_SYSTEM_370,
  RTEMS_RECORD_SYSTEM_371,
  RTEMS_RECORD_SYSTEM_372,
  RTEMS_RECORD_SYSTEM_373,
  RTEMS_RECORD_SYSTEM_374,
  RTEMS_RECORD_SYSTEM_375,
  RTEMS_RECORD_SYSTEM_376,
  RTEMS_RECORD_SYSTEM_377,
  RTEMS_RECORD_SYSTEM_378,
  RTEMS_RECORD_SYSTEM_379,
  RTEMS_RECORD_SYSTEM_380,
  RTEMS_RECORD_SYSTEM_381,
  RTEMS_RECORD_SYSTEM_382,
  RTEMS_RECORD_SYSTEM_383,
  RTEMS_RECORD_SYSTEM_384,
  RTEMS_RECORD_SYSTEM_385,
  RTEMS_RECORD_SYSTEM_386,
  RTEMS_RECORD_SYSTEM_387,
  RTEMS_RECORD_SYSTEM_388,
  RTEMS_RECORD_SYSTEM_389,
  RTEMS_RECORD_SYSTEM_390,
  RTEMS_RECORD_SYSTEM_391,
  RTEMS_RECORD_SYSTEM_392,
  RTEMS_RECORD_SYSTEM_393,
  RTEMS_RECORD_SYSTEM_394,
  RTEMS_RECORD_SYSTEM_395,
  RTEMS_RECORD_SYSTEM_396,
  RTEMS_RECORD_SYSTEM_397,
  RTEMS_RECORD_SYSTEM_398,
  RTEMS_RECORD_SYSTEM_399,
  RTEMS_RECORD_SYSTEM_400,
  RTEMS_RECORD_SYSTEM_401,
  RTEMS_RECORD_SYSTEM_402,
  RTEMS_RECORD_SYSTEM_403,
  RTEMS_RECORD_SYSTEM_404,
  RTEMS_RECORD_SYSTEM_405,
  RTEMS_RECORD_SYSTEM_406,
  RTEMS_RECORD_SYSTEM_407,
  RTEMS_RECORD_SYSTEM_408,
  RTEMS_RECORD_SYSTEM_409,
  RTEMS_RECORD_SYSTEM_410,
  RTEMS_RECORD_SYSTEM_411,
  RTEMS_RECORD_SYSTEM_412,
  RTEMS_RECORD_SYSTEM_413,
  RTEMS_RECORD_SYSTEM_414,
  RTEMS_RECORD_SYSTEM_415,
  RTEMS_RECORD_SYSTEM_416,
  RTEMS_RECORD_SYSTEM_417,
  RTEMS_RECORD_SYSTEM_418,
  RTEMS_RECORD_SYSTEM_419,
  RTEMS_RECORD_SYSTEM_420,
  RTEMS_RECORD_SYSTEM_421,
  RTEMS_RECORD_SYSTEM_422,
  RTEMS_RECORD_SYSTEM_423,
  RTEMS_RECORD_SYSTEM_424,
  RTEMS_RECORD_SYSTEM_425,
  RTEMS_RECORD_SYSTEM_426,
  RTEMS_RECORD_SYSTEM_427,
  RTEMS_RECORD_SYSTEM_428,
  RTEMS_RECORD_SYSTEM_429,
  RTEMS_RECORD_SYSTEM_430,
  RTEMS_RECORD_SYSTEM_431,
  RTEMS_RECORD_SYSTEM_432,
  RTEMS_RECORD_SYSTEM_433,
  RTEMS_RECORD_SYSTEM_434,
  RTEMS_RECORD_SYSTEM_435,
  RTEMS_RECORD_SYSTEM_436,
  RTEMS_RECORD_SYSTEM_437,
  RTEMS_RECORD_SYSTEM_438,
  RTEMS_RECORD_SYSTEM_439,
  RTEMS_RECORD_SYSTEM_440,
  RTEMS_RECORD_SYSTEM_441,
  RTEMS_RECORD_SYSTEM_442,
  RTEMS_RECORD_SYSTEM_443,
  RTEMS_RECORD_SYSTEM_444,
  RTEMS_RECORD_SYSTEM_445,
  RTEMS_RECORD_SYSTEM_446,
  RTEMS_RECORD_SYSTEM_447,
  RTEMS_RECORD_SYSTEM_448,
  RTEMS_RECORD_SYSTEM_449,
  RTEMS_RECORD_SYSTEM_450,
  RTEMS_RECORD_SYSTEM_451,
  RTEMS_RECORD_SYSTEM_452,
  RTEMS_RECORD_SYSTEM_453,
  RTEMS_RECORD_SYSTEM_454,
  RTEMS_RECORD_SYSTEM_455,
  RTEMS_RECORD_SYSTEM_456,
  RTEMS_RECORD_SYSTEM_457,
  RTEMS_RECORD_SYSTEM_458,
  RTEMS_RECORD_SYSTEM_459,
  RTEMS_RECORD_SYSTEM_460,
  RTEMS_RECORD_SYSTEM_461,
  RTEMS_RECORD_SYSTEM_462,
  RTEMS_RECORD_SYSTEM_463,
  RTEMS_RECORD_SYSTEM_464,
  RTEMS_RECORD_SYSTEM_465,
  RTEMS_RECORD_SYSTEM_466,
  RTEMS_RECORD_SYSTEM_467,
  RTEMS_RECORD_SYSTEM_468,
  RTEMS_RECORD_SYSTEM_469,
  RTEMS_RECORD_SYSTEM_470,
  RTEMS_RECORD_SYSTEM_471,
  RTEMS_RECORD_SYSTEM_472,
  RTEMS_RECORD_SYSTEM_473,
  RTEMS_RECORD_SYSTEM_474,
  RTEMS_RECORD_SYSTEM_475,
  RTEMS_RECORD_SYSTEM_476,
  RTEMS_RECORD_SYSTEM_477,
  RTEMS_RECORD_SYSTEM_478,
  RTEMS_RECORD_SYSTEM_479,
  RTEMS_RECORD_SYSTEM_480,
  RTEMS_RECORD_SYSTEM_481,
  RTEMS_RECORD_SYSTEM_482,
  RTEMS_RECORD_SYSTEM_483,
  RTEMS_RECORD_SYSTEM_484,
  RTEMS_RECORD_SYSTEM_485,
  RTEMS_RECORD_SYSTEM_486,
  RTEMS_RECORD_SYSTEM_487,
  RTEMS_RECORD_SYSTEM_488,
  RTEMS_RECORD_SYSTEM_489,
  RTEMS_RECORD_SYSTEM_490,
  RTEMS_RECORD_SYSTEM_491,
  RTEMS_RECORD_SYSTEM_492,
  RTEMS_RECORD_SYSTEM_493,
  RTEMS_RECORD_SYSTEM_494,
  RTEMS_RECORD_SYSTEM_495,
  RTEMS_RECORD_SYSTEM_496,
  RTEMS_RECORD_SYSTEM_497,
  RTEMS_RECORD_SYSTEM_498,
  RTEMS_RECORD_SYSTEM_499,
  RTEMS_RECORD_SYSTEM_500,
  RTEMS_RECORD_SYSTEM_501,
  RTEMS_RECORD_SYSTEM_502,
  RTEMS_RECORD_SYSTEM_503,
  RTEMS_RECORD_SYSTEM_504,
  RTEMS_RECORD_SYSTEM_505,
  RTEMS_RECORD_SYSTEM_506,
  RTEMS_RECORD_SYSTEM_507,
  RTEMS_RECORD_SYSTEM_508,
  RTEMS_RECORD_SYSTEM_509,
  RTEMS_RECORD_SYSTEM_510,
  RTEMS_RECORD_SYSTEM_511,

  /* There are 512 events reserved for the user */
  RTEMS_RECORD_USER_0,
  RTEMS_RECORD_USER_1,
  RTEMS_RECORD_USER_2,
  RTEMS_RECORD_USER_3,
  RTEMS_RECORD_USER_4,
  RTEMS_RECORD_USER_5,
  RTEMS_RECORD_USER_6,
  RTEMS_RECORD_USER_7,
  RTEMS_RECORD_USER_8,
  RTEMS_RECORD_USER_9,
  RTEMS_RECORD_USER_10,
  RTEMS_RECORD_USER_11,
  RTEMS_RECORD_USER_12,
  RTEMS_RECORD_USER_13,
  RTEMS_RECORD_USER_14,
  RTEMS_RECORD_USER_15,
  RTEMS_RECORD_USER_16,
  RTEMS_RECORD_USER_17,
  RTEMS_RECORD_USER_18,
  RTEMS_RECORD_USER_19,
  RTEMS_RECORD_USER_20,
  RTEMS_RECORD_USER_21,
  RTEMS_RECORD_USER_22,
  RTEMS_RECORD_USER_23,
  RTEMS_RECORD_USER_24,
  RTEMS_RECORD_USER_25,
  RTEMS_RECORD_USER_26,
  RTEMS_RECORD_USER_27,
  RTEMS_RECORD_USER_28,
  RTEMS_RECORD_USER_29,
  RTEMS_RECORD_USER_30,
  RTEMS_RECORD_USER_31,
  RTEMS_RECORD_USER_32,
  RTEMS_RECORD_USER_33,
  RTEMS_RECORD_USER_34,
  RTEMS_RECORD_USER_35,
  RTEMS_RECORD_USER_36,
  RTEMS_RECORD_USER_37,
  RTEMS_RECORD_USER_38,
  RTEMS_RECORD_USER_39,
  RTEMS_RECORD_USER_40,
  RTEMS_RECORD_USER_41,
  RTEMS_RECORD_USER_42,
  RTEMS_RECORD_USER_43,
  RTEMS_RECORD_USER_44,
  RTEMS_RECORD_USER_45,
  RTEMS_RECORD_USER_46,
  RTEMS_RECORD_USER_47,
  RTEMS_RECORD_USER_48,
  RTEMS_RECORD_USER_49,
  RTEMS_RECORD_USER_50,
  RTEMS_RECORD_USER_51,
  RTEMS_RECORD_USER_52,
  RTEMS_RECORD_USER_53,
  RTEMS_RECORD_USER_54,
  RTEMS_RECORD_USER_55,
  RTEMS_RECORD_USER_56,
  RTEMS_RECORD_USER_57,
  RTEMS_RECORD_USER_58,
  RTEMS_RECORD_USER_59,
  RTEMS_RECORD_USER_60,
  RTEMS_RECORD_USER_61,
  RTEMS_RECORD_USER_62,
  RTEMS_RECORD_USER_63,
  RTEMS_RECORD_USER_64,
  RTEMS_RECORD_USER_65,
  RTEMS_RECORD_USER_66,
  RTEMS_RECORD_USER_67,
  RTEMS_RECORD_USER_68,
  RTEMS_RECORD_USER_69,
  RTEMS_RECORD_USER_70,
  RTEMS_RECORD_USER_71,
  RTEMS_RECORD_USER_72,
  RTEMS_RECORD_USER_73,
  RTEMS_RECORD_USER_74,
  RTEMS_RECORD_USER_75,
  RTEMS_RECORD_USER_76,
  RTEMS_RECORD_USER_77,
  RTEMS_RECORD_USER_78,
  RTEMS_RECORD_USER_79,
  RTEMS_RECORD_USER_80,
  RTEMS_RECORD_USER_81,
  RTEMS_RECORD_USER_82,
  RTEMS_RECORD_USER_83,
  RTEMS_RECORD_USER_84,
  RTEMS_RECORD_USER_85,
  RTEMS_RECORD_USER_86,
  RTEMS_RECORD_USER_87,
  RTEMS_RECORD_USER_88,
  RTEMS_RECORD_USER_89,
  RTEMS_RECORD_USER_90,
  RTEMS_RECORD_USER_91,
  RTEMS_RECORD_USER_92,
  RTEMS_RECORD_USER_93,
  RTEMS_RECORD_USER_94,
  RTEMS_RECORD_USER_95,
  RTEMS_RECORD_USER_96,
  RTEMS_RECORD_USER_97,
  RTEMS_RECORD_USER_98,
  RTEMS_RECORD_USER_99,
  RTEMS_RECORD_USER_100,
  RTEMS_RECORD_USER_101,
  RTEMS_RECORD_USER_102,
  RTEMS_RECORD_USER_103,
  RTEMS_RECORD_USER_104,
  RTEMS_RECORD_USER_105,
  RTEMS_RECORD_USER_106,
  RTEMS_RECORD_USER_107,
  RTEMS_RECORD_USER_108,
  RTEMS_RECORD_USER_109,
  RTEMS_RECORD_USER_110,
  RTEMS_RECORD_USER_111,
  RTEMS_RECORD_USER_112,
  RTEMS_RECORD_USER_113,
  RTEMS_RECORD_USER_114,
  RTEMS_RECORD_USER_115,
  RTEMS_RECORD_USER_116,
  RTEMS_RECORD_USER_117,
  RTEMS_RECORD_USER_118,
  RTEMS_RECORD_USER_119,
  RTEMS_RECORD_USER_120,
  RTEMS_RECORD_USER_121,
  RTEMS_RECORD_USER_122,
  RTEMS_RECORD_USER_123,
  RTEMS_RECORD_USER_124,
  RTEMS_RECORD_USER_125,
  RTEMS_RECORD_USER_126,
  RTEMS_RECORD_USER_127,
  RTEMS_RECORD_USER_128,
  RTEMS_RECORD_USER_129,
  RTEMS_RECORD_USER_130,
  RTEMS_RECORD_USER_131,
  RTEMS_RECORD_USER_132,
  RTEMS_RECORD_USER_133,
  RTEMS_RECORD_USER_134,
  RTEMS_RECORD_USER_135,
  RTEMS_RECORD_USER_136,
  RTEMS_RECORD_USER_137,
  RTEMS_RECORD_USER_138,
  RTEMS_RECORD_USER_139,
  RTEMS_RECORD_USER_140,
  RTEMS_RECORD_USER_141,
  RTEMS_RECORD_USER_142,
  RTEMS_RECORD_USER_143,
  RTEMS_RECORD_USER_144,
  RTEMS_RECORD_USER_145,
  RTEMS_RECORD_USER_146,
  RTEMS_RECORD_USER_147,
  RTEMS_RECORD_USER_148,
  RTEMS_RECORD_USER_149,
  RTEMS_RECORD_USER_150,
  RTEMS_RECORD_USER_151,
  RTEMS_RECORD_USER_152,
  RTEMS_RECORD_USER_153,
  RTEMS_RECORD_USER_154,
  RTEMS_RECORD_USER_155,
  RTEMS_RECORD_USER_156,
  RTEMS_RECORD_USER_157,
  RTEMS_RECORD_USER_158,
  RTEMS_RECORD_USER_159,
  RTEMS_RECORD_USER_160,
  RTEMS_RECORD_USER_161,
  RTEMS_RECORD_USER_162,
  RTEMS_RECORD_USER_163,
  RTEMS_RECORD_USER_164,
  RTEMS_RECORD_USER_165,
  RTEMS_RECORD_USER_166,
  RTEMS_RECORD_USER_167,
  RTEMS_RECORD_USER_168,
  RTEMS_RECORD_USER_169,
  RTEMS_RECORD_USER_170,
  RTEMS_RECORD_USER_171,
  RTEMS_RECORD_USER_172,
  RTEMS_RECORD_USER_173,
  RTEMS_RECORD_USER_174,
  RTEMS_RECORD_USER_175,
  RTEMS_RECORD_USER_176,
  RTEMS_RECORD_USER_177,
  RTEMS_RECORD_USER_178,
  RTEMS_RECORD_USER_179,
  RTEMS_RECORD_USER_180,
  RTEMS_RECORD_USER_181,
  RTEMS_RECORD_USER_182,
  RTEMS_RECORD_USER_183,
  RTEMS_RECORD_USER_184,
  RTEMS_RECORD_USER_185,
  RTEMS_RECORD_USER_186,
  RTEMS_RECORD_USER_187,
  RTEMS_RECORD_USER_188,
  RTEMS_RECORD_USER_189,
  RTEMS_RECORD_USER_190,
  RTEMS_RECORD_USER_191,
  RTEMS_RECORD_USER_192,
  RTEMS_RECORD_USER_193,
  RTEMS_RECORD_USER_194,
  RTEMS_RECORD_USER_195,
  RTEMS_RECORD_USER_196,
  RTEMS_RECORD_USER_197,
  RTEMS_RECORD_USER_198,
  RTEMS_RECORD_USER_199,
  RTEMS_RECORD_USER_200,
  RTEMS_RECORD_USER_201,
  RTEMS_RECORD_USER_202,
  RTEMS_RECORD_USER_203,
  RTEMS_RECORD_USER_204,
  RTEMS_RECORD_USER_205,
  RTEMS_RECORD_USER_206,
  RTEMS_RECORD_USER_207,
  RTEMS_RECORD_USER_208,
  RTEMS_RECORD_USER_209,
  RTEMS_RECORD_USER_210,
  RTEMS_RECORD_USER_211,
  RTEMS_RECORD_USER_212,
  RTEMS_RECORD_USER_213,
  RTEMS_RECORD_USER_214,
  RTEMS_RECORD_USER_215,
  RTEMS_RECORD_USER_216,
  RTEMS_RECORD_USER_217,
  RTEMS_RECORD_USER_218,
  RTEMS_RECORD_USER_219,
  RTEMS_RECORD_USER_220,
  RTEMS_RECORD_USER_221,
  RTEMS_RECORD_USER_222,
  RTEMS_RECORD_USER_223,
  RTEMS_RECORD_USER_224,
  RTEMS_RECORD_USER_225,
  RTEMS_RECORD_USER_226,
  RTEMS_RECORD_USER_227,
  RTEMS_RECORD_USER_228,
  RTEMS_RECORD_USER_229,
  RTEMS_RECORD_USER_230,
  RTEMS_RECORD_USER_231,
  RTEMS_RECORD_USER_232,
  RTEMS_RECORD_USER_233,
  RTEMS_RECORD_USER_234,
  RTEMS_RECORD_USER_235,
  RTEMS_RECORD_USER_236,
  RTEMS_RECORD_USER_237,
  RTEMS_RECORD_USER_238,
  RTEMS_RECORD_USER_239,
  RTEMS_RECORD_USER_240,
  RTEMS_RECORD_USER_241,
  RTEMS_RECORD_USER_242,
  RTEMS_RECORD_USER_243,
  RTEMS_RECORD_USER_244,
  RTEMS_RECORD_USER_245,
  RTEMS_RECORD_USER_246,
  RTEMS_RECORD_USER_247,
  RTEMS_RECORD_USER_248,
  RTEMS_RECORD_USER_249,
  RTEMS_RECORD_USER_250,
  RTEMS_RECORD_USER_251,
  RTEMS_RECORD_USER_252,
  RTEMS_RECORD_USER_253,
  RTEMS_RECORD_USER_254,
  RTEMS_RECORD_USER_255,
  RTEMS_RECORD_USER_256,
  RTEMS_RECORD_USER_257,
  RTEMS_RECORD_USER_258,
  RTEMS_RECORD_USER_259,
  RTEMS_RECORD_USER_260,
  RTEMS_RECORD_USER_261,
  RTEMS_RECORD_USER_262,
  RTEMS_RECORD_USER_263,
  RTEMS_RECORD_USER_264,
  RTEMS_RECORD_USER_265,
  RTEMS_RECORD_USER_266,
  RTEMS_RECORD_USER_267,
  RTEMS_RECORD_USER_268,
  RTEMS_RECORD_USER_269,
  RTEMS_RECORD_USER_270,
  RTEMS_RECORD_USER_271,
  RTEMS_RECORD_USER_272,
  RTEMS_RECORD_USER_273,
  RTEMS_RECORD_USER_274,
  RTEMS_RECORD_USER_275,
  RTEMS_RECORD_USER_276,
  RTEMS_RECORD_USER_277,
  RTEMS_RECORD_USER_278,
  RTEMS_RECORD_USER_279,
  RTEMS_RECORD_USER_280,
  RTEMS_RECORD_USER_281,
  RTEMS_RECORD_USER_282,
  RTEMS_RECORD_USER_283,
  RTEMS_RECORD_USER_284,
  RTEMS_RECORD_USER_285,
  RTEMS_RECORD_USER_286,
  RTEMS_RECORD_USER_287,
  RTEMS_RECORD_USER_288,
  RTEMS_RECORD_USER_289,
  RTEMS_RECORD_USER_290,
  RTEMS_RECORD_USER_291,
  RTEMS_RECORD_USER_292,
  RTEMS_RECORD_USER_293,
  RTEMS_RECORD_USER_294,
  RTEMS_RECORD_USER_295,
  RTEMS_RECORD_USER_296,
  RTEMS_RECORD_USER_297,
  RTEMS_RECORD_USER_298,
  RTEMS_RECORD_USER_299,
  RTEMS_RECORD_USER_300,
  RTEMS_RECORD_USER_301,
  RTEMS_RECORD_USER_302,
  RTEMS_RECORD_USER_303,
  RTEMS_RECORD_USER_304,
  RTEMS_RECORD_USER_305,
  RTEMS_RECORD_USER_306,
  RTEMS_RECORD_USER_307,
  RTEMS_RECORD_USER_308,
  RTEMS_RECORD_USER_309,
  RTEMS_RECORD_USER_310,
  RTEMS_RECORD_USER_311,
  RTEMS_RECORD_USER_312,
  RTEMS_RECORD_USER_313,
  RTEMS_RECORD_USER_314,
  RTEMS_RECORD_USER_315,
  RTEMS_RECORD_USER_316,
  RTEMS_RECORD_USER_317,
  RTEMS_RECORD_USER_318,
  RTEMS_RECORD_USER_319,
  RTEMS_RECORD_USER_320,
  RTEMS_RECORD_USER_321,
  RTEMS_RECORD_USER_322,
  RTEMS_RECORD_USER_323,
  RTEMS_RECORD_USER_324,
  RTEMS_RECORD_USER_325,
  RTEMS_RECORD_USER_326,
  RTEMS_RECORD_USER_327,
  RTEMS_RECORD_USER_328,
  RTEMS_RECORD_USER_329,
  RTEMS_RECORD_USER_330,
  RTEMS_RECORD_USER_331,
  RTEMS_RECORD_USER_332,
  RTEMS_RECORD_USER_333,
  RTEMS_RECORD_USER_334,
  RTEMS_RECORD_USER_335,
  RTEMS_RECORD_USER_336,
  RTEMS_RECORD_USER_337,
  RTEMS_RECORD_USER_338,
  RTEMS_RECORD_USER_339,
  RTEMS_RECORD_USER_340,
  RTEMS_RECORD_USER_341,
  RTEMS_RECORD_USER_342,
  RTEMS_RECORD_USER_343,
  RTEMS_RECORD_USER_344,
  RTEMS_RECORD_USER_345,
  RTEMS_RECORD_USER_346,
  RTEMS_RECORD_USER_347,
  RTEMS_RECORD_USER_348,
  RTEMS_RECORD_USER_349,
  RTEMS_RECORD_USER_350,
  RTEMS_RECORD_USER_351,
  RTEMS_RECORD_USER_352,
  RTEMS_RECORD_USER_353,
  RTEMS_RECORD_USER_354,
  RTEMS_RECORD_USER_355,
  RTEMS_RECORD_USER_356,
  RTEMS_RECORD_USER_357,
  RTEMS_RECORD_USER_358,
  RTEMS_RECORD_USER_359,
  RTEMS_RECORD_USER_360,
  RTEMS_RECORD_USER_361,
  RTEMS_RECORD_USER_362,
  RTEMS_RECORD_USER_363,
  RTEMS_RECORD_USER_364,
  RTEMS_RECORD_USER_365,
  RTEMS_RECORD_USER_366,
  RTEMS_RECORD_USER_367,
  RTEMS_RECORD_USER_368,
  RTEMS_RECORD_USER_369,
  RTEMS_RECORD_USER_370,
  RTEMS_RECORD_USER_371,
  RTEMS_RECORD_USER_372,
  RTEMS_RECORD_USER_373,
  RTEMS_RECORD_USER_374,
  RTEMS_RECORD_USER_375,
  RTEMS_RECORD_USER_376,
  RTEMS_RECORD_USER_377,
  RTEMS_RECORD_USER_378,
  RTEMS_RECORD_USER_379,
  RTEMS_RECORD_USER_380,
  RTEMS_RECORD_USER_381,
  RTEMS_RECORD_USER_382,
  RTEMS_RECORD_USER_383,
  RTEMS_RECORD_USER_384,
  RTEMS_RECORD_USER_385,
  RTEMS_RECORD_USER_386,
  RTEMS_RECORD_USER_387,
  RTEMS_RECORD_USER_388,
  RTEMS_RECORD_USER_389,
  RTEMS_RECORD_USER_390,
  RTEMS_RECORD_USER_391,
  RTEMS_RECORD_USER_392,
  RTEMS_RECORD_USER_393,
  RTEMS_RECORD_USER_394,
  RTEMS_RECORD_USER_395,
  RTEMS_RECORD_USER_396,
  RTEMS_RECORD_USER_397,
  RTEMS_RECORD_USER_398,
  RTEMS_RECORD_USER_399,
  RTEMS_RECORD_USER_400,
  RTEMS_RECORD_USER_401,
  RTEMS_RECORD_USER_402,
  RTEMS_RECORD_USER_403,
  RTEMS_RECORD_USER_404,
  RTEMS_RECORD_USER_405,
  RTEMS_RECORD_USER_406,
  RTEMS_RECORD_USER_407,
  RTEMS_RECORD_USER_408,
  RTEMS_RECORD_USER_409,
  RTEMS_RECORD_USER_410,
  RTEMS_RECORD_USER_411,
  RTEMS_RECORD_USER_412,
  RTEMS_RECORD_USER_413,
  RTEMS_RECORD_USER_414,
  RTEMS_RECORD_USER_415,
  RTEMS_RECORD_USER_416,
  RTEMS_RECORD_USER_417,
  RTEMS_RECORD_USER_418,
  RTEMS_RECORD_USER_419,
  RTEMS_RECORD_USER_420,
  RTEMS_RECORD_USER_421,
  RTEMS_RECORD_USER_422,
  RTEMS_RECORD_USER_423,
  RTEMS_RECORD_USER_424,
  RTEMS_RECORD_USER_425,
  RTEMS_RECORD_USER_426,
  RTEMS_RECORD_USER_427,
  RTEMS_RECORD_USER_428,
  RTEMS_RECORD_USER_429,
  RTEMS_RECORD_USER_430,
  RTEMS_RECORD_USER_431,
  RTEMS_RECORD_USER_432,
  RTEMS_RECORD_USER_433,
  RTEMS_RECORD_USER_434,
  RTEMS_RECORD_USER_435,
  RTEMS_RECORD_USER_436,
  RTEMS_RECORD_USER_437,
  RTEMS_RECORD_USER_438,
  RTEMS_RECORD_USER_439,
  RTEMS_RECORD_USER_440,
  RTEMS_RECORD_USER_441,
  RTEMS_RECORD_USER_442,
  RTEMS_RECORD_USER_443,
  RTEMS_RECORD_USER_444,
  RTEMS_RECORD_USER_445,
  RTEMS_RECORD_USER_446,
  RTEMS_RECORD_USER_447,
  RTEMS_RECORD_USER_448,
  RTEMS_RECORD_USER_449,
  RTEMS_RECORD_USER_450,
  RTEMS_RECORD_USER_451,
  RTEMS_RECORD_USER_452,
  RTEMS_RECORD_USER_453,
  RTEMS_RECORD_USER_454,
  RTEMS_RECORD_USER_455,
  RTEMS_RECORD_USER_456,
  RTEMS_RECORD_USER_457,
  RTEMS_RECORD_USER_458,
  RTEMS_RECORD_USER_459,
  RTEMS_RECORD_USER_460,
  RTEMS_RECORD_USER_461,
  RTEMS_RECORD_USER_462,
  RTEMS_RECORD_USER_463,
  RTEMS_RECORD_USER_464,
  RTEMS_RECORD_USER_465,
  RTEMS_RECORD_USER_466,
  RTEMS_RECORD_USER_467,
  RTEMS_RECORD_USER_468,
  RTEMS_RECORD_USER_469,
  RTEMS_RECORD_USER_470,
  RTEMS_RECORD_USER_471,
  RTEMS_RECORD_USER_472,
  RTEMS_RECORD_USER_473,
  RTEMS_RECORD_USER_474,
  RTEMS_RECORD_USER_475,
  RTEMS_RECORD_USER_476,
  RTEMS_RECORD_USER_477,
  RTEMS_RECORD_USER_478,
  RTEMS_RECORD_USER_479,
  RTEMS_RECORD_USER_480,
  RTEMS_RECORD_USER_481,
  RTEMS_RECORD_USER_482,
  RTEMS_RECORD_USER_483,
  RTEMS_RECORD_USER_484,
  RTEMS_RECORD_USER_485,
  RTEMS_RECORD_USER_486,
  RTEMS_RECORD_USER_487,
  RTEMS_RECORD_USER_488,
  RTEMS_RECORD_USER_489,
  RTEMS_RECORD_USER_490,
  RTEMS_RECORD_USER_491,
  RTEMS_RECORD_USER_492,
  RTEMS_RECORD_USER_493,
  RTEMS_RECORD_USER_494,
  RTEMS_RECORD_USER_495,
  RTEMS_RECORD_USER_496,
  RTEMS_RECORD_USER_497,
  RTEMS_RECORD_USER_498,
  RTEMS_RECORD_USER_499,
  RTEMS_RECORD_USER_500,
  RTEMS_RECORD_USER_501,
  RTEMS_RECORD_USER_502,
  RTEMS_RECORD_USER_503,
  RTEMS_RECORD_USER_504,
  RTEMS_RECORD_USER_505,
  RTEMS_RECORD_USER_506,
  RTEMS_RECORD_USER_507,
  RTEMS_RECORD_USER_508,
  RTEMS_RECORD_USER_509,
  RTEMS_RECORD_USER_510,
  RTEMS_RECORD_USER_511
} rtems_record_event;

#define RTEMS_RECORD_LAST RTEMS_RECORD_USER_511

#define RTEMS_RECORD_USER( index ) ( RTEMS_RECORD_USER_0 + ( index ) )

/**
 * @brief Bits in the record item event member reserved for the actual event.
 */
#define RTEMS_RECORD_EVENT_BITS 10

/**
 * @brief Bits in the record item event member reserved for the time of the
 * event.
 */
#define RTEMS_RECORD_TIME_BITS 22

/**
 * @brief Builds a time event for the specified time stamp and event.
 *
 * The events are stored in the record item with a time stamp.  There are 22
 * bits allocated to the time stamp and 10 bits allocated to the event.  The 22
 * bits are enough to get reliable time stamps on a system with a 4GHz CPU
 * counter and a 1000Hz clock tick.
 */
#define RTEMS_RECORD_TIME_EVENT( time, event ) \
  ( ( ( time ) << RTEMS_RECORD_EVENT_BITS ) | ( event ) )

/**
 * @brief Gets the time of a time event.
 */
#define RTEMS_RECORD_GET_TIME( time_event ) \
  ( ( time_event ) >> RTEMS_RECORD_EVENT_BITS )

/**
 * @brief Gets the event of a time event.
 */
#define RTEMS_RECORD_GET_EVENT( time_event ) \
  ( ( time_event ) & ( ( 1U << RTEMS_RECORD_EVENT_BITS ) - 1U ) )

/**
 * @brief The record data integer type.
 *
 * It is big enough to store 32-bit integers and pointers.
 */
typedef unsigned long rtems_record_data;

/**
 * @brief The native record item.
 */
typedef struct __attribute__((__packed__)) {
  uint32_t          event;
  rtems_record_data data;
} rtems_record_item;

/**
 * @brief The 32-bit format record item.
 */
typedef struct {
  uint32_t event;
  uint32_t data;
} rtems_record_item_32;

/**
 * @brief The 64-bit format record item.
 */
typedef struct __attribute__((__packed__)) {
  uint32_t event;
  uint64_t data;
} rtems_record_item_64;

const char *rtems_record_event_text( rtems_record_event event );

/** @} */

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* _RTEMS_RECORDDATA_H */
