/* PCI IRQ Library
 *
 * COPYRIGHT (c) 2010 Cobham Gaisler AB.
 *
 * The license and distribution terms for this file may be
 * found in the file LICENSE in this distribution or at
 * http://www.rtems.org/license/LICENSE.
 */

/* IRQ handling does not have so much with PCI to do, this library depends
 * on the BSP to implement shared interrupts.
 */

#ifndef __PCI_IRQ_H__
#define __PCI_IRQ_H__

#include <rtems/irq-extension.h>
#include <rtems/score/basedefs.h>

/*
 * FIXME: This should be available via the IRQ extensions API.
 *
 * https://devel.rtems.org/ticket/3269
 */
void BSP_shared_interrupt_clear(int irq);
void BSP_shared_interrupt_unmask(int irq);
void BSP_shared_interrupt_mask(int irq);

/* PCI Handler (ISR) called when IRQ is generated by any of the PCI devices
 * connected to the same PCI IRQ Pin. This has been defined the same way as
 * rtems_interrupt_handler in order for BSPs to "direct-map" the register
 * and unregister functions rtems_interrupt_handler_install/remove
 */
typedef void (*pci_isr)(void *arg);

/* Get assigned system IRQ to a PCI Device. If no IRQ 0 is returned */
extern int pci_dev_irq(pci_dev_t dev);

/* Register shared PCI IRQ handler, but does not enable it. The system interrupt
 * number is read from the PCI board's PCI configuration space header iline
 * field. The iline field is initialized by the PCI subsystem during start up,
 * the ipin field is translated into a system IRQ and written to iline. The
 * board's driver should use the iline field as the irq argument to this
 * function.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
RTEMS_INLINE_ROUTINE int pci_interrupt_register(int irq, const char *info,
						pci_isr isr, void *arg)
{
	return rtems_interrupt_handler_install(irq, info,
					       RTEMS_INTERRUPT_SHARED, isr,
					       arg);
}

/* Unregister previously registered shared PCI IRQ handler
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
RTEMS_INLINE_ROUTINE int pci_interrupt_unregister(int irq, pci_isr isr,
						  void *arg)
{
	return rtems_interrupt_handler_remove(irq, isr, arg);
}

/* Enable shared PCI IRQ handler. This function will unmask the interrupt
 * controller and mark this interrupt handler ready to handle interrupts. Note
 * that since it is a shared interrupt handler service the interrupt may
 * already be enabled, however no calls to this specific handler is made
 * until it is enabled.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
RTEMS_INLINE_ROUTINE void pci_interrupt_unmask(int irq)
{
	BSP_shared_interrupt_unmask(irq);
}

/* Disable shared PCI IRQ handler. This function will mask the interrupt
 * controller and mark this interrupt handler not ready to receive interrupts.
 * Note that since it is a shared interrupt handler service the interrupt may
 * still be enabled, however no calls to this specific handler is made
 * while it is disabled.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
RTEMS_INLINE_ROUTINE void pci_interrupt_mask(int irq)
{
	BSP_shared_interrupt_mask(irq);
}

/* Acknowledge the interrupt controller by writing to the interrupt controller.
 * Note that since it is a shared interrupt handler service, clearing the
 * interrupt source may affect other ISRs registered to this IRQ.
 *
 * Arguments
 *  irq       System IRQ number, normally taken from the PCI configuration area
 *  isr       Function pointer to the ISR
 *  arg       Second argument to function isr
 */
RTEMS_INLINE_ROUTINE void pci_interrupt_clear(int irq)
{
	BSP_shared_interrupt_clear(irq);
}

#endif /* !__PCI_IRQ_H__ */
