
#include <stdio.h>
#include <stdlib.h>

#include "canbtrs.h"

//#define DISABLE_OCCAN 1
//#define DISABLE_GRCAN 1

#define OCCAN_SPEED_1000K 1000000
#define OCCAN_SPEED_800K  800000
#define OCCAN_SPEED_500K  500000
#define OCCAN_SPEED_250K  250000
#define OCCAN_SPEED_125K  125000
#define OCCAN_SPEED_75K   75000
#define OCCAN_SPEED_50K   50000
#define OCCAN_SPEED_25K   25000
#define OCCAN_SPEED_10K   10000


/*  register bit definitions */
#define OCCAN_BUSTIM_SJW       0xc0
#define OCCAN_BUSTIM_BRP       0x3f
#define OCCAN_BUSTIM_SJW_BIT   6

#define OCCAN_BUSTIM_SAM       0x80
#define OCCAN_BUSTIM_TSEG2     0x70
#define OCCAN_BUSTIM_TSEG2_BIT 4
#define OCCAN_BUSTIM_TSEG1     0x0f

#define MAX_TSEG1 15
#define MAX_TSEG2 7

typedef struct {
	unsigned char btr0;
	unsigned char btr1;
} occan_speed_regs;

unsigned int steplist[] = 
{
  OCCAN_SPEED_1000K,
  OCCAN_SPEED_800K,
  OCCAN_SPEED_500K,
  OCCAN_SPEED_250K,
  OCCAN_SPEED_125K,
  OCCAN_SPEED_75K,
  OCCAN_SPEED_50K,
  OCCAN_SPEED_25K,
  OCCAN_SPEED_10K,
  0
};

void print_timing(unsigned int corefreq, unsigned int rate, occan_speed_regs *timing)
{
  unsigned int baud;
  unsigned int sjw, brp, tseg1, tseg2;
  double err;
  
  printf("BTR0: 0x%x\n",timing->btr0);
  printf("BTR1: 0x%x\n",timing->btr1);
  
  
  tseg1 = timing->btr1 & OCCAN_BUSTIM_TSEG1;
  tseg2 = (timing->btr1>>OCCAN_BUSTIM_TSEG2_BIT) & 0x7;
  brp   = timing->btr0 & OCCAN_BUSTIM_BRP;
  sjw   = timing->btr0 >> OCCAN_BUSTIM_SJW_BIT;
  printf("tseg1: %d, tseg2: %d, brp: %d, sjw: %d\n",tseg1,tseg2,brp,sjw);
  
  baud = corefreq / 2;
  baud = baud / (brp+1);
  baud = baud / (3+tseg1+tseg2);
  
  printf("Baud:   %d\n",baud);
  printf("Sampl:  %d%%\n",(100*(1+tseg1+1))/(1+tseg1+1+tseg2+1));
  err = baud;
  err = err/rate;
  err = err*100-100; /* to % */
  printf("Err:    %f%%\n",err);
}

/* 60000000/(scaler+1)/(2^bpr)/(ps1+ps2+2) */

void print_grtiming(unsigned int corefreq, unsigned int rate, struct grlib_canbtrs_timing *timing)
{
  unsigned int baud;
  double err;
  printf("SCALER: 0x%x\n",timing->scaler);
  printf("PS1:    0x%x\n",timing->ps1);
  printf("PS2:    0x%x\n",timing->ps2);
  printf("RSJ:    0x%x\n",timing->rsj);
  printf("BPR:    0x%x\n",timing->bpr);
  
  baud = corefreq/(timing->scaler+1);
  baud = baud/(1<<timing->bpr);
  baud = baud/(2+timing->ps1+timing->ps2);
  printf("Baud:   %d\n",baud);
  printf("Sampl:  %d%%\n",(100*(1+timing->ps1))/(2+timing->ps1+timing->ps2));
  err = baud;
  err = err/rate;
  err = err*100-100; /* to % */
  printf("Err:    %f%%\n",err);
}

/* OCCAN baud-rate paramter boundaries */
struct grlib_canbtrs_ranges occan_btrs_ranges = {
	.max_scaler = 64,
	.has_bpr = 0,
	.divfactor = 1,
	.min_tseg1 = 1,
	.max_tseg1 = 16,
	.min_tseg2 = 1,
	.max_tseg2 = 8,
};

/* GRCAN nominal boundaries for baud-rate paramters */
struct grlib_canbtrs_ranges grcan_btrs_ranges = {
	.max_scaler = 256*8, /* scaler is multiplied by BPR in steps 1,2,4,8 */
	.has_bpr = 1,
	.divfactor = 2,
	.min_tseg1 = 1,
	.max_tseg1 = 15,
	.min_tseg2 = 2,
	.max_tseg2 = 8,
};

/* GRCANFD nominal boundaries */
struct grlib_canbtrs_ranges grcanfd_nom_btrs_ranges = {
	.max_scaler = 256,
	.has_bpr = 0,
	.divfactor = 1,
	.min_tseg1 = 2,
	.max_tseg1 = 63,
	.min_tseg2 = 2,
	.max_tseg2 = 16,
};

static void convert_timing_to_btrs(
	struct grlib_canbtrs_timing *t,
	occan_speed_regs *btrs)
{
	btrs->btr0 = (t->rsj << OCCAN_BUSTIM_SJW_BIT) |
	             (t->scaler & OCCAN_BUSTIM_BRP);
	btrs->btr1 = (0<<7) | ((t->ps2 - 1) << OCCAN_BUSTIM_TSEG2_BIT) |
	             (t->ps1 - 1);
}

int main(int argc, char *argv[]){
  unsigned int clock_hz=40000000;
  int i,ret;
  struct grlib_canbtrs_timing timing;
  occan_speed_regs occan_btrs;
  unsigned int occan_clock;
  
  if ( argc > 2 ){
    printf("usage: %s [can_core_clock_in_Hz]\n",argv[0]);
    return 0;
  }
  if ( argc == 2 ){
    clock_hz = strtoul(argv[1],NULL,10);
  }

#ifndef DISABLE_OCCAN
  printf("Calculating OC-CAN baud rates for %dkHz\n",clock_hz/1000);
  
  printf("\n");
  i=0;
  while( steplist[i] != 0 ){
    occan_clock = clock_hz / 2; /* BPR is always divided by 2 */
    ret = grlib_canbtrs_calc_timing(steplist[i], occan_clock, 80,
                                    &occan_btrs_ranges, &timing);
    if ( !ret ){
      printf("---------- %d bits/s ----------\n",steplist[i]);
      convert_timing_to_btrs(&timing, &occan_btrs);
      print_timing(clock_hz,steplist[i],&occan_btrs);
    }else{
      printf("Error getting OCCAN Baud rate for %d @ %dkHz\n",steplist[i],clock_hz/1000);
    }
    i++;
  }
#endif
#ifndef DISABLE_GRCAN
  printf("\n\n\nCalculating GRCAN baud rates for %dkHz\n",clock_hz/1000);
  
  printf("\n");
  
  i=0;
  while( steplist[i] != 0 ){
    ret = grlib_canbtrs_calc_timing(steplist[i], clock_hz, 80,
                                    &grcan_btrs_ranges, &timing);

    if ( !ret ){
      printf("---------- %d bits/s ----------\n",steplist[i]);
      print_grtiming(clock_hz,steplist[i],&timing);
    }else{
      printf("Error getting GRCAN Baud rate for %d @ %dkHz\n",steplist[i],clock_hz/1000);
    }
    i++;
  }
#endif
  
  return 0;
}
