/*
 *  AMBA Plag & Play Bus Driver
 *
 *  This driver hook performs bus scanning.
 *
 *  COPYRIGHT (c) 2004.
 *  Gaisler Research
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 *
 *  $Id: amba.c,v 1.1.2.3 2006/04/24 16:58:41 joel Exp $
 */

#include <bsp.h>

#define amba_insert_device(tab, address) \
{ \
  if (*(address)) \
  { \
    (tab)->addr[(tab)->devnr] = (address); \
    (tab)->devnr ++; \
  } \
} while(0)


/* Structure containing address to devices found on the Amba Plug&Play bus */
amba_confarea_type amba_conf;

/* Pointers to Interrupt Controller configuration registers */
volatile LEON3_IrqCtrl_Regs_Map *LEON3_IrqCtrl_Regs;

int LEON3_Cpu_Index = 0;
static int apb_init = 0;

/*
 *  bsp_leon3_predriver_hook
 *
 *  BSP predriver hook.  Called just before drivers are initialized.
 *  Used to scan system bus. Probes for AHB masters, AHB slaves and 
 *  APB slaves. Addresses to configuration areas of the AHB masters,
 *  AHB slaves, APB slaves and APB master are storeds in 
 *  amba_ahb_masters, amba_ahb_slaves and amba.
 */

unsigned int getasr17();

asm(" .text  \n"
    "getasr17:   \n"
    "retl \n"
    "mov %asr17, %o0\n"
);

extern rtems_configuration_table Configuration;

void bsp_leon3_predriver_hook(void)
{
  unsigned int *cfg_area;  /* address to configuration area */
  unsigned int mbar, iobar, conf;
  int i, j;
  unsigned int tmp;

  amba_conf.ahbmst.devnr = 0; amba_conf.ahbslv.devnr = 0; amba_conf.apbslv.devnr = 0;
  cfg_area = (unsigned int *) (LEON3_IO_AREA | LEON3_CONF_AREA);
	amba_conf.ioarea = LEON3_IO_AREA;

  for (i = 0; i < LEON3_AHB_MASTERS; i++) 
  {
    amba_insert_device(&amba_conf.ahbmst, cfg_area);
    cfg_area += LEON3_AHB_CONF_WORDS;
  }

  cfg_area = (unsigned int *) (LEON3_IO_AREA | LEON3_CONF_AREA | LEON3_AHB_SLAVE_CONF_AREA);
  for (i = 0; i < LEON3_AHB_SLAVES; i++) 
  {
    amba_insert_device(&amba_conf.ahbslv, cfg_area);
    cfg_area += LEON3_AHB_CONF_WORDS;
  }  

  for (i = 0; i < amba_conf.ahbslv.devnr; i ++) 
  {
    conf = amba_get_confword(amba_conf.ahbslv, i, 0);
    mbar = amba_ahb_get_membar(amba_conf.ahbslv, i, 0);
    if ((amba_vendor(conf) == VENDOR_GAISLER) && (amba_device(conf) == GAISLER_APBMST) &&
	(apb_init == 0))
    {
      amba_conf.apbmst = amba_membar_start(mbar);
      cfg_area = (unsigned int *) (amba_conf.apbmst | LEON3_CONF_AREA);
      for (j = amba_conf.apbslv.devnr; j < LEON3_APB_SLAVES; j++)
      {
	amba_insert_device(&amba_conf.apbslv, cfg_area);
	cfg_area += LEON3_APB_CONF_WORDS;
      }
      apb_init = 1;
    }
  }    

  /* Find LEON3 Interrupt controler */
  i = 0;
  while (i < amba_conf.apbslv.devnr) 
  {
    conf = amba_get_confword(amba_conf.apbslv, i, 0);
    if ((amba_vendor(conf) == VENDOR_GAISLER) && (amba_device(conf) == GAISLER_IRQMP))
    {
      iobar = amba_apb_get_membar(amba_conf.apbslv, i);
      LEON3_IrqCtrl_Regs = (volatile LEON3_IrqCtrl_Regs_Map *) amba_iobar_start(amba_conf.apbmst, iobar);
      /* asm("mov %%asr17, %0": : "r" (tmp)); */
      if (Configuration.User_multiprocessing_table != NULL)
      {	
	tmp = getasr17();
	LEON3_Cpu_Index = (tmp >> 28) & 3;
      }
      break;
    }
    i++;
  }
}

/**** APB Slaves ****/

/* Return number of APB Slave devices which has given vendor and device */
int amba_get_number_apbslv_devices(amba_confarea_type *amba_conf, int vendor, int device)
{
	unsigned int conf;
	int cnt,i;
	
	for (cnt=i=0; i < amba_conf->apbslv.devnr; i++)
  {
    conf = amba_get_confword(amba_conf->apbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device))
			cnt++;
  }
	return cnt;
}

/* Get First APB Slave device of this vendor&device id */
int amba_find_apbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_apb_device *dev)
{
	unsigned int conf,iobar;
	int i;
	
	for (i=0; i < amba_conf->apbslv.devnr; i++)
  {
    conf = amba_get_confword(amba_conf->apbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device)){		
      iobar = amba_apb_get_membar(amba_conf->apbslv, i);
      dev->start = amba_iobar_start(amba_conf->apbmst, iobar);
			dev->irq = amba_irq(conf);
			return 1;
		}
  }
	return 0;
}

/* Get APB Slave device of this vendor&device id. (setting nr to 0 is eqivalent to calling amba_find_apbslv() ) */
int amba_find_next_apbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_apb_device *dev, int index)
{
	unsigned int conf,iobar;
	int cnt,i;
	
	for (cnt=i=0; i < amba_conf->apbslv.devnr; i++)
  {
    conf = amba_get_confword(amba_conf->apbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device)){
			if ( cnt == index ){
				/* found device */
				iobar = amba_apb_get_membar(amba_conf->apbslv, i);
  	    dev->start = amba_iobar_start(amba_conf->apbmst, iobar);
				dev->irq = amba_irq(conf);
				return 1;
			}
			cnt++;
		}
  }
	return 0;
}

/* Get first nr APB Slave devices, put them into dev (which is an array of nr length) */
int amba_find_apbslvs(amba_confarea_type *amba_conf, int vendor, int device, amba_apb_device *devs, int maxno)
{
	unsigned int conf,iobar;
	int cnt,i;
	
	for (cnt=i=0; (i<amba_conf->apbslv.devnr) && (cnt<maxno) ; i++)
  {
    conf = amba_get_confword(amba_conf->apbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device)){
			/* found device */
			iobar = amba_apb_get_membar(amba_conf->apbslv, i);
 	    devs[cnt].start = amba_iobar_start(amba_conf->apbmst, iobar);
			devs[cnt].irq = amba_irq(conf);
			cnt++;
		}
  }
	return cnt;
}

/***** AHB SLAVES *****/

/* Return number of AHB Slave devices which has given vendor and device */
int amba_get_number_ahbslv_devices(amba_confarea_type *amba_conf, int vendor, int device)
{
	unsigned int conf;
	int cnt,i;
	
	for (cnt=i=0; i < amba_conf->ahbslv.devnr; i++)
  {
    conf = amba_get_confword(amba_conf->ahbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device))
			cnt++;
  }
	return cnt;
}

/* Get First AHB Slave device of this vendor&device id */
int amba_find_ahbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_ahb_device *dev)
{
	unsigned int conf,mbar,addr;
	int j,i;
	
	for (i=0; i < amba_conf->ahbslv.devnr; i++)
  {
    conf = amba_get_confword(amba_conf->ahbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device)){
			for(j=0; j<4; j++){
				mbar = amba_ahb_get_membar(amba_conf->ahbslv, i, j);
				addr = amba_membar_start(mbar);
				if ( amba_membar_type(mbar) == AMBA_TYPE_AHBIO ){
					addr = AMBA_TYPE_AHBIO_ADDR(addr,amba_conf->ioarea);
				}
	      dev->start[j] = addr;
			}
			dev->irq = amba_irq(conf);
			dev->ver = amba_ver(conf);
			return 1;
		}
  }
	return 0;
}

/* Get AHB Slave device of this vendor&device id. (setting nr to 0 is eqivalent to calling amba_find_ahbslv() ) */
int amba_find_next_ahbslv(amba_confarea_type *amba_conf, int vendor, int device, amba_ahb_device *dev, int index)
{
	unsigned int conf,mbar,addr;
	int i,j,cnt;
	
	for (cnt=i=0; i < amba_conf->ahbslv.devnr; i++)
  {
    conf = amba_get_confword(amba_conf->ahbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device)){
			if ( cnt == index ){
				for(j=0; j<4; j++){
					mbar = amba_ahb_get_membar(amba_conf->ahbslv, i, j);
					addr = amba_membar_start(mbar);
					if ( amba_membar_type(mbar) == AMBA_TYPE_AHBIO ){
						addr = AMBA_TYPE_AHBIO_ADDR(addr,amba_conf->ioarea);
					}
		      dev->start[j] = addr;
				}
				dev->irq = amba_irq(conf);
				dev->ver = amba_ver(conf);
				return 1;
			}
			cnt++;
		}
  }
	return 0;
}

/* Get first nr AHB Slave devices, put them into dev (which is an array of nr length) */
int amba_find_ahbslvs(amba_confarea_type *amba_conf, int vendor, int device, amba_ahb_device *devs, int maxno)
{
	unsigned int conf,mbar,addr;
	int i,j,cnt;
	
	for (cnt=i=0; (i < amba_conf->ahbslv.devnr) && (maxno<cnt); i++)
  {
    conf = amba_get_confword(amba_conf->ahbslv, i, 0);
    if ((amba_vendor(conf) == vendor) && (amba_device(conf) == device)){
			for(j=0; j<4; j++){
				mbar = amba_ahb_get_membar(amba_conf->ahbslv, i, j);
				addr = amba_membar_start(mbar);
				if ( amba_membar_type(mbar) == AMBA_TYPE_AHBIO ){
					addr = AMBA_TYPE_AHBIO_ADDR(addr,amba_conf->ioarea);
				}
	      devs[cnt].start[j] = addr;
			}
			devs[cnt].irq = amba_irq(conf);
			devs[cnt].ver = amba_ver(conf);
			cnt++;
		}
  }
	return cnt;
}

