/*
 *  AMBA Plag & Play Bus Driver Macros for LEON2
 *
 *  Macros used for AMBA Plug & Play bus scanning
 *
 *  COPYRIGHT (c) 2007.
 *  Gaisler Research
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.com/license/LICENSE.
 *
 *  $Id: ambapp.h,v 1.1.2.2 2007/12/05 15:41:28 joel Exp $
 */

#ifndef __AMBAPP_H__
#define __AMBAPP_H__

/* Include VENDOR and DEVICE definitions */
#include <ambapp_ids.h>
#include <ambapp_new.h>
#include <grlib.h>

#ifdef __cplusplus
extern "C" {
#endif

#define AMBA_CONF_AREA 0xff000
#define AMBA_AHB_SLAVE_CONF_AREA (1 << 11)

#define AMBA_AHB_CONF_WORDS 8
#define AMBA_APB_CONF_WORDS 2
#define AMBA_AHB_MASTERS 16
#define AMBA_AHB_SLAVES 16
#define AMBA_APB_SLAVES_PER_BUS 16
#define AMBA_APB_SLAVES 32
#define AMBA_APBUARTS 8

/* 
 *
 * Macros for manipulating Configuration registers  
 *
 */
#define amba_get_confword(tab, index, word) (*((tab).addr[(index)]+(word)))

#define amba_vendor(x) (((x) >> 24) & 0xff)

#define amba_device(x) (((x) >> 12) & 0xfff)

#define amba_ahb_get_membar(tab, index, nr) (*((tab).addr[(index)]+4+(nr)))

#define amba_ahb_get_custom(tab, index, nr) (*((tab).addr[(index)]+1+(nr)))

#define amba_apb_get_membar(tab, index) (*((tab).addr[(index)]+1))

#define amba_membar_start(mbar) (((mbar) & 0xfff00000) & (((mbar) & 0xfff0) << 16))

#define amba_iobar_start(base, iobar) ((base) | ((((iobar) & 0xfff00000)>>12) & (((iobar) & 0xfff0)<<4)) )

#define amba_irq(conf) ((conf) & 0x1f)

#define amba_ver(conf) (((conf)>>5) & 0x1f)

#define amba_membar_type(mbar) ((mbar) & 0xf)

#define amba_membar_mask(mbar) (((mbar)>>4) & 0xfff)

#define AMBA_TYPE_APBIO 0x1
#define AMBA_TYPE_MEM   0x2
#define AMBA_TYPE_AHBIO 0x3

#define AMBA_TYPE_AHBIO_ADDR(addr,base_ioarea) ((unsigned int)(base_ioarea) | ((addr) >> 12))

/*
 *  Types and structure used for AMBA Plug & Play bus scanning 
 *
 */
typedef struct amba_device_table {
  int devnr;                    /* numbrer of devices on AHB or APB bus */
  unsigned int *addr[16];       /* addresses to the devices configuration tables */
} amba_device_table;

typedef struct {
  int devnr;
  unsigned int *addr[AMBA_APB_SLAVES];  /* addresses to the devices configuration tables */
  unsigned int apbmst[AMBA_APB_SLAVES]; /* pointer to AHB slave (which is a APB master) */
} amba_apb_dev;

struct amba_mmap {
  unsigned int cpu_adr;
  unsigned int size;
  unsigned int remote_amba_adr;
};

typedef struct _amba_confarea_type amba_confarea_type;

struct _amba_confarea_type {
  amba_confarea_type  *next;   /* next bus in chain */
  int                 notroot; /* is root of a bus (mother AHB has 64 masters/slaves rest 16) */
	unsigned int        ioarea;
  struct amba_mmap    *mmaps;
  amba_device_table   ahbmst;
  amba_device_table   ahbslv;
  amba_apb_dev        apbslv;
};

typedef struct {
  unsigned int start, irq, bus_id;
} amba_apb_device;

typedef struct {
  unsigned int start[4], mask[4], irq, ver;
} amba_ahb_device;

/* Scans AMBA Plug&Play Information and convers that information
 * to a more readable format in RAM.
 *
 * Will scan for - AHB Masters
 *               - AHB Slaves
 *               - APB Slaves (if a AHB/APB bridge is found)
 *
 * \param amba_conf AMBA P&P device info is placed here.
 * \param ioarea address of AMBA Plug&Play information,
 *             on LEON3 systems default is 0xfff00000
 * \param mmaps Memory mmap specific to this amba bus, 
 *              if NULL no translation will be made (default).
 *              A array of maps, ending with a entry with size=0.
 */
void amba_scan (amba_confarea_type * amba_conf, unsigned int ioarea,
                struct amba_mmap *mmaps);

/* Print AMBA Plug&Play info on terminal */
void amba_print_conf (amba_confarea_type * amba_conf);




/***** APB SLAVES *****/

/* Return number of APB Slave devices which has given vendor and device */
int amba_get_number_apbslv_devices (amba_confarea_type * amba_conf, int vendor,
                                    int device);

/* Get First APB Slave device of this vendor&device id */
int amba_find_apbslv (amba_confarea_type * amba_conf, int vendor, int device,
                      amba_apb_device * dev);

/* Get APB Slave device of this vendor&device id. (setting nr to 0 is eqivalent to calling amba_find_apbslv() ) */
int amba_find_next_apbslv (amba_confarea_type * amba_conf, int vendor,
                           int device, amba_apb_device * dev, int index);

/* Get first nr APB Slave devices, put them into dev (which is an array of nr length) */
int amba_find_apbslvs (amba_confarea_type * amba_conf, int vendor, int device,
                       amba_apb_device * devs, int maxno);



/***** AHB SLAVES *****/

/* Return number of AHB Slave devices which has given vendor and device */
int amba_get_number_ahbslv_devices (amba_confarea_type * amba_conf, int vendor,
                                    int device);

/* Get First AHB Slave device of this vendor&device id */
int amba_find_ahbslv (amba_confarea_type * amba_conf, int vendor, int device,
                      amba_ahb_device * dev);

/* Get AHB Slave device of this vendor&device id. (setting nr to 0 is eqivalent to calling amba_find_ahbslv() ) */
int amba_find_next_ahbslv (amba_confarea_type * amba_conf, int vendor,
                           int device, amba_ahb_device * dev, int index);

/* Get first nr AHB Slave devices, put them into dev (which is an array of nr length) */
int amba_find_ahbslvs (amba_confarea_type * amba_conf, int vendor, int device,
                       amba_ahb_device * devs, int maxno);



/***** AHB MASTERS *****/

/* Return number of AHB Master devices which has given vendor and device */
int amba_get_number_ahbmst_devices (amba_confarea_type * amba_conf, int vendor,
                                    int device);

/* Get First AHB Master device of this vendor&device id */
int amba_find_ahbmst (amba_confarea_type * amba_conf, int vendor, int device,
                      amba_ahb_device * dev);

/* Get AHB Master device of this vendor&device id. (setting nr to 0 is eqivalent to calling amba_find_ahbmst() ) */
int amba_find_next_ahbmst (amba_confarea_type * amba_conf, int vendor,
                           int device, amba_ahb_device * dev, int index);

/* Get first nr AHB Master devices, put them into dev (which is an array of nr length) */
int amba_find_ahbmsts (amba_confarea_type * amba_conf, int vendor, int device,
                       amba_ahb_device * devs, int maxno);

#ifdef __cplusplus
}
#endif

#endif
