#! /usr/bin/env python
# ----------------------------------------------------------------------------
# GAISLER_LICENSE
#-----------------------------------------------------------------------------
# File:        grscrub_ctrl.tcl
# Author:      Adria Barros de Oliveira - Cobham Gaisler AB
# Description: Python script to control the FPGA error injection campaign
#
# DISCLAIMER:
# THIS CODE, AND ALL ACCOMPANYING FILES, DATA AND MATERIALS,
# ARE DISTRIBUTED "AS IS" AND WITH NO WARRANTIES OF ANY KIND,
# WHETHER EXPRESS OR IMPLIED. Good data processing procedure dictates
# that any program be thoroughly tested with non-critical data
# before relying on it. The user must assume the entire risk of
# using the program. THIS DISCLAIMER OF WARRANTY CONSTITUTES AN
# ESSENTIAL PART OF THE AGREEMENT. 
#-----------------------------------------------------------------------------


from __future__ import print_function
import os
import time
import serial
import io
from datetime import datetime
import logging
import struct
from array import *
from subprocess import Popen
import random
from random import randrange, uniform
import sys   
import ast   


##################################################
# TO BE CONFIGURED: UART port parameters
##################################################

port_inj  = '/dev/ttyUSB5' # serial port
baud_inj  = 57600          # baud rate for serial port
uart_dev_timeout = 2       # timeout in sec

##################################################
# Target FPGA data
##################################################

map_file = "./DUT/ku060_map/ku060_mapdata.txt"

##################################################
# Scripts arguments
##################################################

# NOTE: if log file "log_name.csv" does not exist, the script creates a new log file, 
#otherwise just attach data at the end 

# Arguments:
RUN_ID_ARGV = 1
NUM_FI_ARGV = 2
LOG_ARGV    = 3
SEQ_ARGV    = 4
FID_ARGV    = 5
BITPOS_ARGV = 6 
if len(sys.argv) < 5 :
   print ("USAGE: python fpgaei_campaign.py RUN_ID number_of_inject_faults log_name SEQ[0-random;1-sequential] frame_id[optional] bit_pos[optional]")
   sys.exit()

RUN_ID = sys.argv[RUN_ID_ARGV]
number_of_inject_faults = int(sys.argv[NUM_FI_ARGV])
seq = int(sys.argv[SEQ_ARGV])

##################################################
# UART control
##################################################

inj_serial = serial.Serial(port=port_inj, baudrate=baud_inj, bytesize=serial.EIGHTBITS, parity=serial.PARITY_NONE, stopbits=serial.STOPBITS_ONE, timeout=uart_dev_timeout,xonxoff = False, rtscts = False, dsrdtr = False, writeTimeout=uart_dev_timeout)

def flush_uart(inj_serial):
   if inj_serial.isOpen():
      try:
         inj_serial.flushInput()
         inj_serial.flushOutput()
      except e:
         print("".join(["[ABORT] Error UART serial port. Error: ", str(e)]))
         sys.exit(0)

flush_uart(inj_serial)

##################################################
# Log file
##################################################

def log_config(log_name):
   global RUN_ID
   new_file = False
   if os.path.exists(''.join([log_name, '.csv']))==False:
      new_file = True
   debug_log = setup_custom_logger(log_name)
   if new_file:
      # create header
      debug_log.info("".join(["FI id, Fault number, Frame id, Frame addr, Bit pos, FI result"]))
   return debug_log

# set logger
def setup_custom_logger(name):
   formatter = logging.Formatter(fmt='%(asctime)s, %(message)s', datefmt='%Y-%m-%d %H:%M:%S')
   handler = logging.FileHandler(''.join([name, '.csv']), mode='a')
   handler.setFormatter(formatter)
   screen_handler = logging.StreamHandler(stream=sys.stdout)
   screen_handler.setFormatter(formatter)
   logger = logging.getLogger(name)
   logger.setLevel(logging.DEBUG)
   logger.addHandler(handler)
   screen_handler.level = logging.INFO
   logger.addHandler(screen_handler)
   return logger

def write_logs( str ):
   print ( str,end='\n' )
   return

def write_debug(fi_num, frame_id, frame_addr, bit_pos, result):
   global debug_log
   global RUN_ID
   debug_log.info("".join(["fi_id_", RUN_ID,", ", str(fi_num),", ", str(frame_id), ", ", str(frame_addr), ", ", str(bit_pos), ", ", str(result)]))


log_name = str(sys.argv[LOG_ARGV])
debug_log = log_config(log_name)


##############################################
# Defining area of injection and frame address
##############################################

#range to inject ku060
min_frame_id = 0
max_frame_id = 37498
min_bit_pos  = 0
max_bit_pos  = 3935

curr_frame_addr = 0

#initial bit_position
curr_bit_pos = 0

#open file with the mapped frames of KU060
ku060_mapdata = open(map_file, "r") 


def SetFrameAddr( frameid ):
   global curr_frame_addr

   ku060_mapdata.seek(9*frameid)
   curr_frame_addr = ku060_mapdata.read(8)

   #Check conversion char to int
   #The charactere must be send as char (or string) through UART in hex format.
   #However, to decode to the ASCII int number in the vhdl design, it is 
   #required to consider char position in the ASCII table.
   frame_addr_string = list(str(curr_frame_addr))
   for i in range(8):
      if int(frame_addr_string[i], 16) > 9:
         frame_addr_string[i] = chr(int(frame_addr_string[i], 16)+48)

   return frame_addr_string


def SetFaultyBit( bitpos ):

   bit_pos_string = str("%x" % bitpos)

   #3 bytes must be sent
   #add zeros to complete the lsb bytes if needed
   while(len(bit_pos_string) < 3):
      bit_pos_string = "0" + bit_pos_string

   #The charactere must be send as char (or string) through UART in hex format.
   #However, to decode to the ASCII int number in the vhdl design, it is 
   #required to consider char position in the ASCII table.
   bit_pos_string = list(str(bit_pos_string))
   for i in range(3):
      if int(bit_pos_string[i], 16) > 9:
         bit_pos_string[i] = chr(int(bit_pos_string[i], 16)+48)

   return bit_pos_string


timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
write_logs( "[INFO] Starting Fault Injection Campaign " + RUN_ID + " at " + timestamp ) # write line of text to file and print to screen

end_condition = 0
repair = 0

num_inj = 0

# frame id
if len(sys.argv) > FID_ARGV :
   #First injection frame fixed
   frame_id = int(sys.argv[FID_ARGV])
else:
   #First injection is always frame 0
   #frame_id = 0
   #First injection random
   frame_id = randrange(min_frame_id, max_frame_id)

# bit pos
if len(sys.argv) > BITPOS_ARGV :
   #First injection bit pos fixed
   curr_bit_pos = int(sys.argv[BITPOS_ARGV])
else:
   #First injection is always bit 0
   #curr_bit_pos = 0
   #First injection random
   curr_bit_pos = (randrange(min_bit_pos, max_bit_pos))

send_frame_addr_string = SetFrameAddr(frame_id)
send_bit_pos_string = SetFaultyBit(curr_bit_pos)


###################################################
## Description of the while loop:
## Injection in the defined CRAM area 
## while (endCondition = false)
##   1. Send the 0x55 byte to sinalize the of start injection
##   2. Send the frame address
##   3. Send the bit postion
##   4. Wait for result of injection
##   5. Write log
##   6. Update to the next frame address and bit position according to the configuration
##   7. Evaluate if endCondition is true
###################################################
   
while(end_condition == 0): 
   flush_uart(inj_serial)

   #send the 0x55 byte through serial to start injection
   sed_byte = 'U'
   inj_serial.write(sed_byte.encode("utf-8")) # send character 'U' 0x55 to serial injector 
   read_byte = inj_serial.read(1)
   read_byte_str = str(read_byte)
   flush_uart(inj_serial)

   #send frame address
   inj_serial.write(send_frame_addr_string)
   read_byte = inj_serial.read(8)
   read_byte_str = str(read_byte)
   flush_uart(inj_serial)

   #send bit pos
   inj_serial.write(send_bit_pos_string)
   read_byte = inj_serial.read(3)
   read_byte_str = str(read_byte)
   
   read_byte = 0

   #read inj sync pckt 
   for i in range(12):
      read_byte = inj_serial.read(1)
      read_byte_str = str(read_byte)

   # read inj result
   read_byte = inj_serial.read(1)
   read_byte_str = str(read_byte)

   if (len(read_byte_str) == 0):
      print("".join(["ERROR - UART serial timeout"]))
      continue

   if read_byte_str == "S":
      num_inj = num_inj + 1

   write_debug("#" + str(num_inj), frame_id, curr_frame_addr, curr_bit_pos, read_byte_str)

   #read the last character meaning the injector has returned to idle
   read_byte = inj_serial.read(1)

   if (num_inj < number_of_inject_faults):

      if seq == 1:
         #Sequential frame injection starting from the configured frame and bit_pos

         if curr_bit_pos < max_bit_pos:
            curr_bit_pos = curr_bit_pos + 1
         else:
            #return to first bit
            curr_bit_pos = 0 
            # end_condition = 1
            if frame_id < max_frame_id:
               frame_id = frame_id + 1
            else:
               #return to first frame
               frame_id = 0 

      else:
         #select the next injection position randonmly
         
         #random selection of frame addr
         frame_id = randrange(min_frame_id, max_frame_id)

         #random selection of bit in the frame
         curr_bit_pos = (randrange(min_bit_pos, max_bit_pos))

         
      send_frame_addr_string = SetFrameAddr(frame_id)
      send_bit_pos_string = SetFaultyBit(curr_bit_pos)

   else:
      end_condition = 1
      

write_logs( "[INFO] Number of injected faults: " + str(num_inj))
timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
write_logs( "[INFO] Ending Fault Injection Campaign at " + timestamp ) 
inj_serial.close()
ku060_mapdata.close()












