# GAISLER_LICENSE
#-----------------------------------------------------------------------------
# File:        fpgaei_ctrl.tcl
# Author:      Adria Barros de Oliveira - Cobham Gaisler AB
# Description: Script to control fault injection experiments
# 
# DISCLAIMER:
# THIS CODE, AND ALL ACCOMPANYING FILES, DATA AND MATERIALS,
# ARE DISTRIBUTED "AS IS" AND WITH NO WARRANTIES OF ANY KIND,
# WHETHER EXPRESS OR IMPLIED. Good data processing procedure dictates
# that any program be thoroughly tested with non-critical data
# before relying on it. The user must assume the entire risk of
# using the program. THIS DISCLAIMER OF WARRANTY CONSTITUTES AN
# ESSENTIAL PART OF THE AGREEMENT. 
#-----------------------------------------------------------------------------


namespace eval ei {
    fconfigure stdin -blocking 0

    # regs from GRSCRUB 
    catch {unset REG}
    
    # EI python script
    set ei_script ./scripts_eifw/fpgaei_campaign.py

    # log files
    set ei_log ""
    set ei_log_name ""

    ### Initial configuration

    proc ei_config { {ci_check 0} {eilog ""}} {
        variable ei_log
        variable ei_log_name

        set ei_log $eilog
        if {$ei_log eq ""} {
            # open a new ei main log
            set ei_log [log log_open "eictrl" $ei_log]
            log log_puts $ei_log [format "ei - ei_config - init configuration"]
        }
        # set log for error injection only
        set systemTime [clock seconds]
        set extension [clock format $systemTime -format "%Y_%b_%d_%H_%M_%S"]
        set ei_log_name [format "./logs/subrun%s/log_ei_data_%s.log" [gr grunid] $extension]
        log log_puts $ei_log [format "ei - ei_config - EI data log name - %s" $ei_log_name]

        if {![grscrub config $ci_check]} {
            log log_puts $ei_log [format "ei - ei_config - Error in the GRSCRUB configuration!!!"]
            return 0
        }

    }

    ####################################################################################
    ### EI for testing procedures
    ####################################################################################

    # random only
    proc ei_random { {current_run 1} {ei_faults 1} } {
		variable ei_script
        variable ei_log
        variable ei_log_name

        log log_puts $ei_log [format "ei - ei_random - EI id %s - num faults %d" $current_run $ei_faults]

        # "USAGE: fpgaei_campaign.py RUN_ID number_of_inject_faults log_folder_path SEQ[0-random;1-sequential] frame_id[optional] bit_pos[optional]"
        set output [exec $ei_script $current_run $ei_faults $ei_log_name 0]
    	log log_puts $ei_log [format "%s" $output]
        log log_puts $ei_log [format "ei - ei_random - End of EI"]
    }

    # deterministic injection
    proc ei_deterministic { {current_run 1} {ei_faults 1} {frame_id 0} {bit_pos 0}} {
		variable ei_script
        variable ei_log
        variable ei_log_name

        log log_puts $ei_log [format "ei - ei_deterministic - EI id %s - num faults %d" 1 $ei_faults]

        # "USAGE: fpgaei_campaign.py RUN_ID number_of_inject_faults log_folder_path SEQ[0-random;1-sequential] frame_id[optional] bit_pos[optional]"
        set output [exec $ei_script $current_run $ei_faults $ei_log_name 1 $frame_id $bit_pos]
    	log log_puts $ei_log [format "%s" $output]
        log log_puts $ei_log [format "ei - ei_deterministic - End of EI"]
    }

    ####################################################################################
    ### EI campaign procedures
    ####################################################################################

    # Example of random injection combined with GRSCRUB readback FFC+CRC
    # num_run: number of executions
    # ei_faults: number of errors injected per execution
    proc ei_run {{num_run 1} {ei_faults 1} {corr_type "all"}} {
        variable ei_log
        set current_run 1
        set stdinx ""
        
        log log_puts $ei_log [format "ei - ei_ctrl_rbk_campaign - Starting Error Injection Campaign"]

        for {set current_run 1} {($current_run <= $num_run) && ($grmon::interrupt != 1) && ($stdinx != "x")} {incr current_run} {
            log log_puts $ei_log [format "#############################################################"]
            ei_random $current_run $ei_faults
            # run GRSCRUB to fix the fault
            grscrub readback_corr $corr_type
            ## change to blind scrubbbing if needed
            # grscrub blindscrubbing
            # grscrub readback_detec all
            set stdinx [gets stdin]
        }

    }

    
    ####################################################################################
    ### Read and Write registers procedures
    ####################################################################################

    # Write a register. Takes a Register name from REG array, and 32-bit value
    proc reg_write {reg val} \
    {
      silent wmem $reg $val

      return 0
    }

    # Read a register. Takes a Register name from REG array
    proc reg_read {reg} \
    {
      
      set val [silent mem $reg 4]
      
      return $val
    }

    ####################################################################################
    ### Misc
    ####################################################################################

    proc ei_get_log_type {} {
      variable ei_log
      return $ei_log
    }

    namespace ensemble create -map {
        {config}        {ei_config}
        {rand}          {ei_random}
        {det}           {ei_deterministic}
        {run}           {ei_run}
        {get_log_type}  {ei_get_log_type}
    }
}


# ================================================================================================
# main

set script_file_name [info script]
puts $script_file_name
