# ================================================================================================
# Copyright (C) 2024, Frontgrade Gaisler AB - All rights reserved
#
# Filename: gr-log.tcl
#
# DISCLAIMER:
# THIS CODE, AND ALL ACCOMPANYING FILES, DATA AND MATERIALS,
# ARE DISTRIBUTED "AS IS" AND WITH NO WARRANTIES OF ANY KIND,
# WHETHER EXPRESS OR IMPLIED. Good data processing procedure dictates
# that any program be thoroughly tested with non-critical data
# before relying on it. The user must assume the entire risk of
# using the program. THIS DISCLAIMER OF WARRANTY CONSTITUTES AN
# ESSENTIAL PART OF THE AGREEMENT. 
# ================================================================================================

namespace eval log {

    #### clear old variables
    catch {unset ECHO}

    # debug flags
    set ECHO   [gr echo]

    #### file-handling procs #############################################################################

    # open log files
    proc log_open { {logtype} {logfile} } {
        variable script_file_name
        variable log_name ""
        set log_file $logfile
		
		if {$log_file eq ""} {		
		    set systemTime [clock seconds]
			set extension [clock format $systemTime -format "%Y-%m-%d_%H-%M-%S"]
			set log_name [format "./logs/subrun%s/log_gr716b_%s_%s.log" [gr grunid] $logtype $extension]
			set log_file [::open $log_name "w"]

            if {$logtype eq "sys" || $logtype eq "fictrl"} {
                log_puts $log_file "log - log_open - ##################################################################"
                log_puts $log_file [format "log - log_open - test subrun - %s" [gr grunid]]
                log_puts $log_file [format "log - log_open - log file created - %s" $log_name]
                log_puts $log_file [format "log - log_open - grmon version - %s" $::grmon_version]
                log_puts $log_file [format "log - log_open - system time - %s" $extension]
                log_puts $log_file "log - log_open - ##################################################################"
            }
		} else {
            puts "WARINING: log already open."
        }
        return $log_file
    }

    # close log files
    proc log_close { {logfile} {silent 0}} {     
        set log_file $logfile
        if {$log_file ne ""} {
            if {!$silent} {
                log_puts $log_file [format "log - log_close - log file closed - %s" $log_file]
            }                
            ::close $log_file
            set log_file ""
        }
        return $log_file
    }

    ### log data procs ###

    proc log_puts { {log_file} {args} } {
        variable ECHO
        
        if {$ECHO} {
            puts -nonewline "\t"
            set systemTime [clock milliseconds]
            puts -nonewline [format "%s.%03d " [clock format [expr {$systemTime/1000}] -format "%H:%M:%S"] [expr {$systemTime % 1000}]]
            puts {*}$args
        }
        if {$log_file ne ""} {
            log_w $log_file {*}$args
            flush $log_file
        }
    }

    proc log_puts_silent { {log_file} {args} } {
        if {$log_file ne ""} {
            log_w $log_file {*}$args
        }
    }
    
    variable log_ts 0
    proc log_w { {log_file} {args} } {
        variable log_ts 
        
        if {$log_file ne ""} {
            if {$log_ts == 0} {
                set systemTime [clock milliseconds]
                puts -nonewline $log_file [format "%s.%03d, " [clock format [expr {$systemTime/1000}] -format "%Y-%m-%d %H:%M:%S"] [expr {$systemTime % 1000}]]
                set log_ts 1
            }
            if {[llength $args] == 1} {
                puts $log_file [lindex $args 0]
                set log_ts 0
            } else {
                puts [lindex $args 0] $log_file [lindex $args 1]
            }
            flush $log_file
        }
    }

    proc log_puts_list { inlist } {
        log_puts [format "log - printlist - "]
        foreach item $inlist {
            log_puts -nonewline $item
        }
    }

    proc spinner {} {
        global spinnerIdx
        if {[incr spinnerIdx] > 3} {
            set spinnerIdx 0
        }
        set spinnerChars {/ - \\ |}
        puts -nonewline "\r[lindex $spinnerChars $spinnerIdx]"
        flush stdout
    }

    namespace ensemble create -map {
        {log_open}     {log_open}
        {log_close}    {log_close}
        {log_puts}     {log_puts}
        {log_silent}   {log_puts_silent}
        {log_list}     {log_puts_list}
        {spinner}      {spinner}
    }
}

# ================================================================================================
# main

set script_file_name [info script]
puts $script_file_name
