# ================================================================================================
# Copyright (C) 2024, Frontgrade Gaisler AB - All rights reserved
#
# Author: Adria Oliveira - Frontgrade Gaisler AB
# Filename: grscrub-ctrl.tcl
# Description: Tcl functions to configure the GR716B/GRSCRUB using GRMON4
# Note that memory addresses may change depending on the setup. Therefore,
# this script should be updated accordingly.
#
# DISCLAIMER:
# THIS CODE, AND ALL ACCOMPANYING FILES, DATA AND MATERIALS,
# ARE DISTRIBUTED "AS IS" AND WITH NO WARRANTIES OF ANY KIND,
# WHETHER EXPRESS OR IMPLIED. Good data processing procedure dictates
# that any program be thoroughly tested with non-critical data
# before relying on it. The user must assume the entire risk of
# using the program. THIS DISCLAIMER OF WARRANTY CONSTITUTES AN
# ESSENTIAL PART OF THE AGREEMENT. 
# ================================================================================================


#####################################################################################
### GRSCRUB CONFIGURATION
#####################################################################################

namespace eval grscrub {

    # Clear old variables
    catch {unset REG}
    catch {unset bitfolder}

    # Component specifics
    set COMPONENT "GR716B"
    set COMP_GRSCRUB_REGADDR 0x80404000
    set COMP_MEMADDR 0x10D00000 ; # golden memory address
    set COMP_RAMADDR 0x30000000 ; # on-chip RAM for validation purposes
    set COMP_SPIM 1 ; # select memory type: (1) SPI Flash; (0) RAM

    # Initialize variables
    variable REG

    # GRSCRUB registers start address
    set GRSCRUB_REGADDR $COMP_GRSCRUB_REGADDR

    # Initialize register offsets for GRSCRUB, based on GRSCRUB_REGADDR
    array set REG {
        GRSCRUB.STAT        0x00000000 
        GRSCRUB.CONFIG      0x00000004
        GRSCRUB.IDCODE      0x00000008
        GRSCRUB.DELAY       0x0000000C 
        GRSCRUB.FCR         0x00000010
        GRSCRUB.LFAR        0x00000014  
        GRSCRUB.LGBAR       0x00000018
        GRSCRUB.HGBAR       0x0000001C
        GRSCRUB.LGSFAR      0x00000020
        GRSCRUB.LMASKAR     0x00000024
        GRSCRUB.LFMAPR      0x00000028
        GRSCRUB.LGCRCAR     0x0000002C
        GRSCRUB.LGRBKAR     0x00000030
        GRSCRUB.ECNT        0x00000034
        GRSCRUB.SETUP       0x00000038 
        GRSCRUB.CAP         0x0000003C
        GRSCRUB.FRAMEID     0x00000040
        GRSCRUB.ERRFRAMEID  0x00000044
    }

    # Status register bits (point to start bit)
    array set STATBITS {
      STAT.STATE    0
      STAT.OPDONE   4
      STAT.SCRERR   5
      STAT.ERRID    6
      STAT.HOLD     11
      STAT.SCRUND   12
    }

    # Setup register bits (point to start bit)
    array set SETUPBITS {
      SETUP.ROWBND    4
      SETUP.TPROG     12
      SETUP.CISEL     20
      SETUP.BITSWPEN  21
      SETUP.BLKFRAME  22
    }

    array set ERRCODE {
      ERRID.NOERROR 0x0
      ERRID.PROG    0x1
      ERRID.UNCBIT  0x2
      ERRID.DMAR    0x3
      ERRID.DMAW    0x4
      ERRID.RBK     0x5
      ERRID.FRAMEW  0x7
    }

    # SelectMap commands
    array set SMAPREG {
        SMAP.DUMMY          0xFFFFFFFF 
        SMAP.WFDR           0x30004000
        SMAP.WCMD           0x30008001
        SMAP.NOOP           0x20000000
        SMAP.SYNC           0xAA995566
        SMAP.DESYNC         0x0000000D
    }


    # Choose correct patch to golden bitfiles.
    variable bitfolder ./DUT


    ### Golden memory address definition ###

    # Golden memory base address
    set MEM_BASE $COMP_MEMADDR

    # Set memory addresses #
    # All default values

    # load configuration bitstream (generated by the synthesis tool)
    set BITPARAMS(LOADAD.BIT)      [expr $MEM_BASE + 0x00000006]

    # load mask data (generated by the synthesis tool)
    set BITPARAMS(LOADAD.MSK)      [expr $MEM_BASE + 0x01800006]

    # Address for GRSCRUB to store the readback data (optional - MEM WRITTING NOT USED WITH SPI FLASH MEM)
    set BITPARAMS(LOADAD.RDBK)     $COMP_RAMADDR ; # using fixed on-chip RAM for test purposes

    # Address for GRSCRUB to read the frame mapped addresses (required for readback mode or any scrubbing mode with BLKFRAME>0).
    set BITPARAMS(LOADAD.MAP)      [expr $MEM_BASE + 0x03000000]

    # Address for GRSCRUB to read the golden CRC data (required for readback mode with CRC check).
    # The GRSCRUB might also store the golden CRC data. (optional - MEM WRITTING NOT USED WITH SPI FLASH MEM)
    set BITPARAMS(LOADAD.CRC)      [expr $MEM_BASE + 0x03200000]
    

    # Define the start addresses in the Golden memory #
    # All default values
    # Use the initialization procedures for adjusting addresses to current bitstream 
    set AUTO_FIX_MEM 1

    # Start address of the configuration bitstream in the Golden memory. 
    # This must be the address of the first dummy word in the configuration bitstream (0xFFFFFFFF), after the initial header.
    # All configuration bitstreams have an initial header with ASCII characters that provides some file information,
    # which is not required to program the FPGA. The synchronization phase starts at the first dummy word (0xFFFFFFFF).
    # The LGBAR register is set with this address.
    set BITPARAMS(START.BIT)       [expr $MEM_BASE + 0x00000090] 

    # Address of the first configuration bitstream frame in the Golden memory
    # The LGSFAR register is set with this address.
    # Note: this is the address of first frame to be scrubbed, here is set to frame 0
    # One should update this address if needed
    set BITPARAMS(START.GOLD)      [expr $MEM_BASE + 0x000001a8]

    # Set the highest configuration bitstream address in the Golden memory
    # The HGBAR register is set with this address.
    set BITPARAMS(END.BIT)         [expr $MEM_BASE + 0x01701e78] 

    # Address of the first mask data related with the first configuration bitstream frame in the Golden memory.
    # The LMASKAR register is set with this address.
    set BITPARAMS(START.MSK)       [expr $MEM_BASE + 0x018001a8] 
  

    ### Other configurations ###

    # Total number of configuration frames of KU060
    set FCNT 49030
   
    # Frame length of KU060
    set FLEN 123 

    # Number of mapped frames (below is set for all CRAM frames)
    # (Only mapped frames can be scrubbed)
    set NUM_SCRUB_FRAMES 37498

    # address of the first FPGA frame to be scrubbed
    set FSTART_ADDR 0x0

    # Define periodic scrubbing runs
    # periodic = 1
    # one time = 0
    set scrun 0

    # clear error counters at every scrubbing cycle
    # useafull in periodic scrubbing
    set clrcnten 1

    # Opdone bitfield position on Status register
    set done 0x10

    # set block frame
    set SET_BLKFRAME 1

    # KU060 FPGA IDCODE
    set FPGA_IDCODE 0x03919093

    variable grlog ""


    ####################################################################################
    ### Log procedures
    ####################################################################################

    proc log_puts { {args} } {
        variable grlog
        set grlog [ei get_log_type]
        log log_puts $grlog {*}$args
    }

    proc log_puts_silent { {args} } {
        variable grlog
        set grlog [ei get_log_type]
        log log_silent $grlog {*}$args
    }

    ####################################################################################
    ### Initialization procedures
    ####################################################################################

    # Initial configuration specific to the current component
    proc component_config { {comp ""} } {

      if {$comp == "GR716B"} {
        ## Config IO matrix
        # set GRSCRUB GPIOs in SYS.CFG.GP3
        set cfg_gp_reg [silent mem 0x8000D00C 4]
        set cfg_gp_reg [expr $cfg_gp_reg & ~0xFFFFFFF0]
        set cfg_gp_reg [expr $cfg_gp_reg | 0xEEEEEEE0]
        silent wmem 0x8000D00C $cfg_gp_reg
        set cfg_gp_reg [silent mem 0x8000D00C 4]
        log_puts_silent [format "grscrub - component_config - GR716B SYS.CFG.GP3 configured for GRSCRUB GPIOs - 0x%08x" $cfg_gp_reg]
        # set GRSCRUB GPIOs in SYS.CFG.GP4
        set cfg_gp_reg [silent mem 0x8000D010 4]
        set cfg_gp_reg [expr $cfg_gp_reg & ~0x0FFFFFFF]
        set cfg_gp_reg [expr $cfg_gp_reg | 0x0EEEEEEE]
        silent wmem 0x8000D010 $cfg_gp_reg
        set cfg_gp_reg [silent mem 0x8000D010 4]
        log_puts_silent [format "grscrub - component_config - GR716B SYS.CFG.GP4 configured for GRSCRUB GPIOs - 0x%08x" $cfg_gp_reg]

        ## Set GPIOs no PULL
        silent wmem 0x8000D028 0x0
        silent wmem 0x8000D02C 0x0
        
        # Clock gates
        # enable SPIM0
        grcg enable 5 grcg0
        set cg_reg [silent mem 0x80006004 4]
        log_puts_silent [format "grscrub - component_config - GR716B SPIM0 core enabled - grcg0 - 0x%08x" $cg_reg]
        # enable GRSCRUB
        grcg enable 22 grcg1
        set cg_reg [silent mem 0x80007004 4]
        log_puts_silent [format "grscrub - component_config - GR716B GRSCRUB core enabled - grcg1- 0x%08x" $cg_reg]

        ## SCRUBBER_CLK config (SMAP CCLK)
        silent wmem 0x8010D010 0x04
        set cg_reg [silent mem 0x8010D010 4]
        log_puts_silent [format "grscrub - component_config - GR716B SMAP CCLK configured - SCLKREF - 0x%08x" $cg_reg]

        ## SPIM0 config
        # force 4 byte address mode (F4B)
        set spim_reg [silent mem 0xfff00100 4]
        silent wmem 0xfff00100 [expr $spim_reg | [expr 1 << 12]]
        # use 4 address bytes (ADDRBYTES)
        set spim_reg [silent mem 0xfff00100 4]
        silent wmem 0xfff00100 [expr $spim_reg & ~[expr 3 << 8]]
        # enable alternate scaler (EAS)
        set spim_reg [silent mem 0xfff00104 4]
        silent wmem 0xfff00104 [expr $spim_reg | [expr 1 << 2]]
        log_puts_silent [format "grscrub - component_config - GR716B SPIM0 configured"]

        # ====================
        # DEBUG OPTIONS (can be removed if no debug is needed)

        #enable test clock
        silent wmem 0x8010D018 0x13
        set cg_reg [silent mem 0x8010D018 4]
        log_puts_silent [format "grscrub - component_config - GR716B test clock enabled - TCTRL - 0x%08x" $cg_reg]
        
        # set test clock GPIOs in SYS.CFG.GP2
        set cfg_gp_reg [silent mem 0x8000D008 4]
        set cfg_gp_reg [expr $cfg_gp_reg & ~0xF00F0000]
        set cfg_gp_reg [expr $cfg_gp_reg | 0xE00E0000]
        silent wmem 0x8000D008 $cfg_gp_reg
        set cfg_gp_reg [silent mem 0x8000D008 4]
        log_puts_silent [format "grscrub - component_config - GR716B SYS.CFG.GP2 configured for test clock GPIOs - 0x%08x" $cfg_gp_reg]
        
        # Configure GR716B AMBA bus to allow GRSCRUB access the LEON3 on-chip mem (this is used for debugging only)
        silent wmem 0x8000E000 0x00600807
        # ====================
        log_puts [format "GR716B-GRSCRUB configured"]
      }
    }

    # Initial configuration
    proc init_config { } {
      variable REG
      variable COMPONENT
      variable AUTO_FIX_MEM
      variable COMP_SPIM

      log_puts_silent [format "grscrub - init_config - init configuration"]

      component_config $COMPONENT     
      grscrub_disable

      if {$AUTO_FIX_MEM} {
        config_gold_addr
      } 

      # configure setup
      set_default_setup
      grscrub_showregs 1

      # check grscrub is disabled
      set status_reg [reg_read $REG(GRSCRUB.STAT)]
      if {$status_reg} {
        while {$status_reg} {
          log_puts_silent [format "grscrub - init_config - waiting GRSCRUB to be disabled... status register - 0x%08x" $status_reg]
          grscrub_disable
          after 500
          grscrub_doneclear
          grscrub_errorclear
          set status_reg [reg_read $REG(GRSCRUB.STAT) 0]
        }
        log_puts_silent [format "grscrub - init_config - GRSCRUB is disabled... status register - 0x%08x" $status_reg]
      }

      puts "\n"  
      log_puts [format "#############################################################"]      
      # Program the target FPGA
      if {![grscrub_progfpga]} {
        log_puts [format "grscrub - init_config - Error to program the FPGA!!!"]
        return 0
      }
      log_puts [format "#############################################################"]
      
      # set one time execution as default
      grscrub_set_scrun 0
      grscrub_disable

      return 1
    }

    # Set the start address of the configuration bitstream in the Golden memory
    proc set_start_bit {} {
      variable MEM_BASE
      variable BITPARAMS
      variable SMAPREG

      set dummy 0
      set addr_shift $MEM_BASE

      while {($dummy ne $SMAPREG(SMAP.DUMMY)) && ($dummy ne [string tolower $SMAPREG(SMAP.DUMMY)])} {
        set addr_shift [expr $addr_shift +4]
        set dummy [silent mem $addr_shift 4]
      }
      set BITPARAMS(START.BIT) $addr_shift 
      log_puts_silent [format "grscrub - set_start_bit    - Start bitstream address set             - 0x%08x" $BITPARAMS(START.BIT)]
    }

    proc set_default_setup { } {
      variable REG
      variable SETUPBITS
      variable SET_BLKFRAME

      # clean setup register
      reg_write $REG(GRSCRUB.SETUP) 0
      set setup [reg_read $REG(GRSCRUB.SETUP)]

      # ROWBND = 2
      reg_write $REG(GRSCRUB.SETUP) [expr $setup | [expr 2 << $SETUPBITS(SETUP.ROWBND)]]
      set setup [reg_read $REG(GRSCRUB.SETUP)]
      log_puts_silent [format "grscrub - set_default_setup - SETUP ROWBND set   - 0x%08X" $setup]

      # TPROG = 150
      reg_write $REG(GRSCRUB.SETUP) [expr $setup | [expr 150 << $SETUPBITS(SETUP.TPROG)]]
      set setup [reg_read $REG(GRSCRUB.SETUP)]
      log_puts_silent [format "grscrub - set_default_setup - SETUP TPROG set    - 0x%08X" $setup]

      # BITSWPEN = 1
      reg_write $REG(GRSCRUB.SETUP) [expr $setup | [expr 1 << $SETUPBITS(SETUP.BITSWPEN)]]
      set setup [reg_read $REG(GRSCRUB.SETUP)]
      log_puts_silent [format "grscrub - set_default_setup - SETUP BITSWPEN set - 0x%08X" $setup]

      if {$SET_BLKFRAME} {
        reg_write $REG(GRSCRUB.SETUP) [expr $setup | [expr $SET_BLKFRAME << $SETUPBITS(SETUP.BLKFRAME)]]
        set setup [reg_read $REG(GRSCRUB.SETUP)]
        log_puts_silent [format "grscrub - set_default_setup - SETUP BLKFRAME set - 0x%08X" $setup]
      }
    }

    # configure the golden memory addresses based on the current bitstream
    proc config_gold_addr {} {
      variable MEM_BASE
      variable BITPARAMS
      variable SMAPREG
      variable FCNT
      variable FLEN

      # fix start bitstream addr
      set_start_bit

      set misc 0
      set addr_shift $MEM_BASE

      # find WFDR word
      while {$misc ne $SMAPREG(SMAP.WFDR)} {
        set addr_shift [expr $addr_shift + 4]
        set misc [silent mem $addr_shift 4]
      }

      # read number of total words
      set addr_shift [expr $addr_shift + 4]
      set misc [silent mem $addr_shift 4]
      set num_words [expr $misc & 0x07FFFFFF]
      log_puts_silent [format "grscrub - config_gold_addr - Total frame words                       - 0x%08x" $num_words]
      # check if values are correct
      if {$num_words ne [expr $FCNT*$FLEN]} {
        log_puts [format "grscrub - config_gold_addr - +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"]
        log_puts [format "grscrub - config_gold_addr - ATTENTION: NUMBER OF FRAME WORDS DOES NOT MATCH WITH PRE-DEFINED FOR THE FPGA"]
        log_puts [format "grscrub - config_gold_addr - fcnt - " $FCNT]
        log_puts [format "grscrub - config_gold_addr - flen - " $FLEN]
        log_puts [format "grscrub - config_gold_addr - +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"]
        return 0
      }

      # Set the address of the first configuration bitstream frame in the Golden memory
      # Note: this is the address of first frame to be scrubbed, here is set to frame 0
      # One should update this address if needed
      set addr_shift [expr $addr_shift + 4]
      set BITPARAMS(START.GOLD) $addr_shift 
      log_puts_silent [format "grscrub - config_gold_addr - Address of first frame of bitstream set - 0x%08x" $BITPARAMS(START.GOLD)]

      # find DESYNC word
      set noend 1
      set misc 0
      set addr_shift [expr $addr_shift + [expr $num_words * 4]]
      while {$noend} {
        # find WCMD
        while {$misc ne $SMAPREG(SMAP.WCMD)} {
          set addr_shift [expr $addr_shift + 4]
          set misc [silent mem $addr_shift 4]
        }

        # find DESYNC
        set addr_shift [expr $addr_shift + 4]
        set misc [silent mem $addr_shift 4]
        if {($misc eq $SMAPREG(SMAP.DESYNC)) || ($misc eq [string tolower $SMAPREG(SMAP.DESYNC)])} {
          set noend 0
        }
      }

      # find last NOOP word
      set addr_shift [expr $addr_shift + 4]
      set misc [silent mem $addr_shift 4]
      while {$misc eq $SMAPREG(SMAP.NOOP)} {
        set addr_shift [expr $addr_shift + 4]
        set misc [silent mem $addr_shift 4]
      }

      # Set end of the bitstream in memory
      set BITPARAMS(END.BIT) [expr $addr_shift - 4]
      log_puts_silent [format "grscrub - config_gold_addr - Address of last word of bitstream set   - 0x%08x" $BITPARAMS(END.BIT)]

      ## Fix mask data
      # Set the address of the mask data of the first configuration bitstream frame in the Golden memory
      # Note: this is the address of first frame to be scrubbed, here is set to frame 0
      # One should update this address if needed
      set addr_shift [expr $BITPARAMS(START.GOLD) & 0x000FFFFF]
      set BITPARAMS(START.MSK) [expr [expr $BITPARAMS(LOADAD.MSK) & 0xFFF00000] + $addr_shift]
      log_puts_silent [format "grscrub - config_gold_addr - Address of first mask data set          - 0x%08x" $BITPARAMS(START.MSK)]
      
      log_puts_silent [format "grscrub - config_gold_addr - Golden addresses configured."]
    }
    

    ####################################################################################
    ### Read and Write registers procedures
    ####################################################################################

    # Write a register. Takes a Register name from REG array, and 32-bit value
    proc reg_write {reg_shift val} \
    {
      variable GRSCRUB_REGADDR

      set reg [expr $GRSCRUB_REGADDR + $reg_shift]
      silent wmem $reg $val
      log_puts_silent [format "grscrub - reg_write - 0x%08X 0x%08X" $reg $val]
    }

    # Read a register. Takes a register name from REG array
    proc reg_read {{reg_shift} {logen 1}} \
    {
      variable GRSCRUB_REGADDR

      set reg [expr $GRSCRUB_REGADDR + $reg_shift]
      set val [silent mem $reg 4]
      if {$logen} {
        log_puts_silent [format "grscrub - reg_read - 0x%08X 0x%08x" $reg $val]
      }
      
      return $val
    }

    ####################################################################################
    ### GRSCRUB STATUS
    ####################################################################################
    
    # GRSCRUB IP enable
    proc grscrub_enable {} \
    {
      variable REG

      set config_reg [expr ([reg_read $REG(GRSCRUB.CONFIG)])]

      #configuration reg -> en bitfild = 1
      reg_write $REG(GRSCRUB.CONFIG) [expr $config_reg | 0x1]

      log_puts_silent [format "grscrub - grscrub_enable - GRSCRUB ip enabled"]
    }

    # GRSCRUB IP disable
    proc grscrub_disable {} \
    {
      variable REG

      set config_reg [expr ([reg_read $REG(GRSCRUB.CONFIG)])]

      #configuration reg -> en bitfild = 0
      reg_write $REG(GRSCRUB.CONFIG) [expr $config_reg & 0xFFFFFFFE]

      log_puts_silent [format "grscrub - grscrub_disable - GRSCRUB ip disabled"]
    }

    # Clean OPDONE and SCRUND bitfields of Status register
    proc grscrub_doneclear {} \
    {
      variable REG

      set status_reg [expr ([reg_read $REG(GRSCRUB.STAT)])]

      #clear both dones
      reg_write $REG(GRSCRUB.STAT) [expr $status_reg | 0x1010]

      grscrub_checkdoneclear
    }

    # Verify if done bitfiled are clean
    proc grscrub_checkdoneclear {} \
    {
      variable REG
      variable done

      set status_reg [expr ([reg_read $REG(GRSCRUB.STAT)])]
      set donecheck [expr $status_reg & $done]

      if {$donecheck == 0x0} {
        log_puts_silent [format "grscrub - grscrub_checkdoneclear - GRSCRUB done is clean!"]
      } else {
        log_puts_silent [format "grscrub - grscrub_checkdoneclear - GRSCRUB done is NOT clean!"]
      }
    }

    # Show GRSCRUB registers
    proc grscrub_showregs { {silent 0} } \
    {
      variable GRSCRUB_REGADDR

      if {$silent} {
        set regs [silent mem $GRSCRUB_REGADDR 72]
        log_puts_silent [format "grscrub - grscrub_showregs - GRSCRUB registers: %s" $regs]
      } else {
        set regs [mem $GRSCRUB_REGADDR 72]
        log_puts [format "grscrub - grscrub_showregs - GRSCRUB registers: %s" $regs]
      }
    }

    # Clear error counter and frame id registers
    # It should be call between periodic executions to be able to get upsets per run
    proc grscrub_clr_counters {} {
      variable REG
      reg_write $REG(GRSCRUB.ECNT)       0x00000000
      reg_write $REG(GRSCRUB.ERRFRAMEID) 0x00000000
      reg_write $REG(GRSCRUB.FRAMEID)    0x00000000
    }

    # Clean the SCRERR bitfield of Status register
    proc grscrub_errorclear {} \
    {
      variable REG
      variable STATBITS

      set status_reg [expr ([reg_read $REG(GRSCRUB.STAT)])]
      reg_write $REG(GRSCRUB.STAT) [expr $status_reg | [expr 1 << $STATBITS(STAT.SCRERR)]]
      set status_reg [expr ([reg_read $REG(GRSCRUB.STAT)])]
      log_puts_silent [format "grscrub - grscrub_errorclear - GRSCRUB error is clean - 0x%08X" $status_reg]
    }

    proc grscrub_set_scrun { {runtype} } {
      variable scrun
      set scrun $runtype
    }

    ####################################################################################
    ### Programming the target FPGA
    ####################################################################################

    # Configure the GRSCRUB for programming operation mode
    proc grscrub_init_progmode {} \
    {
      variable REG
      variable BITPARAMS
      variable FPGA_IDCODE

      log_puts_silent [format  "GRSCRUB init program mode"]

      #configuration reg -> opmode = 0001
      reg_write $REG(GRSCRUB.CONFIG) 0x00000010

      #golden bitstream addresses
      reg_write $REG(GRSCRUB.LGBAR) $BITPARAMS(START.BIT)
      reg_write $REG(GRSCRUB.HGBAR) $BITPARAMS(END.BIT)     
      
      reg_write $REG(GRSCRUB.IDCODE) $FPGA_IDCODE

      #map addr in the golden memory
      reg_write $REG(GRSCRUB.LFMAPR) $BITPARAMS(LOADAD.MAP)
    }

    # Configure GRSCRUB to program the target FPGA
    proc grscrub_progfpga {} \
    {
      variable REG
      variable done

      log_puts [format  "STARTING FPGA PROGRAMMING MODE"]

      grscrub_disable
      grscrub_doneclear
      grscrub_errorclear
      grscrub_init_progmode

      # wait
      after 100
      set ret 0
      
      grscrub_enable

      set start_time [start_timer]
            
      log_puts [format "Executing FPGA Programming..."]

      # wait OPDONE or SCRERR bitfield of Status register
      while {([expr { $done & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != $done) &&    
             ([expr [reg_read $REG(GRSCRUB.STAT) 0]] != 0x14) &&
             ([expr [reg_read $REG(GRSCRUB.STAT) 0]] != 0x00000060) && 
             ([expr [reg_read $REG(GRSCRUB.STAT) 0]] != 0x80000060) && 
             ($grmon::interrupt != 1)} {
        log spinner
        # wait if not done
        after 100
      }
      set end_time [end_timer] 
            
      # check if programmed successfully 
      if {([expr { 0x00000060 & [expr [reg_read $REG(GRSCRUB.STAT)]]}] != 0x00000060)} {
        log_puts [format "GRSCRUB FPGA programmed successfully!"]
        log_puts [format "FPGA Programming time: %s ms" [expr $end_time-$start_time]]
        set ret 1
      } else {
        log_puts [format  "ERROR to program FPGA!!!"]
        set ret 0
      }

      grscrub_disable
      return $ret       
    }


    ####################################################################################
    ### Readback Scrubbing
    ####################################################################################

    # Configure the GRSCRUB for readback scrubbing operation mode
    proc grscrub_init_readbackmode {} \
    {
      variable REG
      variable BITPARAMS
      variable FCNT 
      variable FLEN
      variable FPGA_IDCODE
      variable NUM_SCRUB_FRAMES
      variable FSTART_ADDR

      log_puts_silent [format "grscrub - grscrub_init_readbackmode - GRSCRUB init readback mode"]

      # clear error counter and frame id registers
      reg_write $REG(GRSCRUB.ECNT)       0x00000000
      reg_write $REG(GRSCRUB.ERRFRAMEID) 0x00000000
      reg_write $REG(GRSCRUB.FRAMEID)    0x00000000

      ########## FPGAEI FRAMEWORK ONLY ########################################
      if {[gr eistat]} {
        # High delay is required during FI since both FI engine and GRSCRUB use the SMAP interface
        reg_write $REG(GRSCRUB.DELAY) 0xFFFFFF
      } else {
        # no delay between executions in periodic scrubbing
        reg_write $REG(GRSCRUB.DELAY) 0x0
      }
      ###############################################################

      #golden bitstream addresses
      reg_write $REG(GRSCRUB.LGBAR) $BITPARAMS(START.BIT)
      reg_write $REG(GRSCRUB.HGBAR) $BITPARAMS(END.BIT)   
      
      ## Full scrubbing of the configuration memory

      #set frame address 0x0 
      reg_write $REG(GRSCRUB.LFAR) $FSTART_ADDR 

      # only mapped frames
      reg_write $REG(GRSCRUB.FCR) [expr [expr $NUM_SCRUB_FRAMES << 9] | [expr $FLEN << 2]]

      #memory addr for write readback data
      reg_write $REG(GRSCRUB.LGRBKAR) $BITPARAMS(LOADAD.RDBK)

      #mask addr
      reg_write $REG(GRSCRUB.LMASKAR) $BITPARAMS(START.MSK)

      #start frame addr
      reg_write $REG(GRSCRUB.LGSFAR) $BITPARAMS(START.GOLD)

      #map addr in the golden memory
      reg_write $REG(GRSCRUB.LFMAPR) $BITPARAMS(LOADAD.MAP)

      reg_write $REG(GRSCRUB.IDCODE) $FPGA_IDCODE

      #required if crc on
      reg_write $REG(GRSCRUB.LGCRCAR) $BITPARAMS(LOADAD.CRC)

    }

    # Configure GRSCRUB to readback only detection  
    proc grscrub_readbackfpga_onlydetection {{datacheck "ffc"}} \
    {
      variable REG
      variable done
      variable scrun

      log_puts [format "Starting readback GRSCRUB IP - only detection"]

      grscrub_disable
      grscrub_doneclear
      grscrub_errorclear

      grscrub_init_readbackmode
      
      # set data verification
      if {$datacheck == "ffc"} {
        reg_write $REG(GRSCRUB.CONFIG) 0x0000102C
        log_puts [format "FFC selected"]
      } elseif  {$datacheck == "crc"} {
        reg_write $REG(GRSCRUB.CONFIG) 0x0000082C
        log_puts [format "CRC selected"]
      } else {
        #all
        reg_write $REG(GRSCRUB.CONFIG) 0x0000182C
        log_puts [format "All methods selected: FFC + CRC"]
      }     

      # if periodic scrubbing
      if {$scrun == 1} {
        set config_reg [expr ([reg_read $REG(GRSCRUB.CONFIG)])]
        #configuration reg -> scrun = 1
        reg_write $REG(GRSCRUB.CONFIG) [expr $config_reg | 0x2]
        log_puts [format "grscrub - grscrub_readbackfpga_onlydetection - Periodic scrubbing enabled"]
      } 

      grscrub_enable
      set start_time [start_timer]

      log_puts [format "Reading FPGA configuration memory..."]
      
      # wait OPDONE or SCRERR bitfield of Status register
      while {([expr { $done & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != $done) && ($grmon::interrupt != 1) &&
             ([expr { 0x00000020 & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != 0x00000020)} {
        # wait if not done
        after 100
      }
      set end_time [end_timer]
                       
      # check if readback successfully 
      if {([expr [reg_read $REG(GRSCRUB.STAT)]] == $done) || 
          ([expr [reg_read $REG(GRSCRUB.STAT)]] == 0x00001010)} {
          set run_error [expr [reg_read $REG(GRSCRUB.ECNT)] & 0x0000FFFF]
          log_puts [format "GRSCRUB FPGA readback successfully"]
          log_puts [format "GRSCRUB Last readback mismatches: $run_error"]
          log_puts [format "FPGA readback time: %s ms" [expr $end_time-$start_time]]
      } else {
          log_puts [format "ERROR to readback FPGA!!!"]
      }

      grscrub_disable
    }

    # Configure GRSCRUB to readback detection and correction 
    # it only enables the readback, it does not monitor here
    proc grscrub_set_readbackfpga_correction {{datacheck "ffc"}} {
      variable REG
      variable scrun

      log_puts_silent [format "grscrub - grscrub_set_readbackfpga_correction - Starting GRSCRUB readback correction %s" $datacheck]

      grscrub_disable
      grscrub_doneclear
      grscrub_errorclear

      grscrub_init_readbackmode
     
      #set data verification
      if {$datacheck == "ffc"} {
        reg_write $REG(GRSCRUB.CONFIG) 0x00001024
        log_puts_silent [format "grscrub - grscrub_set_readbackfpga_correction - FFC selected"]
      } elseif  {$datacheck == "crc"} {
        reg_write $REG(GRSCRUB.CONFIG) 0x00000824
        log_puts_silent [format "grscrub - grscrub_set_readbackfpga_correction - CRC selected"]
      } else {
        #all
        reg_write $REG(GRSCRUB.CONFIG) 0x00001824
        log_puts_silent [format "grscrub - grscrub_set_readbackfpga_correction - All methods selected: FFC + CRC"]
      }

      # if periodic scrubbing
      if {$scrun == 1} {
        set config_reg [expr ([reg_read $REG(GRSCRUB.CONFIG)])]
        #configuration reg -> scrun = 1
        reg_write $REG(GRSCRUB.CONFIG) [expr $config_reg | 0x2]
        log_puts_silent [format "grscrub - grscrub_set_readbackfpga_correction - Periodic scrubbing enabled"]
      } 

      grscrub_enable
      log_puts [format "grscrub - grscrub_set_readbackfpga_correction - Readback is enabled"]
    }

    # Configure GRSCRUB to readback detection and correction 
    # Stays monitoring continually
    # Otherwise, use grscrub_set_readbackfpga_correction for only enabling it and do the monitoring somewhere else
    proc grscrub_readbackfpga_correction {{datacheck "ffc"}} \
    {
      variable REG
      variable done
      variable scrun

      log_puts [format "    GRSCRUB readback correction"]

      grscrub_disable
      grscrub_doneclear
      grscrub_errorclear

      grscrub_init_readbackmode
     
      # set data verification
      if {$datacheck == "ffc"} {
        reg_write $REG(GRSCRUB.CONFIG) 0x00001024
        log_puts [format "      FFC detection selected"]
      } elseif  {$datacheck == "crc"} {
        reg_write $REG(GRSCRUB.CONFIG) 0x00000824
        log_puts [format "      CRC detection selected"]
      } else {
        #all
        reg_write $REG(GRSCRUB.CONFIG) 0x00001824
        log_puts [format "      All detection methods selected: FFC + CRC"]
      }

      # if periodic scrubbing
      if {$scrun == 1} {
        set config_reg [expr ([reg_read $REG(GRSCRUB.CONFIG)])]
        #configuration reg -> scrun = 1
        reg_write $REG(GRSCRUB.CONFIG) [expr $config_reg | 0x2]
        log_puts_silent [format "grscrub - grscrub_readbackfpga_correction - Periodic scrubbing enabled"]
      } 

      grscrub_enable
      set start_time [start_timer]
      
      log_puts [format "    Scrubbing FPGA configuration memory..."]

      # wait OPDONE or SCRERR bitfield of Status register
      while {([expr { 0x00000010 & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != 0x00000010) && ($grmon::interrupt != 1) &&
             (([expr { 0x000001E0 & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] == 0x00000000) || 
              ([expr { 0x000001E0 & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] == 0x000000A0))}  {
        log spinner
        # wait if not done
        after 100
      }
      set end_time [end_timer]

      # check error
      if {(([expr { 0x00000020 & [expr [reg_read $REG(GRSCRUB.STAT)]]}] == 0x00000020) && 
           ([expr { 0x000001E0 & [expr [reg_read $REG(GRSCRUB.STAT)]]}] != 0x000000A0))} {
        log_puts [format "ERROR to readback FPGA!!!"]
      } else {
        set run_error [expr [reg_read $REG(GRSCRUB.ECNT)] & 0x0000FFFF]
        set uncor_error [expr [expr [reg_read $REG(GRSCRUB.ECNT)] & 0xFFFF0000] >> 16]
        set correct_errors [expr $run_error-$uncor_error]
        log_puts [format "    GRSCRUB FPGA readback successfully"]
        log_puts [format "      Readback mismatches:  $run_error"]
        log_puts [format "      Correctable errors:   $correct_errors"]
        log_puts [format "      Uncorrectable errors: $uncor_error"]
        log_puts [format "      FPGA readback time:   %s ms" [expr $end_time-$start_time]]
      }

      grscrub_disable
    }


    ####################################################################################
    ### Blind Scrubbing
    ####################################################################################

    ## Configure the GRSCRUB for blind scrubbing operation mode
    proc grscrub_init_blindscrubmode {} \
    {
      variable REG
      variable BITPARAMS
      variable FCNT 
      variable FLEN
      variable scrun
      variable FPGA_IDCODE
      variable NUM_SCRUB_FRAMES
      variable FSTART_ADDR

      log_puts_silent [format "GRSCRUB init blind scrub mode"]

      # if periodic scrubbing
      if {$scrun == 1} {
        reg_write $REG(GRSCRUB.CONFIG) 0x00000022
      } else {
        reg_write $REG(GRSCRUB.CONFIG) 0x00000020
      }

      ########## FOR FPGAEI ##########################################
      if {[gr eistat]} {
        # High delay is required during FI since both FI engine and GRSCRUB use the SMAP interface
        reg_write $REG(GRSCRUB.DELAY) 0xFFFFFF
      } else {
        # no delay between executions in periodic scrubbing
        reg_write $REG(GRSCRUB.DELAY) 0x0
      }
      ###############################################################

      #golden bitstream addresses
      reg_write $REG(GRSCRUB.LGBAR) $BITPARAMS(START.BIT)
      reg_write $REG(GRSCRUB.HGBAR) $BITPARAMS(END.BIT)    

      ## Full scrubbing of the configuration memory

      #set frame address 0x0 
      reg_write $REG(GRSCRUB.LFAR) $FSTART_ADDR 

      # only mapped frames
      reg_write $REG(GRSCRUB.FCR) [expr [expr $NUM_SCRUB_FRAMES << 9] | [expr $FLEN << 2]] 

      #memory addr for write readback data
      reg_write $REG(GRSCRUB.LGRBKAR) $BITPARAMS(LOADAD.RDBK)

      #mask addr
      reg_write $REG(GRSCRUB.LMASKAR) $BITPARAMS(START.MSK)

      #start frame addr
      reg_write $REG(GRSCRUB.LGSFAR) $BITPARAMS(START.GOLD)

      #map addr in the golden memory
      reg_write $REG(GRSCRUB.LFMAPR) $BITPARAMS(LOADAD.MAP)

      reg_write $REG(GRSCRUB.IDCODE) $FPGA_IDCODE
    }

    proc grscrub_set_blindscrubbingfpga {} {
      variable REG
      variable done

      log_puts_silent [format "grscrub - grscrub_set_blindscrubbingfpga - Starting GRSCRUB blind scrubbing"]

      grscrub_disable
      grscrub_doneclear
      grscrub_errorclear

      grscrub_init_blindscrubmode

      grscrub_enable
      
      log_puts [format "grscrub - grscrub_set_blindscrubbingfpga - Blind Scrubbing is enabled"]
    }

    # Configure GRSCRUB to blind scrubbing
    # stays monitoring continually
    # use grscrub_set_blindscrubbingfpga for only enabling it and do the monitoring somewhere else
    proc grscrub_blindscrubbingfpga {} \
    {
      variable REG
      variable done

      log_puts [format "    GRSCRUB blind scrubbing"]

      grscrub_disable
      grscrub_doneclear
      grscrub_errorclear

      grscrub_init_blindscrubmode

      grscrub_enable
      set start_time [start_timer]
      
      log_puts [format "    Blind scrubbing FPGA configuration memory..."]

      # wait OPDONE or SCRERR bitfield of Status register
      while {([expr { $done & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != $done) && ($grmon::interrupt != 1) && 
             ([expr { 0x00000020 & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != 0x00000020)} {
        log spinner
        #wait if not done
        after 100
      }
      set end_time [end_timer]

      # check error 
      if {([expr [reg_read $REG(GRSCRUB.STAT)]] == $done) || 
          ([expr [reg_read $REG(GRSCRUB.STAT)]] == 0x00001010)} {
        log_puts [format "    GRSCRUB FPGA blind scrubbing successfully"]
        log_puts [format "      FPGA blind scrubbing time: %s ms" [expr $end_time-$start_time]]
      } else {
        log_puts [format "ERROR to Blind Scrubbing FPGA!!!"]
      }

      grscrub_disable
    }


    ##########################################################################################
    # Self error injection (for validation purposes only)
    ##########################################################################################

    # Configure the GRSCRUB for readback scrubbing operation mode for error injection
    proc grscrub_init_readbackmode_ei { {faddr 0x0} } \
    {
      variable REG
      variable BITPARAMS
      variable FCNT 
      variable FLEN
      variable FPGA_IDCODE
      variable NUM_SCRUB_FRAMES

      log_puts_silent [format "grscrub - grscrub_init_readbackmode - GRSCRUB init readback mode"]

      # clear error counter and frame id registers
      reg_write $REG(GRSCRUB.ECNT)       0x00000000
      reg_write $REG(GRSCRUB.ERRFRAMEID) 0x00000000
      reg_write $REG(GRSCRUB.FRAMEID)    0x00000000

      # no delay between executions in periodic scrubbing
      reg_write $REG(GRSCRUB.DELAY) 0x0

      #golden bitstream addresses
      reg_write $REG(GRSCRUB.LGBAR) $BITPARAMS(START.BIT)
      reg_write $REG(GRSCRUB.HGBAR) $BITPARAMS(END.BIT)   
      
      # read one specific frame
      reg_write $REG(GRSCRUB.LFAR) $faddr
      reg_write $REG(GRSCRUB.FCR) [expr [expr 1 << 9] | [expr $FLEN << 2]]

      #memory addr for write readback data
      reg_write $REG(GRSCRUB.LGRBKAR) $BITPARAMS(LOADAD.RDBK)

      #mask addr
      reg_write $REG(GRSCRUB.LMASKAR) $BITPARAMS(START.MSK)

      #start frame addr
      reg_write $REG(GRSCRUB.LGSFAR) $BITPARAMS(START.GOLD)

      #map addr in the golden memory
      reg_write $REG(GRSCRUB.LFMAPR) $BITPARAMS(LOADAD.MAP)

      reg_write $REG(GRSCRUB.IDCODE) $FPGA_IDCODE

      #required if crc on
      reg_write $REG(GRSCRUB.LGCRCAR) $BITPARAMS(LOADAD.CRC)
    }

    # Store readback data in the memory (NOT USED WITH SPI FLASH MEM - for debug only)
    proc grscrub_readbackfpga_tomemory_ei { {faddr 0x0} } \
    {
        variable REG
        variable done

        log_puts_silent [format "ei-grscrub - reading back selected frame to memory"]

        grscrub_disable
        grscrub_doneclear
        grscrub_errorclear

        grscrub_init_readbackmode_ei $faddr ; # used for self error injection
        reg_write $REG(GRSCRUB.CONFIG) 0x00000424

        grscrub_enable

        set start_time [start_timer]

        while {([expr { $done & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != $done) && ($grmon::interrupt != 1) &&
               ([expr { 0x00000020 & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != 0x00000020)} {
            after 100
        }

        set end_time [end_timer]  
                         

        if {([expr [reg_read $REG(GRSCRUB.STAT)]] == $done)} {
            log_puts_silent [format "ei-grscrub - FPGA frame readback time: %s ms" [expr $end_time-$start_time]] 
        } else {
            log_puts [format "ei-grscrub - ERROR to readback FPGA!!!"] 
        }

        grscrub_disable
    }

    # Configure the GRSCRUB for blind scrubbing operation mode during error injection
    proc grscrub_init_blindscrubmode_ei { {faddr 0x0} } \
    {
      variable REG
      variable FLEN
      variable FPGA_IDCODE
      variable COMP_RAMADDR

      log_puts_silent [format "GRSCRUB init blind scrub mode"]

      # no periodic scrubbing
      reg_write $REG(GRSCRUB.CONFIG) 0x00000020
      # no delay between executions in periodic scrubbing
      reg_write $REG(GRSCRUB.DELAY) 0x0

      # Set partial scrubbing for specific frame
      reg_write $REG(GRSCRUB.LFAR) $faddr  
      reg_write $REG(GRSCRUB.FCR) [expr [expr 1 << 9] | [expr $FLEN << 2]]

      #start frame addr 
      #it is assumed that the faulty frame data is stored at the beginning of the on-chip memory
      reg_write $REG(GRSCRUB.LGSFAR) $COMP_RAMADDR

      reg_write $REG(GRSCRUB.IDCODE) $FPGA_IDCODE
    }


    # Configure GRSCRUB to blind scrubbing in order to write a faulty frame to the FPGA
    proc grscrub_writeback_ei { {faddr 0x0} } \
    {
      variable REG
      variable done

      log_puts_silent [format "ei-grscrub - writting back faulty frame to FPGA via blind scrubbing"]

      grscrub_disable
      grscrub_doneclear
      grscrub_errorclear

      grscrub_init_blindscrubmode_ei $faddr

      grscrub_enable
      set start_time [start_timer]
      
      # wait OPDONE or SCRERR bitfield of Status register
      while {([expr { $done & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != $done) && ($grmon::interrupt != 1) && 
             ([expr { 0x00000020 & [expr [reg_read $REG(GRSCRUB.STAT) 0]]}] != 0x00000020)} {
        #wait if not done
        after 10
      }
      set end_time [end_timer]

      # check error 
      if {([expr [reg_read $REG(GRSCRUB.STAT)]] == $done) || 
          ([expr [reg_read $REG(GRSCRUB.STAT)]] == 0x00001010)} {
        log_puts_silent [format "ei-grscrub - FPGA frame blind crubbing time: %s ms" [expr $end_time-$start_time]]
      } else {
        log_puts [format "ei-grscrub - ERROR to Blind Scrubbing FPGA!!!"]
      }

      grscrub_disable
    }


    ####################################################################################
    ### Auxiliar procedures
    ####################################################################################

    #Return a high-resolution time value as a system-dependent integer value.
    #The unit of the value is system-dependent but should be the highest resolution
    #clock available on the system such as a CPU cycle counter. If -milliseconds is
    #specified, then the value is guaranteed to be of millisecond granularity. This
    #value should only be used for the relative measurement of elapsed time.
    proc start_timer {} {
        set start_time [clock clicks -milliseconds]
        return $start_time
    }

    proc end_timer {} {
        set end_time [clock clicks -milliseconds]
        return $end_time
    }

    # get the configured map address
    proc get_map_addr {} {
      variable REG
      return $REG(GRSCRUB.LFMAPR)
    }


    namespace ensemble create -map {
        {config}         {init_config}
        {readback_corr}  {grscrub_readbackfpga_correction}
        {readback_detec} {grscrub_readbackfpga_onlydetection}
        {blindscrubbing} {grscrub_blindscrubbingfpga}
        {get_mapaddr}    {get_map_addr}
        {get_reg}        {reg_read}
        {readframe_ei}   {grscrub_readbackfpga_tomemory_ei}
        {writeback_ei}   {grscrub_writeback_ei}
    }
}

# ================================================================================================
# main

set script_file_name [info script]
puts $script_file_name
