# ================================================================================================
# Copyright (C) 2024, Frontgrade Gaisler AB - All rights reserved
#
# Author: Adria Oliveira - Frontgrade Gaisler AB
# Filename: grscrub-ei.tcl
# Description: Script to control fault injection from GRSCRUB itself.
#              Useaful while testing with the GR-CPCI-KU060 board featuring a GR716B.
#              Note that no SW can be executed on GR716B while running this error
#              injection because the on-chip memory is used to store the faulty frame.
#
# DISCLAIMER:
# THIS CODE, AND ALL ACCOMPANYING FILES, DATA AND MATERIALS,
# ARE DISTRIBUTED "AS IS" AND WITH NO WARRANTIES OF ANY KIND,
# WHETHER EXPRESS OR IMPLIED. Good data processing procedure dictates
# that any program be thoroughly tested with non-critical data
# before relying on it. The user must assume the entire risk of
# using the program. THIS DISCLAIMER OF WARRANTY CONSTITUTES AN
# ESSENTIAL PART OF THE AGREEMENT. 
# ================================================================================================


namespace eval ei {
    fconfigure stdin -blocking 0

    # regs from GRSCRUB 
    catch {unset REG}
    
    # log files
    set ei_log ""
    set log_file_inj ""

    ### Initial configuration

    proc ei_config { {eilog ""} } {
        variable ei_log
        variable log_file_inj

        set ei_log $eilog
        if {$ei_log eq ""} {
            # open a new ei main log
            set ei_log [log log_open "eictrl" $ei_log]
            log log_silent $ei_log [format "ei - ei_config - init configuration"]
        }
        log log_puts $ei_log [format "STARTING SYSTEM CONFIGURATION"]

        # set log for error injection only
        set systemTime [clock seconds]
        set extension [clock format $systemTime -format "%Y_%b_%d_%H_%M_%S"]
        set ei_log_name [format "./logs/subrun%s/log_gr716b_eidata_%s.log" [gr grunid] $extension]
        log log_silent $ei_log [format "ei - ei_config - EI data log name - %s" $ei_log_name]
        set log_file_inj [::open $ei_log_name "w"]
        log log_silent $log_file_inj [format "EI_RUN,FRAME_ID,FRAME_ADDR,BIT_POS,WORD_ID,BIT_ID"]

        if {![grscrub config]} {
            log log_puts $ei_log [format "ei - ei_config - Error in the GRSCRUB configuration!!!"]
            return 0
        }
    }

    ####################################################################################
    ### Obtain target EI data 
    ####################################################################################

    proc get_frame_addr { {frameid} } {
        set map_addr [grscrub get_mapaddr]
        set curr_frame_addr [reg_read [expr {[grscrub get_reg $map_addr]+[expr $frameid*0x4]} ]]
        
        return $curr_frame_addr
    }

    proc get_rand_number {min max} {
        set range [expr {$max - $min + 1}]
        return [expr {$min + int(rand() * $range)}]
    }


    ####################################################################################
    ### EI for testing procedures
    ####################################################################################

    proc ei_bitflip { {current_run 1} {frame_id} {bit_pos} } {
        variable ei_log
        variable log_file_inj
        variable ONCHIP_RAM $::grscrub::COMP_RAMADDR
        
        set faddr [get_frame_addr $frame_id]
        set word_id [expr $bit_pos >> 5]
        set bit_id  [expr $bit_pos & 0x1F]
        log log_puts $ei_log [format "    Injection details" $frame_id]
        log log_puts $ei_log [format "      Faulty frame id   - %d" $frame_id]
        log log_puts $ei_log [format "      Faulty frame addr - 0x%08x" $faddr]
        log log_puts $ei_log [format "      Faulty frame bit  - %d" $bit_pos]
        log log_puts $ei_log [format "      Faulty word id    - %d" $word_id]
        log log_puts $ei_log [format "      Faulty bit id     - %d" $bit_id]
        log log_silent $log_file_inj [format $current_run,$frame_id,$faddr,$bit_pos,$word_id,$bit_id]

        # read target frame and save in the on-chip memory
        grscrub readframe_ei $faddr

        # flip target bit
        set faulty_word [reg_read [expr $ONCHIP_RAM+[expr $word_id*0x4]]]
        log log_puts $ei_log [format "    Original data word  - 0x%08x" $faulty_word]
        set faulty_word [expr $faulty_word ^ [expr 1<<$bit_id]]
        log log_puts $ei_log [format "    Faulty data word    - 0x%08x" $faulty_word]
        reg_write [expr $ONCHIP_RAM+[expr $word_id*0x4]] $faulty_word

        # writeback faulty frame to the FPGA from the on-chip memory
        grscrub writeback_ei $faddr
    }

    # random only
    proc ei_random { {current_run 1} {ei_faults 1} } {
        variable ei_log
        variable maxframe $::grscrub::NUM_SCRUB_FRAMES
        variable maxbit [expr [expr $::grscrub::FLEN*32] - 1]

        log log_puts $ei_log [format "ERROR INJECTION EXECUTION: %d" $current_run]
        log log_silent $ei_log [format "ei - ei_random - EI id %d - num rand faults in a frame %d" $current_run $ei_faults]
        
        # generate inj target data
        set frame_id [get_rand_number 0 $maxframe]
        #set faddr [get_frame_addr $frame_id]
        set bit_pos [get_rand_number 0 $maxbit]
        
        # inject error bit
        ei_bitflip $current_run $frame_id $bit_pos
        log log_silent $ei_log [format "ei - ei_random - error injected successfully"]
        log log_puts $ei_log [format "END OF ERROR INJECTION!"]
    }

    # deterministic injection
    proc ei_deterministic { {current_run 1} {ei_faults 1} {frame_id 0} {bit_pos 0}} {
        variable ei_log

        log log_puts $ei_log [format "ei - ei_deterministic - EI id %s - num sequential faults in a frame %d" $current_run $ei_faults]

        # inject error bit
        ei_bitflip $frame_id $bit_pos
        log log_puts $ei_log [format "ei - ei_deterministic - error injected successfully"]
    }

    ####################################################################################
    ### EI campaign procedures
    ####################################################################################

    # Example of random injection combined with GRSCRUB readback FFC+CRC
    # num_run: number of executions
    # ei_faults: number of errors injected per execution
    # currently only one fault is injected at once
    proc ei_run {{num_run 1} {ei_faults 1} {scrub "rdbk"} {corr_type "all"}} {
        variable ei_log
        variable log_file_inj
        set current_run 1
        set stdinx ""

        puts "\n"
        ei_config   
        puts "\n"     
        log log_puts $ei_log [format "Starting Error Injection Campaign"]      

        for {set current_run 1} {($current_run <= $num_run) && ($grmon::interrupt != 1) && ($stdinx != "x")} {incr current_run} {
            puts "\n"
            log log_puts $ei_log [format "#############################################################"]
            ei_random $current_run $ei_faults
            log log_puts $ei_log [format "============================================================="]
            log log_puts $ei_log [format "STARTING SCRUBBING MODE"]
            # run GRSCRUB to fix the fault
            if {$scrub == "rdbk"} {
                grscrub readback_corr $corr_type
            } else {
                grscrub blindscrubbing   
            }
            log log_puts $ei_log [format "END OF SCRUBBING MODE"]
            log log_puts $ei_log [format "#############################################################"]
            set stdinx [gets stdin]
        }
        puts "\n"
        log log_puts $ei_log [format "#############################################################"]
        log log_puts $ei_log [format "Running last readback correction"]          
        # last readback to confirm no errors remained
        grscrub readback_corr $corr_type
        log log_puts $ei_log [format "#############################################################"]
        puts "\n"
        log log_puts $ei_log [format "End of Error Injection Campaign"] 
        puts "\n" 
    }

    
    ####################################################################################
    ### Read and Write registers procedures
    ####################################################################################

    # Write a register. Takes a Register name from REG array, and 32-bit value
    proc reg_write {reg val} \
    {
      silent wmem $reg $val

      return 0
    }

    # Read a register. Takes a Register name from REG array
    proc reg_read {reg} \
    {
      
      set val [silent mem $reg 4]
      
      return $val
    }

    ####################################################################################
    ### Misc
    ####################################################################################

    proc ei_get_log_type {} {
      variable ei_log
      return $ei_log
    }


    # namespace

    namespace ensemble create -map {
        {config}        {ei_config}
        {rand}          {ei_random}
        {det}           {ei_deterministic}
        {run}           {ei_run}
        {get_log_type}  {ei_get_log_type}
    }
}


# ================================================================================================
# main

set script_file_name [info script]
puts $script_file_name